/*  $Id$
 *
 *  Copyright © 2009 Jérôme Guelfucci <jeromeg@xfce.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include <libxfcegui4/libxfcegui4.h>
#include <libxfce4panel/xfce-panel-plugin.h>
#include <libxfce4panel/xfce-panel-convenience.h>

#include <sys/types.h>
#include <sys/stat.h>

/* Struct containing all panel plugin data */
typedef struct
{
  XfcePanelPlugin *plugin;

  GtkWidget *button;
  GtkWidget *image;

  int style_id;
}
PluginData;



/* Protoypes */

static void
xfswitch_plugin_construct       (XfcePanelPlugin      *plugin);

static gboolean
cb_set_size                          (XfcePanelPlugin      *plugin,
                                      int                   size,
                                      PluginData           *pd);

static void
cb_free_data                         (XfcePanelPlugin      *plugin,
                                      PluginData           *pd);

static void
cb_button_clicked                    (GtkWidget            *button,
                                      PluginData           *pd);

static void
cb_response                          (GtkDialog            *dialog,
                                      gint                  response,
                                      gpointer              user_data);

static void
cb_style_set                         (XfcePanelPlugin      *plugin,
                                      gpointer              ignored,
                                      PluginData           *pd);



static gboolean
cb_set_size (XfcePanelPlugin *plugin, int size, PluginData *pd)
{
  GdkPixbuf *pb;

  int width = size - 2 - 2 * MAX (pd->button->style->xthickness,
                                  pd->button->style->ythickness);

  pb = xfce_themed_icon_load ("system-users", width);
  gtk_image_set_from_pixbuf (GTK_IMAGE (pd->image), pb);

  g_object_unref (pb);

  gtk_widget_set_size_request (GTK_WIDGET (plugin), size, size);

  return TRUE;
}



static void
cb_free_data (XfcePanelPlugin *plugin, PluginData *pd)
{
  if (pd->style_id)
  	g_signal_handler_disconnect (plugin, pd->style_id);

  pd->style_id = 0;

  g_free (pd);
}



static void
cb_response (GtkDialog *dialog, gint response, gpointer user_data)
{
    if (response == GTK_RESPONSE_YES)
    {
      GError *error = NULL;

      if (!g_spawn_command_line_async ("gdmflexiserver --new",
                                       &error))
        {
          xfce_err (error->message);
          g_error_free (error);
        }
    }

    gtk_widget_destroy (GTK_WIDGET (dialog));
}



static void
cb_button_clicked (GtkWidget *button, PluginData *pd)
{
  GtkWidget *dialog;

  dialog =
    gtk_message_dialog_new (NULL,
                            GTK_DIALOG_MODAL,
                            GTK_MESSAGE_QUESTION,
                            GTK_BUTTONS_YES_NO,
                            _("Do you really want to switch to another user?"));

  gtk_window_set_icon_name (GTK_WINDOW (dialog), "system-users");

  gtk_widget_show (GTK_WIDGET (dialog));

  g_signal_connect (G_OBJECT (dialog),
                    "response",
                    G_CALLBACK (cb_response),
                    NULL);
}



/* Set the style of the panel plugin.
plugin: a XfcePanelPlugin (a screenshooter one).
pd: the associated PluginData.
*/
static void
cb_style_set (XfcePanelPlugin *plugin, gpointer ignored, PluginData *pd)
{
  cb_set_size (plugin, xfce_panel_plugin_get_size (plugin), pd);
}



/* Create the plugin button */
static void
xfswitch_plugin_construct (XfcePanelPlugin *plugin)
{
  /* Initialise the data structs */
  PluginData *pd = g_new0 (PluginData, 1);

  xfce_textdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");

  pd->plugin = plugin;

  /* Create the panel button */
  pd->button = xfce_create_panel_button ();

  pd->image = gtk_image_new ();

  gtk_container_add (GTK_CONTAINER (pd->button),
                     GTK_WIDGET (pd->image));

  gtk_widget_set_tooltip_text (GTK_WIDGET (pd->button),
    _("Switch to another user without closing the current session"));

  gtk_widget_show_all (pd->button);

  gtk_container_add (GTK_CONTAINER (plugin), pd->button);

  xfce_panel_plugin_add_action_widget (plugin, pd->button);

  /* Set the callbacks */
  g_signal_connect (pd->button, "clicked",
                    G_CALLBACK (cb_button_clicked), pd);

  g_signal_connect (plugin, "free-data",
                    G_CALLBACK (cb_free_data), pd);

  g_signal_connect (plugin, "size-changed",
                    G_CALLBACK (cb_set_size), pd);

  pd->style_id =
      g_signal_connect (plugin, "style-set",
                        G_CALLBACK (cb_style_set), pd);
}
XFCE_PANEL_PLUGIN_REGISTER_EXTERNAL (xfswitch_plugin_construct);
