/* sp.c
 *
 * Copyright (C) 2006-2019 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */

/* Implementation by Sean Parkinson. */

#ifdef HAVE_CONFIG_H
    #include <config.h>
#endif

#include <wolfssl/wolfcrypt/settings.h>
#include <wolfssl/wolfcrypt/error-crypt.h>
#include <wolfssl/wolfcrypt/cpuid.h>
#ifdef NO_INLINE
    #include <wolfssl/wolfcrypt/misc.h>
#else
    #define WOLFSSL_MISC_INCLUDED
    #include <wolfcrypt/src/misc.c>
#endif

#if defined(WOLFSSL_HAVE_SP_RSA) || defined(WOLFSSL_HAVE_SP_DH) || \
                                    defined(WOLFSSL_HAVE_SP_ECC)

#ifdef RSA_LOW_MEM
#ifndef WOLFSSL_SP_SMALL
#define WOLFSSL_SP_SMALL
#endif
#endif

#include <wolfssl/wolfcrypt/sp.h>

#ifdef WOLFSSL_SP_ARM64_ASM
#if defined(WOLFSSL_HAVE_SP_RSA) || defined(WOLFSSL_HAVE_SP_DH)
#ifndef WOLFSSL_SP_NO_2048
/* Read big endian unsigned byte array into r.
 *
 * r  A single precision integer.
 * size  Maximum number of bytes to convert
 * a  Byte array.
 * n  Number of bytes in array to read.
 */
static void sp_2048_from_bin(sp_digit* r, int size, const byte* a, int n)
{
    int i, j = 0;
    word32 s = 0;

    r[0] = 0;
    for (i = n-1; i >= 0; i--) {
        r[j] |= (((sp_digit)a[i]) << s);
        if (s >= 56U) {
            r[j] &= 0xffffffffffffffffl;
            s = 64U - s;
            if (j + 1 >= size) {
                break;
            }
            r[++j] = (sp_digit)a[i] >> s;
            s = 8U - s;
        }
        else {
            s += 8U;
        }
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
}

/* Convert an mp_int to an array of sp_digit.
 *
 * r  A single precision integer.
 * size  Maximum number of bytes to convert
 * a  A multi-precision integer.
 */
static void sp_2048_from_mp(sp_digit* r, int size, const mp_int* a)
{
#if DIGIT_BIT == 64
    int j;

    XMEMCPY(r, a->dp, sizeof(sp_digit) * a->used);

    for (j = a->used; j < size; j++) {
        r[j] = 0;
    }
#elif DIGIT_BIT > 64
    int i, j = 0;
    word32 s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i] << s);
        r[j] &= 0xffffffffffffffffl;
        s = 64U - s;
        if (j + 1 >= size) {
            break;
        }
        /* lint allow cast of mismatch word32 and mp_digit */
        r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
        while ((s + 64U) <= (word32)DIGIT_BIT) {
            s += 64U;
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            if (s < (word32)DIGIT_BIT) {
                /* lint allow cast of mismatch word32 and mp_digit */
                r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
            }
            else {
                r[++j] = 0L;
            }
        }
        s = (word32)DIGIT_BIT - s;
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#else
    int i, j = 0, s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i]) << s;
        if (s + DIGIT_BIT >= 64) {
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            s = 64 - s;
            if (s == DIGIT_BIT) {
                r[++j] = 0;
                s = 0;
            }
            else {
                r[++j] = a->dp[i] >> s;
                s = DIGIT_BIT - s;
            }
        }
        else {
            s += DIGIT_BIT;
        }
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#endif
}

/* Write r as big endian to byte array.
 * Fixed length number of bytes written: 256
 *
 * r  A single precision integer.
 * a  Byte array.
 */
static void sp_2048_to_bin(sp_digit* r, byte* a)
{
    int i, j, s = 0, b;

    j = 2048 / 8 - 1;
    a[j] = 0;
    for (i=0; i<32 && j>=0; i++) {
        b = 0;
        /* lint allow cast of mismatch sp_digit and int */
        a[j--] |= (byte)(r[i] << s); b += 8 - s; /*lint !e9033*/
        if (j < 0) {
            break;
        }
        while (b < 64) {
            a[j--] = r[i] >> b; b += 8;
            if (j < 0) {
                break;
            }
        }
        s = 8 - (b - 64);
        if (j >= 0) {
            a[j] = 0;
        }
        if (s != 0) {
            j++;
        }
    }
}

#ifndef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_2048_mul_8(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[8];

    __asm__ __volatile__ (
        "ldp	x8, x9, [%[a], 0]\n\t"
        "ldp	x10, x11, [%[a], 16]\n\t"
        "ldp	x12, x13, [%[a], 32]\n\t"
        "ldp	x14, x15, [%[a], 48]\n\t"
        "ldp	x16, x17, [%[b], 0]\n\t"
        "ldp	x19, x20, [%[b], 16]\n\t"
        "ldp	x21, x22, [%[b], 32]\n\t"
        "ldp	x23, x24, [%[b], 48]\n\t"
        "#  A[0] * B[0]\n\t"
        "mul	x3, x8, x16\n\t"
        "umulh	x4, x8, x16\n\t"
        "str	x3, [%[tmp]]\n\t"
        "#  A[0] * B[1]\n\t"
        "mul	x6, x8, x17\n\t"
        "umulh	x7, x8, x17\n\t"
        "adds	x4, x4, x6\n\t"
        "adc	x5, xzr, x7\n\t"
        "#  A[1] * B[0]\n\t"
        "mul	x6, x9, x16\n\t"
        "umulh	x7, x9, x16\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "str	x4, [%[tmp], 8]\n\t"
        "#  A[0] * B[2]\n\t"
        "mul	x6, x8, x19\n\t"
        "umulh	x7, x8, x19\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[1]\n\t"
        "mul	x6, x9, x17\n\t"
        "umulh	x7, x9, x17\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[0]\n\t"
        "mul	x6, x10, x16\n\t"
        "umulh	x7, x10, x16\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 16]\n\t"
        "#  A[0] * B[3]\n\t"
        "mul	x6, x8, x20\n\t"
        "umulh	x7, x8, x20\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[2]\n\t"
        "mul	x6, x9, x19\n\t"
        "umulh	x7, x9, x19\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[1]\n\t"
        "mul	x6, x10, x17\n\t"
        "umulh	x7, x10, x17\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[0]\n\t"
        "mul	x6, x11, x16\n\t"
        "umulh	x7, x11, x16\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 24]\n\t"
        "#  A[0] * B[4]\n\t"
        "mul	x6, x8, x21\n\t"
        "umulh	x7, x8, x21\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[3]\n\t"
        "mul	x6, x9, x20\n\t"
        "umulh	x7, x9, x20\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[2]\n\t"
        "mul	x6, x10, x19\n\t"
        "umulh	x7, x10, x19\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[1]\n\t"
        "mul	x6, x11, x17\n\t"
        "umulh	x7, x11, x17\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[0]\n\t"
        "mul	x6, x12, x16\n\t"
        "umulh	x7, x12, x16\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 32]\n\t"
        "#  A[0] * B[5]\n\t"
        "mul	x6, x8, x22\n\t"
        "umulh	x7, x8, x22\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[4]\n\t"
        "mul	x6, x9, x21\n\t"
        "umulh	x7, x9, x21\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[3]\n\t"
        "mul	x6, x10, x20\n\t"
        "umulh	x7, x10, x20\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[2]\n\t"
        "mul	x6, x11, x19\n\t"
        "umulh	x7, x11, x19\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[1]\n\t"
        "mul	x6, x12, x17\n\t"
        "umulh	x7, x12, x17\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[0]\n\t"
        "mul	x6, x13, x16\n\t"
        "umulh	x7, x13, x16\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 40]\n\t"
        "#  A[0] * B[6]\n\t"
        "mul	x6, x8, x23\n\t"
        "umulh	x7, x8, x23\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[5]\n\t"
        "mul	x6, x9, x22\n\t"
        "umulh	x7, x9, x22\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[4]\n\t"
        "mul	x6, x10, x21\n\t"
        "umulh	x7, x10, x21\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[3]\n\t"
        "mul	x6, x11, x20\n\t"
        "umulh	x7, x11, x20\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[2]\n\t"
        "mul	x6, x12, x19\n\t"
        "umulh	x7, x12, x19\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[1]\n\t"
        "mul	x6, x13, x17\n\t"
        "umulh	x7, x13, x17\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[0]\n\t"
        "mul	x6, x14, x16\n\t"
        "umulh	x7, x14, x16\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 48]\n\t"
        "#  A[0] * B[7]\n\t"
        "mul	x6, x8, x24\n\t"
        "umulh	x7, x8, x24\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[6]\n\t"
        "mul	x6, x9, x23\n\t"
        "umulh	x7, x9, x23\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[5]\n\t"
        "mul	x6, x10, x22\n\t"
        "umulh	x7, x10, x22\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[4]\n\t"
        "mul	x6, x11, x21\n\t"
        "umulh	x7, x11, x21\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[3]\n\t"
        "mul	x6, x12, x20\n\t"
        "umulh	x7, x12, x20\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[2]\n\t"
        "mul	x6, x13, x19\n\t"
        "umulh	x7, x13, x19\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[1]\n\t"
        "mul	x6, x14, x17\n\t"
        "umulh	x7, x14, x17\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[0]\n\t"
        "mul	x6, x15, x16\n\t"
        "umulh	x7, x15, x16\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 56]\n\t"
        "#  A[1] * B[7]\n\t"
        "mul	x6, x9, x24\n\t"
        "umulh	x7, x9, x24\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[2] * B[6]\n\t"
        "mul	x6, x10, x23\n\t"
        "umulh	x7, x10, x23\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[5]\n\t"
        "mul	x6, x11, x22\n\t"
        "umulh	x7, x11, x22\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[4]\n\t"
        "mul	x6, x12, x21\n\t"
        "umulh	x7, x12, x21\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[3]\n\t"
        "mul	x6, x13, x20\n\t"
        "umulh	x7, x13, x20\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[2]\n\t"
        "mul	x6, x14, x19\n\t"
        "umulh	x7, x14, x19\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[1]\n\t"
        "mul	x6, x15, x17\n\t"
        "umulh	x7, x15, x17\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 64]\n\t"
        "#  A[2] * B[7]\n\t"
        "mul	x6, x10, x24\n\t"
        "umulh	x7, x10, x24\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[3] * B[6]\n\t"
        "mul	x6, x11, x23\n\t"
        "umulh	x7, x11, x23\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[5]\n\t"
        "mul	x6, x12, x22\n\t"
        "umulh	x7, x12, x22\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[4]\n\t"
        "mul	x6, x13, x21\n\t"
        "umulh	x7, x13, x21\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[3]\n\t"
        "mul	x6, x14, x20\n\t"
        "umulh	x7, x14, x20\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[2]\n\t"
        "mul	x6, x15, x19\n\t"
        "umulh	x7, x15, x19\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 72]\n\t"
        "#  A[3] * B[7]\n\t"
        "mul	x6, x11, x24\n\t"
        "umulh	x7, x11, x24\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[4] * B[6]\n\t"
        "mul	x6, x12, x23\n\t"
        "umulh	x7, x12, x23\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[5]\n\t"
        "mul	x6, x13, x22\n\t"
        "umulh	x7, x13, x22\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[4]\n\t"
        "mul	x6, x14, x21\n\t"
        "umulh	x7, x14, x21\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[3]\n\t"
        "mul	x6, x15, x20\n\t"
        "umulh	x7, x15, x20\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 80]\n\t"
        "#  A[4] * B[7]\n\t"
        "mul	x6, x12, x24\n\t"
        "umulh	x7, x12, x24\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[5] * B[6]\n\t"
        "mul	x6, x13, x23\n\t"
        "umulh	x7, x13, x23\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[5]\n\t"
        "mul	x6, x14, x22\n\t"
        "umulh	x7, x14, x22\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[4]\n\t"
        "mul	x6, x15, x21\n\t"
        "umulh	x7, x15, x21\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 88]\n\t"
        "#  A[5] * B[7]\n\t"
        "mul	x6, x13, x24\n\t"
        "umulh	x7, x13, x24\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[6] * B[6]\n\t"
        "mul	x6, x14, x23\n\t"
        "umulh	x7, x14, x23\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[5]\n\t"
        "mul	x6, x15, x22\n\t"
        "umulh	x7, x15, x22\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 96]\n\t"
        "#  A[6] * B[7]\n\t"
        "mul	x6, x14, x24\n\t"
        "umulh	x7, x14, x24\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[7] * B[6]\n\t"
        "mul	x6, x15, x23\n\t"
        "umulh	x7, x15, x23\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 104]\n\t"
        "#  A[7] * B[7]\n\t"
        "mul	x6, x15, x24\n\t"
        "umulh	x7, x15, x24\n\t"
        "adds	x5, x5, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "stp	x5, x3, [%[r], 112]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [tmp] "r" (tmp)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_2048_sqr_8(sp_digit* r, const sp_digit* a)
{
    sp_digit tmp[8];

    __asm__ __volatile__ (
        "ldp	x10, x11, [%[a], 0]\n\t"
        "ldp	x12, x13, [%[a], 16]\n\t"
        "ldp	x14, x15, [%[a], 32]\n\t"
        "ldp	x16, x17, [%[a], 48]\n\t"
        "#  A[0] * A[0]\n\t"
        "mul	x2, x10, x10\n\t"
        "umulh	x3, x10, x10\n\t"
        "str	x2, [%[tmp]]\n\t"
        "mov	x4, 0\n\t"
        "#  A[0] * A[1]\n\t"
        "mul	x8, x10, x11\n\t"
        "umulh	x9, x10, x11\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "str	x3, [%[tmp], 8]\n\t"
        "#  A[0] * A[2]\n\t"
        "mul	x8, x10, x12\n\t"
        "umulh	x9, x10, x12\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[1] * A[1]\n\t"
        "mul	x8, x11, x11\n\t"
        "umulh	x9, x11, x11\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 16]\n\t"
        "#  A[0] * A[3]\n\t"
        "mul	x8, x10, x13\n\t"
        "umulh	x9, x10, x13\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[1] * A[2]\n\t"
        "mul	x8, x11, x12\n\t"
        "umulh	x9, x11, x12\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x2, [%[tmp], 24]\n\t"
        "#  A[0] * A[4]\n\t"
        "mul	x8, x10, x14\n\t"
        "umulh	x9, x10, x14\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "#  A[1] * A[3]\n\t"
        "mul	x8, x11, x13\n\t"
        "umulh	x9, x11, x13\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "#  A[2] * A[2]\n\t"
        "mul	x8, x12, x12\n\t"
        "umulh	x9, x12, x12\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "str	x3, [%[tmp], 32]\n\t"
        "#  A[0] * A[5]\n\t"
        "mul	x5, x10, x15\n\t"
        "umulh	x6, x10, x15\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[4]\n\t"
        "mul	x8, x11, x14\n\t"
        "umulh	x9, x11, x14\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[3]\n\t"
        "mul	x8, x12, x13\n\t"
        "umulh	x9, x12, x13\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 40]\n\t"
        "#  A[0] * A[6]\n\t"
        "mul	x5, x10, x16\n\t"
        "umulh	x6, x10, x16\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[5]\n\t"
        "mul	x8, x11, x15\n\t"
        "umulh	x9, x11, x15\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[4]\n\t"
        "mul	x8, x12, x14\n\t"
        "umulh	x9, x12, x14\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[3]\n\t"
        "mul	x8, x13, x13\n\t"
        "umulh	x9, x13, x13\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 48]\n\t"
        "#  A[0] * A[7]\n\t"
        "mul	x5, x10, x17\n\t"
        "umulh	x6, x10, x17\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[6]\n\t"
        "mul	x8, x11, x16\n\t"
        "umulh	x9, x11, x16\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[5]\n\t"
        "mul	x8, x12, x15\n\t"
        "umulh	x9, x12, x15\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[4]\n\t"
        "mul	x8, x13, x14\n\t"
        "umulh	x9, x13, x14\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 56]\n\t"
        "#  A[1] * A[7]\n\t"
        "mul	x5, x11, x17\n\t"
        "umulh	x6, x11, x17\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[2] * A[6]\n\t"
        "mul	x8, x12, x16\n\t"
        "umulh	x9, x12, x16\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[5]\n\t"
        "mul	x8, x13, x15\n\t"
        "umulh	x9, x13, x15\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[4]\n\t"
        "mul	x8, x14, x14\n\t"
        "umulh	x9, x14, x14\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 64]\n\t"
        "#  A[2] * A[7]\n\t"
        "mul	x5, x12, x17\n\t"
        "umulh	x6, x12, x17\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[3] * A[6]\n\t"
        "mul	x8, x13, x16\n\t"
        "umulh	x9, x13, x16\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[5]\n\t"
        "mul	x8, x14, x15\n\t"
        "umulh	x9, x14, x15\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 72]\n\t"
        "#  A[3] * A[7]\n\t"
        "mul	x8, x13, x17\n\t"
        "umulh	x9, x13, x17\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "#  A[4] * A[6]\n\t"
        "mul	x8, x14, x16\n\t"
        "umulh	x9, x14, x16\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "#  A[5] * A[5]\n\t"
        "mul	x8, x15, x15\n\t"
        "umulh	x9, x15, x15\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "str	x3, [%[r], 80]\n\t"
        "#  A[4] * A[7]\n\t"
        "mul	x8, x14, x17\n\t"
        "umulh	x9, x14, x17\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * A[6]\n\t"
        "mul	x8, x15, x16\n\t"
        "umulh	x9, x15, x16\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 88]\n\t"
        "#  A[5] * A[7]\n\t"
        "mul	x8, x15, x17\n\t"
        "umulh	x9, x15, x17\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * A[6]\n\t"
        "mul	x8, x16, x16\n\t"
        "umulh	x9, x16, x16\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x2, [%[r], 96]\n\t"
        "#  A[6] * A[7]\n\t"
        "mul	x8, x16, x17\n\t"
        "umulh	x9, x16, x17\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "str	x3, [%[r], 104]\n\t"
        "#  A[7] * A[7]\n\t"
        "mul	x8, x17, x17\n\t"
        "umulh	x9, x17, x17\n\t"
        "adds	x4, x4, x8\n\t"
        "adc	x2, x2, x9\n\t"
        "stp	x4, x2, [%[r], 112]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [tmp] "r" (tmp)
        : "memory", "x2", "x3", "x4", "x8", "x9", "x10", "x5", "x6", "x7", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_add_8(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adds	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "cset	%[c], cs\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer and result.
 * b  A single precision integer.
 */
static sp_digit sp_2048_sub_in_place_16(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "ldp	x2, x3, [%[a], 0]\n\t"
        "ldp	x4, x5, [%[a], 16]\n\t"
        "ldp	x6, x7, [%[b], 0]\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "subs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 0]\n\t"
        "stp	x4, x5, [%[a], 16]\n\t"
        "ldp	x2, x3, [%[a], 32]\n\t"
        "ldp	x4, x5, [%[a], 48]\n\t"
        "ldp	x6, x7, [%[b], 32]\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 32]\n\t"
        "stp	x4, x5, [%[a], 48]\n\t"
        "ldp	x2, x3, [%[a], 64]\n\t"
        "ldp	x4, x5, [%[a], 80]\n\t"
        "ldp	x6, x7, [%[b], 64]\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 64]\n\t"
        "stp	x4, x5, [%[a], 80]\n\t"
        "ldp	x2, x3, [%[a], 96]\n\t"
        "ldp	x4, x5, [%[a], 112]\n\t"
        "ldp	x6, x7, [%[b], 96]\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 96]\n\t"
        "stp	x4, x5, [%[a], 112]\n\t"
        "csetm	%[c], cc\n\t"
        : [c] "+r" (c)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    return c;
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_add_16(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adds	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "cset	%[c], cs\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_2048_mask_8(sp_digit* r, const sp_digit* a, sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    int i;

    for (i=0; i<8; i++) {
        r[i] = a[i] & m;
    }
#else
    r[0] = a[0] & m;
    r[1] = a[1] & m;
    r[2] = a[2] & m;
    r[3] = a[3] & m;
    r[4] = a[4] & m;
    r[5] = a[5] & m;
    r[6] = a[6] & m;
    r[7] = a[7] & m;
#endif
}

/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
SP_NOINLINE static void sp_2048_mul_16(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit* z0 = r;
    sp_digit z1[16];
    sp_digit a1[8];
    sp_digit b1[8];
    sp_digit z2[16];
    sp_digit u, ca, cb;

    ca = sp_2048_add_8(a1, a, &a[8]);
    cb = sp_2048_add_8(b1, b, &b[8]);
    u  = ca & cb;
    sp_2048_mul_8(z1, a1, b1);
    sp_2048_mul_8(z2, &a[8], &b[8]);
    sp_2048_mul_8(z0, a, b);
    sp_2048_mask_8(r + 16, a1, 0 - cb);
    sp_2048_mask_8(b1, b1, 0 - ca);
    u += sp_2048_add_8(r + 16, r + 16, b1);
    u += sp_2048_sub_in_place_16(z1, z2);
    u += sp_2048_sub_in_place_16(z1, z0);
    u += sp_2048_add_16(r + 8, r + 8, z1);
    r[24] = u;
    XMEMSET(r + 24 + 1, 0, sizeof(sp_digit) * (8 - 1));
    (void)sp_2048_add_16(r + 16, r + 16, z2);
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
SP_NOINLINE static void sp_2048_sqr_16(sp_digit* r, const sp_digit* a)
{
    sp_digit* z0 = r;
    sp_digit z2[16];
    sp_digit z1[16];
    sp_digit a1[8];
    sp_digit u;

    u = sp_2048_add_8(a1, a, &a[8]);
    sp_2048_sqr_8(z1, a1);
    sp_2048_sqr_8(z2, &a[8]);
    sp_2048_sqr_8(z0, a);
    sp_2048_mask_8(r + 16, a1, 0 - u);
    u += sp_2048_add_8(r + 16, r + 16, r + 16);
    u += sp_2048_sub_in_place_16(z1, z2);
    u += sp_2048_sub_in_place_16(z1, z0);
    u += sp_2048_add_16(r + 8, r + 8, z1);
    r[24] = u;
    XMEMSET(r + 24 + 1, 0, sizeof(sp_digit) * (8 - 1));
    (void)sp_2048_add_16(r + 16, r + 16, z2);
}

/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer and result.
 * b  A single precision integer.
 */
static sp_digit sp_2048_sub_in_place_32(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "ldp	x2, x3, [%[a], 0]\n\t"
        "ldp	x4, x5, [%[a], 16]\n\t"
        "ldp	x6, x7, [%[b], 0]\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "subs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 0]\n\t"
        "stp	x4, x5, [%[a], 16]\n\t"
        "ldp	x2, x3, [%[a], 32]\n\t"
        "ldp	x4, x5, [%[a], 48]\n\t"
        "ldp	x6, x7, [%[b], 32]\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 32]\n\t"
        "stp	x4, x5, [%[a], 48]\n\t"
        "ldp	x2, x3, [%[a], 64]\n\t"
        "ldp	x4, x5, [%[a], 80]\n\t"
        "ldp	x6, x7, [%[b], 64]\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 64]\n\t"
        "stp	x4, x5, [%[a], 80]\n\t"
        "ldp	x2, x3, [%[a], 96]\n\t"
        "ldp	x4, x5, [%[a], 112]\n\t"
        "ldp	x6, x7, [%[b], 96]\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 96]\n\t"
        "stp	x4, x5, [%[a], 112]\n\t"
        "ldp	x2, x3, [%[a], 128]\n\t"
        "ldp	x4, x5, [%[a], 144]\n\t"
        "ldp	x6, x7, [%[b], 128]\n\t"
        "ldp	x8, x9, [%[b], 144]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 128]\n\t"
        "stp	x4, x5, [%[a], 144]\n\t"
        "ldp	x2, x3, [%[a], 160]\n\t"
        "ldp	x4, x5, [%[a], 176]\n\t"
        "ldp	x6, x7, [%[b], 160]\n\t"
        "ldp	x8, x9, [%[b], 176]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 160]\n\t"
        "stp	x4, x5, [%[a], 176]\n\t"
        "ldp	x2, x3, [%[a], 192]\n\t"
        "ldp	x4, x5, [%[a], 208]\n\t"
        "ldp	x6, x7, [%[b], 192]\n\t"
        "ldp	x8, x9, [%[b], 208]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 192]\n\t"
        "stp	x4, x5, [%[a], 208]\n\t"
        "ldp	x2, x3, [%[a], 224]\n\t"
        "ldp	x4, x5, [%[a], 240]\n\t"
        "ldp	x6, x7, [%[b], 224]\n\t"
        "ldp	x8, x9, [%[b], 240]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 224]\n\t"
        "stp	x4, x5, [%[a], 240]\n\t"
        "csetm	%[c], cc\n\t"
        : [c] "+r" (c)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    return c;
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_add_32(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adds	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "ldp	x7, x8, [%[b], 128]\n\t"
        "ldp	x9, x10, [%[b], 144]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "ldp	x7, x8, [%[b], 160]\n\t"
        "ldp	x9, x10, [%[b], 176]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        "ldp	x3, x4, [%[a], 192]\n\t"
        "ldp	x5, x6, [%[a], 208]\n\t"
        "ldp	x7, x8, [%[b], 192]\n\t"
        "ldp	x9, x10, [%[b], 208]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 192]\n\t"
        "stp	x5, x6, [%[r], 208]\n\t"
        "ldp	x3, x4, [%[a], 224]\n\t"
        "ldp	x5, x6, [%[a], 240]\n\t"
        "ldp	x7, x8, [%[b], 224]\n\t"
        "ldp	x9, x10, [%[b], 240]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 224]\n\t"
        "stp	x5, x6, [%[r], 240]\n\t"
        "cset	%[c], cs\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_2048_mask_16(sp_digit* r, const sp_digit* a, sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    int i;

    for (i=0; i<16; i++) {
        r[i] = a[i] & m;
    }
#else
    int i;

    for (i = 0; i < 16; i += 8) {
        r[i+0] = a[i+0] & m;
        r[i+1] = a[i+1] & m;
        r[i+2] = a[i+2] & m;
        r[i+3] = a[i+3] & m;
        r[i+4] = a[i+4] & m;
        r[i+5] = a[i+5] & m;
        r[i+6] = a[i+6] & m;
        r[i+7] = a[i+7] & m;
    }
#endif
}

/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
SP_NOINLINE static void sp_2048_mul_32(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit* z0 = r;
    sp_digit z1[32];
    sp_digit a1[16];
    sp_digit b1[16];
    sp_digit z2[32];
    sp_digit u, ca, cb;

    ca = sp_2048_add_16(a1, a, &a[16]);
    cb = sp_2048_add_16(b1, b, &b[16]);
    u  = ca & cb;
    sp_2048_mul_16(z1, a1, b1);
    sp_2048_mul_16(z2, &a[16], &b[16]);
    sp_2048_mul_16(z0, a, b);
    sp_2048_mask_16(r + 32, a1, 0 - cb);
    sp_2048_mask_16(b1, b1, 0 - ca);
    u += sp_2048_add_16(r + 32, r + 32, b1);
    u += sp_2048_sub_in_place_32(z1, z2);
    u += sp_2048_sub_in_place_32(z1, z0);
    u += sp_2048_add_32(r + 16, r + 16, z1);
    r[48] = u;
    XMEMSET(r + 48 + 1, 0, sizeof(sp_digit) * (16 - 1));
    (void)sp_2048_add_32(r + 32, r + 32, z2);
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
SP_NOINLINE static void sp_2048_sqr_32(sp_digit* r, const sp_digit* a)
{
    sp_digit* z0 = r;
    sp_digit z2[32];
    sp_digit z1[32];
    sp_digit a1[16];
    sp_digit u;

    u = sp_2048_add_16(a1, a, &a[16]);
    sp_2048_sqr_16(z1, a1);
    sp_2048_sqr_16(z2, &a[16]);
    sp_2048_sqr_16(z0, a);
    sp_2048_mask_16(r + 32, a1, 0 - u);
    u += sp_2048_add_16(r + 32, r + 32, r + 32);
    u += sp_2048_sub_in_place_32(z1, z2);
    u += sp_2048_sub_in_place_32(z1, z0);
    u += sp_2048_add_32(r + 16, r + 16, z1);
    r[48] = u;
    XMEMSET(r + 48 + 1, 0, sizeof(sp_digit) * (16 - 1));
    (void)sp_2048_add_32(r + 32, r + 32, z2);
}

#endif /* !WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_add_32(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x11, %[a], 256\n\t"
        "\n1:\n\t"
        "adds	%[c], %[c], #-1\n\t"
        "ldp	x3, x4, [%[a]], #16\n\t"
        "ldp	x5, x6, [%[a]], #16\n\t"
        "ldp	x7, x8, [%[b]], #16\n\t"
        "ldp	x9, x10, [%[b]], #16\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r]], #16\n\t"
        "stp	x5, x6, [%[r]], #16\n\t"
        "cset	%[c], cs\n\t"
        "cmp	%[a], x11\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [r] "+r" (r), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_sub_in_place_32(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x10, %[a], 256\n\t"
        "\n1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldp	x2, x3, [%[a]]\n\t"
        "ldp	x4, x5, [%[a], #16]\n\t"
        "ldp	x6, x7, [%[b]], #16\n\t"
        "ldp	x8, x9, [%[b]], #16\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a]], #16\n\t"
        "stp	x4, x5, [%[a]], #16\n\t"
        "csetm	%[c], cc\n\t"
        "cmp	%[a], x10\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_2048_mul_32(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[64];

    __asm__ __volatile__ (
        "mov	x5, 0\n\t"
        "mov	x6, 0\n\t"
        "mov	x7, 0\n\t"
        "mov	x8, 0\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 248\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[b], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 256\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 496\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_2048_sqr_32(sp_digit* r, const sp_digit* a)
{
    sp_digit tmp[64];

    __asm__ __volatile__ (
        "mov	x6, 0\n\t"
        "mov	x7, 0\n\t"
        "mov	x8, 0\n\t"
        "mov	x5, 0\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 248\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "cmp	x4, x3\n\t"
        "b.eq	4f\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[a], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "b.al	5f\n\t"
        "\n4:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "mul	x9, x10, x10\n\t"
        "umulh	x10, x10, x10\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "\n5:\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 256\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x4\n\t"
        "b.gt	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 496\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

#endif /* WOLFSSL_SP_SMALL */
#if (defined(WOLFSSL_HAVE_SP_RSA) || defined(WOLFSSL_HAVE_SP_DH)) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)
#ifdef WOLFSSL_SP_SMALL
/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_2048_mask_16(sp_digit* r, const sp_digit* a, sp_digit m)
{
    int i;

    for (i=0; i<16; i++) {
        r[i] = a[i] & m;
    }
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_add_16(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x11, %[a], 128\n\t"
        "\n1:\n\t"
        "adds	%[c], %[c], #-1\n\t"
        "ldp	x3, x4, [%[a]], #16\n\t"
        "ldp	x5, x6, [%[a]], #16\n\t"
        "ldp	x7, x8, [%[b]], #16\n\t"
        "ldp	x9, x10, [%[b]], #16\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r]], #16\n\t"
        "stp	x5, x6, [%[r]], #16\n\t"
        "cset	%[c], cs\n\t"
        "cmp	%[a], x11\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [r] "+r" (r), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_sub_in_place_16(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x10, %[a], 128\n\t"
        "\n1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldp	x2, x3, [%[a]]\n\t"
        "ldp	x4, x5, [%[a], #16]\n\t"
        "ldp	x6, x7, [%[b]], #16\n\t"
        "ldp	x8, x9, [%[b]], #16\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a]], #16\n\t"
        "stp	x4, x5, [%[a]], #16\n\t"
        "csetm	%[c], cc\n\t"
        "cmp	%[a], x10\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_2048_mul_16(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[32];

    __asm__ __volatile__ (
        "mov	x5, 0\n\t"
        "mov	x6, 0\n\t"
        "mov	x7, 0\n\t"
        "mov	x8, 0\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 120\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[b], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 128\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 240\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_2048_sqr_16(sp_digit* r, const sp_digit* a)
{
    sp_digit tmp[32];

    __asm__ __volatile__ (
        "mov	x6, 0\n\t"
        "mov	x7, 0\n\t"
        "mov	x8, 0\n\t"
        "mov	x5, 0\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 120\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "cmp	x4, x3\n\t"
        "b.eq	4f\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[a], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "b.al	5f\n\t"
        "\n4:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "mul	x9, x10, x10\n\t"
        "umulh	x10, x10, x10\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "\n5:\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 128\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x4\n\t"
        "b.gt	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 240\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

#endif /* WOLFSSL_SP_SMALL */
#endif /* (WOLFSSL_HAVE_SP_RSA || WOLFSSL_HAVE_SP_DH) && !WOLFSSL_RSA_PUBLIC_ONLY */

/* Caclulate the bottom digit of -1/a mod 2^n.
 *
 * a    A single precision number.
 * rho  Bottom word of inverse.
 */
static void sp_2048_mont_setup(const sp_digit* a, sp_digit* rho)
{
    sp_digit x, b;

    b = a[0];
    x = (((b + 2) & 4) << 1) + b; /* here x*a==1 mod 2**4 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**8 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**16 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**32 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**64 */

    /* rho = -1/m mod b */
    *rho = -x;
}

/* Mul a by digit b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision digit.
 */
static void sp_2048_mul_d_32(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x5, %[b], x8\n\t"
        "umulh	x3, %[b], x8\n\t"
        "mov	x4, 0\n\t"
        "str	x5, [%[r]]\n\t"
        "mov	x5, 0\n\t"
        "mov	x9, #8\n\t"
        "1:\n\t"
        "ldr	x8, [%[a], x9]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "str	x3, [%[r], x9]\n\t"
        "mov	x3, x4\n\t"
        "mov	x4, x5\n\t"
        "mov	x5, #0\n\t"
        "add	x9, x9, #8\n\t"
        "cmp	x9, 256\n\t"
        "b.lt	1b\n\t"
        "str	x3, [%[r], 256]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8"
    );
#else
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x3, %[b], x8\n\t"
        "umulh	x4, %[b], x8\n\t"
        "mov	x5, 0\n\t"
        "str	x3, [%[r]]\n\t"
        "# A[1] * B\n\t"
        "ldr		x8, [%[a], 8]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 8]\n\t"
        "# A[2] * B\n\t"
        "ldr		x8, [%[a], 16]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 16]\n\t"
        "# A[3] * B\n\t"
        "ldr		x8, [%[a], 24]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 24]\n\t"
        "# A[4] * B\n\t"
        "ldr		x8, [%[a], 32]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 32]\n\t"
        "# A[5] * B\n\t"
        "ldr		x8, [%[a], 40]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 40]\n\t"
        "# A[6] * B\n\t"
        "ldr		x8, [%[a], 48]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 48]\n\t"
        "# A[7] * B\n\t"
        "ldr		x8, [%[a], 56]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 56]\n\t"
        "# A[8] * B\n\t"
        "ldr		x8, [%[a], 64]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 64]\n\t"
        "# A[9] * B\n\t"
        "ldr		x8, [%[a], 72]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 72]\n\t"
        "# A[10] * B\n\t"
        "ldr		x8, [%[a], 80]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 80]\n\t"
        "# A[11] * B\n\t"
        "ldr		x8, [%[a], 88]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 88]\n\t"
        "# A[12] * B\n\t"
        "ldr		x8, [%[a], 96]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 96]\n\t"
        "# A[13] * B\n\t"
        "ldr		x8, [%[a], 104]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 104]\n\t"
        "# A[14] * B\n\t"
        "ldr		x8, [%[a], 112]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 112]\n\t"
        "# A[15] * B\n\t"
        "ldr		x8, [%[a], 120]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 120]\n\t"
        "# A[16] * B\n\t"
        "ldr		x8, [%[a], 128]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 128]\n\t"
        "# A[17] * B\n\t"
        "ldr		x8, [%[a], 136]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 136]\n\t"
        "# A[18] * B\n\t"
        "ldr		x8, [%[a], 144]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 144]\n\t"
        "# A[19] * B\n\t"
        "ldr		x8, [%[a], 152]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 152]\n\t"
        "# A[20] * B\n\t"
        "ldr		x8, [%[a], 160]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 160]\n\t"
        "# A[21] * B\n\t"
        "ldr		x8, [%[a], 168]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 168]\n\t"
        "# A[22] * B\n\t"
        "ldr		x8, [%[a], 176]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 176]\n\t"
        "# A[23] * B\n\t"
        "ldr		x8, [%[a], 184]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 184]\n\t"
        "# A[24] * B\n\t"
        "ldr		x8, [%[a], 192]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 192]\n\t"
        "# A[25] * B\n\t"
        "ldr		x8, [%[a], 200]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 200]\n\t"
        "# A[26] * B\n\t"
        "ldr		x8, [%[a], 208]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 208]\n\t"
        "# A[27] * B\n\t"
        "ldr		x8, [%[a], 216]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 216]\n\t"
        "# A[28] * B\n\t"
        "ldr		x8, [%[a], 224]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 224]\n\t"
        "# A[29] * B\n\t"
        "ldr		x8, [%[a], 232]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 232]\n\t"
        "# A[30] * B\n\t"
        "ldr		x8, [%[a], 240]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 240]\n\t"
        "# A[31] * B\n\t"
        "ldr	x8, [%[a], 248]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adc	x5, x5, x7\n\t"
        "str	x4, [%[r], 248]\n\t"
        "str	x5, [%[r], 256]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8"
    );
#endif
}

#if (defined(WOLFSSL_HAVE_SP_RSA) || defined(WOLFSSL_HAVE_SP_DH)) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)
/* r = 2^n mod m where n is the number of bits to reduce by.
 * Given m must be 2048 bits, just need to subtract.
 *
 * r  A single precision number.
 * m  A signle precision number.
 */
static void sp_2048_mont_norm_16(sp_digit* r, const sp_digit* m)
{
    XMEMSET(r, 0, sizeof(sp_digit) * 16);

    /* r = 2^n mod m */
    sp_2048_sub_in_place_16(r, m);
}

/* Conditionally subtract b from a using the mask m.
 * m is -1 to subtract and 0 when not copying.
 *
 * r  A single precision number representing condition subtract result.
 * a  A single precision number to subtract from.
 * b  A single precision number to subtract.
 * m  Mask value to apply.
 */
static sp_digit sp_2048_cond_sub_16(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    sp_digit c = 0;

#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x8, #0\n\t"
        "1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldr	x4, [%[a], x8]\n\t"
        "ldr	x5, [%[b], x8]\n\t"
        "and	x5, x5, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "csetm	%[c], cc\n\t"
        "str	x4, [%[r], x8]\n\t"
        "add	x8, x8, #8\n\t"
        "cmp	x8, 128\n\t"
        "b.lt	1b\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8"
    );
#else
    __asm__ __volatile__ (

        "ldr		x4, [%[a], 0]\n\t"
        "ldr		x6, [%[a], 8]\n\t"
        "ldr		x5, [%[b], 0]\n\t"
        "ldr		x7, [%[b], 8]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "subs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 0]\n\t"
        "str		x6, [%[r], 8]\n\t"
        "ldr		x4, [%[a], 16]\n\t"
        "ldr		x6, [%[a], 24]\n\t"
        "ldr		x5, [%[b], 16]\n\t"
        "ldr		x7, [%[b], 24]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 16]\n\t"
        "str		x6, [%[r], 24]\n\t"
        "ldr		x4, [%[a], 32]\n\t"
        "ldr		x6, [%[a], 40]\n\t"
        "ldr		x5, [%[b], 32]\n\t"
        "ldr		x7, [%[b], 40]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 32]\n\t"
        "str		x6, [%[r], 40]\n\t"
        "ldr		x4, [%[a], 48]\n\t"
        "ldr		x6, [%[a], 56]\n\t"
        "ldr		x5, [%[b], 48]\n\t"
        "ldr		x7, [%[b], 56]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 48]\n\t"
        "str		x6, [%[r], 56]\n\t"
        "ldr		x4, [%[a], 64]\n\t"
        "ldr		x6, [%[a], 72]\n\t"
        "ldr		x5, [%[b], 64]\n\t"
        "ldr		x7, [%[b], 72]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 64]\n\t"
        "str		x6, [%[r], 72]\n\t"
        "ldr		x4, [%[a], 80]\n\t"
        "ldr		x6, [%[a], 88]\n\t"
        "ldr		x5, [%[b], 80]\n\t"
        "ldr		x7, [%[b], 88]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 80]\n\t"
        "str		x6, [%[r], 88]\n\t"
        "ldr		x4, [%[a], 96]\n\t"
        "ldr		x6, [%[a], 104]\n\t"
        "ldr		x5, [%[b], 96]\n\t"
        "ldr		x7, [%[b], 104]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 96]\n\t"
        "str		x6, [%[r], 104]\n\t"
        "ldr		x4, [%[a], 112]\n\t"
        "ldr		x6, [%[a], 120]\n\t"
        "ldr		x5, [%[b], 112]\n\t"
        "ldr		x7, [%[b], 120]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 112]\n\t"
        "str		x6, [%[r], 120]\n\t"
        "csetm	%[c], cc\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8"
    );
#endif /* WOLFSSL_SP_SMALL */

    return c;
}

/* Reduce the number back to 2048 bits using Montgomery reduction.
 *
 * a   A single precision number to reduce in place.
 * m   The single precision number representing the modulus.
 * mp  The digit representing the negative inverse of m mod 2^n.
 */
SP_NOINLINE static void sp_2048_mont_reduce_16(sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    sp_digit ca = 0;

    __asm__ __volatile__ (
        "ldp       x12, x13, [%[m], 0]\n\t"
        "ldp       x14, x15, [%[m], 16]\n\t"
        "ldp       x16, x17, [%[m], 32]\n\t"
        "ldp       x19, x20, [%[m], 48]\n\t"
        "ldp       x21, x22, [%[m], 64]\n\t"
        "ldp       x23, x24, [%[m], 80]\n\t"
        "ldp       x25, x26, [%[m], 96]\n\t"
        "ldp       x27, x28, [%[m], 112]\n\t"
        "# i = 0\n\t"
        "mov	x3, 0\n\t"
        "ldp	x10, x11, [%[a], 0]\n\t"
        "\n1:\n\t"
        "# mu = a[i] * mp\n\t"
        "mul	x8, %[mp], x10\n\t"
        "# a[i+0] += m[0] * mu\n\t"
        "ldr	x9, [%[a], 0]\n\t"
        "mul		x6, x12, x8\n\t"
        "umulh	x7, x12, x8\n\t"
        "adds	x10, x10, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "# a[i+1] += m[1] * mu\n\t"
        "ldr	x9, [%[a], 8]\n\t"
        "mul		x6, x13, x8\n\t"
        "umulh	x7, x13, x8\n\t"
        "adds	x10, x11, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x10, x10, x5\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+2] += m[2] * mu\n\t"
        "ldr	x11, [%[a], 16]\n\t"
        "mul		x6, x14, x8\n\t"
        "umulh	x7, x14, x8\n\t"
        "adds	x11, x11, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x11, x11, x4\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+3] += m[3] * mu\n\t"
        "ldr	x9, [%[a], 24]\n\t"
        "mul		x6, x15, x8\n\t"
        "umulh	x7, x15, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 24]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+4] += m[4] * mu\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "mul		x6, x16, x8\n\t"
        "umulh	x7, x16, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 32]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+5] += m[5] * mu\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "mul		x6, x17, x8\n\t"
        "umulh	x7, x17, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 40]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+6] += m[6] * mu\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "mul		x6, x19, x8\n\t"
        "umulh	x7, x19, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 48]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+7] += m[7] * mu\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "mul		x6, x20, x8\n\t"
        "umulh	x7, x20, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 56]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+8] += m[8] * mu\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "mul		x6, x21, x8\n\t"
        "umulh	x7, x21, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 64]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+9] += m[9] * mu\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "mul		x6, x22, x8\n\t"
        "umulh	x7, x22, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 72]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+10] += m[10] * mu\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "mul		x6, x23, x8\n\t"
        "umulh	x7, x23, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 80]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+11] += m[11] * mu\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "mul		x6, x24, x8\n\t"
        "umulh	x7, x24, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 88]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+12] += m[12] * mu\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "mul		x6, x25, x8\n\t"
        "umulh	x7, x25, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 96]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+13] += m[13] * mu\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "mul		x6, x26, x8\n\t"
        "umulh	x7, x26, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 104]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+14] += m[14] * mu\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "mul		x6, x27, x8\n\t"
        "umulh	x7, x27, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 112]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+15] += m[15] * mu\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "mul	x6, x28, x8\n\t"
        "umulh	x7, x28, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x7, x7, %[ca]\n\t"
        "cset  %[ca], cs\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 120]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "adcs	x9, x9, x7\n\t"
        "str	x9, [%[a], 128]\n\t"
        "adc	%[ca], %[ca], xzr\n\t"
        "# i += 1\n\t"
        "add	%[a], %[a], 8\n\t"
        "add	x3, x3, 8\n\t"
        "cmp	x3, 128\n\t"
        "blt	1b\n\t"
        "str	x10, [%[a], 0]\n\t"
        "str	x11, [%[a], 8]\n\t"
        : [ca] "+r" (ca), [a] "+r" (a)
        : [m] "r" (m), [mp] "r" (mp)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );

    sp_2048_cond_sub_16(a - 16, a, m, (sp_digit)0 - ca);
}

/* Multiply two Montogmery form numbers mod the modulus (prime).
 * (r = a * b mod m)
 *
 * r   Result of multiplication.
 * a   First number to multiply in Montogmery form.
 * b   Second number to multiply in Montogmery form.
 * m   Modulus (prime).
 * mp  Montogmery mulitplier.
 */
static void sp_2048_mont_mul_16(sp_digit* r, const sp_digit* a, const sp_digit* b,
        const sp_digit* m, sp_digit mp)
{
    sp_2048_mul_16(r, a, b);
    sp_2048_mont_reduce_16(r, m, mp);
}

/* Square the Montgomery form number. (r = a * a mod m)
 *
 * r   Result of squaring.
 * a   Number to square in Montogmery form.
 * m   Modulus (prime).
 * mp  Montogmery mulitplier.
 */
static void sp_2048_mont_sqr_16(sp_digit* r, const sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    sp_2048_sqr_16(r, a);
    sp_2048_mont_reduce_16(r, m, mp);
}

/* Mul a by digit b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision digit.
 */
static void sp_2048_mul_d_16(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x5, %[b], x8\n\t"
        "umulh	x3, %[b], x8\n\t"
        "mov	x4, 0\n\t"
        "str	x5, [%[r]]\n\t"
        "mov	x5, 0\n\t"
        "mov	x9, #8\n\t"
        "1:\n\t"
        "ldr	x8, [%[a], x9]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "str	x3, [%[r], x9]\n\t"
        "mov	x3, x4\n\t"
        "mov	x4, x5\n\t"
        "mov	x5, #0\n\t"
        "add	x9, x9, #8\n\t"
        "cmp	x9, 128\n\t"
        "b.lt	1b\n\t"
        "str	x3, [%[r], 128]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8"
    );
#else
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x3, %[b], x8\n\t"
        "umulh	x4, %[b], x8\n\t"
        "mov	x5, 0\n\t"
        "str	x3, [%[r]]\n\t"
        "# A[1] * B\n\t"
        "ldr		x8, [%[a], 8]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 8]\n\t"
        "# A[2] * B\n\t"
        "ldr		x8, [%[a], 16]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 16]\n\t"
        "# A[3] * B\n\t"
        "ldr		x8, [%[a], 24]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 24]\n\t"
        "# A[4] * B\n\t"
        "ldr		x8, [%[a], 32]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 32]\n\t"
        "# A[5] * B\n\t"
        "ldr		x8, [%[a], 40]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 40]\n\t"
        "# A[6] * B\n\t"
        "ldr		x8, [%[a], 48]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 48]\n\t"
        "# A[7] * B\n\t"
        "ldr		x8, [%[a], 56]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 56]\n\t"
        "# A[8] * B\n\t"
        "ldr		x8, [%[a], 64]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 64]\n\t"
        "# A[9] * B\n\t"
        "ldr		x8, [%[a], 72]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 72]\n\t"
        "# A[10] * B\n\t"
        "ldr		x8, [%[a], 80]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 80]\n\t"
        "# A[11] * B\n\t"
        "ldr		x8, [%[a], 88]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 88]\n\t"
        "# A[12] * B\n\t"
        "ldr		x8, [%[a], 96]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 96]\n\t"
        "# A[13] * B\n\t"
        "ldr		x8, [%[a], 104]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 104]\n\t"
        "# A[14] * B\n\t"
        "ldr		x8, [%[a], 112]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 112]\n\t"
        "# A[15] * B\n\t"
        "ldr	x8, [%[a], 120]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x3, [%[r], 120]\n\t"
        "str	x4, [%[r], 128]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8"
    );
#endif
}

/* Divide the double width number (d1|d0) by the dividend. (d1|d0 / div)
 *
 * d1   The high order half of the number to divide.
 * d0   The low order half of the number to divide.
 * div  The dividend.
 * returns the result of the division.
 */
static sp_digit div_2048_word_16(sp_digit d1, sp_digit d0, sp_digit div)
{
    sp_digit r;

    __asm__ __volatile__ (
        "lsr	x5, %[div], 32\n\t"
        "add	x5, x5, 1\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x6, x3, 32\n\t"
        "mul	x4, %[div], x6\n\t"
        "umulh	x3, %[div], x6\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x3, x3, 32\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "lsl	x3, %[d1], 32\n\t"
        "orr	x3, x3, %[d0], lsr 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "lsl	x3, %[d1], 32\n\t"
        "orr	x3, x3, %[d0], lsr 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "udiv	x3, %[d0], %[div]\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x3, %[div], x3\n\t"
        "sub	%[d0], %[d0], x3\n\t"
        "mov	%[r], x6\n\t"

        : [r] "=r" (r)
        : [d1] "r" (d1), [d0] "r" (d0), [div] "r" (div)
        : "x3", "x4", "x5", "x6"
    );

    return r;
}

/* Compare a with b in constant time.
 *
 * a  A single precision integer.
 * b  A single precision integer.
 * return -ve, 0 or +ve if a is less than, equal to or greater than b
 * respectively.
 */
static int64_t sp_2048_cmp_16(const sp_digit* a, const sp_digit* b)
{
    sp_digit r = -1;
    sp_digit one = 1;

#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x3, -1\n\t"
        "mov	x6, 120\n\t"
        "1:\n\t"
        "ldr	x4, [%[a], x6]\n\t"
        "ldr	x5, [%[b], x6]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "subs	x6, x6, #8\n\t"
        "b.cs	1b\n\t"
        "eor	%[r], %[r], x3\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [one] "r" (one)
        : "x2", "x3", "x4", "x5", "x6"
    );
#else
    __asm__ __volatile__ (
        "mov	x3, -1\n\t"
        "ldr		x4, [%[a], 120]\n\t"
        "ldr		x5, [%[b], 120]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 112]\n\t"
        "ldr		x5, [%[b], 112]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 104]\n\t"
        "ldr		x5, [%[b], 104]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 96]\n\t"
        "ldr		x5, [%[b], 96]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 88]\n\t"
        "ldr		x5, [%[b], 88]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 80]\n\t"
        "ldr		x5, [%[b], 80]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 72]\n\t"
        "ldr		x5, [%[b], 72]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 64]\n\t"
        "ldr		x5, [%[b], 64]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 56]\n\t"
        "ldr		x5, [%[b], 56]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 48]\n\t"
        "ldr		x5, [%[b], 48]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 40]\n\t"
        "ldr		x5, [%[b], 40]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 32]\n\t"
        "ldr		x5, [%[b], 32]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 24]\n\t"
        "ldr		x5, [%[b], 24]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 16]\n\t"
        "ldr		x5, [%[b], 16]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 8]\n\t"
        "ldr		x5, [%[b], 8]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 0]\n\t"
        "ldr		x5, [%[b], 0]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "eor	%[r], %[r], x3\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [one] "r" (one)
        : "x2", "x3", "x4", "x5", "x6"
    );
#endif

    return r;
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Nmber to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_2048_div_16(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[32], t2[17];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[15];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 16);
    for (i=15; i>=0; i--) {
        r1 = div_2048_word_16(t1[16 + i], t1[16 + i - 1], div);

        sp_2048_mul_d_16(t2, d, r1);
        t1[16 + i] += sp_2048_sub_in_place_16(&t1[i], t2);
        t1[16 + i] -= t2[16];
        sp_2048_mask_16(t2, d, t1[16 + i]);
        t1[16 + i] += sp_2048_add_16(&t1[i], &t1[i], t2);
        sp_2048_mask_16(t2, d, t1[16 + i]);
        t1[16 + i] += sp_2048_add_16(&t1[i], &t1[i], t2);
    }

    r1 = sp_2048_cmp_16(t1, d) >= 0;
    sp_2048_cond_sub_16(r, t1, d, (sp_digit)0 - r1);

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_2048_mod_16(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_2048_div_16(a, m, NULL, r);
}

#ifdef WOLFSSL_SP_SMALL
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns 0 on success and MEMORY_E on dynamic memory allocation failure.
 */
static int sp_2048_mod_exp_16(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#ifndef WOLFSSL_SMALL_STACK
    sp_digit t[16][32];
#else
    sp_digit* t[16];
    sp_digit* td;
#endif
    sp_digit* norm;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c, y;
    int err = MP_OKAY;

#ifdef WOLFSSL_SMALL_STACK
    td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 16 * 32, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER);
    if (td == NULL) {
        err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
#ifdef WOLFSSL_SMALL_STACK
        for (i=0; i<16; i++) {
            t[i] = td + i * 32;
        }
#endif
        norm = t[0];

        sp_2048_mont_setup(m, &mp);
        sp_2048_mont_norm_16(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 16U);
        if (reduceA != 0) {
            err = sp_2048_mod_16(t[1] + 16, a, m);
            if (err == MP_OKAY) {
                err = sp_2048_mod_16(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 16, a, sizeof(sp_digit) * 16);
            err = sp_2048_mod_16(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_2048_mont_sqr_16(t[ 2], t[ 1], m, mp);
        sp_2048_mont_mul_16(t[ 3], t[ 2], t[ 1], m, mp);
        sp_2048_mont_sqr_16(t[ 4], t[ 2], m, mp);
        sp_2048_mont_mul_16(t[ 5], t[ 3], t[ 2], m, mp);
        sp_2048_mont_sqr_16(t[ 6], t[ 3], m, mp);
        sp_2048_mont_mul_16(t[ 7], t[ 4], t[ 3], m, mp);
        sp_2048_mont_sqr_16(t[ 8], t[ 4], m, mp);
        sp_2048_mont_mul_16(t[ 9], t[ 5], t[ 4], m, mp);
        sp_2048_mont_sqr_16(t[10], t[ 5], m, mp);
        sp_2048_mont_mul_16(t[11], t[ 6], t[ 5], m, mp);
        sp_2048_mont_sqr_16(t[12], t[ 6], m, mp);
        sp_2048_mont_mul_16(t[13], t[ 7], t[ 6], m, mp);
        sp_2048_mont_sqr_16(t[14], t[ 7], m, mp);
        sp_2048_mont_mul_16(t[15], t[ 8], t[ 7], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 4;
        if (c == 64) {
            c = 60;
        }
        y = (int)(n >> c);
        n <<= 64 - c;
        XMEMCPY(r, t[y], sizeof(sp_digit) * 16);
        for (; i>=0 || c>=4; ) {
            if (c == 0) {
                n = e[i--];
                y = n >> 60;
                n <<= 4;
                c = 60;
            }
            else if (c < 4) {
                y = n >> 60;
                n = e[i--];
                c = 4 - c;
                y |= n >> (64 - c);
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (n >> 60) & 0xf;
                n <<= 4;
                c -= 4;
            }

            sp_2048_mont_sqr_16(r, r, m, mp);
            sp_2048_mont_sqr_16(r, r, m, mp);
            sp_2048_mont_sqr_16(r, r, m, mp);
            sp_2048_mont_sqr_16(r, r, m, mp);

            sp_2048_mont_mul_16(r, r, t[y], m, mp);
        }

        XMEMSET(&r[16], 0, sizeof(sp_digit) * 16U);
        sp_2048_mont_reduce_16(r, m, mp);

        mask = 0 - (sp_2048_cmp_16(r, m) >= 0);
        sp_2048_cond_sub_16(r, r, m, mask);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (td != NULL) {
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return err;
}
#else
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns 0 on success and MEMORY_E on dynamic memory allocation failure.
 */
static int sp_2048_mod_exp_16(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#ifndef WOLFSSL_SMALL_STACK
    sp_digit t[32][32];
#else
    sp_digit* t[32];
    sp_digit* td;
#endif
    sp_digit* norm;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c, y;
    int err = MP_OKAY;

#ifdef WOLFSSL_SMALL_STACK
    td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 32 * 32, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER);
    if (td == NULL) {
        err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
#ifdef WOLFSSL_SMALL_STACK
        for (i=0; i<32; i++) {
            t[i] = td + i * 32;
        }
#endif
        norm = t[0];

        sp_2048_mont_setup(m, &mp);
        sp_2048_mont_norm_16(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 16U);
        if (reduceA != 0) {
            err = sp_2048_mod_16(t[1] + 16, a, m);
            if (err == MP_OKAY) {
                err = sp_2048_mod_16(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 16, a, sizeof(sp_digit) * 16);
            err = sp_2048_mod_16(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_2048_mont_sqr_16(t[ 2], t[ 1], m, mp);
        sp_2048_mont_mul_16(t[ 3], t[ 2], t[ 1], m, mp);
        sp_2048_mont_sqr_16(t[ 4], t[ 2], m, mp);
        sp_2048_mont_mul_16(t[ 5], t[ 3], t[ 2], m, mp);
        sp_2048_mont_sqr_16(t[ 6], t[ 3], m, mp);
        sp_2048_mont_mul_16(t[ 7], t[ 4], t[ 3], m, mp);
        sp_2048_mont_sqr_16(t[ 8], t[ 4], m, mp);
        sp_2048_mont_mul_16(t[ 9], t[ 5], t[ 4], m, mp);
        sp_2048_mont_sqr_16(t[10], t[ 5], m, mp);
        sp_2048_mont_mul_16(t[11], t[ 6], t[ 5], m, mp);
        sp_2048_mont_sqr_16(t[12], t[ 6], m, mp);
        sp_2048_mont_mul_16(t[13], t[ 7], t[ 6], m, mp);
        sp_2048_mont_sqr_16(t[14], t[ 7], m, mp);
        sp_2048_mont_mul_16(t[15], t[ 8], t[ 7], m, mp);
        sp_2048_mont_sqr_16(t[16], t[ 8], m, mp);
        sp_2048_mont_mul_16(t[17], t[ 9], t[ 8], m, mp);
        sp_2048_mont_sqr_16(t[18], t[ 9], m, mp);
        sp_2048_mont_mul_16(t[19], t[10], t[ 9], m, mp);
        sp_2048_mont_sqr_16(t[20], t[10], m, mp);
        sp_2048_mont_mul_16(t[21], t[11], t[10], m, mp);
        sp_2048_mont_sqr_16(t[22], t[11], m, mp);
        sp_2048_mont_mul_16(t[23], t[12], t[11], m, mp);
        sp_2048_mont_sqr_16(t[24], t[12], m, mp);
        sp_2048_mont_mul_16(t[25], t[13], t[12], m, mp);
        sp_2048_mont_sqr_16(t[26], t[13], m, mp);
        sp_2048_mont_mul_16(t[27], t[14], t[13], m, mp);
        sp_2048_mont_sqr_16(t[28], t[14], m, mp);
        sp_2048_mont_mul_16(t[29], t[15], t[14], m, mp);
        sp_2048_mont_sqr_16(t[30], t[15], m, mp);
        sp_2048_mont_mul_16(t[31], t[16], t[15], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 5;
        if (c == 64) {
            c = 59;
        }
        y = (int)(n >> c);
        n <<= 64 - c;
        XMEMCPY(r, t[y], sizeof(sp_digit) * 16);
        for (; i>=0 || c>=5; ) {
            if (c == 0) {
                n = e[i--];
                y = n >> 59;
                n <<= 5;
                c = 59;
            }
            else if (c < 5) {
                y = n >> 59;
                n = e[i--];
                c = 5 - c;
                y |= n >> (64 - c);
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (n >> 59) & 0x1f;
                n <<= 5;
                c -= 5;
            }

            sp_2048_mont_sqr_16(r, r, m, mp);
            sp_2048_mont_sqr_16(r, r, m, mp);
            sp_2048_mont_sqr_16(r, r, m, mp);
            sp_2048_mont_sqr_16(r, r, m, mp);
            sp_2048_mont_sqr_16(r, r, m, mp);

            sp_2048_mont_mul_16(r, r, t[y], m, mp);
        }

        XMEMSET(&r[16], 0, sizeof(sp_digit) * 16U);
        sp_2048_mont_reduce_16(r, m, mp);

        mask = 0 - (sp_2048_cmp_16(r, m) >= 0);
        sp_2048_cond_sub_16(r, r, m, mask);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (td != NULL) {
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return err;
}
#endif /* WOLFSSL_SP_SMALL */

#endif /* (WOLFSSL_HAVE_SP_RSA || WOLFSSL_HAVE_SP_DH) && !WOLFSSL_RSA_PUBLIC_ONLY */

#if defined(WOLFSSL_HAVE_SP_RSA) || defined(WOLFSSL_HAVE_SP_DH)
/* r = 2^n mod m where n is the number of bits to reduce by.
 * Given m must be 2048 bits, just need to subtract.
 *
 * r  A single precision number.
 * m  A signle precision number.
 */
static void sp_2048_mont_norm_32(sp_digit* r, const sp_digit* m)
{
    XMEMSET(r, 0, sizeof(sp_digit) * 32);

    /* r = 2^n mod m */
    sp_2048_sub_in_place_32(r, m);
}

#endif /* WOLFSSL_HAVE_SP_RSA || WOLFSSL_HAVE_SP_DH */
/* Conditionally subtract b from a using the mask m.
 * m is -1 to subtract and 0 when not copying.
 *
 * r  A single precision number representing condition subtract result.
 * a  A single precision number to subtract from.
 * b  A single precision number to subtract.
 * m  Mask value to apply.
 */
static sp_digit sp_2048_cond_sub_32(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    sp_digit c = 0;

#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x8, #0\n\t"
        "1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldr	x4, [%[a], x8]\n\t"
        "ldr	x5, [%[b], x8]\n\t"
        "and	x5, x5, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "csetm	%[c], cc\n\t"
        "str	x4, [%[r], x8]\n\t"
        "add	x8, x8, #8\n\t"
        "cmp	x8, 256\n\t"
        "b.lt	1b\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8"
    );
#else
    __asm__ __volatile__ (

        "ldr		x4, [%[a], 0]\n\t"
        "ldr		x6, [%[a], 8]\n\t"
        "ldr		x5, [%[b], 0]\n\t"
        "ldr		x7, [%[b], 8]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "subs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 0]\n\t"
        "str		x6, [%[r], 8]\n\t"
        "ldr		x4, [%[a], 16]\n\t"
        "ldr		x6, [%[a], 24]\n\t"
        "ldr		x5, [%[b], 16]\n\t"
        "ldr		x7, [%[b], 24]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 16]\n\t"
        "str		x6, [%[r], 24]\n\t"
        "ldr		x4, [%[a], 32]\n\t"
        "ldr		x6, [%[a], 40]\n\t"
        "ldr		x5, [%[b], 32]\n\t"
        "ldr		x7, [%[b], 40]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 32]\n\t"
        "str		x6, [%[r], 40]\n\t"
        "ldr		x4, [%[a], 48]\n\t"
        "ldr		x6, [%[a], 56]\n\t"
        "ldr		x5, [%[b], 48]\n\t"
        "ldr		x7, [%[b], 56]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 48]\n\t"
        "str		x6, [%[r], 56]\n\t"
        "ldr		x4, [%[a], 64]\n\t"
        "ldr		x6, [%[a], 72]\n\t"
        "ldr		x5, [%[b], 64]\n\t"
        "ldr		x7, [%[b], 72]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 64]\n\t"
        "str		x6, [%[r], 72]\n\t"
        "ldr		x4, [%[a], 80]\n\t"
        "ldr		x6, [%[a], 88]\n\t"
        "ldr		x5, [%[b], 80]\n\t"
        "ldr		x7, [%[b], 88]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 80]\n\t"
        "str		x6, [%[r], 88]\n\t"
        "ldr		x4, [%[a], 96]\n\t"
        "ldr		x6, [%[a], 104]\n\t"
        "ldr		x5, [%[b], 96]\n\t"
        "ldr		x7, [%[b], 104]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 96]\n\t"
        "str		x6, [%[r], 104]\n\t"
        "ldr		x4, [%[a], 112]\n\t"
        "ldr		x6, [%[a], 120]\n\t"
        "ldr		x5, [%[b], 112]\n\t"
        "ldr		x7, [%[b], 120]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 112]\n\t"
        "str		x6, [%[r], 120]\n\t"
        "ldr		x4, [%[a], 128]\n\t"
        "ldr		x6, [%[a], 136]\n\t"
        "ldr		x5, [%[b], 128]\n\t"
        "ldr		x7, [%[b], 136]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 128]\n\t"
        "str		x6, [%[r], 136]\n\t"
        "ldr		x4, [%[a], 144]\n\t"
        "ldr		x6, [%[a], 152]\n\t"
        "ldr		x5, [%[b], 144]\n\t"
        "ldr		x7, [%[b], 152]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 144]\n\t"
        "str		x6, [%[r], 152]\n\t"
        "ldr		x4, [%[a], 160]\n\t"
        "ldr		x6, [%[a], 168]\n\t"
        "ldr		x5, [%[b], 160]\n\t"
        "ldr		x7, [%[b], 168]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 160]\n\t"
        "str		x6, [%[r], 168]\n\t"
        "ldr		x4, [%[a], 176]\n\t"
        "ldr		x6, [%[a], 184]\n\t"
        "ldr		x5, [%[b], 176]\n\t"
        "ldr		x7, [%[b], 184]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 176]\n\t"
        "str		x6, [%[r], 184]\n\t"
        "ldr		x4, [%[a], 192]\n\t"
        "ldr		x6, [%[a], 200]\n\t"
        "ldr		x5, [%[b], 192]\n\t"
        "ldr		x7, [%[b], 200]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 192]\n\t"
        "str		x6, [%[r], 200]\n\t"
        "ldr		x4, [%[a], 208]\n\t"
        "ldr		x6, [%[a], 216]\n\t"
        "ldr		x5, [%[b], 208]\n\t"
        "ldr		x7, [%[b], 216]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 208]\n\t"
        "str		x6, [%[r], 216]\n\t"
        "ldr		x4, [%[a], 224]\n\t"
        "ldr		x6, [%[a], 232]\n\t"
        "ldr		x5, [%[b], 224]\n\t"
        "ldr		x7, [%[b], 232]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 224]\n\t"
        "str		x6, [%[r], 232]\n\t"
        "ldr		x4, [%[a], 240]\n\t"
        "ldr		x6, [%[a], 248]\n\t"
        "ldr		x5, [%[b], 240]\n\t"
        "ldr		x7, [%[b], 248]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 240]\n\t"
        "str		x6, [%[r], 248]\n\t"
        "csetm	%[c], cc\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8"
    );
#endif /* WOLFSSL_SP_SMALL */

    return c;
}

/* Reduce the number back to 2048 bits using Montgomery reduction.
 *
 * a   A single precision number to reduce in place.
 * m   The single precision number representing the modulus.
 * mp  The digit representing the negative inverse of m mod 2^n.
 */
SP_NOINLINE static void sp_2048_mont_reduce_32(sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    sp_digit ca = 0;

    __asm__ __volatile__ (
        "ldp       x12, x13, [%[m], 0]\n\t"
        "ldp       x14, x15, [%[m], 16]\n\t"
        "ldp       x16, x17, [%[m], 32]\n\t"
        "ldp       x19, x20, [%[m], 48]\n\t"
        "ldp       x21, x22, [%[m], 64]\n\t"
        "ldp       x23, x24, [%[m], 80]\n\t"
        "ldp       x25, x26, [%[m], 96]\n\t"
        "ldp       x27, x28, [%[m], 112]\n\t"
        "# i = 0\n\t"
        "mov	x3, 0\n\t"
        "ldp	x10, x11, [%[a], 0]\n\t"
        "\n1:\n\t"
        "# mu = a[i] * mp\n\t"
        "mul	x8, %[mp], x10\n\t"
        "# a[i+0] += m[0] * mu\n\t"
        "ldr	x9, [%[a], 0]\n\t"
        "mul		x6, x12, x8\n\t"
        "umulh	x7, x12, x8\n\t"
        "adds	x10, x10, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "# a[i+1] += m[1] * mu\n\t"
        "ldr	x9, [%[a], 8]\n\t"
        "mul		x6, x13, x8\n\t"
        "umulh	x7, x13, x8\n\t"
        "adds	x10, x11, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x10, x10, x5\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+2] += m[2] * mu\n\t"
        "ldr	x11, [%[a], 16]\n\t"
        "mul		x6, x14, x8\n\t"
        "umulh	x7, x14, x8\n\t"
        "adds	x11, x11, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x11, x11, x4\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+3] += m[3] * mu\n\t"
        "ldr	x9, [%[a], 24]\n\t"
        "mul		x6, x15, x8\n\t"
        "umulh	x7, x15, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 24]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+4] += m[4] * mu\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "mul		x6, x16, x8\n\t"
        "umulh	x7, x16, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 32]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+5] += m[5] * mu\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "mul		x6, x17, x8\n\t"
        "umulh	x7, x17, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 40]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+6] += m[6] * mu\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "mul		x6, x19, x8\n\t"
        "umulh	x7, x19, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 48]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+7] += m[7] * mu\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "mul		x6, x20, x8\n\t"
        "umulh	x7, x20, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 56]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+8] += m[8] * mu\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "mul		x6, x21, x8\n\t"
        "umulh	x7, x21, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 64]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+9] += m[9] * mu\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "mul		x6, x22, x8\n\t"
        "umulh	x7, x22, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 72]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+10] += m[10] * mu\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "mul		x6, x23, x8\n\t"
        "umulh	x7, x23, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 80]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+11] += m[11] * mu\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "mul		x6, x24, x8\n\t"
        "umulh	x7, x24, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 88]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+12] += m[12] * mu\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "mul		x6, x25, x8\n\t"
        "umulh	x7, x25, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 96]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+13] += m[13] * mu\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "mul		x6, x26, x8\n\t"
        "umulh	x7, x26, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 104]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+14] += m[14] * mu\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "mul		x6, x27, x8\n\t"
        "umulh	x7, x27, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 112]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+15] += m[15] * mu\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "mul		x6, x28, x8\n\t"
        "umulh	x7, x28, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 120]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+16] += m[16] * mu\n\t"
        "ldr		x7, [%[m], 128]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 128]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+17] += m[17] * mu\n\t"
        "ldr		x7, [%[m], 136]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 136]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+18] += m[18] * mu\n\t"
        "ldr		x7, [%[m], 144]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 144]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+19] += m[19] * mu\n\t"
        "ldr		x7, [%[m], 152]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 152]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+20] += m[20] * mu\n\t"
        "ldr		x7, [%[m], 160]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 160]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+21] += m[21] * mu\n\t"
        "ldr		x7, [%[m], 168]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 168]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+22] += m[22] * mu\n\t"
        "ldr		x7, [%[m], 176]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 176]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+23] += m[23] * mu\n\t"
        "ldr		x7, [%[m], 184]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 184]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+24] += m[24] * mu\n\t"
        "ldr		x7, [%[m], 192]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 192]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+25] += m[25] * mu\n\t"
        "ldr		x7, [%[m], 200]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 200]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+26] += m[26] * mu\n\t"
        "ldr		x7, [%[m], 208]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 208]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+27] += m[27] * mu\n\t"
        "ldr		x7, [%[m], 216]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 216]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+28] += m[28] * mu\n\t"
        "ldr		x7, [%[m], 224]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 224]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+29] += m[29] * mu\n\t"
        "ldr		x7, [%[m], 232]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 232]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+30] += m[30] * mu\n\t"
        "ldr		x7, [%[m], 240]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 240]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+31] += m[31] * mu\n\t"
        "ldr	x7, [%[m], 248]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x7, x7, %[ca]\n\t"
        "cset  %[ca], cs\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 248]\n\t"
        "ldr	x9, [%[a], 256]\n\t"
        "adcs	x9, x9, x7\n\t"
        "str	x9, [%[a], 256]\n\t"
        "adc	%[ca], %[ca], xzr\n\t"
        "# i += 1\n\t"
        "add	%[a], %[a], 8\n\t"
        "add	x3, x3, 8\n\t"
        "cmp	x3, 256\n\t"
        "blt	1b\n\t"
        "str	x10, [%[a], 0]\n\t"
        "str	x11, [%[a], 8]\n\t"
        : [ca] "+r" (ca), [a] "+r" (a)
        : [m] "r" (m), [mp] "r" (mp)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );

    sp_2048_cond_sub_32(a - 32, a, m, (sp_digit)0 - ca);
}

/* Multiply two Montogmery form numbers mod the modulus (prime).
 * (r = a * b mod m)
 *
 * r   Result of multiplication.
 * a   First number to multiply in Montogmery form.
 * b   Second number to multiply in Montogmery form.
 * m   Modulus (prime).
 * mp  Montogmery mulitplier.
 */
static void sp_2048_mont_mul_32(sp_digit* r, const sp_digit* a, const sp_digit* b,
        const sp_digit* m, sp_digit mp)
{
    sp_2048_mul_32(r, a, b);
    sp_2048_mont_reduce_32(r, m, mp);
}

/* Square the Montgomery form number. (r = a * a mod m)
 *
 * r   Result of squaring.
 * a   Number to square in Montogmery form.
 * m   Modulus (prime).
 * mp  Montogmery mulitplier.
 */
static void sp_2048_mont_sqr_32(sp_digit* r, const sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    sp_2048_sqr_32(r, a);
    sp_2048_mont_reduce_32(r, m, mp);
}

/* Divide the double width number (d1|d0) by the dividend. (d1|d0 / div)
 *
 * d1   The high order half of the number to divide.
 * d0   The low order half of the number to divide.
 * div  The dividend.
 * returns the result of the division.
 */
static sp_digit div_2048_word_32(sp_digit d1, sp_digit d0, sp_digit div)
{
    sp_digit r;

    __asm__ __volatile__ (
        "lsr	x5, %[div], 32\n\t"
        "add	x5, x5, 1\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x6, x3, 32\n\t"
        "mul	x4, %[div], x6\n\t"
        "umulh	x3, %[div], x6\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x3, x3, 32\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "lsl	x3, %[d1], 32\n\t"
        "orr	x3, x3, %[d0], lsr 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "lsl	x3, %[d1], 32\n\t"
        "orr	x3, x3, %[d0], lsr 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "udiv	x3, %[d0], %[div]\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x3, %[div], x3\n\t"
        "sub	%[d0], %[d0], x3\n\t"
        "mov	%[r], x6\n\t"

        : [r] "=r" (r)
        : [d1] "r" (d1), [d0] "r" (d0), [div] "r" (div)
        : "x3", "x4", "x5", "x6"
    );

    return r;
}

/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_2048_mask_32(sp_digit* r, const sp_digit* a, sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    int i;

    for (i=0; i<32; i++) {
        r[i] = a[i] & m;
    }
#else
    int i;

    for (i = 0; i < 32; i += 8) {
        r[i+0] = a[i+0] & m;
        r[i+1] = a[i+1] & m;
        r[i+2] = a[i+2] & m;
        r[i+3] = a[i+3] & m;
        r[i+4] = a[i+4] & m;
        r[i+5] = a[i+5] & m;
        r[i+6] = a[i+6] & m;
        r[i+7] = a[i+7] & m;
    }
#endif
}

/* Compare a with b in constant time.
 *
 * a  A single precision integer.
 * b  A single precision integer.
 * return -ve, 0 or +ve if a is less than, equal to or greater than b
 * respectively.
 */
static int64_t sp_2048_cmp_32(const sp_digit* a, const sp_digit* b)
{
    sp_digit r = -1;
    sp_digit one = 1;

#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x3, -1\n\t"
        "mov	x6, 248\n\t"
        "1:\n\t"
        "ldr	x4, [%[a], x6]\n\t"
        "ldr	x5, [%[b], x6]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "subs	x6, x6, #8\n\t"
        "b.cs	1b\n\t"
        "eor	%[r], %[r], x3\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [one] "r" (one)
        : "x2", "x3", "x4", "x5", "x6"
    );
#else
    __asm__ __volatile__ (
        "mov	x3, -1\n\t"
        "ldr		x4, [%[a], 248]\n\t"
        "ldr		x5, [%[b], 248]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 240]\n\t"
        "ldr		x5, [%[b], 240]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 232]\n\t"
        "ldr		x5, [%[b], 232]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 224]\n\t"
        "ldr		x5, [%[b], 224]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 216]\n\t"
        "ldr		x5, [%[b], 216]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 208]\n\t"
        "ldr		x5, [%[b], 208]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 200]\n\t"
        "ldr		x5, [%[b], 200]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 192]\n\t"
        "ldr		x5, [%[b], 192]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 184]\n\t"
        "ldr		x5, [%[b], 184]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 176]\n\t"
        "ldr		x5, [%[b], 176]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 168]\n\t"
        "ldr		x5, [%[b], 168]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 160]\n\t"
        "ldr		x5, [%[b], 160]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 152]\n\t"
        "ldr		x5, [%[b], 152]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 144]\n\t"
        "ldr		x5, [%[b], 144]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 136]\n\t"
        "ldr		x5, [%[b], 136]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 128]\n\t"
        "ldr		x5, [%[b], 128]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 120]\n\t"
        "ldr		x5, [%[b], 120]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 112]\n\t"
        "ldr		x5, [%[b], 112]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 104]\n\t"
        "ldr		x5, [%[b], 104]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 96]\n\t"
        "ldr		x5, [%[b], 96]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 88]\n\t"
        "ldr		x5, [%[b], 88]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 80]\n\t"
        "ldr		x5, [%[b], 80]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 72]\n\t"
        "ldr		x5, [%[b], 72]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 64]\n\t"
        "ldr		x5, [%[b], 64]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 56]\n\t"
        "ldr		x5, [%[b], 56]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 48]\n\t"
        "ldr		x5, [%[b], 48]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 40]\n\t"
        "ldr		x5, [%[b], 40]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 32]\n\t"
        "ldr		x5, [%[b], 32]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 24]\n\t"
        "ldr		x5, [%[b], 24]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 16]\n\t"
        "ldr		x5, [%[b], 16]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 8]\n\t"
        "ldr		x5, [%[b], 8]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 0]\n\t"
        "ldr		x5, [%[b], 0]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "eor	%[r], %[r], x3\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [one] "r" (one)
        : "x2", "x3", "x4", "x5", "x6"
    );
#endif

    return r;
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Nmber to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_2048_div_32(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[64], t2[33];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[31];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 32);
    for (i=31; i>=0; i--) {
        r1 = div_2048_word_32(t1[32 + i], t1[32 + i - 1], div);

        sp_2048_mul_d_32(t2, d, r1);
        t1[32 + i] += sp_2048_sub_in_place_32(&t1[i], t2);
        t1[32 + i] -= t2[32];
        sp_2048_mask_32(t2, d, t1[32 + i]);
        t1[32 + i] += sp_2048_add_32(&t1[i], &t1[i], t2);
        sp_2048_mask_32(t2, d, t1[32 + i]);
        t1[32 + i] += sp_2048_add_32(&t1[i], &t1[i], t2);
    }

    r1 = sp_2048_cmp_32(t1, d) >= 0;
    sp_2048_cond_sub_32(r, t1, d, (sp_digit)0 - r1);

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_2048_mod_32(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_2048_div_32(a, m, NULL, r);
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Nmber to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_2048_div_32_cond(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[64], t2[33];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[31];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 32);
    for (i=31; i>=0; i--) {
        r1 = div_2048_word_32(t1[32 + i], t1[32 + i - 1], div);

        sp_2048_mul_d_32(t2, d, r1);
        t1[32 + i] += sp_2048_sub_in_place_32(&t1[i], t2);
        t1[32 + i] -= t2[32];
        if (t1[32 + i] != 0) {
            t1[32 + i] += sp_2048_add_32(&t1[i], &t1[i], d);
            if (t1[32 + i] != 0)
                t1[32 + i] += sp_2048_add_32(&t1[i], &t1[i], d);
        }
    }

    r1 = sp_2048_cmp_32(t1, d) >= 0;
    sp_2048_cond_sub_32(r, t1, d, (sp_digit)0 - r1);

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_2048_mod_32_cond(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_2048_div_32_cond(a, m, NULL, r);
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || \
                                                     defined(WOLFSSL_HAVE_SP_DH)
#ifdef WOLFSSL_SP_SMALL
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns 0 on success and MEMORY_E on dynamic memory allocation failure.
 */
static int sp_2048_mod_exp_32(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#ifndef WOLFSSL_SMALL_STACK
    sp_digit t[16][64];
#else
    sp_digit* t[16];
    sp_digit* td;
#endif
    sp_digit* norm;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c, y;
    int err = MP_OKAY;

#ifdef WOLFSSL_SMALL_STACK
    td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 16 * 64, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER);
    if (td == NULL) {
        err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
#ifdef WOLFSSL_SMALL_STACK
        for (i=0; i<16; i++) {
            t[i] = td + i * 64;
        }
#endif
        norm = t[0];

        sp_2048_mont_setup(m, &mp);
        sp_2048_mont_norm_32(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 32U);
        if (reduceA != 0) {
            err = sp_2048_mod_32(t[1] + 32, a, m);
            if (err == MP_OKAY) {
                err = sp_2048_mod_32(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 32, a, sizeof(sp_digit) * 32);
            err = sp_2048_mod_32(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_2048_mont_sqr_32(t[ 2], t[ 1], m, mp);
        sp_2048_mont_mul_32(t[ 3], t[ 2], t[ 1], m, mp);
        sp_2048_mont_sqr_32(t[ 4], t[ 2], m, mp);
        sp_2048_mont_mul_32(t[ 5], t[ 3], t[ 2], m, mp);
        sp_2048_mont_sqr_32(t[ 6], t[ 3], m, mp);
        sp_2048_mont_mul_32(t[ 7], t[ 4], t[ 3], m, mp);
        sp_2048_mont_sqr_32(t[ 8], t[ 4], m, mp);
        sp_2048_mont_mul_32(t[ 9], t[ 5], t[ 4], m, mp);
        sp_2048_mont_sqr_32(t[10], t[ 5], m, mp);
        sp_2048_mont_mul_32(t[11], t[ 6], t[ 5], m, mp);
        sp_2048_mont_sqr_32(t[12], t[ 6], m, mp);
        sp_2048_mont_mul_32(t[13], t[ 7], t[ 6], m, mp);
        sp_2048_mont_sqr_32(t[14], t[ 7], m, mp);
        sp_2048_mont_mul_32(t[15], t[ 8], t[ 7], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 4;
        if (c == 64) {
            c = 60;
        }
        y = (int)(n >> c);
        n <<= 64 - c;
        XMEMCPY(r, t[y], sizeof(sp_digit) * 32);
        for (; i>=0 || c>=4; ) {
            if (c == 0) {
                n = e[i--];
                y = n >> 60;
                n <<= 4;
                c = 60;
            }
            else if (c < 4) {
                y = n >> 60;
                n = e[i--];
                c = 4 - c;
                y |= n >> (64 - c);
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (n >> 60) & 0xf;
                n <<= 4;
                c -= 4;
            }

            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);

            sp_2048_mont_mul_32(r, r, t[y], m, mp);
        }

        XMEMSET(&r[32], 0, sizeof(sp_digit) * 32U);
        sp_2048_mont_reduce_32(r, m, mp);

        mask = 0 - (sp_2048_cmp_32(r, m) >= 0);
        sp_2048_cond_sub_32(r, r, m, mask);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (td != NULL) {
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return err;
}
#else
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns 0 on success and MEMORY_E on dynamic memory allocation failure.
 */
static int sp_2048_mod_exp_32(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#ifndef WOLFSSL_SMALL_STACK
    sp_digit t[32][64];
#else
    sp_digit* t[32];
    sp_digit* td;
#endif
    sp_digit* norm;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c, y;
    int err = MP_OKAY;

#ifdef WOLFSSL_SMALL_STACK
    td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 32 * 64, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER);
    if (td == NULL) {
        err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
#ifdef WOLFSSL_SMALL_STACK
        for (i=0; i<32; i++) {
            t[i] = td + i * 64;
        }
#endif
        norm = t[0];

        sp_2048_mont_setup(m, &mp);
        sp_2048_mont_norm_32(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 32U);
        if (reduceA != 0) {
            err = sp_2048_mod_32(t[1] + 32, a, m);
            if (err == MP_OKAY) {
                err = sp_2048_mod_32(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 32, a, sizeof(sp_digit) * 32);
            err = sp_2048_mod_32(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_2048_mont_sqr_32(t[ 2], t[ 1], m, mp);
        sp_2048_mont_mul_32(t[ 3], t[ 2], t[ 1], m, mp);
        sp_2048_mont_sqr_32(t[ 4], t[ 2], m, mp);
        sp_2048_mont_mul_32(t[ 5], t[ 3], t[ 2], m, mp);
        sp_2048_mont_sqr_32(t[ 6], t[ 3], m, mp);
        sp_2048_mont_mul_32(t[ 7], t[ 4], t[ 3], m, mp);
        sp_2048_mont_sqr_32(t[ 8], t[ 4], m, mp);
        sp_2048_mont_mul_32(t[ 9], t[ 5], t[ 4], m, mp);
        sp_2048_mont_sqr_32(t[10], t[ 5], m, mp);
        sp_2048_mont_mul_32(t[11], t[ 6], t[ 5], m, mp);
        sp_2048_mont_sqr_32(t[12], t[ 6], m, mp);
        sp_2048_mont_mul_32(t[13], t[ 7], t[ 6], m, mp);
        sp_2048_mont_sqr_32(t[14], t[ 7], m, mp);
        sp_2048_mont_mul_32(t[15], t[ 8], t[ 7], m, mp);
        sp_2048_mont_sqr_32(t[16], t[ 8], m, mp);
        sp_2048_mont_mul_32(t[17], t[ 9], t[ 8], m, mp);
        sp_2048_mont_sqr_32(t[18], t[ 9], m, mp);
        sp_2048_mont_mul_32(t[19], t[10], t[ 9], m, mp);
        sp_2048_mont_sqr_32(t[20], t[10], m, mp);
        sp_2048_mont_mul_32(t[21], t[11], t[10], m, mp);
        sp_2048_mont_sqr_32(t[22], t[11], m, mp);
        sp_2048_mont_mul_32(t[23], t[12], t[11], m, mp);
        sp_2048_mont_sqr_32(t[24], t[12], m, mp);
        sp_2048_mont_mul_32(t[25], t[13], t[12], m, mp);
        sp_2048_mont_sqr_32(t[26], t[13], m, mp);
        sp_2048_mont_mul_32(t[27], t[14], t[13], m, mp);
        sp_2048_mont_sqr_32(t[28], t[14], m, mp);
        sp_2048_mont_mul_32(t[29], t[15], t[14], m, mp);
        sp_2048_mont_sqr_32(t[30], t[15], m, mp);
        sp_2048_mont_mul_32(t[31], t[16], t[15], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 5;
        if (c == 64) {
            c = 59;
        }
        y = (int)(n >> c);
        n <<= 64 - c;
        XMEMCPY(r, t[y], sizeof(sp_digit) * 32);
        for (; i>=0 || c>=5; ) {
            if (c == 0) {
                n = e[i--];
                y = n >> 59;
                n <<= 5;
                c = 59;
            }
            else if (c < 5) {
                y = n >> 59;
                n = e[i--];
                c = 5 - c;
                y |= n >> (64 - c);
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (n >> 59) & 0x1f;
                n <<= 5;
                c -= 5;
            }

            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);

            sp_2048_mont_mul_32(r, r, t[y], m, mp);
        }

        XMEMSET(&r[32], 0, sizeof(sp_digit) * 32U);
        sp_2048_mont_reduce_32(r, m, mp);

        mask = 0 - (sp_2048_cmp_32(r, m) >= 0);
        sp_2048_cond_sub_32(r, r, m, mask);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (td != NULL) {
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return err;
}
#endif /* WOLFSSL_SP_SMALL */
#endif /* (WOLFSSL_HAVE_SP_RSA && !WOLFSSL_RSA_PUBLIC_ONLY) || WOLFSSL_HAVE_SP_DH */

#ifdef WOLFSSL_HAVE_SP_RSA
/* RSA public key operation.
 *
 * in      Array of bytes representing the number to exponentiate, base.
 * inLen   Number of bytes in base.
 * em      Public exponent.
 * mm      Modulus.
 * out     Buffer to hold big-endian bytes of exponentiation result.
 *         Must be at least 256 bytes long.
 * outLen  Number of bytes in result.
 * returns 0 on success, MP_TO_E when the outLen is too small, MP_READ_E when
 * an array is too long and MEMORY_E when dynamic memory allocation fails.
 */
int sp_RsaPublic_2048(const byte* in, word32 inLen, mp_int* em, mp_int* mm,
    byte* out, word32* outLen)
{
#if !defined(WOLFSSL_SP_SMALL) && !defined(WOLFSSL_SMALL_STACK)
    sp_digit ad[64], md[32], rd[64];
#else
    sp_digit* d = NULL;
#endif
    sp_digit* a;
    sp_digit *ah;
    sp_digit* m;
    sp_digit* r;
    sp_digit e[1];
    int err = MP_OKAY;

    if (*outLen < 256)
        err = MP_TO_E;
    if (err == MP_OKAY && (mp_count_bits(em) > 64 || inLen > 256 ||
                                                     mp_count_bits(mm) != 2048))
        err = MP_READ_E;

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    if (err == MP_OKAY) {
        d = (sp_digit*)XMALLOC(sizeof(sp_digit) * 32 * 5, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (d == NULL)
            err = MEMORY_E;
    }

    if (err == MP_OKAY) {
        a = d;
        r = a + 32 * 2;
        m = r + 32 * 2;
        ah = a + 32;
    }
#else
    a = ad;
    m = md;
    r = rd;
    ah = a + 32;
#endif

    if (err == MP_OKAY) {
        sp_2048_from_bin(ah, 32, in, inLen);
#if DIGIT_BIT >= 64
        e[0] = em->dp[0];
#else
        e[0] = em->dp[0];
        if (em->used > 1)
            e[0] |= ((sp_digit)em->dp[1]) << DIGIT_BIT;
#endif
        if (e[0] == 0)
            err = MP_EXPTMOD_E;
    }
    if (err == MP_OKAY) {
        sp_2048_from_mp(m, 32, mm);

        if (e[0] == 0x3) {
            if (err == MP_OKAY) {
                sp_2048_sqr_32(r, ah);
                err = sp_2048_mod_32_cond(r, r, m);
            }
            if (err == MP_OKAY) {
                sp_2048_mul_32(r, ah, r);
                err = sp_2048_mod_32_cond(r, r, m);
            }
        }
        else {
            int i;
            sp_digit mp;

            sp_2048_mont_setup(m, &mp);

            /* Convert to Montgomery form. */
            XMEMSET(a, 0, sizeof(sp_digit) * 32);
            err = sp_2048_mod_32_cond(a, a, m);

            if (err == MP_OKAY) {
                for (i=63; i>=0; i--)
                    if (e[0] >> i)
                        break;

                XMEMCPY(r, a, sizeof(sp_digit) * 32);
                for (i--; i>=0; i--) {
                    sp_2048_mont_sqr_32(r, r, m, mp);
                    if (((e[0] >> i) & 1) == 1)
                        sp_2048_mont_mul_32(r, r, a, m, mp);
                }
                XMEMSET(&r[32], 0, sizeof(sp_digit) * 32);
                sp_2048_mont_reduce_32(r, m, mp);

                for (i = 31; i > 0; i--) {
                    if (r[i] != m[i])
                        break;
                }
                if (r[i] >= m[i])
                    sp_2048_sub_in_place_32(r, m);
            }
        }
    }

    if (err == MP_OKAY) {
        sp_2048_to_bin(r, out);
        *outLen = 256;
    }

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    if (d != NULL)
        XFREE(d, NULL, DYNAMIC_TYPE_RSA);
#endif

    return err;
}

/* RSA private key operation.
 *
 * in      Array of bytes representing the number to exponentiate, base.
 * inLen   Number of bytes in base.
 * dm      Private exponent.
 * pm      First prime.
 * qm      Second prime.
 * dpm     First prime's CRT exponent.
 * dqm     Second prime's CRT exponent.
 * qim     Inverse of second prime mod p.
 * mm      Modulus.
 * out     Buffer to hold big-endian bytes of exponentiation result.
 *         Must be at least 256 bytes long.
 * outLen  Number of bytes in result.
 * returns 0 on success, MP_TO_E when the outLen is too small, MP_READ_E when
 * an array is too long and MEMORY_E when dynamic memory allocation fails.
 */
int sp_RsaPrivate_2048(const byte* in, word32 inLen, mp_int* dm,
    mp_int* pm, mp_int* qm, mp_int* dpm, mp_int* dqm, mp_int* qim, mp_int* mm,
    byte* out, word32* outLen)
{
#if !defined(WOLFSSL_SP_SMALL) && !defined(WOLFSSL_SMALL_STACK)
    sp_digit ad[32 * 2];
    sp_digit pd[16], qd[16], dpd[16];
    sp_digit tmpad[32], tmpbd[32];
#else
    sp_digit* t = NULL;
#endif
    sp_digit* a;
    sp_digit* p;
    sp_digit* q;
    sp_digit* dp;
    sp_digit* dq;
    sp_digit* qi;
    sp_digit* tmp;
    sp_digit* tmpa;
    sp_digit* tmpb;
    sp_digit* r;
    sp_digit c;
    int err = MP_OKAY;

    (void)dm;
    (void)mm;

    if (*outLen < 256)
        err = MP_TO_E;
    if (err == MP_OKAY && (inLen > 256 || mp_count_bits(mm) != 2048))
        err = MP_READ_E;

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    if (err == MP_OKAY) {
        t = (sp_digit*)XMALLOC(sizeof(sp_digit) * 16 * 11, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (t == NULL)
            err = MEMORY_E;
    }
    if (err == MP_OKAY) {
        a = t;
        p = a + 32 * 2;
        q = p + 16;
        qi = dq = dp = q + 16;
        tmpa = qi + 16;
        tmpb = tmpa + 32;

        tmp = t;
        r = tmp + 32;
    }
#else
    r = a = ad;
    p = pd;
    q = qd;
    qi = dq = dp = dpd;
    tmpa = tmpad;
    tmpb = tmpbd;
    tmp = a + 32;
#endif

    if (err == MP_OKAY) {
        sp_2048_from_bin(a, 32, in, inLen);
        sp_2048_from_mp(p, 16, pm);
        sp_2048_from_mp(q, 16, qm);
        sp_2048_from_mp(dp, 16, dpm);

        err = sp_2048_mod_exp_16(tmpa, a, dp, 1024, p, 1);
    }
    if (err == MP_OKAY) {
        sp_2048_from_mp(dq, 16, dqm);
        err = sp_2048_mod_exp_16(tmpb, a, dq, 1024, q, 1);
    }

    if (err == MP_OKAY) {
        c = sp_2048_sub_in_place_16(tmpa, tmpb);
        sp_2048_mask_16(tmp, p, c);
        sp_2048_add_16(tmpa, tmpa, tmp);

        sp_2048_from_mp(qi, 16, qim);
        sp_2048_mul_16(tmpa, tmpa, qi);
        err = sp_2048_mod_16(tmpa, tmpa, p);
    }

    if (err == MP_OKAY) {
        sp_2048_mul_16(tmpa, q, tmpa);
        XMEMSET(&tmpb[16], 0, sizeof(sp_digit) * 16);
        sp_2048_add_32(r, tmpb, tmpa);

        sp_2048_to_bin(r, out);
        *outLen = 256;
    }

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    if (t != NULL) {
        XMEMSET(t, 0, sizeof(sp_digit) * 16 * 11);
        XFREE(t, NULL, DYNAMIC_TYPE_RSA);
    }
#else
    XMEMSET(tmpad, 0, sizeof(tmpad));
    XMEMSET(tmpbd, 0, sizeof(tmpbd));
    XMEMSET(pd, 0, sizeof(pd));
    XMEMSET(qd, 0, sizeof(qd));
    XMEMSET(dpd, 0, sizeof(dpd));
#endif

    return err;
}
#endif /* WOLFSSL_HAVE_SP_RSA */
#if defined(WOLFSSL_HAVE_SP_DH) || (defined(WOLFSSL_HAVE_SP_RSA) && \
                                              !defined(WOLFSSL_RSA_PUBLIC_ONLY))
/* Convert an array of sp_digit to an mp_int.
 *
 * a  A single precision integer.
 * r  A multi-precision integer.
 */
static int sp_2048_to_mp(const sp_digit* a, mp_int* r)
{
    int err;

    err = mp_grow(r, (2048 + DIGIT_BIT - 1) / DIGIT_BIT);
    if (err == MP_OKAY) { /*lint !e774 case where err is always MP_OKAY*/
#if DIGIT_BIT == 64
        XMEMCPY(r->dp, a, sizeof(sp_digit) * 32);
        r->used = 32;
        mp_clamp(r);
#elif DIGIT_BIT < 64
        int i, j = 0, s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 32; i++) {
            r->dp[j] |= a[i] << s;
            r->dp[j] &= (1L << DIGIT_BIT) - 1;
            s = DIGIT_BIT - s;
            r->dp[++j] = a[i] >> s;
            while (s + DIGIT_BIT <= 64) {
                s += DIGIT_BIT;
                r->dp[j++] &= (1L << DIGIT_BIT) - 1;
                if (s == SP_WORD_SIZE) {
                    r->dp[j] = 0;
                }
                else {
                    r->dp[j] = a[i] >> s;
                }
            }
            s = 64 - s;
        }
        r->used = (2048 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#else
        int i, j = 0, s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 32; i++) {
            r->dp[j] |= ((mp_digit)a[i]) << s;
            if (s + 64 >= DIGIT_BIT) {
    #if DIGIT_BIT != 32 && DIGIT_BIT != 64
                r->dp[j] &= (1L << DIGIT_BIT) - 1;
    #endif
                s = DIGIT_BIT - s;
                r->dp[++j] = a[i] >> s;
                s = 64 - s;
            }
            else {
                s += 64;
            }
        }
        r->used = (2048 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#endif
    }

    return err;
}

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base  Base. MP integer.
 * exp   Exponent. MP integer.
 * mod   Modulus. MP integer.
 * res   Result. MP integer.
 * returs 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_ModExp_2048(mp_int* base, mp_int* exp, mp_int* mod, mp_int* res)
{
    int err = MP_OKAY;
    sp_digit b[64], e[32], m[32];
    sp_digit* r = b;
    int expBits = mp_count_bits(exp);

    if (mp_count_bits(base) > 2048) {
        err = MP_READ_E;
    }

    if (err == MP_OKAY) {
        if (expBits > 2048) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        if (mp_count_bits(mod) != 2048) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        sp_2048_from_mp(b, 32, base);
        sp_2048_from_mp(e, 32, exp);
        sp_2048_from_mp(m, 32, mod);

        err = sp_2048_mod_exp_32(r, b, e, expBits, m, 0);
    }

    if (err == MP_OKAY) {
        err = sp_2048_to_mp(r, res);
    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}

#ifdef WOLFSSL_HAVE_SP_DH

#ifdef HAVE_FFDHE_2048
static void sp_2048_lshift_32(sp_digit* r, sp_digit* a, byte n)
{
    __asm__ __volatile__ (
        "mov	x6, 63\n\t"
        "sub	x6, x6, %[n]\n\t"
        "ldr	x3, [%[a], 248]\n\t"
        "lsr	x4, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x4, x4, x6\n\t"
        "ldr	x2, [%[a], 240]\n\t"
        "str	x4, [%[r], 256]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 232]\n\t"
        "str	x3, [%[r], 248]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 224]\n\t"
        "str	x2, [%[r], 240]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 216]\n\t"
        "str	x4, [%[r], 232]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 208]\n\t"
        "str	x3, [%[r], 224]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 200]\n\t"
        "str	x2, [%[r], 216]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 192]\n\t"
        "str	x4, [%[r], 208]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 184]\n\t"
        "str	x3, [%[r], 200]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 176]\n\t"
        "str	x2, [%[r], 192]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 168]\n\t"
        "str	x4, [%[r], 184]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 160]\n\t"
        "str	x3, [%[r], 176]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 152]\n\t"
        "str	x2, [%[r], 168]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 144]\n\t"
        "str	x4, [%[r], 160]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 136]\n\t"
        "str	x3, [%[r], 152]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 128]\n\t"
        "str	x2, [%[r], 144]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 120]\n\t"
        "str	x4, [%[r], 136]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 112]\n\t"
        "str	x3, [%[r], 128]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 104]\n\t"
        "str	x2, [%[r], 120]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 96]\n\t"
        "str	x4, [%[r], 112]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 88]\n\t"
        "str	x3, [%[r], 104]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 80]\n\t"
        "str	x2, [%[r], 96]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 72]\n\t"
        "str	x4, [%[r], 88]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 64]\n\t"
        "str	x3, [%[r], 80]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 56]\n\t"
        "str	x2, [%[r], 72]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 48]\n\t"
        "str	x4, [%[r], 64]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 40]\n\t"
        "str	x3, [%[r], 56]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 32]\n\t"
        "str	x2, [%[r], 48]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 24]\n\t"
        "str	x4, [%[r], 40]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 16]\n\t"
        "str	x3, [%[r], 32]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 8]\n\t"
        "str	x2, [%[r], 24]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 0]\n\t"
        "str	x4, [%[r], 16]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "str	x2, [%[r]]\n\t"
        "str	x3, [%[r], 8]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [n] "r" (n)
        : "memory", "x2", "x3", "x4", "x5", "x6"
    );
}

/* Modular exponentiate 2 to the e mod m. (r = 2^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns 0 on success and MEMORY_E on dynamic memory allocation failure.
 */
static int sp_2048_mod_exp_2_32(sp_digit* r, const sp_digit* e, int bits,
        const sp_digit* m)
{
#ifndef WOLFSSL_SMALL_STACK
    sp_digit nd[64];
    sp_digit td[33];
#else
    sp_digit* td;
#endif
    sp_digit* norm;
    sp_digit* tmp;
    sp_digit mp = 1;
    sp_digit n, o;
    sp_digit mask;
    int i;
    int c, y;
    int err = MP_OKAY;

#ifdef WOLFSSL_SMALL_STACK
    td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 97, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER);
    if (td == NULL) {
        err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
#ifdef WOLFSSL_SMALL_STACK
        norm = td;
        tmp  = td + 64;
#else
        norm = nd;
        tmp  = td;
#endif

        sp_2048_mont_setup(m, &mp);
        sp_2048_mont_norm_32(norm, m);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 6;
        if (c == 64) {
            c = 58;
        }
        y = (int)(n >> c);
        n <<= 64 - c;
        sp_2048_lshift_32(r, norm, y);
        for (; i>=0 || c>=6; ) {
            if (c == 0) {
                n = e[i--];
                y = n >> 58;
                n <<= 6;
                c = 58;
            }
            else if (c < 6) {
                y = n >> 58;
                n = e[i--];
                c = 6 - c;
                y |= n >> (64 - c);
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (n >> 58) & 0x3f;
                n <<= 6;
                c -= 6;
            }

            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);

            sp_2048_lshift_32(r, r, y);
            sp_2048_mul_d_32(tmp, norm, r[32]);
            r[32] = 0;
            o = sp_2048_add_32(r, r, tmp);
            sp_2048_cond_sub_32(r, r, m, (sp_digit)0 - o);
        }

        XMEMSET(&r[32], 0, sizeof(sp_digit) * 32U);
        sp_2048_mont_reduce_32(r, m, mp);

        mask = 0 - (sp_2048_cmp_32(r, m) >= 0);
        sp_2048_cond_sub_32(r, r, m, mask);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (td != NULL) {
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return err;
}
#endif /* HAVE_FFDHE_2048 */

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base     Base.
 * exp      Array of bytes that is the exponent.
 * expLen   Length of data, in bytes, in exponent.
 * mod      Modulus.
 * out      Buffer to hold big-endian bytes of exponentiation result.
 *          Must be at least 256 bytes long.
 * outLen   Length, in bytes, of exponentiation result.
 * returs 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_DhExp_2048(mp_int* base, const byte* exp, word32 expLen,
    mp_int* mod, byte* out, word32* outLen)
{
    int err = MP_OKAY;
    sp_digit b[64], e[32], m[32];
    sp_digit* r = b;
    word32 i;

    if (mp_count_bits(base) > 2048) {
        err = MP_READ_E;
    }

    if (err == MP_OKAY) {
        if (expLen > 256) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        if (mp_count_bits(mod) != 2048) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        sp_2048_from_mp(b, 32, base);
        sp_2048_from_bin(e, 32, exp, expLen);
        sp_2048_from_mp(m, 32, mod);

    #ifdef HAVE_FFDHE_2048
        if (base->used == 1 && base->dp[0] == 2 && m[31] == (sp_digit)-1)
            err = sp_2048_mod_exp_2_32(r, e, expLen * 8, m);
        else
    #endif
            err = sp_2048_mod_exp_32(r, b, e, expLen * 8, m, 0);

    }

    if (err == MP_OKAY) {
        sp_2048_to_bin(r, out);
        *outLen = 256;
        for (i=0; i<256 && out[i] == 0; i++) {
        }
        *outLen -= i;
        XMEMMOVE(out, out + i, *outLen);

    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}
#endif /* WOLFSSL_HAVE_SP_DH */

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base  Base. MP integer.
 * exp   Exponent. MP integer.
 * mod   Modulus. MP integer.
 * res   Result. MP integer.
 * returs 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_ModExp_1024(mp_int* base, mp_int* exp, mp_int* mod, mp_int* res)
{
    int err = MP_OKAY;
    sp_digit b[32], e[16], m[16];
    sp_digit* r = b;
    int expBits = mp_count_bits(exp);

    if (mp_count_bits(base) > 1024) {
        err = MP_READ_E;
    }

    if (err == MP_OKAY) {
        if (expBits > 1024) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        if (mp_count_bits(mod) != 1024) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        sp_2048_from_mp(b, 16, base);
        sp_2048_from_mp(e, 16, exp);
        sp_2048_from_mp(m, 16, mod);

        err = sp_2048_mod_exp_16(r, b, e, expBits, m, 0);
    }

    if (err == MP_OKAY) {
        XMEMSET(r + 16, 0, sizeof(*r) * 16U);
        err = sp_2048_to_mp(r, res);
        res->used = mod->used;
        mp_clamp(res);
    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}

#endif /* WOLFSSL_HAVE_SP_DH || (WOLFSSL_HAVE_SP_RSA && !WOLFSSL_RSA_PUBLIC_ONLY) */

#endif /* !WOLFSSL_SP_NO_2048 */

#ifndef WOLFSSL_SP_NO_3072
/* Read big endian unsigned byte array into r.
 *
 * r  A single precision integer.
 * size  Maximum number of bytes to convert
 * a  Byte array.
 * n  Number of bytes in array to read.
 */
static void sp_3072_from_bin(sp_digit* r, int size, const byte* a, int n)
{
    int i, j = 0;
    word32 s = 0;

    r[0] = 0;
    for (i = n-1; i >= 0; i--) {
        r[j] |= (((sp_digit)a[i]) << s);
        if (s >= 56U) {
            r[j] &= 0xffffffffffffffffl;
            s = 64U - s;
            if (j + 1 >= size) {
                break;
            }
            r[++j] = (sp_digit)a[i] >> s;
            s = 8U - s;
        }
        else {
            s += 8U;
        }
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
}

/* Convert an mp_int to an array of sp_digit.
 *
 * r  A single precision integer.
 * size  Maximum number of bytes to convert
 * a  A multi-precision integer.
 */
static void sp_3072_from_mp(sp_digit* r, int size, const mp_int* a)
{
#if DIGIT_BIT == 64
    int j;

    XMEMCPY(r, a->dp, sizeof(sp_digit) * a->used);

    for (j = a->used; j < size; j++) {
        r[j] = 0;
    }
#elif DIGIT_BIT > 64
    int i, j = 0;
    word32 s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i] << s);
        r[j] &= 0xffffffffffffffffl;
        s = 64U - s;
        if (j + 1 >= size) {
            break;
        }
        /* lint allow cast of mismatch word32 and mp_digit */
        r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
        while ((s + 64U) <= (word32)DIGIT_BIT) {
            s += 64U;
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            if (s < (word32)DIGIT_BIT) {
                /* lint allow cast of mismatch word32 and mp_digit */
                r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
            }
            else {
                r[++j] = 0L;
            }
        }
        s = (word32)DIGIT_BIT - s;
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#else
    int i, j = 0, s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i]) << s;
        if (s + DIGIT_BIT >= 64) {
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            s = 64 - s;
            if (s == DIGIT_BIT) {
                r[++j] = 0;
                s = 0;
            }
            else {
                r[++j] = a->dp[i] >> s;
                s = DIGIT_BIT - s;
            }
        }
        else {
            s += DIGIT_BIT;
        }
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#endif
}

/* Write r as big endian to byte array.
 * Fixed length number of bytes written: 384
 *
 * r  A single precision integer.
 * a  Byte array.
 */
static void sp_3072_to_bin(sp_digit* r, byte* a)
{
    int i, j, s = 0, b;

    j = 3072 / 8 - 1;
    a[j] = 0;
    for (i=0; i<48 && j>=0; i++) {
        b = 0;
        /* lint allow cast of mismatch sp_digit and int */
        a[j--] |= (byte)(r[i] << s); b += 8 - s; /*lint !e9033*/
        if (j < 0) {
            break;
        }
        while (b < 64) {
            a[j--] = r[i] >> b; b += 8;
            if (j < 0) {
                break;
            }
        }
        s = 8 - (b - 64);
        if (j >= 0) {
            a[j] = 0;
        }
        if (s != 0) {
            j++;
        }
    }
}

#ifndef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_3072_mul_12(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[12];

    __asm__ __volatile__ (
        "#  A[0] * B[0]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x3, x7, x8\n\t"
        "umulh	x4, x7, x8\n\t"
        "mov	x5, 0\n\t"
        "str	x3, [%[tmp]]\n\t"
        "#  A[0] * B[1]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[0]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 8]\n\t"
        "#  A[0] * B[2]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[1]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[0]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 16]\n\t"
        "#  A[0] * B[3]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[2]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[1]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[0]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 24]\n\t"
        "#  A[0] * B[4]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[3]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[2]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[1]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[0]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 32]\n\t"
        "#  A[0] * B[5]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[4]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[3]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[2]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[1]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[0]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 40]\n\t"
        "#  A[0] * B[6]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[5]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[4]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[3]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[2]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[1]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[0]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 48]\n\t"
        "#  A[0] * B[7]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[6]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[5]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[4]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[3]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[2]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[1]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[0]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 56]\n\t"
        "#  A[0] * B[8]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[7]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[6]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[5]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[4]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[3]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[2]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[1]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * B[0]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 64]\n\t"
        "#  A[0] * B[9]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[8]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[7]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[6]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[5]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[4]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[3]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[2]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[8] * B[1]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[9] * B[0]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 72]\n\t"
        "#  A[0] * B[10]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[9]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[8]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[7]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[6]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[5]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[4]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[3]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[8] * B[2]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[9] * B[1]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * B[0]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 80]\n\t"
        "#  A[0] * B[11]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[10]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[9]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[8]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[7]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[6]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[5]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[4]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * B[3]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[9] * B[2]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[10] * B[1]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[11] * B[0]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 88]\n\t"
        "#  A[1] * B[11]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[2] * B[10]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[9]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[8]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[7]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[6]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[5]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[8] * B[4]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[9] * B[3]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[10] * B[2]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[11] * B[1]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 96]\n\t"
        "#  A[2] * B[11]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[3] * B[10]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[9]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[8]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[7]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[6]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[8] * B[5]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[9] * B[4]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * B[3]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[11] * B[2]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 104]\n\t"
        "#  A[3] * B[11]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[4] * B[10]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[9]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[8]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[7]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * B[6]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[9] * B[5]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[10] * B[4]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[11] * B[3]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 112]\n\t"
        "#  A[4] * B[11]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[5] * B[10]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[9]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[8]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[8] * B[7]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[9] * B[6]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[10] * B[5]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[11] * B[4]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 120]\n\t"
        "#  A[5] * B[11]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[6] * B[10]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[9]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[8] * B[8]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[9] * B[7]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * B[6]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[11] * B[5]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 128]\n\t"
        "#  A[6] * B[11]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[7] * B[10]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * B[9]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[9] * B[8]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[10] * B[7]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[11] * B[6]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 136]\n\t"
        "#  A[7] * B[11]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[8] * B[10]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[9] * B[9]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[10] * B[8]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[11] * B[7]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 144]\n\t"
        "#  A[8] * B[11]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[9] * B[10]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * B[9]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[11] * B[8]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 152]\n\t"
        "#  A[9] * B[11]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[10] * B[10]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[11] * B[9]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 160]\n\t"
        "#  A[10] * B[11]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[11] * B[10]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 168]\n\t"
        "#  A[11] * B[11]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adc	x5, x5, x7\n\t"
        "stp	x4, x5, [%[r], 176]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [tmp] "r" (tmp)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_3072_sqr_12(sp_digit* r, const sp_digit* a)
{
    sp_digit tmp[12];

    __asm__ __volatile__ (
        "ldp	x10, x11, [%[a], 0]\n\t"
        "ldp	x12, x13, [%[a], 16]\n\t"
        "ldp	x14, x15, [%[a], 32]\n\t"
        "ldp	x16, x17, [%[a], 48]\n\t"
        "ldp	x19, x20, [%[a], 64]\n\t"
        "ldp	x21, x22, [%[a], 80]\n\t"
        "#  A[0] * A[0]\n\t"
        "mul	x2, x10, x10\n\t"
        "umulh	x3, x10, x10\n\t"
        "str	x2, [%[tmp]]\n\t"
        "mov	x4, 0\n\t"
        "#  A[0] * A[1]\n\t"
        "mul	x8, x10, x11\n\t"
        "umulh	x9, x10, x11\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "str	x3, [%[tmp], 8]\n\t"
        "#  A[0] * A[2]\n\t"
        "mul	x8, x10, x12\n\t"
        "umulh	x9, x10, x12\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[1] * A[1]\n\t"
        "mul	x8, x11, x11\n\t"
        "umulh	x9, x11, x11\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 16]\n\t"
        "#  A[0] * A[3]\n\t"
        "mul	x8, x10, x13\n\t"
        "umulh	x9, x10, x13\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[1] * A[2]\n\t"
        "mul	x8, x11, x12\n\t"
        "umulh	x9, x11, x12\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x2, [%[tmp], 24]\n\t"
        "#  A[0] * A[4]\n\t"
        "mul	x8, x10, x14\n\t"
        "umulh	x9, x10, x14\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "#  A[1] * A[3]\n\t"
        "mul	x8, x11, x13\n\t"
        "umulh	x9, x11, x13\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "#  A[2] * A[2]\n\t"
        "mul	x8, x12, x12\n\t"
        "umulh	x9, x12, x12\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "str	x3, [%[tmp], 32]\n\t"
        "#  A[0] * A[5]\n\t"
        "mul	x5, x10, x15\n\t"
        "umulh	x6, x10, x15\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[4]\n\t"
        "mul	x8, x11, x14\n\t"
        "umulh	x9, x11, x14\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[3]\n\t"
        "mul	x8, x12, x13\n\t"
        "umulh	x9, x12, x13\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 40]\n\t"
        "#  A[0] * A[6]\n\t"
        "mul	x5, x10, x16\n\t"
        "umulh	x6, x10, x16\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[5]\n\t"
        "mul	x8, x11, x15\n\t"
        "umulh	x9, x11, x15\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[4]\n\t"
        "mul	x8, x12, x14\n\t"
        "umulh	x9, x12, x14\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[3]\n\t"
        "mul	x8, x13, x13\n\t"
        "umulh	x9, x13, x13\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 48]\n\t"
        "#  A[0] * A[7]\n\t"
        "mul	x5, x10, x17\n\t"
        "umulh	x6, x10, x17\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[6]\n\t"
        "mul	x8, x11, x16\n\t"
        "umulh	x9, x11, x16\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[5]\n\t"
        "mul	x8, x12, x15\n\t"
        "umulh	x9, x12, x15\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[4]\n\t"
        "mul	x8, x13, x14\n\t"
        "umulh	x9, x13, x14\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 56]\n\t"
        "#  A[0] * A[8]\n\t"
        "mul	x5, x10, x19\n\t"
        "umulh	x6, x10, x19\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[7]\n\t"
        "mul	x8, x11, x17\n\t"
        "umulh	x9, x11, x17\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[6]\n\t"
        "mul	x8, x12, x16\n\t"
        "umulh	x9, x12, x16\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[5]\n\t"
        "mul	x8, x13, x15\n\t"
        "umulh	x9, x13, x15\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[4]\n\t"
        "mul	x8, x14, x14\n\t"
        "umulh	x9, x14, x14\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 64]\n\t"
        "#  A[0] * A[9]\n\t"
        "mul	x5, x10, x20\n\t"
        "umulh	x6, x10, x20\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[8]\n\t"
        "mul	x8, x11, x19\n\t"
        "umulh	x9, x11, x19\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[7]\n\t"
        "mul	x8, x12, x17\n\t"
        "umulh	x9, x12, x17\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[6]\n\t"
        "mul	x8, x13, x16\n\t"
        "umulh	x9, x13, x16\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[5]\n\t"
        "mul	x8, x14, x15\n\t"
        "umulh	x9, x14, x15\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 72]\n\t"
        "#  A[0] * A[10]\n\t"
        "mul	x5, x10, x21\n\t"
        "umulh	x6, x10, x21\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[9]\n\t"
        "mul	x8, x11, x20\n\t"
        "umulh	x9, x11, x20\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[8]\n\t"
        "mul	x8, x12, x19\n\t"
        "umulh	x9, x12, x19\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[7]\n\t"
        "mul	x8, x13, x17\n\t"
        "umulh	x9, x13, x17\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[6]\n\t"
        "mul	x8, x14, x16\n\t"
        "umulh	x9, x14, x16\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[5]\n\t"
        "mul	x8, x15, x15\n\t"
        "umulh	x9, x15, x15\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 80]\n\t"
        "#  A[0] * A[11]\n\t"
        "mul	x5, x10, x22\n\t"
        "umulh	x6, x10, x22\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[10]\n\t"
        "mul	x8, x11, x21\n\t"
        "umulh	x9, x11, x21\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[9]\n\t"
        "mul	x8, x12, x20\n\t"
        "umulh	x9, x12, x20\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[8]\n\t"
        "mul	x8, x13, x19\n\t"
        "umulh	x9, x13, x19\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[7]\n\t"
        "mul	x8, x14, x17\n\t"
        "umulh	x9, x14, x17\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[6]\n\t"
        "mul	x8, x15, x16\n\t"
        "umulh	x9, x15, x16\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 88]\n\t"
        "#  A[1] * A[11]\n\t"
        "mul	x5, x11, x22\n\t"
        "umulh	x6, x11, x22\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[2] * A[10]\n\t"
        "mul	x8, x12, x21\n\t"
        "umulh	x9, x12, x21\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[9]\n\t"
        "mul	x8, x13, x20\n\t"
        "umulh	x9, x13, x20\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[8]\n\t"
        "mul	x8, x14, x19\n\t"
        "umulh	x9, x14, x19\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[7]\n\t"
        "mul	x8, x15, x17\n\t"
        "umulh	x9, x15, x17\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[6]\n\t"
        "mul	x8, x16, x16\n\t"
        "umulh	x9, x16, x16\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 96]\n\t"
        "#  A[2] * A[11]\n\t"
        "mul	x5, x12, x22\n\t"
        "umulh	x6, x12, x22\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[3] * A[10]\n\t"
        "mul	x8, x13, x21\n\t"
        "umulh	x9, x13, x21\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[9]\n\t"
        "mul	x8, x14, x20\n\t"
        "umulh	x9, x14, x20\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[8]\n\t"
        "mul	x8, x15, x19\n\t"
        "umulh	x9, x15, x19\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[7]\n\t"
        "mul	x8, x16, x17\n\t"
        "umulh	x9, x16, x17\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 104]\n\t"
        "#  A[3] * A[11]\n\t"
        "mul	x5, x13, x22\n\t"
        "umulh	x6, x13, x22\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[4] * A[10]\n\t"
        "mul	x8, x14, x21\n\t"
        "umulh	x9, x14, x21\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[9]\n\t"
        "mul	x8, x15, x20\n\t"
        "umulh	x9, x15, x20\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[8]\n\t"
        "mul	x8, x16, x19\n\t"
        "umulh	x9, x16, x19\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[7]\n\t"
        "mul	x8, x17, x17\n\t"
        "umulh	x9, x17, x17\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 112]\n\t"
        "#  A[4] * A[11]\n\t"
        "mul	x5, x14, x22\n\t"
        "umulh	x6, x14, x22\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[5] * A[10]\n\t"
        "mul	x8, x15, x21\n\t"
        "umulh	x9, x15, x21\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[9]\n\t"
        "mul	x8, x16, x20\n\t"
        "umulh	x9, x16, x20\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[8]\n\t"
        "mul	x8, x17, x19\n\t"
        "umulh	x9, x17, x19\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 120]\n\t"
        "#  A[5] * A[11]\n\t"
        "mul	x5, x15, x22\n\t"
        "umulh	x6, x15, x22\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[6] * A[10]\n\t"
        "mul	x8, x16, x21\n\t"
        "umulh	x9, x16, x21\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[9]\n\t"
        "mul	x8, x17, x20\n\t"
        "umulh	x9, x17, x20\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[8]\n\t"
        "mul	x8, x19, x19\n\t"
        "umulh	x9, x19, x19\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 128]\n\t"
        "#  A[6] * A[11]\n\t"
        "mul	x5, x16, x22\n\t"
        "umulh	x6, x16, x22\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[7] * A[10]\n\t"
        "mul	x8, x17, x21\n\t"
        "umulh	x9, x17, x21\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[9]\n\t"
        "mul	x8, x19, x20\n\t"
        "umulh	x9, x19, x20\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 136]\n\t"
        "#  A[7] * A[11]\n\t"
        "mul	x8, x17, x22\n\t"
        "umulh	x9, x17, x22\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * A[10]\n\t"
        "mul	x8, x19, x21\n\t"
        "umulh	x9, x19, x21\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[9] * A[9]\n\t"
        "mul	x8, x20, x20\n\t"
        "umulh	x9, x20, x20\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x2, [%[r], 144]\n\t"
        "#  A[8] * A[11]\n\t"
        "mul	x8, x19, x22\n\t"
        "umulh	x9, x19, x22\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "#  A[9] * A[10]\n\t"
        "mul	x8, x20, x21\n\t"
        "umulh	x9, x20, x21\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "str	x3, [%[r], 152]\n\t"
        "#  A[9] * A[11]\n\t"
        "mul	x8, x20, x22\n\t"
        "umulh	x9, x20, x22\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * A[10]\n\t"
        "mul	x8, x21, x21\n\t"
        "umulh	x9, x21, x21\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 160]\n\t"
        "#  A[10] * A[11]\n\t"
        "mul	x8, x21, x22\n\t"
        "umulh	x9, x21, x22\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x2, [%[r], 168]\n\t"
        "#  A[11] * A[11]\n\t"
        "mul	x8, x22, x22\n\t"
        "umulh	x9, x22, x22\n\t"
        "adds	x3, x3, x8\n\t"
        "adc	x4, x4, x9\n\t"
        "stp	x3, x4, [%[r], 176]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [tmp] "r" (tmp)
        : "memory", "x2", "x3", "x4", "x8", "x9", "x10", "x5", "x6", "x7", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21", "x22"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_add_12(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adds	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "cset	%[c], cs\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer and result.
 * b  A single precision integer.
 */
static sp_digit sp_3072_sub_in_place_24(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "ldp	x2, x3, [%[a], 0]\n\t"
        "ldp	x4, x5, [%[a], 16]\n\t"
        "ldp	x6, x7, [%[b], 0]\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "subs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 0]\n\t"
        "stp	x4, x5, [%[a], 16]\n\t"
        "ldp	x2, x3, [%[a], 32]\n\t"
        "ldp	x4, x5, [%[a], 48]\n\t"
        "ldp	x6, x7, [%[b], 32]\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 32]\n\t"
        "stp	x4, x5, [%[a], 48]\n\t"
        "ldp	x2, x3, [%[a], 64]\n\t"
        "ldp	x4, x5, [%[a], 80]\n\t"
        "ldp	x6, x7, [%[b], 64]\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 64]\n\t"
        "stp	x4, x5, [%[a], 80]\n\t"
        "ldp	x2, x3, [%[a], 96]\n\t"
        "ldp	x4, x5, [%[a], 112]\n\t"
        "ldp	x6, x7, [%[b], 96]\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 96]\n\t"
        "stp	x4, x5, [%[a], 112]\n\t"
        "ldp	x2, x3, [%[a], 128]\n\t"
        "ldp	x4, x5, [%[a], 144]\n\t"
        "ldp	x6, x7, [%[b], 128]\n\t"
        "ldp	x8, x9, [%[b], 144]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 128]\n\t"
        "stp	x4, x5, [%[a], 144]\n\t"
        "ldp	x2, x3, [%[a], 160]\n\t"
        "ldp	x4, x5, [%[a], 176]\n\t"
        "ldp	x6, x7, [%[b], 160]\n\t"
        "ldp	x8, x9, [%[b], 176]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 160]\n\t"
        "stp	x4, x5, [%[a], 176]\n\t"
        "csetm	%[c], cc\n\t"
        : [c] "+r" (c)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    return c;
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_add_24(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adds	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "ldp	x7, x8, [%[b], 128]\n\t"
        "ldp	x9, x10, [%[b], 144]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "ldp	x7, x8, [%[b], 160]\n\t"
        "ldp	x9, x10, [%[b], 176]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        "cset	%[c], cs\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_3072_mask_12(sp_digit* r, const sp_digit* a, sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    int i;

    for (i=0; i<12; i++) {
        r[i] = a[i] & m;
    }
#else
    r[0] = a[0] & m;
    r[1] = a[1] & m;
    r[2] = a[2] & m;
    r[3] = a[3] & m;
    r[4] = a[4] & m;
    r[5] = a[5] & m;
    r[6] = a[6] & m;
    r[7] = a[7] & m;
    r[8] = a[8] & m;
    r[9] = a[9] & m;
    r[10] = a[10] & m;
    r[11] = a[11] & m;
#endif
}

/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
SP_NOINLINE static void sp_3072_mul_24(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit* z0 = r;
    sp_digit z1[24];
    sp_digit a1[12];
    sp_digit b1[12];
    sp_digit z2[24];
    sp_digit u, ca, cb;

    ca = sp_3072_add_12(a1, a, &a[12]);
    cb = sp_3072_add_12(b1, b, &b[12]);
    u  = ca & cb;
    sp_3072_mul_12(z1, a1, b1);
    sp_3072_mul_12(z2, &a[12], &b[12]);
    sp_3072_mul_12(z0, a, b);
    sp_3072_mask_12(r + 24, a1, 0 - cb);
    sp_3072_mask_12(b1, b1, 0 - ca);
    u += sp_3072_add_12(r + 24, r + 24, b1);
    u += sp_3072_sub_in_place_24(z1, z2);
    u += sp_3072_sub_in_place_24(z1, z0);
    u += sp_3072_add_24(r + 12, r + 12, z1);
    r[36] = u;
    XMEMSET(r + 36 + 1, 0, sizeof(sp_digit) * (12 - 1));
    (void)sp_3072_add_24(r + 24, r + 24, z2);
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
SP_NOINLINE static void sp_3072_sqr_24(sp_digit* r, const sp_digit* a)
{
    sp_digit* z0 = r;
    sp_digit z2[24];
    sp_digit z1[24];
    sp_digit a1[12];
    sp_digit u;

    u = sp_3072_add_12(a1, a, &a[12]);
    sp_3072_sqr_12(z1, a1);
    sp_3072_sqr_12(z2, &a[12]);
    sp_3072_sqr_12(z0, a);
    sp_3072_mask_12(r + 24, a1, 0 - u);
    u += sp_3072_add_12(r + 24, r + 24, r + 24);
    u += sp_3072_sub_in_place_24(z1, z2);
    u += sp_3072_sub_in_place_24(z1, z0);
    u += sp_3072_add_24(r + 12, r + 12, z1);
    r[36] = u;
    XMEMSET(r + 36 + 1, 0, sizeof(sp_digit) * (12 - 1));
    (void)sp_3072_add_24(r + 24, r + 24, z2);
}

/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer and result.
 * b  A single precision integer.
 */
static sp_digit sp_3072_sub_in_place_48(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "ldp	x2, x3, [%[a], 0]\n\t"
        "ldp	x4, x5, [%[a], 16]\n\t"
        "ldp	x6, x7, [%[b], 0]\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "subs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 0]\n\t"
        "stp	x4, x5, [%[a], 16]\n\t"
        "ldp	x2, x3, [%[a], 32]\n\t"
        "ldp	x4, x5, [%[a], 48]\n\t"
        "ldp	x6, x7, [%[b], 32]\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 32]\n\t"
        "stp	x4, x5, [%[a], 48]\n\t"
        "ldp	x2, x3, [%[a], 64]\n\t"
        "ldp	x4, x5, [%[a], 80]\n\t"
        "ldp	x6, x7, [%[b], 64]\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 64]\n\t"
        "stp	x4, x5, [%[a], 80]\n\t"
        "ldp	x2, x3, [%[a], 96]\n\t"
        "ldp	x4, x5, [%[a], 112]\n\t"
        "ldp	x6, x7, [%[b], 96]\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 96]\n\t"
        "stp	x4, x5, [%[a], 112]\n\t"
        "ldp	x2, x3, [%[a], 128]\n\t"
        "ldp	x4, x5, [%[a], 144]\n\t"
        "ldp	x6, x7, [%[b], 128]\n\t"
        "ldp	x8, x9, [%[b], 144]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 128]\n\t"
        "stp	x4, x5, [%[a], 144]\n\t"
        "ldp	x2, x3, [%[a], 160]\n\t"
        "ldp	x4, x5, [%[a], 176]\n\t"
        "ldp	x6, x7, [%[b], 160]\n\t"
        "ldp	x8, x9, [%[b], 176]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 160]\n\t"
        "stp	x4, x5, [%[a], 176]\n\t"
        "ldp	x2, x3, [%[a], 192]\n\t"
        "ldp	x4, x5, [%[a], 208]\n\t"
        "ldp	x6, x7, [%[b], 192]\n\t"
        "ldp	x8, x9, [%[b], 208]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 192]\n\t"
        "stp	x4, x5, [%[a], 208]\n\t"
        "ldp	x2, x3, [%[a], 224]\n\t"
        "ldp	x4, x5, [%[a], 240]\n\t"
        "ldp	x6, x7, [%[b], 224]\n\t"
        "ldp	x8, x9, [%[b], 240]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 224]\n\t"
        "stp	x4, x5, [%[a], 240]\n\t"
        "ldp	x2, x3, [%[a], 256]\n\t"
        "ldp	x4, x5, [%[a], 272]\n\t"
        "ldp	x6, x7, [%[b], 256]\n\t"
        "ldp	x8, x9, [%[b], 272]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 256]\n\t"
        "stp	x4, x5, [%[a], 272]\n\t"
        "ldp	x2, x3, [%[a], 288]\n\t"
        "ldp	x4, x5, [%[a], 304]\n\t"
        "ldp	x6, x7, [%[b], 288]\n\t"
        "ldp	x8, x9, [%[b], 304]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 288]\n\t"
        "stp	x4, x5, [%[a], 304]\n\t"
        "ldp	x2, x3, [%[a], 320]\n\t"
        "ldp	x4, x5, [%[a], 336]\n\t"
        "ldp	x6, x7, [%[b], 320]\n\t"
        "ldp	x8, x9, [%[b], 336]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 320]\n\t"
        "stp	x4, x5, [%[a], 336]\n\t"
        "ldp	x2, x3, [%[a], 352]\n\t"
        "ldp	x4, x5, [%[a], 368]\n\t"
        "ldp	x6, x7, [%[b], 352]\n\t"
        "ldp	x8, x9, [%[b], 368]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 352]\n\t"
        "stp	x4, x5, [%[a], 368]\n\t"
        "csetm	%[c], cc\n\t"
        : [c] "+r" (c)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    return c;
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_add_48(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adds	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "ldp	x7, x8, [%[b], 128]\n\t"
        "ldp	x9, x10, [%[b], 144]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "ldp	x7, x8, [%[b], 160]\n\t"
        "ldp	x9, x10, [%[b], 176]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        "ldp	x3, x4, [%[a], 192]\n\t"
        "ldp	x5, x6, [%[a], 208]\n\t"
        "ldp	x7, x8, [%[b], 192]\n\t"
        "ldp	x9, x10, [%[b], 208]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 192]\n\t"
        "stp	x5, x6, [%[r], 208]\n\t"
        "ldp	x3, x4, [%[a], 224]\n\t"
        "ldp	x5, x6, [%[a], 240]\n\t"
        "ldp	x7, x8, [%[b], 224]\n\t"
        "ldp	x9, x10, [%[b], 240]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 224]\n\t"
        "stp	x5, x6, [%[r], 240]\n\t"
        "ldp	x3, x4, [%[a], 256]\n\t"
        "ldp	x5, x6, [%[a], 272]\n\t"
        "ldp	x7, x8, [%[b], 256]\n\t"
        "ldp	x9, x10, [%[b], 272]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 256]\n\t"
        "stp	x5, x6, [%[r], 272]\n\t"
        "ldp	x3, x4, [%[a], 288]\n\t"
        "ldp	x5, x6, [%[a], 304]\n\t"
        "ldp	x7, x8, [%[b], 288]\n\t"
        "ldp	x9, x10, [%[b], 304]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 288]\n\t"
        "stp	x5, x6, [%[r], 304]\n\t"
        "ldp	x3, x4, [%[a], 320]\n\t"
        "ldp	x5, x6, [%[a], 336]\n\t"
        "ldp	x7, x8, [%[b], 320]\n\t"
        "ldp	x9, x10, [%[b], 336]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 320]\n\t"
        "stp	x5, x6, [%[r], 336]\n\t"
        "ldp	x3, x4, [%[a], 352]\n\t"
        "ldp	x5, x6, [%[a], 368]\n\t"
        "ldp	x7, x8, [%[b], 352]\n\t"
        "ldp	x9, x10, [%[b], 368]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 352]\n\t"
        "stp	x5, x6, [%[r], 368]\n\t"
        "cset	%[c], cs\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_3072_mask_24(sp_digit* r, const sp_digit* a, sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    int i;

    for (i=0; i<24; i++) {
        r[i] = a[i] & m;
    }
#else
    int i;

    for (i = 0; i < 24; i += 8) {
        r[i+0] = a[i+0] & m;
        r[i+1] = a[i+1] & m;
        r[i+2] = a[i+2] & m;
        r[i+3] = a[i+3] & m;
        r[i+4] = a[i+4] & m;
        r[i+5] = a[i+5] & m;
        r[i+6] = a[i+6] & m;
        r[i+7] = a[i+7] & m;
    }
#endif
}

/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
SP_NOINLINE static void sp_3072_mul_48(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit* z0 = r;
    sp_digit z1[48];
    sp_digit a1[24];
    sp_digit b1[24];
    sp_digit z2[48];
    sp_digit u, ca, cb;

    ca = sp_3072_add_24(a1, a, &a[24]);
    cb = sp_3072_add_24(b1, b, &b[24]);
    u  = ca & cb;
    sp_3072_mul_24(z1, a1, b1);
    sp_3072_mul_24(z2, &a[24], &b[24]);
    sp_3072_mul_24(z0, a, b);
    sp_3072_mask_24(r + 48, a1, 0 - cb);
    sp_3072_mask_24(b1, b1, 0 - ca);
    u += sp_3072_add_24(r + 48, r + 48, b1);
    u += sp_3072_sub_in_place_48(z1, z2);
    u += sp_3072_sub_in_place_48(z1, z0);
    u += sp_3072_add_48(r + 24, r + 24, z1);
    r[72] = u;
    XMEMSET(r + 72 + 1, 0, sizeof(sp_digit) * (24 - 1));
    (void)sp_3072_add_48(r + 48, r + 48, z2);
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
SP_NOINLINE static void sp_3072_sqr_48(sp_digit* r, const sp_digit* a)
{
    sp_digit* z0 = r;
    sp_digit z2[48];
    sp_digit z1[48];
    sp_digit a1[24];
    sp_digit u;

    u = sp_3072_add_24(a1, a, &a[24]);
    sp_3072_sqr_24(z1, a1);
    sp_3072_sqr_24(z2, &a[24]);
    sp_3072_sqr_24(z0, a);
    sp_3072_mask_24(r + 48, a1, 0 - u);
    u += sp_3072_add_24(r + 48, r + 48, r + 48);
    u += sp_3072_sub_in_place_48(z1, z2);
    u += sp_3072_sub_in_place_48(z1, z0);
    u += sp_3072_add_48(r + 24, r + 24, z1);
    r[72] = u;
    XMEMSET(r + 72 + 1, 0, sizeof(sp_digit) * (24 - 1));
    (void)sp_3072_add_48(r + 48, r + 48, z2);
}

#endif /* !WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_add_48(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x11, %[a], 384\n\t"
        "\n1:\n\t"
        "adds	%[c], %[c], #-1\n\t"
        "ldp	x3, x4, [%[a]], #16\n\t"
        "ldp	x5, x6, [%[a]], #16\n\t"
        "ldp	x7, x8, [%[b]], #16\n\t"
        "ldp	x9, x10, [%[b]], #16\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r]], #16\n\t"
        "stp	x5, x6, [%[r]], #16\n\t"
        "cset	%[c], cs\n\t"
        "cmp	%[a], x11\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [r] "+r" (r), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_sub_in_place_48(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x10, %[a], 384\n\t"
        "\n1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldp	x2, x3, [%[a]]\n\t"
        "ldp	x4, x5, [%[a], #16]\n\t"
        "ldp	x6, x7, [%[b]], #16\n\t"
        "ldp	x8, x9, [%[b]], #16\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a]], #16\n\t"
        "stp	x4, x5, [%[a]], #16\n\t"
        "csetm	%[c], cc\n\t"
        "cmp	%[a], x10\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_3072_mul_48(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[96];

    __asm__ __volatile__ (
        "mov	x5, 0\n\t"
        "mov	x6, 0\n\t"
        "mov	x7, 0\n\t"
        "mov	x8, 0\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 376\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[b], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 384\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 752\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_3072_sqr_48(sp_digit* r, const sp_digit* a)
{
    sp_digit tmp[96];

    __asm__ __volatile__ (
        "mov	x6, 0\n\t"
        "mov	x7, 0\n\t"
        "mov	x8, 0\n\t"
        "mov	x5, 0\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 376\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "cmp	x4, x3\n\t"
        "b.eq	4f\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[a], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "b.al	5f\n\t"
        "\n4:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "mul	x9, x10, x10\n\t"
        "umulh	x10, x10, x10\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "\n5:\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 384\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x4\n\t"
        "b.gt	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 752\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

#endif /* WOLFSSL_SP_SMALL */
#if (defined(WOLFSSL_HAVE_SP_RSA) || defined(WOLFSSL_HAVE_SP_DH)) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)
#ifdef WOLFSSL_SP_SMALL
/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_3072_mask_24(sp_digit* r, const sp_digit* a, sp_digit m)
{
    int i;

    for (i=0; i<24; i++) {
        r[i] = a[i] & m;
    }
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_add_24(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x11, %[a], 192\n\t"
        "\n1:\n\t"
        "adds	%[c], %[c], #-1\n\t"
        "ldp	x3, x4, [%[a]], #16\n\t"
        "ldp	x5, x6, [%[a]], #16\n\t"
        "ldp	x7, x8, [%[b]], #16\n\t"
        "ldp	x9, x10, [%[b]], #16\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r]], #16\n\t"
        "stp	x5, x6, [%[r]], #16\n\t"
        "cset	%[c], cs\n\t"
        "cmp	%[a], x11\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [r] "+r" (r), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_sub_in_place_24(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x10, %[a], 192\n\t"
        "\n1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldp	x2, x3, [%[a]]\n\t"
        "ldp	x4, x5, [%[a], #16]\n\t"
        "ldp	x6, x7, [%[b]], #16\n\t"
        "ldp	x8, x9, [%[b]], #16\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a]], #16\n\t"
        "stp	x4, x5, [%[a]], #16\n\t"
        "csetm	%[c], cc\n\t"
        "cmp	%[a], x10\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_3072_mul_24(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[48];

    __asm__ __volatile__ (
        "mov	x5, 0\n\t"
        "mov	x6, 0\n\t"
        "mov	x7, 0\n\t"
        "mov	x8, 0\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 184\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[b], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 192\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 368\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_3072_sqr_24(sp_digit* r, const sp_digit* a)
{
    sp_digit tmp[48];

    __asm__ __volatile__ (
        "mov	x6, 0\n\t"
        "mov	x7, 0\n\t"
        "mov	x8, 0\n\t"
        "mov	x5, 0\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 184\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "cmp	x4, x3\n\t"
        "b.eq	4f\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[a], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "b.al	5f\n\t"
        "\n4:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "mul	x9, x10, x10\n\t"
        "umulh	x10, x10, x10\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "\n5:\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 192\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x4\n\t"
        "b.gt	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 368\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

#endif /* WOLFSSL_SP_SMALL */
#endif /* (WOLFSSL_HAVE_SP_RSA || WOLFSSL_HAVE_SP_DH) && !WOLFSSL_RSA_PUBLIC_ONLY */

/* Caclulate the bottom digit of -1/a mod 2^n.
 *
 * a    A single precision number.
 * rho  Bottom word of inverse.
 */
static void sp_3072_mont_setup(const sp_digit* a, sp_digit* rho)
{
    sp_digit x, b;

    b = a[0];
    x = (((b + 2) & 4) << 1) + b; /* here x*a==1 mod 2**4 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**8 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**16 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**32 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**64 */

    /* rho = -1/m mod b */
    *rho = -x;
}

/* Mul a by digit b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision digit.
 */
static void sp_3072_mul_d_48(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x5, %[b], x8\n\t"
        "umulh	x3, %[b], x8\n\t"
        "mov	x4, 0\n\t"
        "str	x5, [%[r]]\n\t"
        "mov	x5, 0\n\t"
        "mov	x9, #8\n\t"
        "1:\n\t"
        "ldr	x8, [%[a], x9]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "str	x3, [%[r], x9]\n\t"
        "mov	x3, x4\n\t"
        "mov	x4, x5\n\t"
        "mov	x5, #0\n\t"
        "add	x9, x9, #8\n\t"
        "cmp	x9, 384\n\t"
        "b.lt	1b\n\t"
        "str	x3, [%[r], 384]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8"
    );
#else
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x3, %[b], x8\n\t"
        "umulh	x4, %[b], x8\n\t"
        "mov	x5, 0\n\t"
        "str	x3, [%[r]]\n\t"
        "# A[1] * B\n\t"
        "ldr		x8, [%[a], 8]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 8]\n\t"
        "# A[2] * B\n\t"
        "ldr		x8, [%[a], 16]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 16]\n\t"
        "# A[3] * B\n\t"
        "ldr		x8, [%[a], 24]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 24]\n\t"
        "# A[4] * B\n\t"
        "ldr		x8, [%[a], 32]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 32]\n\t"
        "# A[5] * B\n\t"
        "ldr		x8, [%[a], 40]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 40]\n\t"
        "# A[6] * B\n\t"
        "ldr		x8, [%[a], 48]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 48]\n\t"
        "# A[7] * B\n\t"
        "ldr		x8, [%[a], 56]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 56]\n\t"
        "# A[8] * B\n\t"
        "ldr		x8, [%[a], 64]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 64]\n\t"
        "# A[9] * B\n\t"
        "ldr		x8, [%[a], 72]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 72]\n\t"
        "# A[10] * B\n\t"
        "ldr		x8, [%[a], 80]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 80]\n\t"
        "# A[11] * B\n\t"
        "ldr		x8, [%[a], 88]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 88]\n\t"
        "# A[12] * B\n\t"
        "ldr		x8, [%[a], 96]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 96]\n\t"
        "# A[13] * B\n\t"
        "ldr		x8, [%[a], 104]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 104]\n\t"
        "# A[14] * B\n\t"
        "ldr		x8, [%[a], 112]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 112]\n\t"
        "# A[15] * B\n\t"
        "ldr		x8, [%[a], 120]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 120]\n\t"
        "# A[16] * B\n\t"
        "ldr		x8, [%[a], 128]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 128]\n\t"
        "# A[17] * B\n\t"
        "ldr		x8, [%[a], 136]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 136]\n\t"
        "# A[18] * B\n\t"
        "ldr		x8, [%[a], 144]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 144]\n\t"
        "# A[19] * B\n\t"
        "ldr		x8, [%[a], 152]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 152]\n\t"
        "# A[20] * B\n\t"
        "ldr		x8, [%[a], 160]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 160]\n\t"
        "# A[21] * B\n\t"
        "ldr		x8, [%[a], 168]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 168]\n\t"
        "# A[22] * B\n\t"
        "ldr		x8, [%[a], 176]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 176]\n\t"
        "# A[23] * B\n\t"
        "ldr		x8, [%[a], 184]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 184]\n\t"
        "# A[24] * B\n\t"
        "ldr		x8, [%[a], 192]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 192]\n\t"
        "# A[25] * B\n\t"
        "ldr		x8, [%[a], 200]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 200]\n\t"
        "# A[26] * B\n\t"
        "ldr		x8, [%[a], 208]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 208]\n\t"
        "# A[27] * B\n\t"
        "ldr		x8, [%[a], 216]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 216]\n\t"
        "# A[28] * B\n\t"
        "ldr		x8, [%[a], 224]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 224]\n\t"
        "# A[29] * B\n\t"
        "ldr		x8, [%[a], 232]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 232]\n\t"
        "# A[30] * B\n\t"
        "ldr		x8, [%[a], 240]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 240]\n\t"
        "# A[31] * B\n\t"
        "ldr		x8, [%[a], 248]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 248]\n\t"
        "# A[32] * B\n\t"
        "ldr		x8, [%[a], 256]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 256]\n\t"
        "# A[33] * B\n\t"
        "ldr		x8, [%[a], 264]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 264]\n\t"
        "# A[34] * B\n\t"
        "ldr		x8, [%[a], 272]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 272]\n\t"
        "# A[35] * B\n\t"
        "ldr		x8, [%[a], 280]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 280]\n\t"
        "# A[36] * B\n\t"
        "ldr		x8, [%[a], 288]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 288]\n\t"
        "# A[37] * B\n\t"
        "ldr		x8, [%[a], 296]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 296]\n\t"
        "# A[38] * B\n\t"
        "ldr		x8, [%[a], 304]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 304]\n\t"
        "# A[39] * B\n\t"
        "ldr		x8, [%[a], 312]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 312]\n\t"
        "# A[40] * B\n\t"
        "ldr		x8, [%[a], 320]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 320]\n\t"
        "# A[41] * B\n\t"
        "ldr		x8, [%[a], 328]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 328]\n\t"
        "# A[42] * B\n\t"
        "ldr		x8, [%[a], 336]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 336]\n\t"
        "# A[43] * B\n\t"
        "ldr		x8, [%[a], 344]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 344]\n\t"
        "# A[44] * B\n\t"
        "ldr		x8, [%[a], 352]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 352]\n\t"
        "# A[45] * B\n\t"
        "ldr		x8, [%[a], 360]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 360]\n\t"
        "# A[46] * B\n\t"
        "ldr		x8, [%[a], 368]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 368]\n\t"
        "# A[47] * B\n\t"
        "ldr	x8, [%[a], 376]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x5, [%[r], 376]\n\t"
        "str	x3, [%[r], 384]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8"
    );
#endif
}

#if (defined(WOLFSSL_HAVE_SP_RSA) || defined(WOLFSSL_HAVE_SP_DH)) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)
/* r = 2^n mod m where n is the number of bits to reduce by.
 * Given m must be 3072 bits, just need to subtract.
 *
 * r  A single precision number.
 * m  A signle precision number.
 */
static void sp_3072_mont_norm_24(sp_digit* r, const sp_digit* m)
{
    XMEMSET(r, 0, sizeof(sp_digit) * 24);

    /* r = 2^n mod m */
    sp_3072_sub_in_place_24(r, m);
}

/* Conditionally subtract b from a using the mask m.
 * m is -1 to subtract and 0 when not copying.
 *
 * r  A single precision number representing condition subtract result.
 * a  A single precision number to subtract from.
 * b  A single precision number to subtract.
 * m  Mask value to apply.
 */
static sp_digit sp_3072_cond_sub_24(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    sp_digit c = 0;

#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x8, #0\n\t"
        "1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldr	x4, [%[a], x8]\n\t"
        "ldr	x5, [%[b], x8]\n\t"
        "and	x5, x5, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "csetm	%[c], cc\n\t"
        "str	x4, [%[r], x8]\n\t"
        "add	x8, x8, #8\n\t"
        "cmp	x8, 192\n\t"
        "b.lt	1b\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8"
    );
#else
    __asm__ __volatile__ (

        "ldr		x4, [%[a], 0]\n\t"
        "ldr		x6, [%[a], 8]\n\t"
        "ldr		x5, [%[b], 0]\n\t"
        "ldr		x7, [%[b], 8]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "subs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 0]\n\t"
        "str		x6, [%[r], 8]\n\t"
        "ldr		x4, [%[a], 16]\n\t"
        "ldr		x6, [%[a], 24]\n\t"
        "ldr		x5, [%[b], 16]\n\t"
        "ldr		x7, [%[b], 24]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 16]\n\t"
        "str		x6, [%[r], 24]\n\t"
        "ldr		x4, [%[a], 32]\n\t"
        "ldr		x6, [%[a], 40]\n\t"
        "ldr		x5, [%[b], 32]\n\t"
        "ldr		x7, [%[b], 40]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 32]\n\t"
        "str		x6, [%[r], 40]\n\t"
        "ldr		x4, [%[a], 48]\n\t"
        "ldr		x6, [%[a], 56]\n\t"
        "ldr		x5, [%[b], 48]\n\t"
        "ldr		x7, [%[b], 56]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 48]\n\t"
        "str		x6, [%[r], 56]\n\t"
        "ldr		x4, [%[a], 64]\n\t"
        "ldr		x6, [%[a], 72]\n\t"
        "ldr		x5, [%[b], 64]\n\t"
        "ldr		x7, [%[b], 72]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 64]\n\t"
        "str		x6, [%[r], 72]\n\t"
        "ldr		x4, [%[a], 80]\n\t"
        "ldr		x6, [%[a], 88]\n\t"
        "ldr		x5, [%[b], 80]\n\t"
        "ldr		x7, [%[b], 88]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 80]\n\t"
        "str		x6, [%[r], 88]\n\t"
        "ldr		x4, [%[a], 96]\n\t"
        "ldr		x6, [%[a], 104]\n\t"
        "ldr		x5, [%[b], 96]\n\t"
        "ldr		x7, [%[b], 104]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 96]\n\t"
        "str		x6, [%[r], 104]\n\t"
        "ldr		x4, [%[a], 112]\n\t"
        "ldr		x6, [%[a], 120]\n\t"
        "ldr		x5, [%[b], 112]\n\t"
        "ldr		x7, [%[b], 120]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 112]\n\t"
        "str		x6, [%[r], 120]\n\t"
        "ldr		x4, [%[a], 128]\n\t"
        "ldr		x6, [%[a], 136]\n\t"
        "ldr		x5, [%[b], 128]\n\t"
        "ldr		x7, [%[b], 136]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 128]\n\t"
        "str		x6, [%[r], 136]\n\t"
        "ldr		x4, [%[a], 144]\n\t"
        "ldr		x6, [%[a], 152]\n\t"
        "ldr		x5, [%[b], 144]\n\t"
        "ldr		x7, [%[b], 152]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 144]\n\t"
        "str		x6, [%[r], 152]\n\t"
        "ldr		x4, [%[a], 160]\n\t"
        "ldr		x6, [%[a], 168]\n\t"
        "ldr		x5, [%[b], 160]\n\t"
        "ldr		x7, [%[b], 168]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 160]\n\t"
        "str		x6, [%[r], 168]\n\t"
        "ldr		x4, [%[a], 176]\n\t"
        "ldr		x6, [%[a], 184]\n\t"
        "ldr		x5, [%[b], 176]\n\t"
        "ldr		x7, [%[b], 184]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 176]\n\t"
        "str		x6, [%[r], 184]\n\t"
        "csetm	%[c], cc\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8"
    );
#endif /* WOLFSSL_SP_SMALL */

    return c;
}

/* Reduce the number back to 3072 bits using Montgomery reduction.
 *
 * a   A single precision number to reduce in place.
 * m   The single precision number representing the modulus.
 * mp  The digit representing the negative inverse of m mod 2^n.
 */
SP_NOINLINE static void sp_3072_mont_reduce_24(sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    sp_digit ca = 0;

    __asm__ __volatile__ (
        "ldp       x12, x13, [%[m], 0]\n\t"
        "ldp       x14, x15, [%[m], 16]\n\t"
        "ldp       x16, x17, [%[m], 32]\n\t"
        "ldp       x19, x20, [%[m], 48]\n\t"
        "ldp       x21, x22, [%[m], 64]\n\t"
        "ldp       x23, x24, [%[m], 80]\n\t"
        "ldp       x25, x26, [%[m], 96]\n\t"
        "ldp       x27, x28, [%[m], 112]\n\t"
        "# i = 0\n\t"
        "mov	x3, 0\n\t"
        "ldp	x10, x11, [%[a], 0]\n\t"
        "\n1:\n\t"
        "# mu = a[i] * mp\n\t"
        "mul	x8, %[mp], x10\n\t"
        "# a[i+0] += m[0] * mu\n\t"
        "ldr	x9, [%[a], 0]\n\t"
        "mul		x6, x12, x8\n\t"
        "umulh	x7, x12, x8\n\t"
        "adds	x10, x10, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "# a[i+1] += m[1] * mu\n\t"
        "ldr	x9, [%[a], 8]\n\t"
        "mul		x6, x13, x8\n\t"
        "umulh	x7, x13, x8\n\t"
        "adds	x10, x11, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x10, x10, x5\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+2] += m[2] * mu\n\t"
        "ldr	x11, [%[a], 16]\n\t"
        "mul		x6, x14, x8\n\t"
        "umulh	x7, x14, x8\n\t"
        "adds	x11, x11, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x11, x11, x4\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+3] += m[3] * mu\n\t"
        "ldr	x9, [%[a], 24]\n\t"
        "mul		x6, x15, x8\n\t"
        "umulh	x7, x15, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 24]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+4] += m[4] * mu\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "mul		x6, x16, x8\n\t"
        "umulh	x7, x16, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 32]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+5] += m[5] * mu\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "mul		x6, x17, x8\n\t"
        "umulh	x7, x17, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 40]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+6] += m[6] * mu\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "mul		x6, x19, x8\n\t"
        "umulh	x7, x19, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 48]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+7] += m[7] * mu\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "mul		x6, x20, x8\n\t"
        "umulh	x7, x20, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 56]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+8] += m[8] * mu\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "mul		x6, x21, x8\n\t"
        "umulh	x7, x21, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 64]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+9] += m[9] * mu\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "mul		x6, x22, x8\n\t"
        "umulh	x7, x22, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 72]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+10] += m[10] * mu\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "mul		x6, x23, x8\n\t"
        "umulh	x7, x23, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 80]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+11] += m[11] * mu\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "mul		x6, x24, x8\n\t"
        "umulh	x7, x24, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 88]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+12] += m[12] * mu\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "mul		x6, x25, x8\n\t"
        "umulh	x7, x25, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 96]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+13] += m[13] * mu\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "mul		x6, x26, x8\n\t"
        "umulh	x7, x26, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 104]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+14] += m[14] * mu\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "mul		x6, x27, x8\n\t"
        "umulh	x7, x27, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 112]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+15] += m[15] * mu\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "mul		x6, x28, x8\n\t"
        "umulh	x7, x28, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 120]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+16] += m[16] * mu\n\t"
        "ldr		x7, [%[m], 128]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 128]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+17] += m[17] * mu\n\t"
        "ldr		x7, [%[m], 136]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 136]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+18] += m[18] * mu\n\t"
        "ldr		x7, [%[m], 144]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 144]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+19] += m[19] * mu\n\t"
        "ldr		x7, [%[m], 152]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 152]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+20] += m[20] * mu\n\t"
        "ldr		x7, [%[m], 160]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 160]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+21] += m[21] * mu\n\t"
        "ldr		x7, [%[m], 168]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 168]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+22] += m[22] * mu\n\t"
        "ldr		x7, [%[m], 176]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 176]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+23] += m[23] * mu\n\t"
        "ldr	x7, [%[m], 184]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x7, x7, %[ca]\n\t"
        "cset  %[ca], cs\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 184]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "adcs	x9, x9, x7\n\t"
        "str	x9, [%[a], 192]\n\t"
        "adc	%[ca], %[ca], xzr\n\t"
        "# i += 1\n\t"
        "add	%[a], %[a], 8\n\t"
        "add	x3, x3, 8\n\t"
        "cmp	x3, 192\n\t"
        "blt	1b\n\t"
        "str	x10, [%[a], 0]\n\t"
        "str	x11, [%[a], 8]\n\t"
        : [ca] "+r" (ca), [a] "+r" (a)
        : [m] "r" (m), [mp] "r" (mp)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );

    sp_3072_cond_sub_24(a - 24, a, m, (sp_digit)0 - ca);
}

/* Multiply two Montogmery form numbers mod the modulus (prime).
 * (r = a * b mod m)
 *
 * r   Result of multiplication.
 * a   First number to multiply in Montogmery form.
 * b   Second number to multiply in Montogmery form.
 * m   Modulus (prime).
 * mp  Montogmery mulitplier.
 */
static void sp_3072_mont_mul_24(sp_digit* r, const sp_digit* a, const sp_digit* b,
        const sp_digit* m, sp_digit mp)
{
    sp_3072_mul_24(r, a, b);
    sp_3072_mont_reduce_24(r, m, mp);
}

/* Square the Montgomery form number. (r = a * a mod m)
 *
 * r   Result of squaring.
 * a   Number to square in Montogmery form.
 * m   Modulus (prime).
 * mp  Montogmery mulitplier.
 */
static void sp_3072_mont_sqr_24(sp_digit* r, const sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    sp_3072_sqr_24(r, a);
    sp_3072_mont_reduce_24(r, m, mp);
}

/* Mul a by digit b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision digit.
 */
static void sp_3072_mul_d_24(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x5, %[b], x8\n\t"
        "umulh	x3, %[b], x8\n\t"
        "mov	x4, 0\n\t"
        "str	x5, [%[r]]\n\t"
        "mov	x5, 0\n\t"
        "mov	x9, #8\n\t"
        "1:\n\t"
        "ldr	x8, [%[a], x9]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "str	x3, [%[r], x9]\n\t"
        "mov	x3, x4\n\t"
        "mov	x4, x5\n\t"
        "mov	x5, #0\n\t"
        "add	x9, x9, #8\n\t"
        "cmp	x9, 192\n\t"
        "b.lt	1b\n\t"
        "str	x3, [%[r], 192]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8"
    );
#else
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x3, %[b], x8\n\t"
        "umulh	x4, %[b], x8\n\t"
        "mov	x5, 0\n\t"
        "str	x3, [%[r]]\n\t"
        "# A[1] * B\n\t"
        "ldr		x8, [%[a], 8]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 8]\n\t"
        "# A[2] * B\n\t"
        "ldr		x8, [%[a], 16]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 16]\n\t"
        "# A[3] * B\n\t"
        "ldr		x8, [%[a], 24]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 24]\n\t"
        "# A[4] * B\n\t"
        "ldr		x8, [%[a], 32]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 32]\n\t"
        "# A[5] * B\n\t"
        "ldr		x8, [%[a], 40]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 40]\n\t"
        "# A[6] * B\n\t"
        "ldr		x8, [%[a], 48]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 48]\n\t"
        "# A[7] * B\n\t"
        "ldr		x8, [%[a], 56]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 56]\n\t"
        "# A[8] * B\n\t"
        "ldr		x8, [%[a], 64]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 64]\n\t"
        "# A[9] * B\n\t"
        "ldr		x8, [%[a], 72]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 72]\n\t"
        "# A[10] * B\n\t"
        "ldr		x8, [%[a], 80]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 80]\n\t"
        "# A[11] * B\n\t"
        "ldr		x8, [%[a], 88]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 88]\n\t"
        "# A[12] * B\n\t"
        "ldr		x8, [%[a], 96]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 96]\n\t"
        "# A[13] * B\n\t"
        "ldr		x8, [%[a], 104]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 104]\n\t"
        "# A[14] * B\n\t"
        "ldr		x8, [%[a], 112]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 112]\n\t"
        "# A[15] * B\n\t"
        "ldr		x8, [%[a], 120]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 120]\n\t"
        "# A[16] * B\n\t"
        "ldr		x8, [%[a], 128]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 128]\n\t"
        "# A[17] * B\n\t"
        "ldr		x8, [%[a], 136]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 136]\n\t"
        "# A[18] * B\n\t"
        "ldr		x8, [%[a], 144]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 144]\n\t"
        "# A[19] * B\n\t"
        "ldr		x8, [%[a], 152]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 152]\n\t"
        "# A[20] * B\n\t"
        "ldr		x8, [%[a], 160]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 160]\n\t"
        "# A[21] * B\n\t"
        "ldr		x8, [%[a], 168]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 168]\n\t"
        "# A[22] * B\n\t"
        "ldr		x8, [%[a], 176]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 176]\n\t"
        "# A[23] * B\n\t"
        "ldr	x8, [%[a], 184]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x5, [%[r], 184]\n\t"
        "str	x3, [%[r], 192]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8"
    );
#endif
}

/* Divide the double width number (d1|d0) by the dividend. (d1|d0 / div)
 *
 * d1   The high order half of the number to divide.
 * d0   The low order half of the number to divide.
 * div  The dividend.
 * returns the result of the division.
 */
static sp_digit div_3072_word_24(sp_digit d1, sp_digit d0, sp_digit div)
{
    sp_digit r;

    __asm__ __volatile__ (
        "lsr	x5, %[div], 32\n\t"
        "add	x5, x5, 1\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x6, x3, 32\n\t"
        "mul	x4, %[div], x6\n\t"
        "umulh	x3, %[div], x6\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x3, x3, 32\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "lsl	x3, %[d1], 32\n\t"
        "orr	x3, x3, %[d0], lsr 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "lsl	x3, %[d1], 32\n\t"
        "orr	x3, x3, %[d0], lsr 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "udiv	x3, %[d0], %[div]\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x3, %[div], x3\n\t"
        "sub	%[d0], %[d0], x3\n\t"
        "mov	%[r], x6\n\t"

        : [r] "=r" (r)
        : [d1] "r" (d1), [d0] "r" (d0), [div] "r" (div)
        : "x3", "x4", "x5", "x6"
    );

    return r;
}

/* Compare a with b in constant time.
 *
 * a  A single precision integer.
 * b  A single precision integer.
 * return -ve, 0 or +ve if a is less than, equal to or greater than b
 * respectively.
 */
static int64_t sp_3072_cmp_24(const sp_digit* a, const sp_digit* b)
{
    sp_digit r = -1;
    sp_digit one = 1;

#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x3, -1\n\t"
        "mov	x6, 184\n\t"
        "1:\n\t"
        "ldr	x4, [%[a], x6]\n\t"
        "ldr	x5, [%[b], x6]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "subs	x6, x6, #8\n\t"
        "b.cs	1b\n\t"
        "eor	%[r], %[r], x3\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [one] "r" (one)
        : "x2", "x3", "x4", "x5", "x6"
    );
#else
    __asm__ __volatile__ (
        "mov	x3, -1\n\t"
        "ldr		x4, [%[a], 184]\n\t"
        "ldr		x5, [%[b], 184]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 176]\n\t"
        "ldr		x5, [%[b], 176]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 168]\n\t"
        "ldr		x5, [%[b], 168]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 160]\n\t"
        "ldr		x5, [%[b], 160]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 152]\n\t"
        "ldr		x5, [%[b], 152]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 144]\n\t"
        "ldr		x5, [%[b], 144]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 136]\n\t"
        "ldr		x5, [%[b], 136]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 128]\n\t"
        "ldr		x5, [%[b], 128]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 120]\n\t"
        "ldr		x5, [%[b], 120]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 112]\n\t"
        "ldr		x5, [%[b], 112]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 104]\n\t"
        "ldr		x5, [%[b], 104]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 96]\n\t"
        "ldr		x5, [%[b], 96]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 88]\n\t"
        "ldr		x5, [%[b], 88]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 80]\n\t"
        "ldr		x5, [%[b], 80]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 72]\n\t"
        "ldr		x5, [%[b], 72]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 64]\n\t"
        "ldr		x5, [%[b], 64]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 56]\n\t"
        "ldr		x5, [%[b], 56]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 48]\n\t"
        "ldr		x5, [%[b], 48]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 40]\n\t"
        "ldr		x5, [%[b], 40]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 32]\n\t"
        "ldr		x5, [%[b], 32]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 24]\n\t"
        "ldr		x5, [%[b], 24]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 16]\n\t"
        "ldr		x5, [%[b], 16]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 8]\n\t"
        "ldr		x5, [%[b], 8]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 0]\n\t"
        "ldr		x5, [%[b], 0]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "eor	%[r], %[r], x3\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [one] "r" (one)
        : "x2", "x3", "x4", "x5", "x6"
    );
#endif

    return r;
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Nmber to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_3072_div_24(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[48], t2[25];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[23];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 24);
    for (i=23; i>=0; i--) {
        r1 = div_3072_word_24(t1[24 + i], t1[24 + i - 1], div);

        sp_3072_mul_d_24(t2, d, r1);
        t1[24 + i] += sp_3072_sub_in_place_24(&t1[i], t2);
        t1[24 + i] -= t2[24];
        sp_3072_mask_24(t2, d, t1[24 + i]);
        t1[24 + i] += sp_3072_add_24(&t1[i], &t1[i], t2);
        sp_3072_mask_24(t2, d, t1[24 + i]);
        t1[24 + i] += sp_3072_add_24(&t1[i], &t1[i], t2);
    }

    r1 = sp_3072_cmp_24(t1, d) >= 0;
    sp_3072_cond_sub_24(r, t1, d, (sp_digit)0 - r1);

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_3072_mod_24(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_3072_div_24(a, m, NULL, r);
}

#ifdef WOLFSSL_SP_SMALL
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns 0 on success and MEMORY_E on dynamic memory allocation failure.
 */
static int sp_3072_mod_exp_24(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#ifndef WOLFSSL_SMALL_STACK
    sp_digit t[16][48];
#else
    sp_digit* t[16];
    sp_digit* td;
#endif
    sp_digit* norm;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c, y;
    int err = MP_OKAY;

#ifdef WOLFSSL_SMALL_STACK
    td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 16 * 48, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER);
    if (td == NULL) {
        err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
#ifdef WOLFSSL_SMALL_STACK
        for (i=0; i<16; i++) {
            t[i] = td + i * 48;
        }
#endif
        norm = t[0];

        sp_3072_mont_setup(m, &mp);
        sp_3072_mont_norm_24(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 24U);
        if (reduceA != 0) {
            err = sp_3072_mod_24(t[1] + 24, a, m);
            if (err == MP_OKAY) {
                err = sp_3072_mod_24(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 24, a, sizeof(sp_digit) * 24);
            err = sp_3072_mod_24(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_3072_mont_sqr_24(t[ 2], t[ 1], m, mp);
        sp_3072_mont_mul_24(t[ 3], t[ 2], t[ 1], m, mp);
        sp_3072_mont_sqr_24(t[ 4], t[ 2], m, mp);
        sp_3072_mont_mul_24(t[ 5], t[ 3], t[ 2], m, mp);
        sp_3072_mont_sqr_24(t[ 6], t[ 3], m, mp);
        sp_3072_mont_mul_24(t[ 7], t[ 4], t[ 3], m, mp);
        sp_3072_mont_sqr_24(t[ 8], t[ 4], m, mp);
        sp_3072_mont_mul_24(t[ 9], t[ 5], t[ 4], m, mp);
        sp_3072_mont_sqr_24(t[10], t[ 5], m, mp);
        sp_3072_mont_mul_24(t[11], t[ 6], t[ 5], m, mp);
        sp_3072_mont_sqr_24(t[12], t[ 6], m, mp);
        sp_3072_mont_mul_24(t[13], t[ 7], t[ 6], m, mp);
        sp_3072_mont_sqr_24(t[14], t[ 7], m, mp);
        sp_3072_mont_mul_24(t[15], t[ 8], t[ 7], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 4;
        if (c == 64) {
            c = 60;
        }
        y = (int)(n >> c);
        n <<= 64 - c;
        XMEMCPY(r, t[y], sizeof(sp_digit) * 24);
        for (; i>=0 || c>=4; ) {
            if (c == 0) {
                n = e[i--];
                y = n >> 60;
                n <<= 4;
                c = 60;
            }
            else if (c < 4) {
                y = n >> 60;
                n = e[i--];
                c = 4 - c;
                y |= n >> (64 - c);
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (n >> 60) & 0xf;
                n <<= 4;
                c -= 4;
            }

            sp_3072_mont_sqr_24(r, r, m, mp);
            sp_3072_mont_sqr_24(r, r, m, mp);
            sp_3072_mont_sqr_24(r, r, m, mp);
            sp_3072_mont_sqr_24(r, r, m, mp);

            sp_3072_mont_mul_24(r, r, t[y], m, mp);
        }

        XMEMSET(&r[24], 0, sizeof(sp_digit) * 24U);
        sp_3072_mont_reduce_24(r, m, mp);

        mask = 0 - (sp_3072_cmp_24(r, m) >= 0);
        sp_3072_cond_sub_24(r, r, m, mask);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (td != NULL) {
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return err;
}
#else
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns 0 on success and MEMORY_E on dynamic memory allocation failure.
 */
static int sp_3072_mod_exp_24(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#ifndef WOLFSSL_SMALL_STACK
    sp_digit t[32][48];
#else
    sp_digit* t[32];
    sp_digit* td;
#endif
    sp_digit* norm;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c, y;
    int err = MP_OKAY;

#ifdef WOLFSSL_SMALL_STACK
    td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 32 * 48, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER);
    if (td == NULL) {
        err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
#ifdef WOLFSSL_SMALL_STACK
        for (i=0; i<32; i++) {
            t[i] = td + i * 48;
        }
#endif
        norm = t[0];

        sp_3072_mont_setup(m, &mp);
        sp_3072_mont_norm_24(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 24U);
        if (reduceA != 0) {
            err = sp_3072_mod_24(t[1] + 24, a, m);
            if (err == MP_OKAY) {
                err = sp_3072_mod_24(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 24, a, sizeof(sp_digit) * 24);
            err = sp_3072_mod_24(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_3072_mont_sqr_24(t[ 2], t[ 1], m, mp);
        sp_3072_mont_mul_24(t[ 3], t[ 2], t[ 1], m, mp);
        sp_3072_mont_sqr_24(t[ 4], t[ 2], m, mp);
        sp_3072_mont_mul_24(t[ 5], t[ 3], t[ 2], m, mp);
        sp_3072_mont_sqr_24(t[ 6], t[ 3], m, mp);
        sp_3072_mont_mul_24(t[ 7], t[ 4], t[ 3], m, mp);
        sp_3072_mont_sqr_24(t[ 8], t[ 4], m, mp);
        sp_3072_mont_mul_24(t[ 9], t[ 5], t[ 4], m, mp);
        sp_3072_mont_sqr_24(t[10], t[ 5], m, mp);
        sp_3072_mont_mul_24(t[11], t[ 6], t[ 5], m, mp);
        sp_3072_mont_sqr_24(t[12], t[ 6], m, mp);
        sp_3072_mont_mul_24(t[13], t[ 7], t[ 6], m, mp);
        sp_3072_mont_sqr_24(t[14], t[ 7], m, mp);
        sp_3072_mont_mul_24(t[15], t[ 8], t[ 7], m, mp);
        sp_3072_mont_sqr_24(t[16], t[ 8], m, mp);
        sp_3072_mont_mul_24(t[17], t[ 9], t[ 8], m, mp);
        sp_3072_mont_sqr_24(t[18], t[ 9], m, mp);
        sp_3072_mont_mul_24(t[19], t[10], t[ 9], m, mp);
        sp_3072_mont_sqr_24(t[20], t[10], m, mp);
        sp_3072_mont_mul_24(t[21], t[11], t[10], m, mp);
        sp_3072_mont_sqr_24(t[22], t[11], m, mp);
        sp_3072_mont_mul_24(t[23], t[12], t[11], m, mp);
        sp_3072_mont_sqr_24(t[24], t[12], m, mp);
        sp_3072_mont_mul_24(t[25], t[13], t[12], m, mp);
        sp_3072_mont_sqr_24(t[26], t[13], m, mp);
        sp_3072_mont_mul_24(t[27], t[14], t[13], m, mp);
        sp_3072_mont_sqr_24(t[28], t[14], m, mp);
        sp_3072_mont_mul_24(t[29], t[15], t[14], m, mp);
        sp_3072_mont_sqr_24(t[30], t[15], m, mp);
        sp_3072_mont_mul_24(t[31], t[16], t[15], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 5;
        if (c == 64) {
            c = 59;
        }
        y = (int)(n >> c);
        n <<= 64 - c;
        XMEMCPY(r, t[y], sizeof(sp_digit) * 24);
        for (; i>=0 || c>=5; ) {
            if (c == 0) {
                n = e[i--];
                y = n >> 59;
                n <<= 5;
                c = 59;
            }
            else if (c < 5) {
                y = n >> 59;
                n = e[i--];
                c = 5 - c;
                y |= n >> (64 - c);
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (n >> 59) & 0x1f;
                n <<= 5;
                c -= 5;
            }

            sp_3072_mont_sqr_24(r, r, m, mp);
            sp_3072_mont_sqr_24(r, r, m, mp);
            sp_3072_mont_sqr_24(r, r, m, mp);
            sp_3072_mont_sqr_24(r, r, m, mp);
            sp_3072_mont_sqr_24(r, r, m, mp);

            sp_3072_mont_mul_24(r, r, t[y], m, mp);
        }

        XMEMSET(&r[24], 0, sizeof(sp_digit) * 24U);
        sp_3072_mont_reduce_24(r, m, mp);

        mask = 0 - (sp_3072_cmp_24(r, m) >= 0);
        sp_3072_cond_sub_24(r, r, m, mask);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (td != NULL) {
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return err;
}
#endif /* WOLFSSL_SP_SMALL */

#endif /* (WOLFSSL_HAVE_SP_RSA || WOLFSSL_HAVE_SP_DH) && !WOLFSSL_RSA_PUBLIC_ONLY */

#if defined(WOLFSSL_HAVE_SP_RSA) || defined(WOLFSSL_HAVE_SP_DH)
/* r = 2^n mod m where n is the number of bits to reduce by.
 * Given m must be 3072 bits, just need to subtract.
 *
 * r  A single precision number.
 * m  A signle precision number.
 */
static void sp_3072_mont_norm_48(sp_digit* r, const sp_digit* m)
{
    XMEMSET(r, 0, sizeof(sp_digit) * 48);

    /* r = 2^n mod m */
    sp_3072_sub_in_place_48(r, m);
}

#endif /* WOLFSSL_HAVE_SP_RSA || WOLFSSL_HAVE_SP_DH */
/* Conditionally subtract b from a using the mask m.
 * m is -1 to subtract and 0 when not copying.
 *
 * r  A single precision number representing condition subtract result.
 * a  A single precision number to subtract from.
 * b  A single precision number to subtract.
 * m  Mask value to apply.
 */
static sp_digit sp_3072_cond_sub_48(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    sp_digit c = 0;

#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x8, #0\n\t"
        "1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldr	x4, [%[a], x8]\n\t"
        "ldr	x5, [%[b], x8]\n\t"
        "and	x5, x5, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "csetm	%[c], cc\n\t"
        "str	x4, [%[r], x8]\n\t"
        "add	x8, x8, #8\n\t"
        "cmp	x8, 384\n\t"
        "b.lt	1b\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8"
    );
#else
    __asm__ __volatile__ (

        "ldr		x4, [%[a], 0]\n\t"
        "ldr		x6, [%[a], 8]\n\t"
        "ldr		x5, [%[b], 0]\n\t"
        "ldr		x7, [%[b], 8]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "subs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 0]\n\t"
        "str		x6, [%[r], 8]\n\t"
        "ldr		x4, [%[a], 16]\n\t"
        "ldr		x6, [%[a], 24]\n\t"
        "ldr		x5, [%[b], 16]\n\t"
        "ldr		x7, [%[b], 24]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 16]\n\t"
        "str		x6, [%[r], 24]\n\t"
        "ldr		x4, [%[a], 32]\n\t"
        "ldr		x6, [%[a], 40]\n\t"
        "ldr		x5, [%[b], 32]\n\t"
        "ldr		x7, [%[b], 40]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 32]\n\t"
        "str		x6, [%[r], 40]\n\t"
        "ldr		x4, [%[a], 48]\n\t"
        "ldr		x6, [%[a], 56]\n\t"
        "ldr		x5, [%[b], 48]\n\t"
        "ldr		x7, [%[b], 56]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 48]\n\t"
        "str		x6, [%[r], 56]\n\t"
        "ldr		x4, [%[a], 64]\n\t"
        "ldr		x6, [%[a], 72]\n\t"
        "ldr		x5, [%[b], 64]\n\t"
        "ldr		x7, [%[b], 72]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 64]\n\t"
        "str		x6, [%[r], 72]\n\t"
        "ldr		x4, [%[a], 80]\n\t"
        "ldr		x6, [%[a], 88]\n\t"
        "ldr		x5, [%[b], 80]\n\t"
        "ldr		x7, [%[b], 88]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 80]\n\t"
        "str		x6, [%[r], 88]\n\t"
        "ldr		x4, [%[a], 96]\n\t"
        "ldr		x6, [%[a], 104]\n\t"
        "ldr		x5, [%[b], 96]\n\t"
        "ldr		x7, [%[b], 104]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 96]\n\t"
        "str		x6, [%[r], 104]\n\t"
        "ldr		x4, [%[a], 112]\n\t"
        "ldr		x6, [%[a], 120]\n\t"
        "ldr		x5, [%[b], 112]\n\t"
        "ldr		x7, [%[b], 120]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 112]\n\t"
        "str		x6, [%[r], 120]\n\t"
        "ldr		x4, [%[a], 128]\n\t"
        "ldr		x6, [%[a], 136]\n\t"
        "ldr		x5, [%[b], 128]\n\t"
        "ldr		x7, [%[b], 136]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 128]\n\t"
        "str		x6, [%[r], 136]\n\t"
        "ldr		x4, [%[a], 144]\n\t"
        "ldr		x6, [%[a], 152]\n\t"
        "ldr		x5, [%[b], 144]\n\t"
        "ldr		x7, [%[b], 152]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 144]\n\t"
        "str		x6, [%[r], 152]\n\t"
        "ldr		x4, [%[a], 160]\n\t"
        "ldr		x6, [%[a], 168]\n\t"
        "ldr		x5, [%[b], 160]\n\t"
        "ldr		x7, [%[b], 168]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 160]\n\t"
        "str		x6, [%[r], 168]\n\t"
        "ldr		x4, [%[a], 176]\n\t"
        "ldr		x6, [%[a], 184]\n\t"
        "ldr		x5, [%[b], 176]\n\t"
        "ldr		x7, [%[b], 184]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 176]\n\t"
        "str		x6, [%[r], 184]\n\t"
        "ldr		x4, [%[a], 192]\n\t"
        "ldr		x6, [%[a], 200]\n\t"
        "ldr		x5, [%[b], 192]\n\t"
        "ldr		x7, [%[b], 200]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 192]\n\t"
        "str		x6, [%[r], 200]\n\t"
        "ldr		x4, [%[a], 208]\n\t"
        "ldr		x6, [%[a], 216]\n\t"
        "ldr		x5, [%[b], 208]\n\t"
        "ldr		x7, [%[b], 216]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 208]\n\t"
        "str		x6, [%[r], 216]\n\t"
        "ldr		x4, [%[a], 224]\n\t"
        "ldr		x6, [%[a], 232]\n\t"
        "ldr		x5, [%[b], 224]\n\t"
        "ldr		x7, [%[b], 232]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 224]\n\t"
        "str		x6, [%[r], 232]\n\t"
        "ldr		x4, [%[a], 240]\n\t"
        "ldr		x6, [%[a], 248]\n\t"
        "ldr		x5, [%[b], 240]\n\t"
        "ldr		x7, [%[b], 248]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 240]\n\t"
        "str		x6, [%[r], 248]\n\t"
        "ldr		x4, [%[a], 256]\n\t"
        "ldr		x6, [%[a], 264]\n\t"
        "ldr		x5, [%[b], 256]\n\t"
        "ldr		x7, [%[b], 264]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 256]\n\t"
        "str		x6, [%[r], 264]\n\t"
        "ldr		x4, [%[a], 272]\n\t"
        "ldr		x6, [%[a], 280]\n\t"
        "ldr		x5, [%[b], 272]\n\t"
        "ldr		x7, [%[b], 280]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 272]\n\t"
        "str		x6, [%[r], 280]\n\t"
        "ldr		x4, [%[a], 288]\n\t"
        "ldr		x6, [%[a], 296]\n\t"
        "ldr		x5, [%[b], 288]\n\t"
        "ldr		x7, [%[b], 296]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 288]\n\t"
        "str		x6, [%[r], 296]\n\t"
        "ldr		x4, [%[a], 304]\n\t"
        "ldr		x6, [%[a], 312]\n\t"
        "ldr		x5, [%[b], 304]\n\t"
        "ldr		x7, [%[b], 312]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 304]\n\t"
        "str		x6, [%[r], 312]\n\t"
        "ldr		x4, [%[a], 320]\n\t"
        "ldr		x6, [%[a], 328]\n\t"
        "ldr		x5, [%[b], 320]\n\t"
        "ldr		x7, [%[b], 328]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 320]\n\t"
        "str		x6, [%[r], 328]\n\t"
        "ldr		x4, [%[a], 336]\n\t"
        "ldr		x6, [%[a], 344]\n\t"
        "ldr		x5, [%[b], 336]\n\t"
        "ldr		x7, [%[b], 344]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 336]\n\t"
        "str		x6, [%[r], 344]\n\t"
        "ldr		x4, [%[a], 352]\n\t"
        "ldr		x6, [%[a], 360]\n\t"
        "ldr		x5, [%[b], 352]\n\t"
        "ldr		x7, [%[b], 360]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 352]\n\t"
        "str		x6, [%[r], 360]\n\t"
        "ldr		x4, [%[a], 368]\n\t"
        "ldr		x6, [%[a], 376]\n\t"
        "ldr		x5, [%[b], 368]\n\t"
        "ldr		x7, [%[b], 376]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 368]\n\t"
        "str		x6, [%[r], 376]\n\t"
        "csetm	%[c], cc\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8"
    );
#endif /* WOLFSSL_SP_SMALL */

    return c;
}

/* Reduce the number back to 3072 bits using Montgomery reduction.
 *
 * a   A single precision number to reduce in place.
 * m   The single precision number representing the modulus.
 * mp  The digit representing the negative inverse of m mod 2^n.
 */
SP_NOINLINE static void sp_3072_mont_reduce_48(sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    sp_digit ca = 0;

    __asm__ __volatile__ (
        "ldp       x12, x13, [%[m], 0]\n\t"
        "ldp       x14, x15, [%[m], 16]\n\t"
        "ldp       x16, x17, [%[m], 32]\n\t"
        "ldp       x19, x20, [%[m], 48]\n\t"
        "ldp       x21, x22, [%[m], 64]\n\t"
        "ldp       x23, x24, [%[m], 80]\n\t"
        "ldp       x25, x26, [%[m], 96]\n\t"
        "ldp       x27, x28, [%[m], 112]\n\t"
        "# i = 0\n\t"
        "mov	x3, 0\n\t"
        "ldp	x10, x11, [%[a], 0]\n\t"
        "\n1:\n\t"
        "# mu = a[i] * mp\n\t"
        "mul	x8, %[mp], x10\n\t"
        "# a[i+0] += m[0] * mu\n\t"
        "ldr	x9, [%[a], 0]\n\t"
        "mul		x6, x12, x8\n\t"
        "umulh	x7, x12, x8\n\t"
        "adds	x10, x10, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "# a[i+1] += m[1] * mu\n\t"
        "ldr	x9, [%[a], 8]\n\t"
        "mul		x6, x13, x8\n\t"
        "umulh	x7, x13, x8\n\t"
        "adds	x10, x11, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x10, x10, x5\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+2] += m[2] * mu\n\t"
        "ldr	x11, [%[a], 16]\n\t"
        "mul		x6, x14, x8\n\t"
        "umulh	x7, x14, x8\n\t"
        "adds	x11, x11, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x11, x11, x4\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+3] += m[3] * mu\n\t"
        "ldr	x9, [%[a], 24]\n\t"
        "mul		x6, x15, x8\n\t"
        "umulh	x7, x15, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 24]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+4] += m[4] * mu\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "mul		x6, x16, x8\n\t"
        "umulh	x7, x16, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 32]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+5] += m[5] * mu\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "mul		x6, x17, x8\n\t"
        "umulh	x7, x17, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 40]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+6] += m[6] * mu\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "mul		x6, x19, x8\n\t"
        "umulh	x7, x19, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 48]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+7] += m[7] * mu\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "mul		x6, x20, x8\n\t"
        "umulh	x7, x20, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 56]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+8] += m[8] * mu\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "mul		x6, x21, x8\n\t"
        "umulh	x7, x21, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 64]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+9] += m[9] * mu\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "mul		x6, x22, x8\n\t"
        "umulh	x7, x22, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 72]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+10] += m[10] * mu\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "mul		x6, x23, x8\n\t"
        "umulh	x7, x23, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 80]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+11] += m[11] * mu\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "mul		x6, x24, x8\n\t"
        "umulh	x7, x24, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 88]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+12] += m[12] * mu\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "mul		x6, x25, x8\n\t"
        "umulh	x7, x25, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 96]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+13] += m[13] * mu\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "mul		x6, x26, x8\n\t"
        "umulh	x7, x26, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 104]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+14] += m[14] * mu\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "mul		x6, x27, x8\n\t"
        "umulh	x7, x27, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 112]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+15] += m[15] * mu\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "mul		x6, x28, x8\n\t"
        "umulh	x7, x28, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 120]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+16] += m[16] * mu\n\t"
        "ldr		x7, [%[m], 128]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 128]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+17] += m[17] * mu\n\t"
        "ldr		x7, [%[m], 136]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 136]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+18] += m[18] * mu\n\t"
        "ldr		x7, [%[m], 144]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 144]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+19] += m[19] * mu\n\t"
        "ldr		x7, [%[m], 152]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 152]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+20] += m[20] * mu\n\t"
        "ldr		x7, [%[m], 160]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 160]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+21] += m[21] * mu\n\t"
        "ldr		x7, [%[m], 168]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 168]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+22] += m[22] * mu\n\t"
        "ldr		x7, [%[m], 176]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 176]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+23] += m[23] * mu\n\t"
        "ldr		x7, [%[m], 184]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 184]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+24] += m[24] * mu\n\t"
        "ldr		x7, [%[m], 192]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 192]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+25] += m[25] * mu\n\t"
        "ldr		x7, [%[m], 200]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 200]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+26] += m[26] * mu\n\t"
        "ldr		x7, [%[m], 208]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 208]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+27] += m[27] * mu\n\t"
        "ldr		x7, [%[m], 216]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 216]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+28] += m[28] * mu\n\t"
        "ldr		x7, [%[m], 224]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 224]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+29] += m[29] * mu\n\t"
        "ldr		x7, [%[m], 232]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 232]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+30] += m[30] * mu\n\t"
        "ldr		x7, [%[m], 240]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 240]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+31] += m[31] * mu\n\t"
        "ldr		x7, [%[m], 248]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 248]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+32] += m[32] * mu\n\t"
        "ldr		x7, [%[m], 256]\n\t"
        "ldr	x9, [%[a], 256]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 256]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+33] += m[33] * mu\n\t"
        "ldr		x7, [%[m], 264]\n\t"
        "ldr	x9, [%[a], 264]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 264]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+34] += m[34] * mu\n\t"
        "ldr		x7, [%[m], 272]\n\t"
        "ldr	x9, [%[a], 272]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 272]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+35] += m[35] * mu\n\t"
        "ldr		x7, [%[m], 280]\n\t"
        "ldr	x9, [%[a], 280]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 280]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+36] += m[36] * mu\n\t"
        "ldr		x7, [%[m], 288]\n\t"
        "ldr	x9, [%[a], 288]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 288]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+37] += m[37] * mu\n\t"
        "ldr		x7, [%[m], 296]\n\t"
        "ldr	x9, [%[a], 296]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 296]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+38] += m[38] * mu\n\t"
        "ldr		x7, [%[m], 304]\n\t"
        "ldr	x9, [%[a], 304]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 304]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+39] += m[39] * mu\n\t"
        "ldr		x7, [%[m], 312]\n\t"
        "ldr	x9, [%[a], 312]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 312]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+40] += m[40] * mu\n\t"
        "ldr		x7, [%[m], 320]\n\t"
        "ldr	x9, [%[a], 320]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 320]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+41] += m[41] * mu\n\t"
        "ldr		x7, [%[m], 328]\n\t"
        "ldr	x9, [%[a], 328]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 328]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+42] += m[42] * mu\n\t"
        "ldr		x7, [%[m], 336]\n\t"
        "ldr	x9, [%[a], 336]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 336]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+43] += m[43] * mu\n\t"
        "ldr		x7, [%[m], 344]\n\t"
        "ldr	x9, [%[a], 344]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 344]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+44] += m[44] * mu\n\t"
        "ldr		x7, [%[m], 352]\n\t"
        "ldr	x9, [%[a], 352]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 352]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+45] += m[45] * mu\n\t"
        "ldr		x7, [%[m], 360]\n\t"
        "ldr	x9, [%[a], 360]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 360]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+46] += m[46] * mu\n\t"
        "ldr		x7, [%[m], 368]\n\t"
        "ldr	x9, [%[a], 368]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 368]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+47] += m[47] * mu\n\t"
        "ldr	x7, [%[m], 376]\n\t"
        "ldr	x9, [%[a], 376]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x7, x7, %[ca]\n\t"
        "cset  %[ca], cs\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 376]\n\t"
        "ldr	x9, [%[a], 384]\n\t"
        "adcs	x9, x9, x7\n\t"
        "str	x9, [%[a], 384]\n\t"
        "adc	%[ca], %[ca], xzr\n\t"
        "# i += 1\n\t"
        "add	%[a], %[a], 8\n\t"
        "add	x3, x3, 8\n\t"
        "cmp	x3, 384\n\t"
        "blt	1b\n\t"
        "str	x10, [%[a], 0]\n\t"
        "str	x11, [%[a], 8]\n\t"
        : [ca] "+r" (ca), [a] "+r" (a)
        : [m] "r" (m), [mp] "r" (mp)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );

    sp_3072_cond_sub_48(a - 48, a, m, (sp_digit)0 - ca);
}

/* Multiply two Montogmery form numbers mod the modulus (prime).
 * (r = a * b mod m)
 *
 * r   Result of multiplication.
 * a   First number to multiply in Montogmery form.
 * b   Second number to multiply in Montogmery form.
 * m   Modulus (prime).
 * mp  Montogmery mulitplier.
 */
static void sp_3072_mont_mul_48(sp_digit* r, const sp_digit* a, const sp_digit* b,
        const sp_digit* m, sp_digit mp)
{
    sp_3072_mul_48(r, a, b);
    sp_3072_mont_reduce_48(r, m, mp);
}

/* Square the Montgomery form number. (r = a * a mod m)
 *
 * r   Result of squaring.
 * a   Number to square in Montogmery form.
 * m   Modulus (prime).
 * mp  Montogmery mulitplier.
 */
static void sp_3072_mont_sqr_48(sp_digit* r, const sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    sp_3072_sqr_48(r, a);
    sp_3072_mont_reduce_48(r, m, mp);
}

/* Divide the double width number (d1|d0) by the dividend. (d1|d0 / div)
 *
 * d1   The high order half of the number to divide.
 * d0   The low order half of the number to divide.
 * div  The dividend.
 * returns the result of the division.
 */
static sp_digit div_3072_word_48(sp_digit d1, sp_digit d0, sp_digit div)
{
    sp_digit r;

    __asm__ __volatile__ (
        "lsr	x5, %[div], 32\n\t"
        "add	x5, x5, 1\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x6, x3, 32\n\t"
        "mul	x4, %[div], x6\n\t"
        "umulh	x3, %[div], x6\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x3, x3, 32\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "lsl	x3, %[d1], 32\n\t"
        "orr	x3, x3, %[d0], lsr 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "lsl	x3, %[d1], 32\n\t"
        "orr	x3, x3, %[d0], lsr 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "udiv	x3, %[d0], %[div]\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x3, %[div], x3\n\t"
        "sub	%[d0], %[d0], x3\n\t"
        "mov	%[r], x6\n\t"

        : [r] "=r" (r)
        : [d1] "r" (d1), [d0] "r" (d0), [div] "r" (div)
        : "x3", "x4", "x5", "x6"
    );

    return r;
}

/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_3072_mask_48(sp_digit* r, const sp_digit* a, sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    int i;

    for (i=0; i<48; i++) {
        r[i] = a[i] & m;
    }
#else
    int i;

    for (i = 0; i < 48; i += 8) {
        r[i+0] = a[i+0] & m;
        r[i+1] = a[i+1] & m;
        r[i+2] = a[i+2] & m;
        r[i+3] = a[i+3] & m;
        r[i+4] = a[i+4] & m;
        r[i+5] = a[i+5] & m;
        r[i+6] = a[i+6] & m;
        r[i+7] = a[i+7] & m;
    }
#endif
}

/* Compare a with b in constant time.
 *
 * a  A single precision integer.
 * b  A single precision integer.
 * return -ve, 0 or +ve if a is less than, equal to or greater than b
 * respectively.
 */
static int64_t sp_3072_cmp_48(const sp_digit* a, const sp_digit* b)
{
    sp_digit r = -1;
    sp_digit one = 1;

#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x3, -1\n\t"
        "mov	x6, 376\n\t"
        "1:\n\t"
        "ldr	x4, [%[a], x6]\n\t"
        "ldr	x5, [%[b], x6]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "subs	x6, x6, #8\n\t"
        "b.cs	1b\n\t"
        "eor	%[r], %[r], x3\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [one] "r" (one)
        : "x2", "x3", "x4", "x5", "x6"
    );
#else
    __asm__ __volatile__ (
        "mov	x3, -1\n\t"
        "ldr		x4, [%[a], 376]\n\t"
        "ldr		x5, [%[b], 376]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 368]\n\t"
        "ldr		x5, [%[b], 368]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 360]\n\t"
        "ldr		x5, [%[b], 360]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 352]\n\t"
        "ldr		x5, [%[b], 352]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 344]\n\t"
        "ldr		x5, [%[b], 344]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 336]\n\t"
        "ldr		x5, [%[b], 336]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 328]\n\t"
        "ldr		x5, [%[b], 328]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 320]\n\t"
        "ldr		x5, [%[b], 320]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 312]\n\t"
        "ldr		x5, [%[b], 312]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 304]\n\t"
        "ldr		x5, [%[b], 304]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 296]\n\t"
        "ldr		x5, [%[b], 296]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 288]\n\t"
        "ldr		x5, [%[b], 288]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 280]\n\t"
        "ldr		x5, [%[b], 280]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 272]\n\t"
        "ldr		x5, [%[b], 272]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 264]\n\t"
        "ldr		x5, [%[b], 264]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 256]\n\t"
        "ldr		x5, [%[b], 256]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 248]\n\t"
        "ldr		x5, [%[b], 248]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 240]\n\t"
        "ldr		x5, [%[b], 240]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 232]\n\t"
        "ldr		x5, [%[b], 232]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 224]\n\t"
        "ldr		x5, [%[b], 224]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 216]\n\t"
        "ldr		x5, [%[b], 216]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 208]\n\t"
        "ldr		x5, [%[b], 208]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 200]\n\t"
        "ldr		x5, [%[b], 200]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 192]\n\t"
        "ldr		x5, [%[b], 192]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 184]\n\t"
        "ldr		x5, [%[b], 184]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 176]\n\t"
        "ldr		x5, [%[b], 176]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 168]\n\t"
        "ldr		x5, [%[b], 168]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 160]\n\t"
        "ldr		x5, [%[b], 160]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 152]\n\t"
        "ldr		x5, [%[b], 152]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 144]\n\t"
        "ldr		x5, [%[b], 144]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 136]\n\t"
        "ldr		x5, [%[b], 136]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 128]\n\t"
        "ldr		x5, [%[b], 128]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 120]\n\t"
        "ldr		x5, [%[b], 120]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 112]\n\t"
        "ldr		x5, [%[b], 112]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 104]\n\t"
        "ldr		x5, [%[b], 104]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 96]\n\t"
        "ldr		x5, [%[b], 96]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 88]\n\t"
        "ldr		x5, [%[b], 88]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 80]\n\t"
        "ldr		x5, [%[b], 80]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 72]\n\t"
        "ldr		x5, [%[b], 72]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 64]\n\t"
        "ldr		x5, [%[b], 64]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 56]\n\t"
        "ldr		x5, [%[b], 56]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 48]\n\t"
        "ldr		x5, [%[b], 48]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 40]\n\t"
        "ldr		x5, [%[b], 40]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 32]\n\t"
        "ldr		x5, [%[b], 32]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 24]\n\t"
        "ldr		x5, [%[b], 24]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 16]\n\t"
        "ldr		x5, [%[b], 16]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 8]\n\t"
        "ldr		x5, [%[b], 8]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 0]\n\t"
        "ldr		x5, [%[b], 0]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "eor	%[r], %[r], x3\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [one] "r" (one)
        : "x2", "x3", "x4", "x5", "x6"
    );
#endif

    return r;
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Nmber to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_3072_div_48(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[96], t2[49];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[47];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 48);
    for (i=47; i>=0; i--) {
        r1 = div_3072_word_48(t1[48 + i], t1[48 + i - 1], div);

        sp_3072_mul_d_48(t2, d, r1);
        t1[48 + i] += sp_3072_sub_in_place_48(&t1[i], t2);
        t1[48 + i] -= t2[48];
        sp_3072_mask_48(t2, d, t1[48 + i]);
        t1[48 + i] += sp_3072_add_48(&t1[i], &t1[i], t2);
        sp_3072_mask_48(t2, d, t1[48 + i]);
        t1[48 + i] += sp_3072_add_48(&t1[i], &t1[i], t2);
    }

    r1 = sp_3072_cmp_48(t1, d) >= 0;
    sp_3072_cond_sub_48(r, t1, d, (sp_digit)0 - r1);

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_3072_mod_48(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_3072_div_48(a, m, NULL, r);
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Nmber to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_3072_div_48_cond(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[96], t2[49];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[47];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 48);
    for (i=47; i>=0; i--) {
        r1 = div_3072_word_48(t1[48 + i], t1[48 + i - 1], div);

        sp_3072_mul_d_48(t2, d, r1);
        t1[48 + i] += sp_3072_sub_in_place_48(&t1[i], t2);
        t1[48 + i] -= t2[48];
        if (t1[48 + i] != 0) {
            t1[48 + i] += sp_3072_add_48(&t1[i], &t1[i], d);
            if (t1[48 + i] != 0)
                t1[48 + i] += sp_3072_add_48(&t1[i], &t1[i], d);
        }
    }

    r1 = sp_3072_cmp_48(t1, d) >= 0;
    sp_3072_cond_sub_48(r, t1, d, (sp_digit)0 - r1);

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_3072_mod_48_cond(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_3072_div_48_cond(a, m, NULL, r);
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || \
                                                     defined(WOLFSSL_HAVE_SP_DH)
#ifdef WOLFSSL_SP_SMALL
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns 0 on success and MEMORY_E on dynamic memory allocation failure.
 */
static int sp_3072_mod_exp_48(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#ifndef WOLFSSL_SMALL_STACK
    sp_digit t[16][96];
#else
    sp_digit* t[16];
    sp_digit* td;
#endif
    sp_digit* norm;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c, y;
    int err = MP_OKAY;

#ifdef WOLFSSL_SMALL_STACK
    td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 16 * 96, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER);
    if (td == NULL) {
        err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
#ifdef WOLFSSL_SMALL_STACK
        for (i=0; i<16; i++) {
            t[i] = td + i * 96;
        }
#endif
        norm = t[0];

        sp_3072_mont_setup(m, &mp);
        sp_3072_mont_norm_48(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 48U);
        if (reduceA != 0) {
            err = sp_3072_mod_48(t[1] + 48, a, m);
            if (err == MP_OKAY) {
                err = sp_3072_mod_48(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 48, a, sizeof(sp_digit) * 48);
            err = sp_3072_mod_48(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_3072_mont_sqr_48(t[ 2], t[ 1], m, mp);
        sp_3072_mont_mul_48(t[ 3], t[ 2], t[ 1], m, mp);
        sp_3072_mont_sqr_48(t[ 4], t[ 2], m, mp);
        sp_3072_mont_mul_48(t[ 5], t[ 3], t[ 2], m, mp);
        sp_3072_mont_sqr_48(t[ 6], t[ 3], m, mp);
        sp_3072_mont_mul_48(t[ 7], t[ 4], t[ 3], m, mp);
        sp_3072_mont_sqr_48(t[ 8], t[ 4], m, mp);
        sp_3072_mont_mul_48(t[ 9], t[ 5], t[ 4], m, mp);
        sp_3072_mont_sqr_48(t[10], t[ 5], m, mp);
        sp_3072_mont_mul_48(t[11], t[ 6], t[ 5], m, mp);
        sp_3072_mont_sqr_48(t[12], t[ 6], m, mp);
        sp_3072_mont_mul_48(t[13], t[ 7], t[ 6], m, mp);
        sp_3072_mont_sqr_48(t[14], t[ 7], m, mp);
        sp_3072_mont_mul_48(t[15], t[ 8], t[ 7], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 4;
        if (c == 64) {
            c = 60;
        }
        y = (int)(n >> c);
        n <<= 64 - c;
        XMEMCPY(r, t[y], sizeof(sp_digit) * 48);
        for (; i>=0 || c>=4; ) {
            if (c == 0) {
                n = e[i--];
                y = n >> 60;
                n <<= 4;
                c = 60;
            }
            else if (c < 4) {
                y = n >> 60;
                n = e[i--];
                c = 4 - c;
                y |= n >> (64 - c);
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (n >> 60) & 0xf;
                n <<= 4;
                c -= 4;
            }

            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);

            sp_3072_mont_mul_48(r, r, t[y], m, mp);
        }

        XMEMSET(&r[48], 0, sizeof(sp_digit) * 48U);
        sp_3072_mont_reduce_48(r, m, mp);

        mask = 0 - (sp_3072_cmp_48(r, m) >= 0);
        sp_3072_cond_sub_48(r, r, m, mask);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (td != NULL) {
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return err;
}
#else
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns 0 on success and MEMORY_E on dynamic memory allocation failure.
 */
static int sp_3072_mod_exp_48(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#ifndef WOLFSSL_SMALL_STACK
    sp_digit t[32][96];
#else
    sp_digit* t[32];
    sp_digit* td;
#endif
    sp_digit* norm;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c, y;
    int err = MP_OKAY;

#ifdef WOLFSSL_SMALL_STACK
    td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 32 * 96, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER);
    if (td == NULL) {
        err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
#ifdef WOLFSSL_SMALL_STACK
        for (i=0; i<32; i++) {
            t[i] = td + i * 96;
        }
#endif
        norm = t[0];

        sp_3072_mont_setup(m, &mp);
        sp_3072_mont_norm_48(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 48U);
        if (reduceA != 0) {
            err = sp_3072_mod_48(t[1] + 48, a, m);
            if (err == MP_OKAY) {
                err = sp_3072_mod_48(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 48, a, sizeof(sp_digit) * 48);
            err = sp_3072_mod_48(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_3072_mont_sqr_48(t[ 2], t[ 1], m, mp);
        sp_3072_mont_mul_48(t[ 3], t[ 2], t[ 1], m, mp);
        sp_3072_mont_sqr_48(t[ 4], t[ 2], m, mp);
        sp_3072_mont_mul_48(t[ 5], t[ 3], t[ 2], m, mp);
        sp_3072_mont_sqr_48(t[ 6], t[ 3], m, mp);
        sp_3072_mont_mul_48(t[ 7], t[ 4], t[ 3], m, mp);
        sp_3072_mont_sqr_48(t[ 8], t[ 4], m, mp);
        sp_3072_mont_mul_48(t[ 9], t[ 5], t[ 4], m, mp);
        sp_3072_mont_sqr_48(t[10], t[ 5], m, mp);
        sp_3072_mont_mul_48(t[11], t[ 6], t[ 5], m, mp);
        sp_3072_mont_sqr_48(t[12], t[ 6], m, mp);
        sp_3072_mont_mul_48(t[13], t[ 7], t[ 6], m, mp);
        sp_3072_mont_sqr_48(t[14], t[ 7], m, mp);
        sp_3072_mont_mul_48(t[15], t[ 8], t[ 7], m, mp);
        sp_3072_mont_sqr_48(t[16], t[ 8], m, mp);
        sp_3072_mont_mul_48(t[17], t[ 9], t[ 8], m, mp);
        sp_3072_mont_sqr_48(t[18], t[ 9], m, mp);
        sp_3072_mont_mul_48(t[19], t[10], t[ 9], m, mp);
        sp_3072_mont_sqr_48(t[20], t[10], m, mp);
        sp_3072_mont_mul_48(t[21], t[11], t[10], m, mp);
        sp_3072_mont_sqr_48(t[22], t[11], m, mp);
        sp_3072_mont_mul_48(t[23], t[12], t[11], m, mp);
        sp_3072_mont_sqr_48(t[24], t[12], m, mp);
        sp_3072_mont_mul_48(t[25], t[13], t[12], m, mp);
        sp_3072_mont_sqr_48(t[26], t[13], m, mp);
        sp_3072_mont_mul_48(t[27], t[14], t[13], m, mp);
        sp_3072_mont_sqr_48(t[28], t[14], m, mp);
        sp_3072_mont_mul_48(t[29], t[15], t[14], m, mp);
        sp_3072_mont_sqr_48(t[30], t[15], m, mp);
        sp_3072_mont_mul_48(t[31], t[16], t[15], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 5;
        if (c == 64) {
            c = 59;
        }
        y = (int)(n >> c);
        n <<= 64 - c;
        XMEMCPY(r, t[y], sizeof(sp_digit) * 48);
        for (; i>=0 || c>=5; ) {
            if (c == 0) {
                n = e[i--];
                y = n >> 59;
                n <<= 5;
                c = 59;
            }
            else if (c < 5) {
                y = n >> 59;
                n = e[i--];
                c = 5 - c;
                y |= n >> (64 - c);
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (n >> 59) & 0x1f;
                n <<= 5;
                c -= 5;
            }

            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);

            sp_3072_mont_mul_48(r, r, t[y], m, mp);
        }

        XMEMSET(&r[48], 0, sizeof(sp_digit) * 48U);
        sp_3072_mont_reduce_48(r, m, mp);

        mask = 0 - (sp_3072_cmp_48(r, m) >= 0);
        sp_3072_cond_sub_48(r, r, m, mask);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (td != NULL) {
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return err;
}
#endif /* WOLFSSL_SP_SMALL */
#endif /* (WOLFSSL_HAVE_SP_RSA && !WOLFSSL_RSA_PUBLIC_ONLY) || WOLFSSL_HAVE_SP_DH */

#ifdef WOLFSSL_HAVE_SP_RSA
/* RSA public key operation.
 *
 * in      Array of bytes representing the number to exponentiate, base.
 * inLen   Number of bytes in base.
 * em      Public exponent.
 * mm      Modulus.
 * out     Buffer to hold big-endian bytes of exponentiation result.
 *         Must be at least 384 bytes long.
 * outLen  Number of bytes in result.
 * returns 0 on success, MP_TO_E when the outLen is too small, MP_READ_E when
 * an array is too long and MEMORY_E when dynamic memory allocation fails.
 */
int sp_RsaPublic_3072(const byte* in, word32 inLen, mp_int* em, mp_int* mm,
    byte* out, word32* outLen)
{
#if !defined(WOLFSSL_SP_SMALL) && !defined(WOLFSSL_SMALL_STACK)
    sp_digit ad[96], md[48], rd[96];
#else
    sp_digit* d = NULL;
#endif
    sp_digit* a;
    sp_digit *ah;
    sp_digit* m;
    sp_digit* r;
    sp_digit e[1];
    int err = MP_OKAY;

    if (*outLen < 384)
        err = MP_TO_E;
    if (err == MP_OKAY && (mp_count_bits(em) > 64 || inLen > 384 ||
                                                     mp_count_bits(mm) != 3072))
        err = MP_READ_E;

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    if (err == MP_OKAY) {
        d = (sp_digit*)XMALLOC(sizeof(sp_digit) * 48 * 5, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (d == NULL)
            err = MEMORY_E;
    }

    if (err == MP_OKAY) {
        a = d;
        r = a + 48 * 2;
        m = r + 48 * 2;
        ah = a + 48;
    }
#else
    a = ad;
    m = md;
    r = rd;
    ah = a + 48;
#endif

    if (err == MP_OKAY) {
        sp_3072_from_bin(ah, 48, in, inLen);
#if DIGIT_BIT >= 64
        e[0] = em->dp[0];
#else
        e[0] = em->dp[0];
        if (em->used > 1)
            e[0] |= ((sp_digit)em->dp[1]) << DIGIT_BIT;
#endif
        if (e[0] == 0)
            err = MP_EXPTMOD_E;
    }
    if (err == MP_OKAY) {
        sp_3072_from_mp(m, 48, mm);

        if (e[0] == 0x3) {
            if (err == MP_OKAY) {
                sp_3072_sqr_48(r, ah);
                err = sp_3072_mod_48_cond(r, r, m);
            }
            if (err == MP_OKAY) {
                sp_3072_mul_48(r, ah, r);
                err = sp_3072_mod_48_cond(r, r, m);
            }
        }
        else {
            int i;
            sp_digit mp;

            sp_3072_mont_setup(m, &mp);

            /* Convert to Montgomery form. */
            XMEMSET(a, 0, sizeof(sp_digit) * 48);
            err = sp_3072_mod_48_cond(a, a, m);

            if (err == MP_OKAY) {
                for (i=63; i>=0; i--)
                    if (e[0] >> i)
                        break;

                XMEMCPY(r, a, sizeof(sp_digit) * 48);
                for (i--; i>=0; i--) {
                    sp_3072_mont_sqr_48(r, r, m, mp);
                    if (((e[0] >> i) & 1) == 1)
                        sp_3072_mont_mul_48(r, r, a, m, mp);
                }
                XMEMSET(&r[48], 0, sizeof(sp_digit) * 48);
                sp_3072_mont_reduce_48(r, m, mp);

                for (i = 47; i > 0; i--) {
                    if (r[i] != m[i])
                        break;
                }
                if (r[i] >= m[i])
                    sp_3072_sub_in_place_48(r, m);
            }
        }
    }

    if (err == MP_OKAY) {
        sp_3072_to_bin(r, out);
        *outLen = 384;
    }

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    if (d != NULL)
        XFREE(d, NULL, DYNAMIC_TYPE_RSA);
#endif

    return err;
}

/* RSA private key operation.
 *
 * in      Array of bytes representing the number to exponentiate, base.
 * inLen   Number of bytes in base.
 * dm      Private exponent.
 * pm      First prime.
 * qm      Second prime.
 * dpm     First prime's CRT exponent.
 * dqm     Second prime's CRT exponent.
 * qim     Inverse of second prime mod p.
 * mm      Modulus.
 * out     Buffer to hold big-endian bytes of exponentiation result.
 *         Must be at least 384 bytes long.
 * outLen  Number of bytes in result.
 * returns 0 on success, MP_TO_E when the outLen is too small, MP_READ_E when
 * an array is too long and MEMORY_E when dynamic memory allocation fails.
 */
int sp_RsaPrivate_3072(const byte* in, word32 inLen, mp_int* dm,
    mp_int* pm, mp_int* qm, mp_int* dpm, mp_int* dqm, mp_int* qim, mp_int* mm,
    byte* out, word32* outLen)
{
#if !defined(WOLFSSL_SP_SMALL) && !defined(WOLFSSL_SMALL_STACK)
    sp_digit ad[48 * 2];
    sp_digit pd[24], qd[24], dpd[24];
    sp_digit tmpad[48], tmpbd[48];
#else
    sp_digit* t = NULL;
#endif
    sp_digit* a;
    sp_digit* p;
    sp_digit* q;
    sp_digit* dp;
    sp_digit* dq;
    sp_digit* qi;
    sp_digit* tmp;
    sp_digit* tmpa;
    sp_digit* tmpb;
    sp_digit* r;
    sp_digit c;
    int err = MP_OKAY;

    (void)dm;
    (void)mm;

    if (*outLen < 384)
        err = MP_TO_E;
    if (err == MP_OKAY && (inLen > 384 || mp_count_bits(mm) != 3072))
        err = MP_READ_E;

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    if (err == MP_OKAY) {
        t = (sp_digit*)XMALLOC(sizeof(sp_digit) * 24 * 11, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (t == NULL)
            err = MEMORY_E;
    }
    if (err == MP_OKAY) {
        a = t;
        p = a + 48 * 2;
        q = p + 24;
        qi = dq = dp = q + 24;
        tmpa = qi + 24;
        tmpb = tmpa + 48;

        tmp = t;
        r = tmp + 48;
    }
#else
    r = a = ad;
    p = pd;
    q = qd;
    qi = dq = dp = dpd;
    tmpa = tmpad;
    tmpb = tmpbd;
    tmp = a + 48;
#endif

    if (err == MP_OKAY) {
        sp_3072_from_bin(a, 48, in, inLen);
        sp_3072_from_mp(p, 24, pm);
        sp_3072_from_mp(q, 24, qm);
        sp_3072_from_mp(dp, 24, dpm);

        err = sp_3072_mod_exp_24(tmpa, a, dp, 1536, p, 1);
    }
    if (err == MP_OKAY) {
        sp_3072_from_mp(dq, 24, dqm);
        err = sp_3072_mod_exp_24(tmpb, a, dq, 1536, q, 1);
    }

    if (err == MP_OKAY) {
        c = sp_3072_sub_in_place_24(tmpa, tmpb);
        sp_3072_mask_24(tmp, p, c);
        sp_3072_add_24(tmpa, tmpa, tmp);

        sp_3072_from_mp(qi, 24, qim);
        sp_3072_mul_24(tmpa, tmpa, qi);
        err = sp_3072_mod_24(tmpa, tmpa, p);
    }

    if (err == MP_OKAY) {
        sp_3072_mul_24(tmpa, q, tmpa);
        XMEMSET(&tmpb[24], 0, sizeof(sp_digit) * 24);
        sp_3072_add_48(r, tmpb, tmpa);

        sp_3072_to_bin(r, out);
        *outLen = 384;
    }

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    if (t != NULL) {
        XMEMSET(t, 0, sizeof(sp_digit) * 24 * 11);
        XFREE(t, NULL, DYNAMIC_TYPE_RSA);
    }
#else
    XMEMSET(tmpad, 0, sizeof(tmpad));
    XMEMSET(tmpbd, 0, sizeof(tmpbd));
    XMEMSET(pd, 0, sizeof(pd));
    XMEMSET(qd, 0, sizeof(qd));
    XMEMSET(dpd, 0, sizeof(dpd));
#endif

    return err;
}
#endif /* WOLFSSL_HAVE_SP_RSA */
#if defined(WOLFSSL_HAVE_SP_DH) || (defined(WOLFSSL_HAVE_SP_RSA) && \
                                              !defined(WOLFSSL_RSA_PUBLIC_ONLY))
/* Convert an array of sp_digit to an mp_int.
 *
 * a  A single precision integer.
 * r  A multi-precision integer.
 */
static int sp_3072_to_mp(const sp_digit* a, mp_int* r)
{
    int err;

    err = mp_grow(r, (3072 + DIGIT_BIT - 1) / DIGIT_BIT);
    if (err == MP_OKAY) { /*lint !e774 case where err is always MP_OKAY*/
#if DIGIT_BIT == 64
        XMEMCPY(r->dp, a, sizeof(sp_digit) * 48);
        r->used = 48;
        mp_clamp(r);
#elif DIGIT_BIT < 64
        int i, j = 0, s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 48; i++) {
            r->dp[j] |= a[i] << s;
            r->dp[j] &= (1L << DIGIT_BIT) - 1;
            s = DIGIT_BIT - s;
            r->dp[++j] = a[i] >> s;
            while (s + DIGIT_BIT <= 64) {
                s += DIGIT_BIT;
                r->dp[j++] &= (1L << DIGIT_BIT) - 1;
                if (s == SP_WORD_SIZE) {
                    r->dp[j] = 0;
                }
                else {
                    r->dp[j] = a[i] >> s;
                }
            }
            s = 64 - s;
        }
        r->used = (3072 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#else
        int i, j = 0, s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 48; i++) {
            r->dp[j] |= ((mp_digit)a[i]) << s;
            if (s + 64 >= DIGIT_BIT) {
    #if DIGIT_BIT != 32 && DIGIT_BIT != 64
                r->dp[j] &= (1L << DIGIT_BIT) - 1;
    #endif
                s = DIGIT_BIT - s;
                r->dp[++j] = a[i] >> s;
                s = 64 - s;
            }
            else {
                s += 64;
            }
        }
        r->used = (3072 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#endif
    }

    return err;
}

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base  Base. MP integer.
 * exp   Exponent. MP integer.
 * mod   Modulus. MP integer.
 * res   Result. MP integer.
 * returs 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_ModExp_3072(mp_int* base, mp_int* exp, mp_int* mod, mp_int* res)
{
    int err = MP_OKAY;
    sp_digit b[96], e[48], m[48];
    sp_digit* r = b;
    int expBits = mp_count_bits(exp);

    if (mp_count_bits(base) > 3072) {
        err = MP_READ_E;
    }

    if (err == MP_OKAY) {
        if (expBits > 3072) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        if (mp_count_bits(mod) != 3072) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        sp_3072_from_mp(b, 48, base);
        sp_3072_from_mp(e, 48, exp);
        sp_3072_from_mp(m, 48, mod);

        err = sp_3072_mod_exp_48(r, b, e, expBits, m, 0);
    }

    if (err == MP_OKAY) {
        err = sp_3072_to_mp(r, res);
    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}

#ifdef WOLFSSL_HAVE_SP_DH

#ifdef HAVE_FFDHE_3072
static void sp_3072_lshift_48(sp_digit* r, sp_digit* a, byte n)
{
    __asm__ __volatile__ (
        "mov	x6, 63\n\t"
        "sub	x6, x6, %[n]\n\t"
        "ldr	x3, [%[a], 376]\n\t"
        "lsr	x4, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x4, x4, x6\n\t"
        "ldr	x2, [%[a], 368]\n\t"
        "str	x4, [%[r], 384]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 360]\n\t"
        "str	x3, [%[r], 376]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 352]\n\t"
        "str	x2, [%[r], 368]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 344]\n\t"
        "str	x4, [%[r], 360]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 336]\n\t"
        "str	x3, [%[r], 352]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 328]\n\t"
        "str	x2, [%[r], 344]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 320]\n\t"
        "str	x4, [%[r], 336]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 312]\n\t"
        "str	x3, [%[r], 328]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 304]\n\t"
        "str	x2, [%[r], 320]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 296]\n\t"
        "str	x4, [%[r], 312]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 288]\n\t"
        "str	x3, [%[r], 304]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 280]\n\t"
        "str	x2, [%[r], 296]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 272]\n\t"
        "str	x4, [%[r], 288]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 264]\n\t"
        "str	x3, [%[r], 280]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 256]\n\t"
        "str	x2, [%[r], 272]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 248]\n\t"
        "str	x4, [%[r], 264]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 240]\n\t"
        "str	x3, [%[r], 256]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 232]\n\t"
        "str	x2, [%[r], 248]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 224]\n\t"
        "str	x4, [%[r], 240]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 216]\n\t"
        "str	x3, [%[r], 232]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 208]\n\t"
        "str	x2, [%[r], 224]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 200]\n\t"
        "str	x4, [%[r], 216]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 192]\n\t"
        "str	x3, [%[r], 208]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 184]\n\t"
        "str	x2, [%[r], 200]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 176]\n\t"
        "str	x4, [%[r], 192]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 168]\n\t"
        "str	x3, [%[r], 184]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 160]\n\t"
        "str	x2, [%[r], 176]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 152]\n\t"
        "str	x4, [%[r], 168]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 144]\n\t"
        "str	x3, [%[r], 160]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 136]\n\t"
        "str	x2, [%[r], 152]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 128]\n\t"
        "str	x4, [%[r], 144]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 120]\n\t"
        "str	x3, [%[r], 136]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 112]\n\t"
        "str	x2, [%[r], 128]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 104]\n\t"
        "str	x4, [%[r], 120]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 96]\n\t"
        "str	x3, [%[r], 112]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 88]\n\t"
        "str	x2, [%[r], 104]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 80]\n\t"
        "str	x4, [%[r], 96]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 72]\n\t"
        "str	x3, [%[r], 88]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 64]\n\t"
        "str	x2, [%[r], 80]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 56]\n\t"
        "str	x4, [%[r], 72]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 48]\n\t"
        "str	x3, [%[r], 64]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 40]\n\t"
        "str	x2, [%[r], 56]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 32]\n\t"
        "str	x4, [%[r], 48]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 24]\n\t"
        "str	x3, [%[r], 40]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 16]\n\t"
        "str	x2, [%[r], 32]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 8]\n\t"
        "str	x4, [%[r], 24]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 0]\n\t"
        "str	x3, [%[r], 16]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "str	x4, [%[r]]\n\t"
        "str	x2, [%[r], 8]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [n] "r" (n)
        : "memory", "x2", "x3", "x4", "x5", "x6"
    );
}

/* Modular exponentiate 2 to the e mod m. (r = 2^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns 0 on success and MEMORY_E on dynamic memory allocation failure.
 */
static int sp_3072_mod_exp_2_48(sp_digit* r, const sp_digit* e, int bits,
        const sp_digit* m)
{
#ifndef WOLFSSL_SMALL_STACK
    sp_digit nd[96];
    sp_digit td[49];
#else
    sp_digit* td;
#endif
    sp_digit* norm;
    sp_digit* tmp;
    sp_digit mp = 1;
    sp_digit n, o;
    sp_digit mask;
    int i;
    int c, y;
    int err = MP_OKAY;

#ifdef WOLFSSL_SMALL_STACK
    td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 145, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER);
    if (td == NULL) {
        err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
#ifdef WOLFSSL_SMALL_STACK
        norm = td;
        tmp  = td + 96;
#else
        norm = nd;
        tmp  = td;
#endif

        sp_3072_mont_setup(m, &mp);
        sp_3072_mont_norm_48(norm, m);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 6;
        if (c == 64) {
            c = 58;
        }
        y = (int)(n >> c);
        n <<= 64 - c;
        sp_3072_lshift_48(r, norm, y);
        for (; i>=0 || c>=6; ) {
            if (c == 0) {
                n = e[i--];
                y = n >> 58;
                n <<= 6;
                c = 58;
            }
            else if (c < 6) {
                y = n >> 58;
                n = e[i--];
                c = 6 - c;
                y |= n >> (64 - c);
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (n >> 58) & 0x3f;
                n <<= 6;
                c -= 6;
            }

            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);

            sp_3072_lshift_48(r, r, y);
            sp_3072_mul_d_48(tmp, norm, r[48]);
            r[48] = 0;
            o = sp_3072_add_48(r, r, tmp);
            sp_3072_cond_sub_48(r, r, m, (sp_digit)0 - o);
        }

        XMEMSET(&r[48], 0, sizeof(sp_digit) * 48U);
        sp_3072_mont_reduce_48(r, m, mp);

        mask = 0 - (sp_3072_cmp_48(r, m) >= 0);
        sp_3072_cond_sub_48(r, r, m, mask);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (td != NULL) {
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return err;
}
#endif /* HAVE_FFDHE_3072 */

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base     Base.
 * exp      Array of bytes that is the exponent.
 * expLen   Length of data, in bytes, in exponent.
 * mod      Modulus.
 * out      Buffer to hold big-endian bytes of exponentiation result.
 *          Must be at least 384 bytes long.
 * outLen   Length, in bytes, of exponentiation result.
 * returs 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_DhExp_3072(mp_int* base, const byte* exp, word32 expLen,
    mp_int* mod, byte* out, word32* outLen)
{
    int err = MP_OKAY;
    sp_digit b[96], e[48], m[48];
    sp_digit* r = b;
    word32 i;

    if (mp_count_bits(base) > 3072) {
        err = MP_READ_E;
    }

    if (err == MP_OKAY) {
        if (expLen > 384) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        if (mp_count_bits(mod) != 3072) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        sp_3072_from_mp(b, 48, base);
        sp_3072_from_bin(e, 48, exp, expLen);
        sp_3072_from_mp(m, 48, mod);

    #ifdef HAVE_FFDHE_3072
        if (base->used == 1 && base->dp[0] == 2 && m[47] == (sp_digit)-1)
            err = sp_3072_mod_exp_2_48(r, e, expLen * 8, m);
        else
    #endif
            err = sp_3072_mod_exp_48(r, b, e, expLen * 8, m, 0);

    }

    if (err == MP_OKAY) {
        sp_3072_to_bin(r, out);
        *outLen = 384;
        for (i=0; i<384 && out[i] == 0; i++) {
        }
        *outLen -= i;
        XMEMMOVE(out, out + i, *outLen);

    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}
#endif /* WOLFSSL_HAVE_SP_DH */

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base  Base. MP integer.
 * exp   Exponent. MP integer.
 * mod   Modulus. MP integer.
 * res   Result. MP integer.
 * returs 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_ModExp_1536(mp_int* base, mp_int* exp, mp_int* mod, mp_int* res)
{
    int err = MP_OKAY;
    sp_digit b[48], e[24], m[24];
    sp_digit* r = b;
    int expBits = mp_count_bits(exp);

    if (mp_count_bits(base) > 1536) {
        err = MP_READ_E;
    }

    if (err == MP_OKAY) {
        if (expBits > 1536) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        if (mp_count_bits(mod) != 1536) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        sp_3072_from_mp(b, 24, base);
        sp_3072_from_mp(e, 24, exp);
        sp_3072_from_mp(m, 24, mod);

        err = sp_3072_mod_exp_24(r, b, e, expBits, m, 0);
    }

    if (err == MP_OKAY) {
        XMEMSET(r + 24, 0, sizeof(*r) * 24U);
        err = sp_3072_to_mp(r, res);
        res->used = mod->used;
        mp_clamp(res);
    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}

#endif /* WOLFSSL_HAVE_SP_DH || (WOLFSSL_HAVE_SP_RSA && !WOLFSSL_RSA_PUBLIC_ONLY) */

#endif /* !WOLFSSL_SP_NO_3072 */

#ifdef WOLFSSL_SP_4096
/* Read big endian unsigned byte array into r.
 *
 * r  A single precision integer.
 * size  Maximum number of bytes to convert
 * a  Byte array.
 * n  Number of bytes in array to read.
 */
static void sp_4096_from_bin(sp_digit* r, int size, const byte* a, int n)
{
    int i, j = 0;
    word32 s = 0;

    r[0] = 0;
    for (i = n-1; i >= 0; i--) {
        r[j] |= (((sp_digit)a[i]) << s);
        if (s >= 56U) {
            r[j] &= 0xffffffffffffffffl;
            s = 64U - s;
            if (j + 1 >= size) {
                break;
            }
            r[++j] = (sp_digit)a[i] >> s;
            s = 8U - s;
        }
        else {
            s += 8U;
        }
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
}

/* Convert an mp_int to an array of sp_digit.
 *
 * r  A single precision integer.
 * size  Maximum number of bytes to convert
 * a  A multi-precision integer.
 */
static void sp_4096_from_mp(sp_digit* r, int size, const mp_int* a)
{
#if DIGIT_BIT == 64
    int j;

    XMEMCPY(r, a->dp, sizeof(sp_digit) * a->used);

    for (j = a->used; j < size; j++) {
        r[j] = 0;
    }
#elif DIGIT_BIT > 64
    int i, j = 0;
    word32 s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i] << s);
        r[j] &= 0xffffffffffffffffl;
        s = 64U - s;
        if (j + 1 >= size) {
            break;
        }
        /* lint allow cast of mismatch word32 and mp_digit */
        r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
        while ((s + 64U) <= (word32)DIGIT_BIT) {
            s += 64U;
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            if (s < (word32)DIGIT_BIT) {
                /* lint allow cast of mismatch word32 and mp_digit */
                r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
            }
            else {
                r[++j] = 0L;
            }
        }
        s = (word32)DIGIT_BIT - s;
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#else
    int i, j = 0, s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i]) << s;
        if (s + DIGIT_BIT >= 64) {
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            s = 64 - s;
            if (s == DIGIT_BIT) {
                r[++j] = 0;
                s = 0;
            }
            else {
                r[++j] = a->dp[i] >> s;
                s = DIGIT_BIT - s;
            }
        }
        else {
            s += DIGIT_BIT;
        }
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#endif
}

/* Write r as big endian to byte array.
 * Fixed length number of bytes written: 512
 *
 * r  A single precision integer.
 * a  Byte array.
 */
static void sp_4096_to_bin(sp_digit* r, byte* a)
{
    int i, j, s = 0, b;

    j = 4096 / 8 - 1;
    a[j] = 0;
    for (i=0; i<64 && j>=0; i++) {
        b = 0;
        /* lint allow cast of mismatch sp_digit and int */
        a[j--] |= (byte)(r[i] << s); b += 8 - s; /*lint !e9033*/
        if (j < 0) {
            break;
        }
        while (b < 64) {
            a[j--] = r[i] >> b; b += 8;
            if (j < 0) {
                break;
            }
        }
        s = 8 - (b - 64);
        if (j >= 0) {
            a[j] = 0;
        }
        if (s != 0) {
            j++;
        }
    }
}

#ifndef WOLFSSL_SP_SMALL
/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_4096_add_32(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adds	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "ldp	x7, x8, [%[b], 128]\n\t"
        "ldp	x9, x10, [%[b], 144]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "ldp	x7, x8, [%[b], 160]\n\t"
        "ldp	x9, x10, [%[b], 176]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        "ldp	x3, x4, [%[a], 192]\n\t"
        "ldp	x5, x6, [%[a], 208]\n\t"
        "ldp	x7, x8, [%[b], 192]\n\t"
        "ldp	x9, x10, [%[b], 208]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 192]\n\t"
        "stp	x5, x6, [%[r], 208]\n\t"
        "ldp	x3, x4, [%[a], 224]\n\t"
        "ldp	x5, x6, [%[a], 240]\n\t"
        "ldp	x7, x8, [%[b], 224]\n\t"
        "ldp	x9, x10, [%[b], 240]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 224]\n\t"
        "stp	x5, x6, [%[r], 240]\n\t"
        "cset	%[c], cs\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer and result.
 * b  A single precision integer.
 */
static sp_digit sp_4096_sub_in_place_64(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "ldp	x2, x3, [%[a], 0]\n\t"
        "ldp	x4, x5, [%[a], 16]\n\t"
        "ldp	x6, x7, [%[b], 0]\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "subs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 0]\n\t"
        "stp	x4, x5, [%[a], 16]\n\t"
        "ldp	x2, x3, [%[a], 32]\n\t"
        "ldp	x4, x5, [%[a], 48]\n\t"
        "ldp	x6, x7, [%[b], 32]\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 32]\n\t"
        "stp	x4, x5, [%[a], 48]\n\t"
        "ldp	x2, x3, [%[a], 64]\n\t"
        "ldp	x4, x5, [%[a], 80]\n\t"
        "ldp	x6, x7, [%[b], 64]\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 64]\n\t"
        "stp	x4, x5, [%[a], 80]\n\t"
        "ldp	x2, x3, [%[a], 96]\n\t"
        "ldp	x4, x5, [%[a], 112]\n\t"
        "ldp	x6, x7, [%[b], 96]\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 96]\n\t"
        "stp	x4, x5, [%[a], 112]\n\t"
        "ldp	x2, x3, [%[a], 128]\n\t"
        "ldp	x4, x5, [%[a], 144]\n\t"
        "ldp	x6, x7, [%[b], 128]\n\t"
        "ldp	x8, x9, [%[b], 144]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 128]\n\t"
        "stp	x4, x5, [%[a], 144]\n\t"
        "ldp	x2, x3, [%[a], 160]\n\t"
        "ldp	x4, x5, [%[a], 176]\n\t"
        "ldp	x6, x7, [%[b], 160]\n\t"
        "ldp	x8, x9, [%[b], 176]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 160]\n\t"
        "stp	x4, x5, [%[a], 176]\n\t"
        "ldp	x2, x3, [%[a], 192]\n\t"
        "ldp	x4, x5, [%[a], 208]\n\t"
        "ldp	x6, x7, [%[b], 192]\n\t"
        "ldp	x8, x9, [%[b], 208]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 192]\n\t"
        "stp	x4, x5, [%[a], 208]\n\t"
        "ldp	x2, x3, [%[a], 224]\n\t"
        "ldp	x4, x5, [%[a], 240]\n\t"
        "ldp	x6, x7, [%[b], 224]\n\t"
        "ldp	x8, x9, [%[b], 240]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 224]\n\t"
        "stp	x4, x5, [%[a], 240]\n\t"
        "ldp	x2, x3, [%[a], 256]\n\t"
        "ldp	x4, x5, [%[a], 272]\n\t"
        "ldp	x6, x7, [%[b], 256]\n\t"
        "ldp	x8, x9, [%[b], 272]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 256]\n\t"
        "stp	x4, x5, [%[a], 272]\n\t"
        "ldp	x2, x3, [%[a], 288]\n\t"
        "ldp	x4, x5, [%[a], 304]\n\t"
        "ldp	x6, x7, [%[b], 288]\n\t"
        "ldp	x8, x9, [%[b], 304]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 288]\n\t"
        "stp	x4, x5, [%[a], 304]\n\t"
        "ldp	x2, x3, [%[a], 320]\n\t"
        "ldp	x4, x5, [%[a], 336]\n\t"
        "ldp	x6, x7, [%[b], 320]\n\t"
        "ldp	x8, x9, [%[b], 336]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 320]\n\t"
        "stp	x4, x5, [%[a], 336]\n\t"
        "ldp	x2, x3, [%[a], 352]\n\t"
        "ldp	x4, x5, [%[a], 368]\n\t"
        "ldp	x6, x7, [%[b], 352]\n\t"
        "ldp	x8, x9, [%[b], 368]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 352]\n\t"
        "stp	x4, x5, [%[a], 368]\n\t"
        "ldp	x2, x3, [%[a], 384]\n\t"
        "ldp	x4, x5, [%[a], 400]\n\t"
        "ldp	x6, x7, [%[b], 384]\n\t"
        "ldp	x8, x9, [%[b], 400]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 384]\n\t"
        "stp	x4, x5, [%[a], 400]\n\t"
        "ldp	x2, x3, [%[a], 416]\n\t"
        "ldp	x4, x5, [%[a], 432]\n\t"
        "ldp	x6, x7, [%[b], 416]\n\t"
        "ldp	x8, x9, [%[b], 432]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 416]\n\t"
        "stp	x4, x5, [%[a], 432]\n\t"
        "ldp	x2, x3, [%[a], 448]\n\t"
        "ldp	x4, x5, [%[a], 464]\n\t"
        "ldp	x6, x7, [%[b], 448]\n\t"
        "ldp	x8, x9, [%[b], 464]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 448]\n\t"
        "stp	x4, x5, [%[a], 464]\n\t"
        "ldp	x2, x3, [%[a], 480]\n\t"
        "ldp	x4, x5, [%[a], 496]\n\t"
        "ldp	x6, x7, [%[b], 480]\n\t"
        "ldp	x8, x9, [%[b], 496]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a], 480]\n\t"
        "stp	x4, x5, [%[a], 496]\n\t"
        "csetm	%[c], cc\n\t"
        : [c] "+r" (c)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    return c;
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_4096_add_64(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adds	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "ldp	x7, x8, [%[b], 128]\n\t"
        "ldp	x9, x10, [%[b], 144]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "ldp	x7, x8, [%[b], 160]\n\t"
        "ldp	x9, x10, [%[b], 176]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        "ldp	x3, x4, [%[a], 192]\n\t"
        "ldp	x5, x6, [%[a], 208]\n\t"
        "ldp	x7, x8, [%[b], 192]\n\t"
        "ldp	x9, x10, [%[b], 208]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 192]\n\t"
        "stp	x5, x6, [%[r], 208]\n\t"
        "ldp	x3, x4, [%[a], 224]\n\t"
        "ldp	x5, x6, [%[a], 240]\n\t"
        "ldp	x7, x8, [%[b], 224]\n\t"
        "ldp	x9, x10, [%[b], 240]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 224]\n\t"
        "stp	x5, x6, [%[r], 240]\n\t"
        "ldp	x3, x4, [%[a], 256]\n\t"
        "ldp	x5, x6, [%[a], 272]\n\t"
        "ldp	x7, x8, [%[b], 256]\n\t"
        "ldp	x9, x10, [%[b], 272]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 256]\n\t"
        "stp	x5, x6, [%[r], 272]\n\t"
        "ldp	x3, x4, [%[a], 288]\n\t"
        "ldp	x5, x6, [%[a], 304]\n\t"
        "ldp	x7, x8, [%[b], 288]\n\t"
        "ldp	x9, x10, [%[b], 304]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 288]\n\t"
        "stp	x5, x6, [%[r], 304]\n\t"
        "ldp	x3, x4, [%[a], 320]\n\t"
        "ldp	x5, x6, [%[a], 336]\n\t"
        "ldp	x7, x8, [%[b], 320]\n\t"
        "ldp	x9, x10, [%[b], 336]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 320]\n\t"
        "stp	x5, x6, [%[r], 336]\n\t"
        "ldp	x3, x4, [%[a], 352]\n\t"
        "ldp	x5, x6, [%[a], 368]\n\t"
        "ldp	x7, x8, [%[b], 352]\n\t"
        "ldp	x9, x10, [%[b], 368]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 352]\n\t"
        "stp	x5, x6, [%[r], 368]\n\t"
        "ldp	x3, x4, [%[a], 384]\n\t"
        "ldp	x5, x6, [%[a], 400]\n\t"
        "ldp	x7, x8, [%[b], 384]\n\t"
        "ldp	x9, x10, [%[b], 400]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 384]\n\t"
        "stp	x5, x6, [%[r], 400]\n\t"
        "ldp	x3, x4, [%[a], 416]\n\t"
        "ldp	x5, x6, [%[a], 432]\n\t"
        "ldp	x7, x8, [%[b], 416]\n\t"
        "ldp	x9, x10, [%[b], 432]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 416]\n\t"
        "stp	x5, x6, [%[r], 432]\n\t"
        "ldp	x3, x4, [%[a], 448]\n\t"
        "ldp	x5, x6, [%[a], 464]\n\t"
        "ldp	x7, x8, [%[b], 448]\n\t"
        "ldp	x9, x10, [%[b], 464]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 448]\n\t"
        "stp	x5, x6, [%[r], 464]\n\t"
        "ldp	x3, x4, [%[a], 480]\n\t"
        "ldp	x5, x6, [%[a], 496]\n\t"
        "ldp	x7, x8, [%[b], 480]\n\t"
        "ldp	x9, x10, [%[b], 496]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 480]\n\t"
        "stp	x5, x6, [%[r], 496]\n\t"
        "cset	%[c], cs\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_4096_mul_32(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[32];

    __asm__ __volatile__ (
        "#  A[0] * B[0]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x3, x7, x8\n\t"
        "umulh	x4, x7, x8\n\t"
        "mov	x5, 0\n\t"
        "str	x3, [%[tmp]]\n\t"
        "#  A[0] * B[1]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[0]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 8]\n\t"
        "#  A[0] * B[2]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[1]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[0]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 16]\n\t"
        "#  A[0] * B[3]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[2]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[1]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[0]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 24]\n\t"
        "#  A[0] * B[4]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[3]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[2]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[1]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[0]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 32]\n\t"
        "#  A[0] * B[5]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[4]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[3]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[2]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[1]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[0]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 40]\n\t"
        "#  A[0] * B[6]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[5]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[4]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[3]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[2]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[1]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[0]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 48]\n\t"
        "#  A[0] * B[7]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[6]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[5]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[4]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[3]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[2]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[1]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[0]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 56]\n\t"
        "#  A[0] * B[8]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[7]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[6]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[5]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[4]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[3]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[2]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[1]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * B[0]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 64]\n\t"
        "#  A[0] * B[9]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[8]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[7]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[6]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[5]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[4]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[3]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[2]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[8] * B[1]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[9] * B[0]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 72]\n\t"
        "#  A[0] * B[10]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[9]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[8]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[7]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[6]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[5]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[4]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[3]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[8] * B[2]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[9] * B[1]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * B[0]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 80]\n\t"
        "#  A[0] * B[11]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[10]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[9]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[8]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[7]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[6]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[5]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[4]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * B[3]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[9] * B[2]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[10] * B[1]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[11] * B[0]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 88]\n\t"
        "#  A[0] * B[12]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[11]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[10]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[9]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[8]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[7]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[6]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[5]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[8] * B[4]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[9] * B[3]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[10] * B[2]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[11] * B[1]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[12] * B[0]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 96]\n\t"
        "#  A[0] * B[13]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[12]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[11]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[10]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[9]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[8]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[7]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[6]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[8] * B[5]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[9] * B[4]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * B[3]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[11] * B[2]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[12] * B[1]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[13] * B[0]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 104]\n\t"
        "#  A[0] * B[14]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[13]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[12]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[11]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[10]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[9]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[8]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[7]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * B[6]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[9] * B[5]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[10] * B[4]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[11] * B[3]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[12] * B[2]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[13] * B[1]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[14] * B[0]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 112]\n\t"
        "#  A[0] * B[15]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[14]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[13]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[12]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[11]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[10]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[9]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[8]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[8] * B[7]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[9] * B[6]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[10] * B[5]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[11] * B[4]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[12] * B[3]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[13] * B[2]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[14] * B[1]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[15] * B[0]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 120]\n\t"
        "#  A[0] * B[16]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[15]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[14]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[13]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[12]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[11]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[10]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[9]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[8] * B[8]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[9] * B[7]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * B[6]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[11] * B[5]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[12] * B[4]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[13] * B[3]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[14] * B[2]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[15] * B[1]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[16] * B[0]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 128]\n\t"
        "#  A[0] * B[17]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[16]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[15]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[14]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[13]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[12]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[11]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[10]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * B[9]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[9] * B[8]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[10] * B[7]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[11] * B[6]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[12] * B[5]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[13] * B[4]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[14] * B[3]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[15] * B[2]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[16] * B[1]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[17] * B[0]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 136]\n\t"
        "#  A[0] * B[18]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[17]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[16]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[15]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[14]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[13]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[12]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[11]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[8] * B[10]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[9] * B[9]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[10] * B[8]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[11] * B[7]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[12] * B[6]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[13] * B[5]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[14] * B[4]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[15] * B[3]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[16] * B[2]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[17] * B[1]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[18] * B[0]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 144]\n\t"
        "#  A[0] * B[19]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[18]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[17]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[16]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[15]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[14]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[13]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[12]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[8] * B[11]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[9] * B[10]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * B[9]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[11] * B[8]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[12] * B[7]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[13] * B[6]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[14] * B[5]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[15] * B[4]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[16] * B[3]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[17] * B[2]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[18] * B[1]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[19] * B[0]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 152]\n\t"
        "#  A[0] * B[20]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[19]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[18]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[17]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[16]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[15]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[14]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[13]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * B[12]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[9] * B[11]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[10] * B[10]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[11] * B[9]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[12] * B[8]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[13] * B[7]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[14] * B[6]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[15] * B[5]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[16] * B[4]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[17] * B[3]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[18] * B[2]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[19] * B[1]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[20] * B[0]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 160]\n\t"
        "#  A[0] * B[21]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[20]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[19]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[18]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[17]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[16]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[15]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[14]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[8] * B[13]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[9] * B[12]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[10] * B[11]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[11] * B[10]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[12] * B[9]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[13] * B[8]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[14] * B[7]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[15] * B[6]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[16] * B[5]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[17] * B[4]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[18] * B[3]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[19] * B[2]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[20] * B[1]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[21] * B[0]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 168]\n\t"
        "#  A[0] * B[22]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[21]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[20]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[19]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[18]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[17]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[16]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[15]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[8] * B[14]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[9] * B[13]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * B[12]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[11] * B[11]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[12] * B[10]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[13] * B[9]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[14] * B[8]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[15] * B[7]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[16] * B[6]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[17] * B[5]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[18] * B[4]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[19] * B[3]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[20] * B[2]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[21] * B[1]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[22] * B[0]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 176]\n\t"
        "#  A[0] * B[23]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[22]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[21]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[20]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[19]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[18]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[17]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[16]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * B[15]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[9] * B[14]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[10] * B[13]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[11] * B[12]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[12] * B[11]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[13] * B[10]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[14] * B[9]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[15] * B[8]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[16] * B[7]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[17] * B[6]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[18] * B[5]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[19] * B[4]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[20] * B[3]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[21] * B[2]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[22] * B[1]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[23] * B[0]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 184]\n\t"
        "#  A[0] * B[24]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[23]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[22]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[21]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[20]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[19]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[18]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[17]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[8] * B[16]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[9] * B[15]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[10] * B[14]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[11] * B[13]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[12] * B[12]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[13] * B[11]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[14] * B[10]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[15] * B[9]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[16] * B[8]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[17] * B[7]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[18] * B[6]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[19] * B[5]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[20] * B[4]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[21] * B[3]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[22] * B[2]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[23] * B[1]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[24] * B[0]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 192]\n\t"
        "#  A[0] * B[25]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[24]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[23]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[22]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[21]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[20]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[19]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[18]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[8] * B[17]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[9] * B[16]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * B[15]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[11] * B[14]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[12] * B[13]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[13] * B[12]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[14] * B[11]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[15] * B[10]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[16] * B[9]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[17] * B[8]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[18] * B[7]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[19] * B[6]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[20] * B[5]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[21] * B[4]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[22] * B[3]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[23] * B[2]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[24] * B[1]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[25] * B[0]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 200]\n\t"
        "#  A[0] * B[26]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[25]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[24]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[23]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[22]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[21]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[20]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[19]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * B[18]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[9] * B[17]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[10] * B[16]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[11] * B[15]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[12] * B[14]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[13] * B[13]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[14] * B[12]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[15] * B[11]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[16] * B[10]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[17] * B[9]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[18] * B[8]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[19] * B[7]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[20] * B[6]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[21] * B[5]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[22] * B[4]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[23] * B[3]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[24] * B[2]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[25] * B[1]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[26] * B[0]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 208]\n\t"
        "#  A[0] * B[27]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[26]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[25]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[24]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[23]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[22]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[21]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[20]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[8] * B[19]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[9] * B[18]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[10] * B[17]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[11] * B[16]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[12] * B[15]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[13] * B[14]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[14] * B[13]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[15] * B[12]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[16] * B[11]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[17] * B[10]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[18] * B[9]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[19] * B[8]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[20] * B[7]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[21] * B[6]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[22] * B[5]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[23] * B[4]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[24] * B[3]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[25] * B[2]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[26] * B[1]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[27] * B[0]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 216]\n\t"
        "#  A[0] * B[28]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[27]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[26]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[25]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[24]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[23]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[22]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[21]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[8] * B[20]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[9] * B[19]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * B[18]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[11] * B[17]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[12] * B[16]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[13] * B[15]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[14] * B[14]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[15] * B[13]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[16] * B[12]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[17] * B[11]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[18] * B[10]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[19] * B[9]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[20] * B[8]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[21] * B[7]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[22] * B[6]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[23] * B[5]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[24] * B[4]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[25] * B[3]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[26] * B[2]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[27] * B[1]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[28] * B[0]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 224]\n\t"
        "#  A[0] * B[29]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[1] * B[28]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[27]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[26]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[25]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[24]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[23]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[22]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * B[21]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[9] * B[20]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[10] * B[19]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[11] * B[18]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[12] * B[17]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[13] * B[16]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[14] * B[15]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[15] * B[14]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[16] * B[13]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[17] * B[12]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[18] * B[11]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[19] * B[10]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[20] * B[9]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[21] * B[8]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[22] * B[7]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[23] * B[6]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[24] * B[5]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[25] * B[4]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[26] * B[3]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[27] * B[2]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[28] * B[1]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[29] * B[0]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[tmp], 232]\n\t"
        "#  A[0] * B[30]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[1] * B[29]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[28]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[27]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[26]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[25]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[24]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[23]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[8] * B[22]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[9] * B[21]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[10] * B[20]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[11] * B[19]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[12] * B[18]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[13] * B[17]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[14] * B[16]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[15] * B[15]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[16] * B[14]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[17] * B[13]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[18] * B[12]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[19] * B[11]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[20] * B[10]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[21] * B[9]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[22] * B[8]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[23] * B[7]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[24] * B[6]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[25] * B[5]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[26] * B[4]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[27] * B[3]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[28] * B[2]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[29] * B[1]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[30] * B[0]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[tmp], 240]\n\t"
        "#  A[0] * B[31]\n\t"
        "ldr	x7, [%[a], 0]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[1] * B[30]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[2] * B[29]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[28]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[27]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[26]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[25]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[24]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[8] * B[23]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[9] * B[22]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * B[21]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[11] * B[20]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[12] * B[19]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[13] * B[18]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[14] * B[17]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[15] * B[16]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[16] * B[15]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[17] * B[14]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[18] * B[13]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[19] * B[12]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[20] * B[11]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[21] * B[10]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[22] * B[9]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[23] * B[8]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[24] * B[7]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[25] * B[6]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[26] * B[5]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[27] * B[4]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[28] * B[3]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[29] * B[2]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[30] * B[1]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[31] * B[0]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 0]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 248]\n\t"
        "#  A[1] * B[31]\n\t"
        "ldr	x7, [%[a], 8]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[2] * B[30]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[3] * B[29]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[28]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[27]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[26]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[25]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * B[24]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[9] * B[23]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[10] * B[22]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[11] * B[21]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[12] * B[20]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[13] * B[19]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[14] * B[18]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[15] * B[17]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[16] * B[16]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[17] * B[15]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[18] * B[14]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[19] * B[13]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[20] * B[12]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[21] * B[11]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[22] * B[10]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[23] * B[9]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[24] * B[8]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[25] * B[7]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[26] * B[6]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[27] * B[5]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[28] * B[4]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[29] * B[3]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[30] * B[2]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[31] * B[1]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 8]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 256]\n\t"
        "#  A[2] * B[31]\n\t"
        "ldr	x7, [%[a], 16]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[3] * B[30]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[4] * B[29]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[28]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[27]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[26]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[8] * B[25]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[9] * B[24]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[10] * B[23]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[11] * B[22]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[12] * B[21]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[13] * B[20]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[14] * B[19]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[15] * B[18]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[16] * B[17]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[17] * B[16]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[18] * B[15]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[19] * B[14]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[20] * B[13]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[21] * B[12]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[22] * B[11]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[23] * B[10]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[24] * B[9]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[25] * B[8]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[26] * B[7]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[27] * B[6]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[28] * B[5]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[29] * B[4]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[30] * B[3]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[31] * B[2]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 16]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 264]\n\t"
        "#  A[3] * B[31]\n\t"
        "ldr	x7, [%[a], 24]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[4] * B[30]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[5] * B[29]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[6] * B[28]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[27]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[8] * B[26]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[9] * B[25]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * B[24]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[11] * B[23]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[12] * B[22]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[13] * B[21]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[14] * B[20]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[15] * B[19]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[16] * B[18]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[17] * B[17]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[18] * B[16]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[19] * B[15]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[20] * B[14]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[21] * B[13]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[22] * B[12]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[23] * B[11]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[24] * B[10]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[25] * B[9]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[26] * B[8]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[27] * B[7]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[28] * B[6]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[29] * B[5]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[30] * B[4]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[31] * B[3]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 24]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 272]\n\t"
        "#  A[4] * B[31]\n\t"
        "ldr	x7, [%[a], 32]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[5] * B[30]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[6] * B[29]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[7] * B[28]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[8] * B[27]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[9] * B[26]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[10] * B[25]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[11] * B[24]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[12] * B[23]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[13] * B[22]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[14] * B[21]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[15] * B[20]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[16] * B[19]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[17] * B[18]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[18] * B[17]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[19] * B[16]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[20] * B[15]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[21] * B[14]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[22] * B[13]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[23] * B[12]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[24] * B[11]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[25] * B[10]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[26] * B[9]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[27] * B[8]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[28] * B[7]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[29] * B[6]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[30] * B[5]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[31] * B[4]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 32]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 280]\n\t"
        "#  A[5] * B[31]\n\t"
        "ldr	x7, [%[a], 40]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[6] * B[30]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[7] * B[29]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[8] * B[28]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[9] * B[27]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[10] * B[26]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[11] * B[25]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[12] * B[24]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[13] * B[23]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[14] * B[22]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[15] * B[21]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[16] * B[20]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[17] * B[19]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[18] * B[18]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[19] * B[17]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[20] * B[16]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[21] * B[15]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[22] * B[14]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[23] * B[13]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[24] * B[12]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[25] * B[11]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[26] * B[10]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[27] * B[9]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[28] * B[8]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[29] * B[7]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[30] * B[6]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[31] * B[5]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 288]\n\t"
        "#  A[6] * B[31]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[7] * B[30]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[8] * B[29]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[9] * B[28]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[10] * B[27]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[11] * B[26]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[12] * B[25]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[13] * B[24]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[14] * B[23]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[15] * B[22]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[16] * B[21]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[17] * B[20]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[18] * B[19]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[19] * B[18]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[20] * B[17]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[21] * B[16]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[22] * B[15]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[23] * B[14]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[24] * B[13]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[25] * B[12]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[26] * B[11]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[27] * B[10]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[28] * B[9]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[29] * B[8]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[30] * B[7]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[31] * B[6]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 48]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 296]\n\t"
        "#  A[7] * B[31]\n\t"
        "ldr	x7, [%[a], 56]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[8] * B[30]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[9] * B[29]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[10] * B[28]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[11] * B[27]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[12] * B[26]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[13] * B[25]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[14] * B[24]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[15] * B[23]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[16] * B[22]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[17] * B[21]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[18] * B[20]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[19] * B[19]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[20] * B[18]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[21] * B[17]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[22] * B[16]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[23] * B[15]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[24] * B[14]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[25] * B[13]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[26] * B[12]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[27] * B[11]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[28] * B[10]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[29] * B[9]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[30] * B[8]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[31] * B[7]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 56]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 304]\n\t"
        "#  A[8] * B[31]\n\t"
        "ldr	x7, [%[a], 64]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[9] * B[30]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[10] * B[29]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[11] * B[28]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[12] * B[27]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[13] * B[26]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[14] * B[25]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[15] * B[24]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[16] * B[23]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[17] * B[22]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[18] * B[21]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[19] * B[20]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[20] * B[19]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[21] * B[18]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[22] * B[17]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[23] * B[16]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[24] * B[15]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[25] * B[14]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[26] * B[13]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[27] * B[12]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[28] * B[11]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[29] * B[10]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[30] * B[9]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[31] * B[8]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 64]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 312]\n\t"
        "#  A[9] * B[31]\n\t"
        "ldr	x7, [%[a], 72]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[10] * B[30]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[11] * B[29]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[12] * B[28]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[13] * B[27]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[14] * B[26]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[15] * B[25]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[16] * B[24]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[17] * B[23]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[18] * B[22]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[19] * B[21]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[20] * B[20]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[21] * B[19]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[22] * B[18]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[23] * B[17]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[24] * B[16]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[25] * B[15]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[26] * B[14]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[27] * B[13]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[28] * B[12]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[29] * B[11]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[30] * B[10]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[31] * B[9]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 72]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 320]\n\t"
        "#  A[10] * B[31]\n\t"
        "ldr	x7, [%[a], 80]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[11] * B[30]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[12] * B[29]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[13] * B[28]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[14] * B[27]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[15] * B[26]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[16] * B[25]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[17] * B[24]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[18] * B[23]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[19] * B[22]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[20] * B[21]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[21] * B[20]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[22] * B[19]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[23] * B[18]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[24] * B[17]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[25] * B[16]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[26] * B[15]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[27] * B[14]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[28] * B[13]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[29] * B[12]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[30] * B[11]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[31] * B[10]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 80]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 328]\n\t"
        "#  A[11] * B[31]\n\t"
        "ldr	x7, [%[a], 88]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[12] * B[30]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[13] * B[29]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[14] * B[28]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[15] * B[27]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[16] * B[26]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[17] * B[25]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[18] * B[24]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[19] * B[23]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[20] * B[22]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[21] * B[21]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[22] * B[20]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[23] * B[19]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[24] * B[18]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[25] * B[17]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[26] * B[16]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[27] * B[15]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[28] * B[14]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[29] * B[13]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[30] * B[12]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[31] * B[11]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 88]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 336]\n\t"
        "#  A[12] * B[31]\n\t"
        "ldr	x7, [%[a], 96]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[13] * B[30]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[14] * B[29]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[15] * B[28]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[16] * B[27]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[17] * B[26]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[18] * B[25]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[19] * B[24]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[20] * B[23]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[21] * B[22]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[22] * B[21]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[23] * B[20]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[24] * B[19]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[25] * B[18]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[26] * B[17]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[27] * B[16]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[28] * B[15]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[29] * B[14]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[30] * B[13]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[31] * B[12]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 96]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 344]\n\t"
        "#  A[13] * B[31]\n\t"
        "ldr	x7, [%[a], 104]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[14] * B[30]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[15] * B[29]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[16] * B[28]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[17] * B[27]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[18] * B[26]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[19] * B[25]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[20] * B[24]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[21] * B[23]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[22] * B[22]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[23] * B[21]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[24] * B[20]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[25] * B[19]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[26] * B[18]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[27] * B[17]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[28] * B[16]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[29] * B[15]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[30] * B[14]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[31] * B[13]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 104]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 352]\n\t"
        "#  A[14] * B[31]\n\t"
        "ldr	x7, [%[a], 112]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[15] * B[30]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[16] * B[29]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[17] * B[28]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[18] * B[27]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[19] * B[26]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[20] * B[25]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[21] * B[24]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[22] * B[23]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[23] * B[22]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[24] * B[21]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[25] * B[20]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[26] * B[19]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[27] * B[18]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[28] * B[17]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[29] * B[16]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[30] * B[15]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[31] * B[14]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 112]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 360]\n\t"
        "#  A[15] * B[31]\n\t"
        "ldr	x7, [%[a], 120]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[16] * B[30]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[17] * B[29]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[18] * B[28]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[19] * B[27]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[20] * B[26]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[21] * B[25]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[22] * B[24]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[23] * B[23]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[24] * B[22]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[25] * B[21]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[26] * B[20]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[27] * B[19]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[28] * B[18]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[29] * B[17]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[30] * B[16]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[31] * B[15]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 120]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 368]\n\t"
        "#  A[16] * B[31]\n\t"
        "ldr	x7, [%[a], 128]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[17] * B[30]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[18] * B[29]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[19] * B[28]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[20] * B[27]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[21] * B[26]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[22] * B[25]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[23] * B[24]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[24] * B[23]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[25] * B[22]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[26] * B[21]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[27] * B[20]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[28] * B[19]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[29] * B[18]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[30] * B[17]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[31] * B[16]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 128]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 376]\n\t"
        "#  A[17] * B[31]\n\t"
        "ldr	x7, [%[a], 136]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[18] * B[30]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[19] * B[29]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[20] * B[28]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[21] * B[27]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[22] * B[26]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[23] * B[25]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[24] * B[24]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[25] * B[23]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[26] * B[22]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[27] * B[21]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[28] * B[20]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[29] * B[19]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[30] * B[18]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[31] * B[17]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 136]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 384]\n\t"
        "#  A[18] * B[31]\n\t"
        "ldr	x7, [%[a], 144]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[19] * B[30]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[20] * B[29]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[21] * B[28]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[22] * B[27]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[23] * B[26]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[24] * B[25]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[25] * B[24]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[26] * B[23]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[27] * B[22]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[28] * B[21]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[29] * B[20]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[30] * B[19]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[31] * B[18]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 144]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 392]\n\t"
        "#  A[19] * B[31]\n\t"
        "ldr	x7, [%[a], 152]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[20] * B[30]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[21] * B[29]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[22] * B[28]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[23] * B[27]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[24] * B[26]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[25] * B[25]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[26] * B[24]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[27] * B[23]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[28] * B[22]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[29] * B[21]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[30] * B[20]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[31] * B[19]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 152]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 400]\n\t"
        "#  A[20] * B[31]\n\t"
        "ldr	x7, [%[a], 160]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[21] * B[30]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[22] * B[29]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[23] * B[28]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[24] * B[27]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[25] * B[26]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[26] * B[25]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[27] * B[24]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[28] * B[23]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[29] * B[22]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[30] * B[21]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[31] * B[20]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 160]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 408]\n\t"
        "#  A[21] * B[31]\n\t"
        "ldr	x7, [%[a], 168]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[22] * B[30]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[23] * B[29]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[24] * B[28]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[25] * B[27]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[26] * B[26]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[27] * B[25]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[28] * B[24]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[29] * B[23]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[30] * B[22]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[31] * B[21]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 168]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 416]\n\t"
        "#  A[22] * B[31]\n\t"
        "ldr	x7, [%[a], 176]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[23] * B[30]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[24] * B[29]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[25] * B[28]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[26] * B[27]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[27] * B[26]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[28] * B[25]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[29] * B[24]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[30] * B[23]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[31] * B[22]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 176]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 424]\n\t"
        "#  A[23] * B[31]\n\t"
        "ldr	x7, [%[a], 184]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[24] * B[30]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[25] * B[29]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[26] * B[28]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[27] * B[27]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[28] * B[26]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[29] * B[25]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[30] * B[24]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[31] * B[23]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 184]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 432]\n\t"
        "#  A[24] * B[31]\n\t"
        "ldr	x7, [%[a], 192]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[25] * B[30]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[26] * B[29]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[27] * B[28]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[28] * B[27]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[29] * B[26]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[30] * B[25]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[31] * B[24]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 192]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 440]\n\t"
        "#  A[25] * B[31]\n\t"
        "ldr	x7, [%[a], 200]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[26] * B[30]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[27] * B[29]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[28] * B[28]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[29] * B[27]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[30] * B[26]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[31] * B[25]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 200]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 448]\n\t"
        "#  A[26] * B[31]\n\t"
        "ldr	x7, [%[a], 208]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[27] * B[30]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[28] * B[29]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[29] * B[28]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[30] * B[27]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[31] * B[26]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 208]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 456]\n\t"
        "#  A[27] * B[31]\n\t"
        "ldr	x7, [%[a], 216]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[28] * B[30]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[29] * B[29]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[30] * B[28]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[31] * B[27]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 216]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 464]\n\t"
        "#  A[28] * B[31]\n\t"
        "ldr	x7, [%[a], 224]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, xzr, xzr\n\t"
        "#  A[29] * B[30]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[30] * B[29]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[31] * B[28]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 224]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x5, [%[r], 472]\n\t"
        "#  A[29] * B[31]\n\t"
        "ldr	x7, [%[a], 232]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "#  A[30] * B[30]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[31] * B[29]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 232]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, x5, xzr\n\t"
        "str	x3, [%[r], 480]\n\t"
        "#  A[30] * B[31]\n\t"
        "ldr	x7, [%[a], 240]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[31] * B[30]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 240]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 488]\n\t"
        "#  A[31] * B[31]\n\t"
        "ldr	x7, [%[a], 248]\n\t"
        "ldr	x8, [%[b], 248]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "stp	x5, x3, [%[r], 496]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [tmp] "r" (tmp)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_4096_mask_32(sp_digit* r, const sp_digit* a, sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    int i;

    for (i=0; i<32; i++) {
        r[i] = a[i] & m;
    }
#else
    int i;

    for (i = 0; i < 32; i += 8) {
        r[i+0] = a[i+0] & m;
        r[i+1] = a[i+1] & m;
        r[i+2] = a[i+2] & m;
        r[i+3] = a[i+3] & m;
        r[i+4] = a[i+4] & m;
        r[i+5] = a[i+5] & m;
        r[i+6] = a[i+6] & m;
        r[i+7] = a[i+7] & m;
    }
#endif
}

/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
SP_NOINLINE static void sp_4096_mul_64(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit* z0 = r;
    sp_digit z1[64];
    sp_digit a1[32];
    sp_digit b1[32];
    sp_digit z2[64];
    sp_digit u, ca, cb;

    ca = sp_2048_add_32(a1, a, &a[32]);
    cb = sp_2048_add_32(b1, b, &b[32]);
    u  = ca & cb;
    sp_2048_mul_32(z1, a1, b1);
    sp_2048_mul_32(z2, &a[32], &b[32]);
    sp_2048_mul_32(z0, a, b);
    sp_2048_mask_32(r + 64, a1, 0 - cb);
    sp_2048_mask_32(b1, b1, 0 - ca);
    u += sp_2048_add_32(r + 64, r + 64, b1);
    u += sp_4096_sub_in_place_64(z1, z2);
    u += sp_4096_sub_in_place_64(z1, z0);
    u += sp_4096_add_64(r + 32, r + 32, z1);
    r[96] = u;
    XMEMSET(r + 96 + 1, 0, sizeof(sp_digit) * (32 - 1));
    (void)sp_4096_add_64(r + 64, r + 64, z2);
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_4096_sqr_32(sp_digit* r, const sp_digit* a)
{
    sp_digit tmp[32];

    __asm__ __volatile__ (
        "#  A[0] * A[0]\n\t"
        "ldr	x9, [%[a], 0]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x3, x9, x9\n\t"
        "mov	x4, 0\n\t"
        "str	x8, [%[tmp]]\n\t"
        "#  A[0] * A[1]\n\t"
        "ldr	x9, [%[a], 8]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "str	x3, [%[tmp], 8]\n\t"
        "#  A[0] * A[2]\n\t"
        "ldr	x9, [%[a], 16]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[1] * A[1]\n\t"
        "ldr	x9, [%[a], 8]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 16]\n\t"
        "#  A[0] * A[3]\n\t"
        "ldr	x9, [%[a], 24]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[1] * A[2]\n\t"
        "ldr	x9, [%[a], 16]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x2, [%[tmp], 24]\n\t"
        "#  A[0] * A[4]\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "#  A[1] * A[3]\n\t"
        "ldr	x9, [%[a], 24]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "#  A[2] * A[2]\n\t"
        "ldr	x9, [%[a], 16]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "str	x3, [%[tmp], 32]\n\t"
        "#  A[0] * A[5]\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[4]\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[3]\n\t"
        "ldr	x9, [%[a], 24]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 40]\n\t"
        "#  A[0] * A[6]\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[5]\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[4]\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[3]\n\t"
        "ldr	x9, [%[a], 24]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 48]\n\t"
        "#  A[0] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[6]\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[5]\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[4]\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 56]\n\t"
        "#  A[0] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[6]\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[5]\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[4]\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 64]\n\t"
        "#  A[0] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[6]\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[5]\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 72]\n\t"
        "#  A[0] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[6]\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[5]\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 80]\n\t"
        "#  A[0] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[6]\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 88]\n\t"
        "#  A[0] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[6]\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 96]\n\t"
        "#  A[0] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 104]\n\t"
        "#  A[0] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 112]\n\t"
        "#  A[0] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 120]\n\t"
        "#  A[0] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 128]\n\t"
        "#  A[0] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 136]\n\t"
        "#  A[0] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 144]\n\t"
        "#  A[0] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 152]\n\t"
        "#  A[0] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 160]\n\t"
        "#  A[0] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 168]\n\t"
        "#  A[0] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 176]\n\t"
        "#  A[0] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 184]\n\t"
        "#  A[0] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 192]\n\t"
        "#  A[0] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 200]\n\t"
        "#  A[0] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 208]\n\t"
        "#  A[0] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 216]\n\t"
        "#  A[0] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 224]\n\t"
        "#  A[0] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 232]\n\t"
        "#  A[0] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 240]\n\t"
        "#  A[0] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[1] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 248]\n\t"
        "#  A[1] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[2] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 256]\n\t"
        "#  A[2] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[3] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 264]\n\t"
        "#  A[3] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[4] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 272]\n\t"
        "#  A[4] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[5] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 280]\n\t"
        "#  A[5] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[6] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 288]\n\t"
        "#  A[6] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[7] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 296]\n\t"
        "#  A[7] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[8] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[19] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 304]\n\t"
        "#  A[8] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[9] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[19] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 312]\n\t"
        "#  A[9] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[10] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[19] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[20] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 320]\n\t"
        "#  A[10] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[11] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[19] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[20] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 160]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 328]\n\t"
        "#  A[11] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[12] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[19] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[20] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 160]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[21] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 336]\n\t"
        "#  A[12] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[13] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[19] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[20] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 160]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[21] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 168]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 344]\n\t"
        "#  A[13] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[14] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[19] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[20] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 160]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[21] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 168]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[22] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 352]\n\t"
        "#  A[14] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[15] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[19] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[20] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 160]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[21] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 168]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[22] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 176]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 360]\n\t"
        "#  A[15] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[16] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[19] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[20] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 160]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[21] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 168]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[22] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 176]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[23] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 368]\n\t"
        "#  A[16] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[17] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[19] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[20] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 160]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[21] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 168]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[22] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 176]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[23] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "ldr	x10, [%[a], 184]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 376]\n\t"
        "#  A[17] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[18] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[19] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[20] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 160]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[21] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 168]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[22] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 176]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[23] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 184]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[24] * A[24]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 384]\n\t"
        "#  A[18] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[19] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[20] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 160]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[21] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 168]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[22] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 176]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[23] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 184]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[24] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "ldr	x10, [%[a], 192]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 392]\n\t"
        "#  A[19] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[20] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 160]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[21] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 168]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[22] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 176]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[23] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 184]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[24] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 192]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[25] * A[25]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 400]\n\t"
        "#  A[20] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 160]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[21] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 168]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[22] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 176]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[23] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 184]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[24] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 192]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[25] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "ldr	x10, [%[a], 200]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 408]\n\t"
        "#  A[21] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 168]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[22] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 176]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[23] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 184]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[24] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 192]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[25] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 200]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[26] * A[26]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 416]\n\t"
        "#  A[22] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 176]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[23] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 184]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[24] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 192]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[25] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 200]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[26] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "ldr	x10, [%[a], 208]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 424]\n\t"
        "#  A[23] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 184]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[24] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 192]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[25] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 200]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[26] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 208]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[27] * A[27]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 432]\n\t"
        "#  A[24] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 192]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[25] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 200]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[26] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 208]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[27] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "ldr	x10, [%[a], 216]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 440]\n\t"
        "#  A[25] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 200]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[26] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 208]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[27] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 216]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[28] * A[28]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 448]\n\t"
        "#  A[26] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 208]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, 0\n\t"
        "mov	x7, 0\n\t"
        "#  A[27] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 216]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[28] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "ldr	x10, [%[a], 224]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 456]\n\t"
        "#  A[27] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 216]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "#  A[28] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 224]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "#  A[29] * A[29]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "str	x3, [%[r], 464]\n\t"
        "#  A[28] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 224]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[29] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "ldr	x10, [%[a], 232]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 472]\n\t"
        "#  A[29] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 232]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[30] * A[30]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x2, [%[r], 480]\n\t"
        "#  A[30] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "ldr	x10, [%[a], 240]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "str	x3, [%[r], 488]\n\t"
        "#  A[31] * A[31]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x4, x4, x8\n\t"
        "adc	x2, x2, x9\n\t"
        "stp	x4, x2, [%[r], 496]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [tmp] "r" (tmp)
        : "memory", "x2", "x3", "x4", "x8", "x9", "x10", "x5", "x6", "x7"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
SP_NOINLINE static void sp_4096_sqr_64(sp_digit* r, const sp_digit* a)
{
    sp_digit* z0 = r;
    sp_digit z2[64];
    sp_digit z1[64];
    sp_digit a1[32];
    sp_digit u;

    u = sp_2048_add_32(a1, a, &a[32]);
    sp_2048_sqr_32(z1, a1);
    sp_2048_sqr_32(z2, &a[32]);
    sp_2048_sqr_32(z0, a);
    sp_2048_mask_32(r + 64, a1, 0 - u);
    u += sp_2048_add_32(r + 64, r + 64, r + 64);
    u += sp_4096_sub_in_place_64(z1, z2);
    u += sp_4096_sub_in_place_64(z1, z0);
    u += sp_4096_add_64(r + 32, r + 32, z1);
    r[96] = u;
    XMEMSET(r + 96 + 1, 0, sizeof(sp_digit) * (32 - 1));
    (void)sp_4096_add_64(r + 64, r + 64, z2);
}

#endif /* !WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_4096_add_64(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x11, %[a], 512\n\t"
        "\n1:\n\t"
        "adds	%[c], %[c], #-1\n\t"
        "ldp	x3, x4, [%[a]], #16\n\t"
        "ldp	x5, x6, [%[a]], #16\n\t"
        "ldp	x7, x8, [%[b]], #16\n\t"
        "ldp	x9, x10, [%[b]], #16\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r]], #16\n\t"
        "stp	x5, x6, [%[r]], #16\n\t"
        "cset	%[c], cs\n\t"
        "cmp	%[a], x11\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [r] "+r" (r), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_4096_sub_in_place_64(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x10, %[a], 512\n\t"
        "\n1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldp	x2, x3, [%[a]]\n\t"
        "ldp	x4, x5, [%[a], #16]\n\t"
        "ldp	x6, x7, [%[b]], #16\n\t"
        "ldp	x8, x9, [%[b]], #16\n\t"
        "sbcs	x2, x2, x6\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x2, x3, [%[a]], #16\n\t"
        "stp	x4, x5, [%[a]], #16\n\t"
        "csetm	%[c], cc\n\t"
        "cmp	%[a], x10\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_4096_mul_64(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[128];

    __asm__ __volatile__ (
        "mov	x5, 0\n\t"
        "mov	x6, 0\n\t"
        "mov	x7, 0\n\t"
        "mov	x8, 0\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 504\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[b], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 512\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 1008\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_4096_sqr_64(sp_digit* r, const sp_digit* a)
{
    sp_digit tmp[128];

    __asm__ __volatile__ (
        "mov	x6, 0\n\t"
        "mov	x7, 0\n\t"
        "mov	x8, 0\n\t"
        "mov	x5, 0\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 504\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "cmp	x4, x3\n\t"
        "b.eq	4f\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[a], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "b.al	5f\n\t"
        "\n4:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "mul	x9, x10, x10\n\t"
        "umulh	x10, x10, x10\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "\n5:\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 512\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x4\n\t"
        "b.gt	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 1008\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

#endif /* WOLFSSL_SP_SMALL */
/* Caclulate the bottom digit of -1/a mod 2^n.
 *
 * a    A single precision number.
 * rho  Bottom word of inverse.
 */
static void sp_4096_mont_setup(const sp_digit* a, sp_digit* rho)
{
    sp_digit x, b;

    b = a[0];
    x = (((b + 2) & 4) << 1) + b; /* here x*a==1 mod 2**4 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**8 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**16 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**32 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**64 */

    /* rho = -1/m mod b */
    *rho = -x;
}

/* Mul a by digit b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision digit.
 */
static void sp_4096_mul_d_64(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x5, %[b], x8\n\t"
        "umulh	x3, %[b], x8\n\t"
        "mov	x4, 0\n\t"
        "str	x5, [%[r]]\n\t"
        "mov	x5, 0\n\t"
        "mov	x9, #8\n\t"
        "1:\n\t"
        "ldr	x8, [%[a], x9]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "str	x3, [%[r], x9]\n\t"
        "mov	x3, x4\n\t"
        "mov	x4, x5\n\t"
        "mov	x5, #0\n\t"
        "add	x9, x9, #8\n\t"
        "cmp	x9, 512\n\t"
        "b.lt	1b\n\t"
        "str	x3, [%[r], 512]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8"
    );
#else
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x3, %[b], x8\n\t"
        "umulh	x4, %[b], x8\n\t"
        "mov	x5, 0\n\t"
        "str	x3, [%[r]]\n\t"
        "# A[1] * B\n\t"
        "ldr		x8, [%[a], 8]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 8]\n\t"
        "# A[2] * B\n\t"
        "ldr		x8, [%[a], 16]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 16]\n\t"
        "# A[3] * B\n\t"
        "ldr		x8, [%[a], 24]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 24]\n\t"
        "# A[4] * B\n\t"
        "ldr		x8, [%[a], 32]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 32]\n\t"
        "# A[5] * B\n\t"
        "ldr		x8, [%[a], 40]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 40]\n\t"
        "# A[6] * B\n\t"
        "ldr		x8, [%[a], 48]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 48]\n\t"
        "# A[7] * B\n\t"
        "ldr		x8, [%[a], 56]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 56]\n\t"
        "# A[8] * B\n\t"
        "ldr		x8, [%[a], 64]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 64]\n\t"
        "# A[9] * B\n\t"
        "ldr		x8, [%[a], 72]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 72]\n\t"
        "# A[10] * B\n\t"
        "ldr		x8, [%[a], 80]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 80]\n\t"
        "# A[11] * B\n\t"
        "ldr		x8, [%[a], 88]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 88]\n\t"
        "# A[12] * B\n\t"
        "ldr		x8, [%[a], 96]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 96]\n\t"
        "# A[13] * B\n\t"
        "ldr		x8, [%[a], 104]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 104]\n\t"
        "# A[14] * B\n\t"
        "ldr		x8, [%[a], 112]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 112]\n\t"
        "# A[15] * B\n\t"
        "ldr		x8, [%[a], 120]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 120]\n\t"
        "# A[16] * B\n\t"
        "ldr		x8, [%[a], 128]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 128]\n\t"
        "# A[17] * B\n\t"
        "ldr		x8, [%[a], 136]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 136]\n\t"
        "# A[18] * B\n\t"
        "ldr		x8, [%[a], 144]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 144]\n\t"
        "# A[19] * B\n\t"
        "ldr		x8, [%[a], 152]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 152]\n\t"
        "# A[20] * B\n\t"
        "ldr		x8, [%[a], 160]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 160]\n\t"
        "# A[21] * B\n\t"
        "ldr		x8, [%[a], 168]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 168]\n\t"
        "# A[22] * B\n\t"
        "ldr		x8, [%[a], 176]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 176]\n\t"
        "# A[23] * B\n\t"
        "ldr		x8, [%[a], 184]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 184]\n\t"
        "# A[24] * B\n\t"
        "ldr		x8, [%[a], 192]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 192]\n\t"
        "# A[25] * B\n\t"
        "ldr		x8, [%[a], 200]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 200]\n\t"
        "# A[26] * B\n\t"
        "ldr		x8, [%[a], 208]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 208]\n\t"
        "# A[27] * B\n\t"
        "ldr		x8, [%[a], 216]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 216]\n\t"
        "# A[28] * B\n\t"
        "ldr		x8, [%[a], 224]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 224]\n\t"
        "# A[29] * B\n\t"
        "ldr		x8, [%[a], 232]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 232]\n\t"
        "# A[30] * B\n\t"
        "ldr		x8, [%[a], 240]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 240]\n\t"
        "# A[31] * B\n\t"
        "ldr		x8, [%[a], 248]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 248]\n\t"
        "# A[32] * B\n\t"
        "ldr		x8, [%[a], 256]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 256]\n\t"
        "# A[33] * B\n\t"
        "ldr		x8, [%[a], 264]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 264]\n\t"
        "# A[34] * B\n\t"
        "ldr		x8, [%[a], 272]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 272]\n\t"
        "# A[35] * B\n\t"
        "ldr		x8, [%[a], 280]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 280]\n\t"
        "# A[36] * B\n\t"
        "ldr		x8, [%[a], 288]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 288]\n\t"
        "# A[37] * B\n\t"
        "ldr		x8, [%[a], 296]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 296]\n\t"
        "# A[38] * B\n\t"
        "ldr		x8, [%[a], 304]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 304]\n\t"
        "# A[39] * B\n\t"
        "ldr		x8, [%[a], 312]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 312]\n\t"
        "# A[40] * B\n\t"
        "ldr		x8, [%[a], 320]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 320]\n\t"
        "# A[41] * B\n\t"
        "ldr		x8, [%[a], 328]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 328]\n\t"
        "# A[42] * B\n\t"
        "ldr		x8, [%[a], 336]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 336]\n\t"
        "# A[43] * B\n\t"
        "ldr		x8, [%[a], 344]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 344]\n\t"
        "# A[44] * B\n\t"
        "ldr		x8, [%[a], 352]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 352]\n\t"
        "# A[45] * B\n\t"
        "ldr		x8, [%[a], 360]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 360]\n\t"
        "# A[46] * B\n\t"
        "ldr		x8, [%[a], 368]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 368]\n\t"
        "# A[47] * B\n\t"
        "ldr		x8, [%[a], 376]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 376]\n\t"
        "# A[48] * B\n\t"
        "ldr		x8, [%[a], 384]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 384]\n\t"
        "# A[49] * B\n\t"
        "ldr		x8, [%[a], 392]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 392]\n\t"
        "# A[50] * B\n\t"
        "ldr		x8, [%[a], 400]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 400]\n\t"
        "# A[51] * B\n\t"
        "ldr		x8, [%[a], 408]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 408]\n\t"
        "# A[52] * B\n\t"
        "ldr		x8, [%[a], 416]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 416]\n\t"
        "# A[53] * B\n\t"
        "ldr		x8, [%[a], 424]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 424]\n\t"
        "# A[54] * B\n\t"
        "ldr		x8, [%[a], 432]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 432]\n\t"
        "# A[55] * B\n\t"
        "ldr		x8, [%[a], 440]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 440]\n\t"
        "# A[56] * B\n\t"
        "ldr		x8, [%[a], 448]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 448]\n\t"
        "# A[57] * B\n\t"
        "ldr		x8, [%[a], 456]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 456]\n\t"
        "# A[58] * B\n\t"
        "ldr		x8, [%[a], 464]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 464]\n\t"
        "# A[59] * B\n\t"
        "ldr		x8, [%[a], 472]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 472]\n\t"
        "# A[60] * B\n\t"
        "ldr		x8, [%[a], 480]\n\t"
        "mov		x5, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc		x5, xzr, xzr\n\t"
        "str		x3, [%[r], 480]\n\t"
        "# A[61] * B\n\t"
        "ldr		x8, [%[a], 488]\n\t"
        "mov		x3, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "adc		x3, xzr, xzr\n\t"
        "str		x4, [%[r], 488]\n\t"
        "# A[62] * B\n\t"
        "ldr		x8, [%[a], 496]\n\t"
        "mov		x4, 0\n\t"
        "mul		x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "adc		x4, xzr, xzr\n\t"
        "str		x5, [%[r], 496]\n\t"
        "# A[63] * B\n\t"
        "ldr	x8, [%[a], 504]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x3, [%[r], 504]\n\t"
        "str	x4, [%[r], 512]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8"
    );
#endif
}

#if defined(WOLFSSL_HAVE_SP_RSA) || defined(WOLFSSL_HAVE_SP_DH)
/* r = 2^n mod m where n is the number of bits to reduce by.
 * Given m must be 4096 bits, just need to subtract.
 *
 * r  A single precision number.
 * m  A signle precision number.
 */
static void sp_4096_mont_norm_64(sp_digit* r, const sp_digit* m)
{
    XMEMSET(r, 0, sizeof(sp_digit) * 64);

    /* r = 2^n mod m */
    sp_4096_sub_in_place_64(r, m);
}

#endif /* WOLFSSL_HAVE_SP_RSA || WOLFSSL_HAVE_SP_DH */
/* Conditionally subtract b from a using the mask m.
 * m is -1 to subtract and 0 when not copying.
 *
 * r  A single precision number representing condition subtract result.
 * a  A single precision number to subtract from.
 * b  A single precision number to subtract.
 * m  Mask value to apply.
 */
static sp_digit sp_4096_cond_sub_64(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    sp_digit c = 0;

#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x8, #0\n\t"
        "1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldr	x4, [%[a], x8]\n\t"
        "ldr	x5, [%[b], x8]\n\t"
        "and	x5, x5, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "csetm	%[c], cc\n\t"
        "str	x4, [%[r], x8]\n\t"
        "add	x8, x8, #8\n\t"
        "cmp	x8, 512\n\t"
        "b.lt	1b\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8"
    );
#else
    __asm__ __volatile__ (

        "ldr		x4, [%[a], 0]\n\t"
        "ldr		x6, [%[a], 8]\n\t"
        "ldr		x5, [%[b], 0]\n\t"
        "ldr		x7, [%[b], 8]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "subs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 0]\n\t"
        "str		x6, [%[r], 8]\n\t"
        "ldr		x4, [%[a], 16]\n\t"
        "ldr		x6, [%[a], 24]\n\t"
        "ldr		x5, [%[b], 16]\n\t"
        "ldr		x7, [%[b], 24]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 16]\n\t"
        "str		x6, [%[r], 24]\n\t"
        "ldr		x4, [%[a], 32]\n\t"
        "ldr		x6, [%[a], 40]\n\t"
        "ldr		x5, [%[b], 32]\n\t"
        "ldr		x7, [%[b], 40]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 32]\n\t"
        "str		x6, [%[r], 40]\n\t"
        "ldr		x4, [%[a], 48]\n\t"
        "ldr		x6, [%[a], 56]\n\t"
        "ldr		x5, [%[b], 48]\n\t"
        "ldr		x7, [%[b], 56]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 48]\n\t"
        "str		x6, [%[r], 56]\n\t"
        "ldr		x4, [%[a], 64]\n\t"
        "ldr		x6, [%[a], 72]\n\t"
        "ldr		x5, [%[b], 64]\n\t"
        "ldr		x7, [%[b], 72]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 64]\n\t"
        "str		x6, [%[r], 72]\n\t"
        "ldr		x4, [%[a], 80]\n\t"
        "ldr		x6, [%[a], 88]\n\t"
        "ldr		x5, [%[b], 80]\n\t"
        "ldr		x7, [%[b], 88]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 80]\n\t"
        "str		x6, [%[r], 88]\n\t"
        "ldr		x4, [%[a], 96]\n\t"
        "ldr		x6, [%[a], 104]\n\t"
        "ldr		x5, [%[b], 96]\n\t"
        "ldr		x7, [%[b], 104]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 96]\n\t"
        "str		x6, [%[r], 104]\n\t"
        "ldr		x4, [%[a], 112]\n\t"
        "ldr		x6, [%[a], 120]\n\t"
        "ldr		x5, [%[b], 112]\n\t"
        "ldr		x7, [%[b], 120]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 112]\n\t"
        "str		x6, [%[r], 120]\n\t"
        "ldr		x4, [%[a], 128]\n\t"
        "ldr		x6, [%[a], 136]\n\t"
        "ldr		x5, [%[b], 128]\n\t"
        "ldr		x7, [%[b], 136]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 128]\n\t"
        "str		x6, [%[r], 136]\n\t"
        "ldr		x4, [%[a], 144]\n\t"
        "ldr		x6, [%[a], 152]\n\t"
        "ldr		x5, [%[b], 144]\n\t"
        "ldr		x7, [%[b], 152]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 144]\n\t"
        "str		x6, [%[r], 152]\n\t"
        "ldr		x4, [%[a], 160]\n\t"
        "ldr		x6, [%[a], 168]\n\t"
        "ldr		x5, [%[b], 160]\n\t"
        "ldr		x7, [%[b], 168]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 160]\n\t"
        "str		x6, [%[r], 168]\n\t"
        "ldr		x4, [%[a], 176]\n\t"
        "ldr		x6, [%[a], 184]\n\t"
        "ldr		x5, [%[b], 176]\n\t"
        "ldr		x7, [%[b], 184]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 176]\n\t"
        "str		x6, [%[r], 184]\n\t"
        "ldr		x4, [%[a], 192]\n\t"
        "ldr		x6, [%[a], 200]\n\t"
        "ldr		x5, [%[b], 192]\n\t"
        "ldr		x7, [%[b], 200]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 192]\n\t"
        "str		x6, [%[r], 200]\n\t"
        "ldr		x4, [%[a], 208]\n\t"
        "ldr		x6, [%[a], 216]\n\t"
        "ldr		x5, [%[b], 208]\n\t"
        "ldr		x7, [%[b], 216]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 208]\n\t"
        "str		x6, [%[r], 216]\n\t"
        "ldr		x4, [%[a], 224]\n\t"
        "ldr		x6, [%[a], 232]\n\t"
        "ldr		x5, [%[b], 224]\n\t"
        "ldr		x7, [%[b], 232]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 224]\n\t"
        "str		x6, [%[r], 232]\n\t"
        "ldr		x4, [%[a], 240]\n\t"
        "ldr		x6, [%[a], 248]\n\t"
        "ldr		x5, [%[b], 240]\n\t"
        "ldr		x7, [%[b], 248]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 240]\n\t"
        "str		x6, [%[r], 248]\n\t"
        "ldr		x4, [%[a], 256]\n\t"
        "ldr		x6, [%[a], 264]\n\t"
        "ldr		x5, [%[b], 256]\n\t"
        "ldr		x7, [%[b], 264]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 256]\n\t"
        "str		x6, [%[r], 264]\n\t"
        "ldr		x4, [%[a], 272]\n\t"
        "ldr		x6, [%[a], 280]\n\t"
        "ldr		x5, [%[b], 272]\n\t"
        "ldr		x7, [%[b], 280]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 272]\n\t"
        "str		x6, [%[r], 280]\n\t"
        "ldr		x4, [%[a], 288]\n\t"
        "ldr		x6, [%[a], 296]\n\t"
        "ldr		x5, [%[b], 288]\n\t"
        "ldr		x7, [%[b], 296]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 288]\n\t"
        "str		x6, [%[r], 296]\n\t"
        "ldr		x4, [%[a], 304]\n\t"
        "ldr		x6, [%[a], 312]\n\t"
        "ldr		x5, [%[b], 304]\n\t"
        "ldr		x7, [%[b], 312]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 304]\n\t"
        "str		x6, [%[r], 312]\n\t"
        "ldr		x4, [%[a], 320]\n\t"
        "ldr		x6, [%[a], 328]\n\t"
        "ldr		x5, [%[b], 320]\n\t"
        "ldr		x7, [%[b], 328]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 320]\n\t"
        "str		x6, [%[r], 328]\n\t"
        "ldr		x4, [%[a], 336]\n\t"
        "ldr		x6, [%[a], 344]\n\t"
        "ldr		x5, [%[b], 336]\n\t"
        "ldr		x7, [%[b], 344]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 336]\n\t"
        "str		x6, [%[r], 344]\n\t"
        "ldr		x4, [%[a], 352]\n\t"
        "ldr		x6, [%[a], 360]\n\t"
        "ldr		x5, [%[b], 352]\n\t"
        "ldr		x7, [%[b], 360]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 352]\n\t"
        "str		x6, [%[r], 360]\n\t"
        "ldr		x4, [%[a], 368]\n\t"
        "ldr		x6, [%[a], 376]\n\t"
        "ldr		x5, [%[b], 368]\n\t"
        "ldr		x7, [%[b], 376]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 368]\n\t"
        "str		x6, [%[r], 376]\n\t"
        "ldr		x4, [%[a], 384]\n\t"
        "ldr		x6, [%[a], 392]\n\t"
        "ldr		x5, [%[b], 384]\n\t"
        "ldr		x7, [%[b], 392]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 384]\n\t"
        "str		x6, [%[r], 392]\n\t"
        "ldr		x4, [%[a], 400]\n\t"
        "ldr		x6, [%[a], 408]\n\t"
        "ldr		x5, [%[b], 400]\n\t"
        "ldr		x7, [%[b], 408]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 400]\n\t"
        "str		x6, [%[r], 408]\n\t"
        "ldr		x4, [%[a], 416]\n\t"
        "ldr		x6, [%[a], 424]\n\t"
        "ldr		x5, [%[b], 416]\n\t"
        "ldr		x7, [%[b], 424]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 416]\n\t"
        "str		x6, [%[r], 424]\n\t"
        "ldr		x4, [%[a], 432]\n\t"
        "ldr		x6, [%[a], 440]\n\t"
        "ldr		x5, [%[b], 432]\n\t"
        "ldr		x7, [%[b], 440]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 432]\n\t"
        "str		x6, [%[r], 440]\n\t"
        "ldr		x4, [%[a], 448]\n\t"
        "ldr		x6, [%[a], 456]\n\t"
        "ldr		x5, [%[b], 448]\n\t"
        "ldr		x7, [%[b], 456]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 448]\n\t"
        "str		x6, [%[r], 456]\n\t"
        "ldr		x4, [%[a], 464]\n\t"
        "ldr		x6, [%[a], 472]\n\t"
        "ldr		x5, [%[b], 464]\n\t"
        "ldr		x7, [%[b], 472]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 464]\n\t"
        "str		x6, [%[r], 472]\n\t"
        "ldr		x4, [%[a], 480]\n\t"
        "ldr		x6, [%[a], 488]\n\t"
        "ldr		x5, [%[b], 480]\n\t"
        "ldr		x7, [%[b], 488]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 480]\n\t"
        "str		x6, [%[r], 488]\n\t"
        "ldr		x4, [%[a], 496]\n\t"
        "ldr		x6, [%[a], 504]\n\t"
        "ldr		x5, [%[b], 496]\n\t"
        "ldr		x7, [%[b], 504]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 496]\n\t"
        "str		x6, [%[r], 504]\n\t"
        "csetm	%[c], cc\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8"
    );
#endif /* WOLFSSL_SP_SMALL */

    return c;
}

/* Reduce the number back to 4096 bits using Montgomery reduction.
 *
 * a   A single precision number to reduce in place.
 * m   The single precision number representing the modulus.
 * mp  The digit representing the negative inverse of m mod 2^n.
 */
SP_NOINLINE static void sp_4096_mont_reduce_64(sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    sp_digit ca = 0;

    __asm__ __volatile__ (
        "ldp       x12, x13, [%[m], 0]\n\t"
        "ldp       x14, x15, [%[m], 16]\n\t"
        "ldp       x16, x17, [%[m], 32]\n\t"
        "ldp       x19, x20, [%[m], 48]\n\t"
        "ldp       x21, x22, [%[m], 64]\n\t"
        "ldp       x23, x24, [%[m], 80]\n\t"
        "ldp       x25, x26, [%[m], 96]\n\t"
        "ldp       x27, x28, [%[m], 112]\n\t"
        "# i = 0\n\t"
        "mov	x3, 0\n\t"
        "ldp	x10, x11, [%[a], 0]\n\t"
        "\n1:\n\t"
        "# mu = a[i] * mp\n\t"
        "mul	x8, %[mp], x10\n\t"
        "# a[i+0] += m[0] * mu\n\t"
        "ldr	x9, [%[a], 0]\n\t"
        "mul		x6, x12, x8\n\t"
        "umulh	x7, x12, x8\n\t"
        "adds	x10, x10, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "# a[i+1] += m[1] * mu\n\t"
        "ldr	x9, [%[a], 8]\n\t"
        "mul		x6, x13, x8\n\t"
        "umulh	x7, x13, x8\n\t"
        "adds	x10, x11, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x10, x10, x5\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+2] += m[2] * mu\n\t"
        "ldr	x11, [%[a], 16]\n\t"
        "mul		x6, x14, x8\n\t"
        "umulh	x7, x14, x8\n\t"
        "adds	x11, x11, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x11, x11, x4\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+3] += m[3] * mu\n\t"
        "ldr	x9, [%[a], 24]\n\t"
        "mul		x6, x15, x8\n\t"
        "umulh	x7, x15, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 24]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+4] += m[4] * mu\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "mul		x6, x16, x8\n\t"
        "umulh	x7, x16, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 32]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+5] += m[5] * mu\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "mul		x6, x17, x8\n\t"
        "umulh	x7, x17, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 40]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+6] += m[6] * mu\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "mul		x6, x19, x8\n\t"
        "umulh	x7, x19, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 48]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+7] += m[7] * mu\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "mul		x6, x20, x8\n\t"
        "umulh	x7, x20, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 56]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+8] += m[8] * mu\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "mul		x6, x21, x8\n\t"
        "umulh	x7, x21, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 64]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+9] += m[9] * mu\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "mul		x6, x22, x8\n\t"
        "umulh	x7, x22, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 72]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+10] += m[10] * mu\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "mul		x6, x23, x8\n\t"
        "umulh	x7, x23, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 80]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+11] += m[11] * mu\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "mul		x6, x24, x8\n\t"
        "umulh	x7, x24, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 88]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+12] += m[12] * mu\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "mul		x6, x25, x8\n\t"
        "umulh	x7, x25, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 96]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+13] += m[13] * mu\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "mul		x6, x26, x8\n\t"
        "umulh	x7, x26, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 104]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+14] += m[14] * mu\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "mul		x6, x27, x8\n\t"
        "umulh	x7, x27, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 112]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+15] += m[15] * mu\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "mul		x6, x28, x8\n\t"
        "umulh	x7, x28, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 120]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+16] += m[16] * mu\n\t"
        "ldr		x7, [%[m], 128]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 128]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+17] += m[17] * mu\n\t"
        "ldr		x7, [%[m], 136]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 136]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+18] += m[18] * mu\n\t"
        "ldr		x7, [%[m], 144]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 144]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+19] += m[19] * mu\n\t"
        "ldr		x7, [%[m], 152]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 152]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+20] += m[20] * mu\n\t"
        "ldr		x7, [%[m], 160]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 160]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+21] += m[21] * mu\n\t"
        "ldr		x7, [%[m], 168]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 168]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+22] += m[22] * mu\n\t"
        "ldr		x7, [%[m], 176]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 176]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+23] += m[23] * mu\n\t"
        "ldr		x7, [%[m], 184]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 184]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+24] += m[24] * mu\n\t"
        "ldr		x7, [%[m], 192]\n\t"
        "ldr	x9, [%[a], 192]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 192]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+25] += m[25] * mu\n\t"
        "ldr		x7, [%[m], 200]\n\t"
        "ldr	x9, [%[a], 200]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 200]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+26] += m[26] * mu\n\t"
        "ldr		x7, [%[m], 208]\n\t"
        "ldr	x9, [%[a], 208]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 208]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+27] += m[27] * mu\n\t"
        "ldr		x7, [%[m], 216]\n\t"
        "ldr	x9, [%[a], 216]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 216]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+28] += m[28] * mu\n\t"
        "ldr		x7, [%[m], 224]\n\t"
        "ldr	x9, [%[a], 224]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 224]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+29] += m[29] * mu\n\t"
        "ldr		x7, [%[m], 232]\n\t"
        "ldr	x9, [%[a], 232]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 232]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+30] += m[30] * mu\n\t"
        "ldr		x7, [%[m], 240]\n\t"
        "ldr	x9, [%[a], 240]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 240]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+31] += m[31] * mu\n\t"
        "ldr		x7, [%[m], 248]\n\t"
        "ldr	x9, [%[a], 248]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 248]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+32] += m[32] * mu\n\t"
        "ldr		x7, [%[m], 256]\n\t"
        "ldr	x9, [%[a], 256]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 256]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+33] += m[33] * mu\n\t"
        "ldr		x7, [%[m], 264]\n\t"
        "ldr	x9, [%[a], 264]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 264]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+34] += m[34] * mu\n\t"
        "ldr		x7, [%[m], 272]\n\t"
        "ldr	x9, [%[a], 272]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 272]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+35] += m[35] * mu\n\t"
        "ldr		x7, [%[m], 280]\n\t"
        "ldr	x9, [%[a], 280]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 280]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+36] += m[36] * mu\n\t"
        "ldr		x7, [%[m], 288]\n\t"
        "ldr	x9, [%[a], 288]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 288]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+37] += m[37] * mu\n\t"
        "ldr		x7, [%[m], 296]\n\t"
        "ldr	x9, [%[a], 296]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 296]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+38] += m[38] * mu\n\t"
        "ldr		x7, [%[m], 304]\n\t"
        "ldr	x9, [%[a], 304]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 304]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+39] += m[39] * mu\n\t"
        "ldr		x7, [%[m], 312]\n\t"
        "ldr	x9, [%[a], 312]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 312]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+40] += m[40] * mu\n\t"
        "ldr		x7, [%[m], 320]\n\t"
        "ldr	x9, [%[a], 320]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 320]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+41] += m[41] * mu\n\t"
        "ldr		x7, [%[m], 328]\n\t"
        "ldr	x9, [%[a], 328]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 328]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+42] += m[42] * mu\n\t"
        "ldr		x7, [%[m], 336]\n\t"
        "ldr	x9, [%[a], 336]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 336]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+43] += m[43] * mu\n\t"
        "ldr		x7, [%[m], 344]\n\t"
        "ldr	x9, [%[a], 344]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 344]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+44] += m[44] * mu\n\t"
        "ldr		x7, [%[m], 352]\n\t"
        "ldr	x9, [%[a], 352]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 352]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+45] += m[45] * mu\n\t"
        "ldr		x7, [%[m], 360]\n\t"
        "ldr	x9, [%[a], 360]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 360]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+46] += m[46] * mu\n\t"
        "ldr		x7, [%[m], 368]\n\t"
        "ldr	x9, [%[a], 368]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 368]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+47] += m[47] * mu\n\t"
        "ldr		x7, [%[m], 376]\n\t"
        "ldr	x9, [%[a], 376]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 376]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+48] += m[48] * mu\n\t"
        "ldr		x7, [%[m], 384]\n\t"
        "ldr	x9, [%[a], 384]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 384]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+49] += m[49] * mu\n\t"
        "ldr		x7, [%[m], 392]\n\t"
        "ldr	x9, [%[a], 392]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 392]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+50] += m[50] * mu\n\t"
        "ldr		x7, [%[m], 400]\n\t"
        "ldr	x9, [%[a], 400]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 400]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+51] += m[51] * mu\n\t"
        "ldr		x7, [%[m], 408]\n\t"
        "ldr	x9, [%[a], 408]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 408]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+52] += m[52] * mu\n\t"
        "ldr		x7, [%[m], 416]\n\t"
        "ldr	x9, [%[a], 416]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 416]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+53] += m[53] * mu\n\t"
        "ldr		x7, [%[m], 424]\n\t"
        "ldr	x9, [%[a], 424]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 424]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+54] += m[54] * mu\n\t"
        "ldr		x7, [%[m], 432]\n\t"
        "ldr	x9, [%[a], 432]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 432]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+55] += m[55] * mu\n\t"
        "ldr		x7, [%[m], 440]\n\t"
        "ldr	x9, [%[a], 440]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 440]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+56] += m[56] * mu\n\t"
        "ldr		x7, [%[m], 448]\n\t"
        "ldr	x9, [%[a], 448]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 448]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+57] += m[57] * mu\n\t"
        "ldr		x7, [%[m], 456]\n\t"
        "ldr	x9, [%[a], 456]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 456]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+58] += m[58] * mu\n\t"
        "ldr		x7, [%[m], 464]\n\t"
        "ldr	x9, [%[a], 464]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 464]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+59] += m[59] * mu\n\t"
        "ldr		x7, [%[m], 472]\n\t"
        "ldr	x9, [%[a], 472]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 472]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+60] += m[60] * mu\n\t"
        "ldr		x7, [%[m], 480]\n\t"
        "ldr	x9, [%[a], 480]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 480]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+61] += m[61] * mu\n\t"
        "ldr		x7, [%[m], 488]\n\t"
        "ldr	x9, [%[a], 488]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x4, x7, xzr\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 488]\n\t"
        "adc	x4, x4, xzr\n\t"
        "# a[i+62] += m[62] * mu\n\t"
        "ldr		x7, [%[m], 496]\n\t"
        "ldr	x9, [%[a], 496]\n\t"
        "mul		x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x9, x9, x6\n\t"
        "adc	x5, x7, xzr\n\t"
        "adds	x9, x9, x4\n\t"
        "str	x9, [%[a], 496]\n\t"
        "adc	x5, x5, xzr\n\t"
        "# a[i+63] += m[63] * mu\n\t"
        "ldr	x7, [%[m], 504]\n\t"
        "ldr	x9, [%[a], 504]\n\t"
        "mul	x6, x7, x8\n\t"
        "umulh	x7, x7, x8\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x7, x7, %[ca]\n\t"
        "cset  %[ca], cs\n\t"
        "adds	x9, x9, x5\n\t"
        "str	x9, [%[a], 504]\n\t"
        "ldr	x9, [%[a], 512]\n\t"
        "adcs	x9, x9, x7\n\t"
        "str	x9, [%[a], 512]\n\t"
        "adc	%[ca], %[ca], xzr\n\t"
        "# i += 1\n\t"
        "add	%[a], %[a], 8\n\t"
        "add	x3, x3, 8\n\t"
        "cmp	x3, 512\n\t"
        "blt	1b\n\t"
        "str	x10, [%[a], 0]\n\t"
        "str	x11, [%[a], 8]\n\t"
        : [ca] "+r" (ca), [a] "+r" (a)
        : [m] "r" (m), [mp] "r" (mp)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );

    sp_4096_cond_sub_64(a - 64, a, m, (sp_digit)0 - ca);
}

/* Multiply two Montogmery form numbers mod the modulus (prime).
 * (r = a * b mod m)
 *
 * r   Result of multiplication.
 * a   First number to multiply in Montogmery form.
 * b   Second number to multiply in Montogmery form.
 * m   Modulus (prime).
 * mp  Montogmery mulitplier.
 */
static void sp_4096_mont_mul_64(sp_digit* r, const sp_digit* a, const sp_digit* b,
        const sp_digit* m, sp_digit mp)
{
    sp_4096_mul_64(r, a, b);
    sp_4096_mont_reduce_64(r, m, mp);
}

/* Square the Montgomery form number. (r = a * a mod m)
 *
 * r   Result of squaring.
 * a   Number to square in Montogmery form.
 * m   Modulus (prime).
 * mp  Montogmery mulitplier.
 */
static void sp_4096_mont_sqr_64(sp_digit* r, const sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    sp_4096_sqr_64(r, a);
    sp_4096_mont_reduce_64(r, m, mp);
}

/* Divide the double width number (d1|d0) by the dividend. (d1|d0 / div)
 *
 * d1   The high order half of the number to divide.
 * d0   The low order half of the number to divide.
 * div  The dividend.
 * returns the result of the division.
 */
static sp_digit div_4096_word_64(sp_digit d1, sp_digit d0, sp_digit div)
{
    sp_digit r;

    __asm__ __volatile__ (
        "lsr	x5, %[div], 32\n\t"
        "add	x5, x5, 1\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x6, x3, 32\n\t"
        "mul	x4, %[div], x6\n\t"
        "umulh	x3, %[div], x6\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x3, x3, 32\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "lsl	x3, %[d1], 32\n\t"
        "orr	x3, x3, %[d0], lsr 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "lsl	x3, %[d1], 32\n\t"
        "orr	x3, x3, %[d0], lsr 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "udiv	x3, %[d0], %[div]\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x3, %[div], x3\n\t"
        "sub	%[d0], %[d0], x3\n\t"
        "mov	%[r], x6\n\t"

        : [r] "=r" (r)
        : [d1] "r" (d1), [d0] "r" (d0), [div] "r" (div)
        : "x3", "x4", "x5", "x6"
    );

    return r;
}

/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_4096_mask_64(sp_digit* r, const sp_digit* a, sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    int i;

    for (i=0; i<64; i++) {
        r[i] = a[i] & m;
    }
#else
    int i;

    for (i = 0; i < 64; i += 8) {
        r[i+0] = a[i+0] & m;
        r[i+1] = a[i+1] & m;
        r[i+2] = a[i+2] & m;
        r[i+3] = a[i+3] & m;
        r[i+4] = a[i+4] & m;
        r[i+5] = a[i+5] & m;
        r[i+6] = a[i+6] & m;
        r[i+7] = a[i+7] & m;
    }
#endif
}

/* Compare a with b in constant time.
 *
 * a  A single precision integer.
 * b  A single precision integer.
 * return -ve, 0 or +ve if a is less than, equal to or greater than b
 * respectively.
 */
static int64_t sp_4096_cmp_64(const sp_digit* a, const sp_digit* b)
{
    sp_digit r = -1;
    sp_digit one = 1;

#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x3, -1\n\t"
        "mov	x6, 504\n\t"
        "1:\n\t"
        "ldr	x4, [%[a], x6]\n\t"
        "ldr	x5, [%[b], x6]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "subs	x6, x6, #8\n\t"
        "b.cs	1b\n\t"
        "eor	%[r], %[r], x3\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [one] "r" (one)
        : "x2", "x3", "x4", "x5", "x6"
    );
#else
    __asm__ __volatile__ (
        "mov	x3, -1\n\t"
        "ldr		x4, [%[a], 504]\n\t"
        "ldr		x5, [%[b], 504]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 496]\n\t"
        "ldr		x5, [%[b], 496]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 488]\n\t"
        "ldr		x5, [%[b], 488]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 480]\n\t"
        "ldr		x5, [%[b], 480]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 472]\n\t"
        "ldr		x5, [%[b], 472]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 464]\n\t"
        "ldr		x5, [%[b], 464]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 456]\n\t"
        "ldr		x5, [%[b], 456]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 448]\n\t"
        "ldr		x5, [%[b], 448]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 440]\n\t"
        "ldr		x5, [%[b], 440]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 432]\n\t"
        "ldr		x5, [%[b], 432]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 424]\n\t"
        "ldr		x5, [%[b], 424]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 416]\n\t"
        "ldr		x5, [%[b], 416]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 408]\n\t"
        "ldr		x5, [%[b], 408]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 400]\n\t"
        "ldr		x5, [%[b], 400]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 392]\n\t"
        "ldr		x5, [%[b], 392]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 384]\n\t"
        "ldr		x5, [%[b], 384]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 376]\n\t"
        "ldr		x5, [%[b], 376]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 368]\n\t"
        "ldr		x5, [%[b], 368]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 360]\n\t"
        "ldr		x5, [%[b], 360]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 352]\n\t"
        "ldr		x5, [%[b], 352]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 344]\n\t"
        "ldr		x5, [%[b], 344]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 336]\n\t"
        "ldr		x5, [%[b], 336]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 328]\n\t"
        "ldr		x5, [%[b], 328]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 320]\n\t"
        "ldr		x5, [%[b], 320]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 312]\n\t"
        "ldr		x5, [%[b], 312]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 304]\n\t"
        "ldr		x5, [%[b], 304]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 296]\n\t"
        "ldr		x5, [%[b], 296]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 288]\n\t"
        "ldr		x5, [%[b], 288]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 280]\n\t"
        "ldr		x5, [%[b], 280]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 272]\n\t"
        "ldr		x5, [%[b], 272]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 264]\n\t"
        "ldr		x5, [%[b], 264]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 256]\n\t"
        "ldr		x5, [%[b], 256]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 248]\n\t"
        "ldr		x5, [%[b], 248]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 240]\n\t"
        "ldr		x5, [%[b], 240]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 232]\n\t"
        "ldr		x5, [%[b], 232]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 224]\n\t"
        "ldr		x5, [%[b], 224]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 216]\n\t"
        "ldr		x5, [%[b], 216]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 208]\n\t"
        "ldr		x5, [%[b], 208]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 200]\n\t"
        "ldr		x5, [%[b], 200]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 192]\n\t"
        "ldr		x5, [%[b], 192]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 184]\n\t"
        "ldr		x5, [%[b], 184]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 176]\n\t"
        "ldr		x5, [%[b], 176]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 168]\n\t"
        "ldr		x5, [%[b], 168]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 160]\n\t"
        "ldr		x5, [%[b], 160]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 152]\n\t"
        "ldr		x5, [%[b], 152]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 144]\n\t"
        "ldr		x5, [%[b], 144]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 136]\n\t"
        "ldr		x5, [%[b], 136]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 128]\n\t"
        "ldr		x5, [%[b], 128]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 120]\n\t"
        "ldr		x5, [%[b], 120]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 112]\n\t"
        "ldr		x5, [%[b], 112]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 104]\n\t"
        "ldr		x5, [%[b], 104]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 96]\n\t"
        "ldr		x5, [%[b], 96]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 88]\n\t"
        "ldr		x5, [%[b], 88]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 80]\n\t"
        "ldr		x5, [%[b], 80]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 72]\n\t"
        "ldr		x5, [%[b], 72]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 64]\n\t"
        "ldr		x5, [%[b], 64]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 56]\n\t"
        "ldr		x5, [%[b], 56]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 48]\n\t"
        "ldr		x5, [%[b], 48]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 40]\n\t"
        "ldr		x5, [%[b], 40]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 32]\n\t"
        "ldr		x5, [%[b], 32]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 24]\n\t"
        "ldr		x5, [%[b], 24]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 16]\n\t"
        "ldr		x5, [%[b], 16]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 8]\n\t"
        "ldr		x5, [%[b], 8]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 0]\n\t"
        "ldr		x5, [%[b], 0]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "eor	%[r], %[r], x3\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [one] "r" (one)
        : "x2", "x3", "x4", "x5", "x6"
    );
#endif

    return r;
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Nmber to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_4096_div_64(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[128], t2[65];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[63];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 64);
    for (i=63; i>=0; i--) {
        r1 = div_4096_word_64(t1[64 + i], t1[64 + i - 1], div);

        sp_4096_mul_d_64(t2, d, r1);
        t1[64 + i] += sp_4096_sub_in_place_64(&t1[i], t2);
        t1[64 + i] -= t2[64];
        sp_4096_mask_64(t2, d, t1[64 + i]);
        t1[64 + i] += sp_4096_add_64(&t1[i], &t1[i], t2);
        sp_4096_mask_64(t2, d, t1[64 + i]);
        t1[64 + i] += sp_4096_add_64(&t1[i], &t1[i], t2);
    }

    r1 = sp_4096_cmp_64(t1, d) >= 0;
    sp_4096_cond_sub_64(r, t1, d, (sp_digit)0 - r1);

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_4096_mod_64(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_4096_div_64(a, m, NULL, r);
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Nmber to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_4096_div_64_cond(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[128], t2[65];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[63];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 64);
    for (i=63; i>=0; i--) {
        r1 = div_4096_word_64(t1[64 + i], t1[64 + i - 1], div);

        sp_4096_mul_d_64(t2, d, r1);
        t1[64 + i] += sp_4096_sub_in_place_64(&t1[i], t2);
        t1[64 + i] -= t2[64];
        if (t1[64 + i] != 0) {
            t1[64 + i] += sp_4096_add_64(&t1[i], &t1[i], d);
            if (t1[64 + i] != 0)
                t1[64 + i] += sp_4096_add_64(&t1[i], &t1[i], d);
        }
    }

    r1 = sp_4096_cmp_64(t1, d) >= 0;
    sp_4096_cond_sub_64(r, t1, d, (sp_digit)0 - r1);

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_4096_mod_64_cond(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_4096_div_64_cond(a, m, NULL, r);
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || \
                                                     defined(WOLFSSL_HAVE_SP_DH)
#ifdef WOLFSSL_SP_SMALL
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns 0 on success and MEMORY_E on dynamic memory allocation failure.
 */
static int sp_4096_mod_exp_64(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#ifndef WOLFSSL_SMALL_STACK
    sp_digit t[16][128];
#else
    sp_digit* t[16];
    sp_digit* td;
#endif
    sp_digit* norm;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c, y;
    int err = MP_OKAY;

#ifdef WOLFSSL_SMALL_STACK
    td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 16 * 128, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER);
    if (td == NULL) {
        err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
#ifdef WOLFSSL_SMALL_STACK
        for (i=0; i<16; i++) {
            t[i] = td + i * 128;
        }
#endif
        norm = t[0];

        sp_4096_mont_setup(m, &mp);
        sp_4096_mont_norm_64(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 64U);
        if (reduceA != 0) {
            err = sp_4096_mod_64(t[1] + 64, a, m);
            if (err == MP_OKAY) {
                err = sp_4096_mod_64(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 64, a, sizeof(sp_digit) * 64);
            err = sp_4096_mod_64(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_4096_mont_sqr_64(t[ 2], t[ 1], m, mp);
        sp_4096_mont_mul_64(t[ 3], t[ 2], t[ 1], m, mp);
        sp_4096_mont_sqr_64(t[ 4], t[ 2], m, mp);
        sp_4096_mont_mul_64(t[ 5], t[ 3], t[ 2], m, mp);
        sp_4096_mont_sqr_64(t[ 6], t[ 3], m, mp);
        sp_4096_mont_mul_64(t[ 7], t[ 4], t[ 3], m, mp);
        sp_4096_mont_sqr_64(t[ 8], t[ 4], m, mp);
        sp_4096_mont_mul_64(t[ 9], t[ 5], t[ 4], m, mp);
        sp_4096_mont_sqr_64(t[10], t[ 5], m, mp);
        sp_4096_mont_mul_64(t[11], t[ 6], t[ 5], m, mp);
        sp_4096_mont_sqr_64(t[12], t[ 6], m, mp);
        sp_4096_mont_mul_64(t[13], t[ 7], t[ 6], m, mp);
        sp_4096_mont_sqr_64(t[14], t[ 7], m, mp);
        sp_4096_mont_mul_64(t[15], t[ 8], t[ 7], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 4;
        if (c == 64) {
            c = 60;
        }
        y = (int)(n >> c);
        n <<= 64 - c;
        XMEMCPY(r, t[y], sizeof(sp_digit) * 64);
        for (; i>=0 || c>=4; ) {
            if (c == 0) {
                n = e[i--];
                y = n >> 60;
                n <<= 4;
                c = 60;
            }
            else if (c < 4) {
                y = n >> 60;
                n = e[i--];
                c = 4 - c;
                y |= n >> (64 - c);
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (n >> 60) & 0xf;
                n <<= 4;
                c -= 4;
            }

            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);

            sp_4096_mont_mul_64(r, r, t[y], m, mp);
        }

        XMEMSET(&r[64], 0, sizeof(sp_digit) * 64U);
        sp_4096_mont_reduce_64(r, m, mp);

        mask = 0 - (sp_4096_cmp_64(r, m) >= 0);
        sp_4096_cond_sub_64(r, r, m, mask);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (td != NULL) {
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return err;
}
#else
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns 0 on success and MEMORY_E on dynamic memory allocation failure.
 */
static int sp_4096_mod_exp_64(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#ifndef WOLFSSL_SMALL_STACK
    sp_digit t[32][128];
#else
    sp_digit* t[32];
    sp_digit* td;
#endif
    sp_digit* norm;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c, y;
    int err = MP_OKAY;

#ifdef WOLFSSL_SMALL_STACK
    td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 32 * 128, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER);
    if (td == NULL) {
        err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
#ifdef WOLFSSL_SMALL_STACK
        for (i=0; i<32; i++) {
            t[i] = td + i * 128;
        }
#endif
        norm = t[0];

        sp_4096_mont_setup(m, &mp);
        sp_4096_mont_norm_64(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 64U);
        if (reduceA != 0) {
            err = sp_4096_mod_64(t[1] + 64, a, m);
            if (err == MP_OKAY) {
                err = sp_4096_mod_64(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 64, a, sizeof(sp_digit) * 64);
            err = sp_4096_mod_64(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_4096_mont_sqr_64(t[ 2], t[ 1], m, mp);
        sp_4096_mont_mul_64(t[ 3], t[ 2], t[ 1], m, mp);
        sp_4096_mont_sqr_64(t[ 4], t[ 2], m, mp);
        sp_4096_mont_mul_64(t[ 5], t[ 3], t[ 2], m, mp);
        sp_4096_mont_sqr_64(t[ 6], t[ 3], m, mp);
        sp_4096_mont_mul_64(t[ 7], t[ 4], t[ 3], m, mp);
        sp_4096_mont_sqr_64(t[ 8], t[ 4], m, mp);
        sp_4096_mont_mul_64(t[ 9], t[ 5], t[ 4], m, mp);
        sp_4096_mont_sqr_64(t[10], t[ 5], m, mp);
        sp_4096_mont_mul_64(t[11], t[ 6], t[ 5], m, mp);
        sp_4096_mont_sqr_64(t[12], t[ 6], m, mp);
        sp_4096_mont_mul_64(t[13], t[ 7], t[ 6], m, mp);
        sp_4096_mont_sqr_64(t[14], t[ 7], m, mp);
        sp_4096_mont_mul_64(t[15], t[ 8], t[ 7], m, mp);
        sp_4096_mont_sqr_64(t[16], t[ 8], m, mp);
        sp_4096_mont_mul_64(t[17], t[ 9], t[ 8], m, mp);
        sp_4096_mont_sqr_64(t[18], t[ 9], m, mp);
        sp_4096_mont_mul_64(t[19], t[10], t[ 9], m, mp);
        sp_4096_mont_sqr_64(t[20], t[10], m, mp);
        sp_4096_mont_mul_64(t[21], t[11], t[10], m, mp);
        sp_4096_mont_sqr_64(t[22], t[11], m, mp);
        sp_4096_mont_mul_64(t[23], t[12], t[11], m, mp);
        sp_4096_mont_sqr_64(t[24], t[12], m, mp);
        sp_4096_mont_mul_64(t[25], t[13], t[12], m, mp);
        sp_4096_mont_sqr_64(t[26], t[13], m, mp);
        sp_4096_mont_mul_64(t[27], t[14], t[13], m, mp);
        sp_4096_mont_sqr_64(t[28], t[14], m, mp);
        sp_4096_mont_mul_64(t[29], t[15], t[14], m, mp);
        sp_4096_mont_sqr_64(t[30], t[15], m, mp);
        sp_4096_mont_mul_64(t[31], t[16], t[15], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 5;
        if (c == 64) {
            c = 59;
        }
        y = (int)(n >> c);
        n <<= 64 - c;
        XMEMCPY(r, t[y], sizeof(sp_digit) * 64);
        for (; i>=0 || c>=5; ) {
            if (c == 0) {
                n = e[i--];
                y = n >> 59;
                n <<= 5;
                c = 59;
            }
            else if (c < 5) {
                y = n >> 59;
                n = e[i--];
                c = 5 - c;
                y |= n >> (64 - c);
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (n >> 59) & 0x1f;
                n <<= 5;
                c -= 5;
            }

            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);

            sp_4096_mont_mul_64(r, r, t[y], m, mp);
        }

        XMEMSET(&r[64], 0, sizeof(sp_digit) * 64U);
        sp_4096_mont_reduce_64(r, m, mp);

        mask = 0 - (sp_4096_cmp_64(r, m) >= 0);
        sp_4096_cond_sub_64(r, r, m, mask);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (td != NULL) {
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return err;
}
#endif /* WOLFSSL_SP_SMALL */
#endif /* (WOLFSSL_HAVE_SP_RSA && !WOLFSSL_RSA_PUBLIC_ONLY) || WOLFSSL_HAVE_SP_DH */

#ifdef WOLFSSL_HAVE_SP_RSA
/* RSA public key operation.
 *
 * in      Array of bytes representing the number to exponentiate, base.
 * inLen   Number of bytes in base.
 * em      Public exponent.
 * mm      Modulus.
 * out     Buffer to hold big-endian bytes of exponentiation result.
 *         Must be at least 512 bytes long.
 * outLen  Number of bytes in result.
 * returns 0 on success, MP_TO_E when the outLen is too small, MP_READ_E when
 * an array is too long and MEMORY_E when dynamic memory allocation fails.
 */
int sp_RsaPublic_4096(const byte* in, word32 inLen, mp_int* em, mp_int* mm,
    byte* out, word32* outLen)
{
#if !defined(WOLFSSL_SP_SMALL) && !defined(WOLFSSL_SMALL_STACK)
    sp_digit ad[128], md[64], rd[128];
#else
    sp_digit* d = NULL;
#endif
    sp_digit* a;
    sp_digit *ah;
    sp_digit* m;
    sp_digit* r;
    sp_digit e[1];
    int err = MP_OKAY;

    if (*outLen < 512)
        err = MP_TO_E;
    if (err == MP_OKAY && (mp_count_bits(em) > 64 || inLen > 512 ||
                                                     mp_count_bits(mm) != 4096))
        err = MP_READ_E;

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    if (err == MP_OKAY) {
        d = (sp_digit*)XMALLOC(sizeof(sp_digit) * 64 * 5, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (d == NULL)
            err = MEMORY_E;
    }

    if (err == MP_OKAY) {
        a = d;
        r = a + 64 * 2;
        m = r + 64 * 2;
        ah = a + 64;
    }
#else
    a = ad;
    m = md;
    r = rd;
    ah = a + 64;
#endif

    if (err == MP_OKAY) {
        sp_4096_from_bin(ah, 64, in, inLen);
#if DIGIT_BIT >= 64
        e[0] = em->dp[0];
#else
        e[0] = em->dp[0];
        if (em->used > 1)
            e[0] |= ((sp_digit)em->dp[1]) << DIGIT_BIT;
#endif
        if (e[0] == 0)
            err = MP_EXPTMOD_E;
    }
    if (err == MP_OKAY) {
        sp_4096_from_mp(m, 64, mm);

        if (e[0] == 0x3) {
            if (err == MP_OKAY) {
                sp_4096_sqr_64(r, ah);
                err = sp_4096_mod_64_cond(r, r, m);
            }
            if (err == MP_OKAY) {
                sp_4096_mul_64(r, ah, r);
                err = sp_4096_mod_64_cond(r, r, m);
            }
        }
        else {
            int i;
            sp_digit mp;

            sp_4096_mont_setup(m, &mp);

            /* Convert to Montgomery form. */
            XMEMSET(a, 0, sizeof(sp_digit) * 64);
            err = sp_4096_mod_64_cond(a, a, m);

            if (err == MP_OKAY) {
                for (i=63; i>=0; i--)
                    if (e[0] >> i)
                        break;

                XMEMCPY(r, a, sizeof(sp_digit) * 64);
                for (i--; i>=0; i--) {
                    sp_4096_mont_sqr_64(r, r, m, mp);
                    if (((e[0] >> i) & 1) == 1)
                        sp_4096_mont_mul_64(r, r, a, m, mp);
                }
                XMEMSET(&r[64], 0, sizeof(sp_digit) * 64);
                sp_4096_mont_reduce_64(r, m, mp);

                for (i = 63; i > 0; i--) {
                    if (r[i] != m[i])
                        break;
                }
                if (r[i] >= m[i])
                    sp_4096_sub_in_place_64(r, m);
            }
        }
    }

    if (err == MP_OKAY) {
        sp_4096_to_bin(r, out);
        *outLen = 512;
    }

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    if (d != NULL)
        XFREE(d, NULL, DYNAMIC_TYPE_RSA);
#endif

    return err;
}

/* RSA private key operation.
 *
 * in      Array of bytes representing the number to exponentiate, base.
 * inLen   Number of bytes in base.
 * dm      Private exponent.
 * pm      First prime.
 * qm      Second prime.
 * dpm     First prime's CRT exponent.
 * dqm     Second prime's CRT exponent.
 * qim     Inverse of second prime mod p.
 * mm      Modulus.
 * out     Buffer to hold big-endian bytes of exponentiation result.
 *         Must be at least 512 bytes long.
 * outLen  Number of bytes in result.
 * returns 0 on success, MP_TO_E when the outLen is too small, MP_READ_E when
 * an array is too long and MEMORY_E when dynamic memory allocation fails.
 */
int sp_RsaPrivate_4096(const byte* in, word32 inLen, mp_int* dm,
    mp_int* pm, mp_int* qm, mp_int* dpm, mp_int* dqm, mp_int* qim, mp_int* mm,
    byte* out, word32* outLen)
{
#if !defined(WOLFSSL_SP_SMALL) && !defined(WOLFSSL_SMALL_STACK)
    sp_digit ad[64 * 2];
    sp_digit pd[32], qd[32], dpd[32];
    sp_digit tmpad[64], tmpbd[64];
#else
    sp_digit* t = NULL;
#endif
    sp_digit* a;
    sp_digit* p;
    sp_digit* q;
    sp_digit* dp;
    sp_digit* dq;
    sp_digit* qi;
    sp_digit* tmp;
    sp_digit* tmpa;
    sp_digit* tmpb;
    sp_digit* r;
    sp_digit c;
    int err = MP_OKAY;

    (void)dm;
    (void)mm;

    if (*outLen < 512)
        err = MP_TO_E;
    if (err == MP_OKAY && (inLen > 512 || mp_count_bits(mm) != 4096))
        err = MP_READ_E;

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    if (err == MP_OKAY) {
        t = (sp_digit*)XMALLOC(sizeof(sp_digit) * 32 * 11, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (t == NULL)
            err = MEMORY_E;
    }
    if (err == MP_OKAY) {
        a = t;
        p = a + 64 * 2;
        q = p + 32;
        qi = dq = dp = q + 32;
        tmpa = qi + 32;
        tmpb = tmpa + 64;

        tmp = t;
        r = tmp + 64;
    }
#else
    r = a = ad;
    p = pd;
    q = qd;
    qi = dq = dp = dpd;
    tmpa = tmpad;
    tmpb = tmpbd;
    tmp = a + 64;
#endif

    if (err == MP_OKAY) {
        sp_4096_from_bin(a, 64, in, inLen);
        sp_4096_from_mp(p, 32, pm);
        sp_4096_from_mp(q, 32, qm);
        sp_4096_from_mp(dp, 32, dpm);

        err = sp_4096_mod_exp_32(tmpa, a, dp, 2048, p, 1);
    }
    if (err == MP_OKAY) {
        sp_4096_from_mp(dq, 32, dqm);
        err = sp_4096_mod_exp_32(tmpb, a, dq, 2048, q, 1);
    }

    if (err == MP_OKAY) {
        c = sp_4096_sub_in_place_32(tmpa, tmpb);
        sp_4096_mask_32(tmp, p, c);
        sp_4096_add_32(tmpa, tmpa, tmp);

        sp_4096_from_mp(qi, 32, qim);
        sp_4096_mul_32(tmpa, tmpa, qi);
        err = sp_4096_mod_32(tmpa, tmpa, p);
    }

    if (err == MP_OKAY) {
        sp_4096_mul_32(tmpa, q, tmpa);
        XMEMSET(&tmpb[32], 0, sizeof(sp_digit) * 32);
        sp_4096_add_64(r, tmpb, tmpa);

        sp_4096_to_bin(r, out);
        *outLen = 512;
    }

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    if (t != NULL) {
        XMEMSET(t, 0, sizeof(sp_digit) * 32 * 11);
        XFREE(t, NULL, DYNAMIC_TYPE_RSA);
    }
#else
    XMEMSET(tmpad, 0, sizeof(tmpad));
    XMEMSET(tmpbd, 0, sizeof(tmpbd));
    XMEMSET(pd, 0, sizeof(pd));
    XMEMSET(qd, 0, sizeof(qd));
    XMEMSET(dpd, 0, sizeof(dpd));
#endif

    return err;
}
#endif /* WOLFSSL_HAVE_SP_RSA */
#if defined(WOLFSSL_HAVE_SP_DH) || (defined(WOLFSSL_HAVE_SP_RSA) && \
                                              !defined(WOLFSSL_RSA_PUBLIC_ONLY))
/* Convert an array of sp_digit to an mp_int.
 *
 * a  A single precision integer.
 * r  A multi-precision integer.
 */
static int sp_4096_to_mp(const sp_digit* a, mp_int* r)
{
    int err;

    err = mp_grow(r, (4096 + DIGIT_BIT - 1) / DIGIT_BIT);
    if (err == MP_OKAY) { /*lint !e774 case where err is always MP_OKAY*/
#if DIGIT_BIT == 64
        XMEMCPY(r->dp, a, sizeof(sp_digit) * 64);
        r->used = 64;
        mp_clamp(r);
#elif DIGIT_BIT < 64
        int i, j = 0, s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 64; i++) {
            r->dp[j] |= a[i] << s;
            r->dp[j] &= (1L << DIGIT_BIT) - 1;
            s = DIGIT_BIT - s;
            r->dp[++j] = a[i] >> s;
            while (s + DIGIT_BIT <= 64) {
                s += DIGIT_BIT;
                r->dp[j++] &= (1L << DIGIT_BIT) - 1;
                if (s == SP_WORD_SIZE) {
                    r->dp[j] = 0;
                }
                else {
                    r->dp[j] = a[i] >> s;
                }
            }
            s = 64 - s;
        }
        r->used = (4096 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#else
        int i, j = 0, s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 64; i++) {
            r->dp[j] |= ((mp_digit)a[i]) << s;
            if (s + 64 >= DIGIT_BIT) {
    #if DIGIT_BIT != 32 && DIGIT_BIT != 64
                r->dp[j] &= (1L << DIGIT_BIT) - 1;
    #endif
                s = DIGIT_BIT - s;
                r->dp[++j] = a[i] >> s;
                s = 64 - s;
            }
            else {
                s += 64;
            }
        }
        r->used = (4096 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#endif
    }

    return err;
}

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base  Base. MP integer.
 * exp   Exponent. MP integer.
 * mod   Modulus. MP integer.
 * res   Result. MP integer.
 * returs 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_ModExp_4096(mp_int* base, mp_int* exp, mp_int* mod, mp_int* res)
{
    int err = MP_OKAY;
    sp_digit b[128], e[64], m[64];
    sp_digit* r = b;
    int expBits = mp_count_bits(exp);

    if (mp_count_bits(base) > 4096) {
        err = MP_READ_E;
    }

    if (err == MP_OKAY) {
        if (expBits > 4096) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        if (mp_count_bits(mod) != 4096) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        sp_4096_from_mp(b, 64, base);
        sp_4096_from_mp(e, 64, exp);
        sp_4096_from_mp(m, 64, mod);

        err = sp_4096_mod_exp_64(r, b, e, expBits, m, 0);
    }

    if (err == MP_OKAY) {
        err = sp_4096_to_mp(r, res);
    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}

#ifdef WOLFSSL_HAVE_SP_DH

#ifdef HAVE_FFDHE_4096
static void sp_4096_lshift_64(sp_digit* r, sp_digit* a, byte n)
{
    __asm__ __volatile__ (
        "mov	x6, 63\n\t"
        "sub	x6, x6, %[n]\n\t"
        "ldr	x3, [%[a], 504]\n\t"
        "lsr	x4, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x4, x4, x6\n\t"
        "ldr	x2, [%[a], 496]\n\t"
        "str	x4, [%[r], 512]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 488]\n\t"
        "str	x3, [%[r], 504]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 480]\n\t"
        "str	x2, [%[r], 496]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 472]\n\t"
        "str	x4, [%[r], 488]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 464]\n\t"
        "str	x3, [%[r], 480]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 456]\n\t"
        "str	x2, [%[r], 472]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 448]\n\t"
        "str	x4, [%[r], 464]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 440]\n\t"
        "str	x3, [%[r], 456]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 432]\n\t"
        "str	x2, [%[r], 448]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 424]\n\t"
        "str	x4, [%[r], 440]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 416]\n\t"
        "str	x3, [%[r], 432]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 408]\n\t"
        "str	x2, [%[r], 424]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 400]\n\t"
        "str	x4, [%[r], 416]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 392]\n\t"
        "str	x3, [%[r], 408]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 384]\n\t"
        "str	x2, [%[r], 400]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 376]\n\t"
        "str	x4, [%[r], 392]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 368]\n\t"
        "str	x3, [%[r], 384]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 360]\n\t"
        "str	x2, [%[r], 376]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 352]\n\t"
        "str	x4, [%[r], 368]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 344]\n\t"
        "str	x3, [%[r], 360]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 336]\n\t"
        "str	x2, [%[r], 352]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 328]\n\t"
        "str	x4, [%[r], 344]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 320]\n\t"
        "str	x3, [%[r], 336]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 312]\n\t"
        "str	x2, [%[r], 328]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 304]\n\t"
        "str	x4, [%[r], 320]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 296]\n\t"
        "str	x3, [%[r], 312]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 288]\n\t"
        "str	x2, [%[r], 304]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 280]\n\t"
        "str	x4, [%[r], 296]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 272]\n\t"
        "str	x3, [%[r], 288]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 264]\n\t"
        "str	x2, [%[r], 280]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 256]\n\t"
        "str	x4, [%[r], 272]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 248]\n\t"
        "str	x3, [%[r], 264]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 240]\n\t"
        "str	x2, [%[r], 256]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 232]\n\t"
        "str	x4, [%[r], 248]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 224]\n\t"
        "str	x3, [%[r], 240]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 216]\n\t"
        "str	x2, [%[r], 232]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 208]\n\t"
        "str	x4, [%[r], 224]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 200]\n\t"
        "str	x3, [%[r], 216]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 192]\n\t"
        "str	x2, [%[r], 208]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 184]\n\t"
        "str	x4, [%[r], 200]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 176]\n\t"
        "str	x3, [%[r], 192]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 168]\n\t"
        "str	x2, [%[r], 184]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 160]\n\t"
        "str	x4, [%[r], 176]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 152]\n\t"
        "str	x3, [%[r], 168]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 144]\n\t"
        "str	x2, [%[r], 160]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 136]\n\t"
        "str	x4, [%[r], 152]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 128]\n\t"
        "str	x3, [%[r], 144]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 120]\n\t"
        "str	x2, [%[r], 136]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 112]\n\t"
        "str	x4, [%[r], 128]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 104]\n\t"
        "str	x3, [%[r], 120]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 96]\n\t"
        "str	x2, [%[r], 112]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 88]\n\t"
        "str	x4, [%[r], 104]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 80]\n\t"
        "str	x3, [%[r], 96]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 72]\n\t"
        "str	x2, [%[r], 88]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 64]\n\t"
        "str	x4, [%[r], 80]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 56]\n\t"
        "str	x3, [%[r], 72]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 48]\n\t"
        "str	x2, [%[r], 64]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 40]\n\t"
        "str	x4, [%[r], 56]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 32]\n\t"
        "str	x3, [%[r], 48]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 24]\n\t"
        "str	x2, [%[r], 40]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 16]\n\t"
        "str	x4, [%[r], 32]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 8]\n\t"
        "str	x3, [%[r], 24]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 0]\n\t"
        "str	x2, [%[r], 16]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "str	x3, [%[r]]\n\t"
        "str	x4, [%[r], 8]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [n] "r" (n)
        : "memory", "x2", "x3", "x4", "x5", "x6"
    );
}

/* Modular exponentiate 2 to the e mod m. (r = 2^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns 0 on success and MEMORY_E on dynamic memory allocation failure.
 */
static int sp_4096_mod_exp_2_64(sp_digit* r, const sp_digit* e, int bits,
        const sp_digit* m)
{
#ifndef WOLFSSL_SMALL_STACK
    sp_digit nd[128];
    sp_digit td[65];
#else
    sp_digit* td;
#endif
    sp_digit* norm;
    sp_digit* tmp;
    sp_digit mp = 1;
    sp_digit n, o;
    sp_digit mask;
    int i;
    int c, y;
    int err = MP_OKAY;

#ifdef WOLFSSL_SMALL_STACK
    td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 193, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER);
    if (td == NULL) {
        err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
#ifdef WOLFSSL_SMALL_STACK
        norm = td;
        tmp  = td + 128;
#else
        norm = nd;
        tmp  = td;
#endif

        sp_4096_mont_setup(m, &mp);
        sp_4096_mont_norm_64(norm, m);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 6;
        if (c == 64) {
            c = 58;
        }
        y = (int)(n >> c);
        n <<= 64 - c;
        sp_4096_lshift_64(r, norm, y);
        for (; i>=0 || c>=6; ) {
            if (c == 0) {
                n = e[i--];
                y = n >> 58;
                n <<= 6;
                c = 58;
            }
            else if (c < 6) {
                y = n >> 58;
                n = e[i--];
                c = 6 - c;
                y |= n >> (64 - c);
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (n >> 58) & 0x3f;
                n <<= 6;
                c -= 6;
            }

            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);

            sp_4096_lshift_64(r, r, y);
            sp_4096_mul_d_64(tmp, norm, r[64]);
            r[64] = 0;
            o = sp_4096_add_64(r, r, tmp);
            sp_4096_cond_sub_64(r, r, m, (sp_digit)0 - o);
        }

        XMEMSET(&r[64], 0, sizeof(sp_digit) * 64U);
        sp_4096_mont_reduce_64(r, m, mp);

        mask = 0 - (sp_4096_cmp_64(r, m) >= 0);
        sp_4096_cond_sub_64(r, r, m, mask);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (td != NULL) {
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return err;
}
#endif /* HAVE_FFDHE_4096 */

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base     Base.
 * exp      Array of bytes that is the exponent.
 * expLen   Length of data, in bytes, in exponent.
 * mod      Modulus.
 * out      Buffer to hold big-endian bytes of exponentiation result.
 *          Must be at least 512 bytes long.
 * outLen   Length, in bytes, of exponentiation result.
 * returs 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_DhExp_4096(mp_int* base, const byte* exp, word32 expLen,
    mp_int* mod, byte* out, word32* outLen)
{
    int err = MP_OKAY;
    sp_digit b[128], e[64], m[64];
    sp_digit* r = b;
    word32 i;

    if (mp_count_bits(base) > 4096) {
        err = MP_READ_E;
    }

    if (err == MP_OKAY) {
        if (expLen > 512) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        if (mp_count_bits(mod) != 4096) {
            err = MP_READ_E;
        }
    }

    if (err == MP_OKAY) {
        sp_4096_from_mp(b, 64, base);
        sp_4096_from_bin(e, 64, exp, expLen);
        sp_4096_from_mp(m, 64, mod);

    #ifdef HAVE_FFDHE_4096
        if (base->used == 1 && base->dp[0] == 2 && m[63] == (sp_digit)-1)
            err = sp_4096_mod_exp_2_64(r, e, expLen * 8, m);
        else
    #endif
            err = sp_4096_mod_exp_64(r, b, e, expLen * 8, m, 0);

    }

    if (err == MP_OKAY) {
        sp_4096_to_bin(r, out);
        *outLen = 512;
        for (i=0; i<512 && out[i] == 0; i++) {
        }
        *outLen -= i;
        XMEMMOVE(out, out + i, *outLen);

    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}
#endif /* WOLFSSL_HAVE_SP_DH */

#endif /* WOLFSSL_HAVE_SP_DH || (WOLFSSL_HAVE_SP_RSA && !WOLFSSL_RSA_PUBLIC_ONLY) */

#endif /* WOLFSSL_SP_4096 */

#endif /* WOLFSSL_HAVE_SP_RSA || WOLFSSL_HAVE_SP_DH */
#ifdef WOLFSSL_HAVE_SP_ECC
#ifndef WOLFSSL_SP_NO_256

/* Point structure to use. */
typedef struct sp_point {
    sp_digit x[2 * 4];
    sp_digit y[2 * 4];
    sp_digit z[2 * 4];
    int infinity;
} sp_point;

/* The modulus (prime) of the curve P256. */
static const sp_digit p256_mod[4] = {
    0xffffffffffffffffL,0x00000000ffffffffL,0x0000000000000000L,
    0xffffffff00000001L
};
/* The Montogmery normalizer for modulus of the curve P256. */
static const sp_digit p256_norm_mod[4] = {
    0x0000000000000001L,0xffffffff00000000L,0xffffffffffffffffL,
    0x00000000fffffffeL
};
/* The Montogmery multiplier for modulus of the curve P256. */
static const sp_digit p256_mp_mod = 0x0000000000000001;
#if defined(WOLFSSL_VALIDATE_ECC_KEYGEN) || defined(HAVE_ECC_SIGN) || \
                                            defined(HAVE_ECC_VERIFY)
/* The order of the curve P256. */
static const sp_digit p256_order[4] = {
    0xf3b9cac2fc632551L,0xbce6faada7179e84L,0xffffffffffffffffL,
    0xffffffff00000000L
};
#endif
/* The order of the curve P256 minus 2. */
static const sp_digit p256_order2[4] = {
    0xf3b9cac2fc63254fL,0xbce6faada7179e84L,0xffffffffffffffffL,
    0xffffffff00000000L
};
#if defined(HAVE_ECC_SIGN) || defined(HAVE_ECC_VERIFY)
/* The Montogmery normalizer for order of the curve P256. */
static const sp_digit p256_norm_order[4] = {
    0x0c46353d039cdaafL,0x4319055258e8617bL,0x0000000000000000L,
    0x00000000ffffffffL
};
#endif
#if defined(HAVE_ECC_SIGN) || defined(HAVE_ECC_VERIFY)
/* The Montogmery multiplier for order of the curve P256. */
static const sp_digit p256_mp_order = 0xccd1c8aaee00bc4fL;
#endif
#ifdef WOLFSSL_SP_SMALL
/* The base point of curve P256. */
static const sp_point p256_base = {
    /* X ordinate */
    {
        0xf4a13945d898c296L,0x77037d812deb33a0L,0xf8bce6e563a440f2L,
        0x6b17d1f2e12c4247L, 0L, 0L, 0L, 0L
    },
    /* Y ordinate */
    {
        0xcbb6406837bf51f5L,0x2bce33576b315eceL,0x8ee7eb4a7c0f9e16L,
        0x4fe342e2fe1a7f9bL, 0L, 0L, 0L, 0L
    },
    /* Z ordinate */
    {
        0x0000000000000001L,0x0000000000000000L,0x0000000000000000L,
        0x0000000000000000L, 0L, 0L, 0L, 0L
    },
    /* infinity */
    0
};
#endif /* WOLFSSL_SP_SMALL */
#if defined(HAVE_ECC_CHECK_KEY) || defined(HAVE_COMP_KEY)
static const sp_digit p256_b[4] = {
    0x3bce3c3e27d2604bL,0x651d06b0cc53b0f6L,0xb3ebbd55769886bcL,
    0x5ac635d8aa3a93e7L
};
#endif

static int sp_ecc_point_new_ex(void* heap, sp_point* sp, sp_point** p)
{
    int ret = MP_OKAY;
    (void)heap;
#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    (void)sp;
    *p = (sp_point*)XMALLOC(sizeof(sp_point), heap, DYNAMIC_TYPE_ECC);
#else
    *p = sp;
#endif
    if (p == NULL) {
        ret = MEMORY_E;
    }
    return ret;
}

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
/* Allocate memory for point and return error. */
#define sp_ecc_point_new(heap, sp, p) sp_ecc_point_new_ex((heap), NULL, &(p))
#else
/* Set pointer to data and return no error. */
#define sp_ecc_point_new(heap, sp, p) sp_ecc_point_new_ex((heap), &(sp), &(p))
#endif


static void sp_ecc_point_free(sp_point* p, int clear, void* heap)
{
#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
/* If valid pointer then clear point data if requested and free data. */
    if (p != NULL) {
        if (clear != 0) {
            XMEMSET(p, 0, sizeof(*p));
        }
        XFREE(p, heap, DYNAMIC_TYPE_ECC);
    }
#else
/* Clear point data if requested. */
    if (clear != 0) {
        XMEMSET(p, 0, sizeof(*p));
    }
#endif
    (void)heap;
}

/* Multiply a number by Montogmery normalizer mod modulus (prime).
 *
 * r  The resulting Montgomery form number.
 * a  The number to convert.
 * m  The modulus (prime).
 */
static int sp_256_mod_mul_norm_4(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    int64_t t[8];
    int64_t a32[8];
    int64_t o;

    (void)m;

    a32[0] = a[0] & 0xffffffff;
    a32[1] = a[0] >> 32;
    a32[2] = a[1] & 0xffffffff;
    a32[3] = a[1] >> 32;
    a32[4] = a[2] & 0xffffffff;
    a32[5] = a[2] >> 32;
    a32[6] = a[3] & 0xffffffff;
    a32[7] = a[3] >> 32;

    /*  1  1  0 -1 -1 -1 -1  0 */
    t[0] = 0 + a32[0] + a32[1] - a32[3] - a32[4] - a32[5] - a32[6];
    /*  0  1  1  0 -1 -1 -1 -1 */
    t[1] = 0 + a32[1] + a32[2] - a32[4] - a32[5] - a32[6] - a32[7];
    /*  0  0  1  1  0 -1 -1 -1 */
    t[2] = 0 + a32[2] + a32[3] - a32[5] - a32[6] - a32[7];
    /* -1 -1  0  2  2  1  0 -1 */
    t[3] = 0 - a32[0] - a32[1] + 2 * a32[3] + 2 * a32[4] + a32[5] - a32[7];
    /*  0 -1 -1  0  2  2  1  0 */
    t[4] = 0 - a32[1] - a32[2] + 2 * a32[4] + 2 * a32[5] + a32[6];
    /*  0  0 -1 -1  0  2  2  1 */
    t[5] = 0 - a32[2] - a32[3] + 2 * a32[5] + 2 * a32[6] + a32[7];
    /* -1 -1  0  0  0  1  3  2 */
    t[6] = 0 - a32[0] - a32[1] + a32[5] + 3 * a32[6] + 2 * a32[7];
    /*  1  0 -1 -1 -1 -1  0  3 */
    t[7] = 0 + a32[0] - a32[2] - a32[3] - a32[4] - a32[5] + 3 * a32[7];

    t[1] += t[0] >> 32; t[0] &= 0xffffffff;
    t[2] += t[1] >> 32; t[1] &= 0xffffffff;
    t[3] += t[2] >> 32; t[2] &= 0xffffffff;
    t[4] += t[3] >> 32; t[3] &= 0xffffffff;
    t[5] += t[4] >> 32; t[4] &= 0xffffffff;
    t[6] += t[5] >> 32; t[5] &= 0xffffffff;
    t[7] += t[6] >> 32; t[6] &= 0xffffffff;
    o     = t[7] >> 32; t[7] &= 0xffffffff;
    t[0] += o;
    t[3] -= o;
    t[6] -= o;
    t[7] += o;
    t[1] += t[0] >> 32; t[0] &= 0xffffffff;
    t[2] += t[1] >> 32; t[1] &= 0xffffffff;
    t[3] += t[2] >> 32; t[2] &= 0xffffffff;
    t[4] += t[3] >> 32; t[3] &= 0xffffffff;
    t[5] += t[4] >> 32; t[4] &= 0xffffffff;
    t[6] += t[5] >> 32; t[5] &= 0xffffffff;
    t[7] += t[6] >> 32; t[6] &= 0xffffffff;
    r[0] = (t[1] << 32) | t[0];
    r[1] = (t[3] << 32) | t[2];
    r[2] = (t[5] << 32) | t[4];
    r[3] = (t[7] << 32) | t[6];

    return MP_OKAY;
}

/* Convert an mp_int to an array of sp_digit.
 *
 * r  A single precision integer.
 * size  Maximum number of bytes to convert
 * a  A multi-precision integer.
 */
static void sp_256_from_mp(sp_digit* r, int size, const mp_int* a)
{
#if DIGIT_BIT == 64
    int j;

    XMEMCPY(r, a->dp, sizeof(sp_digit) * a->used);

    for (j = a->used; j < size; j++) {
        r[j] = 0;
    }
#elif DIGIT_BIT > 64
    int i, j = 0;
    word32 s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i] << s);
        r[j] &= 0xffffffffffffffffl;
        s = 64U - s;
        if (j + 1 >= size) {
            break;
        }
        /* lint allow cast of mismatch word32 and mp_digit */
        r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
        while ((s + 64U) <= (word32)DIGIT_BIT) {
            s += 64U;
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            if (s < (word32)DIGIT_BIT) {
                /* lint allow cast of mismatch word32 and mp_digit */
                r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
            }
            else {
                r[++j] = 0L;
            }
        }
        s = (word32)DIGIT_BIT - s;
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#else
    int i, j = 0, s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i]) << s;
        if (s + DIGIT_BIT >= 64) {
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            s = 64 - s;
            if (s == DIGIT_BIT) {
                r[++j] = 0;
                s = 0;
            }
            else {
                r[++j] = a->dp[i] >> s;
                s = DIGIT_BIT - s;
            }
        }
        else {
            s += DIGIT_BIT;
        }
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#endif
}

/* Convert a point of type ecc_point to type sp_point.
 *
 * p   Point of type sp_point (result).
 * pm  Point of type ecc_point.
 */
static void sp_256_point_from_ecc_point_4(sp_point* p, const ecc_point* pm)
{
    XMEMSET(p->x, 0, sizeof(p->x));
    XMEMSET(p->y, 0, sizeof(p->y));
    XMEMSET(p->z, 0, sizeof(p->z));
    sp_256_from_mp(p->x, 4, pm->x);
    sp_256_from_mp(p->y, 4, pm->y);
    sp_256_from_mp(p->z, 4, pm->z);
    p->infinity = 0;
}

/* Convert an array of sp_digit to an mp_int.
 *
 * a  A single precision integer.
 * r  A multi-precision integer.
 */
static int sp_256_to_mp(const sp_digit* a, mp_int* r)
{
    int err;

    err = mp_grow(r, (256 + DIGIT_BIT - 1) / DIGIT_BIT);
    if (err == MP_OKAY) { /*lint !e774 case where err is always MP_OKAY*/
#if DIGIT_BIT == 64
        XMEMCPY(r->dp, a, sizeof(sp_digit) * 4);
        r->used = 4;
        mp_clamp(r);
#elif DIGIT_BIT < 64
        int i, j = 0, s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 4; i++) {
            r->dp[j] |= a[i] << s;
            r->dp[j] &= (1L << DIGIT_BIT) - 1;
            s = DIGIT_BIT - s;
            r->dp[++j] = a[i] >> s;
            while (s + DIGIT_BIT <= 64) {
                s += DIGIT_BIT;
                r->dp[j++] &= (1L << DIGIT_BIT) - 1;
                if (s == SP_WORD_SIZE) {
                    r->dp[j] = 0;
                }
                else {
                    r->dp[j] = a[i] >> s;
                }
            }
            s = 64 - s;
        }
        r->used = (256 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#else
        int i, j = 0, s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 4; i++) {
            r->dp[j] |= ((mp_digit)a[i]) << s;
            if (s + 64 >= DIGIT_BIT) {
    #if DIGIT_BIT != 32 && DIGIT_BIT != 64
                r->dp[j] &= (1L << DIGIT_BIT) - 1;
    #endif
                s = DIGIT_BIT - s;
                r->dp[++j] = a[i] >> s;
                s = 64 - s;
            }
            else {
                s += 64;
            }
        }
        r->used = (256 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#endif
    }

    return err;
}

/* Convert a point of type sp_point to type ecc_point.
 *
 * p   Point of type sp_point.
 * pm  Point of type ecc_point (result).
 * returns MEMORY_E when allocation of memory in ecc_point fails otherwise
 * MP_OKAY.
 */
static int sp_256_point_to_ecc_point_4(const sp_point* p, ecc_point* pm)
{
    int err;

    err = sp_256_to_mp(p->x, pm->x);
    if (err == MP_OKAY) {
        err = sp_256_to_mp(p->y, pm->y);
    }
    if (err == MP_OKAY) {
        err = sp_256_to_mp(p->z, pm->z);
    }

    return err;
}

/* Conditionally copy a into r using the mask m.
 * m is -1 to copy and 0 when not.
 *
 * r  A single precision number to copy over.
 * a  A single precision number to copy.
 * m  Mask value to apply.
 */
static void sp_256_cond_copy_4(sp_digit* r, const sp_digit* a, sp_digit m)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[r], 0]\n\t"
        "ldp	x5, x6, [%[r], 16]\n\t"
        "ldp	x7, x8, [%[a], 0]\n\t"
        "ldp	x9, x10, [%[a], 16]\n\t"
        "eor	x7, x7, x3\n\t"
        "eor	x8, x8, x4\n\t"
        "eor	x9, x9, x5\n\t"
        "eor	x10, x10, x6\n\t"
        "and	x7, x7, %[m]\n\t"
        "and	x8, x8, %[m]\n\t"
        "and	x9, x9, %[m]\n\t"
        "and	x10, x10, %[m]\n\t"
        "eor	x3, x3, x7\n\t"
        "eor	x4, x4, x8\n\t"
        "eor	x5, x5, x9\n\t"
        "eor	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [m] "r" (m)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );
}

/* Compare a with b in constant time.
 *
 * a  A single precision integer.
 * b  A single precision integer.
 * return -ve, 0 or +ve if a is less than, equal to or greater than b
 * respectively.
 */
static int64_t sp_256_cmp_4(const sp_digit* a, const sp_digit* b)
{
    sp_digit r = -1;
    sp_digit one = 1;

#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x3, -1\n\t"
        "mov	x6, 24\n\t"
        "1:\n\t"
        "ldr	x4, [%[a], x6]\n\t"
        "ldr	x5, [%[b], x6]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "subs	x6, x6, #8\n\t"
        "b.cs	1b\n\t"
        "eor	%[r], %[r], x3\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [one] "r" (one)
        : "x2", "x3", "x4", "x5", "x6"
    );
#else
    __asm__ __volatile__ (
        "mov	x3, -1\n\t"
        "ldr		x4, [%[a], 24]\n\t"
        "ldr		x5, [%[b], 24]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 16]\n\t"
        "ldr		x5, [%[b], 16]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 8]\n\t"
        "ldr		x5, [%[b], 8]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "ldr		x4, [%[a], 0]\n\t"
        "ldr		x5, [%[b], 0]\n\t"
        "and		x4, x4, x3\n\t"
        "and		x5, x5, x3\n\t"
        "subs	x4, x4, x5\n\t"
        "csel	%[r], %[one], %[r], hi\n\t"
        "csel	%[r], x3, %[r], lo\n\t"
        "csel	x3, x3, xzr, eq\n\t"
        "eor	%[r], %[r], x3\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [one] "r" (one)
        : "x2", "x3", "x4", "x5", "x6"
    );
#endif

    return r;
}

/* Normalize the values in each word to 64.
 *
 * a  Array of sp_digit to normalize.
 */
#define sp_256_norm_4(a)

/* Conditionally subtract b from a using the mask m.
 * m is -1 to subtract and 0 when not copying.
 *
 * r  A single precision number representing condition subtract result.
 * a  A single precision number to subtract from.
 * b  A single precision number to subtract.
 * m  Mask value to apply.
 */
static sp_digit sp_256_cond_sub_4(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    sp_digit c = 0;

    __asm__ __volatile__ (

        "ldr		x4, [%[a], 0]\n\t"
        "ldr		x6, [%[a], 8]\n\t"
        "ldr		x5, [%[b], 0]\n\t"
        "ldr		x7, [%[b], 8]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "subs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 0]\n\t"
        "str		x6, [%[r], 8]\n\t"
        "ldr		x4, [%[a], 16]\n\t"
        "ldr		x6, [%[a], 24]\n\t"
        "ldr		x5, [%[b], 16]\n\t"
        "ldr		x7, [%[b], 24]\n\t"
        "and		x5, x5, %[m]\n\t"
        "and		x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "sbcs	x6, x6, x7\n\t"
        "str		x4, [%[r], 16]\n\t"
        "str		x6, [%[r], 24]\n\t"
        "csetm	%[c], cc\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8"
    );

    return c;
}

/* Sub b from a into r. (r = a - b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_256_sub_4(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "subs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "csetm	%[c], cc\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

#define sp_256_mont_reduce_order_4    sp_256_mont_reduce_4

/* Reduce the number back to 256 bits using Montgomery reduction.
 *
 * a   A single precision number to reduce in place.
 * m   The single precision number representing the modulus.
 * mp  The digit representing the negative inverse of m mod 2^n.
 */
SP_NOINLINE static void sp_256_mont_reduce_4(sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    __asm__ __volatile__ (
        "# i = 0\n\t"
        "mov	x9, xzr\n\t"
        "mov	x8, xzr\n\t"
        "mov	x6, %[a]\n\t"
        "\n1:\n\t"
        "# mu = a[i] * mp\n\t"
        "ldr	x5, [x6, 0]\n\t"
        "mov	x7, x5\n\t"
        "mul	x5, %[mp], x5\n\t"
        "# a[i+0] += m[0] * mu\n\t"
        "ldr	x4, [%[m], 0]\n\t"
        "ldr	x11, [%[m], 8]\n\t"
        "mul	x3, x4, x5\n\t"
        "umulh	x10, x4, x5\n\t"
        "adds	x7, x7, x3\n\t"
        "str	x7, [x6, 0]\n\t"
        "adc	x10, x10, xzr\n\t"
        "# a[i+1] += m[1] * mu\n\t"
        "mul	x3, x11, x5\n\t"
        "umulh	x12, x11, x5\n\t"
        "ldr	x11, [%[m], 16]\n\t"
        "ldr	x7, [x6, 8]\n\t"
        "adds	x3, x3, x10\n\t"
        "adc	x12, x12, xzr\n\t"
        "adds	x7, x7, x3\n\t"
        "str	x7, [x6, 8]\n\t"
        "adc	x12, x12, xzr\n\t"
        "# a[i+2] += m[2] * mu\n\t"
        "mul	x3, x11, x5\n\t"
        "umulh	x10, x11, x5\n\t"
        "ldr	x11, [%[m], 24]\n\t"
        "ldr	x7, [x6, 16]\n\t"
        "adds	x3, x3, x12\n\t"
        "adc	x10, x10, xzr\n\t"
        "adds	x7, x7, x3\n\t"
        "str	x7, [x6, 16]\n\t"
        "adc	x10, x10, xzr\n\t"
        "# a[i+3] += m[3] * mu\n\t"
        "mul	x3, x11, x5\n\t"
        "umulh	x4, x11, x5\n\t"
        "ldr	x7, [x6, 24]\n\t"
        "ldr	x12, [x6, 32]\n\t"
        "adds	x3, x3, x10\n\t"
        "adcs	x4, x4, x9\n\t"
        "cset	x9, cs\n\t"
        "adds	x7, x7, x3\n\t"
        "str	x7, [x6, 24]\n\t"
        "adcs	x12, x12, x4\n\t"
        "str	x12, [x6, 32]\n\t"
        "adc	x9, x9, xzr\n\t"
        "# i += 1\n\t"
        "add	x6, x6, 8\n\t"
        "add	x8, x8, 8\n\t"
        "cmp	x8, 32\n\t"
        "b.lt	1b\n\t"
        "ldr	x5, [%[a], 32]\n\t"
        "ldr	x6, [%[a], 40]\n\t"
        "ldr	x7, [%[a], 48]\n\t"
        "ldr	x8, [%[a], 56]\n\t"
        "sub	x3, xzr, x9\n\t"
        "ldr	x9, [%[m], 0]\n\t"
        "ldr	x10, [%[m], 8]\n\t"
        "ldr	x11, [%[m], 16]\n\t"
        "ldr	x12, [%[m], 24]\n\t"
        "and	x9, x9, x3\n\t"
        "and	x10, x10, x3\n\t"
        "and	x11, x11, x3\n\t"
        "and	x12, x12, x3\n\t"
        "subs	x5, x5, x9\n\t"
        "sbcs	x6, x6, x10\n\t"
        "sbcs	x7, x7, x11\n\t"
        "sbc	x8, x8, x12\n\t"
        "str	x5, [%[a], 0]\n\t"
        "str	x6, [%[a], 8]\n\t"
        "str	x7, [%[a], 16]\n\t"
        "str	x8, [%[a], 24]\n\t"
        :
        : [a] "r" (a), [m] "r" (m), [mp] "r" (mp)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11",
          "x12"
    );
}

/* Multiply two Montogmery form numbers mod the modulus (prime).
 * (r = a * b mod m)
 *
 * r   Result of multiplication.
 * a   First number to multiply in Montogmery form.
 * b   Second number to multiply in Montogmery form.
 * m   Modulus (prime).
 * mp  Montogmery mulitplier.
 */
SP_NOINLINE static void sp_256_mont_mul_4(sp_digit* r, const sp_digit* a, const sp_digit* b,
        const sp_digit* m, sp_digit mp)
{
    (void)mp;

    __asm__ __volatile__ (
        "ldr	x19, [%[a], 0]\n\t"
        "ldr	x20, [%[a], 8]\n\t"
        "ldr	x21, [%[a], 16]\n\t"
        "ldr	x22, [%[a], 24]\n\t"
        "ldr	x23, [%[b], 0]\n\t"
        "ldr	x24, [%[b], 8]\n\t"
        "ldr	x25, [%[b], 16]\n\t"
        "ldr	x26, [%[b], 24]\n\t"
        "#  A[0] * B[0]\n\t"
        "mul	x10, x19, x23\n\t"
        "umulh	x11, x19, x23\n\t"
        "#  A[0] * B[1]\n\t"
        "mul	x5, x19, x24\n\t"
        "umulh	x6, x19, x24\n\t"
        "adds	x11, x11, x5\n\t"
        "adc	x12, xzr, x6\n\t"
        "#  A[1] * B[0]\n\t"
        "mul	x5, x20, x23\n\t"
        "umulh	x6, x20, x23\n\t"
        "adds	x11, x11, x5\n\t"
        "adcs	x12, x12, x6\n\t"
        "adc	x13, xzr, xzr\n\t"
        "#  A[0] * B[2]\n\t"
        "mul	x5, x19, x25\n\t"
        "umulh	x6, x19, x25\n\t"
        "adds	x12, x12, x5\n\t"
        "adc	x13, x13, x6\n\t"
        "#  A[1] * B[1]\n\t"
        "mul	x5, x20, x24\n\t"
        "umulh	x6, x20, x24\n\t"
        "adds	x12, x12, x5\n\t"
        "adcs	x13, x13, x6\n\t"
        "adc	x14, xzr, xzr\n\t"
        "#  A[2] * B[0]\n\t"
        "mul	x5, x21, x23\n\t"
        "umulh	x6, x21, x23\n\t"
        "adds	x12, x12, x5\n\t"
        "adcs	x13, x13, x6\n\t"
        "adc	x14, x14, xzr\n\t"
        "#  A[0] * B[3]\n\t"
        "mul	x5, x19, x26\n\t"
        "umulh	x6, x19, x26\n\t"
        "adds	x13, x13, x5\n\t"
        "adcs	x14, x14, x6\n\t"
        "adc	x15, xzr, xzr\n\t"
        "#  A[1] * B[2]\n\t"
        "mul	x5, x20, x25\n\t"
        "umulh	x6, x20, x25\n\t"
        "adds	x13, x13, x5\n\t"
        "adcs	x14, x14, x6\n\t"
        "adc	x15, x15, xzr\n\t"
        "#  A[2] * B[1]\n\t"
        "mul	x5, x21, x24\n\t"
        "umulh	x6, x21, x24\n\t"
        "adds	x13, x13, x5\n\t"
        "adcs	x14, x14, x6\n\t"
        "adc	x15, x15, xzr\n\t"
        "#  A[3] * B[0]\n\t"
        "mul	x5, x22, x23\n\t"
        "umulh	x6, x22, x23\n\t"
        "adds	x13, x13, x5\n\t"
        "adcs	x14, x14, x6\n\t"
        "adc	x15, x15, xzr\n\t"
        "#  A[1] * B[3]\n\t"
        "mul	x5, x20, x26\n\t"
        "umulh	x6, x20, x26\n\t"
        "adds	x14, x14, x5\n\t"
        "adcs	x15, x15, x6\n\t"
        "adc	x16, xzr, xzr\n\t"
        "#  A[2] * B[2]\n\t"
        "mul	x5, x21, x25\n\t"
        "umulh	x6, x21, x25\n\t"
        "adds	x14, x14, x5\n\t"
        "adcs	x15, x15, x6\n\t"
        "adc	x16, x16, xzr\n\t"
        "#  A[3] * B[1]\n\t"
        "mul	x5, x22, x24\n\t"
        "umulh	x6, x22, x24\n\t"
        "adds	x14, x14, x5\n\t"
        "adcs	x15, x15, x6\n\t"
        "adc	x16, x16, xzr\n\t"
        "#  A[2] * B[3]\n\t"
        "mul	x5, x21, x26\n\t"
        "umulh	x6, x21, x26\n\t"
        "adds	x15, x15, x5\n\t"
        "adcs	x16, x16, x6\n\t"
        "adc	x17, xzr, xzr\n\t"
        "#  A[3] * B[2]\n\t"
        "mul	x5, x22, x25\n\t"
        "umulh	x6, x22, x25\n\t"
        "adds	x15, x15, x5\n\t"
        "adcs	x16, x16, x6\n\t"
        "adc	x17, x17, xzr\n\t"
        "#  A[3] * B[3]\n\t"
        "mul	x5, x22, x26\n\t"
        "umulh	x6, x22, x26\n\t"
        "adds	x16, x16, x5\n\t"
        "adc	x17, x17, x6\n\t"
        "# Start Reduction\n\t"
        "mov	x5, x10\n\t"
        "mov	x6, x11\n\t"
        "mov	x7, x12\n\t"
        "mov	x8, x13\n\t"
        "# mu = a[0]-a[3] + a[0]-a[2] << 32 << 64 + (a[0] * 2) << 192\n\t"
        "#    - a[0] << 32 << 192\n\t"
        "#   + (a[0] * 2) << 192\n\t"
        "add	x8, x8, x10\n\t"
        "add	x8, x8, x10\n\t"
        "#   a[0]-a[2] << 32\n\t"
        "lsl	x10, x10, 32\n\t"
        "lsr	x19, x5, 32\n\t"
        "lsl	x11, x6, 32\n\t"
        "lsr	x20, x6, 32\n\t"
        "lsl	x12, x7, 32\n\t"
        "eor	x11, x11, x19\n\t"
        "eor	x12, x12, x20\n\t"
        "#   - a[0] << 32 << 192\n\t"
        "sub	x8, x8, x10\n\t"
        "#   + a[0]-a[2] << 32 << 64\n\t"
        "adds	x6, x6, x10\n\t"
        "adcs	x7, x7, x11\n\t"
        "adc	x8, x8, x12\n\t"
        "# a += (mu << 256) - (mu << 224) + (mu << 192) + (mu << 96) - mu\n\t"
        "#   a += mu << 256\n\t"
        "adds	x14, x14, x5\n\t"
        "adcs	x15, x15, x6\n\t"
        "adcs	x16, x16, x7\n\t"
        "adcs	x17, x17, x8\n\t"
        "csetm	x10, cs\n\t"
        "#   a += mu << 192\n\t"
        "adds	x13, x13, x5\n\t"
        "adcs	x14, x14, x6\n\t"
        "adcs	x15, x15, x7\n\t"
        "adcs	x16, x16, x8\n\t"
        "adcs	x17, x17, xzr\n\t"
        "csetm	x21, cs\n\t"
        "add	x10, x10, x21\n\t"
        "# mu <<= 32\n\t"
        "lsr	x9, x8, 32\n\t"
        "lsr	x19, x5, 32\n\t"
        "lsl	x5, x5, 32\n\t"
        "lsr	x20, x6, 32\n\t"
        "lsl	x6, x6, 32\n\t"
        "lsr	x21, x7, 32\n\t"
        "lsl	x7, x7, 32\n\t"
        "lsl	x8, x8, 32\n\t"
        "eor	x6, x6, x19\n\t"
        "eor	x7, x7, x20\n\t"
        "eor	x8, x8, x21\n\t"
        "#   a += (mu << 32) << 64\n\t"
        "adds	x13, x13, x7\n\t"
        "adcs	x14, x14, x8\n\t"
        "adcs	x15, x15, x9\n\t"
        "adcs	x16, x16, xzr\n\t"
        "adcs	x17, x17, xzr\n\t"
        "csetm	x21, cs\n\t"
        "add	x10, x10, x21\n\t"
        "#   a -= (mu << 32) << 192\n\t"
        "subs	x13, x13, x5\n\t"
        "mov	x19, 0xffffffff\n\t"
        "sbcs	x14, x14, x6\n\t"
        "mov	x20, 0xffffffff00000001\n\t"
        "sbcs	x15, x15, x7\n\t"
        "sbcs	x16, x16, x8\n\t"
        "sbcs	x17, x17, x9\n\t"
        "cset	x21, cc\n\t"
        "add	x10, x10, x21\n\t"
        "# mask m and sub from result if overflow\n\t"
        "#  m[0] = -1 & mask = mask\n\t"
        "and	x19, x19, x10\n\t"
        "#  m[2] =  0 & mask = 0\n\t"
        "and	x20, x20, x10\n\t"
        "subs	x14, x14, x10\n\t"
        "sbcs	x15, x15, x19\n\t"
        "sbcs	x16, x16, xzr\n\t"
        "sbc	x17, x17, x20\n\t"
        "str	x14, [%[r], 0]\n\t"
        "str	x15, [%[r], 8]\n\t"
        "str	x16, [%[r], 16]\n\t"
        "str	x17, [%[r], 24]\n\t"
        : [m] "+r" (m), [a] "+r" (a), [b] "+r" (b)
        : [r] "r" (r)
        : "memory", "x5", "x6", "x7", "x8", "x9",
          "x19", "x20", "x21", "x22",
          "x23", "x24", "x25", "x26",
          "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17"
    );
}

/* Square the Montgomery form number mod the modulus (prime). (r = a * a mod m)
 *
 * r   Result of squaring.
 * a   Number to square in Montogmery form.
 * m   Modulus (prime).
 * mp  Montogmery mulitplier.
 */
SP_NOINLINE static void sp_256_mont_sqr_4(sp_digit* r, const sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    __asm__ __volatile__ (
        "ldr	x19, [%[a], 0]\n\t"
        "ldr	x20, [%[a], 8]\n\t"
        "ldr	x21, [%[a], 16]\n\t"
        "ldr	x22, [%[a], 24]\n\t"
        "#  A[0] * A[1]\n\t"
        "mul	x11, x19, x20\n\t"
        "umulh	x12, x19, x20\n\t"
        "#  A[0] * A[2]\n\t"
        "mul	x5, x19, x21\n\t"
        "umulh	x6, x19, x21\n\t"
        "adds	x12, x12, x5\n\t"
        "adc	x13, xzr, x6\n\t"
        "#  A[0] * A[3]\n\t"
        "mul	x5, x19, x22\n\t"
        "umulh	x6, x19, x22\n\t"
        "adds	x13, x13, x5\n\t"
        "adc	x14, xzr, x6\n\t"
        "#  A[1] * A[2]\n\t"
        "mul	x5, x20, x21\n\t"
        "umulh	x6, x20, x21\n\t"
        "adds	x13, x13, x5\n\t"
        "adcs	x14, x14, x6\n\t"
        "adc	x15, xzr, xzr\n\t"
        "#  A[1] * A[3]\n\t"
        "mul	x5, x20, x22\n\t"
        "umulh	x6, x20, x22\n\t"
        "adds	x14, x14, x5\n\t"
        "adc	x15, x15, x6\n\t"
        "#  A[2] * A[3]\n\t"
        "mul	x5, x21, x22\n\t"
        "umulh	x6, x21, x22\n\t"
        "adds	x15, x15, x5\n\t"
        "adc	x16, xzr, x6\n\t"
        "# Double\n\t"
        "adds	x11, x11, x11\n\t"
        "adcs	x12, x12, x12\n\t"
        "adcs	x13, x13, x13\n\t"
        "adcs	x14, x14, x14\n\t"
        "adcs	x15, x15, x15\n\t"
        "adcs	x16, x16, x16\n\t"
        "cset	x17, cs\n\t"
        "#  A[0] * A[0]\n\t"
        "mul	x10, x19, x19\n\t"
        "umulh	x4, x19, x19\n\t"
        "#  A[1] * A[1]\n\t"
        "mul	x5, x20, x20\n\t"
        "umulh	x6, x20, x20\n\t"
        "#  A[2] * A[2]\n\t"
        "mul	x7, x21, x21\n\t"
        "umulh	x8, x21, x21\n\t"
        "#  A[3] * A[3]\n\t"
        "mul	x9, x22, x22\n\t"
        "umulh	x19, x22, x22\n\t"
        "adds	x11, x11, x4\n\t"
        "adcs	x12, x12, x5\n\t"
        "adcs	x13, x13, x6\n\t"
        "adcs	x14, x14, x7\n\t"
        "adcs	x15, x15, x8\n\t"
        "adcs	x16, x16, x9\n\t"
        "adc	x17, x17, x19\n\t"
        "# Start Reduction\n\t"
        "mov	x5, x10\n\t"
        "mov	x6, x11\n\t"
        "mov	x7, x12\n\t"
        "mov	x8, x13\n\t"
        "# mu = a[0]-a[3] + a[0]-a[2] << 32 << 64 + (a[0] * 2) << 192\n\t"
        "#    - a[0] << 32 << 192\n\t"
        "#   + (a[0] * 2) << 192\n\t"
        "add	x8, x8, x10\n\t"
        "add	x8, x8, x10\n\t"
        "#   a[0]-a[2] << 32\n\t"
        "lsl	x10, x10, 32\n\t"
        "lsr	x19, x5, 32\n\t"
        "lsl	x11, x6, 32\n\t"
        "lsr	x20, x6, 32\n\t"
        "lsl	x12, x7, 32\n\t"
        "eor	x11, x11, x19\n\t"
        "eor	x12, x12, x20\n\t"
        "#   - a[0] << 32 << 192\n\t"
        "sub	x8, x8, x10\n\t"
        "#   + a[0]-a[2] << 32 << 64\n\t"
        "adds	x6, x6, x10\n\t"
        "adcs	x7, x7, x11\n\t"
        "adc	x8, x8, x12\n\t"
        "# a += (mu << 256) - (mu << 224) + (mu << 192) + (mu << 96) - mu\n\t"
        "#   a += mu << 256\n\t"
        "adds	x14, x14, x5\n\t"
        "adcs	x15, x15, x6\n\t"
        "adcs	x16, x16, x7\n\t"
        "adcs	x17, x17, x8\n\t"
        "csetm	x10, cs\n\t"
        "#   a += mu << 192\n\t"
        "adds	x13, x13, x5\n\t"
        "adcs	x14, x14, x6\n\t"
        "adcs	x15, x15, x7\n\t"
        "adcs	x16, x16, x8\n\t"
        "adcs	x17, x17, xzr\n\t"
        "csetm	x21, cs\n\t"
        "add	x10, x10, x21\n\t"
        "# mu <<= 32\n\t"
        "lsr	x9, x8, 32\n\t"
        "lsr	x19, x5, 32\n\t"
        "lsl	x5, x5, 32\n\t"
        "lsr	x20, x6, 32\n\t"
        "lsl	x6, x6, 32\n\t"
        "lsr	x21, x7, 32\n\t"
        "lsl	x7, x7, 32\n\t"
        "lsl	x8, x8, 32\n\t"
        "eor	x6, x6, x19\n\t"
        "eor	x7, x7, x20\n\t"
        "eor	x8, x8, x21\n\t"
        "#   a += (mu << 32) << 64\n\t"
        "adds	x13, x13, x7\n\t"
        "adcs	x14, x14, x8\n\t"
        "adcs	x15, x15, x9\n\t"
        "adcs	x16, x16, xzr\n\t"
        "adcs	x17, x17, xzr\n\t"
        "csetm	x21, cs\n\t"
        "add	x10, x10, x21\n\t"
        "#   a -= (mu << 32) << 192\n\t"
        "subs	x13, x13, x5\n\t"
        "mov	x19, 0xffffffff\n\t"
        "sbcs	x14, x14, x6\n\t"
        "mov	x20, 0xffffffff00000001\n\t"
        "sbcs	x15, x15, x7\n\t"
        "sbcs	x16, x16, x8\n\t"
        "sbcs	x17, x17, x9\n\t"
        "cset	x21, cc\n\t"
        "add	x10, x10, x21\n\t"
        "# mask m and sub from result if overflow\n\t"
        "#  m[0] = -1 & mask = mask\n\t"
        "and	x19, x19, x10\n\t"
        "#  m[2] =  0 & mask = 0\n\t"
        "and	x20, x20, x10\n\t"
        "subs	x14, x14, x10\n\t"
        "sbcs	x15, x15, x19\n\t"
        "sbcs	x16, x16, xzr\n\t"
        "sbc	x17, x17, x20\n\t"
        "str	x14, [%[r], 0]\n\t"
        "str	x15, [%[r], 8]\n\t"
        "str	x16, [%[r], 16]\n\t"
        "str	x17, [%[r], 24]\n\t"
        : [m] "+r" (m), [a] "+r" (a), [mp] "+r" (mp)
        : [r] "r" (r)
        : "memory", "x4", "x5", "x6", "x7", "x8", "x9",
          "x19", "x20", "x21", "x22",
          "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17"
    );
}

#if !defined(WOLFSSL_SP_SMALL) || defined(HAVE_COMP_KEY)
/* Square the Montgomery form number a number of times. (r = a ^ n mod m)
 *
 * r   Result of squaring.
 * a   Number to square in Montogmery form.
 * n   Number of times to square.
 * m   Modulus (prime).
 * mp  Montogmery mulitplier.
 */
static void sp_256_mont_sqr_n_4(sp_digit* r, const sp_digit* a, int n,
        const sp_digit* m, sp_digit mp)
{
    sp_256_mont_sqr_4(r, a, m, mp);
    for (; n > 1; n--) {
        sp_256_mont_sqr_4(r, r, m, mp);
    }
}

#endif /* !WOLFSSL_SP_SMALL || HAVE_COMP_KEY */
#ifdef WOLFSSL_SP_SMALL
/* Mod-2 for the P256 curve. */
static const uint64_t p256_mod_2[4] = {
    0xfffffffffffffffdU,0x00000000ffffffffU,0x0000000000000000U,
    0xffffffff00000001U
};
#endif /* !WOLFSSL_SP_SMALL */

/* Invert the number, in Montgomery form, modulo the modulus (prime) of the
 * P256 curve. (r = 1 / a mod m)
 *
 * r   Inverse result.
 * a   Number to invert.
 * td  Temporary data.
 */
static void sp_256_mont_inv_4(sp_digit* r, const sp_digit* a, sp_digit* td)
{
#ifdef WOLFSSL_SP_SMALL
    sp_digit* t = td;
    int i;

    XMEMCPY(t, a, sizeof(sp_digit) * 4);
    for (i=254; i>=0; i--) {
        sp_256_mont_sqr_4(t, t, p256_mod, p256_mp_mod);
        if (p256_mod_2[i / 64] & ((sp_digit)1 << (i % 64)))
            sp_256_mont_mul_4(t, t, a, p256_mod, p256_mp_mod);
    }
    XMEMCPY(r, t, sizeof(sp_digit) * 4);
#else
    sp_digit* t = td;
    sp_digit* t2 = td + 2 * 4;
    sp_digit* t3 = td + 4 * 4;

    /* t = a^2 */
    sp_256_mont_sqr_4(t, a, p256_mod, p256_mp_mod);
    /* t = a^3 = t * a */
    sp_256_mont_mul_4(t, t, a, p256_mod, p256_mp_mod);
    /* t2= a^c = t ^ 2 ^ 2 */
    sp_256_mont_sqr_n_4(t2, t, 2, p256_mod, p256_mp_mod);
    /* t3= a^d = t2 * a */
    sp_256_mont_mul_4(t3, t2, a, p256_mod, p256_mp_mod);
    /* t = a^f = t2 * t */
    sp_256_mont_mul_4(t, t2, t, p256_mod, p256_mp_mod);
    /* t2= a^f0 = t ^ 2 ^ 4 */
    sp_256_mont_sqr_n_4(t2, t, 4, p256_mod, p256_mp_mod);
    /* t3= a^fd = t2 * t3 */
    sp_256_mont_mul_4(t3, t2, t3, p256_mod, p256_mp_mod);
    /* t = a^ff = t2 * t */
    sp_256_mont_mul_4(t, t2, t, p256_mod, p256_mp_mod);
    /* t2= a^ff00 = t ^ 2 ^ 8 */
    sp_256_mont_sqr_n_4(t2, t, 8, p256_mod, p256_mp_mod);
    /* t3= a^fffd = t2 * t3 */
    sp_256_mont_mul_4(t3, t2, t3, p256_mod, p256_mp_mod);
    /* t = a^ffff = t2 * t */
    sp_256_mont_mul_4(t, t2, t, p256_mod, p256_mp_mod);
    /* t2= a^ffff0000 = t ^ 2 ^ 16 */
    sp_256_mont_sqr_n_4(t2, t, 16, p256_mod, p256_mp_mod);
    /* t3= a^fffffffd = t2 * t3 */
    sp_256_mont_mul_4(t3, t2, t3, p256_mod, p256_mp_mod);
    /* t = a^ffffffff = t2 * t */
    sp_256_mont_mul_4(t, t2, t, p256_mod, p256_mp_mod);
    /* t = a^ffffffff00000000 = t ^ 2 ^ 32  */
    sp_256_mont_sqr_n_4(t2, t, 32, p256_mod, p256_mp_mod);
    /* t2= a^ffffffffffffffff = t2 * t */
    sp_256_mont_mul_4(t, t2, t, p256_mod, p256_mp_mod);
    /* t2= a^ffffffff00000001 = t2 * a */
    sp_256_mont_mul_4(t2, t2, a, p256_mod, p256_mp_mod);
    /* t2= a^ffffffff000000010000000000000000000000000000000000000000
     *   = t2 ^ 2 ^ 160 */
    sp_256_mont_sqr_n_4(t2, t2, 160, p256_mod, p256_mp_mod);
    /* t2= a^ffffffff00000001000000000000000000000000ffffffffffffffff
     *   = t2 * t */
    sp_256_mont_mul_4(t2, t2, t, p256_mod, p256_mp_mod);
    /* t2= a^ffffffff00000001000000000000000000000000ffffffffffffffff00000000
     *   = t2 ^ 2 ^ 32 */
    sp_256_mont_sqr_n_4(t2, t2, 32, p256_mod, p256_mp_mod);
    /* r = a^ffffffff00000001000000000000000000000000fffffffffffffffffffffffd
     *   = t2 * t3 */
    sp_256_mont_mul_4(r, t2, t3, p256_mod, p256_mp_mod);
#endif /* WOLFSSL_SP_SMALL */
}

/* Map the Montgomery form projective co-ordinate point to an affine point.
 *
 * r  Resulting affine co-ordinate point.
 * p  Montgomery form projective co-ordinate point.
 * t  Temporary ordinate data.
 */
static void sp_256_map_4(sp_point* r, const sp_point* p, sp_digit* t)
{
    sp_digit* t1 = t;
    sp_digit* t2 = t + 2*4;
    int64_t n;

    sp_256_mont_inv_4(t1, p->z, t + 2*4);

    sp_256_mont_sqr_4(t2, t1, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t1, t2, t1, p256_mod, p256_mp_mod);

    /* x /= z^2 */
    sp_256_mont_mul_4(r->x, p->x, t2, p256_mod, p256_mp_mod);
    XMEMSET(r->x + 4, 0, sizeof(r->x) / 2U);
    sp_256_mont_reduce_4(r->x, p256_mod, p256_mp_mod);
    /* Reduce x to less than modulus */
    n = sp_256_cmp_4(r->x, p256_mod);
    sp_256_cond_sub_4(r->x, r->x, p256_mod, 0 - ((n >= 0) ?
                (sp_digit)1 : (sp_digit)0));
    sp_256_norm_4(r->x);

    /* y /= z^3 */
    sp_256_mont_mul_4(r->y, p->y, t1, p256_mod, p256_mp_mod);
    XMEMSET(r->y + 4, 0, sizeof(r->y) / 2U);
    sp_256_mont_reduce_4(r->y, p256_mod, p256_mp_mod);
    /* Reduce y to less than modulus */
    n = sp_256_cmp_4(r->y, p256_mod);
    sp_256_cond_sub_4(r->y, r->y, p256_mod, 0 - ((n >= 0) ?
                (sp_digit)1 : (sp_digit)0));
    sp_256_norm_4(r->y);

    XMEMSET(r->z, 0, sizeof(r->z));
    r->z[0] = 1;

}

/* Add two Montgomery form numbers (r = a + b % m).
 *
 * r   Result of addition.
 * a   First number to add in Montogmery form.
 * b   Second number to add in Montogmery form.
 * m   Modulus (prime).
 */
static void sp_256_mont_add_4(sp_digit* r, const sp_digit* a, const sp_digit* b,
        const sp_digit* m)
{
    __asm__ __volatile__ (
        "ldr	x4, [%[a],0]\n\t"
        "ldr	x5, [%[a],8]\n\t"
        "ldr	x6, [%[a],16]\n\t"
        "ldr	x7, [%[a],24]\n\t"
        "ldr	x8, [%[b],0]\n\t"
        "ldr	x9, [%[b],8]\n\t"
        "ldr	x10, [%[b],16]\n\t"
        "ldr	x11, [%[b],24]\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "mov	x12, 0xffffffff\n\t"
        "adcs	x6, x6, x10\n\t"
        "adcs	x7, x7, x11\n\t"
        "mov	x13, 0xffffffff00000001\n\t"
        "csetm	x14, cs\n\t"
        "and	x12, x12, x14\n\t"
        "and	x13, x13, x14\n\t"
        "subs	x4, x4, x14\n\t"
        "sbcs	x5, x5, x12\n\t"
        "str	x4, [%[r],0]\n\t"
        "sbcs	x6, x6, xzr\n\t"
        "str	x5, [%[r],8]\n\t"
        "sbc	x7, x7, x13\n\t"
        "str	x6, [%[r],16]\n\t"
        "str	x7, [%[r],24]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14"
    );
}

/* Double a Montgomery form number (r = a + a % m).
 *
 * r   Result of doubling.
 * a   Number to double in Montogmery form.
 * m   Modulus (prime).
 */
static void sp_256_mont_dbl_4(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    __asm__ __volatile__ (
        "ldr	x3, [%[a]]\n\t"
        "ldr	x4, [%[a],8]\n\t"
        "ldr	x5, [%[a],16]\n\t"
        "ldr	x6, [%[a],24]\n\t"
        "adds	x3, x3, x3\n\t"
        "adcs	x4, x4, x4\n\t"
        "mov	x7, 0xffffffff\n\t"
        "adcs	x5, x5, x5\n\t"
        "mov	x8, 0xffffffff00000001\n\t"
        "adcs	x6, x6, x6\n\t"
        "csetm	x9, cs\n\t"
        "and	x7, x7, x9\n\t"
        "and	x8, x8, x9\n\t"
        "subs	x3, x3, x9\n\t"
        "sbcs	x4, x4, x7\n\t"
        "str	x3, [%[r],0]\n\t"
        "sbcs	x5, x5, xzr\n\t"
        "str	x4, [%[r],8]\n\t"
        "sbc	x6, x6, x8\n\t"
        "str	x5, [%[r],16]\n\t"
        "str	x6, [%[r],24]\n\t"
        :
        : [r] "r" (r), [a] "r" (a)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    (void)m;
}

/* Triple a Montgomery form number (r = a + a + a % m).
 *
 * r   Result of Tripling.
 * a   Number to triple in Montogmery form.
 * m   Modulus (prime).
 */
static void sp_256_mont_tpl_4(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    __asm__ __volatile__ (
        "ldr	x10, [%[a]]\n\t"
        "ldr	x11, [%[a],8]\n\t"
        "ldr	x12, [%[a],16]\n\t"
        "ldr	x13, [%[a],24]\n\t"
        "adds	x3, x10, x10\n\t"
        "adcs	x4, x11, x11\n\t"
        "mov	x7, 0xffffffff\n\t"
        "adcs	x5, x12, x12\n\t"
        "mov	x8, 0xffffffff00000001\n\t"
        "adcs	x6, x13, x13\n\t"
        "csetm	x9, cs\n\t"
        "and	x7, x7, x9\n\t"
        "and	x8, x8, x9\n\t"
        "subs	x3, x3, x9\n\t"
        "sbcs	x4, x4, x7\n\t"
        "sbcs	x5, x5, xzr\n\t"
        "sbc	x6, x6, x8\n\t"
        "adds	x3, x3, x10\n\t"
        "adcs	x4, x4, x11\n\t"
        "mov	x7, 0xffffffff\n\t"
        "adcs	x5, x5, x12\n\t"
        "mov	x8, 0xffffffff00000001\n\t"
        "adcs	x6, x6, x13\n\t"
        "csetm	x9, cs\n\t"
        "and	x7, x7, x9\n\t"
        "and	x8, x8, x9\n\t"
        "subs	x3, x3, x9\n\t"
        "sbcs	x4, x4, x7\n\t"
        "sbcs	x5, x5, xzr\n\t"
        "sbc	x6, x6, x8\n\t"
        "str	x3, [%[r], 0]\n\t"
        "str	x4, [%[r], 8]\n\t"
        "str	x5, [%[r], 16]\n\t"
        "str	x6, [%[r], 24]\n\t"
        :
        : [r] "r" (r), [a] "r" (a)
        : "memory", "x10", "x11", "x12", "x13", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    (void)m;
}

/* Subtract two Montgomery form numbers (r = a - b % m).
 *
 * r   Result of subtration.
 * a   Number to subtract from in Montogmery form.
 * b   Number to subtract with in Montogmery form.
 * m   Modulus (prime).
 */
static void sp_256_mont_sub_4(sp_digit* r, const sp_digit* a, const sp_digit* b,
        const sp_digit* m)
{
    __asm__ __volatile__ (
        "ldr	x4, [%[a],0]\n\t"
        "ldr	x5, [%[a],8]\n\t"
        "ldr	x6, [%[a],16]\n\t"
        "ldr	x7, [%[a],24]\n\t"
        "ldr	x8, [%[b],0]\n\t"
        "ldr	x9, [%[b],8]\n\t"
        "ldr	x10, [%[b],16]\n\t"
        "ldr	x11, [%[b],24]\n\t"
        "subs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "mov	x12, 0xffffffff\n\t"
        "sbcs	x6, x6, x10\n\t"
        "sbcs	x7, x7, x11\n\t"
        "mov	x13, 0xffffffff00000001\n\t"
        "csetm	x14, cc\n\t"
        "and	x12, x12, x14\n\t"
        "and	x13, x13, x14\n\t"
        "adds	x4, x4, x14\n\t"
        "adcs	x5, x5, x12\n\t"
        "str	x4, [%[r],0]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "str	x5, [%[r],8]\n\t"
        "adc	x7, x7, x13\n\t"
        "str	x6, [%[r],16]\n\t"
        "str	x7, [%[r],24]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14"
    );
}

/* Divide the number by 2 mod the modulus (prime). (r = a / 2 % m)
 *
 * r  Result of division by 2.
 * a  Number to divide.
 * m  Modulus (prime).
 */
static void sp_256_div2_4(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    __asm__ __volatile__ (
        "ldr	x3, [%[a], 0]\n\t"
        "ldr	x4, [%[a], 8]\n\t"
        "ldr	x5, [%[a], 16]\n\t"
        "ldr	x6, [%[a], 24]\n\t"
        "and	x9, x3, 1\n\t"
        "sub	x10, xzr, x9\n\t"
        "and	x7, x10, 0xffffffff\n\t"
        "and	x8, x10, 0xffffffff00000001\n\t"
        "adds	x3, x3, x10\n\t"
        "adcs	x4, x4, x7\n\t"
        "adcs	x5, x5, xzr\n\t"
        "adcs	x6, x6, x8\n\t"
        "cset	x9, cs\n\t"
        "lsr	x3, x3, 1\n\t"
        "lsr	x7, x4, 1\n\t"
        "lsr	x8, x5, 1\n\t"
        "lsr	x10, x6, 1\n\t"
        "orr	x3, x3, x4, lsl 63\n\t"
        "orr	x4, x7, x5, lsl 63\n\t"
        "orr	x5, x8, x6, lsl 63\n\t"
        "orr	x6, x10, x9, lsl 63\n\t"
        "str	x3, [%[r], 0]\n\t"
        "str	x4, [%[r], 8]\n\t"
        "str	x5, [%[r], 16]\n\t"
        "str	x6, [%[r], 24]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [m] "r" (m)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

}

/* Double the Montgomery form projective point p.
 *
 * r  Result of doubling point.
 * p  Point to double.
 * t  Temporary ordinate data.
 */
static void sp_256_proj_point_dbl_4(sp_point* r, const sp_point* p, sp_digit* t)
{
    sp_point* rp[2];
    sp_digit* t1 = t;
    sp_digit* t2 = t + 2*4;
    sp_digit* x;
    sp_digit* y;
    sp_digit* z;
    int i;

    /* When infinity don't double point passed in - constant time. */
    rp[0] = r;

    /*lint allow cast to different type of pointer*/
    rp[1] = (sp_point*)t; /*lint !e9087 !e740*/
    XMEMSET(rp[1], 0, sizeof(sp_point));
    x = rp[p->infinity]->x;
    y = rp[p->infinity]->y;
    z = rp[p->infinity]->z;
    /* Put point to double into result - good for infinty. */
    if (r != p) {
        for (i=0; i<4; i++) {
            r->x[i] = p->x[i];
        }
        for (i=0; i<4; i++) {
            r->y[i] = p->y[i];
        }
        for (i=0; i<4; i++) {
            r->z[i] = p->z[i];
        }
        r->infinity = p->infinity;
    }

    /* T1 = Z * Z */
    sp_256_mont_sqr_4(t1, z, p256_mod, p256_mp_mod);
    /* Z = Y * Z */
    sp_256_mont_mul_4(z, y, z, p256_mod, p256_mp_mod);
    /* Z = 2Z */
    sp_256_mont_dbl_4(z, z, p256_mod);
    /* T2 = X - T1 */
    sp_256_mont_sub_4(t2, x, t1, p256_mod);
    /* T1 = X + T1 */
    sp_256_mont_add_4(t1, x, t1, p256_mod);
    /* T2 = T1 * T2 */
    sp_256_mont_mul_4(t2, t1, t2, p256_mod, p256_mp_mod);
    /* T1 = 3T2 */
    sp_256_mont_tpl_4(t1, t2, p256_mod);
    /* Y = 2Y */
    sp_256_mont_dbl_4(y, y, p256_mod);
    /* Y = Y * Y */
    sp_256_mont_sqr_4(y, y, p256_mod, p256_mp_mod);
    /* T2 = Y * Y */
    sp_256_mont_sqr_4(t2, y, p256_mod, p256_mp_mod);
    /* T2 = T2/2 */
    sp_256_div2_4(t2, t2, p256_mod);
    /* Y = Y * X */
    sp_256_mont_mul_4(y, y, x, p256_mod, p256_mp_mod);
    /* X = T1 * T1 */
    sp_256_mont_mul_4(x, t1, t1, p256_mod, p256_mp_mod);
    /* X = X - Y */
    sp_256_mont_sub_4(x, x, y, p256_mod);
    /* X = X - Y */
    sp_256_mont_sub_4(x, x, y, p256_mod);
    /* Y = Y - X */
    sp_256_mont_sub_4(y, y, x, p256_mod);
    /* Y = Y * T1 */
    sp_256_mont_mul_4(y, y, t1, p256_mod, p256_mp_mod);
    /* Y = Y - T2 */
    sp_256_mont_sub_4(y, y, t2, p256_mod);

}

/* Double the Montgomery form projective point p a number of times.
 *
 * r  Result of repeated doubling of point.
 * p  Point to double.
 * n  Number of times to double
 * t  Temporary ordinate data.
 */
static void sp_256_proj_point_dbl_n_4(sp_point* r, const sp_point* p, int n,
        sp_digit* t)
{
    sp_point* rp[2];
    sp_digit* w = t;
    sp_digit* a = t + 2*4;
    sp_digit* b = t + 4*4;
    sp_digit* t1 = t + 6*4;
    sp_digit* t2 = t + 8*4;
    sp_digit* x;
    sp_digit* y;
    sp_digit* z;
    int i;

    rp[0] = r;

    /*lint allow cast to different type of pointer*/
    rp[1] = (sp_point*)t; /*lint !e9087 !e740*/
    XMEMSET(rp[1], 0, sizeof(sp_point));
    x = rp[p->infinity]->x;
    y = rp[p->infinity]->y;
    z = rp[p->infinity]->z;
    if (r != p) {
        for (i=0; i<4; i++) {
            r->x[i] = p->x[i];
        }
        for (i=0; i<4; i++) {
            r->y[i] = p->y[i];
        }
        for (i=0; i<4; i++) {
            r->z[i] = p->z[i];
        }
        r->infinity = p->infinity;
    }

    /* Y = 2*Y */
    sp_256_mont_dbl_4(y, y, p256_mod);
    /* W = Z^4 */
    sp_256_mont_sqr_4(w, z, p256_mod, p256_mp_mod);
    sp_256_mont_sqr_4(w, w, p256_mod, p256_mp_mod);
    while (n-- > 0) {
        /* A = 3*(X^2 - W) */
        sp_256_mont_sqr_4(t1, x, p256_mod, p256_mp_mod);
        sp_256_mont_sub_4(t1, t1, w, p256_mod);
        sp_256_mont_tpl_4(a, t1, p256_mod);
        /* B = X*Y^2 */
        sp_256_mont_sqr_4(t2, y, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(b, t2, x, p256_mod, p256_mp_mod);
        /* X = A^2 - 2B */
        sp_256_mont_sqr_4(x, a, p256_mod, p256_mp_mod);
        sp_256_mont_dbl_4(t1, b, p256_mod);
        sp_256_mont_sub_4(x, x, t1, p256_mod);
        /* Z = Z*Y */
        sp_256_mont_mul_4(z, z, y, p256_mod, p256_mp_mod);
        /* t2 = Y^4 */
        sp_256_mont_sqr_4(t2, t2, p256_mod, p256_mp_mod);
        if (n != 0) {
            /* W = W*Y^4 */
            sp_256_mont_mul_4(w, w, t2, p256_mod, p256_mp_mod);
        }
        /* y = 2*A*(B - X) - Y^4 */
        sp_256_mont_sub_4(y, b, x, p256_mod);
        sp_256_mont_mul_4(y, y, a, p256_mod, p256_mp_mod);
        sp_256_mont_dbl_4(y, y, p256_mod);
        sp_256_mont_sub_4(y, y, t2, p256_mod);
    }
    /* Y = Y/2 */
    sp_256_div2_4(y, y, p256_mod);
}

/* Compare two numbers to determine if they are equal.
 * Constant time implementation.
 *
 * a  First number to compare.
 * b  Second number to compare.
 * returns 1 when equal and 0 otherwise.
 */
static int sp_256_cmp_equal_4(const sp_digit* a, const sp_digit* b)
{
    return ((a[0] ^ b[0]) | (a[1] ^ b[1]) | (a[2] ^ b[2]) | (a[3] ^ b[3])) == 0;
}

/* Add two Montgomery form projective points.
 *
 * r  Result of addition.
 * p  Frist point to add.
 * q  Second point to add.
 * t  Temporary ordinate data.
 */
static void sp_256_proj_point_add_4(sp_point* r, const sp_point* p, const sp_point* q,
        sp_digit* t)
{
    const sp_point* ap[2];
    sp_point* rp[2];
    sp_digit* t1 = t;
    sp_digit* t2 = t + 2*4;
    sp_digit* t3 = t + 4*4;
    sp_digit* t4 = t + 6*4;
    sp_digit* t5 = t + 8*4;
    sp_digit* x;
    sp_digit* y;
    sp_digit* z;
    int i;

    /* Ensure only the first point is the same as the result. */
    if (q == r) {
        const sp_point* a = p;
        p = q;
        q = a;
    }

    /* Check double */
    (void)sp_256_sub_4(t1, p256_mod, q->y);
    sp_256_norm_4(t1);
    if ((sp_256_cmp_equal_4(p->x, q->x) & sp_256_cmp_equal_4(p->z, q->z) &
        (sp_256_cmp_equal_4(p->y, q->y) | sp_256_cmp_equal_4(p->y, t1))) != 0) {
        sp_256_proj_point_dbl_4(r, p, t);
    }
    else {
        rp[0] = r;

        /*lint allow cast to different type of pointer*/
        rp[1] = (sp_point*)t; /*lint !e9087 !e740*/
        XMEMSET(rp[1], 0, sizeof(sp_point));
        x = rp[p->infinity | q->infinity]->x;
        y = rp[p->infinity | q->infinity]->y;
        z = rp[p->infinity | q->infinity]->z;

        ap[0] = p;
        ap[1] = q;
        for (i=0; i<4; i++) {
            r->x[i] = ap[p->infinity]->x[i];
        }
        for (i=0; i<4; i++) {
            r->y[i] = ap[p->infinity]->y[i];
        }
        for (i=0; i<4; i++) {
            r->z[i] = ap[p->infinity]->z[i];
        }
        r->infinity = ap[p->infinity]->infinity;

        /* U1 = X1*Z2^2 */
        sp_256_mont_sqr_4(t1, q->z, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t3, t1, q->z, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t1, t1, x, p256_mod, p256_mp_mod);
        /* U2 = X2*Z1^2 */
        sp_256_mont_sqr_4(t2, z, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t4, t2, z, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t2, t2, q->x, p256_mod, p256_mp_mod);
        /* S1 = Y1*Z2^3 */
        sp_256_mont_mul_4(t3, t3, y, p256_mod, p256_mp_mod);
        /* S2 = Y2*Z1^3 */
        sp_256_mont_mul_4(t4, t4, q->y, p256_mod, p256_mp_mod);
        /* H = U2 - U1 */
        sp_256_mont_sub_4(t2, t2, t1, p256_mod);
        /* R = S2 - S1 */
        sp_256_mont_sub_4(t4, t4, t3, p256_mod);
        /* Z3 = H*Z1*Z2 */
        sp_256_mont_mul_4(z, z, q->z, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(z, z, t2, p256_mod, p256_mp_mod);
        /* X3 = R^2 - H^3 - 2*U1*H^2 */
        sp_256_mont_sqr_4(x, t4, p256_mod, p256_mp_mod);
        sp_256_mont_sqr_4(t5, t2, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(y, t1, t5, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t5, t5, t2, p256_mod, p256_mp_mod);
        sp_256_mont_sub_4(x, x, t5, p256_mod);
        sp_256_mont_dbl_4(t1, y, p256_mod);
        sp_256_mont_sub_4(x, x, t1, p256_mod);
        /* Y3 = R*(U1*H^2 - X3) - S1*H^3 */
        sp_256_mont_sub_4(y, y, x, p256_mod);
        sp_256_mont_mul_4(y, y, t4, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t5, t5, t3, p256_mod, p256_mp_mod);
        sp_256_mont_sub_4(y, y, t5, p256_mod);
    }
}

/* Double the Montgomery form projective point p a number of times.
 *
 * r  Result of repeated doubling of point.
 * p  Point to double.
 * n  Number of times to double
 * t  Temporary ordinate data.
 */
static void sp_256_proj_point_dbl_n_store_4(sp_point* r, const sp_point* p,
        int n, int m, sp_digit* t)
{
    sp_digit* w = t;
    sp_digit* a = t + 2*4;
    sp_digit* b = t + 4*4;
    sp_digit* t1 = t + 6*4;
    sp_digit* t2 = t + 8*4;
    sp_digit* x = r[2*m].x;
    sp_digit* y = r[(1<<n)*m].y;
    sp_digit* z = r[2*m].z;
    int i;

    for (i=0; i<4; i++) {
        x[i] = p->x[i];
    }
    for (i=0; i<4; i++) {
        y[i] = p->y[i];
    }
    for (i=0; i<4; i++) {
        z[i] = p->z[i];
    }

    /* Y = 2*Y */
    sp_256_mont_dbl_4(y, y, p256_mod);
    /* W = Z^4 */
    sp_256_mont_sqr_4(w, z, p256_mod, p256_mp_mod);
    sp_256_mont_sqr_4(w, w, p256_mod, p256_mp_mod);
    for (i=1; i<=n; i++) {
        /* A = 3*(X^2 - W) */
        sp_256_mont_sqr_4(t1, x, p256_mod, p256_mp_mod);
        sp_256_mont_sub_4(t1, t1, w, p256_mod);
        sp_256_mont_tpl_4(a, t1, p256_mod);
        /* B = X*Y^2 */
        sp_256_mont_sqr_4(t2, y, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(b, t2, x, p256_mod, p256_mp_mod);
        x = r[(1<<i)*m].x;
        /* X = A^2 - 2B */
        sp_256_mont_sqr_4(x, a, p256_mod, p256_mp_mod);
        sp_256_mont_dbl_4(t1, b, p256_mod);
        sp_256_mont_sub_4(x, x, t1, p256_mod);
        /* Z = Z*Y */
        sp_256_mont_mul_4(r[(1<<i)*m].z, z, y, p256_mod, p256_mp_mod);
        z = r[(1<<i)*m].z;
        /* t2 = Y^4 */
        sp_256_mont_sqr_4(t2, t2, p256_mod, p256_mp_mod);
        if (i != n) {
            /* W = W*Y^4 */
            sp_256_mont_mul_4(w, w, t2, p256_mod, p256_mp_mod);
        }
        /* y = 2*A*(B - X) - Y^4 */
        sp_256_mont_sub_4(y, b, x, p256_mod);
        sp_256_mont_mul_4(y, y, a, p256_mod, p256_mp_mod);
        sp_256_mont_dbl_4(y, y, p256_mod);
        sp_256_mont_sub_4(y, y, t2, p256_mod);

        /* Y = Y/2 */
        sp_256_div2_4(r[(1<<i)*m].y, y, p256_mod);
        r[(1<<i)*m].infinity = 0;
    }
}

/* Add two Montgomery form projective points.
 *
 * ra  Result of addition.
 * rs  Result of subtraction.
 * p   Frist point to add.
 * q   Second point to add.
 * t   Temporary ordinate data.
 */
static void sp_256_proj_point_add_sub_4(sp_point* ra, sp_point* rs,
        const sp_point* p, const sp_point* q, sp_digit* t)
{
    sp_digit* t1 = t;
    sp_digit* t2 = t + 2*4;
    sp_digit* t3 = t + 4*4;
    sp_digit* t4 = t + 6*4;
    sp_digit* t5 = t + 8*4;
    sp_digit* t6 = t + 10*4;
    sp_digit* x = ra->x;
    sp_digit* y = ra->y;
    sp_digit* z = ra->z;
    sp_digit* xs = rs->x;
    sp_digit* ys = rs->y;
    sp_digit* zs = rs->z;


    XMEMCPY(x, p->x, sizeof(p->x) / 2);
    XMEMCPY(y, p->y, sizeof(p->y) / 2);
    XMEMCPY(z, p->z, sizeof(p->z) / 2);
    ra->infinity = 0;
    rs->infinity = 0;

    /* U1 = X1*Z2^2 */
    sp_256_mont_sqr_4(t1, q->z, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t3, t1, q->z, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t1, t1, x, p256_mod, p256_mp_mod);
    /* U2 = X2*Z1^2 */
    sp_256_mont_sqr_4(t2, z, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t4, t2, z, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t2, t2, q->x, p256_mod, p256_mp_mod);
    /* S1 = Y1*Z2^3 */
    sp_256_mont_mul_4(t3, t3, y, p256_mod, p256_mp_mod);
    /* S2 = Y2*Z1^3 */
    sp_256_mont_mul_4(t4, t4, q->y, p256_mod, p256_mp_mod);
    /* H = U2 - U1 */
    sp_256_mont_sub_4(t2, t2, t1, p256_mod);
    /* RS = S2 + S1 */
    sp_256_mont_add_4(t6, t4, t3, p256_mod);
    /* R = S2 - S1 */
    sp_256_mont_sub_4(t4, t4, t3, p256_mod);
    /* Z3 = H*Z1*Z2 */
    /* ZS = H*Z1*Z2 */
    sp_256_mont_mul_4(z, z, q->z, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(z, z, t2, p256_mod, p256_mp_mod);
    XMEMCPY(zs, z, sizeof(p->z)/2);
    /* X3 = R^2 - H^3 - 2*U1*H^2 */
    /* XS = RS^2 - H^3 - 2*U1*H^2 */
    sp_256_mont_sqr_4(x, t4, p256_mod, p256_mp_mod);
    sp_256_mont_sqr_4(xs, t6, p256_mod, p256_mp_mod);
    sp_256_mont_sqr_4(t5, t2, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(y, t1, t5, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t5, t5, t2, p256_mod, p256_mp_mod);
    sp_256_mont_sub_4(x, x, t5, p256_mod);
    sp_256_mont_sub_4(xs, xs, t5, p256_mod);
    sp_256_mont_dbl_4(t1, y, p256_mod);
    sp_256_mont_sub_4(x, x, t1, p256_mod);
    sp_256_mont_sub_4(xs, xs, t1, p256_mod);
    /* Y3 = R*(U1*H^2 - X3) - S1*H^3 */
    /* YS = -RS*(U1*H^2 - XS) - S1*H^3 */
    sp_256_mont_sub_4(ys, y, xs, p256_mod);
    sp_256_mont_sub_4(y, y, x, p256_mod);
    sp_256_mont_mul_4(y, y, t4, p256_mod, p256_mp_mod);
    sp_256_sub_4(t6, p256_mod, t6);
    sp_256_mont_mul_4(ys, ys, t6, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t5, t5, t3, p256_mod, p256_mp_mod);
    sp_256_mont_sub_4(y, y, t5, p256_mod);
    sp_256_mont_sub_4(ys, ys, t5, p256_mod);
}

/* Structure used to describe recoding of scalar multiplication. */
typedef struct ecc_recode {
    /* Index into pre-computation table. */
    uint8_t i;
    /* Use the negative of the point. */
    uint8_t neg;
} ecc_recode;

/* The index into pre-computation table to use. */
static const uint8_t recode_index_4_6[66] = {
     0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
    32, 31, 30, 29, 28, 27, 26, 25, 24, 23, 22, 21, 20, 19, 18, 17,
    16, 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,
     0,  1,
};

/* Whether to negate y-ordinate. */
static const uint8_t recode_neg_4_6[66] = {
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     0,  0,
};

/* Recode the scalar for multiplication using pre-computed values and
 * subtraction.
 *
 * k  Scalar to multiply by.
 * v  Vector of operations to peform.
 */
static void sp_256_ecc_recode_6_4(const sp_digit* k, ecc_recode* v)
{
    int i, j;
    uint8_t y;
    int carry = 0;
    int o;
    sp_digit n;

    j = 0;
    n = k[j];
    o = 0;
    for (i=0; i<43; i++) {
        y = n;
        if (o + 6 < 64) {
            y &= 0x3f;
            n >>= 6;
            o += 6;
        }
        else if (o + 6 == 64) {
            n >>= 6;
            if (++j < 4)
                n = k[j];
            o = 0;
        }
        else if (++j < 4) {
            n = k[j];
            y |= (n << (64 - o)) & 0x3f;
            o -= 58;
            n >>= o;
        }

        y += carry;
        v[i].i = recode_index_4_6[y];
        v[i].neg = recode_neg_4_6[y];
        carry = (y >> 6) + v[i].neg;
    }
}

/* Multiply the point by the scalar and return the result.
 * If map is true then convert result to affine co-ordinates.
 *
 * r     Resulting point.
 * g     Point to multiply.
 * k     Scalar to multiply by.
 * map   Indicates whether to convert result to affine.
 * heap  Heap to use for allocation.
 * returns MEMORY_E when memory allocation fails and MP_OKAY on success.
 */
static int sp_256_ecc_mulmod_win_add_sub_4(sp_point* r, const sp_point* g,
        const sp_digit* k, int map, void* heap)
{
#if !defined(WOLFSSL_SP_SMALL) && !defined(WOLFSSL_SMALL_STACK)
    sp_point td[33];
    sp_point rtd, pd;
    sp_digit tmpd[2 * 4 * 6];
#endif
    sp_point* t;
    sp_point* rt;
    sp_point* p = NULL;
    sp_digit* tmp;
    sp_digit* negy;
    int i;
    ecc_recode v[43];
    int err;

    (void)heap;

    err = sp_ecc_point_new(heap, rtd, rt);
    if (err == MP_OKAY)
        err = sp_ecc_point_new(heap, pd, p);
#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    t = (sp_point*)XMALLOC(sizeof(sp_point) * 33, heap, DYNAMIC_TYPE_ECC);
    if (t == NULL)
        err = MEMORY_E;
    tmp = (sp_digit*)XMALLOC(sizeof(sp_digit) * 2 * 4 * 6, heap,
                             DYNAMIC_TYPE_ECC);
    if (tmp == NULL)
        err = MEMORY_E;
#else
    t = td;
    tmp = tmpd;
#endif


    if (err == MP_OKAY) {
        /* t[0] = {0, 0, 1} * norm */
        XMEMSET(&t[0], 0, sizeof(t[0]));
        t[0].infinity = 1;
        /* t[1] = {g->x, g->y, g->z} * norm */
        err = sp_256_mod_mul_norm_4(t[1].x, g->x, p256_mod);
    }
    if (err == MP_OKAY) {
        err = sp_256_mod_mul_norm_4(t[1].y, g->y, p256_mod);
    }
    if (err == MP_OKAY) {
        err = sp_256_mod_mul_norm_4(t[1].z, g->z, p256_mod);
    }

    if (err == MP_OKAY) {
        t[1].infinity = 0;
        /* t[2] ... t[32]  */
    sp_256_proj_point_dbl_n_store_4(t, &t[ 1], 5, 1, tmp);
    sp_256_proj_point_add_4(&t[ 3], &t[ 2], &t[ 1], tmp);
    sp_256_proj_point_dbl_4(&t[ 6], &t[ 3], tmp);
    sp_256_proj_point_add_sub_4(&t[ 7], &t[ 5], &t[ 6], &t[ 1], tmp);
    sp_256_proj_point_dbl_4(&t[10], &t[ 5], tmp);
    sp_256_proj_point_add_sub_4(&t[11], &t[ 9], &t[10], &t[ 1], tmp);
    sp_256_proj_point_dbl_4(&t[12], &t[ 6], tmp);
    sp_256_proj_point_dbl_4(&t[14], &t[ 7], tmp);
    sp_256_proj_point_add_sub_4(&t[15], &t[13], &t[14], &t[ 1], tmp);
    sp_256_proj_point_dbl_4(&t[18], &t[ 9], tmp);
    sp_256_proj_point_add_sub_4(&t[19], &t[17], &t[18], &t[ 1], tmp);
    sp_256_proj_point_dbl_4(&t[20], &t[10], tmp);
    sp_256_proj_point_dbl_4(&t[22], &t[11], tmp);
    sp_256_proj_point_add_sub_4(&t[23], &t[21], &t[22], &t[ 1], tmp);
    sp_256_proj_point_dbl_4(&t[24], &t[12], tmp);
    sp_256_proj_point_dbl_4(&t[26], &t[13], tmp);
    sp_256_proj_point_add_sub_4(&t[27], &t[25], &t[26], &t[ 1], tmp);
    sp_256_proj_point_dbl_4(&t[28], &t[14], tmp);
    sp_256_proj_point_dbl_4(&t[30], &t[15], tmp);
    sp_256_proj_point_add_sub_4(&t[31], &t[29], &t[30], &t[ 1], tmp);

        negy = t[0].y;

        sp_256_ecc_recode_6_4(k, v);

        i = 42;
        XMEMCPY(rt, &t[v[i].i], sizeof(sp_point));
        for (--i; i>=0; i--) {
            sp_256_proj_point_dbl_n_4(rt, rt, 6, tmp);

            XMEMCPY(p, &t[v[i].i], sizeof(sp_point));
            sp_256_sub_4(negy, p256_mod, p->y);
            sp_256_cond_copy_4(p->y, negy, (sp_digit)0 - v[i].neg);
            sp_256_proj_point_add_4(rt, rt, p, tmp);
        }

        if (map != 0) {
            sp_256_map_4(r, rt, tmp);
        }
        else {
            XMEMCPY(r, rt, sizeof(sp_point));
        }
    }

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    if (t != NULL)
        XFREE(t, heap, DYNAMIC_TYPE_ECC);
    if (tmp != NULL)
        XFREE(tmp, heap, DYNAMIC_TYPE_ECC);
#endif
    sp_ecc_point_free(p, 0, heap);
    sp_ecc_point_free(rt, 0, heap);

    return err;
}

/* A table entry for pre-computed points. */
typedef struct sp_table_entry {
    sp_digit x[4];
    sp_digit y[4];
} sp_table_entry;

#if defined(FP_ECC) || defined(WOLFSSL_SP_SMALL)
#endif /* FP_ECC || WOLFSSL_SP_SMALL */
/* Add two Montgomery form projective points. The second point has a q value of
 * one.
 * Only the first point can be the same pointer as the result point.
 *
 * r  Result of addition.
 * p  Frist point to add.
 * q  Second point to add.
 * t  Temporary ordinate data.
 */
static void sp_256_proj_point_add_qz1_4(sp_point* r, const sp_point* p,
        const sp_point* q, sp_digit* t)
{
    const sp_point* ap[2];
    sp_point* rp[2];
    sp_digit* t1 = t;
    sp_digit* t2 = t + 2*4;
    sp_digit* t3 = t + 4*4;
    sp_digit* t4 = t + 6*4;
    sp_digit* t5 = t + 8*4;
    sp_digit* x;
    sp_digit* y;
    sp_digit* z;
    int i;

    /* Check double */
    (void)sp_256_sub_4(t1, p256_mod, q->y);
    sp_256_norm_4(t1);
    if ((sp_256_cmp_equal_4(p->x, q->x) & sp_256_cmp_equal_4(p->z, q->z) &
        (sp_256_cmp_equal_4(p->y, q->y) | sp_256_cmp_equal_4(p->y, t1))) != 0) {
        sp_256_proj_point_dbl_4(r, p, t);
    }
    else {
        rp[0] = r;

        /*lint allow cast to different type of pointer*/
        rp[1] = (sp_point*)t; /*lint !e9087 !e740*/
        XMEMSET(rp[1], 0, sizeof(sp_point));
        x = rp[p->infinity | q->infinity]->x;
        y = rp[p->infinity | q->infinity]->y;
        z = rp[p->infinity | q->infinity]->z;

        ap[0] = p;
        ap[1] = q;
        for (i=0; i<4; i++) {
            r->x[i] = ap[p->infinity]->x[i];
        }
        for (i=0; i<4; i++) {
            r->y[i] = ap[p->infinity]->y[i];
        }
        for (i=0; i<4; i++) {
            r->z[i] = ap[p->infinity]->z[i];
        }
        r->infinity = ap[p->infinity]->infinity;

        /* U2 = X2*Z1^2 */
        sp_256_mont_sqr_4(t2, z, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t4, t2, z, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t2, t2, q->x, p256_mod, p256_mp_mod);
        /* S2 = Y2*Z1^3 */
        sp_256_mont_mul_4(t4, t4, q->y, p256_mod, p256_mp_mod);
        /* H = U2 - X1 */
        sp_256_mont_sub_4(t2, t2, x, p256_mod);
        /* R = S2 - Y1 */
        sp_256_mont_sub_4(t4, t4, y, p256_mod);
        /* Z3 = H*Z1 */
        sp_256_mont_mul_4(z, z, t2, p256_mod, p256_mp_mod);
        /* X3 = R^2 - H^3 - 2*X1*H^2 */
        sp_256_mont_sqr_4(t1, t4, p256_mod, p256_mp_mod);
        sp_256_mont_sqr_4(t5, t2, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t3, x, t5, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t5, t5, t2, p256_mod, p256_mp_mod);
        sp_256_mont_sub_4(x, t1, t5, p256_mod);
        sp_256_mont_dbl_4(t1, t3, p256_mod);
        sp_256_mont_sub_4(x, x, t1, p256_mod);
        /* Y3 = R*(X1*H^2 - X3) - Y1*H^3 */
        sp_256_mont_sub_4(t3, t3, x, p256_mod);
        sp_256_mont_mul_4(t3, t3, t4, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t5, t5, y, p256_mod, p256_mp_mod);
        sp_256_mont_sub_4(y, t3, t5, p256_mod);
    }
}

#ifdef FP_ECC
/* Convert the projective point to affine.
 * Ordinates are in Montgomery form.
 *
 * a  Point to convert.
 * t  Temprorary data.
 */
static void sp_256_proj_to_affine_4(sp_point* a, sp_digit* t)
{
    sp_digit* t1 = t;
    sp_digit* t2 = t + 2 * 4;
    sp_digit* tmp = t + 4 * 4;

    sp_256_mont_inv_4(t1, a->z, tmp);

    sp_256_mont_sqr_4(t2, t1, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t1, t2, t1, p256_mod, p256_mp_mod);

    sp_256_mont_mul_4(a->x, a->x, t2, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(a->y, a->y, t1, p256_mod, p256_mp_mod);
    XMEMCPY(a->z, p256_norm_mod, sizeof(p256_norm_mod));
}

/* Generate the pre-computed table of points for the base point.
 *
 * a      The base point.
 * table  Place to store generated point data.
 * tmp    Temprorary data.
 * heap  Heap to use for allocation.
 */
static int sp_256_gen_stripe_table_4(const sp_point* a,
        sp_table_entry* table, sp_digit* tmp, void* heap)
{
#if !defined(WOLFSSL_SP_SMALL) && !defined(WOLFSSL_SMALL_STACK)
    sp_point td, s1d, s2d;
#endif
    sp_point* t;
    sp_point* s1 = NULL;
    sp_point* s2 = NULL;
    int i, j;
    int err;

    (void)heap;

    err = sp_ecc_point_new(heap, td, t);
    if (err == MP_OKAY) {
        err = sp_ecc_point_new(heap, s1d, s1);
    }
    if (err == MP_OKAY) {
        err = sp_ecc_point_new(heap, s2d, s2);
    }

    if (err == MP_OKAY) {
        err = sp_256_mod_mul_norm_4(t->x, a->x, p256_mod);
    }
    if (err == MP_OKAY) {
        err = sp_256_mod_mul_norm_4(t->y, a->y, p256_mod);
    }
    if (err == MP_OKAY) {
        err = sp_256_mod_mul_norm_4(t->z, a->z, p256_mod);
    }
    if (err == MP_OKAY) {
        t->infinity = 0;
        sp_256_proj_to_affine_4(t, tmp);

        XMEMCPY(s1->z, p256_norm_mod, sizeof(p256_norm_mod));
        s1->infinity = 0;
        XMEMCPY(s2->z, p256_norm_mod, sizeof(p256_norm_mod));
        s2->infinity = 0;

        /* table[0] = {0, 0, infinity} */
        XMEMSET(&table[0], 0, sizeof(sp_table_entry));
        /* table[1] = Affine version of 'a' in Montgomery form */
        XMEMCPY(table[1].x, t->x, sizeof(table->x));
        XMEMCPY(table[1].y, t->y, sizeof(table->y));

        for (i=1; i<8; i++) {
            sp_256_proj_point_dbl_n_4(t, t, 32, tmp);
            sp_256_proj_to_affine_4(t, tmp);
            XMEMCPY(table[1<<i].x, t->x, sizeof(table->x));
            XMEMCPY(table[1<<i].y, t->y, sizeof(table->y));
        }

        for (i=1; i<8; i++) {
            XMEMCPY(s1->x, table[1<<i].x, sizeof(table->x));
            XMEMCPY(s1->y, table[1<<i].y, sizeof(table->y));
            for (j=(1<<i)+1; j<(1<<(i+1)); j++) {
                XMEMCPY(s2->x, table[j-(1<<i)].x, sizeof(table->x));
                XMEMCPY(s2->y, table[j-(1<<i)].y, sizeof(table->y));
                sp_256_proj_point_add_qz1_4(t, s1, s2, tmp);
                sp_256_proj_to_affine_4(t, tmp);
                XMEMCPY(table[j].x, t->x, sizeof(table->x));
                XMEMCPY(table[j].y, t->y, sizeof(table->y));
            }
        }
    }

    sp_ecc_point_free(s2, 0, heap);
    sp_ecc_point_free(s1, 0, heap);
    sp_ecc_point_free( t, 0, heap);

    return err;
}

#endif /* FP_ECC */
#if defined(FP_ECC) || defined(WOLFSSL_SP_SMALL)
/* Multiply the point by the scalar and return the result.
 * If map is true then convert result to affine co-ordinates.
 *
 * r     Resulting point.
 * k     Scalar to multiply by.
 * map   Indicates whether to convert result to affine.
 * heap  Heap to use for allocation.
 * returns MEMORY_E when memory allocation fails and MP_OKAY on success.
 */
static int sp_256_ecc_mulmod_stripe_4(sp_point* r, const sp_point* g,
        const sp_table_entry* table, const sp_digit* k, int map, void* heap)
{
#if !defined(WOLFSSL_SP_SMALL) && !defined(WOLFSSL_SMALL_STACK)
    sp_point rtd;
    sp_point pd;
    sp_digit td[2 * 4 * 5];
#endif
    sp_point* rt;
    sp_point* p = NULL;
    sp_digit* t;
    int i, j;
    int y, x;
    int err;

    (void)g;
    (void)heap;

    err = sp_ecc_point_new(heap, rtd, rt);
    if (err == MP_OKAY) {
        err = sp_ecc_point_new(heap, pd, p);
    }
#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    t = (sp_digit*)XMALLOC(sizeof(sp_digit) * 2 * 4 * 5, heap,
                           DYNAMIC_TYPE_ECC);
    if (t == NULL) {
        err = MEMORY_E;
    }
#else
    t = td;
#endif

    if (err == MP_OKAY) {
        XMEMCPY(p->z, p256_norm_mod, sizeof(p256_norm_mod));
        XMEMCPY(rt->z, p256_norm_mod, sizeof(p256_norm_mod));

        y = 0;
        for (j=0,x=31; j<8; j++,x+=32) {
            y |= ((k[x / 64] >> (x % 64)) & 1) << j;
        }
        XMEMCPY(rt->x, table[y].x, sizeof(table[y].x));
        XMEMCPY(rt->y, table[y].y, sizeof(table[y].y));
        rt->infinity = !y;
        for (i=30; i>=0; i--) {
            y = 0;
            for (j=0,x=i; j<8; j++,x+=32) {
                y |= ((k[x / 64] >> (x % 64)) & 1) << j;
            }

            sp_256_proj_point_dbl_4(rt, rt, t);
            XMEMCPY(p->x, table[y].x, sizeof(table[y].x));
            XMEMCPY(p->y, table[y].y, sizeof(table[y].y));
            p->infinity = !y;
            sp_256_proj_point_add_qz1_4(rt, rt, p, t);
        }

        if (map != 0) {
            sp_256_map_4(r, rt, t);
        }
        else {
            XMEMCPY(r, rt, sizeof(sp_point));
        }
    }

#if defined(WOLFSSL_SP_SMALL) || defined(WOLFSSL_SMALL_STACK)
    if (t != NULL) {
        XFREE(t, heap, DYNAMIC_TYPE_ECC);
    }
#endif
    sp_ecc_point_free(p, 0, heap);
    sp_ecc_point_free(rt, 0, heap);

    return err;
}

#endif /* FP_ECC || WOLFSSL_SP_SMALL */
#ifdef FP_ECC
#ifndef FP_ENTRIES
    #define FP_ENTRIES 16
#endif

typedef struct sp_cache_t {
    sp_digit x[4];
    sp_digit y[4];
    sp_table_entry table[256];
    uint32_t cnt;
    int set;
} sp_cache_t;

static THREAD_LS_T sp_cache_t sp_cache[FP_ENTRIES];
static THREAD_LS_T int sp_cache_last = -1;
static THREAD_LS_T int sp_cache_inited = 0;

#ifndef HAVE_THREAD_LS
    static volatile int initCacheMutex = 0;
    static wolfSSL_Mutex sp_cache_lock;
#endif

static void sp_ecc_get_cache(const sp_point* g, sp_cache_t** cache)
{
    int i, j;
    uint32_t least;

    if (sp_cache_inited == 0) {
        for (i=0; i<FP_ENTRIES; i++) {
            sp_cache[i].set = 0;
        }
        sp_cache_inited = 1;
    }

    /* Compare point with those in cache. */
    for (i=0; i<FP_ENTRIES; i++) {
        if (!sp_cache[i].set)
            continue;

        if (sp_256_cmp_equal_4(g->x, sp_cache[i].x) &
                           sp_256_cmp_equal_4(g->y, sp_cache[i].y)) {
            sp_cache[i].cnt++;
            break;
        }
    }

    /* No match. */
    if (i == FP_ENTRIES) {
        /* Find empty entry. */
        i = (sp_cache_last + 1) % FP_ENTRIES;
        for (; i != sp_cache_last; i=(i+1)%FP_ENTRIES) {
            if (!sp_cache[i].set) {
                break;
            }
        }

        /* Evict least used. */
        if (i == sp_cache_last) {
            least = sp_cache[0].cnt;
            for (j=1; j<FP_ENTRIES; j++) {
                if (sp_cache[j].cnt < least) {
                    i = j;
                    least = sp_cache[i].cnt;
                }
            }
        }

        XMEMCPY(sp_cache[i].x, g->x, sizeof(sp_cache[i].x));
        XMEMCPY(sp_cache[i].y, g->y, sizeof(sp_cache[i].y));
        sp_cache[i].set = 1;
        sp_cache[i].cnt = 1;
    }

    *cache = &sp_cache[i];
    sp_cache_last = i;
}
#endif /* FP_ECC */

/* Multiply the base point of P256 by the scalar and return the result.
 * If map is true then convert result to affine co-ordinates.
 *
 * r     Resulting point.
 * g     Point to multiply.
 * k     Scalar to multiply by.
 * map   Indicates whether to convert result to affine.
 * heap  Heap to use for allocation.
 * returns MEMORY_E when memory allocation fails and MP_OKAY on success.
 */
static int sp_256_ecc_mulmod_4(sp_point* r, const sp_point* g, const sp_digit* k,
        int map, void* heap)
{
#ifndef FP_ECC
    return sp_256_ecc_mulmod_win_add_sub_4(r, g, k, map, heap);
#else
    sp_digit tmp[2 * 4 * 5];
    sp_cache_t* cache;
    int err = MP_OKAY;

#ifndef HAVE_THREAD_LS
    if (initCacheMutex == 0) {
         wc_InitMutex(&sp_cache_lock);
         initCacheMutex = 1;
    }
    if (wc_LockMutex(&sp_cache_lock) != 0)
       err = BAD_MUTEX_E;
#endif /* HAVE_THREAD_LS */

    if (err == MP_OKAY) {
        sp_ecc_get_cache(g, &cache);
        if (cache->cnt == 2)
            sp_256_gen_stripe_table_4(g, cache->table, tmp, heap);

#ifndef HAVE_THREAD_LS
        wc_UnLockMutex(&sp_cache_lock);
#endif /* HAVE_THREAD_LS */

        if (cache->cnt < 2) {
            err = sp_256_ecc_mulmod_win_add_sub_4(r, g, k, map, heap);
        }
        else {
            err = sp_256_ecc_mulmod_stripe_4(r, g, cache->table, k,
                    map, heap);
        }
    }

    return err;
#endif
}

/* Multiply the point by the scalar and return the result.
 * If map is true then convert result to affine co-ordinates.
 *
 * k