/* Copyright 2016 The Chromium OS Authors. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 */

#include "2sysincludes.h"
#include "2hmac.h"
#include "2sha.h"
#include "bdb_api.h"
#include "bdb_struct.h"
#include "bdb.h"
#include "secrets.h"

const uint8_t secret_constant_a[] = {
	0xad, 0xf8, 0xd1, 0xd9, 0x48, 0xe6, 0xb3, 0xe4, 0xe0, 0xc4,
	0xd8, 0x66, 0x97, 0x95, 0x71, 0xa8, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0x01};

const uint8_t secret_constant_b[] = {
	0xba, 0x9d, 0x1d, 0x8b, 0x12, 0xbd, 0x8d, 0xcd, 0x4c, 0x89,
	0xd8, 0x18, 0x72, 0x98, 0xb5, 0x18, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0x02};

const uint8_t secret_constant_x[] = {
	0xfd, 0xc1, 0xe5, 0x57, 0x34, 0xf4, 0xf6, 0x89, 0x6d, 0x1b,
	0x6f, 0xf2, 0xd0, 0x36, 0xdb, 0xf4, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0x09};

const uint8_t secret_constant_y[] = {
	0x18, 0xef, 0x01, 0x8e, 0xcd, 0x62, 0xf1, 0xb0, 0x2d, 0xd4,
	0x11, 0xa4, 0xb5, 0x6e, 0x38, 0xf6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0x0a};

const uint8_t secret_constant_c[] = {
	0x46, 0xda, 0x52, 0x8d, 0x08, 0x56, 0x14, 0xde, 0x75, 0x9c,
	0x9a, 0xeb, 0x08, 0x93, 0x3d, 0xd6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0x0b};

const uint8_t secret_constant_fv0[] = {
	0x93, 0x32, 0xf7, 0x8d, 0xec, 0x4b, 0x26, 0x2e, 0xb3, 0x5c,
	0x39, 0xd7, 0xfc, 0xc6, 0x9f, 0x09, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0x05};

const uint8_t secret_constant_fv1[] = {
	0x60, 0x8d, 0x96, 0x35, 0xdf, 0xf6, 0x31, 0x67, 0xab, 0xb8,
	0x9f, 0x50, 0x81, 0x28, 0x82, 0xec, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0x06};

const uint8_t secret_constant_kv0[] = {
	0x46, 0x6d, 0xef, 0x2c, 0x05, 0xc9, 0xbf, 0xa9, 0x6b, 0xee,
	0xaa, 0x6c, 0xb9, 0xb4, 0x6d, 0x37, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0x07};

const uint8_t secret_constant_kv1[] = {
	0x0a, 0x9e, 0xc9, 0x20, 0x29, 0xa3, 0x5d, 0xd7, 0x27, 0x55,
	0xb6, 0xa6, 0xb4, 0x80, 0x7c, 0x73, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0x08};

const uint8_t secret_constant_k[] = {
	/*
	 * Digest of kernel data key struct fills first 32 bytes
	 */
	            0x1e, 0x1d, 0xec, 0xf2, 0x6d, 0x27, 0xa6, 0xd9,
	0x67, 0x0f, 0x34, 0xc5, 0xfa, 0x01, 0x68, 0xf6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0x03};

const uint8_t secret_constant_l[] = {
	/*
	 * Digest of kernel data key struct fills first 32 bytes
	 */
	            0x9b, 0xc0, 0x29, 0xd3, 0xc3, 0x90, 0x7f, 0x82,
	0x56, 0xe2, 0x67, 0x79, 0x11, 0x74, 0xbe, 0xd0, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0x04};

const uint8_t secret_constant_p[] = {
	/*
	 * Digest of KDB key struct fills first 32 bytes
	 */
	            0xfe, 0x31, 0xed, 0xed, 0x45, 0xfd, 0x8a, 0x5d,
	0x87, 0x90, 0xac, 0x17, 0x02, 0x89, 0x2c, 0xba, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0x0c};

const uint8_t secret_constant_q[] = {
	/*
	 * Digest of KDB key struct fills first 32 bytes
	 */
	            0xc7, 0x60, 0x83, 0x0f, 0x20, 0x44, 0x5d, 0x9c,
	0x70, 0x96, 0x05, 0x2d, 0x51, 0x4b, 0x15, 0x99, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	0xc6, 0xc6, 0xc6, 0x0d};

/**
 * Get constant with digest
 *
 * This function computes a digest of the given buffer and concatenates it
 * to the given constant.
 *
 * @param buf		Data from which a digest is computed
 * @param buf_size	Size of <buf>
 * @param constant	Buffer containing constant
 * @param out		Buffer where the result is stored
 * @return		BDB_SUCCESS on success or !BDB_SUCCESS otherwise
 */
static int get_constant_with_digest(const uint8_t *buf, uint32_t buf_size,
				    const uint8_t *constant, uint8_t *out)
{
	int digest_size = vb2_digest_size(VB2_HASH_SHA256);
	const struct bdb_key *key = (const struct bdb_key *)buf;

	if (!buf)
		return !BDB_SUCCESS;

	if (bdb_check_key(key, buf_size))
		return !BDB_SUCCESS;

	if (vb2_digest_buffer(buf, buf_size, VB2_HASH_SHA256, out, digest_size))
		return !BDB_SUCCESS;

	memcpy(out + digest_size, constant,
	       BDB_CONSTANT_BLOCK_SIZE - digest_size);

	return BDB_SUCCESS;
}

/**
 * Derive secrets for SP-RO
 *
 * This function extends a BDS to derive secrets as done by SP-RO (a.k.a. mask
 * rom).
 *
 * @param ctx		VBoot context
 * @param type		Type of secret to derive
 * @param wsr		Work secret register
 * @param buf		Data from which a digest is computed
 * @param buf_size	Size of <buf>
 * @param extend	sha256 extension function to be used
 * @return		BDB_SUCCESS on success or BDB_ERROR_* otherwise
 */
static int derive_secret_ro(struct vba_context *ctx, enum bdb_secret_type type,
			    uint8_t *wsr, const uint8_t *buf, uint32_t buf_size,
			    f_extend extend)
{
	uint8_t c[BDB_CONSTANT_BLOCK_SIZE];
	uint8_t *from;
	const uint8_t *by = (const uint8_t *)c;
	uint8_t *to;

	switch (type) {
	case BDB_SECRET_TYPE_WSR:
		from = to = wsr;
		by = secret_constant_x;
		break;
	case BDB_SECRET_TYPE_BDB:
		from = wsr;
		to = ctx->secrets->bdb;
		if (get_constant_with_digest(buf, buf_size,
					     secret_constant_p, c))
			return BDB_ERROR_SECRET_BDB;
		break;
	case BDB_SECRET_TYPE_BOOT_PATH:
		from = wsr;
		to = ctx->secrets->boot_path;
		if (get_constant_with_digest(buf, buf_size,
					     secret_constant_k, c))
			return BDB_ERROR_SECRET_BOOT_PATH;
		break;
	case BDB_SECRET_TYPE_BOOT_VERIFIED:
		from = wsr;
		to = ctx->secrets->boot_verified;
		if (ctx->flags & VBA_CONTEXT_FLAG_BDB_KEY_EFUSED)
			by = secret_constant_fv0;
		else
			by = secret_constant_fv1;
		break;
	case BDB_SECRET_TYPE_NVM_WP:
		from = wsr;
		by = secret_constant_a;
		to = ctx->secrets->nvm_wp;
		break;
	case BDB_SECRET_TYPE_NVM_RW:
		from = ctx->secrets->nvm_wp;
		by = secret_constant_b;
		to = ctx->secrets->nvm_rw;
		break;
	default:
		return BDB_ERROR_SECRET_TYPE;
	}

	if (extend)
		extend(from, by, to);
	else
		vb2_sha256_extend(from, by, to);

	return BDB_SUCCESS;
}

int vba_derive_secret(struct vba_context *ctx, enum bdb_secret_type type,
		      uint8_t *wsr, const uint8_t *buf, uint32_t buf_size)
{
	uint8_t c[BDB_CONSTANT_BLOCK_SIZE];
	uint8_t *from;
	const uint8_t *by = (const uint8_t *)c;
	uint8_t *to;

	switch (type) {
	case BDB_SECRET_TYPE_WSR:
		from = to = wsr;
		by = secret_constant_y;
		break;
	case BDB_SECRET_TYPE_BDB:
		from = to = ctx->secrets->bdb;
		if (get_constant_with_digest(buf, buf_size,
					     secret_constant_q, c))
			return BDB_ERROR_SECRET_BDB;
		break;
	case BDB_SECRET_TYPE_BOOT_PATH:
		from = to = ctx->secrets->boot_path;
		if (get_constant_with_digest(buf, buf_size,
					     secret_constant_l, c))
			return BDB_ERROR_SECRET_BOOT_PATH;
		break;
	case BDB_SECRET_TYPE_BOOT_VERIFIED:
		from = to = ctx->secrets->boot_verified;
		if (ctx->flags & VBA_CONTEXT_FLAG_KERNEL_DATA_KEY_VERIFIED)
			by = secret_constant_kv1;
		else
			by = secret_constant_kv0;
		break;
	case BDB_SECRET_TYPE_BUC:
		from = ctx->secrets->boot_verified;
		by = secret_constant_c;
		to = ctx->secrets->buc;
		break;
	default:
		return BDB_ERROR_SECRET_TYPE;
	}

	vb2_sha256_extend(from, by, to);

	return BDB_SUCCESS;
}

int vba_clear_secret(struct vba_context *ctx, enum bdb_secret_type type)
{
	uint8_t *s;

	switch (type) {
	case BDB_SECRET_TYPE_NVM_RW:
		s = ctx->secrets->nvm_rw;
		break;
	case BDB_SECRET_TYPE_BDB:
		s = ctx->secrets->bdb;
		break;
	case BDB_SECRET_TYPE_BOOT_PATH:
		s = ctx->secrets->boot_path;
		break;
	case BDB_SECRET_TYPE_BOOT_VERIFIED:
		s = ctx->secrets->boot_verified;
		break;
	case BDB_SECRET_TYPE_BUC:
		s = ctx->secrets->buc;
		break;
	default:
		return BDB_ERROR_SECRET_TYPE;
	}

	memset(s, 0, BDB_SECRET_SIZE);
	return BDB_SUCCESS;
}

int vba_extend_secrets_ro(struct vba_context *ctx, const uint8_t *bdb,
			  uint8_t *wsr, f_extend extend)
{
	const struct bdb_key *bdbkey = bdb_get_bdbkey(bdb);
	const struct bdb_key *datakey = bdb_get_datakey(bdb);

	derive_secret_ro(ctx, BDB_SECRET_TYPE_BDB, wsr, (const uint8_t *)bdbkey,
			 bdbkey->struct_size, extend);
	derive_secret_ro(ctx, BDB_SECRET_TYPE_BOOT_PATH, wsr,
			 (const uint8_t *)datakey, datakey->struct_size,
			 extend);
	derive_secret_ro(ctx, BDB_SECRET_TYPE_BOOT_VERIFIED, wsr, NULL, 0,
			 extend);
	derive_secret_ro(ctx, BDB_SECRET_TYPE_NVM_WP, wsr, NULL, 0, extend);
	/* Deriving NVM-RW has to be done after NVM-WP */
	derive_secret_ro(ctx, BDB_SECRET_TYPE_NVM_RW, wsr, NULL, 0, extend);
	/* Extending WSR has to be done last. */
	derive_secret_ro(ctx, BDB_SECRET_TYPE_WSR, wsr, NULL, 0, extend);

	return BDB_SUCCESS;
}
