// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

use std::sync::Arc;

use async_trait::async_trait;
use http::StatusCode;

use crate::raw::*;
use crate::*;

use super::core::ChainsafeCore;
use super::error::parse_error;

pub type ChainsafeWriters = oio::OneShotWriter<ChainsafeWriter>;

pub struct ChainsafeWriter {
    core: Arc<ChainsafeCore>,
    _op: OpWrite,
    path: String,
}

impl ChainsafeWriter {
    pub fn new(core: Arc<ChainsafeCore>, op: OpWrite, path: String) -> Self {
        ChainsafeWriter {
            core,
            _op: op,
            path,
        }
    }
}

#[async_trait]
impl oio::OneShotWrite for ChainsafeWriter {
    async fn write_once(&self, bs: &dyn oio::WriteBuf) -> Result<()> {
        let bs = bs.bytes(bs.remaining());

        let resp = self.core.upload_object(&self.path, bs).await?;

        let status = resp.status();

        match status {
            StatusCode::OK => {
                resp.into_body().consume().await?;
                Ok(())
            }
            _ => Err(parse_error(resp).await?),
        }
    }
}
