# ifndef _RHEO_NEWTON_H
# define _RHEO_NEWTON_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

namespace rheolef { 

/*Class:newton
NAME: @code{newton} -- Newton nonlinear algorithm
@findex newton
@cindex nonlinear problem
@cindex Newton method
DESCRIPTION: 
  @noindent
  Nonlinear Newton algorithm for the resolution of the following problem:
  @example
       F(u) = 0
  @end example
  A simple call to the algorithm writes:
  @example
    my_problem P;
    field uh (Vh);
    newton (P, uh, tol, max_iter);
  @end example
  The @code{my_problem} class may contains methods for the evaluation
  of F (aka residue) and its derivative:
  @example
    class my_problem @{
    public:
      my_problem();
      field residue          (const field& uh) const;
      Float dual_space_norm  (const field& mrh) const;
      void update_derivative (const field& uh) const;
      field derivative_solve (const field& mrh) const;
    @};
  @end example
  The @code{dual_space_norm} returns a scalar from the weighted residual field term
  @code{mrh} returned by the @code{residue} function:
  this scalar is used as stopping criteria for the algorithm.
  The @code{update_derivative} and @code{derivative_solver}
  members are called at each step of the Newton algorithm.
  See the example @code{p_laplacian.h} in the user's documentation for more.
AUTHOR: 
   | Pierre.Saramito@imag.fr
    LJK, 38041 Grenoble cedex 9, France
DATE:   14 oct 2009
METHODS: @newton
End:
*/

//<newton:
template <class Problem, class Field>
int newton (Problem P, Field& uh, Float& tol, size_t& max_iter, odiststream *p_derr = 0) {
    if (p_derr) *p_derr << "# Newton:" << std::endl << "# n r" << std::endl << std::flush;
    for (size_t n = 0; true; n++) {
      Field rh = P.residue(uh);
      Float r = P.dual_space_norm(rh);
      if (p_derr) *p_derr << n << " " << r << std::endl << std::flush;
      if (r <= tol) { tol = r; max_iter = n; return 0; }
      if (n == max_iter) { tol = r; return 1; }
      P.update_derivative (uh);
      Field delta_uh = P.derivative_solve (-rh);
      uh += delta_uh;
    }
}
//>newton:
}// namespace rheolef
# endif // _RHEO_NEWTON_H
