% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/p_significance.R
\name{p_significance.lm}
\alias{p_significance.lm}
\title{Practical Significance (ps)}
\usage{
\method{p_significance}{lm}(x, threshold = "default", ci = 0.95, verbose = TRUE, ...)
}
\arguments{
\item{x}{A statistical model.}

\item{threshold}{The threshold value that separates significant from
negligible effect, which can have following possible values:
\itemize{
\item \code{"default"}, in which case the range is set to \code{0.1} if input is a vector,
and based on \code{\link[bayestestR:rope_range]{rope_range()}} if a (Bayesian) model is provided.
\item a single numeric value (e.g., 0.1), which is used as range around zero
(i.e. the threshold range is set to -0.1 and 0.1, i.e. reflects a symmetric
interval)
\item a numeric vector of length two (e.g., \code{c(-0.2, 0.1)}), useful for
asymmetric intervals.
}}

\item{ci}{Confidence Interval (CI) level. Default to \code{0.95} (\verb{95\%}).}

\item{verbose}{Toggle warnings and messages.}

\item{...}{Arguments passed to other methods, e.g. \code{ci()}.}
}
\value{
A data frame.
}
\description{
Compute the probability of \strong{Practical Significance} (\emph{ps}),
which can be conceptualized as a unidirectional equivalence test. It returns
the probability that an effect is above a given threshold corresponding to a
negligible effect in the median's direction, considering a parameter's \emph{full}
confidence interval. In comparison the the \code{\link[=equivalence_test]{equivalence_test()}} function,
where the \emph{SGPV} (second generation p-value) describes the proportion of the
\emph{full} confidence interval that is \emph{inside} the ROPE, the value returned by
\code{p_significance()} describes the \emph{larger} proportion of the \emph{full} confidence
interval that is \emph{outside} the ROPE. This makes \code{p_significance()} comparable
to \code{\link[bayestestR:p_direction]{bayestestR::p_direction()}}, however, while \code{p_direction()} compares to
a point-null by default, \code{p_significance()} compares to a range-null.
}
\details{
\code{p_significance()} returns the proportion of the \emph{full} confidence
interval range (assuming a normally distributed, equal-tailed interval) that
is outside a certain range (the negligible effect, or ROPE, see argument
\code{threshold}). If there are values of the distribution both below and above
the ROPE, \code{p_significance()} returns the higher probability of a value being
outside the ROPE. Typically, this value should be larger than 0.5 to indicate
practical significance. However, if the range of the negligible effect is
rather large compared to the range of the confidence interval,
\code{p_significance()} will be less than 0.5, which indicates no clear practical
significance.

Note that the assumed interval, which is used to calculate the practical
significance, is an \emph{approximation} of the \emph{full interval} based on the
chosen confidence level. For example, if the 95\% confidence interval of a
coefficient ranges from -1 to 1, the underlying \emph{full (normally distributed)
interval} approximately ranges from -1.9 to 1.9, see also following code:

\if{html}{\out{<div class="sourceCode">}}\preformatted{# simulate full normal distribution
out <- bayestestR::distribution_normal(10000, 0, 0.5)
# range of "full" distribution
range(out)
# range of 95\% CI
round(quantile(out, probs = c(0.025, 0.975)), 2)
}\if{html}{\out{</div>}}

This ensures that the practical significance always refers to the general
compatible parameter space of coefficients. Therefore, the \emph{full interval} is
similar to a Bayesian posterior distribution of an equivalent Bayesian model,
see following code:

\if{html}{\out{<div class="sourceCode">}}\preformatted{library(bayestestR)
library(brms)
m <- lm(mpg ~ gear + wt + cyl + hp, data = mtcars)
m2 <- brm(mpg ~ gear + wt + cyl + hp, data = mtcars)
# probability of significance (ps) for frequentist model
p_significance(m)
# similar to ps of Bayesian models
p_significance(m2)
# similar to ps of simulated draws / bootstrap samples
p_significance(simulate_model(m))
}\if{html}{\out{</div>}}
}
\note{
There is also a \href{https://easystats.github.io/see/articles/bayestestR.html}{\code{plot()}-method}
implemented in the \href{https://easystats.github.io/see/}{\strong{see}-package}.
}
\section{Statistical inference - how to quantify evidence}{

There is no standardized approach to drawing conclusions based on the
available data and statistical models. A frequently chosen but also much
criticized approach is to evaluate results based on their statistical
significance (\emph{Amrhein et al. 2017}).

A more sophisticated way would be to test whether estimated effects exceed
the "smallest effect size of interest", to avoid even the smallest effects
being considered relevant simply because they are statistically significant,
but clinically or practically irrelevant (\emph{Lakens et al. 2018, Lakens 2024}).

A rather unconventional approach, which is nevertheless advocated by various
authors, is to interpret results from classical regression models in terms of
probabilities, similar to the usual approach in Bayesian statistics
(\emph{Greenland et al. 2022; Rafi and Greenland 2020; Schweder 2018; Schweder and
Hjort 2003; Vos 2022}).

The \strong{parameters} package provides several options or functions to aid
statistical inference. These are, for example:
\itemize{
\item \code{\link[=equivalence_test]{equivalence_test()}}, to compute the (conditional) equivalence test for
frequentist models
\item \code{\link[=p_significance]{p_significance()}}, to compute the probability of \emph{practical significance},
which can be conceptualized as a unidirectional equivalence test
\item \code{\link[=p_function]{p_function()}}, or \emph{consonance function}, to compute p-values and
compatibility (confidence) intervals for statistical models
\item the \code{pd} argument (setting \code{pd = TRUE}) in \code{model_parameters()} includes
a column with the \emph{probability of direction}, i.e. the probability that a
parameter is strictly positive or negative. See \code{\link[bayestestR:p_direction]{bayestestR::p_direction()}}
for details.
\item the \code{s_value} argument (setting \code{s_value = TRUE}) in \code{model_parameters()}
replaces the p-values with their related \emph{S}-values (\emph{Rafi and Greenland 2020})
\item finally, it is possible to generate distributions of model coefficients by
generating bootstrap-samples (setting \code{bootstrap = TRUE}) or simulating
draws from model coefficients using \code{\link[=simulate_model]{simulate_model()}}. These samples
can then be treated as "posterior samples" and used in many functions from
the \strong{bayestestR} package.
}

Most of the above shown options or functions derive from methods originally
implemented for Bayesian models (\emph{Makowski et al. 2019}). However, assuming
that model assumptions are met (which means, the model fits well to the data,
the correct model is chosen that reflects the data generating process
(distributional model family) etc.), it seems appropriate to interpret
results from classical frequentist models in a "Bayesian way" (more details:
documentation in \code{\link[=p_function]{p_function()}}).
}

\examples{
\dontshow{if (requireNamespace("bayestestR") && packageVersion("bayestestR") > "0.14.0") (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
data(qol_cancer)
model <- lm(QoL ~ time + age + education, data = qol_cancer)

p_significance(model)
p_significance(model, threshold = c(-0.5, 1.5))

# plot method
if (require("see", quietly = TRUE)) {
  result <- p_significance(model)
  plot(result)
}
\dontshow{\}) # examplesIf}
}
\seealso{
For more details, see \code{\link[bayestestR:p_significance]{bayestestR::p_significance()}}. See also
\code{\link[=equivalence_test]{equivalence_test()}}.
}
