// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_CAST_RECEIVER_RENDERER_PUBLIC_CONTENT_RENDERER_CLIENT_MIXINS_H_
#define COMPONENTS_CAST_RECEIVER_RENDERER_PUBLIC_CONTENT_RENDERER_CLIENT_MIXINS_H_

#include <memory>

#include "base/functional/callback_forward.h"
#include "base/strings/string_piece.h"

namespace blink {
class URLLoaderThrottleProvider;
}  // namespace blink

namespace content {
class RenderFrame;
}  // namespace content

namespace cast_receiver {

// Functions to provide additional ContentRendererClient functionality as
// required for a functioning Cast receiver. The lifetime of this object is
// expected to match that of the ContentRendererClient with which it is
// associated.
//
// TODO(crbug.com/1359580): Use this class in the
// CastRuntimeContentRendererClient.
class ContentRendererClientMixins {
 public:
  using IsCorsExemptHeadersCallback =
      base::RepeatingCallback<bool(base::StringPiece)>;
  static std::unique_ptr<ContentRendererClientMixins> Create(
      IsCorsExemptHeadersCallback is_cors_exempt_header_callback);

  virtual ~ContentRendererClientMixins() = default;

  // To be called by the ContentRendererClient function of the same name.
  virtual void RenderFrameCreated(content::RenderFrame& render_frame) = 0;
  virtual bool DeferMediaLoad(content::RenderFrame& render_frame,
                              base::OnceClosure closure) = 0;
  virtual std::unique_ptr<blink::URLLoaderThrottleProvider>
  CreateURLLoaderThrottleProvider() = 0;

  // Wraps |delegated_load_provider| with a new URLLoaderThrottleProvider to
  // apply any UrlRewriteRules previously set. |delegated_load_provider| may not
  // be empty.
  //
  // Call this from ContentRendererClient::CreateURLLoaderThrottleProvider() as
  // an alternative to calling CreateURLLoaderThrottleProvider() above by
  // passing in a provider which creates any remaining URL Loader Throttles for
  // this ContentRendererClient. The URLRewriteRules providers will be appended
  // to those generated by |delegated_load_provider|.
  virtual std::unique_ptr<blink::URLLoaderThrottleProvider>
  ExtendURLLoaderThrottleProvider(
      std::unique_ptr<blink::URLLoaderThrottleProvider>
          delegated_load_provider = nullptr) = 0;
};

}  // namespace cast_receiver

#endif  // COMPONENTS_CAST_RECEIVER_RENDERER_PUBLIC_CONTENT_RENDERER_CLIENT_MIXINS_H_
