(ns leiningen.debian.dh-clojure-lein-test
  (:require
   [clojure.string :as str]
   [clojure.test :refer [deftest is testing]]
   [debian.dh-clojure-lein.client :as client :refer [add-dep del-dep set-dep]]
   [leiningen.core.project :as proj]
   [leiningen.debian.dh-clojure-lein :as dcl])
  (:import
   [clojure.lang ExceptionInfo]))

;; FIXME: add some plugins and managed-dependencies tests
;; FIXME: test validate-dep-vec or related add/set/del-dep behavior

;; Tests rely on these private bindings
(def dep-vecs->map #'dcl/dep-vecs->map)
(def map->dep-vecs #'dcl/map->dep-vecs)

(deftest profile-handling
  (let [proj
        ^{:active-profiles '(:base :system :user :provided :dev)}
        {:dependencies
         '([org.clojure/clojure "1.x"]
           [nrepl/nrepl "1.0.0" :exclusions ([org.clojure/clojure])])}]
    (is (= [:base :system :user :provided :dev]
           (client/active-profiles proj)))))

(def mini-deps
  ;; [[org.clojure/clojure "42.x"]]
  {{:group-id "org.clojure", :artifact-id "clojure"}
   {:artifact-id "clojure", :group-id "org.clojure", :version "42.x"}})

(deftest add-dep-behavior
  (is (= (merge mini-deps (dep-vecs->map '[[x "2.x"]]))
         (-> (add-dep mini-deps '[x "2.x"]))))
  (is (= (merge mini-deps (dep-vecs->map '[[x "2.x" :exclusions [y]]]))
         (-> (add-dep mini-deps '[x "2.x" :exclusions [y]]))))
  (is (thrown-with-msg?
       ExceptionInfo #"add-dep applied to existing dependency org.clojure"
       (-> (add-dep mini-deps '[org.clojure/clojure "42.x"]))))
  (is (thrown-with-msg?
       ExceptionInfo #"add-dep applied to existing dependency org.clojure"
       (-> (add-dep (assoc-in mini-deps
                              [{:group-id "org.clojure", :artifact-id "clojure"}
                               :exclusions]
                              (dep-vecs->map '[x]))
                    '[org.clojure/clojure "42.x"])))))

(deftest del-dep-behavior
  (is (= (merge mini-deps (dep-vecs->map '[[y "2.x"]]))
         (-> (merge mini-deps (dep-vecs->map '[[x "1.x"] [y "2.x"]]))
             (del-dep 'x))))
  (is (= (merge mini-deps (dep-vecs->map '[[y "2.x"]]))
         (-> (merge mini-deps (dep-vecs->map '[[x "1.x" :classifier "test"]
                                               [y "2.x"]]))
             (del-dep '[x :classifier "test"]))))
  (is (= (merge mini-deps (dep-vecs->map '[[y "2.x"]]))
         (-> (merge mini-deps (dep-vecs->map '[[x "1.x" :extension "jar"]
                                               [y "2.x"]]))
             (del-dep '[x :extension "jar"]))))
  (is (thrown-with-msg?
       ExceptionInfo #"del-dep applied to missing dependency x"
       (del-dep mini-deps 'x)))
  (is (thrown-with-msg?
       ExceptionInfo #"del-dep applied to missing dependency x"
       (-> (merge mini-deps (dep-vecs->map '[[x "1.x" :classifier "test"]]))
           (del-dep 'x))))
  (is (thrown-with-msg?
       ExceptionInfo #"del-dep applied to missing dependency x"
       (-> (merge mini-deps (dep-vecs->map '[[x "1.x" :extension "jar"]]))
           (del-dep 'x)))))

(deftest set-dep-behavior
  (is (thrown-with-msg?
       ExceptionInfo #"set-dep applied to missing dependency x"
       (set-dep mini-deps 'x :version "1.x")))

  (is (= (dep-vecs->map '[[x "3.x"]])
         (set-dep (dep-vecs->map '[[x "1.x"]]) 'x :version "3.x")))
  (is (= (dep-vecs->map '[[x "1.x"] [x "3.x" :classifier "test"]])
         (set-dep (dep-vecs->map '[[x "1.x"] [x "2.x" :classifier "test"]])
                  '[x :classifier "test"] :version "3.x")))
  (is (= (dep-vecs->map '[[x "1.x"] [x "3.x" :extension "jar"]])
         (set-dep (dep-vecs->map '[[x "1.x"] [x "2.x" :extension "jar"]])
                  '[x :extension "jar"] :version "3.x")))

  (is (= (dep-vecs->map '[[x "1.x" :exclusions [y]]])
         (set-dep (dep-vecs->map '[[x "1.x"]]) 'x :exclusions '[y])))
  (is (= (dep-vecs->map '[[x "1.x"]])
         (set-dep (dep-vecs->map '[[x "1.x" :exclusions [w y/z]]])
                  'x :exclusions nil)))

  (is (= (dep-vecs->map '[[x "1.x" :classifier "test" :exclusions [y]]])
         (set-dep (dep-vecs->map '[[x "1.x" :classifier "test"]])
                  '[x :classifier "test"] :exclusions '[y])))
  (is (= (dep-vecs->map '[[x "1.x" :classifier "test"]])
         (set-dep (dep-vecs->map '[[x "1.x" :classifier "test"
                                    :exclusions [w y/z]]])
                  '[x :classifier "test"] :exclusions nil)))

  (is (= (dep-vecs->map '[[x "1.x" :extension "jar" :exclusions [y]]])
         (set-dep (dep-vecs->map '[[x "1.x" :extension "jar"]])
                  '[x :extension "jar"] :exclusions '[y])))
  (is (= (dep-vecs->map '[[x "1.x" :extension "jar"]])
         (set-dep (dep-vecs->map '[[x "1.x" :extension "jar"
                                    :exclusions [w y/z]]])
                  '[x :extension "jar"] :exclusions nil))))

(defn lein-read-proj [form]
  (with-in-str (pr-str form)
    (proj/read-raw *in*)))

(deftest exec-config-behavior
  (let [exc #'dcl/exec-config
        proj (lein-read-proj '(defproject p "1.x" :dependencies [[x "2.x"]]))]
    (is (thrown-with-msg?
         ExceptionInfo #"dhclj-adjust-project was not defined"
         (exc '[] proj)))
    (is (= '[[org.clojure/clojure "1.x"]]
           (->> (lein-read-proj
                 '(defproject p "1.x"
                    :dependencies [[org.clojure/clojure "1.12.0"]]))
                (exc nil)
                :dependencies)))
    (is (= '[[x/x "debian"]]
           (->  nil (exc proj) :dependencies)))
    (is (= "new-version"
           (-> '[(defn dhclj-adjust-project [proj]
                   (assoc (proj) :version "new-version"))]
               (exc proj)
               :version)))
    (is (= '[[x/x "debian"] [y/y "3.x"]]
           (-> '[(require '[debian.dh-clojure-lein.client :as deb])
                 (defn dhclj-adjust-project [proj]
                   (update (proj) :dependencies deb/add-dep '[y "3.x"]))]
               (exc proj)
               :dependencies)))
    (is (= []
           (-> '[(require '[debian.dh-clojure-lein.client :as deb])
                 (defn dhclj-adjust-project [proj]
                   (update (proj) :dependencies deb/del-dep 'x))]
               (exc proj)
               :dependencies)))
    (is (= '[[x/x "yep"]]
           (-> '[(require '[debian.dh-clojure-lein.client :as deb])
                 (defn dhclj-adjust-project [proj]
                   (update (proj) :dependencies deb/set-dep 'x :version "yep"))]
               (exc proj)
               :dependencies)))))
