/*
   Copyright The containerd Authors.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package podsandbox

import (
	"context"
	"fmt"

	"github.com/containerd/containerd"
	"github.com/containerd/containerd/errdefs"
	"github.com/containerd/containerd/log"
)

func (c *Controller) Shutdown(ctx context.Context, sandboxID string) error {
	sandbox, err := c.sandboxStore.Get(sandboxID)
	if err != nil {
		if !errdefs.IsNotFound(err) {
			return fmt.Errorf("an error occurred when try to find sandbox %q: %w", sandboxID, err)
		}
		// Do not return error if the id doesn't exist.
		log.G(ctx).Tracef("Sandbox controller Delete called for sandbox %q that does not exist", sandboxID)
		return nil
	}

	// Cleanup the sandbox root directories.
	sandboxRootDir := c.getSandboxRootDir(sandboxID)
	if err := ensureRemoveAll(ctx, sandboxRootDir); err != nil {
		return fmt.Errorf("failed to remove sandbox root directory %q: %w", sandboxRootDir, err)
	}
	volatileSandboxRootDir := c.getVolatileSandboxRootDir(sandboxID)
	if err := ensureRemoveAll(ctx, volatileSandboxRootDir); err != nil {
		return fmt.Errorf("failed to remove volatile sandbox root directory %q: %w",
			volatileSandboxRootDir, err)
	}

	// Delete sandbox container.
	if sandbox.Container != nil {
		if err := sandbox.Container.Delete(ctx, containerd.WithSnapshotCleanup); err != nil {
			if !errdefs.IsNotFound(err) {
				return fmt.Errorf("failed to delete sandbox container %q: %w", sandboxID, err)
			}
			log.G(ctx).Tracef("Sandbox controller Delete called for sandbox container %q that does not exist", sandboxID)
		}
	}

	return nil
}
