/*
 * This file is part of roccat-tools.
 *
 * roccat-tools is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * roccat-tools is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with roccat-tools. If not, see <http://www.gnu.org/licenses/>.
 */

#include "nythconfig_window.h"
#include "nyth_configuration_dialog.h"
#include "nyth_profile_page.h"
#include "nyth_config.h"
#include "nyth_device.h"
#include "nyth_firmware.h"
#include "nyth_profile_data.h"
#include "nyth_dbus_services.h"
#include "nyth_profile.h"
#include "nyth_info.h"
#include "nyth_tcu_dcu_dialog.h"
#include "roccat_helper.h"
#include "roccat_warning_dialog.h"
#include "roccat_save_dialog.h"
#include "roccat_continue_dialog.h"
#include "roccat_info_dialog.h"
#include "roccat_update_assistant.h"
#include "g_roccat_helper.h"
#include "g_dbus_roccat_helper.h"
#include "roccat.h"
#include "i18n.h"

#define NYTHCONFIG_WINDOW_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST((klass), NYTHCONFIG_WINDOW_TYPE, NythconfigWindowClass))
#define IS_NYTHCONFIG_WINDOW_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE((klass), NYTHCONFIG_WINDOW_TYPE))
#define NYTHCONFIG_WINDOW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE((obj), NYTHCONFIG_WINDOW_TYPE, NythconfigWindowPrivate))

typedef struct _NythconfigWindowClass NythconfigWindowClass;
typedef struct _NythconfigWindowPrivate NythconfigWindowPrivate;

struct _NythconfigWindow {
	RoccatConfigWindow parent;
	NythconfigWindowPrivate *priv;
};

struct _NythconfigWindowClass {
	RoccatConfigWindowClass parent_class;
};

struct _NythconfigWindowPrivate {
	NythProfilePage *profile_pages[NYTH_PROFILE_NUM];
	RoccatKeyFile *config;
	DBusGProxy *dbus_proxy;
	guint actual_profile_index;
	gulong device_add_handler_id;
};

G_DEFINE_TYPE(NythconfigWindow, nythconfig_window, ROCCAT_CONFIG_WINDOW_TYPE);

static gboolean change_profile_handler(gpointer user_data) {
	NythconfigWindow *window = NYTHCONFIG_WINDOW(user_data);
	NythconfigWindowPrivate *priv = window->priv;
	guint profile_index = roccat_config_window_get_active_page(ROCCAT_CONFIG_WINDOW(window));
	guint profile_number = profile_index + 1;
	GError *local_error = NULL;

	if (!roccat_config_window_warn_if_no_device(ROCCAT_CONFIG_WINDOW(window)))
		return FALSE;

	if (priv->actual_profile_index != profile_index) {
		priv->actual_profile_index = profile_index;
		nyth_profile_write_index(roccat_config_window_get_device(ROCCAT_CONFIG_WINDOW(window)), profile_index, &local_error);
		if (!roccat_handle_error_dialog(GTK_WINDOW(window), _("Could not activate profile"), &local_error))
			return FALSE;
		nyth_dbus_emit_profile_changed_outside(priv->dbus_proxy, profile_number);
	}

	return FALSE;
}

static void window_active_page_changed_cb(RoccatConfigWindow *roccat_window, gpointer user_data) {
	g_idle_add(change_profile_handler, roccat_window);
}

static void actual_profile_changed_from_device_cb(DBusGProxy *proxy, guchar profile_index, gpointer user_data) {
	NythconfigWindow *window = NYTHCONFIG_WINDOW(user_data);
	NythconfigWindowPrivate *priv = window->priv;

	if (priv->actual_profile_index != profile_index) {
		priv->actual_profile_index = profile_index;
		roccat_config_window_set_active_page(ROCCAT_CONFIG_WINDOW(window), profile_index);
	}
}

static void device_add_cb(RoccatConfigWindow *roccat_window, gpointer user_data) {
	NythconfigWindow *window = NYTHCONFIG_WINDOW(roccat_window);
	NythconfigWindowPrivate *priv = window->priv;
	GError *local_error = NULL;
	guint i;
	NythProfileData *profile_data;
	RoccatDevice *device;

	device = roccat_config_window_get_device(roccat_window);

	priv->actual_profile_index = nyth_profile_read_index(device, &local_error);
	if (!roccat_handle_error_dialog(GTK_WINDOW(window), _("Could not read actual profile"), &local_error))
		return;
	roccat_config_window_set_active_page(roccat_window, priv->actual_profile_index);

	for (i = 0; i < NYTH_PROFILE_NUM; ++i) {

		profile_data = nyth_profile_page_get_profile_data(priv->profile_pages[i]);

		nyth_profile_data_update_filesystem(profile_data, i, &local_error);
		if (!roccat_handle_error_dialog(GTK_WINDOW(window), _("Could not load filesystem data for profile"), &local_error)) {
			g_free(profile_data);
			break;
		}

		nyth_profile_data_update_hardware(profile_data, device, i, &local_error);
		if (!roccat_handle_error_dialog(GTK_WINDOW(window), _("Could not load hardware data for profile"), &local_error)) {
			g_free(profile_data);
			break;
		}

		nyth_profile_page_set_profile_data(priv->profile_pages[i], profile_data);

		g_free(profile_data);
	}

	dbus_g_proxy_connect_signal(priv->dbus_proxy, "ProfileChanged", G_CALLBACK(actual_profile_changed_from_device_cb), window, NULL);
}

static void device_remove_cb(RoccatConfigWindow *roccat_window, gpointer user_data) {
	NythconfigWindow *window = NYTHCONFIG_WINDOW(roccat_window);
	NythconfigWindowPrivate *priv = window->priv;

	dbus_g_proxy_disconnect_signal(priv->dbus_proxy, "ProfileChanged", G_CALLBACK(actual_profile_changed_from_device_cb), window);
}

static gboolean save_unsaved(NythconfigWindow *window, NythProfileData *profile_data, guint profile_index, GError **error) {
	NythconfigWindowPrivate *priv = window->priv;
	RoccatDevice *device = roccat_config_window_get_device(ROCCAT_CONFIG_WINDOW(window));
	guint profile_number = profile_index + 1;

	if (device)
		nyth_profile_data_save(device, profile_data, profile_index, error);
	else
		nyth_profile_data_eventhandler_save(&profile_data->eventhandler, profile_index, error);

	if (*error)
		return FALSE;
	else {
		nyth_dbus_emit_profile_data_changed_outside(priv->dbus_proxy, profile_number);
		nyth_profile_page_set_profile_data(priv->profile_pages[profile_index], profile_data);
		return TRUE;
	}
}

static gboolean display_quit_dialog(NythconfigWindow *window) {
	guint i;
	gboolean modified;
	NythProfileData *profile_datas[NYTH_PROFILE_NUM];
	gboolean retval;
	GError *error = NULL;

	modified = FALSE;
	for (i = 0; i < NYTH_PROFILE_NUM; ++i) {
		profile_datas[i] = nyth_profile_page_get_profile_data(window->priv->profile_pages[i]);
		if (nyth_profile_data_get_modified(profile_datas[i]))
			modified = TRUE;
	}
	if (!modified) {
		for (i = 0; i < NYTH_PROFILE_NUM; ++i)
			g_free(profile_datas[i]);
		return TRUE;
	}

	switch (roccat_save_unsaved_dialog(GTK_WINDOW(window), TRUE)) {
	case GTK_RESPONSE_CANCEL:
		retval = FALSE;
		break;
	case GTK_RESPONSE_ACCEPT:
		retval = TRUE;
		roccat_config_window_warn_if_no_device(ROCCAT_CONFIG_WINDOW(window));
		for (i = 0; i < NYTH_PROFILE_NUM; ++i) {
			if (!save_unsaved(window, profile_datas[i], i, &error)) {
				retval = FALSE;
				break;
			}
		}
		roccat_handle_error_dialog(GTK_WINDOW(window), _("Could not save unsaved data"), &error);
		break;
	default:
		retval = TRUE;
		break;
	}

	for (i = 0; i < NYTH_PROFILE_NUM; ++i)
		g_free(profile_datas[i]);

	return retval;
}

static gboolean delete_event_cb(GtkWidget *window, GdkEvent *event, gpointer user_data) {
	return !display_quit_dialog(NYTHCONFIG_WINDOW(window));
}

static void destroy_event_cb(GtkWidget *window, gpointer user_data) {
	gtk_main_quit();
}

static void save_profile_to_device_cb(RoccatProfilePage *profile_page, gpointer user_data) {
	NythconfigWindow *window = NYTHCONFIG_WINDOW(user_data);
	NythProfileData *profile_data = nyth_profile_page_get_profile_data(NYTH_PROFILE_PAGE(profile_page));
	guint profile_index = roccat_profile_page_get_index(profile_page);
	GError *error = NULL;

	roccat_config_window_warn_if_no_device(ROCCAT_CONFIG_WINDOW(window));
	save_unsaved(window, profile_data, profile_index, &error);
	roccat_handle_error_dialog(GTK_WINDOW(window), _("Could not save unsaved data"), &error);

	g_free(profile_data);
}

static void load_profile_from_file_cb(RoccatProfilePage *profile_page, gpointer user_data) {
	NythconfigWindow *window = NYTHCONFIG_WINDOW(user_data);
	roccat_warning_dialog(GTK_WINDOW(window), _("Not yet implemented"), NULL);
}

static void save_profile_to_file_cb(RoccatProfilePage *profile_page, gpointer user_data) {
	NythconfigWindow *window = NYTHCONFIG_WINDOW(user_data);
	roccat_warning_dialog(GTK_WINDOW(window), _("Not yet implemented"), NULL);
}

static void save_all_cb(RoccatConfigWindow *roccat_window, gpointer user_data) {
	NythconfigWindow *window = NYTHCONFIG_WINDOW(roccat_window);
	NythProfileData *profile_data;
	guint i;
	GError *error = NULL;

	roccat_config_window_warn_if_no_device(roccat_window);
	for (i = 0; i < NYTH_PROFILE_NUM; ++i) {
		profile_data = nyth_profile_page_get_profile_data(window->priv->profile_pages[i]);
		save_unsaved(window, profile_data, i, &error);
		roccat_handle_error_dialog(GTK_WINDOW(window), _("Could not save unsaved data"), &error);

		g_free(profile_data);
	}
}

static void menu_edit_preferences_cb(GtkMenuItem *item, gpointer user_data) {
	NythconfigWindow *window = NYTHCONFIG_WINDOW(user_data);
	NythconfigWindowPrivate *priv = window->priv;
	nyth_configuration_dialog(GTK_WINDOW(window), priv->config);
	if (TRUE)
		nyth_dbus_emit_configuration_changed_outside(priv->dbus_proxy);
}

static void connect_device_handlers(NythconfigWindow *window) {
	NythconfigWindowPrivate *priv = window->priv;
	priv->device_add_handler_id = g_signal_connect(G_OBJECT(window), "device-added", G_CALLBACK(device_add_cb), NULL);
}

static void disconnect_device_handlers(NythconfigWindow *window) {
	NythconfigWindowPrivate *priv = window->priv;
	if (priv->device_add_handler_id) {
		g_signal_handler_disconnect(G_OBJECT(window), priv->device_add_handler_id);
		priv->device_add_handler_id = 0;
	}
}

static void G_GNUC_UNUSED pre_firmware_update_cb(RoccatUpdateAssistant *assistant, gpointer user_data) {
	disconnect_device_handlers(NYTHCONFIG_WINDOW(user_data));
}

static void G_GNUC_UNUSED post_firmware_update_cb(RoccatUpdateAssistant *assistant, gpointer user_data) {
	connect_device_handlers(NYTHCONFIG_WINDOW(user_data));
}

static void G_GNUC_UNUSED menu_device_firmware_update_cb(GtkMenuItem *item, gpointer user_data) {
	NythconfigWindow *window = NYTHCONFIG_WINDOW(user_data);
	GError *local_error = NULL;
	RoccatDevice *nyth;
	GtkWidget *assistant;
	guint firmware_version;
	gchar *version_string;

	if (!roccat_config_window_warn_if_no_device(ROCCAT_CONFIG_WINDOW(window)))
		return;

	nyth = roccat_config_window_get_device(ROCCAT_CONFIG_WINDOW(window));
	firmware_version = nyth_firmware_version_read(nyth, &local_error);
	if (local_error) {
		roccat_warning_dialog(GTK_WINDOW(window), _("Could not read version info"), local_error->message);
		g_error_free(local_error);
		return;
	}

	assistant = roccat_update_assistant_new(GTK_WINDOW(window), nyth);
	g_object_set(G_OBJECT(assistant),
			"wait-0", NYTH_FIRMWARE_UPDATE_WAIT_0,
			"wait-1", NYTH_FIRMWARE_UPDATE_WAIT_1,
			"wait-2", NYTH_FIRMWARE_UPDATE_WAIT_2,
			"wait-34f", NYTH_FIRMWARE_UPDATE_WAIT_34F,
			NULL);

	version_string = roccat_firmware_version_to_string(firmware_version);
	roccat_update_assistant_add_firmware(ROCCAT_UPDATE_ASSISTANT(assistant), _("Main firmware"), version_string,
			NYTH_FIRMWARE_SIZE, ROCCAT_FIRMWARE_NUMBER_DEFAULT);
	g_free(version_string);

	g_signal_connect(G_OBJECT(assistant), "pre-action", G_CALLBACK(pre_firmware_update_cb), window);
	g_signal_connect(G_OBJECT(assistant), "post-action", G_CALLBACK(post_firmware_update_cb), window);

	gtk_widget_show_all(assistant);
}

static void menu_device_info_cb(GtkMenuItem *item, gpointer user_data) {
	NythconfigWindow *window = NYTHCONFIG_WINDOW(user_data);
	RoccatInfoDialog *dialog;
	GError *local_error = NULL;
	NythInfo *info;
	gchar *string;

	if (!roccat_config_window_warn_if_no_device(ROCCAT_CONFIG_WINDOW(window)))
		return;

	info = nyth_info_read(roccat_config_window_get_device(ROCCAT_CONFIG_WINDOW(window)), &local_error);
	if (!roccat_handle_error_dialog(GTK_WINDOW(window), _("Could not read informations"), &local_error))
		return;

	dialog = ROCCAT_INFO_DIALOG(roccat_info_dialog_new(GTK_WINDOW(window)));

	string = roccat_firmware_version_to_string(info->firmware_version);
	roccat_info_dialog_add_line(dialog, _("Firmware version"), gtk_label_new(string));
	g_free(string);

	g_free(info);

	(void)gtk_dialog_run(GTK_DIALOG(dialog));
	gtk_widget_destroy(GTK_WIDGET(dialog));
}

static void menu_device_tcu_dcu_cb(GtkMenuItem *item, gpointer user_data) {
	NythconfigWindow *window = NYTHCONFIG_WINDOW(user_data);
	GtkDialog *dialog;

	if (!roccat_config_window_warn_if_no_device(ROCCAT_CONFIG_WINDOW(window)))
		return;

	dialog = GTK_DIALOG(nyth_tcu_dcu_dialog_new(GTK_WINDOW(window), roccat_config_window_get_device(ROCCAT_CONFIG_WINDOW(window))));
	(void)gtk_dialog_run(dialog);
	gtk_widget_destroy(GTK_WIDGET(dialog));
}

static void menu_device_reset_cb(GtkMenuItem *item, gpointer user_data) {
	NythconfigWindow *window = NYTHCONFIG_WINDOW(user_data);
	NythconfigWindowPrivate *priv = window->priv;
	NythProfileData *profile_data;
	guint i;
	GError *error = NULL;
	RoccatDevice *device;

	if (!roccat_config_window_warn_if_no_device(ROCCAT_CONFIG_WINDOW(window)))
		return;

	if (!roccat_continue_dialog(GTK_WINDOW(window), _("This resets the device to factory defaults.")))
		return;

	device = roccat_config_window_get_device(ROCCAT_CONFIG_WINDOW(window));
	nyth_reset(device, &error);
	if (!roccat_handle_error_dialog(GTK_WINDOW(window), _("Could not reset device"), &error))
		return;

	for (i = 0; i < NYTH_PROFILE_NUM; ++i) {
		profile_data = nyth_profile_page_get_profile_data(priv->profile_pages[i]);
		nyth_profile_data_update_hardware(profile_data, device, i, &error);
		if (roccat_handle_error_dialog(GTK_WINDOW(window), _("Could not read hardware profile data"), &error))
			nyth_profile_page_set_profile_data(priv->profile_pages[i], profile_data);
		g_free(profile_data);

		nyth_dbus_emit_profile_data_changed_outside(priv->dbus_proxy, i + 1);
	}
}

GtkWidget *nythconfig_window_new(void) {
	return GTK_WIDGET(g_object_new(NYTHCONFIG_WINDOW_TYPE,
			"device-name", NYTH_DEVICE_NAME_COMBINED,
			"default-height", 480,
			"default-width", 640,
			NULL));
}

static GObject *constructor(GType gtype, guint n_properties, GObjectConstructParam *properties) {
	GObject *obj;
	NythconfigWindow *window;
	RoccatConfigWindow *roccat_window;
	NythconfigWindowPrivate *priv;
	NythProfileData *profile_data;
	guint i;
	GtkMenuItem *menu_item;

	obj = G_OBJECT_CLASS(nythconfig_window_parent_class)->constructor(gtype, n_properties, properties);
	window = NYTHCONFIG_WINDOW(obj);
	roccat_window = ROCCAT_CONFIG_WINDOW(obj);
	priv = window->priv;

	priv->config = nyth_configuration_load();

	for (i = 0; i < NYTH_PROFILE_NUM; ++i) {
		priv->profile_pages[i] = NYTH_PROFILE_PAGE(nyth_profile_page_new(i));
		roccat_config_window_append_page(roccat_window, ROCCAT_PROFILE_PAGE(priv->profile_pages[i]));

		profile_data = nyth_profile_data_new();
		nyth_profile_page_set_profile_data(priv->profile_pages[i], profile_data);
		g_free(profile_data);

		g_signal_connect(G_OBJECT(priv->profile_pages[i]), "load-from-file", G_CALLBACK(load_profile_from_file_cb), window);
		g_signal_connect(G_OBJECT(priv->profile_pages[i]), "save-to-file", G_CALLBACK(save_profile_to_file_cb), window);
		g_signal_connect(G_OBJECT(priv->profile_pages[i]), "save-to-device", G_CALLBACK(save_profile_to_device_cb), window);
	}

	menu_item = GTK_MENU_ITEM(gtk_image_menu_item_new_from_stock(GTK_STOCK_PREFERENCES, NULL));
	g_signal_connect(G_OBJECT(menu_item), "activate", G_CALLBACK(menu_edit_preferences_cb), window);
	roccat_config_window_edit_menu_append(roccat_window, menu_item);

/* Firmware update doesn't work on newer kernels
	menu_item = GTK_MENU_ITEM(gtk_image_menu_item_new_with_label(_("Firmware update")));
	g_signal_connect(G_OBJECT(menu_item), "activate", G_CALLBACK(menu_device_firmware_update_cb), window);
	roccat_config_window_device_menu_append(roccat_window, menu_item);
*/

	menu_item = GTK_MENU_ITEM(gtk_image_menu_item_new_with_label(_("Reset")));
	g_signal_connect(G_OBJECT(menu_item), "activate", G_CALLBACK(menu_device_reset_cb), window);
	roccat_config_window_device_menu_append(roccat_window, menu_item);

	menu_item = GTK_MENU_ITEM(gtk_image_menu_item_new_with_label(_("TCU/DCU")));
	g_signal_connect(G_OBJECT(menu_item), "activate", G_CALLBACK(menu_device_tcu_dcu_cb), window);
	roccat_config_window_device_menu_append(roccat_window, menu_item);

	menu_item = GTK_MENU_ITEM(gtk_image_menu_item_new_with_label(_("Info")));
	g_signal_connect(G_OBJECT(menu_item), "activate", G_CALLBACK(menu_device_info_cb), window);
	roccat_config_window_device_menu_append(roccat_window, menu_item);

	g_signal_connect(G_OBJECT(roccat_window), "active-changed", G_CALLBACK(window_active_page_changed_cb), NULL);
	g_signal_connect(G_OBJECT(roccat_window), "destroy", G_CALLBACK(destroy_event_cb), NULL);
	g_signal_connect(G_OBJECT(roccat_window), "delete-event", G_CALLBACK(delete_event_cb), NULL);
	g_signal_connect(G_OBJECT(roccat_window), "save-all", G_CALLBACK(save_all_cb), NULL);
	g_signal_connect(G_OBJECT(roccat_window), "device-removed", G_CALLBACK(device_remove_cb), NULL);
	connect_device_handlers(window);

	/* keep this order */
	priv->dbus_proxy = nyth_dbus_proxy_new();

	roccat_config_window_set_device_scanner(roccat_window, ROCCAT_DEVICE_SCANNER_INTERFACE(nyth_device_scanner_new()));

	gtk_widget_show(GTK_WIDGET(window));

	/* if a device is already plugged in, add_cb has already been executed */
	roccat_config_window_warn_if_no_device(roccat_window);

	return obj;
}

static void nythconfig_window_init(NythconfigWindow *window) {
	window->priv = NYTHCONFIG_WINDOW_GET_PRIVATE(window);
}

static void finalize(GObject *object) {
	NythconfigWindowPrivate *priv = NYTHCONFIG_WINDOW(object)->priv;

	nyth_configuration_dialog_save(GTK_WINDOW(object), priv->config);
	nyth_dbus_emit_configuration_changed_outside(priv->dbus_proxy);
	g_clear_pointer(&priv->config, roccat_configuration_free);
	g_clear_pointer(&priv->dbus_proxy, dbus_roccat_proxy_free);

	G_OBJECT_CLASS(nythconfig_window_parent_class)->finalize(object);
}

static void nythconfig_window_class_init(NythconfigWindowClass *klass) {
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->constructor = constructor;
	gobject_class->finalize = finalize;

	g_type_class_add_private(klass, sizeof(NythconfigWindowClass));
}
