/*
 *  Copyright (c) Stephan Arts 2006-2012 <stephan@xfce.org>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 *  02110-1301, USA.
 */

#include "util.h"
#include "file.h"
#include "thumbnailer.h"
#include "main_window.h"

#ifdef HAVE_MAGIC_H
#include <magic.h>
#endif



static guint rstto_thumbnail_size[] =
{
    THUMBNAIL_SIZE_VERY_SMALL_SIZE,
    THUMBNAIL_SIZE_SMALLER_SIZE,
    THUMBNAIL_SIZE_SMALL_SIZE,
    THUMBNAIL_SIZE_NORMAL_SIZE,
    THUMBNAIL_SIZE_LARGE_SIZE,
    THUMBNAIL_SIZE_LARGER_SIZE,
    THUMBNAIL_SIZE_VERY_LARGE_SIZE
};

enum
{
    RSTTO_FILE_SIGNAL_CHANGED = 0,
    RSTTO_FILE_SIGNAL_COUNT
};

static gint rstto_file_signals[RSTTO_FILE_SIGNAL_COUNT];



static void
rstto_file_finalize (GObject *object);



struct _RsttoFilePrivate
{
    GFile *file;

    gchar *display_name;
    gchar *content_type;
    gboolean final_content_type;

    gchar *uri;
    gchar *path;
    gchar *collate_key;

    gchar *thumbnail_path;
    GdkPixbuf *thumbnails[THUMBNAIL_SIZE_COUNT];

    ExifData *exif_data;
    RsttoImageOrientation orientation;
    gdouble scale;
    RsttoScale auto_scale;
};



G_DEFINE_TYPE_WITH_PRIVATE (RsttoFile, rstto_file, G_TYPE_OBJECT)



static void
rstto_file_init (RsttoFile *r_file)
{
    r_file->priv = rstto_file_get_instance_private (r_file);
    r_file->priv->final_content_type = FALSE;
    r_file->priv->orientation = RSTTO_IMAGE_ORIENT_NOT_DETERMINED;
    r_file->priv->scale = RSTTO_SCALE_NONE;
    r_file->priv->auto_scale = RSTTO_SCALE_NONE;
}


static void
rstto_file_class_init (RsttoFileClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    object_class->finalize = rstto_file_finalize;

    rstto_file_signals[RSTTO_FILE_SIGNAL_CHANGED] = g_signal_new ("changed",
            G_TYPE_FROM_CLASS (object_class),
            G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
            0,
            NULL,
            NULL,
            g_cclosure_marshal_VOID__VOID,
            G_TYPE_NONE,
            0,
            NULL);
}

/**
 * rstto_file_finalize:
 * @object:
 *
 */
static void
rstto_file_finalize (GObject *object)
{
    RsttoFile *r_file = RSTTO_FILE (object);
    gint i = 0;

    if (r_file->priv->file)
    {
        g_object_unref (r_file->priv->file);
        r_file->priv->file = NULL;
    }
    if (r_file->priv->display_name)
    {
        g_free (r_file->priv->display_name);
        r_file->priv->display_name = NULL;
    }
    if (r_file->priv->content_type)
    {
        g_free (r_file->priv->content_type);
        r_file->priv->content_type = NULL;
    }
    if (r_file->priv->path)
    {
        g_free (r_file->priv->path);
        r_file->priv->path = NULL;
    }
    if (r_file->priv->thumbnail_path)
    {
        g_free (r_file->priv->thumbnail_path);
        r_file->priv->thumbnail_path = NULL;
    }
    if (r_file->priv->uri)
    {
        g_free (r_file->priv->uri);
        r_file->priv->uri = NULL;
    }
    if (r_file->priv->collate_key)
    {
        g_free (r_file->priv->collate_key);
        r_file->priv->collate_key = NULL;
    }
    if (r_file->priv->exif_data)
    {
        exif_data_free (r_file->priv->exif_data);
        r_file->priv->exif_data = NULL;
    }

    for (i = 0; i < THUMBNAIL_SIZE_COUNT; ++i)
    {
        if (r_file->priv->thumbnails[i])
        {
            g_object_unref (r_file->priv->thumbnails[i]);
            r_file->priv->thumbnails[i] = NULL;
        }
    }

    G_OBJECT_CLASS (rstto_file_parent_class)->finalize (object);
}

/**
 * rstto_file_new:
 *
 *
 * Singleton
 */
RsttoFile *
rstto_file_new (GFile *file)
{
    RsttoFile *r_file;

    r_file = g_object_new (RSTTO_TYPE_FILE, NULL);
    r_file->priv->file = file;
    g_object_ref (file);

    return r_file;
}

gboolean
rstto_file_is_valid (RsttoFile *r_file)
{
    GtkFileFilter *filter;
    GtkFileFilterInfo filter_info;
    const gchar *content_type = NULL;

    if (! r_file->priv->final_content_type)
    {
#ifdef HAVE_MAGIC_H
        magic_t magic = magic_open (MAGIC_MIME_TYPE | MAGIC_SYMLINK);
        if (NULL != magic)
        {
            const gchar *file_path = rstto_file_get_path (r_file);
            if (NULL != file_path && magic_load (magic, NULL) == 0)
            {
                content_type = magic_file (magic, file_path);
                if (NULL != content_type)
                {
                    /* mime types that require post-processing */
                    if (g_strcmp0 (content_type, "image/x-ms-bmp") == 0)
                    {
                        /* translation for the pixbuf loader: see
                         * https://bugzilla.xfce.org/show_bug.cgi?id=13489 */
                        content_type = "image/bmp";
                    }
                    else if (g_strcmp0 (content_type, "image/x-portable-greymap") == 0)
                    {
                        /* translation for the pixbuf loader: see
                         * https://bugzilla.xfce.org/show_bug.cgi?id=14709 */
                        content_type = "image/x-portable-graymap";
                    }
                    else if (g_strcmp0 (content_type, "application/gzip") == 0)
                    {
                        /* see if it is a compressed SVG file */
                        magic_setflags (magic, magic_getflags (magic) | MAGIC_COMPRESS);
                        if (g_strcmp0 (magic_file (magic, file_path), "image/svg+xml") == 0)
                            content_type = "image/svg+xml";
                    }

                    g_free (r_file->priv->content_type);
                    r_file->priv->content_type = g_strdup (content_type);
                }
            }
            magic_close (magic);
        }
#endif

        if (NULL == content_type)
        {
            GFileInfo *file_info = g_file_query_info (
                    r_file->priv->file,
                    "standard::content-type",
                    0,
                    NULL,
                    NULL);
            if (NULL != file_info)
            {
                content_type = g_file_info_get_content_type (file_info);
                if (NULL != content_type)
                {
                    g_free (r_file->priv->content_type);
                    r_file->priv->content_type = g_strdup (content_type);
                }
                g_object_unref (file_info);
            }
        }

        r_file->priv->final_content_type = TRUE;
    }

    filter = rstto_main_window_get_app_file_filter ();
    filter_info.contains = GTK_FILE_FILTER_MIME_TYPE;
    filter_info.mime_type = r_file->priv->content_type;

    return gtk_file_filter_filter (filter, &filter_info);
}

GFile *
rstto_file_get_file (RsttoFile *r_file)
{
    return r_file->priv->file;
}

gboolean
rstto_file_equal (RsttoFile *r_file_a, RsttoFile *r_file_b)
{
    return r_file_a == r_file_b;
}

const gchar *
rstto_file_get_display_name (RsttoFile *r_file)
{
    GFileInfo *file_info = NULL;
    const gchar *display_name;

    if (NULL == r_file->priv->display_name)
    {
        file_info = g_file_query_info (
                r_file->priv->file,
                G_FILE_ATTRIBUTE_STANDARD_DISPLAY_NAME,
                0,
                NULL,
                NULL);
        if (NULL != file_info)
        {
            display_name = g_file_info_get_display_name (file_info);
            if (NULL != display_name)
            {
                r_file->priv->display_name = g_strdup (display_name);
            }
            g_object_unref (file_info);
        }
    }

    return r_file->priv->display_name;
}

const gchar *
rstto_file_get_path (RsttoFile *r_file)
{
    g_return_val_if_fail (RSTTO_IS_FILE (r_file), NULL);
    g_return_val_if_fail (G_IS_FILE (r_file->priv->file), NULL);

    if (NULL == r_file->priv->path)
    {
        r_file->priv->path = g_file_get_path (r_file->priv->file);
    }
    return r_file->priv->path;
}

const gchar *
rstto_file_get_uri (RsttoFile *r_file)
{
    if (NULL == r_file->priv->uri)
    {
        r_file->priv->uri = g_file_get_uri (r_file->priv->file);
    }
    return r_file->priv->uri;
}

const gchar *
rstto_file_get_collate_key (RsttoFile *r_file)
{
    if (NULL == r_file->priv->collate_key)
    {
        gchar *basename = g_file_get_basename (rstto_file_get_file (r_file));
        if (NULL != basename)
        {
            if (g_utf8_validate (basename, -1, NULL))
            {
                /* If we can use casefold for case insenstivie sorting, then
                 * do so */
                gchar *casefold = g_utf8_casefold (basename, -1);
                if (NULL != casefold)
                {
                    r_file->priv->collate_key = g_utf8_collate_key_for_filename (casefold, -1);
                    g_free (casefold);
                }
                else
                {
                    r_file->priv->collate_key = g_utf8_collate_key_for_filename (basename, -1);
                }
            }
            else
            {
                r_file->priv->collate_key = g_strdup (basename);
            }
            g_free (basename);
        }
    }
    return r_file->priv->collate_key;
}

const gchar *
rstto_file_get_content_type (RsttoFile *r_file)
{
    return r_file->priv->content_type;
}

void
rstto_file_set_content_type (RsttoFile *r_file,
                             const gchar *type)
{
    g_free (r_file->priv->content_type);
    r_file->priv->content_type = g_strdup (type);
}

guint64
rstto_file_get_modified_time (RsttoFile *r_file)
{
    guint64 time_ = 0;
    GFileInfo *file_info = g_file_query_info (r_file->priv->file, "time::modified", 0, NULL, NULL);

    time_ = g_file_info_get_attribute_uint64 (file_info, "time::modified");

    g_object_unref (file_info);

    return time_;
}

goffset
rstto_file_get_size (RsttoFile *r_file)
{
    goffset size = 0;
    GFileInfo *file_info = g_file_query_info (r_file->priv->file, G_FILE_ATTRIBUTE_STANDARD_SIZE, 0, NULL, NULL);

    size = g_file_info_get_size (file_info);

    g_object_unref (file_info);

    return size;
}

ExifEntry *
rstto_file_get_exif (RsttoFile *r_file, ExifTag id)
{
    /* If there is no exif-data object, try to create it */
    if (NULL == r_file->priv->exif_data)
    {
        r_file->priv->exif_data = exif_data_new_from_file (
                rstto_file_get_path (r_file));
    }

    if (NULL != r_file->priv->exif_data)
    {
        return exif_data_get_entry (
                r_file->priv->exif_data,
                id);
    }

    /* If there is no exif-data, return NULL */
    return NULL;
}

RsttoImageOrientation
rstto_file_get_orientation (RsttoFile *r_file)
{
    ExifEntry *exif_entry = NULL;
    if (r_file->priv->orientation == RSTTO_IMAGE_ORIENT_NOT_DETERMINED)
    {
        /* Try to get the default orientation from the EXIF tag */
        exif_entry = rstto_file_get_exif (r_file, EXIF_TAG_ORIENTATION);
        if (NULL != exif_entry)
        {
            r_file->priv->orientation = exif_get_short (
                    exif_entry->data,
                    exif_data_get_byte_order (exif_entry->parent->parent));
        }

        /* If the orientation-tag is not set, default to NONE */
        if (r_file->priv->orientation == RSTTO_IMAGE_ORIENT_NOT_DETERMINED)
        {
            /* Default orientation */
            r_file->priv->orientation = RSTTO_IMAGE_ORIENT_NONE;
        }

    }
    return r_file->priv->orientation;
}

void
rstto_file_set_orientation (
        RsttoFile *r_file,
        RsttoImageOrientation orientation)
{
    r_file->priv->orientation = orientation;
}

gdouble
rstto_file_get_scale (RsttoFile *r_file)
{
    return r_file->priv->scale;
}

void
rstto_file_set_scale (RsttoFile *r_file,
                      gdouble scale)
{
    r_file->priv->scale = scale;
}

RsttoScale
rstto_file_get_auto_scale (RsttoFile *r_file)
{
    return r_file->priv->auto_scale;
}

void
rstto_file_set_auto_scale (RsttoFile *r_file,
                           RsttoScale auto_scale)
{
    r_file->priv->auto_scale = auto_scale;
}

gboolean
rstto_file_has_exif (RsttoFile *r_file)
{
    if (NULL == r_file->priv->exif_data)
    {
        r_file->priv->exif_data = exif_data_new_from_file (rstto_file_get_path (r_file));
    }
    if (NULL == r_file->priv->exif_data)
    {
        return FALSE;
    }
    return TRUE;
}

const gchar *
rstto_file_get_thumbnail_path (RsttoFile *r_file)
{
    const gchar *uri;
    gchar *checksum, *filename, *cache_dir;

    if (NULL == r_file->priv->thumbnail_path)
    {
        uri = rstto_file_get_uri (r_file);
        checksum = g_compute_checksum_for_string (G_CHECKSUM_MD5, uri, strlen (uri));
        filename = g_strconcat (checksum, ".png", NULL);

        /* build and check if the thumbnail is in the new location */
        if (g_file_test ("/.flatpak-info", G_FILE_TEST_EXISTS))
        {
            cache_dir = g_strdup (g_getenv ("HOST_XDG_CACHE_HOME"));
            if (cache_dir == NULL)
                cache_dir =  g_build_path ("/", g_get_home_dir (), ".cache", NULL);
        }
        else
            cache_dir = g_strdup (g_get_user_cache_dir ());

        r_file->priv->thumbnail_path = g_build_path ("/", cache_dir, "thumbnails", "normal", filename, NULL);

        if (!g_file_test (r_file->priv->thumbnail_path, G_FILE_TEST_EXISTS))
        {
            /* Fallback to old version */
            g_free (r_file->priv->thumbnail_path);

            r_file->priv->thumbnail_path = g_build_path ("/", g_get_home_dir (), ".thumbnails", "normal", filename, NULL);
            if (!g_file_test (r_file->priv->thumbnail_path, G_FILE_TEST_EXISTS))
            {
                /* Thumbnail doesn't exist in either spot */
                g_free (r_file->priv->thumbnail_path);
                r_file->priv->thumbnail_path = NULL;
            }
        }

        g_free (checksum);
        g_free (filename);
        g_free (cache_dir);
    }

    return r_file->priv->thumbnail_path;
}

const GdkPixbuf *
rstto_file_get_thumbnail (
        RsttoFile *r_file,
        RsttoThumbnailSize size)
{
    const gchar *thumbnail_path;
    RsttoThumbnailer *thumbnailer;

    if (r_file->priv->thumbnails[size])
        return r_file->priv->thumbnails[size];

    thumbnail_path = rstto_file_get_thumbnail_path (r_file);

    thumbnailer = rstto_thumbnailer_new ();
    rstto_thumbnailer_queue_file (thumbnailer, r_file);

    if (NULL == thumbnail_path)
    {
        /* No thumbnail to return at this time */
        g_object_unref (thumbnailer);
        return NULL;
    }

    /* FIXME:
     *
     * The thumbnail should be updated on the "ready" signal
     * of the thumbnailer, to account for changed thumbnails
     * aswell as missing ones.
     */
    r_file->priv->thumbnails[size] = gdk_pixbuf_new_from_file_at_scale (
            thumbnail_path,
            rstto_thumbnail_size[size],
            rstto_thumbnail_size[size],
            TRUE,
            NULL);

    g_object_unref (thumbnailer);

    return r_file->priv->thumbnails[size];
}

void
rstto_file_changed (RsttoFile *r_file)
{
    g_signal_emit (r_file, rstto_file_signals[RSTTO_FILE_SIGNAL_CHANGED], 0, NULL);
}
