// license:BSD-3-Clause
// copyright-holders:Aaron Giles
/***************************************************************************

    Midway MCR-3 system

    driver by Christopher Kirmse, Aaron Giles

    Games supported:
        * Demolition Derby (Monoboard version) (Turbo Cheap Squeak)
        * Sarge (Turbo Cheap Squeak)
        * Max RPM (Turbo Cheap Squeak)
        * Rampage (Sounds Good)
        * Power Drive (Sounds Good)
        * Star Guards (Sounds Good)
        * Spy Hunter (Cheap Squeak Deluxe)
        * Crater Raider
        * Turbo Tag (prototype) (Cheap Squeak Deluxe)

    Known bugs:
        * Spy Hunter crashes at the end of the boat level
        * sprite placement on the scrolling games seems a bit off


****************************************************************************

    Game Hardware

    Sarge:
        * Mono Board (A080-91787-C000)
        * Turbo Cheap Squeak (A080-91779-B000)
        * MCR/2 Dual Power Amp (A082-90910-F000)

    Spy Hunter:
        * MCR/III CPU (A080-91442-J000)
        * Super Sound I/O (A082-90913-F000)
        * MCR/III Video Gen (A082-91433-J000)
        * Cheap Squeak Deluxe (A080-91671-E000)
        * Absolute Position Board (A080-91649-C000)
        * Lamp Driver Board (A080-91673-B000)
        * MCR/2 Dual Power Amp (A082-90910-F000)

****************************************************************************

    Memory map

****************************************************************************

    ========================================================================
    CPU #1
    ========================================================================
    0000-DFFF   R     xxxxxxxx    Program ROM
    E000-E7FF   R/W   xxxxxxxx    NVRAM
    E800-E9FF   R/W   xxxxxxxx    Sprite RAM
    F000-F7FF   R/W   xxxxxxxx    Background video RAM
    F800-F8FF     W   xxxxxxxx    Palette RAM
    ========================================================================
    0000        R     x-xxxxxx    Input ports
                R     x-------    Service switch (active low)
                R     --x-----    Tilt
                R     ---xxx--    External inputs
                R     ------x-    Right coin
                R     -------x    Left coin
    0000        W     xxxxxxxx    Data latch OP0 (coin meters, 2 led's and cocktail 'flip')
    0001        R     xxxxxxxx    External inputs
    0002        R     xxxxxxxx    External inputs
    0003        R     xxxxxxxx    DIP switches
    0004        R     xxxxxxxx    External inputs
    0004        W     xxxxxxxx    Data latch OP4 (comm. with external hardware)
    0007        R     xxxxxxxx    Audio status
    001C-001F   W     xxxxxxxx    Audio latches 1-4
    00E0        W     --------    Watchdog reset
    00E8        W     xxxxxxxx    Unknown (written at initialization time)
    00F0-00F3   W     xxxxxxxx    CTC communications
    ========================================================================
    Interrupts:
        NMI ???
        INT generated by CTC
    ========================================================================


    ========================================================================
    CPU #2 (Super Sound I/O)
    ========================================================================
    0000-3FFF   R     xxxxxxxx    Program ROM
    8000-83FF   R/W   xxxxxxxx    Program RAM
    9000-9003   R     xxxxxxxx    Audio latches 1-4
    A000          W   xxxxxxxx    AY-8910 #1 control
    A001        R     xxxxxxxx    AY-8910 #1 status
    A002          W   xxxxxxxx    AY-8910 #1 data
    B000          W   xxxxxxxx    AY-8910 #2 control
    B001        R     xxxxxxxx    AY-8910 #2 status
    B002          W   xxxxxxxx    AY-8910 #2 data
    C000          W   xxxxxxxx    Audio status
    E000          W   xxxxxxxx    Unknown
    F000        R     xxxxxxxx    Audio board switches
    ========================================================================
    Interrupts:
        NMI ???
        INT generated by external circuitry 780 times/second
    ========================================================================

***************************************************************************/


#include "emu.h"
#include "includes/mcr3.h"

#include "machine/nvram.h"

#include "speaker.h"

#include "spyhunt.lh"
#include "turbotag.lh"


#define MASTER_CLOCK        XTAL(20'000'000)



WRITE8_MEMBER(mcr3_state::mcrmono_control_port_w)
{
	/*
	    Bit layout is as follows:
	        D7 = n/c
	        D6 = cocktail flip
	        D5 = n/c
	        D4 = n/c
	        D3 = n/c
	        D2 = n/c
	        D1 = n/c
	        D0 = coin meter 1
	*/

	machine().bookkeeping().coin_counter_w(0, (data >> 0) & 1);
	m_mcr_cocktail_flip = (data >> 6) & 1;
}


/*************************************
 *
 *  Demolition Derby (mono) I/O ports
 *
 *************************************/

READ8_MEMBER(mcr3_state::demoderm_ip1_r)
{
	return ioport("MONO.IP1")->read() |
		(ioport(m_input_mux ? "MONO.IP1.ALT2" : "MONO.IP1.ALT1")->read() << 2);
}


READ8_MEMBER(mcr3_state::demoderm_ip2_r)
{
	return ioport("MONO.IP2")->read() |
		(ioport(m_input_mux ? "MONO.IP2.ALT2" : "MONO.IP2.ALT1")->read() << 2);
}


WRITE8_MEMBER(mcr3_state::demoderm_op6_w)
{
	/* top 2 bits select the input */
	if (data & 0x80) m_input_mux = 0;
	if (data & 0x40) m_input_mux = 1;

	/* low 5 bits control the turbo CS */
	m_turbo_cheap_squeak->write(space, offset, data);
}



/*************************************
 *
 *  Max RPM I/O ports
 *
 *************************************/

READ8_MEMBER(mcr3_state::maxrpm_ip1_r)
{
	return m_latched_input;
}


READ8_MEMBER(mcr3_state::maxrpm_ip2_r)
{
	/* this is a blatant hack, should really do a better implementation */
	static const uint8_t shift_bits[5] = { 0x00, 0x05, 0x06, 0x01, 0x02 };
	uint8_t start = ioport("MONO.IP0")->read();
	uint8_t shift = ioport("SHIFT")->read();

	/* reset on a start */
	if (!(start & 0x08))
		m_maxrpm_p1_shift = 0;
	if (!(start & 0x04))
		m_maxrpm_p2_shift = 0;

	/* increment, decrement on falling edge */
	if (!(shift & 0x01) && (m_maxrpm_last_shift & 0x01))
	{
		m_maxrpm_p1_shift++;
		if (m_maxrpm_p1_shift > 4)
			m_maxrpm_p1_shift = 4;
	}
	if (!(shift & 0x02) && (m_maxrpm_last_shift & 0x02))
	{
		m_maxrpm_p1_shift--;
		if (m_maxrpm_p1_shift < 0)
			m_maxrpm_p1_shift = 0;
	}
	if (!(shift & 0x04) && (m_maxrpm_last_shift & 0x04))
	{
		m_maxrpm_p2_shift++;
		if (m_maxrpm_p2_shift > 4)
			m_maxrpm_p2_shift = 4;
	}
	if (!(shift & 0x08) && (m_maxrpm_last_shift & 0x08))
	{
		m_maxrpm_p2_shift--;
		if (m_maxrpm_p2_shift < 0)
			m_maxrpm_p2_shift = 0;
	}

	m_maxrpm_last_shift = shift;

	return ~((shift_bits[m_maxrpm_p1_shift] << 4) + shift_bits[m_maxrpm_p2_shift]);
}


WRITE8_MEMBER(mcr3_state::maxrpm_op5_w)
{
	/* latch bits 1-4 as input to the ADC0844 */
	m_maxrpm_adc_control = (data >> 1) & 0x0f;

	/* remaining bits go to standard connections */
	mcrmono_control_port_w(space, offset, data);
}


WRITE8_MEMBER(mcr3_state::maxrpm_op6_w)
{
	/*
	    Reflective Sensor Control:
	        4 bits of input from OP5 are routed to a transceiver at U2, and
	        ultimately on to the low 4 I/O pins of the ADC0844. The /EN on
	        the transceiver is directly connected to J2-2. Note that two bits
	            get swapped in the process: OP53 = MA3 and OP54 = MA2.

	        In order to perform a read or a write to the ADC0844, the /RD and
	        /WR signals are directly controlled via J2-8 and J2-7 respectively.
	        The input from the /WR is controlled by enabling the transceiver
	        above to allow the 4 bits of input to flow through. The output
	        from an /RD is controlled by disabling the transceiver and allowing
	        the 8 bits of output to flow through J2-13 through J2-20. These are
	        read via IP1.
	*/
	/* bit 7 = /RD (J2-8) on ADC0844 */
	/* bit 6 = /WR (J2-7) on ADC0844 */
	/* bit 5 = /EN (J2-2) on input latch */

	/* when the read is toggled is when the ADC value is latched */
	if (!(data & 0x80))
		m_latched_input = m_maxrpm_adc->read();

	/* when both the write and the enable are low, it's a write to the ADC0844 */
	if (!(data & 0x40) && !(data & 0x20))
		m_maxrpm_adc->write(bitswap<4>(m_maxrpm_adc_control, 2, 3, 1, 0));

	/* low 5 bits control the turbo CS */
	m_turbo_cheap_squeak->write(space, offset, data);
}



/*************************************
 *
 *  Rampage I/O ports
 *
 *************************************/

READ8_MEMBER(mcr3_state::rampage_ip4_r)
{
	return ioport("MONO.IP4")->read() | (m_sounds_good->read(space,0) << 7);
}


WRITE8_MEMBER(mcr3_state::rampage_op6_w)
{
	/* bit 5 controls reset of the Sounds Good board */
	m_sounds_good->reset_write((~data >> 5) & 1);

	/* low 5 bits go directly to the Sounds Good board */
	m_sounds_good->write(space, offset, data);
}



/*************************************
 *
 *  Power Drive I/O ports
 *
 *************************************/

READ8_MEMBER(mcr3_state::powerdrv_ip2_r)
{
	return ioport("MONO.IP2")->read() | (m_sounds_good->read(space, 0) << 7);
}


WRITE8_MEMBER(mcr3_state::powerdrv_op5_w)
{
	/*
	    Lamp Board:
	        Very simple board with direct lamp controls.
	        Pin J1-10 controls lamp 1.
	        Pin J1-8 controls lamp 2.
	        Pin J1-6 controls lamp 3.
	*/
	/* bit 3 -> J1-10 = lamp 1 */
	/* bit 2 -> J1-8 = lamp 2 */
	/* bit 1 -> J1-6 = lamp 3 */
	m_lamps[0] = BIT(data, 3);
	m_lamps[1] = BIT(data, 2);
	m_lamps[2] = BIT(data, 1);

	/* remaining bits go to standard connections */
	mcrmono_control_port_w(space, offset, data);
}


WRITE8_MEMBER(mcr3_state::powerdrv_op6_w)
{
	/* bit 5 controls reset of the Sounds Good board */
	m_sounds_good->reset_write((~data >> 5) & 1);

	/* low 5 bits go directly to the Sounds Good board */
	m_sounds_good->write(space, offset, data);
}



/*************************************
 *
 *  Star Guards I/O ports
 *
 *************************************/

READ8_MEMBER(mcr3_state::stargrds_ip0_r)
{
	uint8_t result = ioport("MONO.IP0")->read();
	if (m_input_mux)
		result = (result & ~0x0a) | (ioport("MONO.IP0.ALT")->read() & 0x0a);
	return (result & ~0x10) | ((m_sounds_good->read(space, 0) << 4) & 0x10);
}


WRITE8_MEMBER(mcr3_state::stargrds_op5_w)
{
	/* bit 1 controls input muxing on port 0 */
	m_input_mux = (data >> 1) & 1;

	/* bit 2 controls light #0 */
	/* bit 3 controls light #1 */
	/* bit 4 controls light #2 */
	m_lamps[0] = BIT(data, 2);
	m_lamps[1] = BIT(data, 3);
	m_lamps[2] = BIT(data, 4);

	/* remaining bits go to standard connections */
	mcrmono_control_port_w(space, offset, data);
}


WRITE8_MEMBER(mcr3_state::stargrds_op6_w)
{
	/* bit 6 controls reset of the Sounds Good board */
	m_sounds_good->reset_write((~data >> 6) & 1);

	/* unline the other games, the STROBE is in the high bit instead of the low bit */
	m_sounds_good->write(space, offset, (data << 1) | (data >> 7));
}



/*************************************
 *
 *  Spy Hunter I/O ports
 *
 *************************************/

READ8_MEMBER(mcr3_state::spyhunt_ip1_r)
{
	return ioport("ssio:IP1")->read() | (m_cheap_squeak_deluxe->stat_r() << 5);
}


READ8_MEMBER(mcr3_state::spyhunt_ip2_r)
{
	/* multiplexed steering wheel/gas pedal */
	return ioport(m_input_mux ? "ssio:IP2.ALT" : "ssio:IP2")->read();
}


WRITE8_MEMBER(mcr3_state::spyhunt_op4_w)
{
	/* Spy Hunter uses port 4 for talking to the Cheap Squeak Deluxe */
	/* (and for toggling the lamps and muxing the analog inputs) */

	/* mux select is in bit 7 */
	m_input_mux = (data >> 7) & 1;

	/*
	    Lamp Driver:
	        A 3-to-8 latching demuxer (MC14099) is connected to the input bits.
	        Three of the inputs (J1-11,10,12) specify which output to write
	        to, and the fourth input (J1-14) is the data value. A fifth input
	        (J1-13) controls the strobe to latch the data value for the
	        demuxer. The eight outputs control 8 lamps through a pair of
	        Darlington drivers (ULN2068B).
	*/
	/* bit 5 = STR1 (J1-13) */
	if (((m_last_op4 ^ data) & 0x20) && !(data & 0x20))
	{
		/* bit 3 -> J1-14 (DATA) */
		/* bit 2 -> J1-11 (A2) */
		/* bit 1 -> J1-10 (A1) */
		/* bit 0 -> J1-12 (A0) */
		m_lamplatch->write_bit(data & 7, BIT(data, 3));
	}
	m_last_op4 = data;

	/* low 5 bits go to control the Cheap Squeak Deluxe */
	m_cheap_squeak_deluxe->sr_w(data & 0x0f);
	m_cheap_squeak_deluxe->sirq_w(BIT(data, 4));
}



/*************************************
 *
 *  Turbo Tag kludges
 *
 *************************************/

READ8_MEMBER(mcr3_state::turbotag_ip2_r)
{
	/* multiplexed steering wheel/gas pedal */
	if (m_input_mux)
		return ioport("ssio:IP2.ALT")->read();

	return ioport("ssio:IP2")->read() + 5 * (m_screen->frame_number() & 1);
}


READ8_MEMBER(mcr3_state::turbotag_kludge_r)
{
	/* The checksum on the ttprog1.bin ROM seems to be bad by 1 bit */
	/* The checksum should come out to $82 but it should be $92     */
	/* Unfortunately, the game refuses to start if any bad ROM is   */
	/* found; to work around this, we catch the checksum byte read  */
	/* and modify it to what we know we will be getting.            */
	if (m_maincpu->pcbase() == 0xb29)
		return 0x82;
	else
		return 0x92;
}



/*************************************
 *
 *  MCR Monoboard CPU memory handlers
 *
 *************************************/

/* address map verified from schematics */
void mcr3_state::mcrmono_map(address_map &map)
{
	map.unmap_value_high();
	map(0x0000, 0xdfff).rom();
	map(0xe000, 0xe7ff).ram().share("nvram");
	map(0xe800, 0xe9ff).ram().share("spriteram");
	map(0xea00, 0xebff).ram();
	map(0xec00, 0xec7f).mirror(0x0380).w(FUNC(mcr3_state::mcr_paletteram9_w)).share("paletteram");
	map(0xf000, 0xf7ff).ram().w(FUNC(mcr3_state::mcr3_videoram_w)).share("videoram");
	map(0xf800, 0xffff).rom();     /* schematics show a 2716 @ 2B here, but nobody used it */
}

/* I/O map verified from schematics */
void mcr3_state::mcrmono_portmap(address_map &map)
{
	map.unmap_value_high();
	map.global_mask(0xff);
	map(0x00, 0x00).mirror(0x78).portr("MONO.IP0");
	map(0x01, 0x01).mirror(0x78).portr("MONO.IP1");
	map(0x02, 0x02).mirror(0x78).portr("MONO.IP2");
	map(0x03, 0x03).mirror(0x78).portr("MONO.IP3");
	map(0x04, 0x04).mirror(0x78).portr("MONO.IP4");
	map(0x05, 0x05).mirror(0x78).w(FUNC(mcr3_state::mcrmono_control_port_w));
	map(0x07, 0x07).mirror(0x78).w("watchdog", FUNC(watchdog_timer_device::reset_w));
	map(0xf0, 0xf3).mirror(0x0c).rw(m_ctc, FUNC(z80ctc_device::read), FUNC(z80ctc_device::write));
}



/*************************************
 *
 *  Spy Hunter main CPU memory handlers
 *
 *************************************/

/* address map verified from schematics */
void mcr3_state::spyhunt_map(address_map &map)
{
	map.unmap_value_high();
	map(0x0000, 0xdfff).rom();
	map(0xe000, 0xe7ff).ram().w(FUNC(mcr3_state::spyhunt_videoram_w)).share("videoram");
	map(0xe800, 0xebff).mirror(0x0400).ram().w(FUNC(mcr3_state::spyhunt_alpharam_w)).share("spyhunt_alpha");
	map(0xf000, 0xf7ff).ram().share("nvram");
	map(0xf800, 0xf9ff).ram().share("spriteram");
	map(0xfa00, 0xfa7f).mirror(0x0180).w(FUNC(mcr3_state::mcr_paletteram9_w)).share("paletteram");
}

/* upper I/O map determined by PAL; only SSIO ports and scroll registers are verified from schematics */
void mcr3_state::spyhunt_portmap(address_map &map)
{
	map.unmap_value_high();
	map.global_mask(0xff);
	midway_ssio_device::ssio_input_ports(map, "ssio");
	map(0x84, 0x86).w(FUNC(mcr3_state::spyhunt_scroll_value_w));
	map(0xe0, 0xe0).w("watchdog", FUNC(watchdog_timer_device::reset_w));
	map(0xe8, 0xe8).nopw();
	map(0xf0, 0xf3).rw(m_ctc, FUNC(z80ctc_device::read), FUNC(z80ctc_device::write));
}


/*************************************
 *
 *  Port definitions
 *
 *************************************/

/* verified from wiring diagram, plus DIP switches from manual */
static INPUT_PORTS_START( demoderm )
	PORT_START("MONO.IP0")  /* J2 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE( 0x20, IP_ACTIVE_LOW )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("MONO.IP1")  /* J2 10-13,15-18 */    /* The high 6 bits contain the steering wheel value */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)

	PORT_START("MONO.IP1.ALT1") /* J2 10-13,15-18 */    /* The high 6 bits contain the steering wheel value */
	PORT_BIT( 0x3f, 0x00, IPT_DIAL ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(1)

	PORT_START("MONO.IP1.ALT2") /* IN1 (muxed) -- the high 6 bits contain the steering wheel value */
	PORT_BIT( 0x3f, 0x00, IPT_DIAL ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(3)

	PORT_START("MONO.IP2")  /* J3 1-8 */    /* The high 6 bits contain the steering wheel value */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)

	PORT_START("MONO.IP2.ALT1") /* J3 1-8 */    /* The high 6 bits contain the steering wheel value */
	PORT_BIT( 0x3f, 0x00, IPT_DIAL ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(2)

	PORT_START("MONO.IP2.ALT2") /* IN2 (muxed) -- the high 6 bits contain the steering wheel value */
	PORT_BIT( 0x3f, 0x00, IPT_DIAL ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(4)

	PORT_START("MONO.IP3")  /* DIPSW @ B3 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x01, "2P Upright" )
	PORT_DIPSETTING(    0x00, "4P Cocktail" )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Harder ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Reward Screen" )
	PORT_DIPSETTING(    0x08, "Expanded" )
	PORT_DIPSETTING(    0x00, "Limited" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x20, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_2C ) )
	PORT_DIPSETTING(    0x30, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("MONO.IP4")  /* J4 1-7,9 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START4 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(3)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(3)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(4)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(4)
INPUT_PORTS_END


/* inputs not verified yet, DIP switches from manual */
static INPUT_PORTS_START( sarge )
	PORT_START("MONO.IP0")  /* J2 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_TILT )
	PORT_SERVICE( 0x20, IP_ACTIVE_LOW )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("MONO.IP1")  /* J2 10-13,15-18 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_2WAY PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_2WAY PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_2WAY PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_2WAY PORT_PLAYER(1)
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)

	PORT_START("MONO.IP2")  /* J3 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_2WAY PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_2WAY PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_2WAY PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_2WAY PORT_PLAYER(2)
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)

	PORT_START("MONO.IP3")  /* DIPSW @ A13 */
	PORT_DIPUNKNOWN( 0x01, 0x01 ) // used, maybe for the "hidden player incentive" easter egg?
	PORT_DIPUNKNOWN( 0x02, 0x02 ) // "
	PORT_DIPUNKNOWN( 0x04, 0x04 ) // "
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x20, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x30, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
//  PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) ) // dupe/invalid
	PORT_DIPUNUSED( 0x40, 0x40 )
	PORT_DIPUNUSED( 0x80, 0x80 )

	PORT_START("MONO.IP4")  /* J4 1-7,9 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


/* verified from wiring diagram, plus DIP switches from manual */
static INPUT_PORTS_START( maxrpm )
	PORT_START("MONO.IP0")  /* J2 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START("MONO.IP1")  /* J2 10-13,15-18 */
	PORT_BIT( 0xff, 0x30, IPT_PEDAL ) PORT_MINMAX(0x30,0xff) PORT_SENSITIVITY(100) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(2)

	PORT_START("MONO.IP2")  /* J3 1-8 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_CUSTOM )

	PORT_START("MONO.IP3")
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x20, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x30, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
//  PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) ) // dupe/invalid
	PORT_BIT( 0xc7, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("MONO.IP4")  /* J4 1-7,9 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("MONO.IP1.ALT1")
	PORT_BIT( 0xff, 0x30, IPT_PEDAL ) PORT_MINMAX(0x30,0xff) PORT_SENSITIVITY(100) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(1)

	PORT_START("MONO.IP1.ALT2")
	PORT_BIT( 0xff, 0x74, IPT_PADDLE ) PORT_MINMAX(0x34,0xb4) PORT_SENSITIVITY(40) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(2)

	PORT_START("MONO.IP1.ALT3")
	PORT_BIT( 0xff, 0x74, IPT_PADDLE ) PORT_MINMAX(0x34,0xb4) PORT_SENSITIVITY(40) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(1)

	PORT_START("SHIFT") /* fake for shifting */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP   ) PORT_2WAY PORT_PLAYER(1) PORT_NAME("P1 Shift Up")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_2WAY PORT_PLAYER(1) PORT_NAME("P1 Shift Down")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP   ) PORT_2WAY PORT_PLAYER(2) PORT_NAME("P2 Shift Up")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_2WAY PORT_PLAYER(2) PORT_NAME("P2 Shift Down")
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


/* verified from wiring diagram, plus DIP switches from manual */
static INPUT_PORTS_START( rampage )
	PORT_START("MONO.IP0")  /* J2 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_TILT )
	PORT_SERVICE( 0x20, IP_ACTIVE_LOW )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("MONO.IP1")  /* J2 10-13,15-18 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("MONO.IP2")  /* J3 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("MONO.IP3")
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x03, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Hard ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x04, 0x04, "Score Option" )
	PORT_DIPSETTING(    0x04, "Keep score when continuing" )
	PORT_DIPSETTING(    0x00, "Lose score when continuing" )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 1C_1C ) )
	PORT_DIPNAME( 0x70, 0x70, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x70, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x60, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x50, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x40, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x30, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(    0x20, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x80, 0x80, "Rack Advance (Cheat)" ) PORT_CODE(KEYCODE_F1)
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START("MONO.IP4")  /* J4 1-7,9 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(3)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(3)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_CUSTOM )   /* status from Sounds Good board */
INPUT_PORTS_END


/* verified from wiring diagram, plus DIP switches from manual */
static INPUT_PORTS_START( powerdrv )
	PORT_START("MONO.IP0")  /* J2 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_TILT )
	PORT_SERVICE( 0x20, IP_ACTIVE_LOW )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("MONO.IP1")  /* J2 10-13,15-18 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_TOGGLE PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_TOGGLE PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)

	PORT_START("MONO.IP2")  /* J3 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(3)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_TOGGLE PORT_PLAYER(3)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(3)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(3)
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_CUSTOM )   /* status from Sounds Good board */

	PORT_START("MONO.IP3")
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 2C_1C ) )
/*  PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )*/
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_2C ) )
	PORT_BIT( 0x0c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x30, "Factory" )
	PORT_DIPSETTING(    0x10, DEF_STR( Hard ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Hardest ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Rack Advance (Cheat)" ) PORT_CODE(KEYCODE_F1)
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START("MONO.IP4")  /* J4 1-7,9 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


/* verified from wiring diagram, plus DIP switches from manual */
static INPUT_PORTS_START( stargrds )
	PORT_START("MONO.IP0")  /* J2 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_CUSTOM )   /* status from Sounds Good board */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START("MONO.IP1")  /* J2 10-13,15-18 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT ) PORT_PLAYER(1)

	PORT_START("MONO.IP2")  /* J3 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT ) PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT ) PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT ) PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT ) PORT_PLAYER(2)

	PORT_START("MONO.IP3")
	PORT_BIT( 0x03, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x0c, 0x0c, "Energy Units" )
	PORT_DIPSETTING(    0x08, "8" )
	PORT_DIPSETTING(    0x0c, "10" )
	PORT_DIPSETTING(    0x04, "12" )
	PORT_DIPSETTING(    0x00, "14" )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0xc0, 0xc0, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0xc0, DEF_STR( Medium ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Hard ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Hardest ) )

	PORT_START("MONO.IP4")  /* J4 1-7,9 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_PLAYER(3)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_PLAYER(3)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT ) PORT_PLAYER(3)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT ) PORT_PLAYER(3)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_PLAYER(3)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_PLAYER(3)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT ) PORT_PLAYER(3)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT ) PORT_PLAYER(3)

	PORT_START("MONO.IP0.ALT")  /* IN0 (muxed) */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )     /* same as MONO.IN0 */
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )     /* same as MONO.IN0 */
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START3 )
	PORT_BIT( 0xf0, IP_ACTIVE_HIGH, IPT_UNUSED )    /* same as MONO.IN0 */
INPUT_PORTS_END


/* verified from wiring diagram, plus DIP switches from manual */
static INPUT_PORTS_START( spyhunt )
	PORT_START("ssio:IP0")  /* J4 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Gear Shift") PORT_TOGGLE
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START("ssio:IP1")  /* J4 10-13,15-18 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_NAME("Left Button / Oil Slick")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("Left Trigger / Missiles")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1  ) PORT_NAME("Center Button / Weapons Van")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_NAME("Right Button / Smoke Screen")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_NAME("Right Trigger / Machine Guns")
	PORT_BIT( 0x60, IP_ACTIVE_HIGH, IPT_CUSTOM )   /* status from CS deluxe, never read */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("ssio:IP2")  /* J5 1-8 */
	PORT_BIT( 0xff, 0x30, IPT_PEDAL ) PORT_MINMAX(0x30,0xff) PORT_SENSITIVITY(100) PORT_KEYDELTA(10)

	PORT_START("ssio:IP3")  /* DIPSW @ B3 */
	PORT_DIPNAME( 0x01, 0x01, "Game Timer" )
	PORT_DIPSETTING(    0x00, "1:00" )
	PORT_DIPSETTING(    0x01, "1:30" )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("ssio:IP4")  /* J6 1-8 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("ssio:DIP")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("ssio:IP2.ALT")
	PORT_BIT( 0xff, 0x74, IPT_PADDLE ) PORT_MINMAX(0x34,0xb4) PORT_SENSITIVITY(40) PORT_KEYDELTA(10)
INPUT_PORTS_END


/* not verified, no manual found */
static INPUT_PORTS_START( crater )
	PORT_START("ssio:IP0")  /* J4 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START("ssio:IP1")  /* J4 10-13,15-18 */
	PORT_BIT( 0xff, 0x00, IPT_DIAL ) PORT_SENSITIVITY(25) PORT_KEYDELTA(10) PORT_CODE_DEC(KEYCODE_Z) PORT_CODE_INC(KEYCODE_X) PORT_REVERSE

	PORT_START("ssio:IP2")  /* J5 1-8 */
	PORT_BIT( 0x03, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_2WAY
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_2WAY
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("ssio:IP3")  /* DIPSW @ B3 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("ssio:IP4")  /* J6 1-8 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("ssio:DIP")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


/* not verified, no manual found */
static INPUT_PORTS_START( turbotag )
	PORT_START("ssio:IP0")  /* J4 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Gear Shift") PORT_TOGGLE
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START("ssio:IP1")  /* J4 10-13,15-18 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1  ) PORT_NAME("Left Button / 1 Player")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("Left Trigger")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_NAME("Center Button")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2  ) PORT_NAME("Right Button / 2 Player")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_NAME("Right Trigger")
	PORT_BIT( 0x60, IP_ACTIVE_HIGH, IPT_CUSTOM )   /* status from CS deluxe, never read */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("ssio:IP2")  /* J5 1-8 */
	PORT_BIT( 0xff, 0x3c, IPT_PEDAL ) PORT_MINMAX(60,180) PORT_SENSITIVITY(100) PORT_KEYDELTA(10)

	PORT_START("ssio:IP3")  /* DIPSW @ B3 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("ssio:IP4")  /* J6 1-8 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("ssio:DIP")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("ssio:IP2.ALT")
	PORT_BIT( 0xff, 0x60, IPT_PADDLE ) PORT_SENSITIVITY(40) PORT_KEYDELTA(10)
INPUT_PORTS_END



/*************************************
 *
 *  Graphics definitions
 *
 *************************************/

static const uint32_t spyhunt_charlayout_xoffset[64] =
{
		0,  0,  2,  2,  4,  4,  6,  6,  8,  8, 10, 10, 12, 12, 14, 14,
		16, 16, 18, 18, 20, 20, 22, 22, 24, 24, 26, 26, 28, 28, 30, 30,
		32, 32, 34, 34, 36, 36, 38, 38, 40, 40, 42, 42, 44, 44, 46, 46,
		48, 48, 50, 50, 52, 52, 54, 54, 56, 56, 58, 58, 60, 60, 62, 62
};

static const gfx_layout spyhunt_charlayout =
{
	64,32,
	RGN_FRAC(1,2),
	4,
	{ RGN_FRAC(1,2), RGN_FRAC(1,2)+1, 0, 1 },
	EXTENDED_XOFFS,
	{
			0*32,  0*32,  2*32,  2*32,  4*32,  4*32,  6*32,  6*32,
			8*32,  8*32, 10*32, 10*32, 12*32, 12*32, 14*32, 14*32,
			16*32, 16*32, 18*32, 18*32, 20*32, 20*32, 22*32, 22*32,
			24*32, 24*32, 26*32, 26*32, 28*32, 28*32, 30*32, 30*32
	},
	128*8,
	spyhunt_charlayout_xoffset,
	nullptr
};


static const gfx_layout spyhunt_alphalayout =
{
	16,16,
	RGN_FRAC(1,1),
	2,
	{ 0, 1 },
	{ 0, 0, 2, 2, 4, 4, 6, 6, 8, 8, 10, 10, 12, 12, 14, 14 },
	{ 0, 0, 2*8, 2*8, 4*8, 4*8, 6*8, 6*8, 8*8, 8*8, 10*8, 10*8, 12*8, 12*8, 14*8, 14*8 },
	16*8
};


static GFXDECODE_START( gfx_mcr3 )
	GFXDECODE_SCALE( "gfx1", 0, mcr_bg_layout,     0, 4, 2, 2 )
	GFXDECODE_ENTRY( "gfx2", 0, mcr_sprite_layout, 0, 4 )
GFXDECODE_END


static GFXDECODE_START( gfx_spyhunt )
	GFXDECODE_ENTRY( "gfx1", 0, spyhunt_charlayout,  3*16, 1 )
	GFXDECODE_ENTRY( "gfx2", 0, mcr_sprite_layout,   0*16, 4 )
	GFXDECODE_ENTRY( "gfx3", 0, spyhunt_alphalayout, 4*16, 1 )
GFXDECODE_END


/*************************************
 *
 *  Machine drivers
 *
 *************************************/

/* Core MCR monoboard system with no sound */
void mcr3_state::mcrmono(machine_config &config)
{
	/* basic machine hardware */
	Z80(config, m_maincpu, MASTER_CLOCK/4);
	m_maincpu->set_addrmap(AS_PROGRAM, &mcr3_state::mcrmono_map);
	m_maincpu->set_addrmap(AS_IO, &mcr3_state::mcrmono_portmap);
	m_maincpu->set_daisy_config(mcr_daisy_chain);

	TIMER(config, "scantimer").configure_scanline(FUNC(mcr3_state::mcr_interrupt), "screen", 0, 1);

	Z80CTC(config, m_ctc, MASTER_CLOCK/4 /* same as "maincpu" */);
	m_ctc->intr_callback().set_inputline(m_maincpu, INPUT_LINE_IRQ0);
	m_ctc->zc_callback<0>().set(m_ctc, FUNC(z80ctc_device::trg1));

	WATCHDOG_TIMER(config, "watchdog").set_vblank_count(m_screen, 16);

	NVRAM(config, "nvram", nvram_device::DEFAULT_ALL_0);

	// sound hardware
	SPEAKER(config, "lspeaker").front_left();
	SPEAKER(config, "rspeaker").front_right();

	/* video hardware */
	SCREEN(config, m_screen, SCREEN_TYPE_RASTER);
	m_screen->set_video_attributes(VIDEO_UPDATE_BEFORE_VBLANK);
	m_screen->set_refresh_hz(30);
	m_screen->set_vblank_time(ATTOSECONDS_IN_USEC(2500) /* not accurate */);
	m_screen->set_size(32*16, 30*16);
	m_screen->set_visarea(0*16, 32*16-1, 0*16, 30*16-1);
	m_screen->set_screen_update(FUNC(mcr3_state::screen_update_mcr3));
	m_screen->set_palette(m_palette);

	GFXDECODE(config, m_gfxdecode, m_palette, gfx_mcr3);
	PALETTE(config, m_palette).set_entries(64);
}


/*************************************/


/* Sarge/Demolition Derby Mono/Max RPM = MCR monoboard with Turbo Cheap Squeak */
void mcr3_state::mono_tcs(machine_config &config)
{
	mcrmono(config);

	/* basic machine hardware */
	MIDWAY_TURBO_CHEAP_SQUEAK(config, m_turbo_cheap_squeak);
	m_turbo_cheap_squeak->add_route(ALL_OUTPUTS, "lspeaker", 1.0);
	m_turbo_cheap_squeak->add_route(ALL_OUTPUTS, "rspeaker", 1.0);
}

void mcr3_state::maxrpm(machine_config &config)
{
	mono_tcs(config);

	ADC0844(config, m_maxrpm_adc);
	m_maxrpm_adc->ch1_callback().set_ioport("MONO.IP1");
	m_maxrpm_adc->ch2_callback().set_ioport("MONO.IP1.ALT1");
	m_maxrpm_adc->ch3_callback().set_ioport("MONO.IP1.ALT2");
	m_maxrpm_adc->ch4_callback().set_ioport("MONO.IP1.ALT3");
}


/* Rampage/Power Drive/Star Guards = MCR monoboard with Sounds Good */
void mcr3_state::mono_sg(machine_config &config)
{
	mcrmono(config);

	/* basic machine hardware */
	MIDWAY_SOUNDS_GOOD(config, m_sounds_good);
	m_sounds_good->add_route(ALL_OUTPUTS, "lspeaker", 1.0);
	m_sounds_good->add_route(ALL_OUTPUTS, "rspeaker", 1.0);
}


/*************************************/


/* Core scrolling system with SSIO sound */
void mcr3_state::mcrscroll(machine_config &config)
{
	mcrmono(config);

	/* basic machine hardware */
	MIDWAY_SSIO(config, m_ssio);
	m_ssio->add_route(0, "lspeaker", 1.0);
	m_ssio->add_route(1, "rspeaker", 1.0);

	m_maincpu->set_addrmap(AS_PROGRAM, &mcr3_state::spyhunt_map);
	m_maincpu->set_addrmap(AS_IO, &mcr3_state::spyhunt_portmap);

	/* video hardware */
	m_screen->set_size(30*16, 30*16);
	m_screen->set_visarea(0, 30*16-1, 0, 30*16-1);
	m_screen->set_screen_update(FUNC(mcr3_state::screen_update_spyhunt));
	m_gfxdecode->set_info(gfx_spyhunt);
	subdevice<palette_device>("palette")->set_entries(64 + 4).set_init(FUNC(mcr3_state::spyhunt_palette));

	MCFG_VIDEO_START_OVERRIDE(mcr3_state,spyhunt)
}


/* Spy Hunter = scrolling system with an SSIO and a cheap squeak deluxe */
void mcr3_state::mcrsc_csd(machine_config &config)
{
	mcrscroll(config);

	/* basic machine hardware */
	MIDWAY_CHEAP_SQUEAK_DELUXE(config, m_cheap_squeak_deluxe);
	m_cheap_squeak_deluxe->add_route(ALL_OUTPUTS, "lspeaker", 1.0);
	m_cheap_squeak_deluxe->add_route(ALL_OUTPUTS, "rspeaker", 1.0);

	CD4099(config, m_lamplatch); // U1 on Lamp Driver Board
	m_lamplatch->q_out_cb<0>().set_output("lamp0");
	m_lamplatch->q_out_cb<1>().set_output("lamp1");
	m_lamplatch->q_out_cb<2>().set_output("lamp2");
	m_lamplatch->q_out_cb<3>().set_output("lamp3");
	m_lamplatch->q_out_cb<4>().set_output("lamp4");
	m_lamplatch->q_out_cb<5>().set_output("lamp5");
	m_lamplatch->q_out_cb<6>().set_output("lamp6");
	m_lamplatch->q_out_cb<7>().set_output("lamp7");
}


/*************************************
 *
 *  ROM definitions
 *
 *************************************/

ROM_START( demoderm ) /* Dipswitch selectable 2 player Upright / 4 player Cocktail */
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "pro0.3b",      0x00000, 0x8000, CRC(2e24527b) SHA1(df8d1129b52ca0f4326c7c7e4f10d81b4ced65b5) )
	ROM_LOAD( "pro1.5b",      0x08000, 0x8000, CRC(034c00fc) SHA1(0f0e8f123a34c330021bce76ed7f38bcb2e9af4e) )
	ROM_FILL(                 0x0e000, 0x2000, 0xff )   /* upper 8k is not mapped on monoboard */

	ROM_REGION( 0x10000, "tcs:cpu", 0 ) /* 64k for the Turbo Cheap Squeak */
	ROM_LOAD( "tcs_u5.bin",   0x0c000, 0x2000, CRC(eca33b2c) SHA1(938b021ea3b0f23aed7a98a930a58af371a02303) )
	ROM_LOAD( "tcs_u4.bin",   0x0e000, 0x2000, CRC(3490289a) SHA1(a9d56ea60bb901267da41ab408f8e1ed3742b0ac) )

	ROM_REGION( 0x04000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "bg0.15a",      0x00000, 0x2000, CRC(a35d13b8) SHA1(18d2d900a787e082ba30c13034ed6c64305a79b4) )
	ROM_LOAD( "bg1.14b",      0x02000, 0x2000, CRC(22ca93f3) SHA1(8a907ee6d1fd88d472d868eb47e7ae7d44097e67) )

	ROM_REGION( 0x20000, "gfx2", 0 )
	ROM_LOAD( "dd_fg-0.a4",   0x00000, 0x4000, CRC(e57a4de6) SHA1(d1b2396a85b984e171d751ef8e1cf970ac4ff9fb) )
	ROM_LOAD( "dd_fg-4.a3",   0x04000, 0x4000, CRC(55aa667f) SHA1(d611dbf9e8ef383d02514b0edb9ea36670193bf0) )
	ROM_LOAD( "dd_fg-1.a6",   0x08000, 0x4000, CRC(70259651) SHA1(55967aaf2a7617c8f5a199d1e07128d79ce16970) )
	ROM_LOAD( "dd_fg-5.a5",   0x0c000, 0x4000, CRC(5fe99007) SHA1(9d640b4715333efdc6300dc353991d6934929399) )
	ROM_LOAD( "dd_fg-2.a8",   0x10000, 0x4000, CRC(6cab7b95) SHA1(8faff7458ab5ff2dd096dd78b1449a4096cc6345) )
	ROM_LOAD( "dd_fg-6.a7",   0x14000, 0x4000, CRC(abfb9a8b) SHA1(14ab416bc76db25ad97353c9072048c64ec95344) )
	ROM_LOAD( "dd_fg-3.a10",  0x18000, 0x4000, CRC(801d9b86) SHA1(5a8c72d1060eea1a3ad67b98aa6eff13f6837af6) )
	ROM_LOAD( "dd_fg-7.a9",   0x1c000, 0x4000, CRC(0ec3f60a) SHA1(4176b246b0ea7bce9498c20e12678f16f7173529) )
ROM_END


ROM_START( sarge )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "cpu_3b.bin",   0x00000, 0x8000, CRC(da31a58f) SHA1(29b97caf61f8f59042519a6b501cd1d15099dd59) )
	ROM_LOAD( "cpu_5b.bin",   0x08000, 0x8000, CRC(6800e746) SHA1(018c2b622b3654530ebc2c299b3f745777163d4b) )
	ROM_FILL(                 0x0e000, 0x2000, 0xff )   /* upper 8k is not mapped on monoboard */

	ROM_REGION( 0x10000, "tcs:cpu", 0 )  /* 64k for the Turbo Cheap Squeak */
	ROM_LOAD( "tcs_u5.bin",   0xc000, 0x2000, CRC(a894ef8a) SHA1(7f53927fc185fff8ba1b1747f0d565e089d879e6) )
	ROM_LOAD( "tcs_u4.bin",   0xe000, 0x2000, CRC(6ca6faf3) SHA1(4647e633dd11f55a65c3acf81adeb3af93624991) )

	ROM_REGION( 0x04000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "til_15a.bin",  0x00000, 0x2000, CRC(685001b8) SHA1(300abd808efe5b44b022082eebc591c7c255697c) )
	ROM_LOAD( "til_14b.bin",  0x02000, 0x2000, CRC(8449eb45) SHA1(6cc43639998d55fe7ffac7e9b091d35ea169e048) )

	ROM_REGION( 0x20000, "gfx2", 0 )
	ROM_LOAD( "spr_8e.bin",   0x00000, 0x8000, CRC(93fac29d) SHA1(3d144208eca3b5377689e69da4505187a3d20d4f) )
	ROM_LOAD( "spr_6e.bin",   0x08000, 0x8000, CRC(7cc6fb28) SHA1(5fe24d5114551b4a4bf303cd612da332555c0f93) )
	ROM_LOAD( "spr_5e.bin",   0x10000, 0x8000, CRC(c832375c) SHA1(dfb7782b13e1e959e0ecd5da771cd38962f6952b) )
	ROM_LOAD( "spr_4e.bin",   0x18000, 0x8000, CRC(c382267d) SHA1(6b459e9ec7948a529b5308357851a0bede085aef) )

	ROM_REGION( 0x0007, "pals", 0) /* PAL's located on the Mono Board (91787) */
	ROM_LOAD( "a59a26axlcxhd.13j.bin",  0x0000, 0x0001, NO_DUMP ) /* PLS153N */
	ROM_LOAD( "a59a26axlbxhd.2j.bin",   0x0000, 0x0001, NO_DUMP ) /* PLS153N */
	ROM_LOAD( "a59a26axlaxhd.3j.bin",   0x0000, 0x0001, NO_DUMP ) /* PLS153N */
	ROM_LOAD( "0066-314bx-xxqx.6h.bin", 0x0000, 0x0001, NO_DUMP ) /* Unknown PAL Type */
	ROM_LOAD( "0066-316bx-xxqx.5h.bin", 0x0000, 0x0001, NO_DUMP ) /* Unknown PAL Type */
	ROM_LOAD( "0066-315bx-xxqx.5g.bin", 0x0000, 0x0001, NO_DUMP ) /* Unknown PAL Type */
	ROM_LOAD( "0066-313bx-xxqx.4g.bin", 0x0000, 0x0001, NO_DUMP ) /* Unknown PAL Type */
ROM_END


ROM_START( maxrpm )
	ROM_REGION( 0x12000, "maincpu", 0 )
	ROM_LOAD( "pro.0",        0x00000, 0x8000, CRC(3f9ec35f) SHA1(ebdcf480aee5569af631106cc6478adc26c4ac24) )
	ROM_LOAD( "pro.1",        0x08000, 0x8000, CRC(f628bb30) SHA1(0514906b9764a7f02a730a610affea4d42a4510d) )
	ROM_FILL(                 0x0e000, 0x2000, 0xff )   /* upper 8k is not mapped on monoboard */

	ROM_REGION( 0x10000, "tcs:cpu", 0 )  /* 64k for the Turbo Cheap Squeak */
	ROM_LOAD( "turbskwk.u5",   0x8000, 0x4000, CRC(55c3b759) SHA1(89d690a007a996e9c7df7b365942e4da755d15d7) )
	ROM_LOAD( "turbskwk.u4",   0xc000, 0x4000, CRC(31a2da2e) SHA1(582434b5d6bc8e84f39191976d8aa0ef9245f253) )

	ROM_REGION( 0x08000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "bg-0",         0x00000, 0x4000, CRC(e3fb693a) SHA1(9543c099cae4f56ef09f4e678891c90ef55928ed) )
	ROM_LOAD( "bg-1",         0x04000, 0x4000, CRC(50d1db6c) SHA1(5ef669b868edf2d0b7f16879523318e8d6a3f9b3) )

	ROM_REGION( 0x20000, "gfx2", 0 )
	ROM_LOAD( "fg-0",         0x00000, 0x8000, CRC(1d1435c1) SHA1(6a53ef047bb763ea30d2e8098cb1dd6b337ccf4f) )
	ROM_LOAD( "fg-1",         0x08000, 0x8000, CRC(e54b7f2a) SHA1(b95f9d71ec98e60db57ce57006c013e5a521d8ab) )
	ROM_LOAD( "fg-2",         0x10000, 0x8000, CRC(38be8505) SHA1(5e01e5e6317dd722d19f23863908ffc5833f22c3) )
	ROM_LOAD( "fg-3",         0x18000, 0x8000, CRC(9ae3eb52) SHA1(a96835caece971692790cd7385ab618373eb821f) )
ROM_END


ROM_START( rampage )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "pro-0_3b_rev_3_8-27-86.3b", 0x00000, 0x8000, CRC(2f7ca03c) SHA1(1e3a1f213fd67938adf14ea0d04dab687ea8f4ef) )
	ROM_LOAD( "pro-1_5b_rev_3_8-27-86.5b", 0x08000, 0x8000, CRC(d89bd9a4) SHA1(3531464ffe49dfaf2755d9e2dc1aea23819b3a5d) )
	ROM_FILL(                              0x0e000, 0x2000, 0xff )  /* upper 8k is not mapped on monoboard */

	ROM_REGION( 0x40000, "sg:cpu", 0 )  /* 256k for the Sounds Good board */
	ROM_LOAD16_BYTE( "u-7_rev.2_8-14-86.u7",   0x00000, 0x8000, CRC(cffd7fa5) SHA1(7c5cecce1d428f847fea37d53eb09c6f62055c6f) )  /* these are Revision 2 sound ROMs */
	ROM_LOAD16_BYTE( "u-17_rev.2_8-14-86.u17", 0x00001, 0x8000, CRC(e92c596b) SHA1(4e2d87398f2e7b637cbad6cb16d832dfa8f8288c) )
	ROM_LOAD16_BYTE( "u-8_rev.2_8-14-86.u8",   0x10000, 0x8000, CRC(11f787e4) SHA1(1fa195bf9169608099d17be5801738a4e17bec3d) )
	ROM_LOAD16_BYTE( "u-18_rev.2_8-14-86.u18", 0x10001, 0x8000, CRC(6b8bf5e1) SHA1(aa8c0260dcd19a795bfc23197cd87348a685d20b) )

	ROM_REGION( 0x08000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "bg-0_u15_7-23-86.15a", 0x00000, 0x04000, CRC(c0d8b7a5) SHA1(692219388a3124fb48db7e35c4127b0fe066a289) )
	ROM_LOAD( "bg-1_u14_7-23-86.14b", 0x04000, 0x04000, CRC(2f6e3aa1) SHA1(ae86ce90bb6bf660e38c0f91e7ce90d44be82d60) )

	ROM_REGION( 0x40000, "gfx2", 0 )
	ROM_LOAD( "fg-0_8e_6-30-86.8e",   0x00000, 0x10000, CRC(0974be5d) SHA1(be347faaa345383dc6e5c2b3789c372d6bd25905) )
	ROM_LOAD( "fg-1_6e_6-30-86.6e",   0x10000, 0x10000, CRC(8728532b) SHA1(327df92db7e3506b827d497859980cd2de51f45d) )
	ROM_LOAD( "fg-2_5e_6-30-86.5e",   0x20000, 0x10000, CRC(9489f714) SHA1(df17a45cdc6a9310856d64f89954be79bbeac12e) )
	ROM_LOAD( "fg-3_4e_6-30-86.4e",   0x30000, 0x10000, CRC(81e1de40) SHA1(7e7818792845ec3687b3202eeade60a298ef513e) )

	ROM_REGION( 0x0001, "sg:pal", 0 ) /* Sounds Good board pal */
	ROM_LOAD( "e36a31axnaxqd.u15.bin", 0x0000, 0x0001, NO_DUMP) /* PAL20L10CNS */
ROM_END


ROM_START( rampage2 )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "pro-0_3b_rev_2_8-4-86.3b", 0x00000, 0x8000, CRC(3f1d0293) SHA1(d68f04b9b3fc377b9e57b823db4e7f9cfedbcf99) )
	ROM_LOAD( "pro-1_5b_rev_2_8-4-86.5b", 0x08000, 0x8000, CRC(58523d75) SHA1(5cd512864568ec7793bda0164f21e7d72a7ea817) )
	ROM_FILL(                             0x0e000, 0x2000, 0xff )   /* upper 8k is not mapped on monoboard */

	ROM_REGION( 0x40000, "sg:cpu", 0 )  /* 256k for the Sounds Good board */
	ROM_LOAD16_BYTE( "u-7_rev.2_8-14-86.u7",   0x00000, 0x8000, CRC(cffd7fa5) SHA1(7c5cecce1d428f847fea37d53eb09c6f62055c6f) )  /* these are Revision 2 sound ROMs */
	ROM_LOAD16_BYTE( "u-17_rev.2_8-14-86.u17", 0x00001, 0x8000, CRC(e92c596b) SHA1(4e2d87398f2e7b637cbad6cb16d832dfa8f8288c) )
	ROM_LOAD16_BYTE( "u-8_rev.2_8-14-86.u8",   0x10000, 0x8000, CRC(11f787e4) SHA1(1fa195bf9169608099d17be5801738a4e17bec3d) )
	ROM_LOAD16_BYTE( "u-18_rev.2_8-14-86.u18", 0x10001, 0x8000, CRC(6b8bf5e1) SHA1(aa8c0260dcd19a795bfc23197cd87348a685d20b) )

	ROM_REGION( 0x08000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "bg-0_u15_7-23-86.15a", 0x00000, 0x04000, CRC(c0d8b7a5) SHA1(692219388a3124fb48db7e35c4127b0fe066a289) )
	ROM_LOAD( "bg-1_u14_7-23-86.14b", 0x04000, 0x04000, CRC(2f6e3aa1) SHA1(ae86ce90bb6bf660e38c0f91e7ce90d44be82d60) )

	ROM_REGION( 0x40000, "gfx2", 0 )
	ROM_LOAD( "fg-0_8e_6-30-86.8e",   0x00000, 0x10000, CRC(0974be5d) SHA1(be347faaa345383dc6e5c2b3789c372d6bd25905) )
	ROM_LOAD( "fg-1_6e_6-30-86.6e",   0x10000, 0x10000, CRC(8728532b) SHA1(327df92db7e3506b827d497859980cd2de51f45d) )
	ROM_LOAD( "fg-2_5e_6-30-86.5e",   0x20000, 0x10000, CRC(9489f714) SHA1(df17a45cdc6a9310856d64f89954be79bbeac12e) )
	ROM_LOAD( "fg-3_4e_6-30-86.4e",   0x30000, 0x10000, CRC(81e1de40) SHA1(7e7818792845ec3687b3202eeade60a298ef513e) )

	ROM_REGION( 0x0001, "sg:pal", 0 ) /* Sounds Good board pal */
	ROM_LOAD( "e36a31axnaxqd.u15.bin", 0x0000, 0x0001, NO_DUMP) /* PAL20L10CNS */
ROM_END


ROM_START( powerdrv )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "pdrv3b.bin",   0x00000, 0x8000, CRC(d870b704) SHA1(300d6ff3c92a45f5c4f28c171280174924aecb6c) )
	ROM_LOAD( "pdrv5b.bin",   0x08000, 0x8000, CRC(fa0544ad) SHA1(55a9cf8c8648761443e4a5a3b214f4d6236cbaff) )
	ROM_FILL(                 0x0e000, 0x2000, 0xff )   /* upper 8k is not mapped on monoboard */

	ROM_REGION( 0x40000, "sg:cpu", 0 )  /* 256k for the Sounds Good board */
	ROM_LOAD16_BYTE( "power_drive_snd_u7.u7",   0x00000, 0x8000, CRC(78713e78) SHA1(11382c024536f743e051ba208ae02d0f5e07cf5e) ) /* Dated 11/21/86 */
	ROM_LOAD16_BYTE( "power_drive_snd_u17.u17", 0x00001, 0x8000, CRC(c41de6e4) SHA1(0391afd96ee80dd1d4a34e661e5df1e01fbbd57a) )
	ROM_LOAD16_BYTE( "power_drive_snd_u8.u8",   0x10000, 0x8000, CRC(15714036) SHA1(77ca5f703eb7f146e13d9c01f4427f6aaa31df39) )
	ROM_LOAD16_BYTE( "power_drive_snd_u18.u18", 0x10001, 0x8000, CRC(cae14c70) SHA1(04e92f1f144cc8ff13a09a3d38aa65ac05c41c0b) )

	ROM_REGION( 0x08000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "pdrv15a.bin",  0x00000, 0x04000, CRC(b858b5a8) SHA1(da622bde13c7156a826d658e176feccf18f33a4b) )
	ROM_LOAD( "pdrv14b.bin",  0x04000, 0x04000, CRC(12ee7fc2) SHA1(aca7b20efa8b0e2217691fd8de6adf68a2048331) )

	ROM_REGION( 0x40000, "gfx2", 0 )
	ROM_LOAD( "pdrv8e.bin",   0x00000, 0x10000, CRC(dd3a2adc) SHA1(0792591eb55603c809d08ee622ecb2c9f5731038) )
	ROM_LOAD( "pdrv6e.bin",   0x10000, 0x10000, CRC(1a1f7f81) SHA1(d66ff3ef3855e086e665531aef0e079842c48fcb) )
	ROM_LOAD( "pdrv5e.bin",   0x20000, 0x10000, CRC(4cb4780e) SHA1(e1fad431257d9ec5e35175e4dc09d21e36ba1fa0) )
	ROM_LOAD( "pdrv4e.bin",   0x30000, 0x10000, CRC(de400335) SHA1(49438bc7c2ba236dcbd4ddc3c985460887dcf110) )
ROM_END


ROM_START( stargrds )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "pro-0.3b",  0x00000, 0x8000, CRC(3ad94aa2) SHA1(1e17ac70fddee1f3d0dd71172e15a7a256168a70) )
	ROM_LOAD( "pro-1.5b",  0x08000, 0x8000, CRC(dba428b0) SHA1(72efa2f02e95f05a5503ced136fbdf3fcdf57554) )
	ROM_FILL(              0x0e000, 0x2000, 0xff )  /* upper 8k is not mapped on monoboard */

	ROM_REGION( 0x40000, "sg:cpu", 0 )  /* 256k for the Sounds Good board */
	ROM_LOAD16_BYTE( "snd0.u7",      0x00000, 0x8000, CRC(7755a493) SHA1(a888fba45a2a31de5b3082bfc5ccef94dafc4d16) )
	ROM_LOAD16_BYTE( "snd1.u17",     0x00001, 0x8000, CRC(d98d14ae) SHA1(51dbb97655ab8a389ca67f0e796ab57894f5bb32) )

	ROM_REGION( 0x10000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "bg-0.15a",     0x00000, 0x08000, CRC(ecfaef3e) SHA1(145b0543552e678ef441e4a20afe80cd21e43cf6) )
	ROM_LOAD( "bg-1.14b",     0x08000, 0x08000, CRC(2c75569d) SHA1(13808ff0fdb413a4ac6ddef1422b86c118277899) )

	ROM_REGION( 0x40000, "gfx2", 0 )
	ROM_LOAD( "fg-0.8e",         0x00000, 0x10000, CRC(22797aaa) SHA1(806ea29f8a43f104d3154322f492e7fa053da751) )
	ROM_LOAD( "fg-1.6e",         0x10000, 0x10000, CRC(413fa119) SHA1(cb609db2e6694a5bb563afd6b28c19afe65a8487) )
	ROM_LOAD( "fg-2.5e",         0x20000, 0x10000, CRC(7036cfe6) SHA1(7778c5a8bf457ba0c1fa0e17978709e1f7ccb8a5) )
	ROM_LOAD( "fg-3.4e",         0x30000, 0x10000, CRC(cc5cc003) SHA1(afe8a0b0542c6f98de9386a07d6586af7de6bdf6) )
ROM_END


/* Spy Hunter labels look like this
   SPY-HUNTER
     C.P.U.
      PG3
     2/9/84
*/

ROM_START( spyhunt )
	ROM_REGION( 0x10000, "maincpu", 0 ) // all dated 2-9-84
	ROM_LOAD( "spy-hunter_cpu_pg0_2-9-84.6d",   0x0000, 0x2000, CRC(1721b88f) SHA1(c7a641f0c05bd343ebc79e1c1be3a26da5fb77f0) )
	ROM_LOAD( "spy-hunter_cpu_pg1_2-9-84.7d",   0x2000, 0x2000, CRC(909d044f) SHA1(67237c3efde568d52e9f8b0d36df726d05a9d9e4) )
	ROM_LOAD( "spy-hunter_cpu_pg2_2-9-84.8d",   0x4000, 0x2000, CRC(afeeb8bd) SHA1(fde32863d08a745dfe19f1c1382810eab6aebcec) )
	ROM_LOAD( "spy-hunter_cpu_pg3_2-9-84.9d",   0x6000, 0x2000, CRC(5e744381) SHA1(5b75e4f44dfd63d6e35294c606b84231c216e57d) )
	ROM_LOAD( "spy-hunter_cpu_pg4_2-9-84.10d",  0x8000, 0x2000, CRC(a3033c15) SHA1(e9811450a7c952561912777d679fe45a6b5a794a) )
	ROM_LOAD( "spy-hunter_cpu_pg5_2-9-84.11d",  0xc000, 0x2000, CRC(88aa1e99) SHA1(c173512ed76973d2f86a74380bb7b7c5bb4f5285) )
	ROM_CONTINUE(                               0xa000, 0x2000 )

	ROM_REGION( 0x10000, "ssio:cpu", 0 ) // all dated 11-18-83
	ROM_LOAD( "spy-hunter_snd_0_sd_11-18-83.a7",   0x0000, 0x1000, CRC(c95cf31e) SHA1(d1b0e299a27e306ddbc0654fd3a9d981c92afe8c) )
	ROM_LOAD( "spy-hunter_snd_1_sd_11-18-83.a8",   0x1000, 0x1000, CRC(12aaa48e) SHA1(c6b835fc45e4484a4d52b682ce015caa242c8b4f) )

	ROM_REGION( 0x8000, "csd:cpu", 0 )  /* 32k for the Cheap Squeak Deluxe */ // all dated 11-18-83
	ROM_LOAD16_BYTE( "spy-hunter_cs_deluxe_u7_a_11-18-83.u7",   0x00000, 0x2000, CRC(6e689fe7) SHA1(38ad2e9f12b9d389fb2568ebcb32c8bd1ac6879e) )
	ROM_LOAD16_BYTE( "spy-hunter_cs_deluxe_u17_b_11-18-83.u17", 0x00001, 0x2000, CRC(0d9ddce6) SHA1(d955c0e67fc78b517cc229601ab4023cc5a644c2) )
	ROM_LOAD16_BYTE( "spy-hunter_cs_deluxe_u8_c_11-18-83.u8",   0x04000, 0x2000, CRC(35563cd0) SHA1(5708d374dd56758194c95118f096ea51bf12bf64) )
	ROM_LOAD16_BYTE( "spy-hunter_cs_deluxe_u18_d_11-18-83.u18", 0x04001, 0x2000, CRC(63d3f5b1) SHA1(5864a7e9b6bc3d2df6891d40965a7a0efbba6837) )

	ROM_REGION( 0x08000, "gfx1", 0 ) // all dated 11-18-83
	ROM_LOAD( "spy-hunter_cpu_bg0_11-18-83.3a",   0x00000, 0x2000, CRC(dea34fed) SHA1(cbbb2ba75e087eebdce79a0016118c327c8f0a96) )
	ROM_LOAD( "spy-hunter_cpu_bg1_11-18-83.4a",   0x02000, 0x2000, CRC(8f64525f) SHA1(d457d12f31a30deb3b4e5b8189c9414aac1ad701) )
	ROM_LOAD( "spy-hunter_cpu_bg2_11-18-83.5a",   0x04000, 0x2000, CRC(ba0fd626) SHA1(f39281feb3fbbbd4234fbb70ee77bab3e1a33e3b) )
	ROM_LOAD( "spy-hunter_cpu_bg3_11-18-83.6a",   0x06000, 0x2000, CRC(7b482d61) SHA1(f6a46690f69a7513a7fbacd0199946f600d796dd) )

	ROM_REGION( 0x20000, "gfx2", 0 ) // all dated 11-18-83
	ROM_LOAD( "spy-hunter_video_1fg_11-18-83.a7",   0x00000, 0x4000, CRC(9fe286ec) SHA1(d72cd7e69ef78e25cf5bc599fb0a7da11bf4657f) )
	ROM_LOAD( "spy-hunter_video_0fg_11-18-83.a8",   0x04000, 0x4000, CRC(292c5466) SHA1(5abb9e2cc592adf81f12bf8ebeaf3e2931a7fa6d) )
	ROM_LOAD( "spy-hunter_video_3fg_11-18-83.a5",   0x08000, 0x4000, CRC(b894934d) SHA1(e7d6db1635d244d002054dd223a2d0713316ef77) )
	ROM_LOAD( "spy-hunter_video_2fg_11-18-83.a6",   0x0c000, 0x4000, CRC(62c8bfa5) SHA1(f245e49c178f846b647d09c32aa97d61333bdd83) )
	ROM_LOAD( "spy-hunter_video_5fg_11-18-83.a3",   0x10000, 0x4000, CRC(2d9fbcec) SHA1(d73862b974726fe50bf011ea7977f8229b8a1e24) )
	ROM_LOAD( "spy-hunter_video_4fg_11-18-83.a4",   0x14000, 0x4000, CRC(7ca4941b) SHA1(068ecd1e91ecfedba2ae542062f8f51f1329725d) )
	ROM_LOAD( "spy-hunter_video_7fg_11-18-83.a1",   0x18000, 0x4000, CRC(940fe17e) SHA1(60d07c10ef5867875d47a4edaa68934e37e2a0aa) )
	ROM_LOAD( "spy-hunter_video_6fg_11-18-83.a2",   0x1c000, 0x4000, CRC(8cb8a066) SHA1(5fa88d471ed8fd18244dd21b976c86530f57c8ac) )

	ROM_REGION( 0x01000, "gfx3", 0 ) // all dated 11-18-83
	ROM_LOAD( "spy-hunter_cpu_alpha-n_11-18-83", 0x00000, 0x1000, CRC(936dc87f) SHA1(cdf73bea82481fbc300ec5a1fbbe8d662007c56b) )
ROM_END


ROM_START( spyhuntp )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "sh_mb_2.bin",                    0x0000, 0x2000, CRC(df6cd642) SHA1(e695aa3e3dc3b9df87bf2c1bcb75edbdf03fde98) )
	ROM_LOAD( "spy-hunter_cpu_pg1_2-9-84.7d",   0x2000, 0x2000, CRC(909d044f) SHA1(67237c3efde568d52e9f8b0d36df726d05a9d9e4) )
	ROM_LOAD( "spy-hunter_cpu_pg2_2-9-84.8d",   0x4000, 0x2000, CRC(afeeb8bd) SHA1(fde32863d08a745dfe19f1c1382810eab6aebcec) )
	ROM_LOAD( "spy-hunter_cpu_pg3_2-9-84.9d",   0x6000, 0x2000, CRC(5e744381) SHA1(5b75e4f44dfd63d6e35294c606b84231c216e57d) )
	ROM_LOAD( "sh_mb_6.bin",                    0x8000, 0x2000, CRC(8cbf04d8) SHA1(e45cb36935367e46ea41de0177b3453cd8bdce85) )
	ROM_LOAD( "spy-hunter_cpu_pg5_2-9-84.11d",  0xc000, 0x2000, CRC(88aa1e99) SHA1(c173512ed76973d2f86a74380bb7b7c5bb4f5285) )
	ROM_CONTINUE(                               0xa000, 0x2000 )

	ROM_REGION( 0x10000, "ssio:cpu", 0 )
	ROM_LOAD( "spy-hunter_snd_0_sd_11-18-83.a7",   0x0000, 0x1000, CRC(c95cf31e) SHA1(d1b0e299a27e306ddbc0654fd3a9d981c92afe8c) )
	ROM_LOAD( "spy-hunter_snd_1_sd_11-18-83.a8",   0x1000, 0x1000, CRC(12aaa48e) SHA1(c6b835fc45e4484a4d52b682ce015caa242c8b4f) )

	ROM_REGION( 0x8000, "csd:cpu", 0 )  /* 32k for the Cheap Squeak Deluxe */
	ROM_LOAD16_BYTE( "spy-hunter_cs_deluxe_u7_a_11-18-83.u7",   0x00000, 0x2000, CRC(6e689fe7) SHA1(38ad2e9f12b9d389fb2568ebcb32c8bd1ac6879e) )
	ROM_LOAD16_BYTE( "spy-hunter_cs_deluxe_u17_b_11-18-83.u17", 0x00001, 0x2000, CRC(0d9ddce6) SHA1(d955c0e67fc78b517cc229601ab4023cc5a644c2) )
	ROM_LOAD16_BYTE( "spy-hunter_cs_deluxe_u8_c_11-18-83.u8",   0x04000, 0x2000, CRC(35563cd0) SHA1(5708d374dd56758194c95118f096ea51bf12bf64) )
	ROM_LOAD16_BYTE( "spy-hunter_cs_deluxe_u18_d_11-18-83.u18", 0x04001, 0x2000, CRC(63d3f5b1) SHA1(5864a7e9b6bc3d2df6891d40965a7a0efbba6837) )

	ROM_REGION( 0x08000, "gfx1", 0 )
	ROM_LOAD( "spy-hunter_cpu_bg0_11-18-83.3a",   0x00000, 0x2000, CRC(dea34fed) SHA1(cbbb2ba75e087eebdce79a0016118c327c8f0a96) )
	ROM_LOAD( "spy-hunter_cpu_bg1_11-18-83.4a",   0x02000, 0x2000, CRC(8f64525f) SHA1(d457d12f31a30deb3b4e5b8189c9414aac1ad701) )
	ROM_LOAD( "spy-hunter_cpu_bg2_11-18-83.5a",   0x04000, 0x2000, CRC(ba0fd626) SHA1(f39281feb3fbbbd4234fbb70ee77bab3e1a33e3b) )
	ROM_LOAD( "spy-hunter_cpu_bg3_11-18-83.6a",   0x06000, 0x2000, CRC(7b482d61) SHA1(f6a46690f69a7513a7fbacd0199946f600d796dd) )

	ROM_REGION( 0x20000, "gfx2", 0 )
	ROM_LOAD( "spy-hunter_video_1fg_11-18-83.a7",   0x00000, 0x4000, CRC(9fe286ec) SHA1(d72cd7e69ef78e25cf5bc599fb0a7da11bf4657f) )
	ROM_LOAD( "spy-hunter_video_0fg_11-18-83.a8",   0x04000, 0x4000, CRC(292c5466) SHA1(5abb9e2cc592adf81f12bf8ebeaf3e2931a7fa6d) )
	ROM_LOAD( "spy-hunter_video_3fg_11-18-83.a5",   0x08000, 0x4000, CRC(b894934d) SHA1(e7d6db1635d244d002054dd223a2d0713316ef77) )
	ROM_LOAD( "spy-hunter_video_2fg_11-18-83.a6",   0x0c000, 0x4000, CRC(62c8bfa5) SHA1(f245e49c178f846b647d09c32aa97d61333bdd83) )
	ROM_LOAD( "spy-hunter_video_5fg_11-18-83.a3",   0x10000, 0x4000, CRC(2d9fbcec) SHA1(d73862b974726fe50bf011ea7977f8229b8a1e24) )
	ROM_LOAD( "spy-hunter_video_4fg_11-18-83.a4",   0x14000, 0x4000, CRC(7ca4941b) SHA1(068ecd1e91ecfedba2ae542062f8f51f1329725d) )
	ROM_LOAD( "spy-hunter_video_7fg_11-18-83.a1",   0x18000, 0x4000, CRC(940fe17e) SHA1(60d07c10ef5867875d47a4edaa68934e37e2a0aa) )
	ROM_LOAD( "spy-hunter_video_6fg_11-18-83.a2",   0x1c000, 0x4000, CRC(8cb8a066) SHA1(5fa88d471ed8fd18244dd21b976c86530f57c8ac) )

	ROM_REGION( 0x01000, "gfx3", 0 )
	ROM_LOAD( "sh_mb_1.bin",  0x00000, 0x0800, CRC(5c74c9f0) SHA1(e42789d14e5510d1dcf4a30f6bd40a40ab46f7f3) )
ROM_END




ROM_START( crater )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "crcpu.6d",     0x0000, 0x2000, CRC(ad31f127) SHA1(d03506570cd08bcdb39d7c6b9de4f9628c7373e9) )
	ROM_LOAD( "crcpu.7d",     0x2000, 0x2000, CRC(3743c78f) SHA1(05605041c7a5ef2a8271cc3dd1aae4c82c8797eb) )
	ROM_LOAD( "crcpu.8d",     0x4000, 0x2000, CRC(c95f9088) SHA1(2162c4ef2cc29857b10d64ec9e09ca6d086fa9e7) )
	ROM_LOAD( "crcpu.9d",     0x6000, 0x2000, CRC(a03c4b11) SHA1(6a442a0828942dc51dbe0d3f19be855a52c12f39) )
	ROM_LOAD( "crcpu.10d",    0x8000, 0x2000, CRC(44ae4cbd) SHA1(2188bf697f1b3fcbb2ad6cbd4d9098e3b8d56a95) )

	ROM_REGION( 0x10000, "ssio:cpu", 0 )
	ROM_LOAD( "crsnd4.a7",    0x0000, 0x1000, CRC(fd666cb5) SHA1(257174266e264db8ac9af5f2296fd0a22847f85f) )
	ROM_LOAD( "crsnd1.a8",    0x1000, 0x1000, CRC(90bf2c4c) SHA1(7adfbf2251b5d46043d614554320e2fe544cc570) )
	ROM_LOAD( "crsnd2.a9",    0x2000, 0x1000, CRC(3b8deef1) SHA1(a14186a33a7b5ca07086ce44fb1c8c64900654d0) )
	ROM_LOAD( "crsnd3.a10",   0x3000, 0x1000, CRC(05803453) SHA1(1bba85e6494b4f67cea82d61e6cd341337bca998) )

	ROM_REGION( 0x08000, "gfx1", 0 )
	ROM_LOAD( "crcpu.3a",     0x00000, 0x2000, CRC(9d73504a) SHA1(23633ce43745c12b59916b85ca92a98e889b884e) )
	ROM_LOAD( "crcpu.4a",     0x02000, 0x2000, CRC(42a47dff) SHA1(cb936ac7ba3dd740c626a11bd6ee3c87dad4e311) )
	ROM_LOAD( "crcpu.5a",     0x04000, 0x2000, CRC(2fe4a6e1) SHA1(329cf9f71b7cbf2a13a05f9cd3397d73c69da893) )
	ROM_LOAD( "crcpu.6a",     0x06000, 0x2000, CRC(d0659042) SHA1(8f28155eb2725eaf6d1c410f76a88969541aeef3) )

	ROM_REGION( 0x20000, "gfx2", 0 )
	ROM_LOAD( "crvid.a4",     0x00000, 0x4000, CRC(579a8e36) SHA1(1053234b82877f0f8d1a2ce745b853899cfbcc22) )
	ROM_LOAD( "crvid.a3",     0x04000, 0x4000, CRC(2c2f5b29) SHA1(f7c4caabd116d3f861e70c6740cd058479477da7) )
	ROM_LOAD( "crvid.a6",     0x08000, 0x4000, CRC(5bf954e0) SHA1(8ff9549a0b5d7e6ad4671c29bf04bb9c26a995ad) )
	ROM_LOAD( "crvid.a5",     0x0c000, 0x4000, CRC(9bdec312) SHA1(3a3b44d4f73aad1f0ab2591be479f8472583f20a) )
	ROM_LOAD( "crvid.a8",     0x10000, 0x4000, CRC(4b913498) SHA1(8aa83fd6f60d6176a8781a906ba75c663ac3f3cb) )
	ROM_LOAD( "crvid.a7",     0x14000, 0x4000, CRC(9fa307d5) SHA1(ec9ba9c372d62132f02e2a87ed5f602c360755e9) )
	ROM_LOAD( "crvid.a10",    0x18000, 0x4000, CRC(7a22d6bc) SHA1(c7f97bdf13d52665e61d83b7a8f1621c6df0dbf2) )
	ROM_LOAD( "crvid.a9",     0x1c000, 0x4000, CRC(811f152d) SHA1(f9dee6f95d903a41890fe58bfc0528fecb1d6902) )

	ROM_REGION( 0x01000, "gfx3", 0 )
	ROM_LOAD( "crcpu.10g",    0x00000, 0x1000, CRC(6fe53c8d) SHA1(ceb04916af42d58f3173e5986756a0db8be11999) )
ROM_END


ROM_START( turbotag )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "ttprog0.bin",  0x0000, 0x2000, CRC(6110fd80) SHA1(544d3cd24b047797c2006a9fac499c14140960db) )
	ROM_LOAD( "ttprog1.bin",  0x2000, 0x2000, BAD_DUMP CRC(b0505e18) SHA1(b6871484f9a0663a9bcffd58368be5ae7717d29c) )
	ROM_LOAD( "ttprog2.bin",  0x4000, 0x2000, CRC(c4141237) SHA1(c1d10da4961e94bd8c2b30a2f9e0cbd8080eb56d) )
	ROM_LOAD( "ttprog3.bin",  0x6000, 0x2000, CRC(af294c6e) SHA1(c8573dd046aa2b071bef3bd86a783ee348c8d6eb) )
	ROM_LOAD( "ttprog4.bin",  0x8000, 0x2000, CRC(8c5bc1a4) SHA1(c38d7aa2639945e705856cf1449faf51a7c82ff0) )
	ROM_LOAD( "ttprog5.bin",  0xa000, 0x2000, CRC(11e62fe4) SHA1(72897702c61486b654e4b4a3f6560c144c862e1f) )
	ROM_RELOAD(               0xc000, 0x2000 )

	ROM_REGION( 0x10000, "ssio:cpu", ROMREGION_ERASE00 )

	ROM_REGION( 0x8000, "csd:cpu", 0 )  /* 32k for the Cheap Squeak Deluxe */
	ROM_LOAD16_BYTE( "ttu7.bin",  0x00000, 0x2000, CRC(8ebb3302) SHA1(c516abdae6eea524a6d2a039ed9bd7dff72ab986) )
	ROM_LOAD16_BYTE( "ttu17.bin", 0x00001, 0x2000, CRC(605d6c74) SHA1(a6c2bc95cca372fa823ab256c9dd1f92b6ba45fd) )
	ROM_LOAD16_BYTE( "ttu8.bin",  0x04000, 0x2000, CRC(6bfcb22a) SHA1(7b895e3ae1e99f195bb32b052f801b58c63a401c) )
	ROM_LOAD16_BYTE( "ttu18.bin", 0x04001, 0x2000, CRC(bb25852c) SHA1(7e0346b5e2fc16a87a157c868936978be6145e3e) )

	ROM_REGION( 0x08000, "gfx1", 0 )
	ROM_LOAD( "ttbg0.bin",    0x00000, 0x2000, CRC(1cd2023f) SHA1(03b4f9d047d4fb3c4889e797e9ffd5edabd8ecff) )
	ROM_LOAD( "ttbg1.bin",    0x02000, 0x2000, CRC(784e84cd) SHA1(cf4ddbd4b8dbfaee2f4e05c89355fad30e264641) )
	ROM_LOAD( "ttbg2.bin",    0x04000, 0x2000, CRC(da9d47d2) SHA1(5af9222f62d9948ec468fb66a5afb46a00f159c3) )
	ROM_LOAD( "ttbg3.bin",    0x06000, 0x2000, CRC(367e06a5) SHA1(24040e29bbe367b497675d155c3e0343399b179b) )

	ROM_REGION( 0x20000, "gfx2", 0 )
	ROM_LOAD( "ttfg1.bin",    0x00000, 0x4000, CRC(9d7e6ebc) SHA1(96c658091cb12d65e41f8ac5f609eb51857cef82) )
	ROM_LOAD( "ttfg0.bin",    0x04000, 0x4000, CRC(ed69e1a8) SHA1(4fc223da52d6df3182ba5ba3db8e793e379030f2) )
	ROM_LOAD( "ttfg3.bin",    0x08000, 0x4000, CRC(74e21c1c) SHA1(42ea0c569cd443866b46bcaac2c60d197fcbc2a9) )
	ROM_LOAD( "ttfg2.bin",    0x0c000, 0x4000, CRC(037ec6fc) SHA1(7bc0bd95cc673d321de33d17764a06f48109b38e) )
	ROM_LOAD( "ttfg5.bin",    0x10000, 0x4000, CRC(8b718879) SHA1(3e8361e4423e6822c09f866a99cf9fc789c99f66) )
	ROM_LOAD( "ttfg4.bin",    0x14000, 0x4000, CRC(6fdb0c13) SHA1(b3e140a838f8deaa26dbc86315603fdcef47f157) )
	ROM_LOAD( "ttfg7.bin",    0x18000, 0x4000, CRC(212019dc) SHA1(831bb83f6301d9f2f06b31a00ea425b13421b573) )
	ROM_LOAD( "ttfg6.bin",    0x1c000, 0x4000, CRC(4094e996) SHA1(f50fa31bc311c16bcd2459668f5a942a1d9de75d) )

	ROM_REGION( 0x01000, "gfx3", 0 )
	ROM_LOAD( "ttan.bin",     0x00000, 0x1000, CRC(aa0b1471) SHA1(e3dd69f1a14926c6b709d6b19d9e90a1f0867465) )
ROM_END



/*************************************
 *
 *  Driver initialization
 *
 *************************************/

void mcr3_state::mcr_common_init()
{
	save_item(NAME(m_input_mux));
	save_item(NAME(m_latched_input));
	save_item(NAME(m_last_op4));
}


void mcr3_state::init_demoderm()
{
	mcr_common_init();
	m_maincpu->space(AS_IO).install_read_handler(0x01, 0x01, read8_delegate(FUNC(mcr3_state::demoderm_ip1_r),this));
	m_maincpu->space(AS_IO).install_read_handler(0x02, 0x02, read8_delegate(FUNC(mcr3_state::demoderm_ip2_r),this));
	m_maincpu->space(AS_IO).install_write_handler(0x06, 0x06, write8_delegate(FUNC(mcr3_state::demoderm_op6_w),this));
}


void mcr3_state::init_sarge()
{
	mcr_common_init();
	m_maincpu->space(AS_IO).install_write_handler(0x06, 0x06, write8_delegate(FUNC(midway_turbo_cheap_squeak_device::write),m_turbo_cheap_squeak.target()));
}


void mcr3_state::init_maxrpm()
{
	mcr_common_init();
	m_maincpu->space(AS_IO).install_read_handler(0x01, 0x01, read8_delegate(FUNC(mcr3_state::maxrpm_ip1_r),this));
	m_maincpu->space(AS_IO).install_read_handler(0x02, 0x02, read8_delegate(FUNC(mcr3_state::maxrpm_ip2_r),this));
	m_maincpu->space(AS_IO).install_write_handler(0x05, 0x05, write8_delegate(FUNC(mcr3_state::maxrpm_op5_w),this));
	m_maincpu->space(AS_IO).install_write_handler(0x06, 0x06, write8_delegate(FUNC(mcr3_state::maxrpm_op6_w),this));

	save_item(NAME(m_maxrpm_adc_control));
	save_item(NAME(m_maxrpm_last_shift));
	save_item(NAME(m_maxrpm_p1_shift));
	save_item(NAME(m_maxrpm_p2_shift));
}


void mcr3_state::init_rampage()
{
	mcr_common_init();
	m_maincpu->space(AS_IO).install_read_handler(0x04, 0x04, read8_delegate(FUNC(mcr3_state::rampage_ip4_r),this));
	m_maincpu->space(AS_IO).install_write_handler(0x06, 0x06, write8_delegate(FUNC(mcr3_state::rampage_op6_w),this));
}


void mcr3_state::init_powerdrv()
{
	mcr_common_init();
	m_maincpu->space(AS_IO).install_read_handler(0x02, 0x02, read8_delegate(FUNC(mcr3_state::powerdrv_ip2_r),this));
	m_maincpu->space(AS_IO).install_write_handler(0x05, 0x05, write8_delegate(FUNC(mcr3_state::powerdrv_op5_w),this));
	m_maincpu->space(AS_IO).install_write_handler(0x06, 0x06, write8_delegate(FUNC(mcr3_state::powerdrv_op6_w),this));
}


void mcr3_state::init_stargrds()
{
	mcr_common_init();
	m_maincpu->space(AS_IO).install_read_handler(0x00, 0x00, read8_delegate(FUNC(mcr3_state::stargrds_ip0_r),this));
	m_maincpu->space(AS_IO).install_write_handler(0x05, 0x05, write8_delegate(FUNC(mcr3_state::stargrds_op5_w),this));
	m_maincpu->space(AS_IO).install_write_handler(0x06, 0x06, write8_delegate(FUNC(mcr3_state::stargrds_op6_w),this));
}


void mcr3_state::init_spyhunt()
{
	mcr_common_init();
	m_ssio->set_custom_input(1, 0x60, read8_delegate(FUNC(mcr3_state::spyhunt_ip1_r),this));
	m_ssio->set_custom_input(2, 0xff, read8_delegate(FUNC(mcr3_state::spyhunt_ip2_r),this));
	m_ssio->set_custom_output(4, 0xff, write8_delegate(FUNC(mcr3_state::spyhunt_op4_w),this));

	m_spyhunt_sprite_color_mask = 0x00;
	m_spyhunt_scroll_offset = 16;
}



void mcr3_state::init_crater()
{
	mcr_common_init();

	m_spyhunt_sprite_color_mask = 0x03;
	m_spyhunt_scroll_offset = 96;
}


void mcr3_state::init_turbotag()
{
	mcr_common_init();
	m_ssio->set_custom_input(1, 0x60, read8_delegate(FUNC(mcr3_state::spyhunt_ip1_r),this));
	m_ssio->set_custom_input(2, 0xff, read8_delegate(FUNC(mcr3_state::turbotag_ip2_r),this));
	m_ssio->set_custom_output(4, 0xff, write8_delegate(FUNC(mcr3_state::spyhunt_op4_w),this));

	m_spyhunt_sprite_color_mask = 0x00;
	m_spyhunt_scroll_offset = 88;

	/* the CSD 68k doesn't have any program to execute */
	m_cheap_squeak_deluxe->suspend_cpu();

	/* kludge for bad ROM read */
	m_maincpu->space(AS_PROGRAM).install_read_handler(0x0b53, 0x0b53, read8_delegate(FUNC(mcr3_state::turbotag_kludge_r),this));
}



/*************************************
 *
 *  Game drivers
 *
 *************************************/

/* MCR monoboard games */
GAME(  1984, demoderm, demoderb, mono_tcs,  demoderm, mcr3_state, init_demoderm, ROT0,  "Bally Midway", "Demolition Derby (MCR-3 Mono Board Version)", MACHINE_SUPPORTS_SAVE )
GAME(  1985, sarge,    0,        mono_tcs,  sarge,    mcr3_state, init_sarge,    ROT0,  "Bally Midway", "Sarge", MACHINE_SUPPORTS_SAVE )
GAME(  1986, maxrpm,   0,        maxrpm,    maxrpm,   mcr3_state, init_maxrpm,   ROT0,  "Bally Midway", "Max RPM (ver 2)", MACHINE_SUPPORTS_SAVE )
GAME(  1986, rampage,  0,        mono_sg,   rampage,  mcr3_state, init_rampage,  ROT0,  "Bally Midway", "Rampage (Rev 3, 8/27/86)", MACHINE_SUPPORTS_SAVE )
GAME(  1986, rampage2, rampage,  mono_sg,   rampage,  mcr3_state, init_rampage,  ROT0,  "Bally Midway", "Rampage (Rev 2, 8/4/86)", MACHINE_SUPPORTS_SAVE )
GAME(  1986, powerdrv, 0,        mono_sg,   powerdrv, mcr3_state, init_powerdrv, ROT0,  "Bally Midway", "Power Drive", MACHINE_SUPPORTS_SAVE )
GAME(  1987, stargrds, 0,        mono_sg,   stargrds, mcr3_state, init_stargrds, ROT0,  "Bally Midway", "Star Guards", MACHINE_SUPPORTS_SAVE )

/* MCR scrolling games */
GAMEL( 1983, spyhunt,  0,        mcrsc_csd, spyhunt,  mcr3_state, init_spyhunt,  ROT90, "Bally Midway", "Spy Hunter", MACHINE_SUPPORTS_SAVE, layout_spyhunt )
GAMEL( 1983, spyhuntp, spyhunt,  mcrsc_csd, spyhunt,  mcr3_state, init_spyhunt,  ROT90, "Bally Midway (Playtronic license)", "Spy Hunter (Playtronic license)", MACHINE_SUPPORTS_SAVE, layout_spyhunt )
GAME(  1984, crater,   0,        mcrscroll, crater,   mcr3_state, init_crater,   ORIENTATION_FLIP_X, "Bally Midway", "Crater Raider", MACHINE_SUPPORTS_SAVE )
GAMEL( 1985, turbotag, 0,        mcrsc_csd, turbotag, mcr3_state, init_turbotag, ROT90, "Bally Midway", "Turbo Tag (prototype)", MACHINE_SUPPORTS_SAVE, layout_turbotag )
