// license:GPL-2.0+
// copyright-holders:Juergen Buchmueller
/******************************************************************************
    Atari 400/800

    MESS Driver

    Juergen Buchmueller, June 1998

    2009-05 FP changes:
     Factored out MESS specific code from MAME
     Added skeleton support for other XL/XE machines (VERY preliminary):
     - a600xl based on maxaflex emulation in MAME
     - a1200xl sharing a800xl code without BASIC
     - a65xe, a65xea, a130xe, a800xe, xegs sharing a800xl code (and this is wrong
      at least for xegs)
     Added proper dumps and labels, thanks to Freddy Offenga researches (a few
     are still marked BAD_DUMP while waiting for crc confirmation, since they
     have been obtained by splitting whole dumps)

     To Do:
     - Find out why a600xl and a800xl don't work (xe machines should then follow)
     - Investigate supported RAM sizes and OS versions in different models
     - Implement differences between various models (currently most of the
      XL/XE are exactly an a800xl, but this will change as soon as emulation
      starts to work)
     - Fix various keyboard differences
     - Freddy emulation for 800XLF?
     - Add support for proto boards and expansions (a1400xl, C/PM board, etc.)
     - Clean up the whole driver + cart + floppy structure

    2013-11-06 Robert Tuccitto:
    Updated Palette per 'CGIA D020577' and 'GTIA C014805', including
    normalized grayscale with proper color gradient.  Added Phase Shift
    values 24.7 thru 27.7 degrees in 0.5 degree increments.  Enabled
    Phase Shift 26.2 degrees as default.

    2013-11-23 Robert Tuccitto:
    Added palette notes

******************************************************************************/

#include "emu.h"
#include "includes/atari400.h"

#include "cpu/m6502/m6502.h"
#include "machine/6821pia.h"
#include "machine/ram.h"
#include "machine/timer.h"
#include "machine/atarifdc.h"
#include "sound/dac.h"
#include "sound/pokey.h"
#include "sound/volt_reg.h"

#include "bus/a800/a800_slot.h"
#include "bus/a800/a800_carts.h"
#include "bus/a800/a8sio.h"

#include "screen.h"
#include "softlist.h"
#include "speaker.h"

/******************************************************************************
    Atari 800 memory map (preliminary)

    ***************** read access *******************
    range     short   description
    0000-9FFF RAM     main memory
    A000-BFFF RAM/ROM RAM or (banked) ROM cartridges
    C000-CFFF ROM     unused or monitor ROM

    ********* GTIA    ********************************

    D000      m0pf    missile 0 playfield collisions
    D001      m1pf    missile 1 playfield collisions
    D002      m2pf    missile 2 playfield collisions
    D003      m3pf    missile 3 playfield collisions
    D004      p0pf    player 0 playfield collisions
    D005      p1pf    player 1 playfield collisions
    D006      p2pf    player 2 playfield collisions
    D007      p3pf    player 3 playfield collisions
    D008      m0pl    missile 0 player collisions
    D009      m1pl    missile 1 player collisions
    D00A      m2pl    missile 2 player collisions
    D00B      m3pl    missile 3 player collisions
    D00C      p0pl    player 0 player collisions
    D00D      p1pl    player 1 player collisions
    D00E      p2pl    player 2 player collisions
    D00F      p3pl    player 3 player collisions
    D010      but0    button stick 0
    D011      but1    button stick 1
    D012      but2    button stick 2
    D013      but3    button stick 3
    D014      xff     unused
    D015      xff     unused
    D016      xff     unused
    D017      xff     unused
    D018      xff     unused
    D019      xff     unused
    D01A      xff     unused
    D01B      xff     unused
    D01C      xff     unused
    D01D      xff     unused
    D01E      xff     unused
    D01F      cons    console keys
    D020-D0FF repeated 7 times

    D100-D1FF xff

    ********* POKEY   ********************************
    D200      pot0    paddle 0
    D201      pot1    paddle 1
    D202      pot2    paddle 2
    D203      pot3    paddle 3
    D204      pot4    paddle 4
    D205      pot5    paddle 5
    D206      pot6    paddle 6
    D207      pot7    paddle 7
    D208      potb    all paddles
    D209      kbcode  keyboard scan code
    D20A      random  random number generator
    D20B      xff     unused
    D20C      xff     unused
    D20D      serin   serial input
    D20E      irqst   IRQ status
    D20F      skstat  sk status
    D210-D2FF repeated 15 times

    ********* PIO     ********************************
    D300      porta   read pio port A
    D301      portb   read pio port B
    D302      pactl   read pio port A control
    D303      pbctl   read pio port B control
    D304-D3FF repeated 63 times

    ********* ANTIC   ********************************
    D400      xff     unused
    D401      xff     unused
    D402      xff     unused
    D403      xff     unused
    D404      xff     unused
    D405      xff     unused
    D406      xff     unused
    D407      xff     unused
    D408      xff     unused
    D409      xff     unused
    D40A      xff     unused
    D40B      vcount  vertical (scanline) counter
    D40C      penh    light pen horizontal pos
    D40D      penv    light pen vertical pos
    D40E      xff     unused
    D40F      nmist   NMI status

    D500-D7FF xff     unused memory

    D800-DFFF ROM     floating point ROM
    E000-FFFF ROM     bios ROM

    ***************** write access *******************
    range     short   description
    0000-9FFF RAM     main memory
    A000-BFFF RAM/ROM RAM or (banked) ROM
    C000-CFFF ROM     unused or monitor ROM

    ********* GTIA    ********************************
    D000      hposp0  player 0 horz position
    D001      hposp1  player 1 horz position
    D002      hposp2  player 2 horz position
    D003      hposp3  player 3 horz position
    D004      hposm0  missile 0 horz position
    D005      hposm1  missile 0 horz position
    D006      hposm2  missile 0 horz position
    D007      hposm3  missile 0 horz position
    D008      sizep0  size player 0
    D009      sizep1  size player 0
    D00A      sizep2  size player 0
    D00B      sizep3  size player 0
    D00C      sizem   size missiles
    D00D      grafp0  graphics data for player 0
    D00E      grafp1  graphics data for player 1
    D00F      grafp2  graphics data for player 2
    D010      grafp3  graphics data for player 3
    D011      grafm   graphics data for missiles
    D012      colpm0  color for player/missile 0
    D013      colpm1  color for player/missile 1
    D014      colpm2  color for player/missile 2
    D015      colpm3  color for player/missile 3
    D016      colpf0  color 0 playfield
    D017      colpf1  color 1 playfield
    D018      colpf2  color 2 playfield
    D019      colpf3  color 3 playfield
    D01A      colbk   background playfield
    D01B      prior   priority select
    D01C      vdelay  delay until vertical retrace
    D01D      gractl  graphics control
    D01E      hitclr  clear collisions
    D01F      wcons   write console (speaker)
    D020-D0FF repeated 7 times

    D100-D1FF xff     unused

    ********* POKEY   ********************************
    D200      audf1   frequency audio chan #1
    D201      audc1   control audio chan #1
    D202      audf2   frequency audio chan #2
    D203      audc2   control audio chan #2
    D204      audf3   frequency audio chan #3
    D205      audc3   control audio chan #3
    D206      audf4   frequency audio chan #4
    D207      audc4   control audio chan #4
    D208      audctl  audio control
    D209      stimer  start timer
    D20A      skres   sk reset
    D20B      potgo   start pot AD conversion
    D20C      xff     unused
    D20D      serout  serial output
    D20E      irqen   IRQ enable
    D20F      skctl   sk control
    D210-D2FF repeated 15 times

    ********* PIO     ********************************
    D300      porta   write pio port A (output or mask)
    D301      portb   write pio port B (output or mask)
    D302      pactl   write pio port A control
    D303      pbctl   write pio port B control
    D304-D3FF         repeated

    ********* ANTIC   ********************************
    D400      dmactl  write DMA control
    D401      chactl  write character control
    D402      dlistl  write display list lo
    D403      dlisth  write display list hi
    D404      hscrol  write horz scroll
    D405      vscrol  write vert scroll
    D406      xff     unused
    D407      pmbash  player/missile base addr hi
    D408      xff     unused
    D409      chbash  character generator base addr hi
    D40A      wsync   wait for hsync
    D40B      xff     unused
    D40C      xff     unused
    D40D      xff     unused
    D40E      nmien   NMI enable
    D40F      nmires  NMI reset

    D500-D7FF xff     unused memory

    D800-DFFF ROM     floating point ROM
    E000-FFFF ROM     BIOS ROM
******************************************************************************/

class a400_state : public atari_common_state
{
public:
	a400_state(const machine_config &mconfig, device_type type, const char *tag) :
		atari_common_state(mconfig, type, tag),
		m_ram(*this, RAM_TAG),
		m_pia(*this, "pia"),
		m_dac(*this, "dac"),
		m_region_maincpu(*this, "maincpu"),
		m_cart(*this, "cartleft"),
		m_cart2(*this, "cartright")
	{ }

	void atari_common_nodac(machine_config &config);
	void atari_common(machine_config &config);
	void a800pal(machine_config &config);
	void a400pal(machine_config &config);
	void a5200(machine_config &config);
	void a800(machine_config &config);
	void a1200xl(machine_config &config);
	void a800xlpal(machine_config &config);
	void a130xe(machine_config &config);
	void a800xl(machine_config &config);
	void a600xl(machine_config &config);
	void xegs(machine_config &config);
	void a400(machine_config &config);

private:
	DECLARE_MACHINE_START(a400);
	DECLARE_MACHINE_START(a800);
	DECLARE_MACHINE_START(a800xl);
	DECLARE_MACHINE_START(a5200);
	void a400_palette(palette_device &palette) const;

	DECLARE_MACHINE_RESET(a400);

	DECLARE_WRITE8_MEMBER(gtia_cb);

	DECLARE_WRITE8_MEMBER(a600xl_pia_pb_w);
	DECLARE_WRITE8_MEMBER(a800xl_pia_pb_w);

	DECLARE_READ8_MEMBER(read_d5xx);    // at least one cart type can enable/disable roms when reading
	DECLARE_WRITE8_MEMBER(disable_cart);

	// these are needed to handle carts which can disable ROM without
	// installing/disinstalling continuously RAM and ROM (with e.g. big
	// preformance hit in Williams carts)
	DECLARE_READ8_MEMBER(special_read_8000);
	DECLARE_WRITE8_MEMBER(special_write_8000);
	DECLARE_READ8_MEMBER(special_read_a000);
	DECLARE_WRITE8_MEMBER(special_write_a000);

	DECLARE_READ8_MEMBER(a600xl_low_r);
	DECLARE_READ8_MEMBER(a1200xl_low_r);
	DECLARE_READ8_MEMBER(a800xl_low_r);
	DECLARE_WRITE8_MEMBER(a800xl_low_w);
	DECLARE_READ8_MEMBER(a800xl_high_r);
	DECLARE_WRITE8_MEMBER(a800xl_high_w);
	DECLARE_READ8_MEMBER(a130xe_low_r);
	DECLARE_WRITE8_MEMBER(a130xe_low_w);
	DECLARE_READ8_MEMBER(xegs_low_r);
	DECLARE_WRITE8_MEMBER(xegs_low_w);

	TIMER_DEVICE_CALLBACK_MEMBER(a400_interrupt);
	TIMER_DEVICE_CALLBACK_MEMBER(a800xl_interrupt);
	TIMER_DEVICE_CALLBACK_MEMBER(a5200_interrupt);

	void a1200xl_mem(address_map &map);
	void a130xe_mem(address_map &map);
	void a400_mem(address_map &map);
	void a5200_mem(address_map &map);
	void a600xl_mem(address_map &map);
	void a800xl_mem(address_map &map);
	void xegs_mem(address_map &map);

	//required_device<cpu_device> m_maincpu;    // maincpu is already contained in atari_common_state
	required_device<ram_device> m_ram;
	required_device<pia6821_device> m_pia;
	optional_device<dac_bit_interface> m_dac;
	required_memory_region m_region_maincpu;
	memory_bank *m_0000 = nullptr;
	memory_bank *m_8000 = nullptr;
	memory_bank *m_a000 = nullptr;
	optional_device<a800_cart_slot_device> m_cart;
	optional_device<a800_cart_slot_device> m_cart2;

	int m_cart_disabled, m_cart_helper;
	int m_last_offs;
	uint8_t m_mmu, m_ext_bank;

	void setup_ram(int bank, uint32_t size);
	void setup_cart(a800_cart_slot_device *slot);
};



/**************************************************************
 *
 * Memory handlers
 *
 **************************************************************/

READ8_MEMBER(a400_state::a600xl_low_r)
{
	if (m_mmu & 0x80)
		return 0xff;
	else
		return m_region_maincpu->base()[0xd000 + (offset & 0x7ff)];
}


READ8_MEMBER(a400_state::a1200xl_low_r)
{
	if (offset < 0x5000)    // 0x0000-0x4fff
		return m_ram->pointer()[offset];
	else if (offset < 0x5800)   // 0x5000-0x57ff
	{
		if (m_mmu & 0x80)
			return m_ram->pointer()[offset];
		else
			return m_region_maincpu->base()[0xd000 + (offset & 0x7ff)];
	}
	else if (offset < 0xc000)   // 0x5800-0xbfff
		return m_ram->pointer()[offset];
	else    // 0xc000-0xcfff
	{
		if (!(m_mmu & 0x01))
			return m_ram->pointer()[offset];
		else
			return m_region_maincpu->base()[offset];
	}
}

READ8_MEMBER(a400_state::a800xl_low_r)
{
	if (offset < 0x5000)    // 0x0000-0x4fff
		return m_ram->pointer()[offset];
	else if (offset < 0x5800)   // 0x5000-0x57ff
	{
		if (m_mmu & 0x80)
			return m_ram->pointer()[offset];
		else
			return m_region_maincpu->base()[0xd000 + (offset & 0x7ff)];
	}
	else if (offset < 0xa000)   // 0x5800-0x9fff
		return m_ram->pointer()[offset];
	else if (offset < 0xc000)   // 0xa000-0xbfff
	{
		if (m_mmu & 0x02)
			return m_ram->pointer()[offset];
		else
			return m_region_maincpu->base()[(offset & 0x1fff) + 0xa000];
	}
	else    // 0xc000-0xcfff
	{
		if (!(m_mmu & 0x01))
			return m_ram->pointer()[offset];
		else
			return m_region_maincpu->base()[offset];
	}
}

WRITE8_MEMBER(a400_state::a800xl_low_w)
{
	if (offset < 0x5000)    // 0x0000-0x4fff
		m_ram->pointer()[offset] = data;
	else if (offset < 0x5800)   // 0x5000-0x57ff
	{
		if (m_mmu & 0x80)
			m_ram->pointer()[offset] = data;
	}
	else if (offset < 0xa000)   // 0x5800-0x7fff
		m_ram->pointer()[offset] = data;
	else if (offset < 0xc000)   // 0xa000-0xbfff
	{
		if (m_mmu & 0x02)
			m_ram->pointer()[offset] = data;
	}
	else    // 0xc000-0xcfff
	{
		if (!(m_mmu & 0x01))
			m_ram->pointer()[offset] = data;
	}
}

READ8_MEMBER(a400_state::a800xl_high_r)
{
	if (m_mmu & 0x01)
		return m_region_maincpu->base()[0xd800 + offset];
	else
		return m_ram->pointer()[0xd800 + offset];
}

WRITE8_MEMBER(a400_state::a800xl_high_w)
{
	if (!(m_mmu & 0x01))
		m_ram->pointer()[0xd800 + offset] = data;
}

READ8_MEMBER(a400_state::a130xe_low_r)
{
	if (offset < 0x4000)    // 0x0000-0x3fff
		return m_ram->pointer()[offset];
	else if (offset < 0x8000)   // 0x4000-0x7fff
	{
		// NOTE: ANTIC accesses to extra RAM are not supported yet!
		if (!(m_mmu & 0x80) && offset >= 0x5000 && offset < 0x5800)
			return m_region_maincpu->base()[0xd000 + (offset & 0x7ff)];
		if (!(m_mmu & 0x10))
			return m_ram->pointer()[offset + 0x10000 + (m_ext_bank * 0x4000)];
		else
			return m_ram->pointer()[offset];
	}
	else if (offset < 0xa000)   // 0x8000-0x9fff
		return m_ram->pointer()[offset];
	else if (offset < 0xc000)   // 0xa000-0xbfff
	{
		if (m_mmu & 0x02)
			return m_ram->pointer()[offset];
		else
			return m_region_maincpu->base()[(offset & 0x1fff) + 0xa000];
	}
	else    // 0xc000-0xcfff
	{
		if (!(m_mmu & 0x01))
			return m_ram->pointer()[offset];
		else
			return m_region_maincpu->base()[offset];
	}
}

WRITE8_MEMBER(a400_state::a130xe_low_w)
{
	if (offset < 0x4000)    // 0x0000-0x3fff
		m_ram->pointer()[offset] = data;
	else if (offset < 0x8000)   // 0x4000-0x7fff
	{
		// NOTE: ANTIC accesses to extra RAM are not supported yet!
		if (!(m_mmu & 0x80) && offset >= 0x5000 && offset < 0x5800)
			return;
		if (!(m_mmu & 0x10))
			m_ram->pointer()[offset + 0x10000 + (m_ext_bank * 0x4000)] = data;
		else
			m_ram->pointer()[offset] = data;
	}
	else if (offset < 0xa000)   // 0x5800-0x7fff
		m_ram->pointer()[offset] = data;
	else if (offset < 0xc000)   // 0xa000-0xbfff
	{
		if (m_mmu & 0x02)
			m_ram->pointer()[offset] = data;
	}
	else    // 0xc000-0xcfff
	{
		if (!(m_mmu & 0x01))
			m_ram->pointer()[offset] = data;
	}
}

READ8_MEMBER(a400_state::xegs_low_r)
{
	if (offset < 0x5000)    // 0x0000-0x4fff
		return m_ram->pointer()[offset];
	else if (offset < 0x5800)   // 0x5000-0x57ff
	{
		if (m_mmu & 0x80)
			return m_ram->pointer()[offset];
		else
			return m_region_maincpu->base()[0xd000 + (offset & 0x7ff)];
	}
	else if (offset < 0x8000)   // 0x5800-0x7fff
		return m_ram->pointer()[offset];
	else if (offset < 0xa000)   // 0x8000-0x9fff
		return m_region_maincpu->base()[0x8000 + (offset & 0x1fff)];
	else if (offset < 0xc000)   // 0xa000-0xbfff
		return m_region_maincpu->base()[0x8000 + (offset & 0x1fff)];
	else    // 0xc000-0xcfff
	{
		if (!(m_mmu & 0x01))
			return m_ram->pointer()[offset];
		else
			return m_region_maincpu->base()[offset];
	}
}

WRITE8_MEMBER(a400_state::xegs_low_w)
{
	if (offset < 0x5000)    // 0x0000-0x4fff
		m_ram->pointer()[offset] = data;
	else if (offset < 0x5800)   // 0x5000-0x57ff
	{
		if (m_mmu & 0x80)
			m_ram->pointer()[offset] = data;
	}
	else if (offset < 0x8000)   // 0x5800-0x7fff
		m_ram->pointer()[offset] = data;
	else if (offset < 0xc000)   // 0xa000-0xbfff
		return;
	else    // 0xc000-0xcfff
	{
		if (!(m_mmu & 0x01))
			m_ram->pointer()[offset] = data;
	}
}

/**************************************************************
 *
 * Memory maps
 *
 **************************************************************/


void a400_state::a400_mem(address_map &map)
{
	map(0x0000, 0xbfff).noprw(); // RAM installed at runtime
	map(0xc000, 0xcfff).rom();
	map(0xd000, 0xd0ff).rw(m_gtia, FUNC(gtia_device::read), FUNC(gtia_device::write));
	map(0xd100, 0xd1ff).noprw();
	map(0xd200, 0xd2ff).rw("pokey", FUNC(pokey_device::read), FUNC(pokey_device::write));
	map(0xd300, 0xd3ff).rw(m_pia, FUNC(pia6821_device::read_alt), FUNC(pia6821_device::write_alt));
	map(0xd400, 0xd4ff).rw(m_antic, FUNC(antic_device::read), FUNC(antic_device::write));
	map(0xd500, 0xd7ff).noprw();
	map(0xd800, 0xffff).rom();
}


void a400_state::a600xl_mem(address_map &map)
{
	map(0x0000, 0x3fff).ram();
	map(0x5000, 0x57ff).r(FUNC(a400_state::a600xl_low_r));    // self test or NOP
	map(0xa000, 0xbfff).rom(); // BASIC
	map(0xc000, 0xcfff).rom(); // OS
	map(0xd000, 0xd0ff).rw(m_gtia, FUNC(gtia_device::read), FUNC(gtia_device::write));
	map(0xd100, 0xd1ff).noprw();
	map(0xd200, 0xd2ff).rw("pokey", FUNC(pokey_device::read), FUNC(pokey_device::write));
	map(0xd300, 0xd3ff).rw(m_pia, FUNC(pia6821_device::read_alt), FUNC(pia6821_device::write_alt));
	map(0xd400, 0xd4ff).rw(m_antic, FUNC(antic_device::read), FUNC(antic_device::write));
	map(0xd500, 0xd7ff).noprw();
	map(0xd800, 0xffff).rom(); // OS
}


void a400_state::a1200xl_mem(address_map &map)
{
	map(0x0000, 0xcfff).rw(FUNC(a400_state::a1200xl_low_r), FUNC(a400_state::xegs_low_w));
	map(0xd000, 0xd0ff).rw(m_gtia, FUNC(gtia_device::read), FUNC(gtia_device::write));
	map(0xd100, 0xd1ff).noprw();
	map(0xd200, 0xd2ff).rw("pokey", FUNC(pokey_device::read), FUNC(pokey_device::write));
	map(0xd300, 0xd3ff).rw(m_pia, FUNC(pia6821_device::read_alt), FUNC(pia6821_device::write_alt));
	map(0xd400, 0xd4ff).rw(m_antic, FUNC(antic_device::read), FUNC(antic_device::write));
	map(0xd500, 0xd7ff).noprw();
	map(0xd800, 0xffff).rw(FUNC(a400_state::a800xl_high_r), FUNC(a400_state::a800xl_high_w));
}


void a400_state::a800xl_mem(address_map &map)
{
	map(0x0000, 0xcfff).rw(FUNC(a400_state::a800xl_low_r), FUNC(a400_state::a800xl_low_w));
	map(0xd000, 0xd0ff).rw(m_gtia, FUNC(gtia_device::read), FUNC(gtia_device::write));
	map(0xd100, 0xd1ff).noprw();
	map(0xd200, 0xd2ff).rw("pokey", FUNC(pokey_device::read), FUNC(pokey_device::write));
	map(0xd300, 0xd3ff).rw(m_pia, FUNC(pia6821_device::read_alt), FUNC(pia6821_device::write_alt));
	map(0xd400, 0xd4ff).rw(m_antic, FUNC(antic_device::read), FUNC(antic_device::write));
	map(0xd500, 0xd7ff).noprw();
	map(0xd800, 0xffff).rw(FUNC(a400_state::a800xl_high_r), FUNC(a400_state::a800xl_high_w));
}


void a400_state::a130xe_mem(address_map &map)
{
	map(0x0000, 0xcfff).rw(FUNC(a400_state::a130xe_low_r), FUNC(a400_state::a800xl_low_w));
	map(0xd000, 0xd0ff).rw(m_gtia, FUNC(gtia_device::read), FUNC(gtia_device::write));
	map(0xd100, 0xd1ff).noprw();
	map(0xd200, 0xd2ff).rw("pokey", FUNC(pokey_device::read), FUNC(pokey_device::write));
	map(0xd300, 0xd3ff).rw(m_pia, FUNC(pia6821_device::read_alt), FUNC(pia6821_device::write_alt));
	map(0xd400, 0xd4ff).rw(m_antic, FUNC(antic_device::read), FUNC(antic_device::write));
	map(0xd500, 0xd7ff).noprw();
	map(0xd800, 0xffff).rw(FUNC(a400_state::a800xl_high_r), FUNC(a400_state::a800xl_high_w));
}


void a400_state::xegs_mem(address_map &map)
{
	map(0x0000, 0xcfff).rw(FUNC(a400_state::xegs_low_r), FUNC(a400_state::xegs_low_w));
	map(0xd000, 0xd0ff).rw(m_gtia, FUNC(gtia_device::read), FUNC(gtia_device::write));
	map(0xd100, 0xd1ff).noprw();
	map(0xd200, 0xd2ff).rw("pokey", FUNC(pokey_device::read), FUNC(pokey_device::write));
	map(0xd300, 0xd3ff).rw(m_pia, FUNC(pia6821_device::read_alt), FUNC(pia6821_device::write_alt));
	map(0xd400, 0xd4ff).rw(m_antic, FUNC(antic_device::read), FUNC(antic_device::write));
	map(0xd500, 0xd7ff).noprw();
	map(0xd800, 0xffff).rw(FUNC(a400_state::a800xl_high_r), FUNC(a400_state::a800xl_high_w));
}


void a400_state::a5200_mem(address_map &map)
{
	map(0x0000, 0x3fff).ram();
	map(0x4000, 0xbfff).noprw(); // ROM installed at machine start
	map(0xc000, 0xcfff).rw(m_gtia, FUNC(gtia_device::read), FUNC(gtia_device::write));
	map(0xd400, 0xdfff).rw(m_antic, FUNC(antic_device::read), FUNC(antic_device::write));
	// 0xe000-0xe7ff - Expansion?
	map(0xe800, 0xefff).rw("pokey", FUNC(pokey_device::read), FUNC(pokey_device::write));
	map(0xf000, 0xffff).rom();
}



/**************************************************************
 *
 * Input ports
 *
 **************************************************************/


#define JOYSTICK_DELTA          10
#define JOYSTICK_SENSITIVITY    200

static INPUT_PORTS_START( atari_artifacting )
	PORT_START("artifacts")
	PORT_CONFNAME(0x40, 0x00, "Television Artifacts" )
	PORT_CONFSETTING(0x00, DEF_STR( Off ) )
	PORT_CONFSETTING(0x40, DEF_STR( On ) )
INPUT_PORTS_END



static INPUT_PORTS_START( atari_console )
	PORT_START("console")
	PORT_BIT(0x04, 0x04, IPT_KEYBOARD) PORT_NAME("CONS.2: Option") PORT_CODE(KEYCODE_F3)
	PORT_BIT(0x02, 0x02, IPT_KEYBOARD) PORT_NAME("CONS.1: Select") PORT_CODE(KEYCODE_F2)
	PORT_BIT(0x01, 0x01, IPT_KEYBOARD) PORT_NAME("CONS.0: Start") PORT_CODE(KEYCODE_F1)
INPUT_PORTS_END



static INPUT_PORTS_START( atari_digital_joystick2 )
	PORT_START("djoy_0_1")
	PORT_BIT(0x01, 0x01, IPT_JOYSTICK_UP)    PORT_CODE(KEYCODE_8_PAD) PORT_CODE(JOYCODE_Y_UP_SWITCH)    PORT_PLAYER(1)
	PORT_BIT(0x02, 0x02, IPT_JOYSTICK_DOWN)  PORT_CODE(KEYCODE_2_PAD) PORT_CODE(JOYCODE_Y_DOWN_SWITCH)  PORT_PLAYER(1)
	PORT_BIT(0x04, 0x04, IPT_JOYSTICK_LEFT)  PORT_CODE(KEYCODE_4_PAD) PORT_CODE(JOYCODE_X_LEFT_SWITCH)  PORT_PLAYER(1)
	PORT_BIT(0x08, 0x08, IPT_JOYSTICK_RIGHT) PORT_CODE(KEYCODE_6_PAD) PORT_CODE(JOYCODE_X_RIGHT_SWITCH) PORT_PLAYER(1)
	PORT_BIT(0x10, 0x10, IPT_JOYSTICK_UP)    PORT_CODE(KEYCODE_8_PAD) PORT_CODE(JOYCODE_Y_UP_SWITCH)    PORT_PLAYER(2)
	PORT_BIT(0x20, 0x20, IPT_JOYSTICK_DOWN)  PORT_CODE(KEYCODE_2_PAD) PORT_CODE(JOYCODE_Y_DOWN_SWITCH)  PORT_PLAYER(2)
	PORT_BIT(0x40, 0x40, IPT_JOYSTICK_LEFT)  PORT_CODE(KEYCODE_4_PAD) PORT_CODE(JOYCODE_X_LEFT_SWITCH)  PORT_PLAYER(2)
	PORT_BIT(0x80, 0x80, IPT_JOYSTICK_RIGHT) PORT_CODE(KEYCODE_6_PAD) PORT_CODE(JOYCODE_X_RIGHT_SWITCH) PORT_PLAYER(2)

	PORT_START("djoy_2_3")
	PORT_BIT(0x01, 0x01, IPT_UNUSED)
	PORT_BIT(0x02, 0x02, IPT_UNUSED)
	PORT_BIT(0x04, 0x04, IPT_UNUSED)
	PORT_BIT(0x08, 0x08, IPT_UNUSED)
	PORT_BIT(0x10, 0x10, IPT_UNUSED)
	PORT_BIT(0x20, 0x20, IPT_UNUSED)
	PORT_BIT(0x40, 0x40, IPT_UNUSED)
	PORT_BIT(0x80, 0x80, IPT_UNUSED)

	PORT_START("djoy_b")
	PORT_BIT(0x01, 0x01, IPT_BUTTON1) PORT_CODE(KEYCODE_0_PAD) PORT_CODE(JOYCODE_BUTTON1) PORT_PLAYER(1)
	PORT_BIT(0x02, 0x02, IPT_BUTTON1) PORT_CODE(JOYCODE_BUTTON1) PORT_PLAYER(2)
	PORT_BIT(0x04, 0x04, IPT_UNUSED)
	PORT_BIT(0x08, 0x08, IPT_UNUSED)
	PORT_BIT(0x10, 0x10, IPT_BUTTON2) PORT_CODE(KEYCODE_DEL_PAD) PORT_CODE(JOYCODE_BUTTON2) PORT_PLAYER(1)
	PORT_BIT(0x20, 0x20, IPT_BUTTON2) PORT_CODE(JOYCODE_BUTTON2) PORT_PLAYER(2)
	PORT_BIT(0x40, 0x40, IPT_UNUSED)
	PORT_BIT(0x80, 0x80, IPT_UNUSED)
INPUT_PORTS_END



static INPUT_PORTS_START( atari_digital_joystick4 )
	PORT_START("djoy_0_1")
	PORT_BIT(0x01, 0x01, IPT_JOYSTICK_UP)    PORT_CODE(KEYCODE_8_PAD) PORT_CODE(JOYCODE_Y_UP_SWITCH)    PORT_PLAYER(1)
	PORT_BIT(0x02, 0x02, IPT_JOYSTICK_DOWN)  PORT_CODE(KEYCODE_2_PAD) PORT_CODE(JOYCODE_Y_DOWN_SWITCH)  PORT_PLAYER(1)
	PORT_BIT(0x04, 0x04, IPT_JOYSTICK_LEFT)  PORT_CODE(KEYCODE_4_PAD) PORT_CODE(JOYCODE_X_LEFT_SWITCH)  PORT_PLAYER(1)
	PORT_BIT(0x08, 0x08, IPT_JOYSTICK_RIGHT) PORT_CODE(KEYCODE_6_PAD) PORT_CODE(JOYCODE_X_RIGHT_SWITCH) PORT_PLAYER(1)
	PORT_BIT(0x10, 0x10, IPT_JOYSTICK_UP)    PORT_CODE(KEYCODE_8_PAD) PORT_CODE(JOYCODE_Y_UP_SWITCH)    PORT_PLAYER(2)
	PORT_BIT(0x20, 0x20, IPT_JOYSTICK_DOWN)  PORT_CODE(KEYCODE_2_PAD) PORT_CODE(JOYCODE_Y_DOWN_SWITCH)  PORT_PLAYER(2)
	PORT_BIT(0x40, 0x40, IPT_JOYSTICK_LEFT)  PORT_CODE(KEYCODE_4_PAD) PORT_CODE(JOYCODE_X_LEFT_SWITCH)  PORT_PLAYER(2)
	PORT_BIT(0x80, 0x80, IPT_JOYSTICK_RIGHT) PORT_CODE(KEYCODE_6_PAD) PORT_CODE(JOYCODE_X_RIGHT_SWITCH) PORT_PLAYER(2)

	PORT_START("djoy_2_3")
	PORT_BIT(0x01, 0x01, IPT_JOYSTICK_UP)    PORT_CODE(KEYCODE_8_PAD) PORT_CODE(JOYCODE_Y_UP_SWITCH)    PORT_PLAYER(3)
	PORT_BIT(0x02, 0x02, IPT_JOYSTICK_DOWN)  PORT_CODE(KEYCODE_2_PAD) PORT_CODE(JOYCODE_Y_DOWN_SWITCH)  PORT_PLAYER(3)
	PORT_BIT(0x04, 0x04, IPT_JOYSTICK_LEFT)  PORT_CODE(KEYCODE_4_PAD) PORT_CODE(JOYCODE_X_LEFT_SWITCH)  PORT_PLAYER(3)
	PORT_BIT(0x08, 0x08, IPT_JOYSTICK_RIGHT) PORT_CODE(KEYCODE_6_PAD) PORT_CODE(JOYCODE_X_RIGHT_SWITCH) PORT_PLAYER(3)
	PORT_BIT(0x10, 0x10, IPT_JOYSTICK_UP)    PORT_CODE(KEYCODE_8_PAD) PORT_CODE(JOYCODE_Y_UP_SWITCH)    PORT_PLAYER(4)
	PORT_BIT(0x20, 0x20, IPT_JOYSTICK_DOWN)  PORT_CODE(KEYCODE_2_PAD) PORT_CODE(JOYCODE_Y_DOWN_SWITCH)  PORT_PLAYER(4)
	PORT_BIT(0x40, 0x40, IPT_JOYSTICK_LEFT)  PORT_CODE(KEYCODE_4_PAD) PORT_CODE(JOYCODE_X_LEFT_SWITCH)  PORT_PLAYER(4)
	PORT_BIT(0x80, 0x80, IPT_JOYSTICK_RIGHT) PORT_CODE(KEYCODE_6_PAD) PORT_CODE(JOYCODE_X_RIGHT_SWITCH) PORT_PLAYER(4)

	PORT_START("djoy_b")
	PORT_BIT(0x01, 0x01, IPT_BUTTON1) PORT_CODE(KEYCODE_0_PAD) PORT_CODE(JOYCODE_BUTTON1) PORT_PLAYER(1)
	PORT_BIT(0x02, 0x02, IPT_BUTTON1) PORT_CODE(KEYCODE_0_PAD) PORT_CODE(JOYCODE_BUTTON1) PORT_PLAYER(2)
	PORT_BIT(0x04, 0x04, IPT_BUTTON1) PORT_CODE(KEYCODE_0_PAD) PORT_CODE(JOYCODE_BUTTON1) PORT_PLAYER(3)
	PORT_BIT(0x08, 0x08, IPT_BUTTON1) PORT_CODE(KEYCODE_0_PAD) PORT_CODE(JOYCODE_BUTTON1) PORT_PLAYER(4)
	PORT_BIT(0x10, 0x10, IPT_BUTTON2) PORT_CODE(KEYCODE_DEL_PAD) PORT_CODE(JOYCODE_BUTTON2) PORT_PLAYER(1)
	PORT_BIT(0x20, 0x20, IPT_BUTTON2) PORT_CODE(KEYCODE_DEL_PAD) PORT_CODE(JOYCODE_BUTTON2) PORT_PLAYER(2)
	PORT_BIT(0x40, 0x40, IPT_BUTTON2) PORT_CODE(KEYCODE_DEL_PAD) PORT_CODE(JOYCODE_BUTTON2) PORT_PLAYER(3)
	PORT_BIT(0x80, 0x80, IPT_BUTTON2) PORT_CODE(KEYCODE_DEL_PAD) PORT_CODE(JOYCODE_BUTTON2) PORT_PLAYER(4)
INPUT_PORTS_END



/* 2009-04 FP:
Small note about natural keyboard support: currently,
- "Break" is mapped to 'F1'
- "Clear" is mapped to 'F2'
- "Atari" is mapped to 'F3'                         */

static INPUT_PORTS_START( atari_keyboard )
	PORT_START("keyboard.0")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_L) PORT_CHAR('l') PORT_CHAR('L')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_J) PORT_CHAR('j') PORT_CHAR('J')
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_COLON) PORT_CHAR(';') PORT_CHAR(':')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Break") PORT_CODE(KEYCODE_BACKSPACE) PORT_CHAR(UCHAR_MAMEKEY(F1))
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_K) PORT_CHAR('k') PORT_CHAR('K')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_QUOTE) PORT_CHAR('+') PORT_CHAR('\\')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_BACKSLASH) PORT_CHAR('*') PORT_CHAR('^')

	PORT_START("keyboard.1")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_O) PORT_CHAR('o') PORT_CHAR('O')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_P) PORT_CHAR('p') PORT_CHAR('P')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_U) PORT_CHAR('u') PORT_CHAR('U')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Return") PORT_CODE(KEYCODE_ENTER) PORT_CHAR(13)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_I) PORT_CHAR('i') PORT_CHAR('I')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_OPENBRACE) PORT_CHAR('-') PORT_CHAR('_')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_CLOSEBRACE) PORT_CHAR('=') PORT_CHAR('|')

	PORT_START("keyboard.2")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_V) PORT_CHAR('v') PORT_CHAR('V')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_C) PORT_CHAR('c') PORT_CHAR('C')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_B) PORT_CHAR('b') PORT_CHAR('B')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_X) PORT_CHAR('x') PORT_CHAR('X')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_Z) PORT_CHAR('z') PORT_CHAR('Z')

	PORT_START("keyboard.3")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_4) PORT_CHAR('4') PORT_CHAR('$')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_3) PORT_CHAR('3') PORT_CHAR('#')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_6) PORT_CHAR('6') PORT_CHAR('&')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Escape") PORT_CODE(KEYCODE_ESC) PORT_CHAR(27)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_5) PORT_CHAR('5') PORT_CHAR('%')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_2) PORT_CHAR('2') PORT_CHAR('\"')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_1) PORT_CHAR('1') PORT_CHAR('!')

	PORT_START("keyboard.4")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_COMMA) PORT_CHAR(',') PORT_CHAR('[')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_SPACE) PORT_CHAR(' ')
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_STOP) PORT_CHAR('.') PORT_CHAR(']')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_N) PORT_CHAR('n') PORT_CHAR('N')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_M) PORT_CHAR('m') PORT_CHAR('M')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_SLASH) PORT_CHAR('/') PORT_CHAR('?')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Atari") PORT_CODE(KEYCODE_RCONTROL) PORT_CHAR(UCHAR_MAMEKEY(F3))

	PORT_START("keyboard.5")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_R) PORT_CHAR('r') PORT_CHAR('R')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_E) PORT_CHAR('e') PORT_CHAR('E')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_Y) PORT_CHAR('y') PORT_CHAR('Y')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Tab") PORT_CODE(KEYCODE_TAB) PORT_CHAR(UCHAR_MAMEKEY(TAB))
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_T) PORT_CHAR('t') PORT_CHAR('T')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_W) PORT_CHAR('w') PORT_CHAR('W')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_Q) PORT_CHAR('q') PORT_CHAR('Q')

	PORT_START("keyboard.6")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_9) PORT_CHAR('9') PORT_CHAR('(')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_0) PORT_CHAR('0') PORT_CHAR(')')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_7) PORT_CHAR('7') PORT_CHAR('\'')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("BackS  Delete") PORT_CODE(KEYCODE_BACKSLASH2) PORT_CHAR(8) PORT_CHAR(UCHAR_MAMEKEY(DEL))
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_8) PORT_CHAR('8') PORT_CHAR('@')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("<  Clear") PORT_CODE(KEYCODE_MINUS) PORT_CHAR('<') PORT_CHAR(UCHAR_MAMEKEY(F2))
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME(">  Insert") PORT_CODE(KEYCODE_EQUALS) PORT_CHAR('>') PORT_CHAR(UCHAR_MAMEKEY(INSERT))

	PORT_START("keyboard.7")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_F) PORT_CHAR('f') PORT_CHAR('F')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_H) PORT_CHAR('h') PORT_CHAR('H')
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_D) PORT_CHAR('d') PORT_CHAR('D')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Caps") PORT_CODE(KEYCODE_CAPSLOCK) PORT_TOGGLE PORT_CHAR(UCHAR_MAMEKEY(CAPSLOCK))
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_G) PORT_CHAR('g') PORT_CHAR('G')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_S) PORT_CHAR('s') PORT_CHAR('S')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_A) PORT_CHAR('a') PORT_CHAR('A')

	PORT_START("fake")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Shift") PORT_CODE(KEYCODE_LSHIFT) PORT_CODE(KEYCODE_RSHIFT) PORT_CHAR(UCHAR_SHIFT_1)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Ctrl") PORT_CODE(KEYCODE_LCONTROL) PORT_CHAR(UCHAR_SHIFT_2)
INPUT_PORTS_END


static INPUT_PORTS_START( atari_analog_paddles )
	PORT_START("analog_0")
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(1) PORT_REVERSE

	PORT_START("analog_1")
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(2) PORT_REVERSE

	PORT_START("analog_2")
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(3) PORT_REVERSE

	PORT_START("analog_3")
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(4) PORT_REVERSE

	PORT_START("analog_4")
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_REVERSE /* PORT_PLAYER(5) */

	PORT_START("analog_5")
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_REVERSE /* PORT_PLAYER(6) */

	PORT_START("analog_6")
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_REVERSE /* PORT_PLAYER(7) */

	PORT_START("analog_7")
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_REVERSE /* PORT_PLAYER(8) */
INPUT_PORTS_END



static INPUT_PORTS_START( a800 )
	PORT_INCLUDE( atari_artifacting )
	PORT_INCLUDE( atari_console )
	PORT_INCLUDE( atari_digital_joystick4 )
	PORT_INCLUDE( atari_keyboard )
	PORT_INCLUDE( atari_analog_paddles )
INPUT_PORTS_END



static INPUT_PORTS_START( a800xl )
	PORT_INCLUDE( atari_artifacting )
	PORT_INCLUDE( atari_console )
	PORT_INCLUDE( atari_digital_joystick2 )
	PORT_INCLUDE( atari_keyboard )
	PORT_INCLUDE( atari_analog_paddles )
INPUT_PORTS_END



static INPUT_PORTS_START( a5200 )
	PORT_INCLUDE( atari_artifacting )

	PORT_START("djoy_b")    /* lower/upper buttons */
	PORT_BIT(0x01, 0x01, IPT_BUTTON1) PORT_PLAYER(1)
	PORT_BIT(0x02, 0x02, IPT_BUTTON1) PORT_PLAYER(2)
	PORT_BIT(0x04, 0x04, IPT_BUTTON1) PORT_PLAYER(3)
	PORT_BIT(0x08, 0x08, IPT_BUTTON1) PORT_PLAYER(4)
	PORT_BIT(0x10, 0x10, IPT_BUTTON2) PORT_PLAYER(1)
	PORT_BIT(0x20, 0x20, IPT_BUTTON2) PORT_PLAYER(2)
	PORT_BIT(0x40, 0x40, IPT_BUTTON2) PORT_PLAYER(3)
	PORT_BIT(0x80, 0x80, IPT_BUTTON2) PORT_PLAYER(4)

	PORT_START("keypad.0")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("(Break)") PORT_CODE(KEYCODE_PAUSE)    // is this correct?
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[#]") PORT_CODE(KEYCODE_ENTER_PAD)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[0]") PORT_CODE(KEYCODE_0_PAD)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[*]") PORT_CODE(KEYCODE_PLUS_PAD)
	PORT_BIT(0xf0, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("keypad.1")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("Reset") PORT_CODE(KEYCODE_F3)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[9]") PORT_CODE(KEYCODE_9_PAD)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[8]") PORT_CODE(KEYCODE_8_PAD)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[7]") PORT_CODE(KEYCODE_7_PAD)
	PORT_BIT(0xf0, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("keypad.2")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME(DEF_STR(Pause)) PORT_CODE(KEYCODE_F2)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[6]") PORT_CODE(KEYCODE_6_PAD)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[5]") PORT_CODE(KEYCODE_5_PAD)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[4]") PORT_CODE(KEYCODE_4_PAD)
	PORT_BIT(0xf0, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("keypad.3")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_START)    PORT_NAME("Start")
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[3]") PORT_CODE(KEYCODE_3_PAD)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[2]") PORT_CODE(KEYCODE_2_PAD)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[1]") PORT_CODE(KEYCODE_1_PAD)
	PORT_BIT(0xf0, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("analog_0")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_X) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(1)

	PORT_START("analog_1")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_Y) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(1)

	PORT_START("analog_2")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_X) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(2)

	PORT_START("analog_3")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_Y) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(2)

	PORT_START("analog_4")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_X) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(3)

	PORT_START("analog_5")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_Y) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(3)

	PORT_START("analog_6")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_X) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(4)

	PORT_START("analog_7")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_Y) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(4)

INPUT_PORTS_END
/***************************************************************
Atari 5200 Palette Notes:

Palette on a modern flat panel display (LCD, LED, Plasma, etc.)
appears different from a traditional CRT. The most outstanding
difference is Hue 1x, the hue begin point. Hue 1x looks very
'green' (~-60 to -45 degrees - depending on how poor or well it
handles the signal conversion and its calibration) on a modern
flat panel display, as opposed to 'gold' (~-33 degrees) on a
CRT.  The official technical document, "GTIA C014805 NTSC"
stipulates Hue 1x as gold.

The "Atari 5200 Field Service Manual" provides two different
sets of instructions in harmony with utilizing the "PAM
Diagnostic SALT Cartridge v1.1".  In one account it states the
color just below and above the reference bar to be within one
shade of each other.

Under the same reference document, directions are given for it
to be the same color.  Phase Shift 25.7 degrees matches Hue 1x,
15x and the color below the reference bar.

However, if the system is adjusted within the first several
minutes of running, the warm up, consistent system run time,
causes Hue 15x (F$) to become stronger/darker gold (More brown
then ultimately red-brown); as well as leans Hue 14x (E$) more
brown than green.  Once achieving a phase shift of 27.7,
Hue 14x (E$) and Hue 15x (F$) near-exact match Hue 1x and 2x
respectively.

Accounting for system 'warm-up', phase shifting, as well as the
instructions for it to be within one shade of each other, would
make Phase Shift 26.2 degrees or 26.7 degrees a realistic
logical choice.

It also collaborates with the official "GTIA C014805 NTSC"
document for color order: Hue 1x = Gold, Hue 2x = Orange,
Hue 15x (F$) = Light-Orange; Phase Shift 26.2 places
Hue 15x (F$) between Hue 1x, Gold and Hue 2x, Orange;
a Light Orange in color.  Color descriptions are best measured
in the middle of the brightness scale.

It should be mentioned that Green-Yellow is referenced at
Hue 13x (D$), nowhere near Hue 1x.  A Green-Yellow Hue 1x is
how the palette is manipulated and modified (in part) under
a modern flat panel display.

Note though, even a properly calibrated console, at power on,
the phase shift appears as low as ~23 degrees and after a
considerable consistent runtime, can be as high as ~28 degrees.
In general, the low end of ~23 degrees lasts for maybe several
seconds, whereas higher values such as ~25-27 degrees is the
most dominant during system run time.

Additionally, the blue to red (And consequently blue to green)
ratio proportions may appear different on a modern flat panel
display than a CRT in some instances for the Atari 5200 system.
Furthermore, you may have some variation of proportions even
within the same display type.

One side effect of this on the console's palette is that some
values of red may appear too pinkish - Too much blue to red.
This is not the same as a traditional tint-hue control
adjustment; rather, can be demonstrated by changing the blue
ratio values via MESS HLSL settings.

Lastly, the Atari 2600 & 7800 NTSC color palettes hold the same
hue structure order and have similar appearance differences
dependent upon display type.
***************************************************************/
/**************************************************************
 *
 * Palette - Phase Shift 26.2
 *
 **************************************************************/

static const uint8_t atari_colors[256*3] =
{
	/* Grey */
	0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
	0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
	0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
	0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,
	/* Gold */
	0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
	0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
	0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
	0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,
	/* Orange */
	0x31,0x00,0x00, 0x42,0x06,0x00, 0x53,0x17,0x00, 0x64,0x28,0x00,
	0x75,0x39,0x00, 0x86,0x4A,0x00, 0x97,0x5B,0x0B, 0xA8,0x6C,0x1C,
	0xB9,0x7D,0x2D, 0xCA,0x8E,0x3E, 0xDB,0x9F,0x4F, 0xEC,0xB0,0x60,
	0xFD,0xC1,0x71, 0xFF,0xD2,0x86, 0xFF,0xE3,0x9D, 0xFF,0xF4,0xB3,
	/* Red-Orange */
	0x3E,0x00,0x00, 0x4F,0x00,0x00, 0x60,0x08,0x00, 0x71,0x19,0x00,
	0x82,0x2A,0x0F, 0x93,0x3B,0x20, 0xA4,0x4C,0x31, 0xB5,0x5D,0x42,
	0xC6,0x6E,0x53, 0xD7,0x7F,0x64, 0xE8,0x90,0x75, 0xF9,0xA1,0x86,
	0xFF,0xB2,0x9A, 0xFF,0xC3,0xB0, 0xFF,0xD4,0xC6, 0xFF,0xE5,0xDC,
	/* Pink */
	0x3E,0x00,0x06, 0x4F,0x00,0x12, 0x60,0x00,0x1E, 0x71,0x0E,0x2E,
	0x82,0x1F,0x3F, 0x93,0x30,0x50, 0xA4,0x41,0x61, 0xB5,0x52,0x72,
	0xC6,0x63,0x83, 0xD7,0x74,0x94, 0xE8,0x85,0xA5, 0xF9,0x96,0xB6,
	0xFF,0xA7,0xCB, 0xFF,0xB8,0xE1, 0xFF,0xC9,0xEF, 0xFF,0xDA,0xF4,
	/* Purple */
	0x32,0x00,0x38, 0x43,0x00,0x44, 0x54,0x00,0x50, 0x65,0x0C,0x5F,
	0x76,0x1D,0x70, 0x87,0x2E,0x81, 0x98,0x3F,0x92, 0xA9,0x50,0xA3,
	0xBA,0x61,0xB4, 0xCB,0x72,0xC5, 0xDC,0x83,0xD6, 0xED,0x94,0xE4,
	0xFE,0xA5,0xE4, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,
	/* Purple-Blue */
	0x1B,0x00,0x5F, 0x2C,0x00,0x6B, 0x3D,0x00,0x77, 0x4E,0x11,0x88,
	0x5F,0x22,0x99, 0x70,0x33,0xAA, 0x81,0x44,0xBB, 0x92,0x55,0xCC,
	0xA3,0x66,0xDD, 0xB4,0x77,0xED, 0xC5,0x88,0xED, 0xD6,0x99,0xED,
	0xE7,0xAA,0xED, 0xF8,0xBB,0xED, 0xFF,0xCC,0xF0, 0xFF,0xDD,0xF5,
	/* Blue 1 */
	0x00,0x00,0x72, 0x10,0x00,0x7E, 0x21,0x0D,0x8E, 0x32,0x1E,0x9F,
	0x43,0x2F,0xB0, 0x54,0x40,0xC1, 0x65,0x51,0xD2, 0x76,0x62,0xE3,
	0x87,0x73,0xF4, 0x98,0x84,0xF9, 0xA9,0x95,0xF9, 0xBA,0xA6,0xF9,
	0xCB,0xB7,0xF9, 0xDC,0xC8,0xF9, 0xED,0xD9,0xF9, 0xFE,0xEA,0xF9,
	/* Blue 2 */
	0x00,0x00,0x65, 0x00,0x0C,0x7A, 0x05,0x1D,0x8E, 0x16,0x2E,0x9F,
	0x27,0x3F,0xB0, 0x38,0x50,0xC1, 0x49,0x61,0xD2, 0x5A,0x72,0xE3,
	0x6B,0x83,0xF4, 0x7C,0x94,0xFF, 0x8D,0xA5,0xFF, 0x9E,0xB6,0xFF,
	0xAF,0xC7,0xFF, 0xC0,0xD8,0xFF, 0xD1,0xE9,0xFF, 0xE2,0xFA,0xFF,
	/* Light-Blue */
	0x00,0x0D,0x48, 0x00,0x1E,0x5E, 0x00,0x2F,0x74, 0x00,0x40,0x8A,
	0x11,0x51,0x9B, 0x22,0x62,0xAC, 0x33,0x73,0xBD, 0x44,0x84,0xCE,
	0x55,0x95,0xDF, 0x66,0xA6,0xF0, 0x77,0xB7,0xFF, 0x88,0xC8,0xFF,
	0x99,0xD9,0xFF, 0xAA,0xEA,0xFF, 0xBB,0xFB,0xFF, 0xCC,0xFF,0xFF,
	/* Turquoise */
	0x00,0x1C,0x1C, 0x00,0x2D,0x32, 0x00,0x3E,0x49, 0x00,0x4F,0x5F,
	0x05,0x60,0x73, 0x16,0x71,0x84, 0x27,0x82,0x95, 0x38,0x93,0xA6,
	0x49,0xA4,0xB7, 0x5A,0xB5,0xC8, 0x6B,0xC6,0xD9, 0x7C,0xD7,0xEA,
	0x8D,0xE8,0xFB, 0x9E,0xF9,0xFF, 0xAF,0xFF,0xFF, 0xC0,0xFF,0xFF,
	/* Green-Blue */
	0x00,0x25,0x0B, 0x00,0x36,0x10, 0x00,0x47,0x18, 0x00,0x58,0x2E,
	0x07,0x69,0x42, 0x18,0x7A,0x53, 0x29,0x8B,0x64, 0x3A,0x9C,0x75,
	0x4B,0xAD,0x86, 0x5C,0xBE,0x97, 0x6D,0xCF,0xA8, 0x7E,0xE0,0xB9,
	0x8F,0xF1,0xCA, 0xA0,0xFF,0xDA, 0xB1,0xFF,0xE6, 0xC2,0xFF,0xF2,
	/* Green */
	0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x04,0x5A,0x1A,
	0x15,0x6B,0x1A, 0x26,0x7C,0x22, 0x37,0x8D,0x33, 0x48,0x9E,0x44,
	0x59,0xAF,0x55, 0x6A,0xC0,0x66, 0x7B,0xD1,0x77, 0x8C,0xE2,0x88,
	0x9D,0xF3,0x99, 0xAE,0xFF,0xA8, 0xBF,0xFF,0xB4, 0xD0,0xFF,0xC0,
	/* Yellow-Green */
	0x00,0x21,0x0A, 0x00,0x32,0x0F, 0x0A,0x43,0x11, 0x1B,0x54,0x11,
	0x2C,0x65,0x11, 0x3D,0x76,0x11, 0x4E,0x87,0x11, 0x5F,0x98,0x1E,
	0x70,0xA9,0x2F, 0x81,0xBA,0x40, 0x92,0xCB,0x51, 0xA3,0xDC,0x62,
	0xB4,0xED,0x73, 0xC5,0xFE,0x84, 0xD6,0xFF,0x90, 0xE7,0xFF,0x9C,
	/* Orange-Green */
	0x05,0x13,0x04, 0x16,0x24,0x04, 0x27,0x35,0x04, 0x38,0x46,0x04,
	0x49,0x57,0x04, 0x5A,0x68,0x04, 0x6B,0x79,0x04, 0x7C,0x8A,0x09,
	0x8D,0x9B,0x1A, 0x9E,0xAC,0x2B, 0xAF,0xBD,0x3C, 0xC0,0xCE,0x4D,
	0xD1,0xDF,0x5E, 0xE2,0xF0,0x6F, 0xF3,0xFF,0x80, 0xFF,0xFF,0x8D,
	/* Light-Orange */
	0x21,0x02,0x00, 0x32,0x13,0x00, 0x43,0x24,0x00, 0x54,0x35,0x00,
	0x65,0x46,0x00, 0x76,0x57,0x00, 0x87,0x68,0x00, 0x98,0x79,0x0C,
	0xA9,0x8A,0x1D, 0xBA,0x9B,0x2E, 0xCB,0xAC,0x3F, 0xDC,0xBD,0x50,
	0xED,0xCE,0x61, 0xFE,0xDF,0x72, 0xFF,0xF0,0x87, 0xFF,0xFF,0x9D
};


/* Initialise the palette */
void a400_state::a400_palette(palette_device &palette) const
{
	for (unsigned i = 0; i < ARRAY_LENGTH(atari_colors) / 3; i++)
		palette.set_pen_color(i, atari_colors[i * 3], atari_colors[i * 3 + 1], atari_colors[i * 3 + 2]);
}
/******************************************************************
    PALETTE - PHASE 24.7 SHIFT

GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,

ORANGE
    0x30,0x00,0x00, 0x41,0x07,0x00, 0x52,0x18,0x00, 0x63,0x29,0x00,
    0x74,0x3A,0x00, 0x85,0x4B,0x00, 0x96,0x5C,0x0A, 0xA7,0x6D,0x1B,
    0xB8,0x7E,0x2C, 0xC9,0x8F,0x3D, 0xDA,0xA0,0x4E, 0xEB,0xB1,0x5F,
    0xFC,0xC2,0x70, 0xFF,0xD3,0x85, 0xFF,0xE4,0x9B, 0xFF,0xF5,0xB1,

RED-ORANGE
    0x3D,0x00,0x00, 0x4E,0x00,0x00, 0x5F,0x09,0x00, 0x70,0x1A,0x00,
    0x81,0x2B,0x09, 0x92,0x3C,0x1A, 0xA3,0x4D,0x2B, 0xB4,0x5E,0x3C,
    0xC5,0x6F,0x4D, 0xD6,0x80,0x5E, 0xE7,0x91,0x6F, 0xF8,0xA2,0x80,
    0xFF,0xB3,0x94, 0xFF,0xC4,0xAA, 0xFF,0xD5,0xC0, 0xFF,0xE6,0xD6,

PINK
    0x3F,0x00,0x00, 0x50,0x00,0x09, 0x61,0x00,0x15, 0x72,0x10,0x26,
    0x83,0x21,0x37, 0x94,0x32,0x48, 0xA5,0x43,0x59, 0xB6,0x54,0x6A,
    0xC7,0x65,0x7B, 0xD8,0x76,0x8C, 0xE9,0x87,0x9D, 0xFA,0x98,0xAE,
    0xFF,0xA9,0xC2, 0xFF,0xBA,0xD8, 0xFF,0xCB,0xEE, 0xFF,0xDC,0xF4,

PURPLE
    0x36,0x00,0x2E, 0x47,0x00,0x3A, 0x58,0x00,0x46, 0x69,0x0C,0x55,
    0x7A,0x1D,0x66, 0x8B,0x2E,0x77, 0x9C,0x3F,0x88, 0xAD,0x50,0x99,
    0xBE,0x61,0xAA, 0xCF,0x72,0xBB, 0xE0,0x83,0xCC, 0xF1,0x94,0xDD,
    0xFF,0xA5,0xE4, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x23,0x00,0x55, 0x34,0x00,0x61, 0x45,0x00,0x6D, 0x56,0x0F,0x7E,
    0x67,0x20,0x8F, 0x78,0x31,0xA0, 0x89,0x42,0xB1, 0x9A,0x53,0xC2,
    0xAB,0x64,0xD3, 0xBC,0x75,0xE4, 0xCD,0x86,0xEA, 0xDE,0x97,0xEA,
    0xEF,0xA8,0xEA, 0xFF,0xB9,0xEA, 0xFF,0xCA,0xEF, 0xFF,0xDB,0xF4,

BLUE1
    0x09,0x00,0x6E, 0x1A,0x00,0x7A, 0x2B,0x08,0x88, 0x3C,0x19,0x99,
    0x4D,0x2A,0xAA, 0x5E,0x3B,0xBB, 0x6F,0x4C,0xCC, 0x80,0x5D,0xDD,
    0x91,0x6E,0xEE, 0xA2,0x7F,0xF4, 0xB3,0x90,0xF4, 0xC4,0xA1,0xF4,
    0xD5,0xB2,0xF4, 0xE6,0xC3,0xF4, 0xF7,0xD4,0xF4, 0xFF,0xE5,0xF7,

BLUE2
    0x00,0x00,0x6D, 0x00,0x05,0x80, 0x10,0x16,0x91, 0x21,0x27,0xA2,
    0x32,0x38,0xB3, 0x43,0x49,0xC4, 0x54,0x5A,0xD5, 0x65,0x6B,0xE6,
    0x76,0x7C,0xF7, 0x87,0x8D,0xFF, 0x98,0x9E,0xFF, 0xA9,0xAF,0xFF,
    0xBA,0xC0,0xFF, 0xCB,0xD1,0xFF, 0xDC,0xE2,0xFF, 0xED,0xF3,0xFF

LIGHT-BLUE
    0x00,0x05,0x57, 0x00,0x16,0x6E, 0x00,0x27,0x84, 0x09,0x38,0x97,
    0x1A,0x49,0xA8, 0x2B,0x5A,0xB9, 0x3C,0x6B,0xCA, 0x4D,0x7C,0xDB,
    0x5E,0x8D,0xEC, 0x6F,0x9E,0xFD, 0x80,0xAF,0xFF, 0x91,0xC0,0xFF,
    0xA2,0xD1,0xFF, 0xB3,0xE2,0xFF, 0xC4,0xF3,0xFF, 0xD5,0xFF,0xFF,

TURQUOISE
    0x00,0x15,0x34, 0x00,0x26,0x4A, 0x00,0x37,0x60, 0x00,0x48,0x77,
    0x0A,0x59,0x8A, 0x1B,0x6A,0x9B, 0x2C,0x7B,0xAC, 0x3D,0x8C,0xBD,
    0x4E,0x9D,0xCE, 0x5F,0xAE,0xDF, 0x70,0xBF,0xF0, 0x81,0xD0,0xFF,
    0x92,0xE1,0xFF, 0xA3,0xF2,0xFF, 0xB4,0xFF,0xFF, 0xC5,0xFF,0xFF

GREEN-BLUE
    0x00,0x21,0x0A, 0x00,0x32,0x1F, 0x00,0x43,0x35, 0x00,0x54,0x4B,
    0x04,0x65,0x60, 0x15,0x76,0x71, 0x26,0x87,0x82, 0x37,0x98,0x93,
    0x48,0xA9,0xA4, 0x59,0xBA,0xB5, 0x6A,0xCB,0xC6, 0x7B,0xDC,0xD7,
    0x8C,0xED,0xE8, 0x9D,0xFE,0xF9, 0xAE,0xFF,0xFF, 0xBF,0xFF,0xFF,

GREEN
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1D,
    0x0A,0x6B,0x30, 0x1B,0x7C,0x41, 0x2C,0x8D,0x52, 0x3D,0x9E,0x63,
    0x4E,0xAF,0x74, 0x5F,0xC0,0x85, 0x70,0xD1,0x96, 0x81,0xE2,0xA7,
    0x92,0xF3,0xB8, 0xA3,0xFF,0xC8, 0xB4,0xFF,0xD3, 0xC5,0xFF,0xDF,

YELLOW-GREEN
    0x00,0x26,0x0B, 0x00,0x37,0x10, 0x00,0x48,0x16, 0x0A,0x59,0x18,
    0x1B,0x6A,0x18, 0x2C,0x7B,0x18, 0x3D,0x8C,0x27, 0x4E,0x9D,0x38,
    0x5F,0xAE,0x49, 0x70,0xBF,0x5A, 0x81,0xD0,0x6B, 0x92,0xE1,0x7C,
    0xA3,0xF2,0x8D, 0xB4,0xFF,0x9C, 0xC5,0xFF,0xA8, 0xD6,0xFF,0xB4,

ORANGE-GREEN
    0x00,0x1E,0x09, 0x00,0x2F,0x0E, 0x11,0x40,0x0E, 0x22,0x51,0x0E,
    0x33,0x62,0x0E, 0x44,0x73,0x0E, 0x55,0x84,0x0E, 0x66,0x95,0x17,
    0x77,0xA6,0x28, 0x88,0xB7,0x39, 0x99,0xC8,0x4A, 0xAA,0xD9,0x5B,
    0xBB,0xEA,0x6C, 0xCC,0xFB,0x7D, 0xDD,0xFF,0x8A, 0xEE,0xFF,0x96,

LIGHT-ORANGE
    0x0A,0x11,0x02, 0x1B,0x22,0x02, 0x2C,0x33,0x02, 0x3D,0x44,0x02,
    0x4E,0x55,0x02, 0x5F,0x66,0x02, 0x70,0x77,0x02, 0x81,0x88,0x09,
    0x92,0x99,0x1A, 0xA3,0xAA,0x2B, 0xB4,0xBB,0x3C, 0xC5,0xCC,0x4D,
    0xD6,0xDD,0x5E, 0xE7,0xEE,0x6F, 0xF8,0xFF,0x80, 0xFF,0xFF,0x8F,
*******************************************************************

*******************************************************************
    PALETTE - PHASE 25.2 SHIFT

GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,

ORANGE
    0x30,0x00,0x00, 0x41,0x07,0x00, 0x52,0x18,0x00, 0x63,0x29,0x00,
    0x74,0x3A,0x00, 0x85,0x4B,0x00, 0x96,0x5C,0x0A, 0xA7,0x6D,0x1B,
    0xB8,0x7E,0x2C, 0xC9,0x8F,0x3D, 0xDA,0xA0,0x4E, 0xEB,0xB1,0x5F,
    0xFC,0xC2,0x70, 0xFF,0xD3,0x85, 0xFF,0xE4,0x9B, 0xFF,0xF5,0xB1,

RED-ORANGE
    0x3E,0x00,0x00, 0x4F,0x00,0x00, 0x60,0x09,0x00, 0x71,0x1A,0x00,
    0x82,0x2B,0x0B, 0x93,0x3C,0x1C, 0xA4,0x4D,0x2D, 0xB5,0x5E,0x3E,
    0xC6,0x6F,0x4F, 0xD7,0x80,0x60, 0xE8,0x91,0x71, 0xF9,0xA2,0x82,
    0xFF,0xB3,0x96, 0xFF,0xC4,0xAC, 0xFF,0xD5,0xC2, 0xFF,0xE6,0xD8,

PINK
    0x3F,0x00,0x00, 0x50,0x00,0x0C, 0x61,0x00,0x18, 0x72,0x0F,0x28,
    0x83,0x20,0x39, 0x94,0x31,0x4A, 0xA5,0x42,0x5B, 0xB6,0x53,0x6C,
    0xC7,0x64,0x7D, 0xD8,0x75,0x8E, 0xE9,0x86,0x9F, 0xFA,0x97,0xB0,
    0xFF,0xA8,0xC5, 0xFF,0xB9,0xDB, 0xFF,0xCA,0xEF, 0xFF,0xDB,0xF4,

PURPLE
    0x35,0x00,0x31, 0x46,0x00,0x3D, 0x57,0x00,0x49, 0x68,0x0C,0x58,
    0x79,0x1D,0x69, 0x8A,0x2E,0x7A, 0x9B,0x3F,0x8B, 0xAC,0x50,0x9C,
    0xBD,0x61,0xAD, 0xCE,0x72,0xBE, 0xDF,0x83,0xCF, 0xF0,0x94,0xE0,
    0xFF,0xA5,0xE4, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x20,0x00,0x59, 0x31,0x00,0x65, 0x42,0x00,0x71, 0x53,0x10,0x82,
    0x64,0x21,0x93, 0x75,0x32,0xA4, 0x86,0x43,0xB5, 0x97,0x54,0xC6,
    0xA8,0x65,0xD7, 0xB9,0x76,0xE8, 0xCA,0x87,0xEB, 0xDB,0x98,0xEB,
    0xEC,0xA9,0xEB, 0xFD,0xBA,0xEB, 0xFF,0xCB,0xEF, 0xFF,0xDC,0xF4,

BLUE1
    0x05,0x00,0x70, 0x16,0x00,0x7C, 0x27,0x09,0x8B, 0x38,0x1A,0x9C,
    0x49,0x2B,0xAD, 0x5A,0x3C,0xBE, 0x6B,0x4D,0xCF, 0x7C,0x5E,0xE0,
    0X8D,0x6F,0xF1, 0x9E,0x80,0xF6, 0xAF,0x91,0xF6, 0xC0,0xA2,0xF6,
    0xD1,0xB3,0xF6, 0xE2,0xC4,0xF6, 0xF3,0xD5,0xF6, 0xFF,0xE6,0xF7,

BLUE2
    0x00,0x00,0x6B, 0x00,0x08,0x7E, 0x0C,0x19,0x91, 0x1D,0x2A,0xA2,
    0x2E,0x3B,0xB3, 0x3F,0x4C,0xC4, 0x50,0x5D,0xD5, 0x61,0x6E,0xE6,
    0x72,0x7F,0xF7, 0x83,0x90,0xFF, 0x94,0xA1,0xFF, 0xA5,0xB2,0xFF,
    0xB6,0xC3,0xFF, 0xC7,0xD4,0xFF, 0xD8,0xE5,0xFF, 0xE9,0xF6,0xFF,

LIGHT-BLUE
    0x00,0x08,0x52, 0x00,0x19,0x68, 0x00,0x2A,0x7F, 0x05,0x3B,0x93,
    0x16,0x4C,0xA4, 0x27,0x5D,0xB5, 0x38,0x6E,0xC6, 0x49,0x7F,0xD7,
    0x5A,0x90,0xE8, 0x6B,0xA1,0xF9, 0x7C,0xB2,0xFF, 0x8D,0xC3,0xFF,
    0x9E,0xD4,0xFF, 0xAF,0xE5,0xFF, 0xC0,0xF6,0xFF, 0xD1,0xFF,0xFF,

TURQUOISE
    0x00,0x17,0x2D, 0x00,0x28,0x43, 0x00,0x39,0x59, 0x00,0x4A,0x6F,
    0x08,0x5B,0x83, 0x19,0x6C,0x94, 0x2A,0x7D,0xA5, 0x3B,0x8E,0xB6,
    0x4C,0x9F,0xC7, 0x5D,0xB0,0xD8, 0x6E,0xC1,0xE9, 0x7F,0xD2,0xFA,
    0x90,0xE3,0xFF, 0xA1,0xF4,0xFF, 0xB2,0xFF,0xFF, 0xC3,0xFF,0xFF,

GREEN-BLUE
    0x00,0x23,0x0A, 0x00,0x34,0x15, 0x00,0x45,0x2B, 0x00,0x56,0x41,
    0x04,0x67,0x56, 0x15,0x78,0x67, 0x26,0x89,0x78, 0x37,0x9A,0x89,
    0x48,0xAB,0x9A, 0x59,0xBC,0xAB, 0x6A,0xCD,0xBC, 0x7B,0xDE,0xCD,
    0x8C,0xEF,0xDE, 0x9D,0xFF,0xEE, 0xAE,0xFF,0xFA, 0xBF,0xFF,0xFF,

GREEN
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1B,
    0x0D,0x6B,0x25, 0x1E,0x7C,0x36, 0x2F,0x8D,0x47, 0x40,0x9E,0x58,
    0x51,0xAF,0x69, 0x62,0xC0,0x7A, 0x73,0xD1,0x8B, 0x84,0xE2,0x9C,
    0x95,0xF3,0xAD, 0xA6,0xFF,0xBD, 0xB7,0xFF,0xC9, 0xC8,0xFF,0xD4,

YELLOW-GREEN
    0x00,0x24,0x0B, 0x00,0x35,0x10, 0x00,0x46,0x15, 0x10,0x57,0x15,
    0x21,0x68,0x15, 0x32,0x79,0x15, 0x43,0x8A,0x1C, 0x54,0x9B,0x2D,
    0x65,0xAC,0x3E, 0x76,0xBD,0x4F, 0x87,0xCE,0x60, 0x98,0xDF,0x71,
    0xA9,0xF0,0x82, 0xBA,0xFF,0x93, 0xCB,0xFF,0x9F, 0xDC,0xFF,0xAA,

ORANGE-GREEN
    0x00,0x1B,0x08, 0x08,0x2C,0x0B, 0x19,0x3D,0x0B, 0x2A,0x4E,0x0B,
    0x3B,0x5F,0x0B, 0x4C,0x70,0x0B, 0x5D,0x81,0x0B, 0x6E,0x92,0x11,
    0x7F,0xA3,0x22, 0x90,0xB4,0x33, 0xA1,0xC5,0x44, 0xB2,0xD6,0x55,
    0xC3,0xE7,0x66, 0xD4,0xF8,0x77, 0xE5,0xFF,0x85, 0xF6,0xFF,0x91,

LIGHT-ORANGE
    0x12,0x0C,0x00, 0x23,0x1D,0x00, 0x34,0x2E,0x00, 0x45,0x3F,0x00,
    0x56,0x50,0x00, 0x67,0x61,0x00, 0x78,0x72,0x00, 0x89,0x83,0x08,
    0x9A,0x94,0x19, 0xAB,0xA5,0x2A, 0xBC,0xB6,0x3B, 0xCD,0xC7,0x4C,
    0xDE,0xD8,0x5D, 0xEF,0xE9,0x6E, 0xFF,0xFA,0x80, 0xFF,0xFF,0x92,
*******************************************************************

*******************************************************************
    PALETTE - PHASE 25.7 SHIFT
GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,

ORANGE
    0x31,0x00,0x00, 0x42,0x06,0x00, 0x53,0x17,0x00, 0x64,0x28,0x00,
    0x75,0x39,0x00, 0x86,0X4A,0x00, 0x97,0x5B,0x0A, 0xA8,0x6C,0x1B,
    0xB9,0x7D,0x2C, 0xCA,0x8E,0x3D, 0xDB,0x9F,0x4E, 0xEC,0xB0,0x5F,
    0xFD,0xC1,0x70, 0xFF,0xD2,0x85, 0xFF,0xE3,0x9C, 0xFF,0xF4,0xB2,

RED-ORANGE
    0x3E,0x00,0x00, 0x4F,0x00,0x00, 0x60,0x08,0x00, 0x71,0x19,0x00,
    0x82,0x2A,0x0D, 0x93,0x3B,0x1E, 0xA4,0x4C,0x2F, 0xB5,0x5D,0x40,
    0xC6,0x6E,0x51, 0xD7,0x7F,0x62, 0xE8,0x90,0x73, 0xF9,0xA1,0x83,
    0xFF,0xB2,0x98, 0xFF,0xC3,0xAE, 0xFF,0xD4,0xC4, 0xFF,0xE5,0xDA,

PINK
    0x3F,0x00,0x03, 0x50,0x00,0x0F, 0x61,0x00,0x1B, 0x72,0x0F,0x2B,
    0x83,0x20,0x3C, 0x94,0x31,0x4D, 0xA5,0x42,0x5E, 0xB6,0x53,0x6F,
    0xC7,0x64,0x80, 0xD8,0x75,0x91, 0xE9,0x86,0xA2, 0xFA,0x97,0xB3,
    0xFF,0xA8,0xC8, 0xFF,0xB9,0xDE, 0xFF,0xCA,0xEF, 0xFF,0xDB,0xF4,

PURPLE
    0x33,0x00,0x35, 0x44,0x00,0x41, 0x55,0x00,0x4C, 0x66,0x0C,0x5C,
    0x77,0x1D,0x6D, 0x88,0x2E,0x7E, 0x99,0x3F,0x8F, 0xAA,0x50,0xA0,
    0xBB,0x61,0xB1, 0xCC,0x72,0xC2, 0xDD,0x83,0xD3, 0xEE,0x94,0xE4,
    0xFF,0xA5,0xE4, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x1D,0x00,0x5C, 0x2E,0x00,0x68, 0x40,0x00,0x74, 0x51,0x10,0x84,
    0x62,0x21,0x95, 0x73,0x32,0xA6, 0x84,0x43,0xB7, 0x95,0x54,0xC8,
    0xA6,0x65,0xD9, 0xB7,0x76,0xEA, 0xC8,0x87,0xEB, 0xD9,0x98,0xEB,
    0xE9,0xA9,0xEC, 0xFB,0xBA,0xEB, 0xFF,0xCB,0xEF, 0xFF,0xDC,0xF4,

BLUE1
    0x02,0x00,0x71, 0x13,0x00,0x7D, 0x24,0x0B,0x8C, 0x35,0x1C,0x9D,
    0x46,0x2D,0xAE, 0x57,0x3E,0xBF, 0x68,0x4F,0xD0, 0x79,0x60,0xE1,
    0x8A,0x71,0xF2, 0x9B,0x82,0xF7, 0xAC,0x93,0xF7, 0xBD,0xA4,0xF7,
    0xCE,0xB5,0xF7, 0xDF,0xC6,0xF7, 0xF0,0xD7,0xF7, 0xFF,0xE8,0xF8,

BLUE2
    0x00,0x00,0x68, 0x00,0x0A,0x7C, 0x08,0x1B,0x90, 0x19,0x2C,0xA1,
    0x2A,0x3D,0xB2, 0x3B,0x4E,0xC3, 0x4C,0x5F,0xD4, 0x5D,0x70,0xE5,
    0x6E,0x81,0xF6, 0x7F,0x92,0xFF, 0x90,0xA3,0xFF, 0xA1,0xB4,0xFF,
    0xB2,0xC5,0xFF, 0xC3,0xD6,0xFF, 0xD4,0xE7,0xFF, 0xE5,0xF8,0xFF,

LIGHT-BLUE
    0x00,0x0A,0x4D, 0x00,0x1B,0x63, 0x00,0x2C,0x79, 0x02,0x3D,0x8F,
    0x13,0x4E,0xA0, 0x24,0x5F,0xB1, 0x35,0x70,0xC2, 0x46,0x81,0xD3,
    0x57,0x92,0xE4, 0x68,0xA3,0xF5, 0x79,0xB4,0xFF, 0x8A,0xC5,0xFF,
    0x9B,0xD6,0xFF, 0xAC,0xE7,0xFF, 0xBD,0xF8,0xFF, 0xCE,0xFF,0xFF,

TURQUOISE
    0x00,0x1A,0x26, 0x00,0x2B,0x3C, 0x00,0x3C,0x52, 0x00,0x4D,0x68,
    0x06,0x5E,0x7C, 0x17,0x6F,0x8D, 0x28,0x80,0x9E, 0x39,0x91,0xAF,
    0x4A,0xA2,0xC0, 0x5B,0xB3,0xD1, 0x6C,0xC4,0xE2, 0x7D,0xD5,0xF3,
    0x8E,0xE6,0xFF, 0x9F,0xF7,0xFF, 0xB0,0xFF,0xFF, 0xC1,0xFF,0xFF,

GREEN-BLUE
    0x00,0x24,0x0B, 0x00,0x35,0x10, 0x00,0x46,0x22, 0x00,0x57,0x38,
    0x05,0x68,0x4D, 0x16,0x79,0x5E, 0x27,0x8A,0x6F, 0x38,0x9B,0x80,
    0x49,0xAC,0x91, 0x5A,0xBD,0xA2, 0x6B,0xCE,0xB3, 0x7C,0xDF,0xC4,
    0x8D,0xF0,0xD5, 0x9E,0xFF,0xE5, 0xAF,0xFF,0xF1, 0xC0,0xFF,0xFD,

GREEN
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1B,
    0x10,0x6B,0x1B, 0x21,0x7C,0x2C, 0x32,0x8D,0x3D, 0x43,0x9E,0x4E,
    0x54,0xAF,0x5F, 0x65,0xC0,0x70, 0x76,0xD1,0x81, 0x87,0xE2,0x92,
    0x98,0xF3,0xA3, 0xA9,0xFF,0xB3, 0xBA,0xFF,0xBF, 0xCB,0xFF,0xCB,

YELLOW-GREEN
    0x00,0x23,0x0A, 0x00,0x34,0x10, 0x04,0x45,0x13, 0x15,0x56,0x13,
    0x26,0x67,0x13, 0x37,0x78,0x13, 0x48,0x89,0x14, 0x59,0x9A,0x25,
    0x6A,0xAB,0x36, 0x7B,0xBC,0x47, 0x8C,0xCD,0x58, 0x9D,0xDE,0x69,
    0xAE,0xEF,0x7A, 0xBF,0xFF,0x8B, 0xD0,0xFF,0x97, 0xE1,0xFF,0xA3,

ORANGE-GREEN
    0x00,0x17,0x07, 0x0E,0x28,0x08, 0x1F,0x39,0x08, 0x30,0x4A,0x08,
    0x41,0x5B,0x08, 0x52,0x6C,0x08, 0x63,0x7D,0x08, 0x74,0x8E,0x0D,
    0x85,0x9F,0x1E, 0x96,0xB0,0x2F, 0xA7,0xC1,0x40, 0xB8,0xD2,0x51,
    0xC9,0xE3,0x62, 0xDA,0xF4,0x73, 0xEB,0xFF,0x82, 0xFC,0xFF,0x8E,

LIGHT-ORANGE
    0x19,0x07,0x00, 0x2A,0x18,0x00, 0x3B,0x29,0x00, 0x4C,0x3A,0x00,
    0x5D,0x4B,0x00, 0x6E,0x5C,0x00, 0x7F,0x6D,0x00, 0x90,0x7E,0x09,
    0xA1,0x8F,0x1A, 0xB2,0xA0,0x2B, 0xC3,0xB1,0x3C, 0xD4,0xC2,0x4D,
    0xE5,0xD3,0x5E, 0xF6,0xE4,0x6F, 0xFF,0xF5,0x82, 0xFF,0xFF,0x96,
*******************************************************************

*******************************************************************
    PALETTE - PHASE 26.7 SHIFT

GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xFF,0x97,

ORANGE
    0x32,0x00,0x00, 0x43,0x06,0x00, 0x54,0x17,0x00, 0x65,0x28,0x00,
    0x79,0x39,0x00, 0x87,0x4A,0x00, 0x98,0x5B,0x0C, 0xA9,0x6C,0x1D,
    0xBA,0x7D,0x2E, 0xCB,0x8E,0x3F, 0xDC,0x9F,0x50, 0xED,0xB0,0x61,
    0xFE,0xC1,0x72, 0xFF,0xD2,0x87, 0xFF,0xE3,0x9E, 0xFF,0xF4,0xB4,

RED-ORANGE
    0x3E,0x00,0x00, 0x4F,0x00,0x00, 0x60,0x07,0x00, 0x71,0x18,0x00,
    0x82,0x29,0x10, 0x93,0x3A,0x21, 0xA4,0x4B,0x32, 0xB5,0x5C,0x43,
    0xC6,0x6D,0x54, 0xD7,0x7E,0x65, 0xE8,0x8F,0x76, 0xF9,0xA0,0x87,
    0xFF,0xB1,0x9C, 0xFF,0xC2,0xB2, 0xFF,0xD3,0xC8, 0xFF,0xE4,0xDE,

PINK
    0x3E,0x00,0x09, 0x4F,0x00,0x15, 0x60,0x00,0x21, 0x71,0x0E,0x31,
    0x82,0x1F,0x42, 0x93,0x30,0x53, 0xA4,0x41,0x64, 0xB5,0x52,0x75,
    0xC6,0x63,0x86, 0xD7,0x74,0x97, 0xE8,0x85,0xA8, 0xF9,0x96,0xB9,
    0xFF,0xA7,0xCE, 0xFF,0xB8,0xE4, 0xFF,0xC9,0xEF, 0xFF,0xDA,0xF4,

PURPLE
    0x30,0x00,0x3D, 0x41,0x00,0x48, 0x52,0x00,0x54, 0x63,0x0C,0x64,
    0x74,0x1D,0x75, 0x85,0x2E,0x86, 0x96,0x3F,0x97, 0xA7,0x50,0xA8,
    0xB8,0x61,0xB9, 0xC9,0x72,0xCA, 0xDA,0x83,0xDB, 0xEB,0x94,0xE5,
    0xFC,0xA5,0xE5, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x18,0x00,0x62, 0x29,0x00,0x6E, 0x3A,0x01,0x7A, 0x4B,0x12,0x8B,
    0x5C,0x23,0x9C, 0x6D,0x34,0xAD, 0x7E,0x45,0xBE, 0x8F,0x56,0xCF,
    0xA0,0x67,0xE0, 0xB1,0x78,0xEE, 0xC2,0x89,0xEE, 0xD3,0x9A,0xEE,
    0xE4,0xAB,0xEE, 0xF5,0xBC,0xEE, 0xFF,0xCD,0xE0, 0xFF,0xDE,0xF5,

BLUE1
    0x00,0x00,0x72, 0x0C,0x00,0x7F, 0x1D,0x0E,0x8F, 0x2E,0x1F,0xA0,
    0x3F,0x30,0xB1, 0x50,0x41,0xC2, 0x61,0x52,0xD3, 0x72,0x63,0xE4,
    0x83,0x74,0xF5, 0x94,0x85,0xFA, 0xA5,0x96,0xFA, 0xB6,0xA7,0xFA,
    0xC7,0xB8,0xFA, 0xD8,0xC9,0xFA, 0xE9,0xDA,0xFA, 0xFA,0xE8,0xFA,

BLUE2
    0x00,0x00,0x62, 0x00,0x0F,0x77, 0x01,0x20,0x8D, 0x12,0x31,0x9E,
    0x23,0x42,0xAF, 0x34,0x53,0xC0, 0x45,0x64,0xD1, 0x56,0x75,0xE2,
    0x67,0x86,0xF3, 0x78,0x97,0xFF, 0x89,0xA8,0xFF, 0x9A,0xB9,0xFF,
    0xAB,0xCA,0xFF, 0xBC,0xDB,0xFF, 0xCD,0xEC,0xFF, 0xDE,0xFD,0xFF,

LIGHT-BLUE
    0x00,0x10,0x42, 0x00,0x21,0x58, 0x00,0x32,0x6E, 0x00,0x43,0x84,
    0x0E,0x54,0x96, 0x1F,0x65,0xA7, 0x30,0x76,0xB8, 0x41,0x87,0xC9,
    0x52,0x98,0xDA, 0x63,0xA9,0xEB, 0x74,0xBA,0xFC, 0x85,0xCB,0xFF,
    0x96,0xDC,0xFF, 0xA7,0xED,0xFF, 0xB8,0xFE,0xFF, 0xC9,0xFF,0xFF,

TURQUOISE
    0x00,0x1E,0x14, 0x00,0x2F,0x2A, 0x00,0x40,0x40, 0x00,0x51,0x56,
    0x04,0x62,0x6B, 0x15,0x73,0x7C, 0x26,0x84,0x8D, 0x37,0x95,0x9E,
    0x48,0xA6,0xAF, 0x59,0xB7,0xC0, 0x6A,0xC8,0xD1, 0x7B,0xD9,0xE2,
    0x8C,0xEA,0xF3, 0x9D,0xFB,0xFF, 0xAE,0xFF,0xFF, 0xBF,0xFF,0xFF,

GREEN-BLUE
    0x00,0x26,0x0B, 0x00,0x37,0x10, 0x00,0x48,0x16, 0x00,0x59,0x25,
    0x08,0x6A,0x38, 0x19,0x7B,0x49, 0x2A,0x8C,0x5A, 0x3B,0x9D,0x6B,
    0x4C,0xAE,0x7C, 0x5D,0xBF,0x8D, 0x6E,0xD0,0x9E, 0x7F,0xE1,0xAF,
    0x90,0xF2,0xC0, 0xA1,0xFF,0xD0, 0xB2,0xFF,0xDC, 0xC3,0xFF,0xE8,

GREEN
    0x00,0x26,0x0B, 0x00,0x37,0x10, 0x00,0x48,0x16, 0x08,0x59,0x18,
    0x19,0x6A,0x18, 0x2A,0x7B,0x18, 0x3B,0x8C,0x29, 0x4C,0x9D,0x3A,
    0x5D,0xAE,0x4B, 0x6E,0xBF,0x5C, 0x7F,0xD0,0x6D, 0x90,0xE1,0x7E,
    0xA1,0xF2,0x8F, 0xB2,0xFF,0x9F, 0xC3,0xFF,0xAB, 0xD4,0xFF,0xB7,

YELLOW-GREEN
    0x00,0x1E,0x09, 0x00,0x2F,0x0E, 0x11,0x40,0x0E, 0x22,0x51,0x0E,
    0x33,0x62,0x0E, 0x44,0x73,0x0E, 0x55,0x84,0x0E, 0x66,0x95,0x17,
    0x77,0xA6,0x28, 0x88,0xB7,0x39, 0x99,0xC8,0x4A, 0xAA,0xD9,0x5B,
    0xBB,0xEA,0x6C, 0xCC,0xFB,0x7D, 0xDD,0xFF,0x8A, 0xEE,0xFF,0x96,

ORANGE-GREEN
    0x0D,0x0F,0x01, 0x1E,0x20,0x01, 0x2F,0x31,0x01, 0x40,0x42,0x01,
    0x51,0x53,0x01, 0x62,0x64,0x01, 0x73,0x75,0x01, 0x84,0x86,0x08,
    0x95,0x97,0x19, 0xA6,0xA8,0x2A, 0xB7,0xB9,0x3B, 0xC8,0xCA,0x4C,
    0xD9,0xDB,0x5D, 0xEA,0xEC,0x6E, 0xFB,0xFD,0x7F, 0xFF,0xFF,0x8F,

LIGHT-ORANGE
    0x28,0x00,0x00, 0x39,0x0E,0x00, 0x4A,0x1F,0x00, 0x5B,0x30,0x00,
    0x6C,0x41,0x00, 0x7D,0x52,0x00, 0x8E,0x63,0x00, 0x9F,0x74,0x10,
    0xB0,0x85,0x21, 0xC1,0x96,0x32, 0xD2,0xA7,0x43, 0xE3,0xB8,0x54,
    0xF4,0xC9,0x65, 0xFF,0xDA,0x78, 0xFF,0xEB,0x8E, 0xFF,0xFC,0xA4,
*******************************************************************

*******************************************************************
    PALETTE - PHASE 27.2 SHIFT


GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,

ORANGE
    0x32,0x00,0x00, 0x43,0x05,0x00, 0x54,0x16,0x00, 0x65,0x27,0x00,
    0x76,0x38,0x00, 0x87,0X49,0x00, 0x98,0x5A,0x0C, 0xA9,0x6B,0x1D,
    0xBA,0x7C,0x2E, 0xCB,0x8D,0x3F, 0xDC,0x9E,0x50, 0xED,0xAF,0x61,
    0xFE,0xC0,0x72, 0xFF,0xD1,0x88, 0xFF,0xE2,0x9E, 0xFF,0xF3,0xB4,

RED-ORANGE
    0x3F,0x00,0x00, 0x50,0x00,0x00, 0x61,0x07,0x00, 0x72,0x18,0x01,
    0x83,0x29,0x12, 0x94,0x3A,0x23, 0xA5,0x4B,0x34, 0xB6,0x5C,0x45,
    0xC7,0x6D,0x56, 0xD8,0x7E,0x67, 0xE9,0x8F,0x78, 0xFA,0xA0,0x89,
    0xFF,0xB1,0x9E, 0xFF,0xC2,0xB4, 0xFF,0xD3,0xCA, 0xFF,0xE4,0xE0,

PINK
    0x3E,0x00,0x0C, 0x4F,0x00,0x18, 0x60,0x00,0x24, 0x71,0x0E,0x34,
    0x82,0x1F,0x45, 0x93,0x30,0x56, 0xA4,0x41,0x67, 0xB5,0x52,0x78,
    0xC6,0x63,0x89, 0xD7,0x74,0x9A, 0xE8,0x85,0xAB, 0xF9,0x96,0xB6,
    0xFF,0xA7,0xD1, 0xFF,0xB8,0xE7, 0xFF,0xC9,0xEF, 0xFF,0xDA,0xF4,

PURPLE
    0x2F,0x00,0x3F, 0x40,0x00,0x4B, 0x51,0x00,0x57, 0x62,0x0C,0x66,
    0x73,0x1D,0x77, 0x84,0x2E,0x88, 0x95,0x3F,0x99, 0xA6,0x50,0xAA,
    0xB7,0x61,0xBB, 0xC8,0x72,0xCC, 0xD9,0x83,0xDD, 0xEA,0x94,0xE5,
    0xFB,0xA5,0xE5, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x16,0x00,0x64, 0x27,0x00,0x70, 0x38,0x02,0x7D, 0x49,0x13,0x8E,
    0x5A,0x24,0x9F, 0x6B,0x35,0xB0, 0x7C,0x46,0xC1, 0x8D,0x57,0xD2,
    0x9E,0x68,0xE3, 0xAF,0x79,0xEF, 0xC0,0x8A,0xEF, 0xD1,0x9D,0xEF,
    0xE2,0xAC,0xEF, 0xF3,0xBD,0xEF, 0xFF,0xCE,0xF0, 0xFF,0xDF,0xF5,

BLUE1
    0x00,0x00,0x71, 0x09,0x00,0x7F, 0x1A,0x10,0x90, 0x2B,0x21,0xA1,
    0x3C,0x32,0xB2, 0x4D,0x43,0xC3, 0x5E,0x54,0xD4, 0x6F,0x65,0xE5,
    0x80,0x76,0xF6, 0x91,0x87,0xFC, 0xA2,0x98,0xFC, 0xB3,0xA9,0xFC,
    0xC4,0xBA,0xFC, 0xD5,0xCB,0xFC, 0xE6,0xDC,0xFC, 0xF7,0xED,0xFC,

BLUE2
    0x00,0x00,0x5E, 0x00,0x11,0x74, 0x00,0x22,0x8A, 0x0F,0x33,0x9C,
    0x20,0x44,0xAD, 0x31,0x55,0xBE, 0x42,0x66,0xCF, 0x53,0x77,0xE0,
    0x64,0x88,0xF1, 0x75,0x99,0xFF, 0x86,0xAA,0xFF, 0x97,0xBB,0xFF,
    0xA8,0xCC,0xFF, 0xB9,0xDD,0xFF, 0xCA,0xEE,0xFF, 0xDB,0xFF,0xFF,

LIGHT-BLUE
    0x00,0x12,0x3B, 0x00,0x23,0x51, 0x00,0x34,0x68, 0x00,0x45,0x7E,
    0x0C,0x56,0x90, 0x1D,0x67,0xA1, 0x2E,0x78,0xB2, 0x3F,0x89,0xC3,
    0x50,0x9A,0xD4, 0x61,0xAB,0xE5, 0x72,0xBC,0xF6, 0x83,0xCD,0xFF,
    0x94,0xDE,0xFF, 0xA5,0xEF,0xFF, 0xB6,0xFF,0xFF, 0xC7,0xFF,0xFF,

TURQUOISE
    0x00,0x20,0x0C, 0x00,0x31,0x22, 0x00,0x42,0x38, 0x00,0x53,0x4E,
    0x04,0x64,0x63, 0x15,0x75,0x74, 0x26,0x86,0x85, 0x37,0x97,0x96,
    0x48,0xA8,0xA7, 0x59,0xB9,0xB8, 0x6A,0xCA,0xC9, 0x7B,0xDB,0xDA,
    0x8C,0xEC,0xEB, 0x9D,0xFD,0xFC, 0xAE,0xFF,0xFF, 0xBF,0xFF,0xFF,

GREEN-BLUE
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1C,
    0x0B,0x6B,0x2F, 0x1C,0x7C,0x40, 0x2D,0x8D,0x51, 0x3E,0x9E,0x62,
    0x4F,0xAF,0x73, 0x60,0xC0,0x84, 0x71,0xD1,0x95, 0x82,0xE2,0xA6,
    0x93,0xF3,0xB7, 0xA4,0xFF,0xC6, 0xB5,0xFF,0xD2, 0xC6,0xFF,0xDE,

GREEN
    0x00,0x25,0x0B, 0x00,0x36,0x10, 0x00,0x47,0x15, 0x0D,0x58,0x16,
    0x1E,0x69,0x16, 0x2F,0x7A,0x16, 0x40,0x8B,0x21, 0x51,0x9C,0x32,
    0x62,0xAD,0x43, 0x73,0xBE,0x54, 0x84,0xCF,0x65, 0x95,0xE0,0x76,
    0xA6,0xF1,0x87, 0xB7,0xFF,0x98, 0xC8,0xFF,0xA3, 0xD9,0xFF,0xAF,

YELLOW-GREEN
    0x00,0x1B,0x08, 0x06,0x2C,0x0B, 0x17,0x3D,0x0B, 0x28,0x4E,0x0B,
    0x39,0x5F,0x0B, 0x4A,0x70,0x0B, 0x5B,0x81,0x0B, 0x6C,0x92,0x12,
    0x7D,0xA3,0x23, 0x8E,0xB4,0x34, 0x9F,0xC5,0x45, 0xB0,0xD6,0x56,
    0xC1,0xE7,0x67, 0xD2,0xF8,0x78, 0xE3,0xFF,0x86, 0xF4,0xFF,0x92,

ORANGE-GREEN
    0x13,0x0B,0x00, 0x24,0x1C,0x00, 0x35,0x2D,0x00, 0x46,0x3E,0x00,
    0x57,0x4F,0x00, 0x68,0x60,0x00, 0x79,0x71,0x00, 0x8A,0x82,0x08,
    0x9B,0x93,0x19, 0xAC,0xA4,0x2A, 0xBD,0xB5,0x3B, 0xCE,0xC6,0x4C,
    0xDF,0xD7,0x5D, 0xF0,0xE8,0x6E, 0xFF,0xF9,0x7F, 0xFF,0xFF,0x92,

LIGHT-ORANGE
    0x2D,0x00,0x00, 0x3E,0x0A,0x00, 0x4F,0x1B,0x00, 0x60,0x2C,0x00,
    0x71,0x3D,0x00, 0x82,0x4E,0x00, 0x93,0x5F,0x05, 0xA4,0x70,0x16,
    0xB5,0x81,0x27, 0xC4,0x90,0x37, 0xD7,0xA3,0x49, 0xE8,0xB4,0x5A,
    0xF9,0xC5,0x6B, 0xFF,0xD6,0x80, 0xFF,0xE7,0x96, 0xFF,0xF8,0xAC,
*******************************************************************

*******************************************************************
    PALETTE - 27.7 PHASE SHIFT

GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xFF,0x97,

ORANGE
    0x32,0x00,0x00, 0x43,0x05,0x00, 0x54,0x16,0x00, 0x65,0x27,0x00,
    0x76,0x38,0x00, 0x87,0x49,0x00, 0x98,0x5A,0x0C, 0xA9,0x6B,0x1D,
    0xBA,0x7C,0x2E, 0xCB,0x8D,0x3F, 0xDC,0x9E,0x50, 0xED,0xAF,0x61,
    0xFE,0xC0,0x72, 0xFF,0xD1,0x88, 0xFF,0xE2,0x9E, 0xFF,0xF3,0xB4,

RED-ORANGE
    0x3F,0x00,0x00, 0x50,0x00,0x00, 0x61,0x06,0x00, 0x72,0x17,0x03,
    0x83,0x28,0x14, 0x94,0x39,0x25, 0xA5,0x4A,0x36, 0xB6,0x5B,0x47,
    0xC7,0x6C,0x58, 0xD8,0x7D,0x69, 0xE9,0x8E,0x7A, 0xFA,0x9F,0x8B,
    0xFF,0xB0,0x9F, 0xFF,0xC1,0xB5, 0xFF,0xD2,0xCB, 0xFF,0xE3,0xE1,

PINK
    0x3D,0x00,0x10, 0x4E,0x00,0x1C, 0x5F,0x00,0x27, 0x70,0x0D,0x37,
    0x81,0x1E,0x48, 0x92,0x2F,0x59, 0xA3,0x40,0x6A, 0xB4,0x51,0x7B,
    0xC5,0x62,0x8C, 0xD6,0x73,0x9D, 0xE7,0x84,0xAE, 0xF8,0x95,0xBF,
    0xFF,0xA6,0xD3, 0xFF,0xB7,0xE9, 0xFF,0xC8,0xEE, 0xFF,0xD9,0xF4,

PURPLE
    0x2D,0x00,0x42, 0x3E,0x00,0x4E, 0x4F,0x00,0x5A, 0x60,0x0C,0x6A,
    0x71,0x1D,0x7B, 0x82,0x2E,0x8C, 0x93,0x3F,0x9D, 0xA4,0x50,0xAE,
    0xB5,0x61,0xBF, 0xC6,0x72,0xD0, 0xD7,0x83,0xE1, 0xE8,0x94,0xE6,
    0xF9,0xA5,0xE6, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x13,0x00,0x67, 0x24,0x00,0x73, 0x35,0x03,0x80, 0x46,0x14,0x91,
    0x57,0x25,0xA2, 0x68,0x36,0xB3, 0x79,0x47,0xC4, 0x8A,0x58,0xD5,
    0x9B,0x69,0xE6, 0xAC,0x7A,0xF0, 0xBD,0x8B,0xF0, 0xCE,0x9C,0xF0,
    0xDF,0xAD,0xF0, 0xF0,0xBE,0xF0, 0xFF,0xCF,0xF1, 0xFF,0xE0,0xF6,

BLUE1
    0x00,0x00,0x70, 0x05,0x01,0x80, 0x16,0x12,0x91, 0x27,0x23,0xA2,
    0x38,0x34,0xB3, 0x49,0x45,0xC4, 0x5A,0x56,0xD5, 0x6B,0x67,0xE6,
    0x7C,0x78,0xF7, 0x8D,0x89,0xFE, 0x9E,0x9A,0xFE, 0xAF,0xAB,0xFE,
    0xC0,0xBC,0xFE, 0xD1,0xCD,0xFE, 0xE2,0xDE,0xFE, 0xF3,0xEF,0xFE,

BLUE2
    0x00,0x03,0x5B, 0x00,0x14,0x71, 0x00,0x25,0x87, 0x0C,0x36,0x9A,
    0x1D,0x47,0xAB, 0x2E,0x58,0xBC, 0x3F,0x69,0xCD, 0x50,0x7A,0xDE,
    0x61,0x8B,0xEF, 0x72,0x9C,0xFF, 0x83,0xAD,0xFF, 0x94,0xBE,0xFF,
    0xA5,0xCF,0xFF, 0xB6,0xE0,0xFF, 0xC7,0xF1,0xFF, 0xD8,0xFF,0xFF,

LIGHT-BLUE
    0x00,0x15,0x35, 0x00,0x26,0x4B, 0x00,0x37,0x61, 0x00,0x48,0x78,
    0x0A,0x59,0x8B, 0x1B,0x6A,0x9C, 0x2C,0x7B,0xAD, 0x3D,0x8C,0xBE,
    0x4E,0x9D,0xCF, 0x5F,0xAE,0xE0, 0x70,0xBF,0xF1, 0x81,0xD0,0xFF,
    0x92,0xE1,0xFF, 0xA3,0xF2,0xFF, 0xB4,0xFF,0xFF, 0xC5,0xFF,0xFF,

TURQUOISE
    0x00,0x22,0x0A, 0x00,0x33,0x19, 0x00,0x44,0x2F, 0x00,0x55,0x45,
    0x04,0x66,0x5A, 0x15,0x77,0x6B, 0x26,0x88,0x7C, 0x37,0x99,0x8D,
    0x48,0xAA,0x9E, 0x59,0xBB,0xAF, 0x6A,0xCC,0xC0, 0x7B,0xDD,0xD1,
    0x8C,0xEE,0xE2, 0x9D,0xFF,0xF3, 0xAE,0xFF,0xFF, 0xBF,0xFF,0xFF,

GREEN-BLUE
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1B,
    0x0D,0x6B,0x25, 0x1E,0x7C,0x36, 0x2F,0x8D,0x47, 0x40,0x9E,0x58,
    0x51,0xAF,0x69, 0x62,0xC0,0x7A, 0x73,0xD1,0x8B, 0x84,0xE2,0x9C,
    0x95,0xF3,0xAD, 0xA6,0xFF,0xBD, 0xB7,0xFF,0xC9, 0xC8,0xFF,0xD4,

GREEN
    0x00,0x24,0x0B, 0x00,0x35,0x10, 0x01,0x46,0x15, 0x12,0x57,0x15,
    0x23,0x68,0x15, 0x34,0x79,0x15, 0x45,0x8A,0x19, 0x56,0x9B,0x2A,
    0x67,0xAC,0x3B, 0x78,0xBD,0x4C, 0x89,0xCE,0x5D, 0x9A,0xDF,0x6E,
    0xAB,0xF0,0x7F, 0xBC,0xFF,0x8F, 0xCD,0xFF,0x9B, 0xDE,0xFF,0xA7,

YELLOW-GREEN
    0x00,0x18,0x07, 0x00,0x29,0x0C, 0x1E,0x3A,0x08, 0x2F,0x4B,0x08,
    0x40,0x5C,0x08, 0x51,0x6D,0x08, 0x62,0x7E,0x08, 0x73,0x8F,0x0D,
    0x84,0xA0,0x1E, 0x95,0xB1,0x2F, 0xA6,0xC2,0x40, 0xB7,0xD3,0x51,
    0xC8,0xE4,0x62, 0xD9,0xF5,0x73, 0xEA,0xFF,0x82, 0xFB,0xFF,0x8E,

ORANGE-GREEN
    0x1B,0x07,0x00, 0x2C,0x18,0x00, 0x3D,0x29,0x00, 0x4E,0x3A,0x00,
    0x5F,0x4B,0x00, 0x70,0x5C,0x00, 0x81,0x6D,0x00, 0x92,0x7E,0x09,
    0xA3,0x8F,0x1A, 0xB4,0xA0,0x2B, 0xC5,0xB1,0x3C, 0xD6,0xC2,0x4D,
    0xE7,0xD3,0x5E, 0xF8,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xFF,0x97,

LIGHT-ORANGE
    0x33,0x00,0x00, 0x44,0x05,0x00, 0x55,0x16,0x00, 0x66,0x27,0x00,
    0x77,0x38,0x00, 0x88,0x49,0x00, 0x99,0x5A,0x0D, 0xAA,0x6B,0x1E,
    0xBB,0x7C,0x2F, 0xCC,0x8D,0x40, 0xDD,0x9E,0x51, 0xEE,0xAF,0x62,
    0xFF,0xC0,0x73, 0xFF,0xD1,0x89, 0xFF,0xE2,0x9F, 0xFF,0xF3,0xB5
*******************************************************************/

/**************************************************************
 *
 *    Memory setup
 *
 **************************************************************/

void a400_state::setup_ram(int bank, uint32_t size)
{
	offs_t ram_top;

	switch (bank)
	{
	case 0: // 0x0000-0x7fff
		ram_top = std::min(size, uint32_t(0x8000)) - 1;
		m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0x0000, ram_top, "0000");
		m_0000 = membank("0000");
		m_0000->set_base(m_ram->pointer());
		break;
	case 1: // 0x8000-0x9fff
		ram_top = std::min(size, uint32_t(0xa000)) - 1;
		if (ram_top > 0x8000)
		{
			m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0x8000, ram_top, "8000");
			m_8000 = membank("8000");
			m_8000->set_base(m_ram->pointer() + 0x8000);
		}
		else
		{
			m_8000 = nullptr;
		}
		break;
	case 2: // 0xa000-0xbfff
		ram_top = std::min(size, uint32_t(0xc000)) - 1;
		if (ram_top > 0xa000)
		{
			m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0xa000, ram_top, "a000");
			m_a000 = membank("a000");
			m_a000->set_base(m_ram->pointer() + 0xa000);
		}
		else
		{
			m_a000 = nullptr;
		}
		break;
	}
}


// these handle cart enable/disable without calling setup_ram thousands of times
READ8_MEMBER(a400_state::special_read_8000)
{
	if (!m_cart_disabled)
		return m_cart->read_80xx(space, offset);
	else
	{
		offset += 0x8000;
		if (m_ram->size() < offset)
			return 0;
		else
			return m_ram->pointer()[offset];
	}
}

WRITE8_MEMBER(a400_state::special_write_8000)
{
	if (m_cart_disabled)
	{
		offset += 0x8000;
		if (m_ram->size() >= offset)
			m_ram->pointer()[offset] = data;
	}
}

READ8_MEMBER(a400_state::special_read_a000)
{
	if (!m_cart_disabled)
		return m_cart->read_80xx(space, offset);
	else
	{
		offset += 0xa000;
		if (m_ram->size() < offset)
			return 0;
		else
			return m_ram->pointer()[offset];
	}
}

WRITE8_MEMBER(a400_state::special_write_a000)
{
	if (m_cart_disabled)
	{
		offset += 0xa000;
		if (m_ram->size() >= offset)
			m_ram->pointer()[offset] = data;
	}
}


READ8_MEMBER(a400_state::read_d5xx)
{
	disable_cart(space, offset, 0);
	return 0xff;
}

WRITE8_MEMBER(a400_state::disable_cart)
{
	if (m_cart->exists())
	{
		switch (m_cart->get_cart_type())
		{
		case A800_PHOENIX:
		case A800_BLIZZARD:
			if (!m_cart_disabled)
				m_cart_disabled = 1;
			break;
		case A800_OSS034M:
		case A800_OSS043M:
		case A800_EXPRESS:
		case A800_DIAMOND:
		case A800_WILLIAMS:
			// use m_cart_disabled & m_last_offs to avoid continuous remapping of
			// the memory space in some games (e.g. dropzone)
			if (offset & 0x8 && !m_cart_disabled)
				m_cart_disabled = 1;
			else if (!(offset & 0x8))
			{
				if (m_cart_disabled)
					m_cart_disabled = 0;

				if ((offset & 0x7) != m_last_offs)
				{
					// we enter here only if we are writing to a different offset than last time
					m_last_offs = offset & 0x7;
					m_cart->write_d5xx(space, offset, data);
				}
			}
			break;
		case A800_TURBO64:
		case A800_TURBO128:
			if (offset & 0x10 && !m_cart_disabled)
				m_cart_disabled = 1;
			else if (!(offset & 0x10))
			{
				if (m_cart_disabled)
					m_cart_disabled = 0;

				if ((offset & 0x0f) != m_last_offs)
				{
					// we enter here only if we are writing to a different offset than last time
					m_last_offs = offset & 0x0f;
					m_cart->write_d5xx(space, offset & 0x0f, data);
				}
			}
			break;
		case A800_SPARTADOS:
			// writes with offset & 8 are also used to enable/disable the subcart, so they go through!
			m_cart->write_d5xx(space, offset, data);
			break;
		case A800_OSSM091:
		case A800_OSS8K:
			if ((offset & 0x9) == 0x08)
				m_cart_disabled = 1;
			else
			{
				m_cart_disabled = 0;
				m_cart->write_d5xx(space, offset, data);
			}
			break;
		case A800_MICROCALC:
			m_cart_helper = (m_cart_helper + 1) % 5;
			if (m_cart_helper == 4)
				m_cart_disabled = 1;
			else
			{
				m_cart_disabled = 0;
				m_cart->write_d5xx(space, offset, m_cart_helper);
			}
			break;
		default:
			break;
		}
	}
}

void a400_state::setup_cart(a800_cart_slot_device *slot)
{
	m_cart_disabled = 0;
	m_last_offs = -1;

	if (slot->exists())
	{
		switch (slot->get_cart_type())
		{
		case A800_8K:
			m_maincpu->space(AS_PROGRAM).install_read_handler(0xa000, 0xbfff, read8_delegate(FUNC(a800_cart_slot_device::read_80xx),(a800_cart_slot_device*)slot));
			m_maincpu->space(AS_PROGRAM).unmap_write(0xa000, 0xbfff);
			break;
		case A800_8K_RIGHT:
			m_maincpu->space(AS_PROGRAM).install_read_handler(0x8000, 0x9fff, read8_delegate(FUNC(a800_cart_slot_device::read_80xx),(a800_cart_slot_device*)slot));
			m_maincpu->space(AS_PROGRAM).unmap_write(0x8000, 0x9fff);
			break;
		case A800_16K:
			m_maincpu->space(AS_PROGRAM).install_read_handler(0x8000, 0xbfff, read8_delegate(FUNC(a800_cart_slot_device::read_80xx),(a800_cart_slot_device*)slot));
			m_maincpu->space(AS_PROGRAM).unmap_write(0x8000, 0xbfff);
			break;
		case A800_BBSB:
			m_maincpu->space(AS_PROGRAM).install_read_handler(0x8000, 0xbfff, read8_delegate(FUNC(a800_cart_slot_device::read_80xx),(a800_cart_slot_device*)slot));
			m_maincpu->space(AS_PROGRAM).install_write_handler(0x8000, 0x9fff, write8_delegate(FUNC(a800_cart_slot_device::write_80xx),(a800_cart_slot_device*)slot));
			m_maincpu->space(AS_PROGRAM).unmap_write(0xa000, 0xbfff);
			break;
		case A800_OSS034M:
		case A800_OSS043M:
		case A800_OSSM091:
		case A800_OSS8K:
		case A800_TURBO64:
		case A800_TURBO128:
		case A800_PHOENIX:
			m_maincpu->space(AS_PROGRAM).install_readwrite_handler(0xa000, 0xbfff, read8_delegate(FUNC(a400_state::special_read_a000), this), write8_delegate(FUNC(a400_state::special_write_a000), this));
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd500, 0xd5ff, write8_delegate(FUNC(a400_state::disable_cart), this));
			break;
		case A800_EXPRESS:
			m_maincpu->space(AS_PROGRAM).install_readwrite_handler(0xa000, 0xbfff, read8_delegate(FUNC(a400_state::special_read_a000), this), write8_delegate(FUNC(a400_state::special_write_a000), this));
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd570, 0xd57f, write8_delegate(FUNC(a400_state::disable_cart), this));
			break;
		case A800_DIAMOND:
			m_maincpu->space(AS_PROGRAM).install_readwrite_handler(0xa000, 0xbfff, read8_delegate(FUNC(a400_state::special_read_a000), this), write8_delegate(FUNC(a400_state::special_write_a000), this));
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd5d0, 0xd5df, write8_delegate(FUNC(a400_state::disable_cart), this));
			break;
		case A800_WILLIAMS:
			m_maincpu->space(AS_PROGRAM).install_readwrite_handler(0xa000, 0xbfff, read8_delegate(FUNC(a400_state::special_read_a000), this), write8_delegate(FUNC(a400_state::special_write_a000), this));
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd500, 0xd50f, write8_delegate(FUNC(a400_state::disable_cart), this));
			break;
		case A800_SPARTADOS:
			m_maincpu->space(AS_PROGRAM).install_readwrite_handler(0xa000, 0xbfff, read8_delegate(FUNC(a400_state::special_read_a000), this), write8_delegate(FUNC(a400_state::special_write_a000), this));
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd5e0, 0xd5ef, write8_delegate(FUNC(a400_state::disable_cart), this));
			break;
		case A800_BLIZZARD:
			m_maincpu->space(AS_PROGRAM).install_readwrite_handler(0x8000, 0xbfff, read8_delegate(FUNC(a400_state::special_read_8000), this), write8_delegate(FUNC(a400_state::special_write_8000), this));
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd500, 0xd5ff, write8_delegate(FUNC(a400_state::disable_cart), this));
			break;
		case A800_MICROCALC:
			// this can also disable ROM when reading in 0xd500-0xd5ff
			m_maincpu->space(AS_PROGRAM).install_readwrite_handler(0xa000, 0xbfff, read8_delegate(FUNC(a400_state::special_read_a000), this), write8_delegate(FUNC(a400_state::special_write_a000), this));
			m_maincpu->space(AS_PROGRAM).install_readwrite_handler(0xd500, 0xd5ff, read8_delegate(FUNC(a400_state::read_d5xx), this), write8_delegate(FUNC(a400_state::disable_cart), this));
			break;
		case A800_TELELINK2:
			m_maincpu->space(AS_PROGRAM).install_read_handler(0x8000, 0xbfff, read8_delegate(FUNC(a800_cart_slot_device::read_80xx),(a800_cart_slot_device*)slot));
			m_maincpu->space(AS_PROGRAM).install_write_handler(0x9000, 0x90ff, write8_delegate(FUNC(a800_cart_slot_device::write_80xx),(a800_cart_slot_device*)slot));
			m_maincpu->space(AS_PROGRAM).unmap_write(0xa000, 0xbfff);
			m_maincpu->space(AS_PROGRAM).install_read_handler(0xd501, 0xd501, read8_delegate(FUNC(a800_cart_slot_device::read_d5xx),(a800_cart_slot_device*)slot));
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd502, 0xd502, write8_delegate(FUNC(a800_cart_slot_device::write_d5xx),(a800_cart_slot_device*)slot));
			break;
		case A800_XEGS:
			m_maincpu->space(AS_PROGRAM).install_read_handler(0x8000, 0xbfff, read8_delegate(FUNC(a800_cart_slot_device::read_80xx),(a800_cart_slot_device*)slot));
			m_maincpu->space(AS_PROGRAM).unmap_write(0x8000, 0xbfff);
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd500, 0xd5ff, write8_delegate(FUNC(a800_cart_slot_device::write_d5xx),(a800_cart_slot_device*)slot));
			break;
		case A5200_4K:
		case A5200_8K:
		case A5200_16K:
		case A5200_32K:
		case A5200_16K_2CHIPS:
			m_maincpu->space(AS_PROGRAM).install_read_handler(0x4000, 0xbfff, read8_delegate(FUNC(a800_cart_slot_device::read_80xx),(a800_cart_slot_device*)slot));
			m_maincpu->space(AS_PROGRAM).unmap_write(0x4000, 0xbfff);
			break;
		case A5200_BBSB:
			m_maincpu->space(AS_PROGRAM).install_read_handler(0x4000, 0xbfff, read8_delegate(FUNC(a800_cart_slot_device::read_80xx),(a800_cart_slot_device*)slot));
			m_maincpu->space(AS_PROGRAM).install_write_handler(0x4000, 0x5fff, write8_delegate(FUNC(a800_cart_slot_device::write_80xx),(a800_cart_slot_device*)slot));
			m_maincpu->space(AS_PROGRAM).unmap_write(0x6000, 0xbfff);
			break;
		}
	}
}


TIMER_DEVICE_CALLBACK_MEMBER( a400_state::a400_interrupt )
{
	m_antic->generic_interrupt(4);
}

TIMER_DEVICE_CALLBACK_MEMBER( a400_state::a800xl_interrupt )
{
	m_antic->generic_interrupt(2);
}

TIMER_DEVICE_CALLBACK_MEMBER( a400_state::a5200_interrupt )
{
	m_antic->generic_interrupt(4);
}

MACHINE_RESET_MEMBER( a400_state, a400 )
{
	m_pokey->write(15, 0);
}


MACHINE_START_MEMBER( a400_state, a400 )
{
	setup_ram(0, m_ram->size());
	setup_ram(1, m_ram->size());
	setup_ram(2, m_ram->size());
	setup_cart(m_cart);

	save_item(NAME(m_cart_disabled));
	save_item(NAME(m_cart_helper));
	save_item(NAME(m_last_offs));
}


MACHINE_START_MEMBER( a400_state, a800 )
{
	setup_ram(0, m_ram->size());
	setup_ram(1, m_ram->size());
	setup_ram(2, m_ram->size());
	setup_cart(m_cart);
	setup_cart(m_cart2);

	save_item(NAME(m_cart_disabled));
	save_item(NAME(m_cart_helper));
	save_item(NAME(m_last_offs));
}

MACHINE_START_MEMBER( a400_state, a800xl )
{
	m_mmu = 0xfd;
	m_ext_bank = 0x03;  // only used by a130xe
	setup_cart(m_cart);

	save_item(NAME(m_cart_disabled));
	save_item(NAME(m_cart_helper));
	save_item(NAME(m_last_offs));
	save_item(NAME(m_mmu));
	save_item(NAME(m_ext_bank));
}


MACHINE_START_MEMBER( a400_state, a5200 )
{
	setup_cart(m_cart);

	save_item(NAME(m_cart_disabled));
	save_item(NAME(m_cart_helper));
	save_item(NAME(m_last_offs));
}


/**************************************************************
 *
 * GTIA interface
 *
 **************************************************************/

WRITE8_MEMBER(a400_state::gtia_cb)
{
	m_dac->write(BIT(data, 3));
}

/**************************************************************
 *
 * PIA interface
 *
 **************************************************************/

WRITE8_MEMBER(a400_state::a600xl_pia_pb_w)
{
	m_mmu = data;
}

WRITE8_MEMBER(a400_state::a800xl_pia_pb_w)
{
	if (m_pia->port_b_z_mask() != 0xff)
	{
		m_mmu = data;
		m_ext_bank = (m_mmu & 0x0c) >> 2;
	}
}

/**************************************************************
 *
 * Machine drivers
 *
 **************************************************************/


void a400_state::atari_common_nodac(machine_config &config)
{
	/* basic machine hardware */
	M6502(config, m_maincpu, pokey_device::FREQ_17_EXACT);

	MCFG_MACHINE_RESET_OVERRIDE( a400_state, a400 )

	/* video hardware */
	SCREEN(config, m_screen, SCREEN_TYPE_RASTER);
	m_screen->set_vblank_time(ATTOSECONDS_IN_USEC(1));
	m_screen->set_visarea(antic_device::MIN_X, antic_device::MAX_X, antic_device::MIN_Y, antic_device::MAX_Y);
	m_screen->set_screen_update("antic", FUNC(antic_device::screen_update));
	m_screen->set_palette("palette");

	PALETTE(config, "palette", FUNC(a400_state::a400_palette), ARRAY_LENGTH(atari_colors) / 3);

	PIA6821(config, m_pia, 0);
	m_pia->readpa_handler().set_ioport("djoy_0_1");
	m_pia->readpb_handler().set_ioport("djoy_2_3");
	m_pia->ca2_handler().set("sio", FUNC(a8sio_device::motor_w));
	m_pia->cb2_handler().set("fdc", FUNC(atari_fdc_device::pia_cb2_w));
	m_pia->cb2_handler().append("sio", FUNC(a8sio_device::command_w));

	a8sio_device &sio(A8SIO(config, "sio", nullptr));
	//sio.clock_in().set("pokey", FUNC(pokey_device::bclk_w));
	sio.data_in().set("pokey", FUNC(pokey_device::sid_w));
	sio.proceed().set(m_pia, FUNC(pia6821_device::ca1_w));
	sio.interrupt().set(m_pia, FUNC(pia6821_device::cb1_w));

	/* sound hardware */
	SPEAKER(config, "speaker").front_center();
	POKEY(config, m_pokey, pokey_device::FREQ_17_EXACT);
	m_pokey->pot_r<0>().set_ioport("analog_0");
	m_pokey->pot_r<1>().set_ioport("analog_1");
	m_pokey->pot_r<2>().set_ioport("analog_2");
	m_pokey->pot_r<3>().set_ioport("analog_3");
	m_pokey->pot_r<4>().set_ioport("analog_4");
	m_pokey->pot_r<5>().set_ioport("analog_5");
	m_pokey->pot_r<6>().set_ioport("analog_6");
	m_pokey->pot_r<7>().set_ioport("analog_7");
	m_pokey->serin_r().set("fdc", FUNC(atari_fdc_device::serin_r));
	m_pokey->serout_w().set("fdc", FUNC(atari_fdc_device::serout_w));
	//m_pokey->oclk_w().set("sio", FUNC(a8sio_device::clock_out_w));
	//m_pokey->sod_w().set("sio", FUNC(a8sio_device::data_out_w));
	m_pokey->set_keyboard_callback(FUNC(a400_state::a800_keyboard));
	m_pokey->set_interrupt_callback(FUNC(a400_state::interrupt_cb));
	m_pokey->add_route(ALL_OUTPUTS, "speaker", 1.0);
}

void a400_state::atari_common(machine_config &config)
{
	atari_common_nodac(config);

	DAC_1BIT(config, "dac", 0).add_route(ALL_OUTPUTS, "speaker", 0.03);
	VOLTAGE_REGULATOR(config, "vref", 0).add_route(0, "dac", 1.0, DAC_VREF_POS_INPUT);

	/* internal ram */
	RAM(config, m_ram).set_default_size("48K");

	ATARI_GTIA(config, m_gtia, 0);
	m_gtia->read_callback().set_ioport("console");
	m_gtia->write_callback().set(FUNC(a400_state::gtia_cb));

	ATARI_ANTIC(config, m_antic, 0);
	m_antic->set_gtia_tag(m_gtia);

	/* devices */
	ATARI_FDC(config, "fdc", 0);

	A800_CART_SLOT(config, "cartleft", a800_left, nullptr);

	/* software lists */
	SOFTWARE_LIST(config, "flop_list").set_original("a800_flop");
	SOFTWARE_LIST(config, "cart_list").set_original("a800");
	SOFTWARE_LIST(config, "xegs_list").set_original("xegs");
}

// memory map A400 + NTSC screen
void a400_state::a400(machine_config &config)
{
	atari_common(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a400_state::a400_mem);
	TIMER(config, "scantimer").configure_scanline(FUNC(a400_state::a400_interrupt), "screen", 0, 1);

	MCFG_MACHINE_START_OVERRIDE( a400_state, a400 )

	m_screen->set_refresh_hz(antic_device::FRAME_RATE_60HZ);
	m_screen->set_size(antic_device::HWIDTH * 8, antic_device::TOTAL_LINES_60HZ);

	m_gtia->set_region(GTIA_NTSC);
}

// memory map A400 + PAL screen
void a400_state::a400pal(machine_config &config)
{
	atari_common(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a400_state::a400_mem);
	TIMER(config, "scantimer").configure_scanline(FUNC(a400_state::a400_interrupt), "screen", 0, 1);

	MCFG_MACHINE_START_OVERRIDE( a400_state, a400 )

	m_screen->set_refresh_hz(antic_device::FRAME_RATE_50HZ);
	m_screen->set_size(antic_device::HWIDTH * 8, antic_device::TOTAL_LINES_50HZ);

	m_gtia->set_region(GTIA_PAL);
}

// memory map A800 + NTSC screen + Right cartslot
void a400_state::a800(machine_config &config)
{
	atari_common(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a400_state::a400_mem);
	TIMER(config, "scantimer").configure_scanline(FUNC(a400_state::a400_interrupt), "screen", 0, 1);

	MCFG_MACHINE_START_OVERRIDE( a400_state, a800 )

	m_screen->set_refresh_hz(antic_device::FRAME_RATE_60HZ);
	m_screen->set_size(antic_device::HWIDTH * 8, antic_device::TOTAL_LINES_60HZ);

	m_gtia->set_region(GTIA_NTSC);

	A800_CART_SLOT(config, "cartright", a800_right, nullptr);
}


// memory map A800 + PAL screen + Right cartslot
void a400_state::a800pal(machine_config &config)
{
	atari_common(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a400_state::a400_mem);
	TIMER(config, "scantimer").configure_scanline(FUNC(a400_state::a400_interrupt), "screen", 0, 1);

	MCFG_MACHINE_START_OVERRIDE( a400_state, a800 )

	m_screen->set_refresh_hz(antic_device::FRAME_RATE_50HZ);
	m_screen->set_size(antic_device::HWIDTH * 8, antic_device::TOTAL_LINES_50HZ);

	m_gtia->set_region(GTIA_PAL);

	A800_CART_SLOT(config, "cartright", a800_right, nullptr);
}


// memory map A600XL (same as 800XL but less RAM) + NTSC screen + MMU via PIA portB
void a400_state::a600xl(machine_config &config)
{
	atari_common(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a400_state::a600xl_mem);
	TIMER(config, "scantimer").configure_scanline(FUNC(a400_state::a800xl_interrupt), "screen", 0, 1);

	m_pia->writepb_handler().set(FUNC(a400_state::a600xl_pia_pb_w));

	MCFG_MACHINE_START_OVERRIDE( a400_state, a800xl )

	m_screen->set_refresh_hz(antic_device::FRAME_RATE_60HZ);
	m_screen->set_size(antic_device::HWIDTH * 8, antic_device::TOTAL_LINES_60HZ);

	m_gtia->set_region(GTIA_NTSC);

	m_ram->set_default_size("16K");
}

// memory map A800XL + NTSC screen + MMU via PIA portB
void a400_state::a800xl(machine_config &config)
{
	atari_common(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a400_state::a800xl_mem);
	TIMER(config, "scantimer").configure_scanline(FUNC(a400_state::a800xl_interrupt), "screen", 0, 1);

	m_pia->writepb_handler().set(FUNC(a400_state::a800xl_pia_pb_w));

	MCFG_MACHINE_START_OVERRIDE( a400_state, a800xl )

	m_ram->set_default_size("64K");

	m_screen->set_refresh_hz(antic_device::FRAME_RATE_60HZ);
	m_screen->set_size(antic_device::HWIDTH * 8, antic_device::TOTAL_LINES_60HZ);

	m_gtia->set_region(GTIA_NTSC);
}


// memory map A800XL + PAL screen + MMU via PIA portB
void a400_state::a800xlpal(machine_config &config)
{
	a800xl(config);

	m_maincpu->set_clock(1773000);

	m_screen->set_refresh_hz(antic_device::FRAME_RATE_50HZ);
	m_screen->set_size(antic_device::HWIDTH * 8, antic_device::TOTAL_LINES_50HZ);

	m_gtia->set_region(GTIA_PAL);

	m_pokey->set_clock(1773000);
}


// memory map A1200XL+ MMU via PIA portB
void a400_state::a1200xl(machine_config &config)
{
	a800xl(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a400_state::a1200xl_mem);
	m_pia->writepb_handler().set(FUNC(a400_state::a800xl_pia_pb_w));
}

// memory map A130XE (extra RAM only partially emulated)
void a400_state::a130xe(machine_config &config)
{
	a800xl(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a400_state::a130xe_mem);
	m_ram->set_default_size("128K");
}

// memory map XEGS, only XEGS bankswitch supported
void a400_state::xegs(machine_config &config)
{
	a800xl(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a400_state::xegs_mem);

	config.device_remove("cartleft");
	config.device_remove("cart_list");

	XEGS_CART_SLOT(config, "cartleft", xegs_carts, nullptr);
}

// memory map A5200, different ports, less RAM
void a400_state::a5200(machine_config &config)
{
	atari_common_nodac(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a400_state::a5200_mem);
	TIMER(config, "scantimer").configure_scanline(FUNC(a400_state::a5200_interrupt), "screen", 0, 1);

	// FIXME: should there be anything connected where other system have the fdc?
	m_pokey->serin_r().set_constant(0);
	m_pokey->serout_w().set_nop();
	m_pokey->set_keyboard_callback(FUNC(a400_state::a5200_keypads));
	m_pokey->set_interrupt_callback(FUNC(a400_state::interrupt_cb));
	m_pokey->add_route(ALL_OUTPUTS, "speaker", 1.0);

	ATARI_GTIA(config, m_gtia, 0);
	m_gtia->set_region(GTIA_NTSC);

	ATARI_ANTIC(config, m_antic, 0);
	m_antic->set_gtia_tag(m_gtia);

	m_pia->readpa_handler().set_constant(0); // FIXME: is there anything connected here
	m_pia->readpb_handler().set_constant(0); // FIXME: is there anything connected here
	m_pia->cb2_handler().set_nop(); // FIXME: is there anything connected here

	MCFG_MACHINE_START_OVERRIDE( a400_state, a5200 )

	m_screen->set_refresh_hz(antic_device::FRAME_RATE_60HZ);
	m_screen->set_size(antic_device::HWIDTH * 8, antic_device::TOTAL_LINES_60HZ);

	A5200_CART_SLOT(config, "cartleft", a5200_carts, nullptr);

	/* Software lists */
	SOFTWARE_LIST(config, "cart_list").set_original("a5200");

	/* internal ram */
	RAM(config, m_ram).set_default_size("16K");
}


/**************************************************************
 *
 * ROM loading
 *
 **************************************************************/

ROM_START(a400)
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co12399b.rom", 0xd800, 0x0800, CRC(6a5d766e) SHA1(01a6044f7a81d409c938e7dfde0a1af5832229d2) )
	ROM_SYSTEM_BIOS(0, "default", "OS Rev. B")
	ROMX_LOAD( "co12499b.rom",  0xe000, 0x1000, BAD_DUMP CRC(d818f3e8) SHA1(bcdec2188f6a6a5bfc1df4e383bd828d34b5c4ac), ROM_BIOS(0) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co14599b.rom",  0xf000, 0x1000, BAD_DUMP CRC(c1690a9b) SHA1(c5248e8565574fd39ae1c3f4f356aa4cac07df95), ROM_BIOS(0) )    // CRC and label waiting for confirmation
	ROM_SYSTEM_BIOS(1, "reva", "OS Rev. A")
	ROMX_LOAD( "co12499a.rom",  0xe000, 0x1000, BAD_DUMP CRC(29f64e17) SHA1(abf7ec488c6b600f1b7f30bdc7f8a2bf6a727675), ROM_BIOS(1) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co14599a.rom",  0xf000, 0x1000, BAD_DUMP CRC(bc533f0c) SHA1(e217148495fa747fe5488132d8d22533e68c7e58), ROM_BIOS(1) )    // CRC and label waiting for confirmation
ROM_END

ROM_START(a400pal)
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co12399b.rom", 0xd800, 0x0800, CRC(6a5d766e) SHA1(01a6044f7a81d409c938e7dfde0a1af5832229d2) )
	ROM_LOAD( "co15199.rom", 0xe000, 0x1000, BAD_DUMP CRC(8e547f56) SHA1(1bd746ea798b723bfb18495a7facca113183d713) )    // Rev. A - CRC and label waiting for confirmation
	ROM_LOAD( "co15299.rom", 0xf000, 0x1000, BAD_DUMP CRC(be55b413) SHA1(d88afae49b08e75943d0258cb580e5d34756414a) )    // Rev. A - CRC and label waiting for confirmation
ROM_END

ROM_START(a800)
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co12399b.rom", 0xd800, 0x0800, CRC(6a5d766e) SHA1(01a6044f7a81d409c938e7dfde0a1af5832229d2) )
	ROM_SYSTEM_BIOS(0, "default", "OS Rev. B")
	ROMX_LOAD( "co12499b.rom",  0xe000, 0x1000, BAD_DUMP CRC(d818f3e8) SHA1(bcdec2188f6a6a5bfc1df4e383bd828d34b5c4ac), ROM_BIOS(0) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co14599b.rom",  0xf000, 0x1000, BAD_DUMP CRC(c1690a9b) SHA1(c5248e8565574fd39ae1c3f4f356aa4cac07df95), ROM_BIOS(0) )    // CRC and label waiting for confirmation
	ROM_SYSTEM_BIOS(1, "reva", "OS Rev. A")
	ROMX_LOAD( "co12499a.rom",  0xe000, 0x1000, BAD_DUMP CRC(29f64e17) SHA1(abf7ec488c6b600f1b7f30bdc7f8a2bf6a727675), ROM_BIOS(1) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co14599a.rom",  0xf000, 0x1000, BAD_DUMP CRC(bc533f0c) SHA1(e217148495fa747fe5488132d8d22533e68c7e58), ROM_BIOS(1) )    // CRC and label waiting for confirmation
ROM_END

ROM_START(a800pal)
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co12399b.rom", 0xd800, 0x0800, CRC(6a5d766e) SHA1(01a6044f7a81d409c938e7dfde0a1af5832229d2) )
	ROM_LOAD( "co15199.rom", 0xe000, 0x1000, BAD_DUMP CRC(8e547f56) SHA1(1bd746ea798b723bfb18495a7facca113183d713) )    // Rev. A - CRC and label waiting for confirmation
	ROM_LOAD( "co15299.rom", 0xf000, 0x1000, BAD_DUMP CRC(be55b413) SHA1(d88afae49b08e75943d0258cb580e5d34756414a) )    // Rev. A - CRC and label waiting for confirmation
ROM_END

ROM_START(a1200xl)
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_SYSTEM_BIOS(0, "default", "OS Rev. 11")
	ROMX_LOAD( "co60616b.rom", 0xc000, 0x2000, BAD_DUMP CRC(6e29ec8d) SHA1(3f9c06d6b4d261f3d5bf4354e3cff0c17b9347b9), ROM_BIOS(0) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co60617b.rom", 0xe000, 0x2000, BAD_DUMP CRC(d73ce29a) SHA1(64790242d902643fe0c40dd842749f1fe461831b), ROM_BIOS(0) )    // CRC and label waiting for confirmation
	ROM_SYSTEM_BIOS(1, "rev10", "OS Rev. 10")
	ROMX_LOAD( "co60616a.rom", 0xc000, 0x2000, BAD_DUMP CRC(0391386b) SHA1(7c176657c88b89b8a69bf021fa8e0939efc0dff2), ROM_BIOS(1) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co60617a.rom", 0xe000, 0x2000, BAD_DUMP CRC(b502f1e7) SHA1(6688db57d97fa570aef5c15cef3e5fb2688879c2), ROM_BIOS(1) )    // CRC and label waiting for confirmation
ROM_END

ROM_START(a600xl)
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co60302a.rom", 0xa000, 0x2000, CRC(f0202fb3) SHA1(7ad88dd99ff4a6ee66f6d162074db6f8bef7a9b6) )    // Rev. B
	ROM_LOAD( "co62024.rom",  0xc000, 0x4000, CRC(643bcc98) SHA1(881d030656b40bbe48f15a696b28f22c0b752ab0) )    // Rev. 1
ROM_END

ROM_START(a800xl)
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co60302a.rom", 0xa000, 0x2000, CRC(f0202fb3) SHA1(7ad88dd99ff4a6ee66f6d162074db6f8bef7a9b6) )   // Rev. B
	ROM_LOAD( "co61598b.rom", 0xc000, 0x4000, CRC(1f9cd270) SHA1(ae4f523ba08b6fd59f3cae515a2b2410bbd98f55) )   // Rev. 2
ROM_END

#define rom_a800xlp rom_a800xl

ROM_START(a65xe)
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co24947a.rom", 0xa000, 0x2000, CRC(7d684184) SHA1(3693c9cb9bf3b41bae1150f7a8264992468fc8c0) )   // Rev. C
	ROM_LOAD( "co61598b.rom", 0xc000, 0x4000, CRC(1f9cd270) SHA1(ae4f523ba08b6fd59f3cae515a2b2410bbd98f55) )   // Rev. 2
ROM_END

ROM_START(a65xea)
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "basic_ar.rom", 0xa000, 0x2000, CRC(c899f4d6) SHA1(043df191d1fe402e792266a108e147ffcda35130) )   // is this correct? or shall we use Rev. C?
//  ROM_LOAD( "c101700.rom",  0xc000, 0x4000, CRC(7f9a76c8) SHA1(57eb6d87850a763f11767f53d4eaede186f831a2) )   // this was from Savetz and has wrong bits!
	ROM_LOAD( "c101700.rom",  0xc000, 0x4000, CRC(45f47988) SHA1(a36b8b20f657580f172749bb0625c08706ed824c) )   // Rev. 3B ?
ROM_END

ROM_START(a130xe)
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co24947a.rom", 0xa000, 0x2000, CRC(7d684184) SHA1(3693c9cb9bf3b41bae1150f7a8264992468fc8c0) )   // Rev. C
	ROM_LOAD( "co61598b.rom", 0xc000, 0x4000, CRC(1f9cd270) SHA1(ae4f523ba08b6fd59f3cae515a2b2410bbd98f55) )   // Rev. 2
ROM_END

ROM_START(a800xe)
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co24947a.rom", 0xa000, 0x2000, CRC(7d684184) SHA1(3693c9cb9bf3b41bae1150f7a8264992468fc8c0) )   // Rev. C
	ROM_LOAD( "c300717.rom",  0xc000, 0x4000, CRC(29f133f7) SHA1(f03b9b93000ee84abb9cf8d6367241006f172182) )   // Rev. 3
ROM_END

ROM_START(xegs)
	ROM_REGION(0x10000, "maincpu", ROMREGION_ERASE00)
	ROM_LOAD( "c101687.rom", 0x8000, 0x8000, CRC(d50260d1) SHA1(0e0625ab2473f8431640df3ac8af61925760b9b9) )    // Rev. C + Rev. 4 + Missile Command
ROM_END


ROM_START(a5200)
	ROM_REGION(0x10000, "maincpu", ROMREGION_ERASEFF)
	ROM_SYSTEM_BIOS(0, "default", "a5200")
	ROMX_LOAD( "5200.rom",  0xf800, 0x0800, CRC(4248d3e3) SHA1(6ad7a1e8c9fad486fbec9498cb48bf5bc3adc530), ROM_BIOS(0) )
	ROM_SYSTEM_BIOS(1, "alt", "a5200 (alt)")
	ROMX_LOAD( "5200a.rom", 0xf800, 0x0800, CRC(c2ba2613) SHA1(1d2a3f00109d75d2d79fecb565775eb95b7d04d5), ROM_BIOS(1) )
ROM_END


/**************************************************************
 *
 * Game driver(s)
 *
 **************************************************************/

/*     YEAR  NAME    PARENT  COMPAT  MACHINE    INPUT   CLASS       INIT        COMPANY  FULLNAME */
COMP( 1979, a400,    0,      0,      a400,      a800,   a400_state, empty_init, "Atari", "Atari 400 (NTSC)",     0)
COMP( 1979, a400pal, a400,   0,      a400pal,   a800,   a400_state, empty_init, "Atari", "Atari 400 (PAL)",      0)
COMP( 1979, a800,    0,      0,      a800,      a800,   a400_state, empty_init, "Atari", "Atari 800 (NTSC)",     0)
COMP( 1979, a800pal, a800,   0,      a800pal,   a800,   a400_state, empty_init, "Atari", "Atari 800 (PAL)",      0)
COMP( 1982, a1200xl, a800,   0,      a1200xl,   a800xl, a400_state, empty_init, "Atari", "Atari 1200XL",         MACHINE_NOT_WORKING )      // 64k RAM
COMP( 1983, a600xl,  a800xl, 0,      a600xl,    a800xl, a400_state, empty_init, "Atari", "Atari 600XL",          MACHINE_IMPERFECT_GRAPHICS )      // 16k RAM
COMP( 1983, a800xl,  0,      0,      a800xl,    a800xl, a400_state, empty_init, "Atari", "Atari 800XL (NTSC)",   MACHINE_IMPERFECT_GRAPHICS )      // 64k RAM
COMP( 1983, a800xlp, a800xl, 0,      a800xlpal, a800xl, a400_state, empty_init, "Atari", "Atari 800XL (PAL)",    MACHINE_IMPERFECT_GRAPHICS )      // 64k RAM
COMP( 1986, a65xe,   a800xl, 0,      a800xl,    a800xl, a400_state, empty_init, "Atari", "Atari 65XE",           MACHINE_IMPERFECT_GRAPHICS )      // 64k RAM
COMP( 1986, a65xea,  a800xl, 0,      a800xl,    a800xl, a400_state, empty_init, "Atari", "Atari 65XE (Arabic)",  MACHINE_NOT_WORKING )
COMP( 1986, a130xe,  a800xl, 0,      a130xe,    a800xl, a400_state, empty_init, "Atari", "Atari 130XE",          MACHINE_NOT_WORKING )      // 128k RAM
COMP( 1986, a800xe,  a800xl, 0,      a800xl,    a800xl, a400_state, empty_init, "Atari", "Atari 800XE",          MACHINE_IMPERFECT_GRAPHICS )      // 64k RAM
COMP( 1987, xegs,    0,      0,      xegs,      a800xl, a400_state, empty_init, "Atari", "Atari XE Game System", MACHINE_IMPERFECT_GRAPHICS )  // 64k RAM

CONS( 1982, a5200,   0,      0,      a5200,     a5200,  a400_state, empty_init, "Atari", "Atari 5200",           0)
