// This file is part of meshoptimizer library and is distributed under the terms of MIT License.
// Copyright (C) 2016-2019, by Arseny Kapoulkine (arseny.kapoulkine@gmail.com)
var MeshoptDecoder = (function() {
	"use strict";
	var wasm = "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";

	var memory = new WebAssembly.Memory({
		initial: 1
	});
	var heap = new Uint8Array(memory.buffer);
	var brk = 32768; // stack top

	var sbrk = function(size) {
		var old = brk;
		brk += size;
		if (brk > heap.length) {
			memory.grow(Math.ceil((brk - heap.length) / 65536));
			heap = new Uint8Array(memory.buffer);
		}
		return old;
	};

	var imports = {
		env: {
			memory: memory,
			emscripten_memcpy_big: function(d, s, n) {
				heap.set(heap.subarray(s, s + n), d);
			},
		}
	};

	var instance = {};
	var promise =
		(typeof fetch === 'function'
			? fetch('data:application/octet-stream;base64,' + wasm)
			  .then(response => response.arrayBuffer())
			: Promise.resolve(Buffer.from(wasm, 'base64').buffer))
		.then(bytes => WebAssembly.instantiate(bytes, imports))
		.then(result => instance = result.instance);

	var decode = function(fun, target, count, size, source) {
		var tp = sbrk(count * size);
		var sp = sbrk(source.length);
		heap.set(source, sp);
		var res = fun(tp, count, size, sp, source.length);
		target.set(heap.subarray(tp, tp + count * size), 0);
		sbrk(tp - sbrk(0));
		if (res != 0) {
			throw new Error("Malformed buffer data: " + res);
		}
	};

	return {
		ready: promise,
		decodeVertexBuffer: function(target, count, size, source) {
			decode(instance.exports["meshopt_decodeVertexBuffer"], target, count, size, source);
		},
		decodeIndexBuffer: function(target, count, size, source) {
			decode(instance.exports["meshopt_decodeIndexBuffer"], target, count, size, source);
		}
	}
})();

if (typeof exports === 'object' && typeof module === 'object')
	module.exports = MeshoptDecoder;
else if (typeof define === 'function' && define['amd'])
	define([], function() {
		return MeshoptDecoder;
	});
else if (typeof exports === 'object')
	exports["MeshoptDecoder"] = MeshoptDecoder;
