// license:BSD-3-Clause
// copyright-holders:R. Belmont
/*********************************************************************

    apple2common.cpp

    Apple II stuff shared between apple2/apple2e/apple2gs.

*********************************************************************/

#include "emu.h"
#include "emuopts.h"

#include "apple2common.h"


// device type definition
DEFINE_DEVICE_TYPE(APPLE2_COMMON, apple2_common_device, "apple2com", "Apple II Common Components")

//-------------------------------------------------
//  apple2_common_device - constructor
//-------------------------------------------------

apple2_common_device::apple2_common_device(const machine_config &mconfig, const char *tag, device_t *owner, uint32_t clock)
	: device_t(mconfig, APPLE2_COMMON, tag, owner, clock)
{
}


//-------------------------------------------------
//  device_start - device-specific startup
//-------------------------------------------------

void apple2_common_device::device_start()
{
	// precalculate joystick time constants
	m_x_calibration = attotime::from_nsec(10800).as_double();
	m_y_calibration = attotime::from_nsec(10800).as_double();
	m_joystick_x1_time = m_joystick_x2_time = m_joystick_y1_time = m_joystick_y2_time = 0.0;
}

struct dasm_data
{
	u16 addr;
	const char *name;
};

static const struct dasm_data p8_calls[] =
{
	{ 0x40, "ALLOC_INTERRUPT" }, { 0x41, "DEALLOC_INTERRUPT" },
	{ 0x65, "QUIT" }, { 0x80, "READ_BLOCK" }, { 0x81, "WRITE_BLOCK" }, { 0x82, "GET_TIME" },
	{ 0x99, "ATINIT" }, { 0xc0, "CREATE" }, { 0xc1, "DESTROY" }, { 0xc2, "RENAME" }, { 0xc3, "SET_FILE_INFO" },
	{ 0xc4, "GET_FILE_INFO" }, { 0xc5, "ONLINE" }, { 0xc6, "SET_PREFIX" }, { 0xc7, "GET_PREFIX" }, { 0xc8, "OPEN" },
	{ 0xc9, "NEWLINE" }, { 0xca, "READ" }, { 0xcb, "WRITE" }, { 0xcc, "CLOSE" }, { 0xcd, "FLUSH" }, { 0xce, "SET_MARK" },
	{ 0xcf, "GET_MARK" }, { 0xd0, "SET_EOF" }, { 0xd1, "GET_EOF" }, { 0xd2, "SET_BUF" }, { 0xd3, "GET_BUF" },

	{ 0xffff, "" }
};

static const struct dasm_data a2_stuff[] =
{
	{ 0x0020, "TXTLEFT" }, { 0x0021, "TXTWIDTH" }, { 0x0022, "TXTTOP" }, { 0x0023, "TXTBOTTOM" },
	{ 0x0024, "CURSORHORZ" }, { 0x0025, "CURSORVERT" }, { 0x0028, "BASEADR" }, { 0x002b, "BOOTSLOT" }, { 0x002c, "GRLINEEND" },
	{ 0x0030, "GRCOLOR" }, { 0x0032, "TEXTFORMAT" }, { 0x0033, "PROMPTCHAR" }, { 0x0036, "OUTHOOK" }, { 0x0038, "INHOOK" },
	{ 0x0067, "BASICPRGBASE" }, { 0x0069, "BASICVARBASE" }, { 0x006b, "BASICARRBASE" }, { 0x6d, "BASICVAREND" },
	{ 0x006f, "BASICSTREND" }, { 0x0075, "CURLINENUM" }, { 0x77, "ENDSTOPLINE" }, { 0x79, "CURLINEADR" },
	{ 0x007b, "CURDATAADR" }, { 0x007d, "NEXTDATAADR" }, { 0x009d, "FAC" }, { 0x00a5, "ARG" }, { 0x00af, "BASICPRGEND" },

	{ 0xc000, "KBD" }, { 0xc001, "80STOREON" }, { 0xc002, "RDMAINRAM" }, {0xc003, "RDCARDRAM" }, {0xc004, "WRMAINRAM" },
	{ 0xc005, "WRCARDRAM" }, { 0xc006, "SETSLOTCXROM" }, { 0xc007, "SETINTCXROM" }, { 0xc008, "SETSTDZP" },
	{ 0xc009, "SETALTZP "}, { 0xc00a, "SETINTC3ROM" }, { 0xc00b, "SETSLOTC3ROM" }, { 0xc00c, "CLR80VID" },
	{ 0xc00d, "SET80VID" }, { 0xc00e, "CLRALTCHAR" }, { 0xc00f, "SETALTCHAR" }, { 0xc010, "KBDSTRB" },
	{ 0xc011, "RDLCBNK2" }, { 0xc012, "RDLCRAM" }, { 0xc013, "RDRAMRD" }, { 0xc014, "RDRAMWRT" },
	{ 0xc015, "RDCXROM" }, { 0xc016, "RDALTZP" }, { 0xc017, "RDC3ROM" }, { 0xc018, "RD80STORE" },
	{ 0xc019, "RDVBL" }, { 0xc01a, "RDTEXT" }, { 0xc01b, "RDMIXED" }, { 0xc01c, "RDPAGE2" },
	{ 0xc01d, "RDHIRES" }, { 0xc01e, "RDALTCHAR" }, { 0xc01f, "RD80VID" }, { 0xc020, "TAPEOUT" },
	{ 0xc030, "SPKR" }, { 0xc040, "STROBE" }, { 0xc050, "TXTCLR" }, { 0xc051, "TXTSET" },
	{ 0xc052, "MIXCLR" }, { 0xc053, "MIXSET" }, { 0xc054, "TXTPAGE1" }, { 0xc055, "TXTPAGE2" },
	{ 0xc056, "LORES" }, { 0xc057, "HIRES" }, { 0xc058, "CLRAN0" }, { 0xc059, "SETAN0" },
	{ 0xc05a, "CLRAN1" }, { 0xc05b, "SETAN1" }, { 0xc05c, "CLRAN2" }, { 0xc05d, "SETAN2" },
	{ 0xc05e, "DHIRESON" }, { 0xc05f, "DHIRESOFF" }, { 0xc060, "TAPEIN" }, { 0xc061, "RDBTN0" },
	{ 0xc062, "BUTN1" }, { 0xc063, "RD63" }, { 0xc064, "PADDL0" }, { 0xc065, "PADDL1" },
	{ 0xc066, "PADDL2" }, { 0xc067, "PADDL3" }, { 0xc070, "PTRIG" }, { 0xc073, "BANKSEL" },
	{ 0xc07e, "IOUDISON" }, { 0xc07f, "IOUDISOFF" }, { 0xc081, "ROMIN" }, { 0xc083, "LCBANK2" },
	{ 0xc085, "ROMIN" }, { 0xc087, "LCBANK2" }, { 0xcfff, "DISCC8ROM" },

	{ 0xF800, "F8ROM:PLOT" }, { 0xF80E, "F8ROM:PLOT1" } , { 0xF819, "F8ROM:HLINE" }, { 0xF828, "F8ROM:VLINE" },
	{ 0xF832, "F8ROM:CLRSCR" }, { 0xF836, "F8ROM:CLRTOP" }, { 0xF847, "F8ROM:GBASCALC" }, { 0xF85F, "F8ROM:NXTCOL" },
	{ 0xF864, "F8ROM:SETCOL" }, { 0xF871, "F8ROM:SCRN" }, { 0xF88C, "F8ROM:INSDS1.2" }, { 0xF88E, "F8ROM:INSDS2" },
	{ 0xF890, "F8ROM:GET816LEN" }, { 0xF8D0, "F8ROM:INSTDSP" }, { 0xF940, "F8ROM:PRNTYX" }, { 0xF941, "F8ROM:PRNTAX" },
	{ 0xF944, "F8ROM:PRNTX" }, { 0xF948, "F8ROM:PRBLNK" }, { 0xF94A, "F8ROM:PRBL2" },  { 0xF953, "F8ROM:PCADJ" },
	{ 0xF962, "F8ROM:TEXT2COPY" }, { 0xFA40, "F8ROM:OLDIRQ" }, { 0xFA4C, "F8ROM:BREAK" }, { 0xFA59, "F8ROM:OLDBRK" },
	{ 0xFA62, "F8ROM:RESET" }, { 0xFAA6, "F8ROM:PWRUP" }, { 0xFABA, "F8ROM:SLOOP" }, { 0xFAD7, "F8ROM:REGDSP" },
	{ 0xFADA, "F8ROM:RGDSP1" },
	{ 0xFB19, "F8ROM:RTBL" }, { 0xFB1E, "F8ROM:PREAD" }, { 0xFB21, "F8ROM:PREAD4" }, { 0xFB2F, "F8ROM:INIT" },
	{ 0xFB39, "F8ROM:SETTXT" }, { 0xFB40, "F8ROM:SETGR" }, { 0xFB4B, "F8ROM:SETWND" }, { 0xFB51, "F8ROM:SETWND2" },
	{ 0xFB5B, "F8ROM:TABV" }, { 0xFB60, "F8ROM:APPLEII" }, { 0xFB6F, "F8ROM:SETPWRC" }, { 0xFB78, "F8ROM:VIDWAIT" },
	{ 0xFB88, "F8ROM:KBDWAIT" }, { 0xFBB3, "F8ROM:VERSION" }, { 0xFBBF, "F8ROM:ZIDBYTE2" }, { 0xFBC0, "F8ROM:ZIDBYTE" },
	{ 0xFBC1, "F8ROM:BASCALC" }, { 0xFBDD, "F8ROM:BELL1" }, { 0xFBE2, "F8ROM:BELL1.2" }, { 0xFBE4, "F8ROM:BELL2" },
	{ 0xFBF0, "F8ROM:STORADV" }, { 0xFBF4, "F8ROM:ADVANCE" }, { 0xFBFD, "F8ROM:VIDOUT" }, { 0xFC10, "F8ROM:BS" },
	{ 0xFC1A, "F8ROM:UP" }, { 0xFC22, "F8ROM:VTAB" }, { 0xFC24, "F8ROM:VTABZ" }, { 0xFC42, "F8ROM:CLREOP" },
	{ 0xFC58, "F8ROM:HOME" }, { 0xFC62, "F8ROM:CR" }, { 0xFC66, "F8ROM:LF" }, { 0xFC70, "F8ROM:SCROLL" },
	{ 0xFC9C, "F8ROM:CLREOL" }, { 0xFC9E, "F8ROM:CLREOLZ" }, {0xFCA8, "F8ROM:WAIT" }, { 0xFCB4, "F8ROM:NXTA4" },
	{ 0xFCBA, "F8ROM:NXTA1" }, { 0xFCC9, "F8ROM:HEADR" }, { 0xFD0C, "F8ROM:RDKEY" }, { 0xFD10, "F8ROM:FD10" },
	{ 0xFD18, "F8ROM:RDKEY1" }, { 0xFD1B, "F8ROM:KEYIN" }, { 0xFD35, "F8ROM:RDCHAR" }, { 0xFD67, "F8ROM:GETLNZ" },
	{ 0xFD6A, "F8ROM:GETLN" }, { 0xFD6C, "F8ROM:GETLN0" }, { 0xFD6F, "F8ROM:GETLN1" }, { 0xFD8B, "F8ROM:CROUT1" },
	{ 0xFD8E, "F8ROM:CROUT" }, { 0xFD92, "F8ROM:PRA1" }, { 0xFDDA, "F8ROM:PRBYTE" }, { 0xFDE3, "F8ROM:PRHEX" },
	{ 0xFDED, "F8ROM:COUT" }, { 0xFDF0, "F8ROM:COUT1" }, { 0xFDF6, "F8ROM:COUTZ" }, { 0xFE1F, "F8ROM:IDROUTINE" },
	{ 0xFE2C, "F8ROM:MOVE" }, { 0xFE5E, "F8ROM:LIST" }, { 0xFE80, "F8ROM:SETINV" }, { 0xFE84, "F8ROM:SETNORM" },
	{ 0xFE89, "F8ROM:SETKBD" }, { 0xFE8B, "F8ROM:INPORT" }, { 0xFE93, "F8ROM:SETVID" }, { 0xFE95, "F8ROM:OUTPORT" },
	{ 0xFEB6, "F8ROM:GO" }, { 0xFECD, "F8ROM:WRITE" }, { 0xFEFD, "F8ROM:READ" }, { 0xFF2D, "F8ROM:PRERR" },
	{ 0xFF3A, "F8ROM:BELL" }, { 0xFF3F, "F8ROM:RESTORE" }, { 0xFF4A, "F8ROM:SAVE" }, { 0xFF58, "F8ROM:IORTS" },
	{ 0xFF59, "F8ROM:OLDRST" }, { 0xFF65, "F8ROM:MON" }, { 0xFF69, "F8ROM:MONZ" }, { 0xFF6C, "F8ROM:MONZ2" },
	{ 0xFF70, "F8ROM:MONZ4" }, { 0xFF8A, "F8ROM:DIG" }, { 0xFFA7, "F8ROM:GETNUM" }, { 0xFFAD, "F8ROM:NXTCHR" },
	{ 0xFFBE, "F8ROM:TOSUB" }, { 0xFFC7, "F8ROM:ZMODE" },

	{ 0xffff, "" }
};

offs_t apple2_common_device::com_2byte_op(std::ostream &stream, offs_t pc, const util::disasm_interface::data_buffer &opcodes, const char *opname)
{
	int item = 0;
	u16 operand = (u16)opcodes.r8(pc+1);
	while ((a2_stuff[item].addr & 0xff00) == 0)
	{
		if (a2_stuff[item].addr == operand)
		{
			stream << opname << " <" << a2_stuff[item].name;
			return 2 | util::disasm_interface::SUPPORTED;
		}
		item++;
	}

	return 0;
}

offs_t apple2_common_device::com_3byte_op(std::ostream &stream, offs_t pc, const util::disasm_interface::data_buffer &opcodes, const char *opname)
{
	int item = 0;
	u16 operand = opcodes.r16(pc+1);
	while (a2_stuff[item].addr != 0xffff)
	{
		if (a2_stuff[item].addr == operand)
		{
			stream << opname << " " << a2_stuff[item].name;
			return 3 | util::disasm_interface::SUPPORTED;
		}
		item++;
	}

	return 0;
}

offs_t apple2_common_device::dasm_override(std::ostream &stream, offs_t pc, const util::disasm_interface::data_buffer &opcodes, const util::disasm_interface::data_buffer &params)
{
	u8 opcode;
	unsigned result = 0;

	opcode = opcodes.r8(pc);

	switch (opcode)
	{
		case 0xad: // LDA
			return com_3byte_op(stream, pc, opcodes, "lda");

		case 0x8d:   // STA
			return com_3byte_op(stream, pc, opcodes, "sta");

		case 0x2c:   // BIT
			return com_3byte_op(stream, pc, opcodes, "bit");

		case 0x20:   // JSR
			{
				u16 operand = opcodes.r16(pc + 1);
				if (operand == 0xbf00)
				{
					u8 p8call = opcodes.r8(pc + 3);
					u16 p8params = opcodes.r16(pc + 4);
					int item = 0;

					while (p8_calls[item].addr != 0xffff)
					{
						if (p8_calls[item].addr == p8call)
						{
							util::stream_format(stream, "jsr ProDOS 8: %s ($%04x)", p8_calls[item].name, p8params);
							return 6 | util::disasm_interface::SUPPORTED | util::disasm_interface::STEP_OVER;
						}
						item++;
					}
				}
				else
				{
					offs_t rv = com_3byte_op(stream, pc, opcodes, "jsr");

					if (rv > 0)
					{
						rv |= util::disasm_interface::STEP_OVER;
					}
					return rv;
				}
			}
			break;

		case 0x4c:   // JMP
			return com_3byte_op(stream, pc, opcodes, "jmp");

		case 0x84:  // STY ZP
			return com_2byte_op(stream, pc, opcodes, "sty");

		case 0x85:  // STA ZP
			return com_2byte_op(stream, pc, opcodes, "sta");

		case 0xa4:  // LDY ZP
			return com_2byte_op(stream, pc, opcodes, "ldy");

		case 0xa5:  // LDA ZP
			return com_2byte_op(stream, pc, opcodes, "lda");

		case 0x65:  // ADC ZP
			return com_2byte_op(stream, pc, opcodes, "adc");

		case 0xc5:  // CMP ZP
			return com_2byte_op(stream, pc, opcodes, "cmp");

		case 0xc6:  // DEC ZP
			return com_2byte_op(stream, pc, opcodes, "dec");

		case 0xe6:  // INC ZP
			return com_2byte_op(stream, pc, opcodes, "inc");

		default:
			break;
	}

	//result = 2 | util::disasm_interface::SUPPORTED;

	return result;
}
