/* -*- Mode: Java; c-basic-offset: 4; tab-width: 20; indent-tabs-mode: nil; -*-
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

package org.mozilla.gecko.lwt;

import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapShader;
import android.graphics.Canvas;
import android.graphics.ColorFilter;
import android.graphics.ComposeShader;
import android.graphics.LinearGradient;
import android.graphics.Paint;
import android.graphics.PixelFormat;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.Rect;
import android.graphics.Shader;
import android.graphics.drawable.Drawable;

/**
 * A special drawable used with lightweight themes. This draws a color
 * (with an optional color-filter) and a bitmap (with a linear gradient
 * to specify the alpha) in order.
 */
public class LightweightThemeDrawable extends Drawable {
    private final Paint mPaint;
    private Paint mColorPaint;

    private final Bitmap mBitmap;
    private final Resources mResources;

    private int mStartColor;
    private int mEndColor;
    private boolean mHorizontalGradient;

    public LightweightThemeDrawable(Resources resources, Bitmap bitmap) {
        mBitmap = bitmap;
        mResources = resources;

        mPaint = new Paint();
        mPaint.setAntiAlias(true);
        mPaint.setStrokeWidth(0.0f);
    }

    @Override
    protected void onBoundsChange(Rect bounds) {
        super.onBoundsChange(bounds);
        initializeBitmapShader();
    }

    @Override
    public void draw(Canvas canvas) {
        // Draw the colors, if available.
        if (mColorPaint != null) {
            canvas.drawPaint(mColorPaint);
        }

        // Draw the bitmap.
        canvas.drawPaint(mPaint);
    }

    @Override
    public int getOpacity() {
        return PixelFormat.TRANSLUCENT;
    }

    @Override
    public void setAlpha(int alpha) {
        // A StateListDrawable will reset the alpha value with 255.
        // We cannot use to be the bitmap alpha.
        mPaint.setAlpha(alpha);
    }

    @Override
    public void setColorFilter(ColorFilter filter) {
        mPaint.setColorFilter(filter);
    }

    /**
     * Creates a paint that paint a particular color.
     *
     * Note that the given color should include an alpha value.
     *
     * @param color The color to be painted.
     */
    public void setColor(int color) {
        mColorPaint = new Paint(mPaint);
        mColorPaint.setColor(color);
    }

    /**
     * Creates a paint that paint a particular color, and a filter for the color.
     *
     * Note that the given color should include an alpha value.
     *
     * @param color The color to be painted.
     * @param filter The filter color to be applied using SRC_OVER mode.
     */
    public void setColorWithFilter(int color, int filter) {
        mColorPaint = new Paint(mPaint);
        mColorPaint.setColor(color);
        mColorPaint.setColorFilter(new PorterDuffColorFilter(filter, PorterDuff.Mode.SRC_OVER));
    }

    public void setAlpha(final int startAlpha, final int endAlpha) {
        // By default we draw a vertical linear gradient.
        setAlpha(startAlpha, endAlpha, false);
    }

    /**
     * Set the alpha for the linear gradient used with the bitmap's shader.
     *
     * @param startAlpha The starting alpha (0..255) value to be applied to the LinearGradient.
     * @param endAlpha The ending alpha (0..255) value to be applied to the LinearGradient.
     * @param horizontalGradient Draw gradient in horizontal direction; otherwise in vertical direction.
     */
    public void setAlpha(final int startAlpha, final int endAlpha, final boolean horizontalGradient) {
        mStartColor = startAlpha << 24;
        mEndColor = endAlpha << 24;
        mHorizontalGradient = horizontalGradient;
        initializeBitmapShader();
    }

    private void initializeBitmapShader() {
        // A bitmap-shader to draw the bitmap.
        // Clamp mode will repeat the last row of pixels.
        // Hence its better to have an endAlpha of 0 for the linear-gradient.
        final BitmapShader bitmapShader = new BitmapShader(mBitmap, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);

        // A linear-gradient to specify the opacity of the bitmap.
        final LinearGradient gradient;
        if (mHorizontalGradient) {
            gradient = new LinearGradient(0, 0, mBitmap.getWidth(), 0, mStartColor, mEndColor, Shader.TileMode.CLAMP);
        } else {
            gradient = new LinearGradient(0, 0, 0, mBitmap.getHeight(), mStartColor, mEndColor, Shader.TileMode.CLAMP);
        }

        // Make a combined shader -- a performance win.
        // The linear-gradient is the 'SRC' and the bitmap-shader is the 'DST'.
        // Drawing the DST in the SRC will provide the opacity.
        mPaint.setShader(new ComposeShader(bitmapShader, gradient, PorterDuff.Mode.DST_IN));
    }
}
