/*
 * crypto module tests
 * Copyright (c) 2014-2015, Jouni Malinen <j@w1.fi>
 *
 * This software may be distributed under the terms of the BSD license.
 * See README for more details.
 */

#include "utils/includes.h"

#include "utils/common.h"
#include "utils/module_tests.h"
#include "crypto/aes_siv.h"
#include "crypto/aes_wrap.h"
#include "crypto/aes.h"
#include "crypto/ms_funcs.h"
#include "crypto/crypto.h"
#include "crypto/sha1.h"
#include "crypto/sha256.h"
#include "crypto/sha384.h"


static int test_siv(void)
{
#ifdef CONFIG_MESH
	/* RFC 5297, A.1. Deterministic Authenticated Encryption Example */
	u8 key[] = {
		0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8,
		0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
		0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
	};
	u8 ad[] = {
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
		0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27
	};
	u8 plaintext[] = {
		0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88,
		0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee
	};
	u8 iv_c[] = {
		0x85, 0x63, 0x2d, 0x07, 0xc6, 0xe8, 0xf3, 0x7f,
		0x95, 0x0a, 0xcd, 0x32, 0x0a, 0x2e, 0xcc, 0x93,
		0x40, 0xc0, 0x2b, 0x96, 0x90, 0xc4, 0xdc, 0x04,
		0xda, 0xef, 0x7f, 0x6a, 0xfe, 0x5c
	};
	/* RFC 5297, A.2. Nonce-Based Authenticated Encryption Example */
	u8 key_2[] = {
		0x7f, 0x7e, 0x7d, 0x7c, 0x7b, 0x7a, 0x79, 0x78,
		0x77, 0x76, 0x75, 0x74, 0x73, 0x72, 0x71, 0x70,
		0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
		0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f
	};
	u8 ad1_2[] = {
		0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
		0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff,
		0xde, 0xad, 0xda, 0xda, 0xde, 0xad, 0xda, 0xda,
		0xff, 0xee, 0xdd, 0xcc, 0xbb, 0xaa, 0x99, 0x88,
		0x77, 0x66, 0x55, 0x44, 0x33, 0x22, 0x11, 0x00
	};
	u8 ad2_2[] = {
		0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80,
		0x90, 0xa0
	};
	u8 nonce_2[] = {
		0x09, 0xf9, 0x11, 0x02, 0x9d, 0x74, 0xe3, 0x5b,
		0xd8, 0x41, 0x56, 0xc5, 0x63, 0x56, 0x88, 0xc0
	};
	u8 plaintext_2[] = {
		0x74, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20,
		0x73, 0x6f, 0x6d, 0x65, 0x20, 0x70, 0x6c, 0x61,
		0x69, 0x6e, 0x74, 0x65, 0x78, 0x74, 0x20, 0x74,
		0x6f, 0x20, 0x65, 0x6e, 0x63, 0x72, 0x79, 0x70,
		0x74, 0x20, 0x75, 0x73, 0x69, 0x6e, 0x67, 0x20,
		0x53, 0x49, 0x56, 0x2d, 0x41, 0x45, 0x53
	};
	u8 iv_c_2[] = {
		0x7b, 0xdb, 0x6e, 0x3b, 0x43, 0x26, 0x67, 0xeb,
		0x06, 0xf4, 0xd1, 0x4b, 0xff, 0x2f, 0xbd, 0x0f,
		0xcb, 0x90, 0x0f, 0x2f, 0xdd, 0xbe, 0x40, 0x43,
		0x26, 0x60, 0x19, 0x65, 0xc8, 0x89, 0xbf, 0x17,
		0xdb, 0xa7, 0x7c, 0xeb, 0x09, 0x4f, 0xa6, 0x63,
		0xb7, 0xa3, 0xf7, 0x48, 0xba, 0x8a, 0xf8, 0x29,
		0xea, 0x64, 0xad, 0x54, 0x4a, 0x27, 0x2e, 0x9c,
		0x48, 0x5b, 0x62, 0xa3, 0xfd, 0x5c, 0x0d
	};
	u8 out[2 * AES_BLOCK_SIZE + sizeof(plaintext_2)];
	const u8 *addr[3];
	size_t len[3];

	/* RFC 5297, A.1. Deterministic Authenticated Encryption Example */
	addr[0] = ad;
	len[0] = sizeof(ad);

	if (aes_siv_encrypt(key, sizeof(key), plaintext, sizeof(plaintext),
			    1, addr, len, out)) {
		wpa_printf(MSG_ERROR, "AES-SIV mode encryption failed");
		return 1;
	}
	if (os_memcmp(out, iv_c, sizeof(iv_c)) != 0) {
		wpa_printf(MSG_ERROR,
			   "AES-SIV mode encryption returned invalid cipher text");
		return 1;
	}

	if (aes_siv_decrypt(key, sizeof(key), iv_c, sizeof(iv_c),
			    1, addr, len, out)) {
		wpa_printf(MSG_ERROR, "AES-SIV mode decryption failed");
		return 1;
	}
	if (os_memcmp(out, plaintext, sizeof(plaintext)) != 0) {
		wpa_printf(MSG_ERROR,
			   "AES-SIV mode decryption returned invalid plain text");
		return 1;
	}

	/* RFC 5297, A.2. Nonce-Based Authenticated Encryption Example */
	addr[0] = ad1_2;
	len[0] = sizeof(ad1_2);
	addr[1] = ad2_2;
	len[1] = sizeof(ad2_2);
	addr[2] = nonce_2;
	len[2] = sizeof(nonce_2);

	if (aes_siv_encrypt(key_2, sizeof(key_2),
			    plaintext_2, sizeof(plaintext_2),
			    3, addr, len, out)) {
		wpa_printf(MSG_ERROR, "AES-SIV mode encryption failed");
		return 1;
	}
	if (os_memcmp(out, iv_c_2, sizeof(iv_c_2)) != 0) {
		wpa_printf(MSG_ERROR,
			   "AES-SIV mode encryption returned invalid cipher text");
		return 1;
	}

	if (aes_siv_decrypt(key_2, sizeof(key_2), iv_c_2, sizeof(iv_c_2),
			    3, addr, len, out)) {
		wpa_printf(MSG_ERROR, "AES-SIV mode decryption failed");
		return 1;
	}
	if (os_memcmp(out, plaintext_2, sizeof(plaintext_2)) != 0) {
		wpa_printf(MSG_ERROR,
			   "AES-SIV mode decryption returned invalid plain text");
		return 1;
	}

	wpa_printf(MSG_INFO, "AES-SIV test cases passed");
#endif /* CONFIG_MESH */

	return 0;
}


/* OMAC1 AES-128 test vectors from
 * http://csrc.nist.gov/CryptoToolkit/modes/proposedmodes/omac/omac-ad.pdf
 * which are same as the examples from NIST SP800-38B
 * http://csrc.nist.gov/CryptoToolkit/modes/800-38_Series_Publications/SP800-38B.pdf
 */

struct omac1_test_vector {
	u8 k[16];
	u8 msg[64];
	int msg_len;
	u8 tag[16];
};

static const struct omac1_test_vector omac1_test_vectors[] =
{
	{
		{ 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
		  0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c },
		{ },
		0,
		{ 0xbb, 0x1d, 0x69, 0x29, 0xe9, 0x59, 0x37, 0x28,
		  0x7f, 0xa3, 0x7d, 0x12, 0x9b, 0x75, 0x67, 0x46 }
	},
	{
		{ 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
		  0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c },
		{ 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
		  0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a},
		16,
		{ 0x07, 0x0a, 0x16, 0xb4, 0x6b, 0x4d, 0x41, 0x44,
		  0xf7, 0x9b, 0xdd, 0x9d, 0xd0, 0x4a, 0x28, 0x7c }
	},
	{
		{ 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
		  0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c },
		{ 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
		  0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
		  0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
		  0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
		  0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11 },
		40,
		{ 0xdf, 0xa6, 0x67, 0x47, 0xde, 0x9a, 0xe6, 0x30,
		  0x30, 0xca, 0x32, 0x61, 0x14, 0x97, 0xc8, 0x27 }
	},
	{
		{ 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
		  0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c },
		{ 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
		  0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
		  0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
		  0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
		  0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
		  0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
		  0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
		  0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 },
		64,
		{ 0x51, 0xf0, 0xbe, 0xbf, 0x7e, 0x3b, 0x9d, 0x92,
		  0xfc, 0x49, 0x74, 0x17, 0x79, 0x36, 0x3c, 0xfe }
	},
};


static int test_omac1_vector(const struct omac1_test_vector *tv,
			     unsigned int i)
{
	u8 key[] = {
		0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
		0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
	};
	u8 msg[] = { 0x12, 0x34, 0x56 };
	u8 result[24], result2[24];
	const u8 *addr[3];
	size_t len[3];

	if (omac1_aes_128(tv->k, tv->msg, tv->msg_len, result) ||
	    os_memcmp(result, tv->tag, 16) != 0) {
		wpa_printf(MSG_ERROR, "OMAC1-AES-128 test vector %u failed", i);
		return 1;
	}

	if (tv->msg_len > 1) {

		addr[0] = tv->msg;
		len[0] = 1;
		addr[1] = tv->msg + 1;
		len[1] = tv->msg_len - 1;

		if (omac1_aes_128_vector(tv->k, 2, addr, len, result) ||
		    os_memcmp(result, tv->tag, 16) != 0) {
			wpa_printf(MSG_ERROR,
				   "OMAC1-AES-128(vector) test vector %u failed",
				   i);
			return 1;
		}

		addr[0] = tv->msg;
		len[0] = tv->msg_len - 2;
		addr[1] = tv->msg + tv->msg_len - 2;
		len[1] = 1;
		addr[2] = tv->msg + tv->msg_len - 1;
		len[2] = 1;

		if (omac1_aes_128_vector(tv->k, 3, addr, len, result) ||
		    os_memcmp(result, tv->tag, 16) != 0) {
			wpa_printf(MSG_ERROR,
				   "OMAC1-AES-128(vector2) test vector %u failed",
				   i);
			return 1;
		}
	}

	addr[0] = &msg[0];
	len[0] = 1;
	addr[1] = &msg[1];
	len[1] = 1;
	addr[2] = &msg[2];
	len[2] = 1;
	if (omac1_aes_128(key, msg, sizeof(msg), result) ||
	    omac1_aes_128_vector(key, 3, addr, len, result2) ||
	    os_memcmp(result, result2, 16) != 0) {
		wpa_printf(MSG_ERROR, "OMAC1-AES-128 short test mismatch");
		return 1;
	}

	return 0;
}


static int test_omac1(void)
{
	unsigned int i;

	for (i = 0; i < ARRAY_SIZE(omac1_test_vectors); i++) {
		if (test_omac1_vector(&omac1_test_vectors[i], i))
			return 1;
	}

	wpa_printf(MSG_INFO, "OMAC1-AES-128 test cases passed");

	return 0;
}


static int test_eax(void)
{
#ifdef EAP_PSK
	u8 msg[] = { 0xF7, 0xFB };
	u8 key[] = { 0x91, 0x94, 0x5D, 0x3F, 0x4D, 0xCB, 0xEE, 0x0B,
		     0xF4, 0x5E, 0xF5, 0x22, 0x55, 0xF0, 0x95, 0xA4 };
	u8 nonce[] = { 0xBE, 0xCA, 0xF0, 0x43, 0xB0, 0xA2, 0x3D, 0x84,
		       0x31, 0x94, 0xBA, 0x97, 0x2C, 0x66, 0xDE, 0xBD };
	u8 hdr[] = { 0xFA, 0x3B, 0xFD, 0x48, 0x06, 0xEB, 0x53, 0xFA };
	u8 cipher[] = { 0x19, 0xDD, 0x5C, 0x4C, 0x93, 0x31, 0x04, 0x9D,
			0x0B, 0xDA, 0xB0, 0x27, 0x74, 0x08, 0xF6, 0x79,
			0x67, 0xE5 };
	u8 data[sizeof(msg)], tag[AES_BLOCK_SIZE];

	os_memcpy(data, msg, sizeof(msg));
	if (aes_128_eax_encrypt(key, nonce, sizeof(nonce), hdr, sizeof(hdr),
				data, sizeof(data), tag)) {
		wpa_printf(MSG_ERROR, "AES-128 EAX mode encryption failed");
		return 1;
	}
	if (os_memcmp(data, cipher, sizeof(data)) != 0) {
		wpa_printf(MSG_ERROR,
			   "AES-128 EAX mode encryption returned invalid cipher text");
		return 1;
	}
	if (os_memcmp(tag, cipher + sizeof(data), AES_BLOCK_SIZE) != 0) {
		wpa_printf(MSG_ERROR,
			   "AES-128 EAX mode encryption returned invalid tag");
		return 1;
	}

	if (aes_128_eax_decrypt(key, nonce, sizeof(nonce), hdr, sizeof(hdr),
				data, sizeof(data), tag)) {
		wpa_printf(MSG_ERROR, "AES-128 EAX mode decryption failed");
		return 1;
	}
	if (os_memcmp(data, msg, sizeof(data)) != 0) {
		wpa_printf(MSG_ERROR,
			   "AES-128 EAX mode decryption returned invalid plain text");
		return 1;
	}

	wpa_printf(MSG_INFO, "AES-128 EAX mode test cases passed");
#endif /* EAP_PSK */

	return 0;
}


static int test_cbc(void)
{
	struct cbc_test_vector {
		u8 key[16];
		u8 iv[16];
		u8 plain[32];
		u8 cipher[32];
		size_t len;
	} vectors[] = {
		{
			{ 0x06, 0xa9, 0x21, 0x40, 0x36, 0xb8, 0xa1, 0x5b,
			  0x51, 0x2e, 0x03, 0xd5, 0x34, 0x12, 0x00, 0x06 },
			{ 0x3d, 0xaf, 0xba, 0x42, 0x9d, 0x9e, 0xb4, 0x30,
			  0xb4, 0x22, 0xda, 0x80, 0x2c, 0x9f, 0xac, 0x41 },
			"Single block msg",
			{ 0xe3, 0x53, 0x77, 0x9c, 0x10, 0x79, 0xae, 0xb8,
			  0x27, 0x08, 0x94, 0x2d, 0xbe, 0x77, 0x18, 0x1a },
			16
		},
		{
			{ 0xc2, 0x86, 0x69, 0x6d, 0x88, 0x7c, 0x9a, 0xa0,
			  0x61, 0x1b, 0xbb, 0x3e, 0x20, 0x25, 0xa4, 0x5a },
			{ 0x56, 0x2e, 0x17, 0x99, 0x6d, 0x09, 0x3d, 0x28,
			  0xdd, 0xb3, 0xba, 0x69, 0x5a, 0x2e, 0x6f, 0x58 },
			{ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			  0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			  0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
			{ 0xd2, 0x96, 0xcd, 0x94, 0xc2, 0xcc, 0xcf, 0x8a,
			  0x3a, 0x86, 0x30, 0x28, 0xb5, 0xe1, 0xdc, 0x0a,
			  0x75, 0x86, 0x60, 0x2d, 0x25, 0x3c, 0xff, 0xf9,
			  0x1b, 0x82, 0x66, 0xbe, 0xa6, 0xd6, 0x1a, 0xb1 },
			32
		}
	};
	int ret = 0;
	u8 *buf;
	unsigned int i;

	for (i = 0; i < ARRAY_SIZE(vectors); i++) {
		struct cbc_test_vector *tv = &vectors[i];

		buf = os_malloc(tv->len);
		if (buf == NULL) {
			ret++;
			break;
		}

		os_memcpy(buf, tv->plain, tv->len);
		if (aes_128_cbc_encrypt(tv->key, tv->iv, buf, tv->len) ||
		    os_memcmp(buf, tv->cipher, tv->len) != 0) {
			wpa_printf(MSG_ERROR, "AES-CBC encrypt %d failed", i);
			ret++;
		}

		os_memcpy(buf, tv->cipher, tv->len);
		if (aes_128_cbc_decrypt(tv->key, tv->iv, buf, tv->len) ||
		    os_memcmp(buf, tv->plain, tv->len) != 0) {
			wpa_printf(MSG_ERROR, "AES-CBC decrypt %d failed", i);
			ret++;
		}

		os_free(buf);
	}

	return ret;
}


static int test_ecb(void)
{
#ifdef EAP_PSK
	struct ecb_test_vector {
		char *key;
		char *plaintext;
		char *ciphertext;
	} vectors[] = {
		/* CAVS 11.1 - ECBGFSbox128.rsp */
		{
			"00000000000000000000000000000000",
			"f34481ec3cc627bacd5dc3fb08f273e6",
			"0336763e966d92595a567cc9ce537f5e"
		},
		{
			"00000000000000000000000000000000",
			"9798c4640bad75c7c3227db910174e72",
			"a9a1631bf4996954ebc093957b234589"
		},
		{
			"00000000000000000000000000000000",
			"96ab5c2ff612d9dfaae8c31f30c42168",
			"ff4f8391a6a40ca5b25d23bedd44a597"
		},
		{
			"00000000000000000000000000000000",
			"6a118a874519e64e9963798a503f1d35",
			"dc43be40be0e53712f7e2bf5ca707209"
		},
		{
			"00000000000000000000000000000000",
			"cb9fceec81286ca3e989bd979b0cb284",
			"92beedab1895a94faa69b632e5cc47ce"
		},
		{
			"00000000000000000000000000000000",
			"b26aeb1874e47ca8358ff22378f09144",
			"459264f4798f6a78bacb89c15ed3d601"
		},
		{
			"00000000000000000000000000000000",
			"58c8e00b2631686d54eab84b91f0aca1",
			"08a4e2efec8a8e3312ca7460b9040bbf"
		},
		/* CAVS 11.1 - ECBKeySbox128.rsp */
		{
			"10a58869d74be5a374cf867cfb473859",
			"00000000000000000000000000000000",
			"6d251e6944b051e04eaa6fb4dbf78465"
		},
		{
			"caea65cdbb75e9169ecd22ebe6e54675",
			"00000000000000000000000000000000",
			"6e29201190152df4ee058139def610bb",
		}
	};
	int ret = 0;
	unsigned int i;
	u8 key[16], plain[16], cipher[16], out[16];

	for (i = 0; i < ARRAY_SIZE(vectors); i++) {
		struct ecb_test_vector *tv = &vectors[i];

		if (hexstr2bin(tv->key, key, sizeof(key)) ||
		    hexstr2bin(tv->plaintext, plain, sizeof(plain)) ||
		    hexstr2bin(tv->ciphertext, cipher, sizeof(cipher))) {
			wpa_printf(MSG_ERROR, "Invalid AES-ECB test vector %u",
				   i);
			ret++;
			continue;
		}

		if (aes_128_encrypt_block(key, plain, out) < 0 ||
		    os_memcmp(out, cipher, 16) != 0) {
			wpa_printf(MSG_ERROR, "AES-ECB encrypt %u failed", i);
			ret++;
		}
	}

	if (!ret)
		wpa_printf(MSG_INFO, "AES ECB mode test cases passed");

	return ret;
#endif /* EAP_PSK */

	return 0;
}


static int test_key_wrap(void)
{
	int ret = 0;

	/* RFC 3394 - Test vector 4.1 */
	u8 kek41[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
	};
	u8 plain41[] = {
		0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
		0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff
	};
	u8 crypt41[] = {
		0x1F, 0xA6, 0x8B, 0x0A, 0x81, 0x12, 0xB4, 0x47,
		0xAE, 0xF3, 0x4B, 0xD8, 0xFB, 0x5A, 0x7B, 0x82,
		0x9D, 0x3E, 0x86, 0x23, 0x71, 0xD2, 0xCF, 0xE5
	};
#ifndef CONFIG_BORINGSSL
	/* RFC 3394 - Test vector 4.2 */
	u8 kek42[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17
	};
	u8 plain42[] = {
		0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
		0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff
	};
	u8 crypt42[] = {
		0x96, 0x77, 0x8B, 0x25, 0xAE, 0x6C, 0xA4, 0x35,
		0xF9, 0x2B, 0x5B, 0x97, 0xC0, 0x50, 0xAE, 0xD2,
		0x46, 0x8A, 0xB8, 0xA1, 0x7A, 0xD8, 0x4E, 0x5D
	};
#endif /* CONFIG_BORINGSSL */
	/* RFC 3394 - Test vector 4.3 */
	u8 kek43[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
		0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F
	};
	u8 plain43[] = {
		0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
		0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff
	};
	u8 crypt43[] = {
		0x64, 0xE8, 0xC3, 0xF9, 0xCE, 0x0F, 0x5B, 0xA2,
		0x63, 0xE9, 0x77, 0x79, 0x05, 0x81, 0x8A, 0x2A,
		0x93, 0xC8, 0x19, 0x1E, 0x7D, 0x6E, 0x8A, 0xE7,
	};
#ifndef CONFIG_BORINGSSL
	/* RFC 3394 - Test vector 4.4 */
	u8 kek44[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17
	};
	u8 plain44[] = {
		0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
		0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
	};
	u8 crypt44[] = {
		0x03, 0x1D, 0x33, 0x26, 0x4E, 0x15, 0xD3, 0x32,
		0x68, 0xF2, 0x4E, 0xC2, 0x60, 0x74, 0x3E, 0xDC,
		0xE1, 0xC6, 0xC7, 0xDD, 0xEE, 0x72, 0x5A, 0x93,
		0x6B, 0xA8, 0x14, 0x91, 0x5C, 0x67, 0x62, 0xD2
	};
#endif /* CONFIG_BORINGSSL */
	/* RFC 3394 - Test vector 4.5 */
	u8 kek45[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
		0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F
	};
	u8 plain45[] = {
		0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
		0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
	};
	u8 crypt45[] = {
		0xA8, 0xF9, 0xBC, 0x16, 0x12, 0xC6, 0x8B, 0x3F,
		0xF6, 0xE6, 0xF4, 0xFB, 0xE3, 0x0E, 0x71, 0xE4,
		0x76, 0x9C, 0x8B, 0x80, 0xA3, 0x2C, 0xB8, 0x95,
		0x8C, 0xD5, 0xD1, 0x7D, 0x6B, 0x25, 0x4D, 0xA1,
	};
	/* RFC 3394 - Test vector 4.6 */
	u8 kek46[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
		0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F
	};
	u8 plain46[] = {
		0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
		0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
	};
	u8 crypt46[] = {
		0x28, 0xC9, 0xF4, 0x04, 0xC4, 0xB8, 0x10, 0xF4,
		0xCB, 0xCC, 0xB3, 0x5C, 0xFB, 0x87, 0xF8, 0x26,
		0x3F, 0x57, 0x86, 0xE2, 0xD8, 0x0E, 0xD3, 0x26,
		0xCB, 0xC7, 0xF0, 0xE7, 0x1A, 0x99, 0xF4, 0x3B,
		0xFB, 0x98, 0x8B, 0x9B, 0x7A, 0x02, 0xDD, 0x21
	};
	u8 result[40];

	wpa_printf(MSG_INFO, "RFC 3394 - Test vector 4.1");
	if (aes_wrap(kek41, sizeof(kek41), sizeof(plain41) / 8, plain41,
		     result)) {
		wpa_printf(MSG_ERROR, "AES-WRAP-128 reported failure");
		ret++;
	}
	if (os_memcmp(result, crypt41, sizeof(crypt41)) != 0) {
		wpa_printf(MSG_ERROR, "AES-WRAP-128 failed");
		ret++;
	}
	if (aes_unwrap(kek41, sizeof(kek41), sizeof(plain41) / 8, crypt41,
		       result)) {
		wpa_printf(MSG_ERROR, "AES-UNWRAP-128 reported failure");
		ret++;
	}
	if (os_memcmp(result, plain41, sizeof(plain41)) != 0) {
		wpa_printf(MSG_ERROR, "AES-UNWRAP-128 failed");
		ret++;
	}

#ifndef CONFIG_BORINGSSL
	wpa_printf(MSG_INFO, "RFC 3394 - Test vector 4.2");
	if (aes_wrap(kek42, sizeof(kek42), sizeof(plain42) / 8, plain42,
		     result)) {
		wpa_printf(MSG_ERROR, "AES-WRAP-192 reported failure");
		ret++;
	}
	if (os_memcmp(result, crypt42, sizeof(crypt42)) != 0) {
		wpa_printf(MSG_ERROR, "AES-WRAP-192 failed");
		ret++;
	}
	if (aes_unwrap(kek42, sizeof(kek42), sizeof(plain42) / 8, crypt42,
		       result)) {
		wpa_printf(MSG_ERROR, "AES-UNWRAP-192 reported failure");
		ret++;
	}
	if (os_memcmp(result, plain42, sizeof(plain42)) != 0) {
		wpa_printf(MSG_ERROR, "AES-UNWRAP-192 failed");
		ret++;
	}
#endif /* CONFIG_BORINGSSL */

	wpa_printf(MSG_INFO, "RFC 3394 - Test vector 4.3");
	if (aes_wrap(kek43, sizeof(kek43), sizeof(plain43) / 8, plain43,
		     result)) {
		wpa_printf(MSG_ERROR, "AES-WRAP-256 reported failure");
		ret++;
	}
	if (os_memcmp(result, crypt43, sizeof(crypt43)) != 0) {
		wpa_printf(MSG_ERROR, "AES-WRAP-256 failed");
		ret++;
	}
	if (aes_unwrap(kek43, sizeof(kek43), sizeof(plain43) / 8, crypt43,
		       result)) {
		wpa_printf(MSG_ERROR, "AES-UNWRAP-256 reported failure");
		ret++;
	}
	if (os_memcmp(result, plain43, sizeof(plain43)) != 0) {
		wpa_printf(MSG_ERROR, "AES-UNWRAP-256 failed");
		ret++;
	}

#ifndef CONFIG_BORINGSSL
	wpa_printf(MSG_INFO, "RFC 3394 - Test vector 4.4");
	if (aes_wrap(kek44, sizeof(kek44), sizeof(plain44) / 8, plain44,
		     result)) {
		wpa_printf(MSG_ERROR, "AES-WRAP-192 reported failure");
		ret++;
	}
	if (os_memcmp(result, crypt44, sizeof(crypt44)) != 0) {
		wpa_printf(MSG_ERROR, "AES-WRAP-192 failed");
		ret++;
	}
	if (aes_unwrap(kek44, sizeof(kek44), sizeof(plain44) / 8, crypt44,
		       result)) {
		wpa_printf(MSG_ERROR, "AES-UNWRAP-192 reported failure");
		ret++;
	}
	if (os_memcmp(result, plain44, sizeof(plain44)) != 0) {
		wpa_printf(MSG_ERROR, "AES-UNWRAP-192 failed");
		ret++;
	}
#endif /* CONFIG_BORINGSSL */

	wpa_printf(MSG_INFO, "RFC 3394 - Test vector 4.5");
	if (aes_wrap(kek45, sizeof(kek45), sizeof(plain45) / 8, plain45,
		     result)) {
		wpa_printf(MSG_ERROR, "AES-WRAP-256 reported failure");
		ret++;
	}
	if (os_memcmp(result, crypt45, sizeof(crypt45)) != 0) {
		wpa_printf(MSG_ERROR, "AES-WRAP-256 failed");
		ret++;
	}
	if (aes_unwrap(kek45, sizeof(kek45), sizeof(plain45) / 8, crypt45,
		       result)) {
		wpa_printf(MSG_ERROR, "AES-UNWRAP-256 reported failure");
		ret++;
	}
	if (os_memcmp(result, plain45, sizeof(plain45)) != 0) {
		wpa_printf(MSG_ERROR, "AES-UNWRAP-256 failed");
		ret++;
	}

	wpa_printf(MSG_INFO, "RFC 3394 - Test vector 4.6");
	if (aes_wrap(kek46, sizeof(kek46), sizeof(plain46) / 8, plain46,
		     result)) {
		wpa_printf(MSG_ERROR, "AES-WRAP-256 reported failure");
		ret++;
	}
	if (os_memcmp(result, crypt46, sizeof(crypt46)) != 0) {
		wpa_printf(MSG_ERROR, "AES-WRAP-256 failed");
		ret++;
	}
	if (aes_unwrap(kek46, sizeof(kek46), sizeof(plain46) / 8, crypt46,
		       result)) {
		wpa_printf(MSG_ERROR, "AES-UNWRAP-256 reported failure");
		ret++;
	}
	if (os_memcmp(result, plain46, sizeof(plain46)) != 0) {
		wpa_printf(MSG_ERROR, "AES-UNWRAP-256 failed");
		ret++;
	}

	if (!ret)
		wpa_printf(MSG_INFO, "AES key wrap/unwrap test cases passed");

	return ret;
}


static int test_aes_ctr(void)
{
	int res = 0;

#if defined(CONFIG_MESH) || defined(CONFIG_PSK)
	/* CTR-AES*.Encrypt test vectors from NIST SP 800-38a */
	const u8 key128[] = {
		0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
		0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
	};
	const u8 counter128[] = {
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
		0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
	};
	const u8 plain128[] = {
		0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
		0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
		0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
		0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
		0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
		0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
		0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
		0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
	};
	const u8 cipher128[] = {
		0x87, 0x4d, 0x61, 0x91, 0xb6, 0x20, 0xe3, 0x26,
		0x1b, 0xef, 0x68, 0x64, 0x99, 0x0d, 0xb6, 0xce,
		0x98, 0x06, 0xf6, 0x6b, 0x79, 0x70, 0xfd, 0xff,
		0x86, 0x17, 0x18, 0x7b, 0xb9, 0xff, 0xfd, 0xff,
		0x5a, 0xe4, 0xdf, 0x3e, 0xdb, 0xd5, 0xd3, 0x5e,
		0x5b, 0x4f, 0x09, 0x02, 0x0d, 0xb0, 0x3e, 0xab,
		0x1e, 0x03, 0x1d, 0xda, 0x2f, 0xbe, 0x03, 0xd1,
		0x79, 0x21, 0x70, 0xa0, 0xf3, 0x00, 0x9c, 0xee
	};
	const u8 key192[] = {
		0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52,
		0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
		0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b
	};
	const u8 counter192[] = {
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
		0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
	};
	const u8 plain192[] = {
		0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
		0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
		0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
		0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
		0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
		0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
		0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
		0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
	};
	const u8 cipher192[] = {
		0x1a, 0xbc, 0x93, 0x24, 0x17, 0x52, 0x1c, 0xa2,
		0x4f, 0x2b, 0x04, 0x59, 0xfe, 0x7e, 0x6e, 0x0b,
		0x09, 0x03, 0x39, 0xec, 0x0a, 0xa6, 0xfa, 0xef,
		0xd5, 0xcc, 0xc2, 0xc6, 0xf4, 0xce, 0x8e, 0x94,
		0x1e, 0x36, 0xb2, 0x6b, 0xd1, 0xeb, 0xc6, 0x70,
		0xd1, 0xbd, 0x1d, 0x66, 0x56, 0x20, 0xab, 0xf7,
		0x4f, 0x78, 0xa7, 0xf6, 0xd2, 0x98, 0x09, 0x58,
		0x5a, 0x97, 0xda, 0xec, 0x58, 0xc6, 0xb0, 0x50
	};
	const u8 key256[] = {
		0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
		0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
		0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
		0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4
	};
	const u8 counter256[] = {
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
		0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
	};
	const u8 plain256[] = {
		0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
		0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
		0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
		0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
		0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
		0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
		0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
		0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
	};
	const u8 cipher256[] = {
		0x60, 0x1e, 0xc3, 0x13, 0x77, 0x57, 0x89, 0xa5,
		0xb7, 0xa7, 0xf5, 0x04, 0xbb, 0xf3, 0xd2, 0x28,
		0xf4, 0x43, 0xe3, 0xca, 0x4d, 0x62, 0xb5, 0x9a,
		0xca, 0x84, 0xe9, 0x90, 0xca, 0xca, 0xf5, 0xc5,
		0x2b, 0x09, 0x30, 0xda, 0xa2, 0x3d, 0xe9, 0x4c,
		0xe8, 0x70, 0x17, 0xba, 0x2d, 0x84, 0x98, 0x8d,
		0xdf, 0xc9, 0xc5, 0x8d, 0xb6, 0x7a, 0xad, 0xa6,
		0x13, 0xc2, 0xdd, 0x08, 0x45, 0x79, 0x41, 0xa6
	};
	size_t len;
	u8 *tmp;

	wpa_printf(MSG_DEBUG, "CTR-AES128.Encrypt");
	len = sizeof(plain128);
	tmp = os_malloc(len);
	if (!tmp)
		return -1;
	os_memcpy(tmp, plain128, len);
	if (aes_ctr_encrypt(key128, sizeof(key128), counter128, tmp, len) < 0) {
		wpa_printf(MSG_ERROR, "aes_ctr_encrypt() failed");
		res = -1;
	} else if (os_memcmp(tmp, cipher128, len) != 0) {
		wpa_printf(MSG_ERROR,
			   "CTR-AES128.Encrypt test vector did not match");
		res = -1;
	}
	os_free(tmp);

	wpa_printf(MSG_DEBUG, "CTR-AES192.Encrypt");
	len = sizeof(plain192);
	tmp = os_malloc(len);
	if (!tmp)
		return -1;
	os_memcpy(tmp, plain192, len);
	if (aes_ctr_encrypt(key192, sizeof(key192), counter192, tmp, len) < 0) {
		wpa_printf(MSG_ERROR, "aes_ctr_encrypt() failed");
		res = -1;
	} else if (os_memcmp(tmp, cipher192, len) != 0) {
		wpa_printf(MSG_ERROR,
			   "CTR-AES192.Encrypt test vector did not match");
		res = -1;
	}
	os_free(tmp);

	wpa_printf(MSG_DEBUG, "CTR-AES256.Encrypt");
	len = sizeof(plain256);
	tmp = os_malloc(len);
	if (!tmp)
		return -1;
	os_memcpy(tmp, plain256, len);
	if (aes_ctr_encrypt(key256, sizeof(key256), counter256, tmp, len) < 0) {
		wpa_printf(MSG_ERROR, "aes_ctr_encrypt() failed");
		res = -1;
	} else if (os_memcmp(tmp, cipher256, len) != 0) {
		wpa_printf(MSG_ERROR,
			   "CTR-AES256.Encrypt test vector did not match");
		res = -1;
	}
	os_free(tmp);
#endif

	return res;
}


static int test_md5(void)
{
#ifndef CONFIG_FIPS
	struct {
		char *data;
		char *hash;
	} tests[] = {
		{
			"",
			"\xd4\x1d\x8c\xd9\x8f\x00\xb2\x04"
			"\xe9\x80\x09\x98\xec\xf8\x42\x7e"
		},
		{
			"a",
			"\x0c\xc1\x75\xb9\xc0\xf1\xb6\xa8"
			"\x31\xc3\x99\xe2\x69\x77\x26\x61"
		},
		{
			"abc",
			"\x90\x01\x50\x98\x3c\xd2\x4f\xb0"
			"\xd6\x96\x3f\x7d\x28\xe1\x7f\x72"
		},
		{
			"message digest",
			"\xf9\x6b\x69\x7d\x7c\xb7\x93\x8d"
			"\x52\x5a\x2f\x31\xaa\xf1\x61\xd0"
		},
		{
			"abcdefghijklmnopqrstuvwxyz",
			"\xc3\xfc\xd3\xd7\x61\x92\xe4\x00"
			"\x7d\xfb\x49\x6c\xca\x67\xe1\x3b"
		},
		{
			"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"
			"0123456789",
			"\xd1\x74\xab\x98\xd2\x77\xd9\xf5"
			"\xa5\x61\x1c\x2c\x9f\x41\x9d\x9f"
		},
		{
			"12345678901234567890123456789012345678901234567890"
			"123456789012345678901234567890",
			"\x57\xed\xf4\xa2\x2b\xe3\xc9\x55"
			"\xac\x49\xda\x2e\x21\x07\xb6\x7a"
		}
	};
	unsigned int i;
	u8 hash[16];
	const u8 *addr[2];
	size_t len[2];
	int errors = 0;

	for (i = 0; i < ARRAY_SIZE(tests); i++) {
		wpa_printf(MSG_INFO, "MD5 test case %d", i);

		addr[0] = (u8 *) tests[i].data;
		len[0] = strlen(tests[i].data);
		if (md5_vector(1, addr, len, hash) < 0 ||
		    os_memcmp(hash, tests[i].hash, 16) != 0) {
			wpa_printf(MSG_INFO, " FAIL");
			errors++;
		} else
			wpa_printf(MSG_INFO, " OK");

		if (len[0]) {
			addr[0] = (u8 *) tests[i].data;
			len[0] = strlen(tests[i].data);
			addr[1] = (u8 *) tests[i].data + 1;
			len[1] = strlen(tests[i].data) - 1;
			if (md5_vector(1, addr, len, hash) < 0 ||
			    os_memcmp(hash, tests[i].hash, 16) != 0) {
				wpa_printf(MSG_INFO, " FAIL");
				errors++;
			} else
				wpa_printf(MSG_INFO, " OK");
		}
	}

	if (!errors)
		wpa_printf(MSG_INFO, "MD5 test cases passed");

	return errors;
#else /* CONFIG_FIPS */
	wpa_printf(MSG_INFO, "MD5 test cases skipped due to CONFIG_FIPS");
	return 0;
#endif /* CONFIG_FIPS */
}


static int test_eap_fast(void)
{
#ifdef EAP_FAST
	/* RFC 4851, Appendix B.1 */
	const u8 pac_key[] = {
		0x0B, 0x97, 0x39, 0x0F, 0x37, 0x51, 0x78, 0x09,
		0x81, 0x1E, 0xFD, 0x9C, 0x6E, 0x65, 0x94, 0x2B,
		0x63, 0x2C, 0xE9, 0x53, 0x89, 0x38, 0x08, 0xBA,
		0x36, 0x0B, 0x03, 0x7C, 0xD1, 0x85, 0xE4, 0x14
	};
	const u8 seed[] = {
		0x3F, 0xFB, 0x11, 0xC4, 0x6C, 0xBF, 0xA5, 0x7A,
		0x54, 0x40, 0xDA, 0xE8, 0x22, 0xD3, 0x11, 0xD3,
		0xF7, 0x6D, 0xE4, 0x1D, 0xD9, 0x33, 0xE5, 0x93,
		0x70, 0x97, 0xEB, 0xA9, 0xB3, 0x66, 0xF4, 0x2A,
		0x00, 0x00, 0x00, 0x02, 0x6A, 0x66, 0x43, 0x2A,
		0x8D, 0x14, 0x43, 0x2C, 0xEC, 0x58, 0x2D, 0x2F,
		0xC7, 0x9C, 0x33, 0x64, 0xBA, 0x04, 0xAD, 0x3A,
		0x52, 0x54, 0xD6, 0xA5, 0x79, 0xAD, 0x1E, 0x00
	};
	const u8 master_secret[] = {
		0x4A, 0x1A, 0x51, 0x2C, 0x01, 0x60, 0xBC, 0x02,
		0x3C, 0xCF, 0xBC, 0x83, 0x3F, 0x03, 0xBC, 0x64,
		0x88, 0xC1, 0x31, 0x2F, 0x0B, 0xA9, 0xA2, 0x77,
		0x16, 0xA8, 0xD8, 0xE8, 0xBD, 0xC9, 0xD2, 0x29,
		0x38, 0x4B, 0x7A, 0x85, 0xBE, 0x16, 0x4D, 0x27,
		0x33, 0xD5, 0x24, 0x79, 0x87, 0xB1, 0xC5, 0xA2
	};
#ifndef CONFIG_FIPS
	const u8 key_block[] = {
		0x59, 0x59, 0xBE, 0x8E, 0x41, 0x3A, 0x77, 0x74,
		0x8B, 0xB2, 0xE5, 0xD3, 0x60, 0xAC, 0x4D, 0x35,
		0xDF, 0xFB, 0xC8, 0x1E, 0x9C, 0x24, 0x9C, 0x8B,
		0x0E, 0xC3, 0x1D, 0x72, 0xC8, 0x84, 0x9D, 0x57,
		0x48, 0x51, 0x2E, 0x45, 0x97, 0x6C, 0x88, 0x70,
		0xBE, 0x5F, 0x01, 0xD3, 0x64, 0xE7, 0x4C, 0xBB,
		0x11, 0x24, 0xE3, 0x49, 0xE2, 0x3B, 0xCD, 0xEF,
		0x7A, 0xB3, 0x05, 0x39, 0x5D, 0x64, 0x8A, 0x44,
		0x11, 0xB6, 0x69, 0x88, 0x34, 0x2E, 0x8E, 0x29,
		0xD6, 0x4B, 0x7D, 0x72, 0x17, 0x59, 0x28, 0x05,
		0xAF, 0xF9, 0xB7, 0xFF, 0x66, 0x6D, 0xA1, 0x96,
		0x8F, 0x0B, 0x5E, 0x06, 0x46, 0x7A, 0x44, 0x84,
		0x64, 0xC1, 0xC8, 0x0C, 0x96, 0x44, 0x09, 0x98,
		0xFF, 0x92, 0xA8, 0xB4, 0xC6, 0x42, 0x28, 0x71
	};
#endif /* CONFIG_FIPS */
	const u8 sks[] = {
		0xD6, 0x4B, 0x7D, 0x72, 0x17, 0x59, 0x28, 0x05,
		0xAF, 0xF9, 0xB7, 0xFF, 0x66, 0x6D, 0xA1, 0x96,
		0x8F, 0x0B, 0x5E, 0x06, 0x46, 0x7A, 0x44, 0x84,
		0x64, 0xC1, 0xC8, 0x0C, 0x96, 0x44, 0x09, 0x98,
		0xFF, 0x92, 0xA8, 0xB4, 0xC6, 0x42, 0x28, 0x71
	};
	const u8 isk[] = {
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	};
	const u8 imck[] = {
		0x16, 0x15, 0x3C, 0x3F, 0x21, 0x55, 0xEF, 0xD9,
		0x7F, 0x34, 0xAE, 0xC8, 0x1A, 0x4E, 0x66, 0x80,
		0x4C, 0xC3, 0x76, 0xF2, 0x8A, 0xA9, 0x6F, 0x96,
		0xC2, 0x54, 0x5F, 0x8C, 0xAB, 0x65, 0x02, 0xE1,
		0x18, 0x40, 0x7B, 0x56, 0xBE, 0xEA, 0xA7, 0xC5,
		0x76, 0x5D, 0x8F, 0x0B, 0xC5, 0x07, 0xC6, 0xB9,
		0x04, 0xD0, 0x69, 0x56, 0x72, 0x8B, 0x6B, 0xB8,
		0x15, 0xEC, 0x57, 0x7B
	};
	const u8 msk[] = {
		0x4D, 0x83, 0xA9, 0xBE, 0x6F, 0x8A, 0x74, 0xED,
		0x6A, 0x02, 0x66, 0x0A, 0x63, 0x4D, 0x2C, 0x33,
		0xC2, 0xDA, 0x60, 0x15, 0xC6, 0x37, 0x04, 0x51,
		0x90, 0x38, 0x63, 0xDA, 0x54, 0x3E, 0x14, 0xB9,
		0x27, 0x99, 0x18, 0x1E, 0x07, 0xBF, 0x0F, 0x5A,
		0x5E, 0x3C, 0x32, 0x93, 0x80, 0x8C, 0x6C, 0x49,
		0x67, 0xED, 0x24, 0xFE, 0x45, 0x40, 0xA0, 0x59,
		0x5E, 0x37, 0xC2, 0xE9, 0xD0, 0x5D, 0x0A, 0xE3
	};
	const u8 emsk[] = {
		0x3A, 0xD4, 0xAB, 0xDB, 0x76, 0xB2, 0x7F, 0x3B,
		0xEA, 0x32, 0x2C, 0x2B, 0x74, 0xF4, 0x28, 0x55,
		0xEF, 0x2D, 0xBA, 0x78, 0xC9, 0x57, 0x2F, 0x0D,
		0x06, 0xCD, 0x51, 0x7C, 0x20, 0x93, 0x98, 0xA9,
		0x76, 0xEA, 0x70, 0x21, 0xD7, 0x0E, 0x25, 0x54,
		0x97, 0xED, 0xB2, 0x8A, 0xF6, 0xED, 0xFD, 0x0A,
		0x2A, 0xE7, 0xA1, 0x58, 0x90, 0x10, 0x50, 0x44,
		0xB3, 0x82, 0x85, 0xDB, 0x06, 0x14, 0xD2, 0xF9
	};
	/* RFC 4851, Appendix B.2 */
	u8 tlv[] = {
		0x80, 0x0C, 0x00, 0x38, 0x00, 0x01, 0x01, 0x00,
		0xD8, 0x6A, 0x8C, 0x68, 0x3C, 0x32, 0x31, 0xA8,
		0x56, 0x63, 0xB6, 0x40, 0x21, 0xFE, 0x21, 0x14,
		0x4E, 0xE7, 0x54, 0x20, 0x79, 0x2D, 0x42, 0x62,
		0xC9, 0xBF, 0x53, 0x7F, 0x54, 0xFD, 0xAC, 0x58,
		0x43, 0x24, 0x6E, 0x30, 0x92, 0x17, 0x6D, 0xCF,
		0xE6, 0xE0, 0x69, 0xEB, 0x33, 0x61, 0x6A, 0xCC,
		0x05, 0xC5, 0x5B, 0xB7
	};
	const u8 compound_mac[] = {
		0x43, 0x24, 0x6E, 0x30, 0x92, 0x17, 0x6D, 0xCF,
		0xE6, 0xE0, 0x69, 0xEB, 0x33, 0x61, 0x6A, 0xCC,
		0x05, 0xC5, 0x5B, 0xB7
	};
	u8 buf[512];
	const u8 *simck, *cmk;
	int errors = 0;

	wpa_printf(MSG_INFO, "EAP-FAST test cases");

	wpa_printf(MSG_INFO, "- T-PRF (SHA1) test case / master_secret");
	if (sha1_t_prf(pac_key, sizeof(pac_key),
		       "PAC to master secret label hash",
		       seed, sizeof(seed), buf, sizeof(master_secret)) < 0 ||
	    os_memcmp(master_secret, buf, sizeof(master_secret)) != 0) {
		wpa_printf(MSG_INFO, "T-PRF test - FAILED!");
		errors++;
	}

#ifndef CONFIG_FIPS
	wpa_printf(MSG_INFO, "- PRF (TLS, SHA1/MD5) test case / key_block");
	if (tls_prf_sha1_md5(master_secret, sizeof(master_secret),
			     "key expansion", seed, sizeof(seed),
			     buf, sizeof(key_block)) ||
	    os_memcmp(key_block, buf, sizeof(key_block)) != 0) {
		wpa_printf(MSG_INFO, "PRF test - FAILED!");
		errors++;
	}
#endif /* CONFIG_FIPS */

	wpa_printf(MSG_INFO, "- T-PRF (SHA1) test case / IMCK");
	if (sha1_t_prf(sks, sizeof(sks), "Inner Methods Compound Keys",
		       isk, sizeof(isk), buf, sizeof(imck)) < 0 ||
	    os_memcmp(imck, buf, sizeof(imck)) != 0) {
		wpa_printf(MSG_INFO, "T-PRF test - FAILED!");
		errors++;
	}

	simck = imck;
	cmk = imck + 40;

	wpa_printf(MSG_INFO, "- T-PRF (SHA1) test case / MSK");
	if (sha1_t_prf(simck, 40, "Session Key Generating Function",
		       (u8 *) "", 0, buf, sizeof(msk)) < 0 ||
	    os_memcmp(msk, buf, sizeof(msk)) != 0) {
		wpa_printf(MSG_INFO, "T-PRF test - FAILED!");
		errors++;
	}

	wpa_printf(MSG_INFO, "- T-PRF (SHA1) test case / EMSK");
	if (sha1_t_prf(simck, 40, "Extended Session Key Generating Function",
		       (u8 *) "", 0, buf, sizeof(msk)) < 0 ||
	    os_memcmp(emsk, buf, sizeof(emsk)) != 0) {
		wpa_printf(MSG_INFO, "T-PRF test - FAILED!");
		errors++;
	}

	wpa_printf(MSG_INFO, "- Compound MAC test case");
	os_memset(tlv + sizeof(tlv) - 20, 0, 20);
	if (hmac_sha1(cmk, 20, tlv, sizeof(tlv), tlv + sizeof(tlv) - 20) < 0 ||
	    os_memcmp(tlv + sizeof(tlv) - 20, compound_mac,
		      sizeof(compound_mac)) != 0) {
		wpa_printf(MSG_INFO, "Compound MAC test - FAILED!");
		errors++;
	}

	return errors;
#else /* EAP_FAST */
	return 0;
#endif /* EAP_FAST */
}


static const u8 key0[] =
{
	0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	0x0b, 0x0b, 0x0b, 0x0b
};
static const u8 data0[] = "Hi There";
static const u8 prf0[] =
{
	0xbc, 0xd4, 0xc6, 0x50, 0xb3, 0x0b, 0x96, 0x84,
	0x95, 0x18, 0x29, 0xe0, 0xd7, 0x5f, 0x9d, 0x54,
	0xb8, 0x62, 0x17, 0x5e, 0xd9, 0xf0, 0x06, 0x06,
	0xe1, 0x7d, 0x8d, 0xa3, 0x54, 0x02, 0xff, 0xee,
	0x75, 0xdf, 0x78, 0xc3, 0xd3, 0x1e, 0x0f, 0x88,
	0x9f, 0x01, 0x21, 0x20, 0xc0, 0x86, 0x2b, 0xeb,
	0x67, 0x75, 0x3e, 0x74, 0x39, 0xae, 0x24, 0x2e,
	0xdb, 0x83, 0x73, 0x69, 0x83, 0x56, 0xcf, 0x5a
};

static const u8 key1[] = "Jefe";
static const u8 data1[] = "what do ya want for nothing?";
static const u8 prf1[] =
{
	0x51, 0xf4, 0xde, 0x5b, 0x33, 0xf2, 0x49, 0xad,
	0xf8, 0x1a, 0xeb, 0x71, 0x3a, 0x3c, 0x20, 0xf4,
	0xfe, 0x63, 0x14, 0x46, 0xfa, 0xbd, 0xfa, 0x58,
	0x24, 0x47, 0x59, 0xae, 0x58, 0xef, 0x90, 0x09,
	0xa9, 0x9a, 0xbf, 0x4e, 0xac, 0x2c, 0xa5, 0xfa,
	0x87, 0xe6, 0x92, 0xc4, 0x40, 0xeb, 0x40, 0x02,
	0x3e, 0x7b, 0xab, 0xb2, 0x06, 0xd6, 0x1d, 0xe7,
	0xb9, 0x2f, 0x41, 0x52, 0x90, 0x92, 0xb8, 0xfc
};


static const u8 key2[] =
{
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	0xaa, 0xaa, 0xaa, 0xaa
};
static const u8 data2[] =
{
	0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	0xdd, 0xdd
};
static const u8 prf2[] =
{
	0xe1, 0xac, 0x54, 0x6e, 0xc4, 0xcb, 0x63, 0x6f,
	0x99, 0x76, 0x48, 0x7b, 0xe5, 0xc8, 0x6b, 0xe1,
	0x7a, 0x02, 0x52, 0xca, 0x5d, 0x8d, 0x8d, 0xf1,
	0x2c, 0xfb, 0x04, 0x73, 0x52, 0x52, 0x49, 0xce,
	0x9d, 0xd8, 0xd1, 0x77, 0xea, 0xd7, 0x10, 0xbc,
	0x9b, 0x59, 0x05, 0x47, 0x23, 0x91, 0x07, 0xae,
	0xf7, 0xb4, 0xab, 0xd4, 0x3d, 0x87, 0xf0, 0xa6,
	0x8f, 0x1c, 0xbd, 0x9e, 0x2b, 0x6f, 0x76, 0x07
};


struct passphrase_test {
	char *passphrase;
	char *ssid;
	char psk[32];
};

static const struct passphrase_test passphrase_tests[] =
{
	{
		"password",
		"IEEE",
		{
			0xf4, 0x2c, 0x6f, 0xc5, 0x2d, 0xf0, 0xeb, 0xef,
			0x9e, 0xbb, 0x4b, 0x90, 0xb3, 0x8a, 0x5f, 0x90,
			0x2e, 0x83, 0xfe, 0x1b, 0x13, 0x5a, 0x70, 0xe2,
			0x3a, 0xed, 0x76, 0x2e, 0x97, 0x10, 0xa1, 0x2e
		}
	},
	{
		"ThisIsAPassword",
		"ThisIsASSID",
		{
			0x0d, 0xc0, 0xd6, 0xeb, 0x90, 0x55, 0x5e, 0xd6,
			0x41, 0x97, 0x56, 0xb9, 0xa1, 0x5e, 0xc3, 0xe3,
			0x20, 0x9b, 0x63, 0xdf, 0x70, 0x7d, 0xd5, 0x08,
			0xd1, 0x45, 0x81, 0xf8, 0x98, 0x27, 0x21, 0xaf
		}
	},
	{
		"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
		"ZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZ",
		{
			0xbe, 0xcb, 0x93, 0x86, 0x6b, 0xb8, 0xc3, 0x83,
			0x2c, 0xb7, 0x77, 0xc2, 0xf5, 0x59, 0x80, 0x7c,
			0x8c, 0x59, 0xaf, 0xcb, 0x6e, 0xae, 0x73, 0x48,
			0x85, 0x00, 0x13, 0x00, 0xa9, 0x81, 0xcc, 0x62
		}
	},
};

#define NUM_PASSPHRASE_TESTS ARRAY_SIZE(passphrase_tests)


struct rfc6070_test {
	char *p;
	char *s;
	int c;
	char dk[32];
	size_t dk_len;
};

static const struct rfc6070_test rfc6070_tests[] =
{
	{
		"password",
		"salt",
		1,
		{
			0x0c, 0x60, 0xc8, 0x0f, 0x96, 0x1f, 0x0e, 0x71,
			0xf3, 0xa9, 0xb5, 0x24, 0xaf, 0x60, 0x12, 0x06,
			0x2f, 0xe0, 0x37, 0xa6
		},
		20
	},
	{
		"password",
		"salt",
		2,
		{
			0xea, 0x6c, 0x01, 0x4d, 0xc7, 0x2d, 0x6f, 0x8c,
			0xcd, 0x1e, 0xd9, 0x2a, 0xce, 0x1d, 0x41, 0xf0,
			0xd8, 0xde, 0x89, 0x57
		},
		20
	},
	{
		"password",
		"salt",
		4096,
		{
			0x4b, 0x00, 0x79, 0x01, 0xb7, 0x65, 0x48, 0x9a,
			0xbe, 0xad, 0x49, 0xd9, 0x26, 0xf7, 0x21, 0xd0,
			0x65, 0xa4, 0x29, 0xc1
		},
		20
	},
#if 0 /* This takes quite long to derive.. */
	{
		"password",
		"salt",
		16777216,
		{
			0xee, 0xfe, 0x3d, 0x61, 0xcd, 0x4d, 0xa4, 0xe4,
			0xe9, 0x94, 0x5b, 0x3d, 0x6b, 0xa2, 0x15, 0x8c,
			0x26, 0x34, 0xe9, 0x84
		},
		20
	},
#endif
	{
		"passwordPASSWORDpassword",
		"saltSALTsaltSALTsaltSALTsaltSALTsalt",
		4096,
		{
			0x3d, 0x2e, 0xec, 0x4f, 0xe4, 0x1c, 0x84, 0x9b,
			0x80, 0xc8, 0xd8, 0x36, 0x62, 0xc0, 0xe4, 0x4a,
			0x8b, 0x29, 0x1a, 0x96, 0x4c, 0xf2, 0xf0, 0x70,
			0x38
		},
		25
	},
#if 0 /* \0 not currently supported in passphrase parameters.. */
	{
		"pass\0word",
		"sa\0lt",
		4096,
		{
			0x56, 0xfa, 0x6a, 0xa7, 0x55, 0x48, 0x09, 0x9d,
			0xcc, 0x37, 0xd7, 0xf0, 0x34, 0x25, 0xe0, 0xc3
		},
		16
	},
#endif
};

#define NUM_RFC6070_TESTS ARRAY_SIZE(rfc6070_tests)


static int test_sha1(void)
{
	u8 res[512];
	int ret = 0;
	unsigned int i;

	wpa_printf(MSG_INFO, "PRF-SHA1 test cases:");

	if (sha1_prf(key0, sizeof(key0), "prefix", data0, sizeof(data0) - 1,
		     res, sizeof(prf0)) == 0 &&
	    os_memcmp(res, prf0, sizeof(prf0)) == 0)
		wpa_printf(MSG_INFO, "Test case 0 - OK");
	else {
		wpa_printf(MSG_INFO, "Test case 0 - FAILED!");
		ret++;
	}

	if (sha1_prf(key1, sizeof(key1) - 1, "prefix", data1, sizeof(data1) - 1,
		     res, sizeof(prf1)) == 0 &&
	    os_memcmp(res, prf1, sizeof(prf1)) == 0)
		wpa_printf(MSG_INFO, "Test case 1 - OK");
	else {
		wpa_printf(MSG_INFO, "Test case 1 - FAILED!");
		ret++;
	}

	if (sha1_prf(key2, sizeof(key2), "prefix", data2, sizeof(data2),
		     res, sizeof(prf2)) == 0 &&
	    os_memcmp(res, prf2, sizeof(prf2)) == 0)
		wpa_printf(MSG_INFO, "Test case 2 - OK");
	else {
		wpa_printf(MSG_INFO, "Test case 2 - FAILED!");
		ret++;
	}

	ret += test_eap_fast();

	wpa_printf(MSG_INFO, "PBKDF2-SHA1 Passphrase test cases:");
	for (i = 0; i < NUM_PASSPHRASE_TESTS; i++) {
		u8 psk[32];
		const struct passphrase_test *test = &passphrase_tests[i];

		if (pbkdf2_sha1(test->passphrase,
				(const u8 *) test->ssid, strlen(test->ssid),
				4096, psk, 32) == 0 &&
		    os_memcmp(psk, test->psk, 32) == 0)
			wpa_printf(MSG_INFO, "Test case %d - OK", i);
		else {
			wpa_printf(MSG_INFO, "Test case %d - FAILED!", i);
			ret++;
		}
	}

	wpa_printf(MSG_INFO, "PBKDF2-SHA1 test cases (RFC 6070):");
	for (i = 0; i < NUM_RFC6070_TESTS; i++) {
		u8 dk[25];
		const struct rfc6070_test *test = &rfc6070_tests[i];

		if (pbkdf2_sha1(test->p, (const u8 *) test->s, strlen(test->s),
				test->c, dk, test->dk_len) == 0 &&
		    os_memcmp(dk, test->dk, test->dk_len) == 0)
			wpa_printf(MSG_INFO, "Test case %d - OK", i);
		else {
			wpa_printf(MSG_INFO, "Test case %d - FAILED!", i);
			ret++;
		}
	}

	if (!ret)
		wpa_printf(MSG_INFO, "SHA1 test cases passed");
	return ret;
}


static const struct {
	char *data;
	u8 hash[32];
} tests[] = {
	{
		"abc",
		{
			0xba, 0x78, 0x16, 0xbf, 0x8f, 0x01, 0xcf, 0xea,
			0x41, 0x41, 0x40, 0xde, 0x5d, 0xae, 0x22, 0x23,
			0xb0, 0x03, 0x61, 0xa3, 0x96, 0x17, 0x7a, 0x9c,
			0xb4, 0x10, 0xff, 0x61, 0xf2, 0x00, 0x15, 0xad
		}
	},
	{
		"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		{
			0x24, 0x8d, 0x6a, 0x61, 0xd2, 0x06, 0x38, 0xb8,
			0xe5, 0xc0, 0x26, 0x93, 0x0c, 0x3e, 0x60, 0x39,
			0xa3, 0x3c, 0xe4, 0x59, 0x64, 0xff, 0x21, 0x67,
			0xf6, 0xec, 0xed, 0xd4, 0x19, 0xdb, 0x06, 0xc1
		}
	}
};

static const struct hmac_test {
	u8 key[150];
	size_t key_len;
	u8 data[160];
	size_t data_len;
	u8 hash[32]; /* HMAC-SHA-256 */
	u8 hash384[48]; /* HMAC-SHA-384 */
} hmac_tests[] = {
	/* draft-ietf-ipsec-ciph-sha-256-01.txt; RFC 4231 */
	{
		{
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20
		},
		32,
		"abc", 3,
		{
			0xa2, 0x1b, 0x1f, 0x5d, 0x4c, 0xf4, 0xf7, 0x3a,
			0x4d, 0xd9, 0x39, 0x75, 0x0f, 0x7a, 0x06, 0x6a,
			0x7f, 0x98, 0xcc, 0x13, 0x1c, 0xb1, 0x6a, 0x66,
			0x92, 0x75, 0x90, 0x21, 0xcf, 0xab, 0x81, 0x81
		},
		{ }
	},
	{
		{
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20
		},
		32,
		"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		56,
		{
			0x10, 0x4f, 0xdc, 0x12, 0x57, 0x32, 0x8f, 0x08,
			0x18, 0x4b, 0xa7, 0x31, 0x31, 0xc5, 0x3c, 0xae,
			0xe6, 0x98, 0xe3, 0x61, 0x19, 0x42, 0x11, 0x49,
			0xea, 0x8c, 0x71, 0x24, 0x56, 0x69, 0x7d, 0x30
		},
		{ }
	},
	{
		{
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20
		},
		32,
		"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"
		"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		112,
		{
			0x47, 0x03, 0x05, 0xfc, 0x7e, 0x40, 0xfe, 0x34,
			0xd3, 0xee, 0xb3, 0xe7, 0x73, 0xd9, 0x5a, 0xab,
			0x73, 0xac, 0xf0, 0xfd, 0x06, 0x04, 0x47, 0xa5,
			0xeb, 0x45, 0x95, 0xbf, 0x33, 0xa9, 0xd1, 0xa3
		},
		{ }
	},
	{
		{
			0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
			0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
			0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
			0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b
		},
		32,
		"Hi There",
		8,
		{
			0x19, 0x8a, 0x60, 0x7e, 0xb4, 0x4b, 0xfb, 0xc6,
			0x99, 0x03, 0xa0, 0xf1, 0xcf, 0x2b, 0xbd, 0xc5,
			0xba, 0x0a, 0xa3, 0xf3, 0xd9, 0xae, 0x3c, 0x1c,
			0x7a, 0x3b, 0x16, 0x96, 0xa0, 0xb6, 0x8c, 0xf7
		},
		{ }
	},
	{ /* RFC 4231 - Test Case 1 */
		{
			0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
			0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
			0x0b, 0x0b, 0x0b, 0x0b
		},
		20,
		"Hi There",
		8,
		{
			0xb0, 0x34, 0x4c, 0x61, 0xd8, 0xdb, 0x38, 0x53,
			0x5c, 0xa8, 0xaf, 0xce, 0xaf, 0x0b, 0xf1, 0x2b,
			0x88, 0x1d, 0xc2, 0x00, 0xc9, 0x83, 0x3d, 0xa7,
			0x26, 0xe9, 0x37, 0x6c, 0x2e, 0x32, 0xcf, 0xf7
		},
		{
			0xaf, 0xd0, 0x39, 0x44, 0xd8, 0x48, 0x95, 0x62,
			0x6b, 0x08, 0x25, 0xf4, 0xab, 0x46, 0x90, 0x7f,
			0x15, 0xf9, 0xda, 0xdb, 0xe4, 0x10, 0x1e, 0xc6,
			0x82, 0xaa, 0x03, 0x4c, 0x7c, 0xeb, 0xc5, 0x9c,
			0xfa, 0xea, 0x9e, 0xa9, 0x07, 0x6e, 0xde, 0x7f,
			0x4a, 0xf1, 0x52, 0xe8, 0xb2, 0xfa, 0x9c, 0xb6
		}
	},
	{ /* RFC 4231 - Test Case 2 */
		"Jefe",
		4,
		"what do ya want for nothing?",
		28,
		{
			0x5b, 0xdc, 0xc1, 0x46, 0xbf, 0x60, 0x75, 0x4e,
			0x6a, 0x04, 0x24, 0x26, 0x08, 0x95, 0x75, 0xc7,
			0x5a, 0x00, 0x3f, 0x08, 0x9d, 0x27, 0x39, 0x83,
			0x9d, 0xec, 0x58, 0xb9, 0x64, 0xec, 0x38, 0x43
		},
		{
			0xaf, 0x45, 0xd2, 0xe3, 0x76, 0x48, 0x40, 0x31,
			0x61, 0x7f, 0x78, 0xd2, 0xb5, 0x8a, 0x6b, 0x1b,
			0x9c, 0x7e, 0xf4, 0x64, 0xf5, 0xa0, 0x1b, 0x47,
			0xe4, 0x2e, 0xc3, 0x73, 0x63, 0x22, 0x44, 0x5e,
			0x8e, 0x22, 0x40, 0xca, 0x5e, 0x69, 0xe2, 0xc7,
			0x8b, 0x32, 0x39, 0xec, 0xfa, 0xb2, 0x16, 0x49
		}
	},
	{
		{
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa
		},
		32,
		{
			0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
			0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
			0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
			0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
			0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
			0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
			0xdd, 0xdd
		},
		50,
		{
			0xcd, 0xcb, 0x12, 0x20, 0xd1, 0xec, 0xcc, 0xea,
			0x91, 0xe5, 0x3a, 0xba, 0x30, 0x92, 0xf9, 0x62,
			0xe5, 0x49, 0xfe, 0x6c, 0xe9, 0xed, 0x7f, 0xdc,
			0x43, 0x19, 0x1f, 0xbd, 0xe4, 0x5c, 0x30, 0xb0
		},
		{ }
	},
	{ /* RFC 4231 - Test Case 3 */
		{
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa
		},
		20,
		{
			0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
			0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
			0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
			0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
			0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
			0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
			0xdd, 0xdd
		},
		50,
		{
			0x77, 0x3e, 0xa9, 0x1e, 0x36, 0x80, 0x0e, 0x46,
			0x85, 0x4d, 0xb8, 0xeb, 0xd0, 0x91, 0x81, 0xa7,
			0x29, 0x59, 0x09, 0x8b, 0x3e, 0xf8, 0xc1, 0x22,
			0xd9, 0x63, 0x55, 0x14, 0xce, 0xd5, 0x65, 0xfe
		},
		{
			0x88, 0x06, 0x26, 0x08, 0xd3, 0xe6, 0xad, 0x8a,
			0x0a, 0xa2, 0xac, 0xe0, 0x14, 0xc8, 0xa8, 0x6f,
			0x0a, 0xa6, 0x35, 0xd9, 0x47, 0xac, 0x9f, 0xeb,
			0xe8, 0x3e, 0xf4, 0xe5, 0x59, 0x66, 0x14, 0x4b,
			0x2a, 0x5a, 0xb3, 0x9d, 0xc1, 0x38, 0x14, 0xb9,
			0x4e, 0x3a, 0xb6, 0xe1, 0x01, 0xa3, 0x4f, 0x27
		}
	},
	{
		{
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
			0x21, 0x22, 0x23, 0x24, 0x25
		},
		37,
		{
			0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
			0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
			0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
			0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
			0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
			0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
			0xcd, 0xcd
		},
		50,
		{
			0xd4, 0x63, 0x3c, 0x17, 0xf6, 0xfb, 0x8d, 0x74,
			0x4c, 0x66, 0xde, 0xe0, 0xf8, 0xf0, 0x74, 0x55,
			0x6e, 0xc4, 0xaf, 0x55, 0xef, 0x07, 0x99, 0x85,
			0x41, 0x46, 0x8e, 0xb4, 0x9b, 0xd2, 0xe9, 0x17
		},
		{ }
	},
	{ /* RFC 4231 - Test Case 4 */
		{
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			0x19,
		},
		25,
		{
			0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
			0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
			0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
			0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
			0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
			0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
			0xcd, 0xcd
		},
		50,
		{
			0x82, 0x55, 0x8a, 0x38, 0x9a, 0x44, 0x3c, 0x0e,
			0xa4, 0xcc, 0x81, 0x98, 0x99, 0xf2, 0x08, 0x3a,
			0x85, 0xf0, 0xfa, 0xa3, 0xe5, 0x78, 0xf8, 0x07,
			0x7a, 0x2e, 0x3f, 0xf4, 0x67, 0x29, 0x66, 0x5b
		},
		{
			0x3e, 0x8a, 0x69, 0xb7, 0x78, 0x3c, 0x25, 0x85,
			0x19, 0x33, 0xab, 0x62, 0x90, 0xaf, 0x6c, 0xa7,
			0x7a, 0x99, 0x81, 0x48, 0x08, 0x50, 0x00, 0x9c,
			0xc5, 0x57, 0x7c, 0x6e, 0x1f, 0x57, 0x3b, 0x4e,
			0x68, 0x01, 0xdd, 0x23, 0xc4, 0xa7, 0xd6, 0x79,
			0xcc, 0xf8, 0xa3, 0x86, 0xc6, 0x74, 0xcf, 0xfb
		}
	},
	{
		{
			0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
			0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
			0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
			0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c
		},
		32,
		"Test With Truncation",
		20,
		{
			0x75, 0x46, 0xaf, 0x01, 0x84, 0x1f, 0xc0, 0x9b,
			0x1a, 0xb9, 0xc3, 0x74, 0x9a, 0x5f, 0x1c, 0x17,
			0xd4, 0xf5, 0x89, 0x66, 0x8a, 0x58, 0x7b, 0x27,
			0x00, 0xa9, 0xc9, 0x7c, 0x11, 0x93, 0xcf, 0x42
		},
		{ }
	},
	{
		{
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa
		},
		80,
		"Test Using Larger Than Block-Size Key - Hash Key First",
		54,
		{
			0x69, 0x53, 0x02, 0x5e, 0xd9, 0x6f, 0x0c, 0x09,
			0xf8, 0x0a, 0x96, 0xf7, 0x8e, 0x65, 0x38, 0xdb,
			0xe2, 0xe7, 0xb8, 0x20, 0xe3, 0xdd, 0x97, 0x0e,
			0x7d, 0xdd, 0x39, 0x09, 0x1b, 0x32, 0x35, 0x2f
		},
		{ }
	},
	{ /* RFC 4231 - Test Case 6 */
		{
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa
		},
		131,
		"Test Using Larger Than Block-Size Key - Hash Key First",
		54,
		{
			0x60, 0xe4, 0x31, 0x59, 0x1e, 0xe0, 0xb6, 0x7f,
			0x0d, 0x8a, 0x26, 0xaa, 0xcb, 0xf5, 0xb7, 0x7f,
			0x8e, 0x0b, 0xc6, 0x21, 0x37, 0x28, 0xc5, 0x14,
			0x05, 0x46, 0x04, 0x0f, 0x0e, 0xe3, 0x7f, 0x54
		},
		{
			0x4e, 0xce, 0x08, 0x44, 0x85, 0x81, 0x3e, 0x90,
			0x88, 0xd2, 0xc6, 0x3a, 0x04, 0x1b, 0xc5, 0xb4,
			0x4f, 0x9e, 0xf1, 0x01, 0x2a, 0x2b, 0x58, 0x8f,
			0x3c, 0xd1, 0x1f, 0x05, 0x03, 0x3a, 0xc4, 0xc6,
			0x0c, 0x2e, 0xf6, 0xab, 0x40, 0x30, 0xfe, 0x82,
			0x96, 0x24, 0x8d, 0xf1, 0x63, 0xf4, 0x49, 0x52
		}
	},
	{
		{
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa
		},
		80,
		"Test Using Larger Than Block-Size Key and Larger Than One "
		"Block-Size Data",
		73,
		{
			0x63, 0x55, 0xac, 0x22, 0xe8, 0x90, 0xd0, 0xa3,
			0xc8, 0x48, 0x1a, 0x5c, 0xa4, 0x82, 0x5b, 0xc8,
			0x84, 0xd3, 0xe7, 0xa1, 0xff, 0x98, 0xa2, 0xfc,
			0x2a, 0xc7, 0xd8, 0xe0, 0x64, 0xc3, 0xb2, 0xe6
		},
		{ }
	},
	{ /* RFC 4231 - Test Case 7 */
		{
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa
		},
		131,
		"This is a test using a larger than block-size key and a larger than block-size data. The key needs to be hashed before being used by the HMAC algorithm.",
		152,
		{
			0x9b, 0x09, 0xff, 0xa7, 0x1b, 0x94, 0x2f, 0xcb,
			0x27, 0x63, 0x5f, 0xbc, 0xd5, 0xb0, 0xe9, 0x44,
			0xbf, 0xdc, 0x63, 0x64, 0x4f, 0x07, 0x13, 0x93,
			0x8a, 0x7f, 0x51, 0x53, 0x5c, 0x3a, 0x35, 0xe2
		},
		{
			0x66, 0x17, 0x17, 0x8e, 0x94, 0x1f, 0x02, 0x0d,
			0x35, 0x1e, 0x2f, 0x25, 0x4e, 0x8f, 0xd3, 0x2c,
			0x60, 0x24, 0x20, 0xfe, 0xb0, 0xb8, 0xfb, 0x9a,
			0xdc, 0xce, 0xbb, 0x82, 0x46, 0x1e, 0x99, 0xc5,
			0xa6, 0x78, 0xcc, 0x31, 0xe7, 0x99, 0x17, 0x6d,
			0x38, 0x60, 0xe6, 0x11, 0x0c, 0x46, 0x52, 0x3e
		}
	}
};


static int test_sha256(void)
{
	unsigned int i;
	u8 hash[32];
	const u8 *addr[2];
	size_t len[2];
	int errors = 0;
	u8 *key;

	for (i = 0; i < ARRAY_SIZE(tests); i++) {
		wpa_printf(MSG_INFO, "SHA256 test case %d:", i + 1);

		addr[0] = (u8 *) tests[i].data;
		len[0] = strlen(tests[i].data);
		sha256_vector(1, addr, len, hash);
		if (memcmp(hash, tests[i].hash, 32) != 0) {
			wpa_printf(MSG_INFO, " FAIL");
			errors++;
		} else
			wpa_printf(MSG_INFO, " OK");

		if (len[0]) {
			addr[0] = (u8 *) tests[i].data;
			len[0] = 1;
			addr[1] = (u8 *) tests[i].data + 1;
			len[1] = strlen(tests[i].data) - 1;
			sha256_vector(2, addr, len, hash);
			if (memcmp(hash, tests[i].hash, 32) != 0) {
				wpa_printf(MSG_INFO, " FAIL");
				errors++;
			} else
				wpa_printf(MSG_INFO, " OK");
		}
	}

	for (i = 0; i < ARRAY_SIZE(hmac_tests); i++) {
		const struct hmac_test *t = &hmac_tests[i];

		wpa_printf(MSG_INFO, "HMAC-SHA256 test case %d:", i + 1);

		if (hmac_sha256(t->key, t->key_len, t->data, t->data_len,
				hash) < 0 ||
		    os_memcmp(hash, t->hash, 32) != 0) {
			wpa_printf(MSG_INFO, " FAIL");
			errors++;
		} else
			wpa_printf(MSG_INFO, " OK");

		addr[0] = t->data;
		len[0] = t->data_len;
		if (hmac_sha256_vector(t->key, t->key_len, 1, addr, len,
				       hash) < 0 ||
		    os_memcmp(hash, t->hash, 32) != 0) {
			wpa_printf(MSG_INFO, " FAIL");
			errors++;
		} else
			wpa_printf(MSG_INFO, " OK");

		if (len[0]) {
			addr[0] = t->data;
			len[0] = 1;
			addr[1] = t->data + 1;
			len[1] = t->data_len - 1;
			if (hmac_sha256_vector(t->key, t->key_len, 2, addr, len,
					       hash) < 0 ||
			    os_memcmp(hash, t->hash, 32) != 0) {
				wpa_printf(MSG_INFO, " FAIL");
				errors++;
			} else
				wpa_printf(MSG_INFO, " OK");
		}
	}

	wpa_printf(MSG_INFO, "Test IEEE 802.11r KDF");
	sha256_prf((u8 *) "abc", 3, "KDF test", (u8 *) "data", 4,
		   hash, sizeof(hash));
	/* TODO: add proper test case for this */

	key = os_malloc(8161);
	if (key) {
#ifdef CONFIG_HMAC_SHA256_KDF
		int res;

		res = hmac_sha256_kdf((u8 *) "secret", 6, "label",
				      (u8 *) "seed", 4, key, 8160);
		if (res) {
			wpa_printf(MSG_INFO,
				   "Unexpected hmac_sha256_kdf(outlen=8160) failure");
			errors++;
		}

		res = hmac_sha256_kdf((u8 *) "secret", 6, "label",
				      (u8 *) "seed", 4, key, 8161);
		if (res == 0) {
			wpa_printf(MSG_INFO,
				   "Unexpected hmac_sha256_kdf(outlen=8161) success");
			errors++;
		}
#endif /* CONFIG_HMAC_SHA256_KDF */

		os_free(key);
	}

	if (!errors)
		wpa_printf(MSG_INFO, "SHA256 test cases passed");
	return errors;
}


static int test_sha384(void)
{
#ifdef CONFIG_SHA384
	unsigned int i;
	u8 hash[48];
	const u8 *addr[2];
	size_t len[2];
	int errors = 0;
	const char *data = "hello";
	const u8 hash_res[] = {
		0x59, 0xe1, 0x74, 0x87, 0x77, 0x44, 0x8c, 0x69,
		0xde, 0x6b, 0x80, 0x0d, 0x7a, 0x33, 0xbb, 0xfb,
		0x9f, 0xf1, 0xb4, 0x63, 0xe4, 0x43, 0x54, 0xc3,
		0x55, 0x3b, 0xcd, 0xb9, 0xc6, 0x66, 0xfa, 0x90,
		0x12, 0x5a, 0x3c, 0x79, 0xf9, 0x03, 0x97, 0xbd,
		0xf5, 0xf6, 0xa1, 0x3d, 0xe8, 0x28, 0x68, 0x4f
	};

	addr[0] = (const u8 *) data;
	len[0] = 5;
	if (sha384_vector(1, addr, len, hash) < 0 ||
	    os_memcmp(hash, hash_res, 48) != 0) {
		wpa_printf(MSG_INFO, "SHA384 test case 1: FAIL");
		errors++;
	} else {
		wpa_printf(MSG_INFO, "SHA384 test case 1: OK");
	}

	addr[0] = (const u8 *) data;
	len[0] = 4;
	addr[1] = (const u8 *) data + 4;
	len[1] = 1;
	if (sha384_vector(2, addr, len, hash) < 0 ||
	    os_memcmp(hash, hash_res, 48) != 0) {
		wpa_printf(MSG_INFO, "SHA384 test case 2: FAIL");
		errors++;
	} else {
		wpa_printf(MSG_INFO, "SHA384 test case 2: OK");
	}

	for (i = 0; i < ARRAY_SIZE(hmac_tests); i++) {
		const struct hmac_test *t = &hmac_tests[i];

		if (t->hash384[0] == 0 && t->hash384[1] == 0 &&
		    t->hash384[2] == 0 && t->hash384[3] == 0)
			continue;
		wpa_printf(MSG_INFO, "HMAC-SHA384 test case %d:", i + 1);

		if (hmac_sha384(t->key, t->key_len, t->data, t->data_len,
				hash) < 0 ||
		    os_memcmp(hash, t->hash384, 48) != 0) {
			wpa_printf(MSG_INFO, " FAIL");
			errors++;
		} else
			wpa_printf(MSG_INFO, " OK");

		addr[0] = t->data;
		len[0] = t->data_len;
		if (hmac_sha384_vector(t->key, t->key_len, 1, addr, len,
				       hash) < 0 ||
		    os_memcmp(hash, t->hash384, 48) != 0) {
			wpa_printf(MSG_INFO, " FAIL");
			errors++;
		} else
			wpa_printf(MSG_INFO, " OK");

		if (len[0]) {
			addr[0] = t->data;
			len[0] = 1;
			addr[1] = t->data + 1;
			len[1] = t->data_len - 1;
			if (hmac_sha384_vector(t->key, t->key_len, 2, addr, len,
					       hash) < 0 ||
			    os_memcmp(hash, t->hash384, 48) != 0) {
				wpa_printf(MSG_INFO, " FAIL");
				errors++;
			} else
				wpa_printf(MSG_INFO, " OK");
		}
	}

	if (!errors)
		wpa_printf(MSG_INFO, "SHA384 test cases passed");
	return errors;
#else /* CONFIG_SHA384 */
	return 0;
#endif /* CONFIG_SHA384 */
}


static int test_fips186_2_prf(void)
{
	/* http://csrc.nist.gov/encryption/dss/Examples-1024bit.pdf */
	u8 xkey[] = {
		0xbd, 0x02, 0x9b, 0xbe, 0x7f, 0x51, 0x96, 0x0b,
		0xcf, 0x9e, 0xdb, 0x2b, 0x61, 0xf0, 0x6f, 0x0f,
		0xeb, 0x5a, 0x38, 0xb6
	};
	u8 w[] = {
		0x20, 0x70, 0xb3, 0x22, 0x3d, 0xba, 0x37, 0x2f,
		0xde, 0x1c, 0x0f, 0xfc, 0x7b, 0x2e, 0x3b, 0x49,
		0x8b, 0x26, 0x06, 0x14, 0x3c, 0x6c, 0x18, 0xba,
		0xcb, 0x0f, 0x6c, 0x55, 0xba, 0xbb, 0x13, 0x78,
		0x8e, 0x20, 0xd7, 0x37, 0xa3, 0x27, 0x51, 0x16
	};
	u8 buf[40];

	wpa_printf(MSG_INFO,
		   "Testing EAP-SIM PRF (FIPS 186-2 + change notice 1)");
	if (fips186_2_prf(xkey, sizeof(xkey), buf, sizeof(buf)) < 0 ||
	    os_memcmp(w, buf, sizeof(w)) != 0) {
		wpa_printf(MSG_INFO, "fips186_2_prf failed");
		return 1;
	}

	return 0;
}


static int test_extract_expand_hkdf(void)
{
	u8 prk[SHA256_MAC_LEN];
	u8 okm[82];

	/* RFC 5869, A.1 */
	u8 ikm1[22] = {
		0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
		0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
		0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b
	};
	u8 salt1[13] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c
	};
	u8 info1[10] = {
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
		0xf8, 0xf9
	};
	u8 prk1[32] = {
		0x07, 0x77, 0x09, 0x36, 0x2c, 0x2e, 0x32, 0xdf,
		0x0d, 0xdc, 0x3f, 0x0d, 0xc4, 0x7b, 0xba, 0x63,
		0x90, 0xb6, 0xc7, 0x3b, 0xb5, 0x0f, 0x9c, 0x31,
		0x22, 0xec, 0x84, 0x4a, 0xd7, 0xc2, 0xb3, 0xe5
	};
	u8 okm1[42] = {
		0x3c, 0xb2, 0x5f, 0x25, 0xfa, 0xac, 0xd5, 0x7a,
		0x90, 0x43, 0x4f, 0x64, 0xd0, 0x36, 0x2f, 0x2a,
		0x2d, 0x2d, 0x0a, 0x90, 0xcf, 0x1a, 0x5a, 0x4c,
		0x5d, 0xb0, 0x2d, 0x56, 0xec, 0xc4, 0xc5, 0xbf,
		0x34, 0x00, 0x72, 0x08, 0xd5, 0xb8, 0x87, 0x18,
		0x58, 0x65
	};

	/* RFC 5869, A.2 */
	u8 ikm2[80] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
		0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
		0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
		0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
		0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f
	};
	u8 salt2[80] = {
		0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
		0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
		0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
		0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
		0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
		0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
		0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf
	};
	u8 info2[80] = {
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
		0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
		0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
		0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
		0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
		0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
	};
	u8 prk2[32] = {
		0x06, 0xa6, 0xb8, 0x8c, 0x58, 0x53, 0x36, 0x1a,
		0x06, 0x10, 0x4c, 0x9c, 0xeb, 0x35, 0xb4, 0x5c,
		0xef, 0x76, 0x00, 0x14, 0x90, 0x46, 0x71, 0x01,
		0x4a, 0x19, 0x3f, 0x40, 0xc1, 0x5f, 0xc2, 0x44
	};
	u8 okm2[82] = {
		0xb1, 0x1e, 0x39, 0x8d, 0xc8, 0x03, 0x27, 0xa1,
		0xc8, 0xe7, 0xf7, 0x8c, 0x59, 0x6a, 0x49, 0x34,
		0x4f, 0x01, 0x2e, 0xda, 0x2d, 0x4e, 0xfa, 0xd8,
		0xa0, 0x50, 0xcc, 0x4c, 0x19, 0xaf, 0xa9, 0x7c,
		0x59, 0x04, 0x5a, 0x99, 0xca, 0xc7, 0x82, 0x72,
		0x71, 0xcb, 0x41, 0xc6, 0x5e, 0x59, 0x0e, 0x09,
		0xda, 0x32, 0x75, 0x60, 0x0c, 0x2f, 0x09, 0xb8,
		0x36, 0x77, 0x93, 0xa9, 0xac, 0xa3, 0xdb, 0x71,
		0xcc, 0x30, 0xc5, 0x81, 0x79, 0xec, 0x3e, 0x87,
		0xc1, 0x4c, 0x01, 0xd5, 0xc1, 0xf3, 0x43, 0x4f,
		0x1d, 0x87
	};

	wpa_printf(MSG_INFO, "Testing Extract-and-Expand HKDF (RFC 5869)");

	wpa_printf(MSG_INFO, "RFC 5869 - Test Case 1");
	if (hmac_sha256(salt1, sizeof(salt1), ikm1, sizeof(ikm1), prk) < 0)
		return -1;
	if (os_memcmp(prk, prk1, SHA256_MAC_LEN) != 0) {
		wpa_printf(MSG_INFO, "HKDF-Extract mismatch in PRK");
		return -1;
	}
	if (hmac_sha256_kdf(prk1, sizeof(prk1), NULL, info1, sizeof(info1),
			    okm, sizeof(okm1)) < 0)
		return -1;
	if (os_memcmp(okm, okm1, sizeof(okm1)) != 0) {
		wpa_printf(MSG_INFO, "HKDF-Expand mismatch in OKM");
		return -1;
	}

	wpa_printf(MSG_INFO, "RFC 5869 - Test Case 2");
	if (hmac_sha256(salt2, sizeof(salt2), ikm2, sizeof(ikm2), prk) < 0)
		return -1;
	if (os_memcmp(prk, prk2, SHA256_MAC_LEN) != 0) {
		wpa_printf(MSG_INFO, "HKDF-Extract mismatch in PRK");
		return -1;
	}
	if (hmac_sha256_kdf(prk2, sizeof(prk2), NULL, info2, sizeof(info2),
			    okm, sizeof(okm2)) < 0)
		return -1;
	if (os_memcmp(okm, okm2, sizeof(okm2)) != 0) {
		wpa_printf(MSG_INFO, "HKDF-Expand mismatch in OKM");
		return -1;
	}

	wpa_printf(MSG_INFO, "Extract-and-Expand HKDF test cases passed");

	return 0;
}


static int test_ms_funcs(void)
{
#ifndef CONFIG_FIPS
	/* Test vector from RFC2759 example */
	char *username = "User";
	char *password = "clientPass";
	u8 auth_challenge[] = {
		0x5B, 0x5D, 0x7C, 0x7D, 0x7B, 0x3F, 0x2F, 0x3E,
		0x3C, 0x2C, 0x60, 0x21, 0x32, 0x26, 0x26, 0x28
	};
	u8 peer_challenge[] = {
		0x21, 0x40, 0x23, 0x24, 0x25, 0x5E, 0x26, 0x2A,
		0x28, 0x29, 0x5F, 0x2B, 0x3A, 0x33, 0x7C, 0x7E
	};
	u8 password_hash[] = {
		0x44, 0xEB, 0xBA, 0x8D, 0x53, 0x12, 0xB8, 0xD6,
		0x11, 0x47, 0x44, 0x11, 0xF5, 0x69, 0x89, 0xAE
	};
	u8 nt_response[] = {
		0x82, 0x30, 0x9E, 0xCD, 0x8D, 0x70, 0x8B, 0x5E,
		0xA0, 0x8F, 0xAA, 0x39, 0x81, 0xCD, 0x83, 0x54,
		0x42, 0x33, 0x11, 0x4A, 0x3D, 0x85, 0xD6, 0xDF
	};
	u8 password_hash_hash[] = {
		0x41, 0xC0, 0x0C, 0x58, 0x4B, 0xD2, 0xD9, 0x1C,
		0x40, 0x17, 0xA2, 0xA1, 0x2F, 0xA5, 0x9F, 0x3F
	};
	u8 authenticator_response[] = {
		0x40, 0x7A, 0x55, 0x89, 0x11, 0x5F, 0xD0, 0xD6,
		0x20, 0x9F, 0x51, 0x0F, 0xE9, 0xC0, 0x45, 0x66,
		0x93, 0x2C, 0xDA, 0x56
	};
	u8 master_key[] = {
		0xFD, 0xEC, 0xE3, 0x71, 0x7A, 0x8C, 0x83, 0x8C,
		0xB3, 0x88, 0xE5, 0x27, 0xAE, 0x3C, 0xDD, 0x31
	};
	u8 send_start_key[] = {
		0x8B, 0x7C, 0xDC, 0x14, 0x9B, 0x99, 0x3A, 0x1B,
		0xA1, 0x18, 0xCB, 0x15, 0x3F, 0x56, 0xDC, 0xCB
	};
	u8 buf[32];
	int errors = 0;

	if (nt_password_hash((u8 *) password, os_strlen(password), buf) ||
	    os_memcmp(password_hash, buf, sizeof(password_hash)) != 0) {
		wpa_printf(MSG_ERROR, "nt_password_hash failed");
		errors++;
	}

	if (generate_nt_response(auth_challenge, peer_challenge,
				 (u8 *) username, os_strlen(username),
				 (u8 *) password, os_strlen(password), buf) ||
	    os_memcmp(nt_response, buf, sizeof(nt_response)) != 0) {
		wpa_printf(MSG_ERROR, "generate_nt_response failed");
		errors++;
	}

	if (hash_nt_password_hash(password_hash, buf) ||
	    os_memcmp(password_hash_hash, buf,
		      sizeof(password_hash_hash)) != 0) {
		wpa_printf(MSG_ERROR, "hash_nt_password_hash failed");
		errors++;
	}

	if (generate_authenticator_response((u8 *) password,
					    os_strlen(password),
					    peer_challenge, auth_challenge,
					    (u8 *) username,
					    os_strlen(username),
					    nt_response, buf) ||
	    os_memcmp(authenticator_response, buf,
		      sizeof(authenticator_response)) != 0) {
		wpa_printf(MSG_ERROR, "generate_authenticator_response failed");
		errors++;
	}

	if (get_master_key(password_hash_hash, nt_response, buf) ||
	    os_memcmp(master_key, buf, sizeof(master_key)) != 0) {
		wpa_printf(MSG_ERROR, "get_master_key failed");
		errors++;
	}

	if (get_asymetric_start_key(master_key, buf, sizeof(send_start_key),
				    1, 1) ||
	    os_memcmp(send_start_key, buf, sizeof(send_start_key)) != 0) {
		wpa_printf(MSG_ERROR, "get_asymetric_start_key failed");
		errors++;
	}

	if (errors)
		wpa_printf(MSG_ERROR, "ms_funcs: %d errors", errors);
	else
		wpa_printf(MSG_INFO, "ms_funcs test cases passed");

	return errors;
#else /* CONFIG_FIPS */
	wpa_printf(MSG_INFO, "ms_funcs test cases skipped due to CONFIG_FIPS");
	return 0;
#endif /* CONFIG_FIPS */
}


int crypto_module_tests(void)
{
	int ret = 0;

	wpa_printf(MSG_INFO, "crypto module tests");
	if (test_siv() ||
	    test_omac1() ||
	    test_eax() ||
	    test_cbc() ||
	    test_ecb() ||
	    test_key_wrap() ||
	    test_aes_ctr() ||
	    test_md5() ||
	    test_sha1() ||
	    test_sha256() ||
	    test_sha384() ||
	    test_fips186_2_prf() ||
	    test_extract_expand_hkdf() ||
	    test_ms_funcs())
		ret = -1;

	return ret;
}
