/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

/* JS reflection package. */

#include "mozilla/ArrayUtils.h"
#include "mozilla/DebugOnly.h"
#include "mozilla/Move.h"

#include <stdlib.h>

#include "jsarray.h"
#include "jsatom.h"
#include "jsobj.h"
#include "jspubtd.h"

#include "builtin/Reflect.h"
#include "frontend/Parser.h"
#include "frontend/TokenStream.h"
#include "js/CharacterEncoding.h"
#include "vm/RegExpObject.h"
#include "vm/BigIntType.h"

#include "jsobjinlines.h"

#include "frontend/ParseNode-inl.h"

using namespace js;
using namespace js::frontend;

using JS::AutoValueArray;
using mozilla::ArrayLength;
using mozilla::DebugOnly;
using mozilla::Forward;

enum ASTType {
    AST_ERROR = -1,
#define ASTDEF(ast, str, method) ast,
#include "jsast.tbl"
#undef ASTDEF
    AST_LIMIT
};

enum AssignmentOperator {
    AOP_ERR = -1,

    /* assign */
    AOP_ASSIGN = 0,
    /* operator-assign */
    AOP_PLUS, AOP_MINUS, AOP_STAR, AOP_DIV, AOP_MOD, AOP_POW,
    /* shift-assign */
    AOP_LSH, AOP_RSH, AOP_URSH,
    /* binary */
    AOP_BITOR, AOP_BITXOR, AOP_BITAND,
    /* short-circuit */
    AOP_COALESCE, AOP_OR, AOP_AND,

    AOP_LIMIT
};

enum BinaryOperator {
    BINOP_ERR = -1,

    /* eq */
    BINOP_EQ = 0, BINOP_NE, BINOP_STRICTEQ, BINOP_STRICTNE,
    /* rel */
    BINOP_LT, BINOP_LE, BINOP_GT, BINOP_GE,
    /* shift */
    BINOP_LSH, BINOP_RSH, BINOP_URSH,
    /* arithmetic */
    BINOP_ADD, BINOP_SUB, BINOP_STAR, BINOP_DIV, BINOP_MOD, BINOP_POW,
    /* binary */
    BINOP_BITOR, BINOP_BITXOR, BINOP_BITAND,
    /* misc */
    BINOP_IN, BINOP_INSTANCEOF, BINOP_COALESCE,

    BINOP_LIMIT
};

enum UnaryOperator {
    UNOP_ERR = -1,

    UNOP_DELETE = 0,
    UNOP_NEG,
    UNOP_POS,
    UNOP_NOT,
    UNOP_BITNOT,
    UNOP_TYPEOF,
    UNOP_VOID,
    UNOP_AWAIT,

    UNOP_LIMIT
};

enum VarDeclKind {
    VARDECL_ERR = -1,
    VARDECL_VAR = 0,
    VARDECL_CONST,
    VARDECL_LET,
    VARDECL_LIMIT
};

enum PropKind {
    PROP_ERR = -1,
    PROP_INIT = 0,
    PROP_GETTER,
    PROP_SETTER,
    PROP_MUTATEPROTO,
    PROP_LIMIT
};

static const char* const aopNames[] = {
    "=",    /* AOP_ASSIGN */
    "+=",   /* AOP_PLUS */
    "-=",   /* AOP_MINUS */
    "*=",   /* AOP_STAR */
    "/=",   /* AOP_DIV */
    "%=",   /* AOP_MOD */
    "**=",  /* AOP_POW */
    "<<=",  /* AOP_LSH */
    ">>=",  /* AOP_RSH */
    ">>>=", /* AOP_URSH */
    "|=",   /* AOP_BITOR */
    "^=",   /* AOP_BITXOR */
    "&="    /* AOP_BITAND */
    "\?\?=", /* AOP_COALESCE */
    "||=",  /* AOP_OR */
    "&&=",  /* AOP_AND */
};

static const char* const binopNames[] = {
    "==",         /* BINOP_EQ */
    "!=",         /* BINOP_NE */
    "===",        /* BINOP_STRICTEQ */
    "!==",        /* BINOP_STRICTNE */
    "<",          /* BINOP_LT */
    "<=",         /* BINOP_LE */
    ">",          /* BINOP_GT */
    ">=",         /* BINOP_GE */
    "<<",         /* BINOP_LSH */
    ">>",         /* BINOP_RSH */
    ">>>",        /* BINOP_URSH */
    "+",          /* BINOP_PLUS */
    "-",          /* BINOP_MINUS */
    "*",          /* BINOP_STAR */
    "/",          /* BINOP_DIV */
    "%",          /* BINOP_MOD */
    "**",         /* BINOP_POW */
    "|",          /* BINOP_BITOR */
    "^",          /* BINOP_BITXOR */
    "&",          /* BINOP_BITAND */
    "in",         /* BINOP_IN */
    "instanceof", /* BINOP_INSTANCEOF */
    "??"          /* BINOP_COALESCE */
};

static const char* const unopNames[] = {
    "delete",  /* UNOP_DELETE */
    "-",       /* UNOP_NEG */
    "+",       /* UNOP_POS */
    "!",       /* UNOP_NOT */
    "~",       /* UNOP_BITNOT */
    "typeof",  /* UNOP_TYPEOF */
    "void",    /* UNOP_VOID */
    "await"    /* UNOP_AWAIT */
};

static const char* const nodeTypeNames[] = {
#define ASTDEF(ast, str, method) str,
#include "jsast.tbl"
#undef ASTDEF
    nullptr
};

static const char* const callbackNames[] = {
#define ASTDEF(ast, str, method) method,
#include "jsast.tbl"
#undef ASTDEF
    nullptr
};

enum YieldKind { Delegating, NotDelegating };

typedef AutoValueVector NodeVector;

/*
 * ParseNode is a somewhat intricate data structure, and its invariants have
 * evolved, making it more likely that there could be a disconnect between the
 * parser and the AST serializer. We use these macros to check invariants on a
 * parse node and raise a dynamic error on failure.
 */
#define LOCAL_ASSERT(expr)                                                                \
    JS_BEGIN_MACRO                                                                        \
        MOZ_ASSERT(expr);                                                                 \
        if (!(expr)) {                                                                    \
            JS_ReportErrorNumberASCII(cx, GetErrorMessage, nullptr, JSMSG_BAD_PARSE_NODE);\
            return false;                                                                 \
        }                                                                                 \
    JS_END_MACRO

#define LOCAL_NOT_REACHED(expr)                                                           \
    JS_BEGIN_MACRO                                                                        \
        MOZ_ASSERT(false);                                                                \
        JS_ReportErrorNumberASCII(cx, GetErrorMessage, nullptr, JSMSG_BAD_PARSE_NODE);    \
        return false;                                                                     \
    JS_END_MACRO

namespace {

/* Set 'result' to obj[id] if any such property exists, else defaultValue. */
static bool
GetPropertyDefault(JSContext* cx, HandleObject obj, HandleId id, HandleValue defaultValue,
                   MutableHandleValue result)
{
    bool found;
    if (!HasProperty(cx, obj, id, &found))
        return false;
    if (!found) {
        result.set(defaultValue);
        return true;
    }
    return GetProperty(cx, obj, obj, id, result);
}

enum class GeneratorStyle
{
    None,
    Legacy,
    ES6
};

/*
 * Builder class that constructs JavaScript AST node objects. See:
 *
 *     https://developer.mozilla.org/en/SpiderMonkey/Parser_API
 *
 * Bug 569487: generalize builder interface
 */
class NodeBuilder
{
    typedef AutoValueArray<AST_LIMIT> CallbackArray;

    JSContext*  cx;
    TokenStream* tokenStream;
    bool        saveLoc;               /* save source location information?     */
    char const* src;                  /* source filename or null               */
    RootedValue srcval;                /* source filename JS value or null      */
    CallbackArray callbacks;           /* user-specified callbacks              */
    RootedValue userv;                 /* user-specified builder object or null */

  public:
    NodeBuilder(JSContext* c, bool l, char const* s)
      : cx(c), tokenStream(nullptr), saveLoc(l), src(s), srcval(c), callbacks(cx),
          userv(c)
    {}

    [[nodiscard]] bool init(HandleObject userobj = nullptr) {
        if (src) {
            if (!atomValue(src, &srcval))
                return false;
        } else {
            srcval.setNull();
        }

        if (!userobj) {
            userv.setNull();
            for (unsigned i = 0; i < AST_LIMIT; i++) {
                callbacks[i].setNull();
            }
            return true;
        }

        userv.setObject(*userobj);

        RootedValue nullVal(cx, NullValue());
        RootedValue funv(cx);
        for (unsigned i = 0; i < AST_LIMIT; i++) {
            const char* name = callbackNames[i];
            RootedAtom atom(cx, Atomize(cx, name, strlen(name)));
            if (!atom)
                return false;
            RootedId id(cx, AtomToId(atom));
            if (!GetPropertyDefault(cx, userobj, id, nullVal, &funv))
                return false;

            if (funv.isNullOrUndefined()) {
                callbacks[i].setNull();
                continue;
            }

            if (!funv.isObject() || !funv.toObject().is<JSFunction>()) {
                ReportValueErrorFlags(cx, JSREPORT_ERROR, JSMSG_NOT_FUNCTION,
                                      JSDVG_SEARCH_STACK, funv, nullptr, nullptr, nullptr);
                return false;
            }

            callbacks[i].set(funv);
        }

        return true;
    }

    void setTokenStream(TokenStream* ts) {
        tokenStream = ts;
    }

  private:
    [[nodiscard]] bool callbackHelper(HandleValue fun, const InvokeArgs& args, size_t i,
                                     TokenPos* pos, MutableHandleValue dst)
    {
        // The end of the implementation of callback(). All arguments except
        // loc have already been stored in range [0, i).
        if (saveLoc) {
            if (!newNodeLoc(pos, args[i]))
                return false;
        }

        return js::Call(cx, fun, userv, args, dst);
    }

    // Helper function for callback(). Note that all Arguments must be types
    // that convert to HandleValue, so this isn't as template-y as it seems,
    // just variadic.
    template <typename... Arguments>
    [[nodiscard]] bool callbackHelper(HandleValue fun, const InvokeArgs& args, size_t i,
                                     HandleValue head, Arguments&&... tail)
    {
        // Recursive loop to store the arguments into args. This eventually
        // bottoms out in a call to the non-template callbackHelper() above.
        args[i].set(head);
        return callbackHelper(fun, args, i + 1, Forward<Arguments>(tail)...);
    }

    // Invoke a user-defined callback. The actual signature is:
    //
    //     bool callback(HandleValue fun, HandleValue... args, TokenPos* pos,
    //                   MutableHandleValue dst);
    template <typename... Arguments>
    [[nodiscard]] bool callback(HandleValue fun, Arguments&&... args) {
        InvokeArgs iargs(cx);
        if (!iargs.init(cx, sizeof...(args) - 2 + size_t(saveLoc)))
            return false;

        return callbackHelper(fun, iargs, 0, Forward<Arguments>(args)...);
    }

    // WARNING: Returning a Handle is non-standard, but it works in this case
    // because both |v| and |UndefinedHandleValue| are definitely rooted on a
    // previous stack frame (i.e. we're just choosing between two
    // already-rooted values).
    HandleValue opt(HandleValue v) {
        MOZ_ASSERT_IF(v.isMagic(), v.whyMagic() == JS_SERIALIZE_NO_NODE);
        return v.isMagic(JS_SERIALIZE_NO_NODE) ? JS::UndefinedHandleValue : v;
    }

    [[nodiscard]] bool atomValue(const char* s, MutableHandleValue dst) {
        /*
         * Bug 575416: instead of Atomize, lookup constant atoms in tbl file
         */
        RootedAtom atom(cx, Atomize(cx, s, strlen(s)));
        if (!atom)
            return false;

        dst.setString(atom);
        return true;
    }

    [[nodiscard]] bool newObject(MutableHandleObject dst) {
        RootedPlainObject nobj(cx, NewBuiltinClassInstance<PlainObject>(cx));
        if (!nobj)
            return false;

        dst.set(nobj);
        return true;
    }

    [[nodiscard]] bool newArray(NodeVector& elts, MutableHandleValue dst);

    [[nodiscard]] bool createNode(ASTType type, TokenPos* pos, MutableHandleObject dst);

    [[nodiscard]] bool newNodeHelper(HandleObject obj, MutableHandleValue dst) {
        // The end of the implementation of newNode().
        MOZ_ASSERT(obj);
        dst.setObject(*obj);
        return true;
    }

    template <typename... Arguments>
    [[nodiscard]] bool newNodeHelper(HandleObject obj, const char *name, HandleValue value,
                                    Arguments&&... rest)
    {
        // Recursive loop to define properties. Note that the newNodeHelper()
        // call below passes two fewer arguments than we received, as we omit
        // `name` and `value`. This eventually bottoms out in a call to the
        // non-template newNodeHelper() above.
        return defineProperty(obj, name, value)
               && newNodeHelper(obj, Forward<Arguments>(rest)...);
    }

    // Create a node object with "type" and "loc" properties, as well as zero
    // or more properties passed in as arguments. The signature is really more
    // like:
    //
    //     bool newNode(ASTType type, TokenPos* pos,
    //                  {const char *name0, HandleValue value0,}...
    //                  MutableHandleValue dst);
    template <typename... Arguments>
    [[nodiscard]] bool newNode(ASTType type, TokenPos* pos, Arguments&&... args) {
        RootedObject node(cx);
        return createNode(type, pos, &node) &&
               newNodeHelper(node, Forward<Arguments>(args)...);
    }

    [[nodiscard]] bool listNode(ASTType type, const char* propName, NodeVector& elts, TokenPos* pos,
                               MutableHandleValue dst) {
        RootedValue array(cx);
        if (!newArray(elts, &array))
            return false;

        RootedValue cb(cx, callbacks[type]);
        if (!cb.isNull())
            return callback(cb, array, pos, dst);

        return newNode(type, pos, propName, array, dst);
    }

    [[nodiscard]] bool defineProperty(HandleObject obj, const char* name, HandleValue val) {
        MOZ_ASSERT_IF(val.isMagic(), val.whyMagic() == JS_SERIALIZE_NO_NODE);

        /*
         * Bug 575416: instead of Atomize, lookup constant atoms in tbl file
         */
        RootedAtom atom(cx, Atomize(cx, name, strlen(name)));
        if (!atom)
            return false;

        /* Represent "no node" as null and ensure users are not exposed to magic values. */
        RootedValue optVal(cx, val.isMagic(JS_SERIALIZE_NO_NODE) ? NullValue() : val);
        return DefineProperty(cx, obj, atom->asPropertyName(), optVal);
    }

    [[nodiscard]] bool newNodeLoc(TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool setNodeLoc(HandleObject node, TokenPos* pos);

  public:
    /*
     * All of the public builder methods take as their last two
     * arguments a nullable token position and a non-nullable, rooted
     * outparam.
     *
     * Any Value arguments representing optional subnodes may be a
     * JS_SERIALIZE_NO_NODE magic value.
     */

    /*
     * misc nodes
     */

    [[nodiscard]] bool program(NodeVector& elts, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool literal(HandleValue val, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool identifier(HandleValue name, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool function(ASTType type, TokenPos* pos,
                               HandleValue id, NodeVector& args, NodeVector& defaults,
                               HandleValue body, HandleValue rest, GeneratorStyle generatorStyle,
                               bool isAsync, bool isExpression, MutableHandleValue dst);

    [[nodiscard]] bool variableDeclarator(HandleValue id, HandleValue init, TokenPos* pos,
                                         MutableHandleValue dst);

    [[nodiscard]] bool switchCase(HandleValue expr, NodeVector& elts, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool catchClause(HandleValue var, HandleValue guard, HandleValue body, TokenPos* pos,
                                  MutableHandleValue dst);

    [[nodiscard]] bool prototypeMutation(HandleValue val, TokenPos* pos, MutableHandleValue dst);
    [[nodiscard]] bool propertyInitializer(HandleValue key, HandleValue val, PropKind kind,
                                          bool isShorthand, bool isMethod, TokenPos* pos,
                                          MutableHandleValue dst);


    /*
     * statements
     */

    [[nodiscard]] bool blockStatement(NodeVector& elts, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool expressionStatement(HandleValue expr, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool emptyStatement(TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool ifStatement(HandleValue test, HandleValue cons, HandleValue alt, TokenPos* pos,
                     MutableHandleValue dst);

    [[nodiscard]] bool breakStatement(HandleValue label, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool continueStatement(HandleValue label, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool labeledStatement(HandleValue label, HandleValue stmt, TokenPos* pos,
                          MutableHandleValue dst);

    [[nodiscard]] bool throwStatement(HandleValue arg, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool returnStatement(HandleValue arg, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool forStatement(HandleValue init, HandleValue test, HandleValue update, HandleValue stmt,
                      TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool forInStatement(HandleValue var, HandleValue expr, HandleValue stmt,
                                     bool isForEach, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool forOfStatement(HandleValue var, HandleValue expr, HandleValue stmt, TokenPos* pos,
                                     MutableHandleValue dst);

    [[nodiscard]] bool withStatement(HandleValue expr, HandleValue stmt, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool whileStatement(HandleValue test, HandleValue stmt, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool doWhileStatement(HandleValue stmt, HandleValue test, TokenPos* pos,
                                       MutableHandleValue dst);

    [[nodiscard]] bool switchStatement(HandleValue disc, NodeVector& elts, bool lexical, TokenPos* pos,
                                      MutableHandleValue dst);

    [[nodiscard]] bool tryStatement(HandleValue body, NodeVector& guarded, HandleValue unguarded,
                                   HandleValue finally, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool debuggerStatement(TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool importDeclaration(NodeVector& elts, HandleValue moduleSpec, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool importSpecifier(HandleValue importName, HandleValue bindingName, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool exportDeclaration(HandleValue decl, NodeVector& elts, HandleValue moduleSpec,
                                        HandleValue isDefault, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool exportSpecifier(HandleValue bindingName, HandleValue exportName, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool exportBatchSpecifier(TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool classDefinition(bool expr, HandleValue name, HandleValue heritage,
                                      HandleValue block, TokenPos* pos, MutableHandleValue dst);
    [[nodiscard]] bool classMembers(NodeVector& members, MutableHandleValue dst);
    [[nodiscard]] bool classMethod(HandleValue name, HandleValue body, PropKind kind, bool isStatic,
                                  TokenPos* pos, MutableHandleValue dst);
    [[nodiscard]] bool classField(HandleValue name, HandleValue initializer,
                                 TokenPos* pos, MutableHandleValue dst);
    [[nodiscard]] bool staticClassBlock(HandleValue body, TokenPos* pos, MutableHandleValue dst);

    /*
     * expressions
     */

    [[nodiscard]] bool binaryExpression(BinaryOperator op, HandleValue left, HandleValue right,
                                       TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool unaryExpression(UnaryOperator op, HandleValue expr, TokenPos* pos,
                                      MutableHandleValue dst);

    [[nodiscard]] bool assignmentExpression(AssignmentOperator op, HandleValue lhs, HandleValue rhs,
                                           TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool updateExpression(HandleValue expr, bool incr, bool prefix, TokenPos* pos,
                                       MutableHandleValue dst);

    [[nodiscard]] bool logicalExpression(ParseNodeKind kind, HandleValue left, HandleValue right, TokenPos* pos,
                                        MutableHandleValue dst);

    [[nodiscard]] bool conditionalExpression(HandleValue test, HandleValue cons, HandleValue alt,
                                            TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool sequenceExpression(NodeVector& elts, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool newExpression(HandleValue callee, NodeVector& args, TokenPos* pos,
                                    MutableHandleValue dst);

    [[nodiscard]] bool callExpression(HandleValue callee, NodeVector& args, TokenPos* pos,
                                     MutableHandleValue dst, bool isOptional = false);

    [[nodiscard]] bool memberExpression(bool computed, HandleValue expr, HandleValue member,
                                       TokenPos* pos, MutableHandleValue dst,
                                       bool isOptional = false);

    [[nodiscard]] bool arrayExpression(NodeVector& elts, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool templateLiteral(NodeVector& elts, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool taggedTemplate(HandleValue callee, NodeVector& args, TokenPos* pos,
                                     MutableHandleValue dst);

    [[nodiscard]] bool callSiteObj(NodeVector& raw, NodeVector& cooked, TokenPos* pos,
                                  MutableHandleValue dst);

    [[nodiscard]] bool spreadExpression(HandleValue expr, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool optionalExpression(HandleValue expr, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool deleteOptionalExpression(HandleValue expr, TokenPos* pos,
                                               MutableHandleValue dst);

    [[nodiscard]] bool computedName(HandleValue name, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool objectExpression(NodeVector& elts, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool thisExpression(TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool yieldExpression(HandleValue arg, YieldKind kind, TokenPos* pos,
                                      MutableHandleValue dst);

    [[nodiscard]] bool comprehensionBlock(HandleValue patt, HandleValue src, bool isForEach,
                                         bool isForOf, TokenPos* pos, MutableHandleValue dst);
    [[nodiscard]] bool comprehensionIf(HandleValue test, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool comprehensionExpression(HandleValue body, NodeVector& blocks,
                                              HandleValue filter, bool isLegacy, TokenPos* pos,
                                              MutableHandleValue dst);

    [[nodiscard]] bool generatorExpression(HandleValue body, NodeVector& blocks, HandleValue filter,
                                          bool isLegacy, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool metaProperty(HandleValue meta, HandleValue property, TokenPos* pos,
                                   MutableHandleValue dst);

    [[nodiscard]] bool callImportExpression(HandleValue ident, HandleValue arg, TokenPos* pos,
                                           MutableHandleValue dst);

    [[nodiscard]] bool super(TokenPos* pos, MutableHandleValue dst);

    /*
     * declarations
     */

    [[nodiscard]] bool variableDeclaration(NodeVector& elts, VarDeclKind kind, TokenPos* pos,
                                          MutableHandleValue dst);

    /*
     * patterns
     */

    [[nodiscard]] bool arrayPattern(NodeVector& elts, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool objectPattern(NodeVector& elts, TokenPos* pos, MutableHandleValue dst);

    [[nodiscard]] bool propertyPattern(HandleValue key, HandleValue patt, bool isShorthand,
                                      TokenPos* pos, MutableHandleValue dst);
};

} /* anonymous namespace */

bool
NodeBuilder::createNode(ASTType type, TokenPos* pos, MutableHandleObject dst)
{
    MOZ_ASSERT(type > AST_ERROR && type < AST_LIMIT);

    RootedValue tv(cx);
    RootedPlainObject node(cx, NewBuiltinClassInstance<PlainObject>(cx));
    if (!node ||
        !setNodeLoc(node, pos) ||
        !atomValue(nodeTypeNames[type], &tv) ||
        !defineProperty(node, "type", tv)) {
        return false;
    }

    dst.set(node);
    return true;
}

bool
NodeBuilder::newArray(NodeVector& elts, MutableHandleValue dst)
{
    const size_t len = elts.length();
    if (len > UINT32_MAX) {
        ReportAllocationOverflow(cx);
        return false;
    }
    RootedObject array(cx, NewDenseFullyAllocatedArray(cx, uint32_t(len)));
    if (!array)
        return false;

    for (size_t i = 0; i < len; i++) {
        RootedValue val(cx, elts[i]);

        MOZ_ASSERT_IF(val.isMagic(), val.whyMagic() == JS_SERIALIZE_NO_NODE);

        /* Represent "no node" as an array hole by not adding the value. */
        if (val.isMagic(JS_SERIALIZE_NO_NODE))
            continue;

        if (!DefineElement(cx, array, i, val))
            return false;
    }

    dst.setObject(*array);
    return true;
}

bool
NodeBuilder::newNodeLoc(TokenPos* pos, MutableHandleValue dst)
{
    if (!pos) {
        dst.setNull();
        return true;
    }

    RootedObject loc(cx);
    RootedObject to(cx);
    RootedValue val(cx);

    if (!newObject(&loc))
        return false;

    dst.setObject(*loc);

    uint32_t startLineNum, startColumnIndex;
    uint32_t endLineNum, endColumnIndex;
    tokenStream->srcCoords.lineNumAndColumnIndex(pos->begin, &startLineNum, &startColumnIndex);
    tokenStream->srcCoords.lineNumAndColumnIndex(pos->end, &endLineNum, &endColumnIndex);

    if (!newObject(&to))
        return false;
    val.setObject(*to);
    if (!defineProperty(loc, "start", val))
        return false;
    val.setNumber(startLineNum);
    if (!defineProperty(to, "line", val))
        return false;
    val.setNumber(startColumnIndex);
    if (!defineProperty(to, "column", val))
        return false;

    if (!newObject(&to))
        return false;
    val.setObject(*to);
    if (!defineProperty(loc, "end", val))
        return false;
    val.setNumber(endLineNum);
    if (!defineProperty(to, "line", val))
        return false;
    val.setNumber(endColumnIndex);
    if (!defineProperty(to, "column", val))
        return false;

    if (!defineProperty(loc, "source", srcval))
        return false;

    return true;
}

bool
NodeBuilder::setNodeLoc(HandleObject node, TokenPos* pos)
{
    if (!saveLoc) {
        RootedValue nullVal(cx, NullValue());
        return defineProperty(node, "loc", nullVal);
    }

    RootedValue loc(cx);
    return newNodeLoc(pos, &loc) &&
           defineProperty(node, "loc", loc);
}

bool
NodeBuilder::program(NodeVector& elts, TokenPos* pos, MutableHandleValue dst)
{
    return listNode(AST_PROGRAM, "body", elts, pos, dst);
}

bool
NodeBuilder::blockStatement(NodeVector& elts, TokenPos* pos, MutableHandleValue dst)
{
    return listNode(AST_BLOCK_STMT, "body", elts, pos, dst);
}

bool
NodeBuilder::expressionStatement(HandleValue expr, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_EXPR_STMT]);
    if (!cb.isNull())
        return callback(cb, expr, pos, dst);

    return newNode(AST_EXPR_STMT, pos, "expression", expr, dst);
}

bool
NodeBuilder::emptyStatement(TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_EMPTY_STMT]);
    if (!cb.isNull())
        return callback(cb, pos, dst);

    return newNode(AST_EMPTY_STMT, pos, dst);
}

bool
NodeBuilder::ifStatement(HandleValue test, HandleValue cons, HandleValue alt, TokenPos* pos,
                         MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_IF_STMT]);
    if (!cb.isNull())
        return callback(cb, test, cons, opt(alt), pos, dst);

    return newNode(AST_IF_STMT, pos,
                   "test", test,
                   "consequent", cons,
                   "alternate", alt,
                   dst);
}

bool
NodeBuilder::breakStatement(HandleValue label, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_BREAK_STMT]);
    if (!cb.isNull())
        return callback(cb, opt(label), pos, dst);

    return newNode(AST_BREAK_STMT, pos, "label", label, dst);
}

bool
NodeBuilder::continueStatement(HandleValue label, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_CONTINUE_STMT]);
    if (!cb.isNull())
        return callback(cb, opt(label), pos, dst);

    return newNode(AST_CONTINUE_STMT, pos, "label", label, dst);
}

bool
NodeBuilder::labeledStatement(HandleValue label, HandleValue stmt, TokenPos* pos,
                              MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_LAB_STMT]);
    if (!cb.isNull())
        return callback(cb, label, stmt, pos, dst);

    return newNode(AST_LAB_STMT, pos,
                   "label", label,
                   "body", stmt,
                   dst);
}

bool
NodeBuilder::throwStatement(HandleValue arg, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_THROW_STMT]);
    if (!cb.isNull())
        return callback(cb, arg, pos, dst);

    return newNode(AST_THROW_STMT, pos, "argument", arg, dst);
}

bool
NodeBuilder::returnStatement(HandleValue arg, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_RETURN_STMT]);
    if (!cb.isNull())
        return callback(cb, opt(arg), pos, dst);

    return newNode(AST_RETURN_STMT, pos, "argument", arg, dst);
}

bool
NodeBuilder::forStatement(HandleValue init, HandleValue test, HandleValue update, HandleValue stmt,
                          TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_FOR_STMT]);
    if (!cb.isNull())
        return callback(cb, opt(init), opt(test), opt(update), stmt, pos, dst);

    return newNode(AST_FOR_STMT, pos,
                   "init", init,
                   "test", test,
                   "update", update,
                   "body", stmt,
                   dst);
}

bool
NodeBuilder::forInStatement(HandleValue var, HandleValue expr, HandleValue stmt, bool isForEach,
                            TokenPos* pos, MutableHandleValue dst)
{
    RootedValue isForEachVal(cx, BooleanValue(isForEach));

    RootedValue cb(cx, callbacks[AST_FOR_IN_STMT]);
    if (!cb.isNull())
        return callback(cb, var, expr, stmt, isForEachVal, pos, dst);

    return newNode(AST_FOR_IN_STMT, pos,
                   "left", var,
                   "right", expr,
                   "body", stmt,
                   "each", isForEachVal,
                   dst);
}

bool
NodeBuilder::forOfStatement(HandleValue var, HandleValue expr, HandleValue stmt, TokenPos* pos,
                            MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_FOR_OF_STMT]);
    if (!cb.isNull())
        return callback(cb, var, expr, stmt, pos, dst);

    return newNode(AST_FOR_OF_STMT, pos,
                   "left", var,
                   "right", expr,
                   "body", stmt,
                   dst);
}

bool
NodeBuilder::withStatement(HandleValue expr, HandleValue stmt, TokenPos* pos,
                           MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_WITH_STMT]);
    if (!cb.isNull())
        return callback(cb, expr, stmt, pos, dst);

    return newNode(AST_WITH_STMT, pos,
                   "object", expr,
                   "body", stmt,
                   dst);
}

bool
NodeBuilder::whileStatement(HandleValue test, HandleValue stmt, TokenPos* pos,
                            MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_WHILE_STMT]);
    if (!cb.isNull())
        return callback(cb, test, stmt, pos, dst);

    return newNode(AST_WHILE_STMT, pos,
                   "test", test,
                   "body", stmt,
                   dst);
}

bool
NodeBuilder::doWhileStatement(HandleValue stmt, HandleValue test, TokenPos* pos,
                              MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_DO_STMT]);
    if (!cb.isNull())
        return callback(cb, stmt, test, pos, dst);

    return newNode(AST_DO_STMT, pos,
                   "body", stmt,
                   "test", test,
                   dst);
}

bool
NodeBuilder::switchStatement(HandleValue disc, NodeVector& elts, bool lexical, TokenPos* pos,
                             MutableHandleValue dst)
{
    RootedValue array(cx);
    if (!newArray(elts, &array))
        return false;

    RootedValue lexicalVal(cx, BooleanValue(lexical));

    RootedValue cb(cx, callbacks[AST_SWITCH_STMT]);
    if (!cb.isNull())
        return callback(cb, disc, array, lexicalVal, pos, dst);

    return newNode(AST_SWITCH_STMT, pos,
                   "discriminant", disc,
                   "cases", array,
                   "lexical", lexicalVal,
                   dst);
}

bool
NodeBuilder::tryStatement(HandleValue body, NodeVector& guarded, HandleValue unguarded,
                          HandleValue finally, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue guardedHandlers(cx);
    if (!newArray(guarded, &guardedHandlers))
        return false;

    RootedValue cb(cx, callbacks[AST_TRY_STMT]);
    if (!cb.isNull())
        return callback(cb, body, guardedHandlers, unguarded, opt(finally), pos, dst);

    return newNode(AST_TRY_STMT, pos,
                   "block", body,
                   "guardedHandlers", guardedHandlers,
                   "handler", unguarded,
                   "finalizer", finally,
                   dst);
}

bool
NodeBuilder::debuggerStatement(TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_DEBUGGER_STMT]);
    if (!cb.isNull())
        return callback(cb, pos, dst);

    return newNode(AST_DEBUGGER_STMT, pos, dst);
}

bool
NodeBuilder::binaryExpression(BinaryOperator op, HandleValue left, HandleValue right, TokenPos* pos,
                              MutableHandleValue dst)
{
    MOZ_ASSERT(op > BINOP_ERR && op < BINOP_LIMIT);

    RootedValue opName(cx);
    if (!atomValue(binopNames[op], &opName))
        return false;

    RootedValue cb(cx, callbacks[AST_BINARY_EXPR]);
    if (!cb.isNull())
        return callback(cb, opName, left, right, pos, dst);

    return newNode(AST_BINARY_EXPR, pos,
                   "operator", opName,
                   "left", left,
                   "right", right,
                   dst);
}

bool
NodeBuilder::unaryExpression(UnaryOperator unop, HandleValue expr, TokenPos* pos,
                             MutableHandleValue dst)
{
    MOZ_ASSERT(unop > UNOP_ERR && unop < UNOP_LIMIT);

    RootedValue opName(cx);
    if (!atomValue(unopNames[unop], &opName))
        return false;

    RootedValue cb(cx, callbacks[AST_UNARY_EXPR]);
    if (!cb.isNull())
        return callback(cb, opName, expr, pos, dst);

    RootedValue trueVal(cx, BooleanValue(true));
    return newNode(AST_UNARY_EXPR, pos,
                   "operator", opName,
                   "argument", expr,
                   "prefix", trueVal,
                   dst);
}

bool
NodeBuilder::assignmentExpression(AssignmentOperator aop, HandleValue lhs, HandleValue rhs,
                                  TokenPos* pos, MutableHandleValue dst)
{
    MOZ_ASSERT(aop > AOP_ERR && aop < AOP_LIMIT);

    RootedValue opName(cx);
    if (!atomValue(aopNames[aop], &opName))
        return false;

    RootedValue cb(cx, callbacks[AST_ASSIGN_EXPR]);
    if (!cb.isNull())
        return callback(cb, opName, lhs, rhs, pos, dst);

    return newNode(AST_ASSIGN_EXPR, pos,
                   "operator", opName,
                   "left", lhs,
                   "right", rhs,
                   dst);
}

bool
NodeBuilder::updateExpression(HandleValue expr, bool incr, bool prefix, TokenPos* pos,
                              MutableHandleValue dst)
{
    RootedValue opName(cx);
    if (!atomValue(incr ? "++" : "--", &opName))
        return false;

    RootedValue prefixVal(cx, BooleanValue(prefix));

    RootedValue cb(cx, callbacks[AST_UPDATE_EXPR]);
    if (!cb.isNull())
        return callback(cb, expr, opName, prefixVal, pos, dst);

    return newNode(AST_UPDATE_EXPR, pos,
                   "operator", opName,
                   "argument", expr,
                   "prefix", prefixVal,
                   dst);
}

bool
NodeBuilder::logicalExpression(ParseNodeKind kind,
                               HandleValue left,
                               HandleValue right,
                               TokenPos* pos,
                               MutableHandleValue dst)
{
    RootedValue opName(cx);
    switch (kind) {
      case PNK_COALESCE:
        if (!atomValue("??", &opName)) {
            return false;
        }
        break;

      case PNK_OR:
        if (!atomValue("||", &opName)) {
            return false;
        }
        break;

      case PNK_AND:
        if (!atomValue("&&", &opName)) {
            return false;
        }
        break;

      default:
        LOCAL_NOT_REACHED("unexpected logical operator type");
    }

    RootedValue cb(cx, callbacks[AST_LOGICAL_EXPR]);
    if (!cb.isNull())
        return callback(cb, opName, left, right, pos, dst);

    return newNode(AST_LOGICAL_EXPR, pos,
                   "operator", opName,
                   "left", left,
                   "right", right,
                   dst);
}

bool
NodeBuilder::conditionalExpression(HandleValue test, HandleValue cons, HandleValue alt,
                                   TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_COND_EXPR]);
    if (!cb.isNull())
        return callback(cb, test, cons, alt, pos, dst);

    return newNode(AST_COND_EXPR, pos,
                   "test", test,
                   "consequent", cons,
                   "alternate", alt,
                   dst);
}

bool
NodeBuilder::sequenceExpression(NodeVector& elts, TokenPos* pos, MutableHandleValue dst)
{
    return listNode(AST_LIST_EXPR, "expressions", elts, pos, dst);
}

bool
NodeBuilder::callExpression(HandleValue callee, NodeVector& args, TokenPos* pos,
                            MutableHandleValue dst, bool isOptional)
{
    RootedValue array(cx);
    if (!newArray(args, &array))
        return false;

    RootedValue cb(cx, callbacks[AST_CALL_EXPR]);
    if (!cb.isNull())
        return callback(cb, callee, array, pos, dst);

    return newNode(isOptional ? AST_OPT_CALL_EXPR : AST_CALL_EXPR,
                   pos,
                   "callee",
                   callee,
                   "arguments",
                   array,
                   dst);
}

bool
NodeBuilder::newExpression(HandleValue callee, NodeVector& args, TokenPos* pos,
                           MutableHandleValue dst)
{
    RootedValue array(cx);
    if (!newArray(args, &array))
        return false;

    RootedValue cb(cx, callbacks[AST_NEW_EXPR]);
    if (!cb.isNull())
        return callback(cb, callee, array, pos, dst);

    return newNode(AST_NEW_EXPR, pos,
                   "callee", callee,
                   "arguments", array,
                   dst);
}

bool
NodeBuilder::memberExpression(bool computed, HandleValue expr, HandleValue member, TokenPos* pos,
                              MutableHandleValue dst, bool isOptional)
{
    RootedValue computedVal(cx, BooleanValue(computed));

    RootedValue cb(cx, callbacks[AST_MEMBER_EXPR]);
    if (!cb.isNull())
        return callback(cb, computedVal, expr, member, pos, dst);

    return newNode(isOptional ? AST_OPT_MEMBER_EXPR : AST_MEMBER_EXPR,
                   pos,
                   "object",
                   expr,
                   "property",
                   member,
                   "computed",
                   computedVal,
                   dst);
}

bool
NodeBuilder::arrayExpression(NodeVector& elts, TokenPos* pos, MutableHandleValue dst)
{
    return listNode(AST_ARRAY_EXPR, "elements", elts, pos, dst);
}

bool
NodeBuilder::callSiteObj(NodeVector& raw, NodeVector& cooked, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue rawVal(cx);
    if (!newArray(raw, &rawVal))
        return false;

    RootedValue cookedVal(cx);
    if (!newArray(cooked, &cookedVal))
        return false;

    return newNode(AST_CALL_SITE_OBJ, pos,
                   "raw", rawVal,
                   "cooked", cookedVal,
                    dst);
}

bool
NodeBuilder::taggedTemplate(HandleValue callee, NodeVector& args, TokenPos* pos,
                            MutableHandleValue dst)
{
    RootedValue array(cx);
    if (!newArray(args, &array))
        return false;

    return newNode(AST_TAGGED_TEMPLATE, pos,
                   "callee", callee,
                   "arguments", array,
                   dst);
}

bool
NodeBuilder::templateLiteral(NodeVector& elts, TokenPos* pos, MutableHandleValue dst)
{
    return listNode(AST_TEMPLATE_LITERAL, "elements", elts, pos, dst);
}

bool
NodeBuilder::computedName(HandleValue name, TokenPos* pos, MutableHandleValue dst)
{
    return newNode(AST_COMPUTED_NAME, pos,
                   "name", name,
                   dst);
}

bool
NodeBuilder::spreadExpression(HandleValue expr, TokenPos* pos, MutableHandleValue dst)
{
    return newNode(AST_SPREAD_EXPR, pos,
                   "expression", expr,
                   dst);
}

bool
NodeBuilder::optionalExpression(HandleValue expr, TokenPos* pos, MutableHandleValue dst)
{
  return newNode(AST_OPTIONAL_EXPR, pos,
                 "expression", expr,
                 dst);
}

bool
NodeBuilder::deleteOptionalExpression(HandleValue expr, TokenPos* pos, MutableHandleValue dst)
{
  return newNode(AST_DELETE_OPTIONAL_EXPR, pos,
                 "expression", expr,
                 dst);
}

bool
NodeBuilder::propertyPattern(HandleValue key, HandleValue patt, bool isShorthand, TokenPos* pos,
                             MutableHandleValue dst)
{
    RootedValue kindName(cx);
    if (!atomValue("init", &kindName))
        return false;

    RootedValue isShorthandVal(cx, BooleanValue(isShorthand));

    RootedValue cb(cx, callbacks[AST_PROP_PATT]);
    if (!cb.isNull())
        return callback(cb, key, patt, pos, dst);

    return newNode(AST_PROP_PATT, pos,
                   "key", key,
                   "value", patt,
                   "kind", kindName,
                   "shorthand", isShorthandVal,
                   dst);
}

bool
NodeBuilder::prototypeMutation(HandleValue val, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_PROTOTYPEMUTATION]);
    if (!cb.isNull())
        return callback(cb, val, pos, dst);

    return newNode(AST_PROTOTYPEMUTATION, pos,
                   "value", val,
                   dst);
}

bool
NodeBuilder::propertyInitializer(HandleValue key, HandleValue val, PropKind kind, bool isShorthand,
                                 bool isMethod, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue kindName(cx);
    if (!atomValue(kind == PROP_INIT
                   ? "init"
                   : kind == PROP_GETTER
                   ? "get"
                   : "set", &kindName)) {
        return false;
    }

    RootedValue isShorthandVal(cx, BooleanValue(isShorthand));
    RootedValue isMethodVal(cx, BooleanValue(isMethod));

    RootedValue cb(cx, callbacks[AST_PROPERTY]);
    if (!cb.isNull())
        return callback(cb, kindName, key, val, pos, dst);

    return newNode(AST_PROPERTY, pos,
                   "key", key,
                   "value", val,
                   "kind", kindName,
                   "method", isMethodVal,
                   "shorthand", isShorthandVal,
                   dst);
}

bool
NodeBuilder::objectExpression(NodeVector& elts, TokenPos* pos, MutableHandleValue dst)
{
    return listNode(AST_OBJECT_EXPR, "properties", elts, pos, dst);
}

bool
NodeBuilder::thisExpression(TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_THIS_EXPR]);
    if (!cb.isNull())
        return callback(cb, pos, dst);

    return newNode(AST_THIS_EXPR, pos, dst);
}

bool
NodeBuilder::yieldExpression(HandleValue arg, YieldKind kind, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_YIELD_EXPR]);
    RootedValue delegateVal(cx);

    switch (kind) {
      case Delegating:
        delegateVal = BooleanValue(true);
        break;
      case NotDelegating:
        delegateVal = BooleanValue(false);
        break;
    }

    if (!cb.isNull())
        return callback(cb, opt(arg), delegateVal, pos, dst);
    return newNode(AST_YIELD_EXPR, pos, "argument", arg, "delegate", delegateVal, dst);
}

bool
NodeBuilder::comprehensionBlock(HandleValue patt, HandleValue src, bool isForEach, bool isForOf, TokenPos* pos,
                                MutableHandleValue dst)
{
    RootedValue isForEachVal(cx, BooleanValue(isForEach));
    RootedValue isForOfVal(cx, BooleanValue(isForOf));

    RootedValue cb(cx, callbacks[AST_COMP_BLOCK]);
    if (!cb.isNull())
        return callback(cb, patt, src, isForEachVal, isForOfVal, pos, dst);

    return newNode(AST_COMP_BLOCK, pos,
                   "left", patt,
                   "right", src,
                   "each", isForEachVal,
                   "of", isForOfVal,
                   dst);
}

bool
NodeBuilder::comprehensionIf(HandleValue test, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_COMP_IF]);
    if (!cb.isNull())
        return callback(cb, test, pos, dst);

    return newNode(AST_COMP_IF, pos,
                   "test", test,
                   dst);
}

bool
NodeBuilder::comprehensionExpression(HandleValue body, NodeVector& blocks, HandleValue filter,
                                     bool isLegacy, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue array(cx);
    if (!newArray(blocks, &array))
        return false;

    RootedValue style(cx);
    if (!atomValue(isLegacy ? "legacy" : "modern", &style))
        return false;

    RootedValue cb(cx, callbacks[AST_COMP_EXPR]);
    if (!cb.isNull())
        return callback(cb, body, array, opt(filter), style, pos, dst);

    return newNode(AST_COMP_EXPR, pos,
                   "body", body,
                   "blocks", array,
                   "filter", filter,
                   "style", style,
                   dst);
}

bool
NodeBuilder::generatorExpression(HandleValue body, NodeVector& blocks, HandleValue filter,
                                 bool isLegacy, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue array(cx);
    if (!newArray(blocks, &array))
        return false;

    RootedValue style(cx);
    if (!atomValue(isLegacy ? "legacy" : "modern", &style))
        return false;

    RootedValue cb(cx, callbacks[AST_GENERATOR_EXPR]);
    if (!cb.isNull())
        return callback(cb, body, array, opt(filter), style, pos, dst);

    return newNode(AST_GENERATOR_EXPR, pos,
                   "body", body,
                   "blocks", array,
                   "filter", filter,
                   "style", style,
                   dst);
}

bool
NodeBuilder::importDeclaration(NodeVector& elts, HandleValue moduleSpec, TokenPos* pos,
                               MutableHandleValue dst)
{
    RootedValue array(cx);
    if (!newArray(elts, &array))
        return false;

    RootedValue cb(cx, callbacks[AST_IMPORT_DECL]);
    if (!cb.isNull())
        return callback(cb, array, moduleSpec, pos, dst);

    return newNode(AST_IMPORT_DECL, pos,
                   "specifiers", array,
                   "source", moduleSpec,
                   dst);
}

bool
NodeBuilder::importSpecifier(HandleValue importName, HandleValue bindingName, TokenPos* pos,
                             MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_IMPORT_SPEC]);
    if (!cb.isNull())
        return callback(cb, importName, bindingName, pos, dst);

    return newNode(AST_IMPORT_SPEC, pos,
                   "id", importName,
                   "name", bindingName,
                   dst);
}

bool
NodeBuilder::exportDeclaration(HandleValue decl, NodeVector& elts, HandleValue moduleSpec,
                               HandleValue isDefault, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue array(cx, NullValue());
    if (decl.isNull() && !newArray(elts, &array))
        return false;

    RootedValue cb(cx, callbacks[AST_EXPORT_DECL]);

    if (!cb.isNull())
        return callback(cb, decl, array, moduleSpec, pos, dst);

    return newNode(AST_EXPORT_DECL, pos,
                   "declaration", decl,
                   "specifiers", array,
                   "source", moduleSpec,
                   "isDefault", isDefault,
                   dst);
}

bool
NodeBuilder::exportSpecifier(HandleValue bindingName, HandleValue exportName, TokenPos* pos,
                             MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_EXPORT_SPEC]);
    if (!cb.isNull())
        return callback(cb, bindingName, exportName, pos, dst);

    return newNode(AST_EXPORT_SPEC, pos,
                   "id", bindingName,
                   "name", exportName,
                   dst);
}

bool
NodeBuilder::exportBatchSpecifier(TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_EXPORT_BATCH_SPEC]);
    if (!cb.isNull())
        return callback(cb, pos, dst);

    return newNode(AST_EXPORT_BATCH_SPEC, pos, dst);
}

bool
NodeBuilder::variableDeclaration(NodeVector& elts, VarDeclKind kind, TokenPos* pos,
                                 MutableHandleValue dst)
{
    MOZ_ASSERT(kind > VARDECL_ERR && kind < VARDECL_LIMIT);

    RootedValue array(cx), kindName(cx);
    if (!newArray(elts, &array) ||
        !atomValue(kind == VARDECL_CONST
                   ? "const"
                   : kind == VARDECL_LET
                   ? "let"
                   : "var", &kindName)) {
        return false;
    }

    RootedValue cb(cx, callbacks[AST_VAR_DECL]);
    if (!cb.isNull())
        return callback(cb, kindName, array, pos, dst);

    return newNode(AST_VAR_DECL, pos,
                   "kind", kindName,
                   "declarations", array,
                   dst);
}

bool
NodeBuilder::variableDeclarator(HandleValue id, HandleValue init, TokenPos* pos,
                                MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_VAR_DTOR]);
    if (!cb.isNull())
        return callback(cb, id, opt(init), pos, dst);

    return newNode(AST_VAR_DTOR, pos, "id", id, "init", init, dst);
}

bool
NodeBuilder::switchCase(HandleValue expr, NodeVector& elts, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue array(cx);
    if (!newArray(elts, &array))
        return false;

    RootedValue cb(cx, callbacks[AST_CASE]);
    if (!cb.isNull())
        return callback(cb, opt(expr), array, pos, dst);

    return newNode(AST_CASE, pos,
                   "test", expr,
                   "consequent", array,
                   dst);
}

bool
NodeBuilder::catchClause(HandleValue var, HandleValue guard, HandleValue body, TokenPos* pos,
                         MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_CATCH]);
    if (!cb.isNull())
        return callback(cb, opt(var), opt(guard), body, pos, dst);

    return newNode(AST_CATCH, pos,
                   "param", var,
                   "guard", guard,
                   "body", body,
                   dst);
}

bool
NodeBuilder::literal(HandleValue val, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_LITERAL]);
    if (!cb.isNull())
        return callback(cb, val, pos, dst);

    return newNode(AST_LITERAL, pos, "value", val, dst);
}

bool
NodeBuilder::identifier(HandleValue name, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_IDENTIFIER]);
    if (!cb.isNull())
        return callback(cb, name, pos, dst);

    return newNode(AST_IDENTIFIER, pos, "name", name, dst);
}

bool
NodeBuilder::objectPattern(NodeVector& elts, TokenPos* pos, MutableHandleValue dst)
{
    return listNode(AST_OBJECT_PATT, "properties", elts, pos, dst);
}

bool
NodeBuilder::arrayPattern(NodeVector& elts, TokenPos* pos, MutableHandleValue dst)
{
    return listNode(AST_ARRAY_PATT, "elements", elts, pos, dst);
}

bool
NodeBuilder::function(ASTType type, TokenPos* pos,
                      HandleValue id, NodeVector& args, NodeVector& defaults,
                      HandleValue body, HandleValue rest,
                      GeneratorStyle generatorStyle, bool isAsync, bool isExpression,
                      MutableHandleValue dst)
{
    RootedValue array(cx), defarray(cx);
    if (!newArray(args, &array))
        return false;
    if (!newArray(defaults, &defarray))
        return false;

    bool isGenerator = generatorStyle != GeneratorStyle::None;
    RootedValue isGeneratorVal(cx, BooleanValue(isGenerator));
    RootedValue isAsyncVal(cx, BooleanValue(isAsync));
    RootedValue isExpressionVal(cx, BooleanValue(isExpression));

    RootedValue cb(cx, callbacks[type]);
    if (!cb.isNull()) {
        return callback(cb, opt(id), array, body, isGeneratorVal, isExpressionVal, pos, dst);
    }

    if (isGenerator) {
        // Distinguish ES6 generators from legacy generators.
        RootedValue styleVal(cx);
        JSAtom* styleStr = generatorStyle == GeneratorStyle::ES6
                           ? Atomize(cx, "es6", 3)
                           : Atomize(cx, "legacy", 6);
        if (!styleStr)
            return false;
        styleVal.setString(styleStr);
        return newNode(type, pos,
                       "id", id,
                       "params", array,
                       "defaults", defarray,
                       "body", body,
                       "rest", rest,
                       "generator", isGeneratorVal,
                       "async", isAsyncVal,
                       "style", styleVal,
                       "expression", isExpressionVal,
                       dst);
    }

    return newNode(type, pos,
                   "id", id,
                   "params", array,
                   "defaults", defarray,
                   "body", body,
                   "rest", rest,
                   "generator", isGeneratorVal,
                   "async", isAsyncVal,
                   "expression", isExpressionVal,
                   dst);
}

bool
NodeBuilder::classMethod(HandleValue name, HandleValue body, PropKind kind, bool isStatic,
                         TokenPos* pos, MutableHandleValue dst)
{
    RootedValue kindName(cx);
    if (!atomValue(kind == PROP_INIT
                   ? "method"
                   : kind == PROP_GETTER
                   ? "get"
                   : "set", &kindName)) {
        return false;
    }

    RootedValue isStaticVal(cx, BooleanValue(isStatic));
    RootedValue cb(cx, callbacks[AST_CLASS_METHOD]);
    if (!cb.isNull())
        return callback(cb, kindName, name, body, isStaticVal, pos, dst);

    return newNode(AST_CLASS_METHOD, pos,
                   "name", name,
                   "body", body,
                   "kind", kindName,
                   "static", isStaticVal,
                   dst);
}

bool
NodeBuilder::classField(HandleValue name, HandleValue initializer,
                        TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_CLASS_FIELD]);
    if (!cb.isNull())
        return callback(cb, name, initializer, pos, dst);

    return newNode(AST_CLASS_FIELD, pos,
                   "name", name,
                   "init", initializer,
                   dst);
}

bool
NodeBuilder::staticClassBlock(HandleValue body, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_STATIC_CLASS_BLOCK]);
    if (!cb.isNull())
        return callback(cb, body, pos, dst);

    return newNode(AST_STATIC_CLASS_BLOCK, pos,
                   "body", body,
                   dst);
}

bool
NodeBuilder::classMembers(NodeVector& members, MutableHandleValue dst)
{
    return newArray(members, dst);
}

bool
NodeBuilder::classDefinition(bool expr, HandleValue name, HandleValue heritage, HandleValue block,
                             TokenPos* pos, MutableHandleValue dst)
{
    ASTType type = expr ? AST_CLASS_EXPR : AST_CLASS_STMT;
    RootedValue cb(cx, callbacks[type]);
    if (!cb.isNull())
        return callback(cb, name, heritage, block, pos, dst);

    return newNode(type, pos,
                   "id", name,
                   "superClass", heritage,
                   "body", block,
                   dst);
}

bool
NodeBuilder::metaProperty(HandleValue meta, HandleValue property, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_METAPROPERTY]);
    if (!cb.isNull())
        return callback(cb, meta, property, pos, dst);

    return newNode(AST_METAPROPERTY, pos,
                   "meta", meta,
                   "property", property,
                   dst);
}

bool
NodeBuilder::callImportExpression(HandleValue ident, HandleValue arg, TokenPos* pos,
                                  MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_CALL_IMPORT]);
    if (!cb.isNull())
        return callback(cb, arg, pos, dst);

    return newNode(AST_CALL_IMPORT, pos,
                   "ident", ident,
                   "arg", arg,
                   dst);
}

bool
NodeBuilder::super(TokenPos* pos, MutableHandleValue dst)
{
    RootedValue cb(cx, callbacks[AST_SUPER]);
    if (!cb.isNull())
        return callback(cb, pos, dst);

    return newNode(AST_SUPER, pos, dst);
}

namespace {

/*
 * Serialization of parse nodes to JavaScript objects.
 *
 * All serialization methods take a non-nullable ParseNode pointer.
 */
class ASTSerializer
{
    JSContext*          cx;
    Parser<FullParseHandler>* parser;
    NodeBuilder         builder;
    DebugOnly<uint32_t> lineno;

    Value unrootedAtomContents(JSAtom* atom) {
        return StringValue(atom ? atom : cx->names().empty);
    }

    BinaryOperator binop(ParseNodeKind kind, JSOp op);
    UnaryOperator unop(ParseNodeKind kind, JSOp op);
    AssignmentOperator aop(JSOp op);

    bool statements(ListNode* stmtList, NodeVector& elts);
    bool expressions(ListNode* exprList, NodeVector& elts);
    bool leftAssociate(ListNode* node, MutableHandleValue dst);
    bool rightAssociate(ListNode* node, MutableHandleValue dst);
    bool functionArgs(ParseNode* pn, ListNode* argsList,
                      NodeVector& args, NodeVector& defaults, MutableHandleValue rest);

    bool sourceElement(ParseNode* pn, MutableHandleValue dst);

    bool declaration(ParseNode* pn, MutableHandleValue dst);
    bool variableDeclaration(ListNode* declList, bool lexical, MutableHandleValue dst);
    bool variableDeclarator(ParseNode* pn, MutableHandleValue dst);
    bool importDeclaration(BinaryNode* importNode, MutableHandleValue dst);
    bool importSpecifier(BinaryNode* importSpec, MutableHandleValue dst);
    bool exportDeclaration(ParseNode* exportNode, MutableHandleValue dst);
    bool exportSpecifier(BinaryNode* exportSpec, MutableHandleValue dst);
    bool classDefinition(ClassNode* pn, bool expr, MutableHandleValue dst);

    bool optStatement(ParseNode* pn, MutableHandleValue dst) {
        if (!pn) {
            dst.setMagic(JS_SERIALIZE_NO_NODE);
            return true;
        }
        return statement(pn, dst);
    }

    bool forInit(ParseNode* pn, MutableHandleValue dst);
    bool forIn(ForNode* loop, ParseNode* iterExpr, HandleValue var, HandleValue stmt,
               MutableHandleValue dst);
    bool forOf(ForNode* loop, ParseNode* iterExpr, HandleValue var, HandleValue stmt,
               MutableHandleValue dst);
    bool statement(ParseNode* pn, MutableHandleValue dst);
    bool blockStatement(ListNode* node, MutableHandleValue dst);
    bool switchStatement(SwitchStatement* switchStmt, MutableHandleValue dst);
    bool switchCase(CaseClause* caseClause, MutableHandleValue dst);
    bool tryStatement(TryNode* tryNode, MutableHandleValue dst);
    bool catchClause(TernaryNode* clauseNode, bool* isGuarded, MutableHandleValue dst);

    bool optExpression(ParseNode* pn, MutableHandleValue dst) {
        if (!pn) {
            dst.setMagic(JS_SERIALIZE_NO_NODE);
            return true;
        }
        return expression(pn, dst);
    }

    bool expression(ParseNode* pn, MutableHandleValue dst);

    bool propertyName(ParseNode* key, MutableHandleValue dst);
    bool property(ParseNode* pn, MutableHandleValue dst);

    bool classMethod(ClassMethod* classMethod, MutableHandleValue dst);
    bool classField(ClassField* classField, MutableHandleValue dst);
    bool staticClassBlock(StaticClassBlock* staticClassBlock, MutableHandleValue dst);

    bool optIdentifier(HandleAtom atom, TokenPos* pos, MutableHandleValue dst) {
        if (!atom) {
            dst.setMagic(JS_SERIALIZE_NO_NODE);
            return true;
        }
        return identifier(atom, pos, dst);
    }

    bool identifier(HandleAtom atom, TokenPos* pos, MutableHandleValue dst);
    bool identifier(NameNode* id, MutableHandleValue dst);
    bool literal(ParseNode* pn, MutableHandleValue dst);

    bool optPattern(ParseNode* pn, MutableHandleValue dst) {
        if (!pn) {
            dst.setMagic(JS_SERIALIZE_NO_NODE);
            return true;
        }
        return pattern(pn, dst);
    }

    bool pattern(ParseNode* pn, MutableHandleValue dst);
    bool arrayPattern(ListNode* array, MutableHandleValue dst);
    bool objectPattern(ListNode* obj, MutableHandleValue dst);

    bool function(FunctionNode* funNode, ASTType type, MutableHandleValue dst);
    bool functionArgsAndBody(ParseNode* pn, NodeVector& args, NodeVector& defaults,
                             bool isAsync, bool isExpression,
                             MutableHandleValue body, MutableHandleValue rest);
    bool functionBody(ParseNode* pn, TokenPos* pos, MutableHandleValue dst);

    bool comprehensionBlock(ForNode* forNode, MutableHandleValue dst);
    bool comprehensionIf(TernaryNode* ifNode, MutableHandleValue dst);
    bool comprehension(ParseNode* pn, MutableHandleValue dst);
    bool generatorExpression(ParseNode* pn, MutableHandleValue dst);

  public:
    ASTSerializer(JSContext* c, bool l, char const* src, uint32_t ln)
        : cx(c)
        , parser(nullptr)
        , builder(c, l, src)
#ifdef DEBUG
        , lineno(ln)
#endif
    {}

    bool init(HandleObject userobj) {
        return builder.init(userobj);
    }

    void setParser(Parser<FullParseHandler>* p) {
        parser = p;
        builder.setTokenStream(&p->tokenStream);
    }

    bool program(ListNode* node, MutableHandleValue dst);
};

} /* anonymous namespace */

AssignmentOperator
ASTSerializer::aop(JSOp op)
{
    switch (op) {
      case JSOP_NOP:
        return AOP_ASSIGN;
      case JSOP_ADD:
        return AOP_PLUS;
      case JSOP_SUB:
        return AOP_MINUS;
      case JSOP_MUL:
        return AOP_STAR;
      case JSOP_DIV:
        return AOP_DIV;
      case JSOP_MOD:
        return AOP_MOD;
      case JSOP_POW:
        return AOP_POW;
      case JSOP_LSH:
        return AOP_LSH;
      case JSOP_RSH:
        return AOP_RSH;
      case JSOP_URSH:
        return AOP_URSH;
      case JSOP_BITOR:
        return AOP_BITOR;
      case JSOP_BITXOR:
        return AOP_BITXOR;
      case JSOP_BITAND:
        return AOP_BITAND;
      case JSOP_COALESCE:
        return AOP_COALESCE;
      case JSOP_OR:
        return AOP_OR;
      case JSOP_AND:
        return AOP_AND;
      default:
        return AOP_ERR;
    }
}

UnaryOperator
ASTSerializer::unop(ParseNodeKind kind, JSOp op)
{
    if (IsDeleteKind(kind))
        return UNOP_DELETE;

    if (IsTypeofKind(kind))
        return UNOP_TYPEOF;

    if (kind == PNK_AWAIT)
        return UNOP_AWAIT;

    switch (op) {
      case JSOP_NEG:
        return UNOP_NEG;
      case JSOP_POS:
        return UNOP_POS;
      case JSOP_NOT:
        return UNOP_NOT;
      case JSOP_BITNOT:
        return UNOP_BITNOT;
      case JSOP_VOID:
        return UNOP_VOID;
      default:
        return UNOP_ERR;
    }
}

BinaryOperator
ASTSerializer::binop(ParseNodeKind kind, JSOp op)
{
    switch (kind) {
      case PNK_LSH:
        return BINOP_LSH;
      case PNK_RSH:
        return BINOP_RSH;
      case PNK_URSH:
        return BINOP_URSH;
      case PNK_LT:
        return BINOP_LT;
      case PNK_LE:
        return BINOP_LE;
      case PNK_GT:
        return BINOP_GT;
      case PNK_GE:
        return BINOP_GE;
      case PNK_EQ:
        return BINOP_EQ;
      case PNK_NE:
        return BINOP_NE;
      case PNK_STRICTEQ:
        return BINOP_STRICTEQ;
      case PNK_STRICTNE:
        return BINOP_STRICTNE;
      case PNK_ADD:
        return BINOP_ADD;
      case PNK_SUB:
        return BINOP_SUB;
      case PNK_STAR:
        return BINOP_STAR;
      case PNK_DIV:
        return BINOP_DIV;
      case PNK_MOD:
        return BINOP_MOD;
      case PNK_POW:
        return BINOP_POW;
      case PNK_BITOR:
        return BINOP_BITOR;
      case PNK_BITXOR:
        return BINOP_BITXOR;
      case PNK_BITAND:
        return BINOP_BITAND;
      case PNK_IN:
        return BINOP_IN;
      case PNK_INSTANCEOF:
        return BINOP_INSTANCEOF;
      case PNK_COALESCE:
        return BINOP_COALESCE;
      default:
        return BINOP_ERR;
    }
}

bool
ASTSerializer::statements(ListNode* stmtList, NodeVector& elts)
{
    MOZ_ASSERT(stmtList->isKind(PNK_STATEMENTLIST));

    if (!elts.reserve(stmtList->count()))
        return false;

    for (ParseNode* stmt : stmtList->contents()) {
        MOZ_ASSERT(stmtList->pn_pos.encloses(stmt->pn_pos));

        RootedValue elt(cx);
        if (!sourceElement(stmt, &elt))
            return false;
        elts.infallibleAppend(elt);
    }

    return true;
}

bool
ASTSerializer::expressions(ListNode* exprList, NodeVector& elts)
{
    if (!elts.reserve(exprList->count()))
        return false;

    for (ParseNode* expr : exprList->contents()) {
        MOZ_ASSERT(exprList->pn_pos.encloses(expr->pn_pos));

        RootedValue elt(cx);
        if (!expression(expr, &elt))
            return false;
        elts.infallibleAppend(elt);
    }

    return true;
}

bool
ASTSerializer::blockStatement(ListNode* node, MutableHandleValue dst)
{
    MOZ_ASSERT(node->isKind(PNK_STATEMENTLIST));

    NodeVector stmts(cx);
    return statements(node, stmts) &&
           builder.blockStatement(stmts, &node->pn_pos, dst);
}

bool
ASTSerializer::program(ListNode* node, MutableHandleValue dst)
{
    MOZ_ASSERT(parser->tokenStream.srcCoords.lineNum(node->pn_pos.begin) == lineno);

    NodeVector stmts(cx);
    return statements(node, stmts) &&
           builder.program(stmts, &node->pn_pos, dst);
}

bool
ASTSerializer::sourceElement(ParseNode* pn, MutableHandleValue dst)
{
    /* SpiderMonkey allows declarations even in pure statement contexts. */
    return statement(pn, dst);
}

bool
ASTSerializer::declaration(ParseNode* pn, MutableHandleValue dst)
{
    MOZ_ASSERT(pn->isKind(PNK_FUNCTION) ||
               pn->isKind(PNK_VAR) ||
               pn->isKind(PNK_LET) ||
               pn->isKind(PNK_CONST));

    switch (pn->getKind()) {
      case PNK_FUNCTION:
        return function(&pn->as<FunctionNode>(), AST_FUNC_DECL, dst);

      case PNK_VAR:
        return variableDeclaration(&pn->as<ListNode>(), false, dst);

      default:
        MOZ_ASSERT(pn->isKind(PNK_LET) || pn->isKind(PNK_CONST));
        return variableDeclaration(&pn->as<ListNode>(), true, dst);
    }
}

bool
ASTSerializer::variableDeclaration(ListNode* declList, bool lexical, MutableHandleValue dst)
{
    MOZ_ASSERT_IF(lexical, declList->isKind(PNK_LET) || declList->isKind(PNK_CONST));
    MOZ_ASSERT_IF(!lexical, declList->isKind(PNK_VAR));

    VarDeclKind kind = VARDECL_ERR;
    // Treat both the toplevel const binding (secretly var-like) and the lexical const
    // the same way
    if (lexical)
        kind = declList->isKind(PNK_LET) ? VARDECL_LET : VARDECL_CONST;
    else
        kind = declList->isKind(PNK_VAR) ? VARDECL_VAR : VARDECL_CONST;

    NodeVector dtors(cx);
    if (!dtors.reserve(declList->count()))
        return false;
    for (ParseNode* decl : declList->contents()) {
        RootedValue child(cx);
        if (!variableDeclarator(decl, &child))
            return false;
        dtors.infallibleAppend(child);
    }
    return builder.variableDeclaration(dtors, kind, &declList->pn_pos, dst);
}

bool
ASTSerializer::variableDeclarator(ParseNode* pn, MutableHandleValue dst)
{
    ParseNode* patternNode;
    ParseNode* initNode;

    if (pn->isKind(PNK_NAME)) {
        patternNode = pn;
        initNode = pn->as<NameNode>().initializer();
        MOZ_ASSERT_IF(initNode, pn->pn_pos.encloses(initNode->pn_pos));
    } else if (pn->isKind(PNK_ASSIGN)) {
        AssignmentNode* assignNode = &pn->as<AssignmentNode>();
        patternNode = assignNode->left();
        initNode = assignNode->right();
        MOZ_ASSERT(pn->pn_pos.encloses(patternNode->pn_pos));
        MOZ_ASSERT(pn->pn_pos.encloses(initNode->pn_pos));
    } else {
        /* This happens for a destructuring declarator in a for-in/of loop. */
        patternNode = pn;
        initNode = nullptr;
    }

    RootedValue patternVal(cx), init(cx);
    return pattern(patternNode, &patternVal) &&
           optExpression(initNode, &init) &&
           builder.variableDeclarator(patternVal, init, &pn->pn_pos, dst);
}

bool
ASTSerializer::importDeclaration(BinaryNode* importNode, MutableHandleValue dst)
{
    MOZ_ASSERT(importNode->isKind(PNK_IMPORT));

    ListNode* specList = &importNode->left()->as<ListNode>();
    MOZ_ASSERT(specList->isKind(PNK_IMPORT_SPEC_LIST));

    ParseNode* moduleSpecNode = importNode->right();
    MOZ_ASSERT(moduleSpecNode->isKind(PNK_STRING));

    NodeVector elts(cx);
    if (!elts.reserve(specList->count()))
        return false;

    for (ParseNode* item : specList->contents()) {
        BinaryNode* spec = &item->as<BinaryNode>();
        RootedValue elt(cx);
        if (!importSpecifier(spec, &elt))
            return false;
        elts.infallibleAppend(elt);
    }

    RootedValue moduleSpec(cx);
    return literal(moduleSpecNode, &moduleSpec) &&
           builder.importDeclaration(elts, moduleSpec, &importNode->pn_pos, dst);
}

bool
ASTSerializer::importSpecifier(BinaryNode* importSpec, MutableHandleValue dst)
{
    MOZ_ASSERT(importSpec->isKind(PNK_IMPORT_SPEC));
    NameNode* importNameNode = &importSpec->left()->as<NameNode>();
    NameNode* bindingNameNode = &importSpec->right()->as<NameNode>();

    RootedValue importName(cx);
    RootedValue bindingName(cx);
    return identifier(importNameNode, &importName) &&
           identifier(bindingNameNode, &bindingName) &&
           builder.importSpecifier(importName, bindingName, &importSpec->pn_pos, dst);
}

bool
ASTSerializer::exportDeclaration(ParseNode* exportNode, MutableHandleValue dst)
{
    MOZ_ASSERT(exportNode->isKind(PNK_EXPORT) ||
               exportNode->isKind(PNK_EXPORT_FROM) ||
               exportNode->isKind(PNK_EXPORT_DEFAULT));
    MOZ_ASSERT_IF(exportNode->isKind(PNK_EXPORT), exportNode->is<UnaryNode>());
    MOZ_ASSERT_IF(exportNode->isKind(PNK_EXPORT_FROM),
                  exportNode->as<BinaryNode>().right()->isKind(PNK_STRING));

    RootedValue decl(cx, NullValue());
    NodeVector elts(cx);

    ParseNode* kid = exportNode->isKind(PNK_EXPORT)
                     ? exportNode->as<UnaryNode>().kid()
                     : exportNode->as<BinaryNode>().left();
    switch (ParseNodeKind kind = kid->getKind()) {
      case PNK_EXPORT_SPEC_LIST: {
        ListNode* specList = &kid->as<ListNode>();
        if (!elts.reserve(specList->count()))
            return false;

        for (ParseNode* spec : specList->contents()) {
            RootedValue elt(cx);
            if (spec->isKind(PNK_EXPORT_SPEC)) {
                if (!exportSpecifier(&spec->as<BinaryNode>(), &elt))
                    return false;
            } else {
                if (!builder.exportBatchSpecifier(&exportNode->pn_pos, &elt))
                    return false;
            }
            elts.infallibleAppend(elt);
        }
        break;
      }

      case PNK_FUNCTION:
        if (!function(&kid->as<FunctionNode>(), AST_FUNC_DECL, &decl))
            return false;
        break;

      case PNK_CLASS:
        if (!classDefinition(&kid->as<ClassNode>(), false, &decl))
            return false;
        break;

      case PNK_VAR:
      case PNK_CONST:
      case PNK_LET:
        if (!variableDeclaration(&kid->as<ListNode>(), kind != PNK_VAR, &decl))
            return false;
        break;

      default:
          if (!expression(kid, &decl))
              return false;
          break;
    }

    RootedValue moduleSpec(cx, NullValue());
    if (exportNode->isKind(PNK_EXPORT_FROM)) {
        if (!literal(exportNode->as<BinaryNode>().right(), &moduleSpec)) {
            return false;
        }
    }

    RootedValue isDefault(cx, BooleanValue(false));
    if (exportNode->isKind(PNK_EXPORT_DEFAULT))
        isDefault.setBoolean(true);

    return builder.exportDeclaration(decl, elts, moduleSpec, isDefault, &exportNode->pn_pos, dst);
}

bool
ASTSerializer::exportSpecifier(BinaryNode* exportSpec, MutableHandleValue dst)
{
    MOZ_ASSERT(exportSpec->isKind(PNK_EXPORT_SPEC));
    NameNode* bindingNameNode = &exportSpec->left()->as<NameNode>();
    NameNode* exportNameNode = &exportSpec->right()->as<NameNode>();

    RootedValue bindingName(cx);
    RootedValue exportName(cx);
    return identifier(bindingNameNode, &bindingName) &&
           identifier(exportNameNode, &exportName) &&
           builder.exportSpecifier(bindingName, exportName, &exportSpec->pn_pos, dst);
}

bool
ASTSerializer::switchCase(CaseClause* caseClause, MutableHandleValue dst)
{
    MOZ_ASSERT_IF(caseClause->caseExpression(),
                  caseClause->pn_pos.encloses(caseClause->caseExpression()->pn_pos));
    MOZ_ASSERT(caseClause->pn_pos.encloses(caseClause->statementList()->pn_pos));

    NodeVector stmts(cx);
    RootedValue expr(cx);
    return optExpression(caseClause->caseExpression(), &expr) &&
           statements(caseClause->statementList(), stmts) &&
           builder.switchCase(expr, stmts, &caseClause->pn_pos, dst);
}

bool
ASTSerializer::switchStatement(SwitchStatement* switchStmt, MutableHandleValue dst)
{
    MOZ_ASSERT(switchStmt->pn_pos.encloses(switchStmt->discriminant().pn_pos));
    MOZ_ASSERT(switchStmt->pn_pos.encloses(switchStmt->lexicalForCaseList().pn_pos));

    RootedValue disc(cx);

    if (!expression(&switchStmt->discriminant(), &disc))
        return false;


    ListNode* caseList = &switchStmt->lexicalForCaseList().scopeBody()->as<ListNode>();

    NodeVector cases(cx);
    if (!cases.reserve(caseList->count()))
        return false;

    for (ParseNode* item : caseList->contents()) {
        CaseClause* caseClause = &item->as<CaseClause>();
        RootedValue child(cx);
        if (!switchCase(caseClause, &child))
            return false;
        cases.infallibleAppend(child);
    }

    return builder.switchStatement(disc, cases, /* lexical = */ true, &switchStmt->pn_pos, dst);
}

bool
ASTSerializer::catchClause(TernaryNode* clauseNode, bool* isGuarded, MutableHandleValue dst)
{
    ParseNode* patNode = clauseNode->kid1();
    MOZ_ASSERT_IF(patNode, clauseNode->pn_pos.encloses(patNode->pn_pos));

    ParseNode* condNode = clauseNode->kid2();
    MOZ_ASSERT_IF(condNode, clauseNode->pn_pos.encloses(condNode->pn_pos));

    ParseNode* bodyNode = clauseNode->kid3();
    MOZ_ASSERT(clauseNode->pn_pos.encloses(bodyNode->pn_pos));

    RootedValue var(cx), guard(cx), body(cx);

    if (!optPattern(patNode, &var) ||
        !optExpression(condNode, &guard)) {
        return false;
    }

    *isGuarded = !guard.isMagic(JS_SERIALIZE_NO_NODE);

    return statement(bodyNode, &body) &&
           builder.catchClause(var, guard, body, &clauseNode->pn_pos, dst);
}

bool
ASTSerializer::tryStatement(TryNode* tryNode, MutableHandleValue dst)
{
    ParseNode* bodyNode = tryNode->body();
    MOZ_ASSERT(tryNode->pn_pos.encloses(bodyNode->pn_pos));

    ListNode* catchList = tryNode->catchList();
    MOZ_ASSERT_IF(catchList, tryNode->pn_pos.encloses(catchList->pn_pos));

    ParseNode* finallyNode = tryNode->finallyBlock();
    MOZ_ASSERT_IF(finallyNode, tryNode->pn_pos.encloses(finallyNode->pn_pos));

    RootedValue body(cx);
    if (!statement(bodyNode, &body))
        return false;

    NodeVector guarded(cx);
    RootedValue unguarded(cx, NullValue());

    if (catchList) {
        if (!guarded.reserve(catchList->count()))
            return false;

        for (ParseNode* catchNode : catchList->contents()) {
            LexicalScopeNode* catchScope = &catchNode->as<LexicalScopeNode>();
            RootedValue clause(cx);
            bool isGuarded;
            if (!catchClause(&catchScope->scopeBody()->as<TernaryNode>(), &isGuarded, &clause))
                return false;
            if (isGuarded)
                guarded.infallibleAppend(clause);
            else
                unguarded = clause;
        }
    }

    RootedValue finally(cx);
    return optStatement(finallyNode, &finally) &&
           builder.tryStatement(body, guarded, unguarded, finally, &tryNode->pn_pos, dst);
}

bool
ASTSerializer::forInit(ParseNode* pn, MutableHandleValue dst)
{
    if (!pn) {
        dst.setMagic(JS_SERIALIZE_NO_NODE);
        return true;
    }

    bool lexical = pn->isKind(PNK_LET) || pn->isKind(PNK_CONST);
    return (lexical || pn->isKind(PNK_VAR))
           ? variableDeclaration(&pn->as<ListNode>(), lexical, dst)
           : expression(pn, dst);
}

bool
ASTSerializer::forOf(ForNode* loop, ParseNode* iterExpr, HandleValue var, HandleValue stmt,
                     MutableHandleValue dst)
{
    RootedValue expr(cx);

    return expression(iterExpr, &expr) &&
        builder.forOfStatement(var, expr, stmt, &loop->pn_pos, dst);
}

bool
ASTSerializer::forIn(ForNode* loop, ParseNode* iterExpr, HandleValue var, HandleValue stmt,
                     MutableHandleValue dst)
{
    RootedValue expr(cx);
    bool isForEach = loop->iflags() & JSITER_FOREACH;

    return expression(iterExpr, &expr) &&
        builder.forInStatement(var, expr, stmt, isForEach, &loop->pn_pos, dst);
}

bool
ASTSerializer::classDefinition(ClassNode* pn, bool expr, MutableHandleValue dst)
{
    RootedValue className(cx, MagicValue(JS_SERIALIZE_NO_NODE));
    RootedValue heritage(cx);
    RootedValue classBody(cx);

    if (ClassNames* names = pn->names()) {
        if (!identifier(names->innerBinding(), &className))
            return false;
    }

    return optExpression(pn->heritage(), &heritage) &&
           statement(pn->memberList(), &classBody) &&
           builder.classDefinition(expr, className, heritage, classBody, &pn->pn_pos, dst);
}

bool
ASTSerializer::statement(ParseNode* pn, MutableHandleValue dst)
{
    JS_CHECK_RECURSION(cx, return false);
    switch (pn->getKind()) {
      case PNK_FUNCTION:
      case PNK_VAR:
        return declaration(pn, dst);

      case PNK_LET:
      case PNK_CONST:
        return declaration(pn, dst);

      case PNK_IMPORT:
        return importDeclaration(&pn->as<BinaryNode>(), dst);

      case PNK_EXPORT:
      case PNK_EXPORT_DEFAULT:
      case PNK_EXPORT_FROM:
        return exportDeclaration(pn, dst);

      case PNK_SEMI:
        if (pn->as<UnaryNode>().kid()) {
            RootedValue expr(cx);
            return expression(pn->as<UnaryNode>().kid(), &expr) &&
                   builder.expressionStatement(expr, &pn->pn_pos, dst);
        }
        return builder.emptyStatement(&pn->pn_pos, dst);

      case PNK_LEXICALSCOPE:
        pn = pn->as<LexicalScopeNode>().scopeBody();
        if (!pn->isKind(PNK_STATEMENTLIST))
            return statement(pn, dst);
        [[fallthrough]];

      case PNK_STATEMENTLIST:
        return blockStatement(&pn->as<ListNode>(), dst);

      case PNK_IF:
      {
        TernaryNode* ifNode = &pn->as<TernaryNode>();

        ParseNode* testNode = ifNode->kid1();
        MOZ_ASSERT(ifNode->pn_pos.encloses(testNode->pn_pos));

        ParseNode* consNode = ifNode->kid2();
        MOZ_ASSERT(ifNode->pn_pos.encloses(consNode->pn_pos));

        ParseNode* altNode = ifNode->kid3();
        MOZ_ASSERT_IF(altNode, ifNode->pn_pos.encloses(altNode->pn_pos));

        RootedValue test(cx), cons(cx), alt(cx);

        return expression(testNode, &test) &&
               statement(consNode, &cons) &&
               optStatement(altNode, &alt) &&
               builder.ifStatement(test, cons, alt, &ifNode->pn_pos, dst);
      }

      case PNK_SWITCH:
        return switchStatement(&pn->as<SwitchStatement>(), dst);

      case PNK_TRY:
        return tryStatement(&pn->as<TryNode>(), dst);

      case PNK_WITH:
      case PNK_WHILE:
      {
        BinaryNode* node = &pn->as<BinaryNode>();

        ParseNode* exprNode = node->left();
        MOZ_ASSERT(node->pn_pos.encloses(exprNode->pn_pos));

        ParseNode* stmtNode = node->right();
        MOZ_ASSERT(node->pn_pos.encloses(stmtNode->pn_pos));

        RootedValue expr(cx), stmt(cx);

        return expression(exprNode, &expr) &&
               statement(stmtNode, &stmt) &&
               (node->isKind(PNK_WITH)
                ? builder.withStatement(expr, stmt, &node->pn_pos, dst)
                : builder.whileStatement(expr, stmt, &node->pn_pos, dst));
      }

      case PNK_DOWHILE:
      {
        BinaryNode* node = &pn->as<BinaryNode>();

        ParseNode* stmtNode = node->left();
        MOZ_ASSERT(node->pn_pos.encloses(stmtNode->pn_pos));

        ParseNode* testNode = node->right();
        MOZ_ASSERT(node->pn_pos.encloses(testNode->pn_pos));

        RootedValue stmt(cx), test(cx);

        return statement(stmtNode, &stmt) &&
               expression(testNode, &test) &&
               builder.doWhileStatement(stmt, test, &node->pn_pos, dst);
      }

      case PNK_FOR:
      case PNK_COMPREHENSIONFOR:
      {
        ForNode* forNode = &pn->as<ForNode>();

        TernaryNode* head = forNode->head();
        MOZ_ASSERT(forNode->pn_pos.encloses(head->pn_pos));

        ParseNode* stmtNode = forNode->right();
        MOZ_ASSERT(forNode->pn_pos.encloses(stmtNode->pn_pos));

        ParseNode* initNode = head->kid1();
        MOZ_ASSERT_IF(initNode, head->pn_pos.encloses(initNode->pn_pos));

        ParseNode* maybeTest = head->kid2();
        MOZ_ASSERT_IF(maybeTest, head->pn_pos.encloses(maybeTest->pn_pos));

        ParseNode* updateOrIter = head->kid3();
        MOZ_ASSERT_IF(updateOrIter, head->pn_pos.encloses(updateOrIter->pn_pos));

        RootedValue stmt(cx);
        if (!statement(stmtNode, &stmt))
            return false;

        if (head->isKind(PNK_FORIN) || head->isKind(PNK_FOROF)) {
            RootedValue var(cx);
            if (initNode->is<LexicalScopeNode>()) {
                LexicalScopeNode* scopeNode = &initNode->as<LexicalScopeNode>();
                if (!variableDeclaration(&scopeNode->scopeBody()->as<ListNode>(), true, &var))
                    return false;
            } else if (!initNode->isKind(PNK_VAR) &&
                       !initNode->isKind(PNK_LET) &&
                       !initNode->isKind(PNK_CONST))
            {
                if (!pattern(initNode, &var))
                    return false;
            } else {
                if (!variableDeclaration(&initNode->as<ListNode>(),
                                         initNode->isKind(PNK_LET) ||
                                         initNode->isKind(PNK_CONST),
                                         &var))
                {
                    return false;
                }
            }
            if (head->isKind(PNK_FORIN))
                return forIn(forNode, updateOrIter, var, stmt, dst);
            return forOf(forNode, updateOrIter, var, stmt, dst);
        }

        RootedValue init(cx), test(cx), update(cx);

        return forInit(initNode, &init) &&
               optExpression(maybeTest, &test) &&
               optExpression(updateOrIter, &update) &&
               builder.forStatement(init, test, update, stmt, &forNode->pn_pos, dst);
      }

      case PNK_BREAK:
      case PNK_CONTINUE:
      {
        LoopControlStatement* node = &pn->as<LoopControlStatement>();
        RootedValue label(cx);
        RootedAtom pnAtom(cx, node->label());
        return optIdentifier(pnAtom, nullptr, &label) &&
               (node->isKind(PNK_BREAK)
                ? builder.breakStatement(label, &node->pn_pos, dst)
                : builder.continueStatement(label, &node->pn_pos, dst));
      }

      case PNK_LABEL:
      {
        LabeledStatement* labelNode = &pn->as<LabeledStatement>();
        ParseNode* stmtNode = labelNode->statement();
        MOZ_ASSERT(labelNode->pn_pos.encloses(stmtNode->pn_pos));

        RootedValue label(cx), stmt(cx);
        RootedAtom pnAtom(cx, labelNode->label());
        return identifier(pnAtom, nullptr, &label) &&
               statement(stmtNode, &stmt) &&
               builder.labeledStatement(label, stmt, &labelNode->pn_pos, dst);
      }

      case PNK_THROW:
      {
        UnaryNode* throwNode = &pn->as<UnaryNode>();
        ParseNode* operand = throwNode->kid();
        MOZ_ASSERT(throwNode->pn_pos.encloses(operand->pn_pos));

        RootedValue arg(cx);

        return expression(operand, &arg) &&
               builder.throwStatement(arg, &throwNode->pn_pos, dst);
      }

      case PNK_RETURN:
      {
        UnaryNode* returnNode = &pn->as<UnaryNode>();
        ParseNode* operand = returnNode->kid();
        MOZ_ASSERT_IF(operand, returnNode->pn_pos.encloses(operand->pn_pos));

        RootedValue arg(cx);

        return optExpression(operand, &arg) &&
               builder.returnStatement(arg, &returnNode->pn_pos, dst);
      }

      case PNK_DEBUGGER:
        return builder.debuggerStatement(&pn->pn_pos, dst);

      case PNK_CLASS:
        return classDefinition(&pn->as<ClassNode>(), false, dst);

      case PNK_CLASSMEMBERLIST:
      {
        ListNode* memberList = &pn->as<ListNode>();
        NodeVector members(cx);
        if (!members.reserve(memberList->count()))
            return false;

        for (ParseNode* item : memberList->contents()) {
            if (item->is<LexicalScopeNode>())
                item = item->as<LexicalScopeNode>().scopeBody();
            if (item->is<ClassField>()) {
                ClassField* field = &item->as<ClassField>();
                MOZ_ASSERT(memberList->pn_pos.encloses(field->pn_pos));

                RootedValue prop(cx);
                if (!classField(field, &prop))
                    return false;
                members.infallibleAppend(prop);
            } else if (item->is<StaticClassBlock>()) {
                StaticClassBlock* scb = &item->as<StaticClassBlock>();
                MOZ_ASSERT(memberList->pn_pos.encloses(scb->pn_pos));
                RootedValue prop(cx);
                if (!staticClassBlock(scb, &prop))
                    return false;
                members.infallibleAppend(prop);
            } else {
                ClassMethod* method = &item->as<ClassMethod>();
                MOZ_ASSERT(memberList->pn_pos.encloses(method->pn_pos));

                RootedValue prop(cx);
                if (!classMethod(method, &prop))
                    return false;
                members.infallibleAppend(prop);
            }
        }

        return builder.classMembers(members, dst);
      }

      case PNK_NOP:
        return builder.emptyStatement(&pn->pn_pos, dst);

      default:
        LOCAL_NOT_REACHED("unexpected statement type");
    }
}

bool
ASTSerializer::classMethod(ClassMethod* classMethod, MutableHandleValue dst)
{
    PropKind kind;
    switch (classMethod->getOp()) {
      case JSOP_INITPROP:
        kind = PROP_INIT;
        break;

      case JSOP_INITPROP_GETTER:
        kind = PROP_GETTER;
        break;

      case JSOP_INITPROP_SETTER:
        kind = PROP_SETTER;
        break;

      default:
        LOCAL_NOT_REACHED("unexpected object-literal property");
    }

    RootedValue key(cx), val(cx);
    bool isStatic = classMethod->isStatic();
    return propertyName(&classMethod->name(), &key) &&
           expression(&classMethod->method(), &val) &&
           builder.classMethod(key, val, kind, isStatic, &classMethod->pn_pos, dst);
}

bool
ASTSerializer::classField(ClassField* classField, MutableHandleValue dst)
{
    RootedValue key(cx), val(cx);
    // Dig through the lambda and get to the actual expression
    ParseNode* value = classField->initializer()
                           ->body()
                           ->head()->as<LexicalScopeNode>()
                           .scopeBody()->as<ListNode>()
                           .head()->as<UnaryNode>()
                           .kid()->as<BinaryNode>()
                           .right();
    // RawUndefinedExpr is the node we use for "there is no initializer". If one
    // writes, literally, `x = undefined;`, it will not be a RawUndefinedExpr
    // node, but rather a variable reference.
    // Behavior for "there is no initializer" should be { ..., "init": null }
    if (value->getKind() != PNK_RAW_UNDEFINED) {
        if (!expression(value, &val))
          return false;
    } else {
        val.setNull();
    }
    return propertyName(&classField->name(), &key) &&
           builder.classField(key, val, &classField->pn_pos, dst);
}

bool
ASTSerializer::staticClassBlock(StaticClassBlock* staticClassBlock, MutableHandleValue dst)
{
    FunctionNode* fun = staticClassBlock->function();

    NodeVector args(cx);
    NodeVector defaults(cx);

    RootedValue body(cx), rest(cx);
    rest.setNull();
    return functionArgsAndBody(fun->body(), args, defaults, false, false,
                               &body, &rest) &&
           builder.staticClassBlock(body, &staticClassBlock->pn_pos, dst);
}

bool
ASTSerializer::leftAssociate(ListNode* node, MutableHandleValue dst)
{
    MOZ_ASSERT(!node->empty());

    ParseNodeKind kind = node->getKind();
    bool logop = (kind == PNK_COALESCE ||
                  kind == PNK_OR ||
                  kind == PNK_AND);

    ParseNode* head = node->head();
    RootedValue left(cx);
    if (!expression(head, &left))
        return false;
    for (ParseNode* next : node->contentsFrom(head->pn_next)) {
        RootedValue right(cx);
        if (!expression(next, &right))
            return false;

        TokenPos subpos(node->pn_pos.begin, next->pn_pos.end);

        if (logop) {
            if (!builder.logicalExpression(kind, left, right, &subpos, &left))
                return false;
        } else {
            BinaryOperator op = binop(node->getKind(), node->getOp());
            LOCAL_ASSERT(op > BINOP_ERR && op < BINOP_LIMIT);

            if (!builder.binaryExpression(op, left, right, &subpos, &left))
                return false;
        }
    }

    dst.set(left);
    return true;
}

bool
ASTSerializer::rightAssociate(ListNode* node, MutableHandleValue dst)
{
    MOZ_ASSERT(!node->empty());

    // First, we need to reverse the list, so that we can traverse it in the right order.
    // It's OK to destructively reverse the list, because there are no other consumers.

    ParseNode* head = node->head();
    ParseNode* prev = nullptr;
    ParseNode* current = head;
    ParseNode* next;
    while (current != nullptr) {
        next = current->pn_next;
        current->pn_next = prev;
        prev = current;
        current = next;
    }

    head = prev;

    RootedValue right(cx);
    if (!expression(head, &right))
        return false;
    for (ParseNode* next = head->pn_next; next; next = next->pn_next) {
        RootedValue left(cx);
        if (!expression(next, &left))
            return false;

        TokenPos subpos(node->pn_pos.begin, next->pn_pos.end);

        BinaryOperator op = binop(node->getKind(), node->getOp());
        LOCAL_ASSERT(op > BINOP_ERR && op < BINOP_LIMIT);

        if (!builder.binaryExpression(op, left, right, &subpos, &right))
            return false;
    }

    dst.set(right);
    return true;
}

bool
ASTSerializer::comprehensionBlock(ForNode* forNode, MutableHandleValue dst)
{
    TernaryNode* in = forNode->head();

    LOCAL_ASSERT(in && (in->isKind(PNK_FORIN) || in->isKind(PNK_FOROF)));

    bool isForEach = in->isKind(PNK_FORIN) && (forNode->iflags() & JSITER_FOREACH);
    bool isForOf = in->isKind(PNK_FOROF);

    ListNode* decl;
    if (in->kid1()->is<LexicalScopeNode>())
        decl = &in->kid1()->as<LexicalScopeNode>().scopeBody()->as<ListNode>();
    else
        decl = &in->kid1()->as<ListNode>();
    MOZ_ASSERT(decl->count() == 1);

    RootedValue patt(cx), src(cx);
    return pattern(decl->head(), &patt) &&
           expression(in->kid3(), &src) &&
           builder.comprehensionBlock(patt, src, isForEach, isForOf, &in->pn_pos, dst);
}

bool
ASTSerializer::comprehensionIf(TernaryNode* ifNode, MutableHandleValue dst)
{
    LOCAL_ASSERT(ifNode->isKind(PNK_IF));
    LOCAL_ASSERT(!ifNode->kid3());

    RootedValue patt(cx);
    return pattern(ifNode->kid1(), &patt) &&
           builder.comprehensionIf(patt, &ifNode->pn_pos, dst);
}

bool
ASTSerializer::comprehension(ParseNode* pn, MutableHandleValue dst)
{
    // There are two array comprehension flavors.
    // 1. The kind that was in ES4 for a while: [z for (x in y)]
    // 2. The kind that was in ES6 for a while: [for (x of y) z]
    // They have slightly different parse trees and scoping.
    bool isLegacy = pn->is<LexicalScopeNode>();
    ParseNode* next = isLegacy ? pn->as<LexicalScopeNode>().scopeBody() : pn;
    LOCAL_ASSERT(next->isKind(PNK_COMPREHENSIONFOR));

    NodeVector blocks(cx);
    RootedValue filter(cx, MagicValue(JS_SERIALIZE_NO_NODE));
    while (true) {
        if (next->isKind(PNK_COMPREHENSIONFOR)) {
            ForNode* forNode = &next->as<ForNode>();
            RootedValue block(cx);
            if (!comprehensionBlock(forNode, &block) || !blocks.append(block))
                return false;
            next = forNode->body();
        } else if (next->isKind(PNK_IF)) {
            TernaryNode* tn = &next->as<TernaryNode>();
            if (isLegacy) {
                MOZ_ASSERT(filter.isMagic(JS_SERIALIZE_NO_NODE));
                if (!optExpression(tn->kid1(), &filter))
                    return false;
            } else {
                // ES7 comprehension can contain multiple ComprehensionIfs.
                RootedValue compif(cx);
                if (!comprehensionIf(tn, &compif) ||
                    !blocks.append(compif))
                    return false;
            }
            next = tn->kid2();
        } else {
            break;
        }
    }

    LOCAL_ASSERT(next->isKind(PNK_ARRAYPUSH));

    RootedValue body(cx);

    return expression(next->as<UnaryNode>().kid(), &body) &&
           builder.comprehensionExpression(body, blocks, filter, isLegacy, &pn->pn_pos, dst);
}

bool
ASTSerializer::generatorExpression(ParseNode* pn, MutableHandleValue dst)
{
    // Just as there are two kinds of array comprehension (see
    // ASTSerializer::comprehension), there are legacy and modern generator
    // expression.
    bool isLegacy = pn->is<LexicalScopeNode>();
    ParseNode* next = isLegacy ? pn->as<LexicalScopeNode>().scopeBody() : pn;
    LOCAL_ASSERT(next->isKind(PNK_COMPREHENSIONFOR));

    NodeVector blocks(cx);
    RootedValue filter(cx, MagicValue(JS_SERIALIZE_NO_NODE));
    while (true) {
        if (next->isKind(PNK_COMPREHENSIONFOR)) {
            ForNode* forNode = &next->as<ForNode>();
            RootedValue block(cx);
            if (!comprehensionBlock(forNode, &block) || !blocks.append(block))
                return false;
            next = forNode->body();
        } else if (next->isKind(PNK_IF)) {
            TernaryNode* tn = &next->as<TernaryNode>();
            if (isLegacy) {
                MOZ_ASSERT(filter.isMagic(JS_SERIALIZE_NO_NODE));
                if (!optExpression(tn->kid1(), &filter))
                    return false;
            } else {
                // ES7 comprehension can contain multiple ComprehensionIfs.
                RootedValue compif(cx);
                if (!comprehensionIf(tn, &compif) ||
                    !blocks.append(compif))
                    return false;
            }
            next = tn->kid2();
        } else {
            break;
        }
    }

    LOCAL_ASSERT(next->isKind(PNK_SEMI));
    ParseNode* yieldNode = next->as<UnaryNode>().kid();
    LOCAL_ASSERT(yieldNode->isKind(PNK_YIELD) &&
                 yieldNode->as<UnaryNode>().kid());

    RootedValue body(cx);

    return expression(yieldNode->as<UnaryNode>().kid(), &body) &&
           builder.generatorExpression(body, blocks, filter, isLegacy, &pn->pn_pos, dst);
}

bool
ASTSerializer::expression(ParseNode* pn, MutableHandleValue dst)
{
    JS_CHECK_RECURSION(cx, return false);
    switch (pn->getKind()) {
      case PNK_FUNCTION:
      {
        FunctionNode* funNode = &pn->as<FunctionNode>();
        ASTType type = funNode->funbox()->function()->isArrow() ? AST_ARROW_EXPR : AST_FUNC_EXPR;
        return function(funNode, type, dst);
      }

      case PNK_COMMA:
      {
        NodeVector exprs(cx);
        return expressions(&pn->as<ListNode>(), exprs) &&
               builder.sequenceExpression(exprs, &pn->pn_pos, dst);
      }

      case PNK_CONDITIONAL:
      {
        ConditionalExpression* condNode = &pn->as<ConditionalExpression>();
        ParseNode* testNode = condNode->kid1();
        ParseNode* consNode = condNode->kid2();
        ParseNode* altNode = condNode->kid3();
        MOZ_ASSERT(condNode->pn_pos.encloses(testNode->pn_pos));
        MOZ_ASSERT(condNode->pn_pos.encloses(consNode->pn_pos));
        MOZ_ASSERT(condNode->pn_pos.encloses(altNode->pn_pos));

        RootedValue test(cx), cons(cx), alt(cx);

        return expression(testNode, &test) &&
               expression(consNode, &cons) &&
               expression(altNode, &alt) &&
               builder.conditionalExpression(test, cons, alt, &condNode->pn_pos, dst);
      }

      case PNK_COALESCE:
      case PNK_OR:
      case PNK_AND:
        return leftAssociate(&pn->as<ListNode>(), dst);

      case PNK_PREINCREMENT:
      case PNK_PREDECREMENT:
      {
        UnaryNode* incDec = &pn->as<UnaryNode>();
        ParseNode* operand = incDec->kid();
        MOZ_ASSERT(incDec->pn_pos.encloses(operand->pn_pos));

        bool inc = incDec->isKind(PNK_PREINCREMENT);
        RootedValue expr(cx);
        return expression(operand, &expr) &&
               builder.updateExpression(expr, inc, true, &incDec->pn_pos, dst);
      }

      case PNK_POSTINCREMENT:
      case PNK_POSTDECREMENT:
      {
        UnaryNode* incDec = &pn->as<UnaryNode>();
        ParseNode* operand = incDec->kid();
        MOZ_ASSERT(incDec->pn_pos.encloses(operand->pn_pos));

        bool inc = incDec->isKind(PNK_POSTINCREMENT);
        RootedValue expr(cx);
        return expression(operand, &expr) &&
               builder.updateExpression(expr, inc, false, &incDec->pn_pos, dst);
      }

      case PNK_ASSIGN:
      case PNK_ADDASSIGN:
      case PNK_SUBASSIGN:
      case PNK_COALESCEASSIGN:
      case PNK_ORASSIGN:
      case PNK_ANDASSIGN:
      case PNK_BITORASSIGN:
      case PNK_BITXORASSIGN:
      case PNK_BITANDASSIGN:
      case PNK_LSHASSIGN:
      case PNK_RSHASSIGN:
      case PNK_URSHASSIGN:
      case PNK_MULASSIGN:
      case PNK_DIVASSIGN:
      case PNK_MODASSIGN:
      case PNK_POWASSIGN:
      {
        AssignmentNode* assignNode = &pn->as<AssignmentNode>();
        ParseNode* lhsNode = assignNode->left();
        ParseNode* rhsNode = assignNode->right();
        MOZ_ASSERT(assignNode->pn_pos.encloses(lhsNode->pn_pos));
        MOZ_ASSERT(assignNode->pn_pos.encloses(rhsNode->pn_pos));

        AssignmentOperator op = aop(assignNode->getOp());
        LOCAL_ASSERT(op > AOP_ERR && op < AOP_LIMIT);

        RootedValue lhs(cx), rhs(cx);
        return pattern(lhsNode, &lhs) &&
               expression(rhsNode, &rhs) &&
               builder.assignmentExpression(op, lhs, rhs, &assignNode->pn_pos, dst);
      }

      case PNK_ADD:
      case PNK_SUB:
      case PNK_STRICTEQ:
      case PNK_EQ:
      case PNK_STRICTNE:
      case PNK_NE:
      case PNK_LT:
      case PNK_LE:
      case PNK_GT:
      case PNK_GE:
      case PNK_LSH:
      case PNK_RSH:
      case PNK_URSH:
      case PNK_STAR:
      case PNK_DIV:
      case PNK_MOD:
      case PNK_BITOR:
      case PNK_BITXOR:
      case PNK_BITAND:
      case PNK_IN:
      case PNK_INSTANCEOF:
        return leftAssociate(&pn->as<ListNode>(), dst);

      case PNK_POW:
        return rightAssociate(&pn->as<ListNode>(), dst);

      case PNK_DELETENAME:
      case PNK_DELETEPROP:
      case PNK_DELETEELEM:
      case PNK_DELETEEXPR:
      case PNK_TYPEOFNAME:
      case PNK_TYPEOFEXPR:
      case PNK_VOID:
      case PNK_NOT:
      case PNK_BITNOT:
      case PNK_POS:
      case PNK_AWAIT:
      case PNK_NEG: {
        UnaryNode* unaryNode = &pn->as<UnaryNode>();
        ParseNode* operand = unaryNode->kid();
        MOZ_ASSERT(unaryNode->pn_pos.encloses(operand->pn_pos));

        UnaryOperator op = unop(unaryNode->getKind(), unaryNode->getOp());
        LOCAL_ASSERT(op > UNOP_ERR && op < UNOP_LIMIT);

        RootedValue expr(cx);
        return expression(operand, &expr) &&
               builder.unaryExpression(op, expr, &unaryNode->pn_pos, dst);
      }

      case PNK_GENEXP: {
        ParseNode* callee = pn->as<BinaryNode>().left();
        MOZ_ASSERT(callee->isKind(PNK_FUNCTION));

        ListNode* paramsBody = callee->as<FunctionNode>().body();
        MOZ_ASSERT(paramsBody->isKind(PNK_PARAMSBODY));

        ListNode* body = &paramsBody->last()->as<ListNode>();
        MOZ_ASSERT(body->isKind(PNK_STATEMENTLIST));
        MOZ_ASSERT(body->last()->isKind(PNK_LEXICALSCOPE) ||
                   body->last()->isKind(PNK_COMPREHENSIONFOR));
        return generatorExpression(body->last(), dst);
      }

      case PNK_DELETEOPTCHAIN: {
        RootedValue expr(cx);
        return expression(pn->as<UnaryNode>().kid(), &expr) &&
               builder.deleteOptionalExpression(expr, &pn->pn_pos, dst);
      }

      case PNK_OPTCHAIN: {
        RootedValue expr(cx);
        return expression(pn->as<UnaryNode>().kid(), &expr) &&
               builder.optionalExpression(expr, &pn->pn_pos, dst);
      }

      case PNK_NEW:
      case PNK_TAGGED_TEMPLATE:
      case PNK_CALL:
      case PNK_OPTCALL:
      case PNK_SUPERCALL:
      {
        BinaryNode* node = &pn->as<BinaryNode>();
        ParseNode* calleeNode = node->left();
        ListNode* argsList = &node->right()->as<ListNode>();
        MOZ_ASSERT(node->pn_pos.encloses(calleeNode->pn_pos));

        RootedValue callee(cx);
        if (node->isKind(PNK_SUPERCALL)) {
            MOZ_ASSERT(calleeNode->isKind(PNK_SUPERBASE));
            if (!builder.super(&calleeNode->pn_pos, &callee))
                return false;
        } else {
            if (!expression(calleeNode, &callee))
                return false;
        }

        NodeVector args(cx);
        if (!args.reserve(argsList->count()))
            return false;

        for (ParseNode* argNode : argsList->contents()) {
            MOZ_ASSERT(node->pn_pos.encloses(argNode->pn_pos));

            RootedValue arg(cx);
            if (!expression(argNode, &arg))
                return false;
            args.infallibleAppend(arg);
        }

        if (node->getKind() == PNK_TAGGED_TEMPLATE)
            return builder.taggedTemplate(callee, args, &node->pn_pos, dst);

        bool isOptional = node->isKind(PNK_OPTCALL);

        // SUPERCALL is Call(super, args)
        return node->isKind(PNK_NEW)
               ? builder.newExpression(callee, args, &node->pn_pos, dst)
               : builder.callExpression(callee, args, &node->pn_pos, dst, isOptional);
      }

      case PNK_OPTDOT:
      case PNK_DOT:
      {
        PropertyAccessBase* prop = &pn->as<PropertyAccessBase>();
        MOZ_ASSERT(prop->pn_pos.encloses(prop->expression().pn_pos));

        RootedValue expr(cx);
        RootedValue propname(cx);
        RootedAtom pnAtom(cx, prop->key().atom());

        if (prop->isSuper()) {
            if (!builder.super(&prop->expression().pn_pos, &expr))
                return false;
        } else {
            if (!expression(&prop->expression(), &expr))
                return false;
        }

        bool isOptional = prop->isKind(PNK_OPTDOT);

        return identifier(pnAtom, nullptr, &propname) &&
               builder.memberExpression(false, expr, propname, &prop->pn_pos,
                                        dst, isOptional);
      }

      case PNK_OPTELEM:
      case PNK_ELEM:
      {
        PropertyByValueBase* elem = &pn->as<PropertyByValueBase>();
        MOZ_ASSERT(elem->pn_pos.encloses(elem->expression().pn_pos));
        MOZ_ASSERT(elem->pn_pos.encloses(elem->key().pn_pos));

        RootedValue expr(cx), key(cx);

        if (elem->isSuper()) {
            if (!builder.super(&elem->expression().pn_pos, &expr))
                return false;
        } else {
            if (!expression(&elem->expression(), &expr))
                return false;
        }

        bool isOptional = elem->isKind(PNK_OPTELEM);

        return expression(&elem->key(), &key) &&
               builder.memberExpression(true, expr, key, &elem->pn_pos, dst,
                                        isOptional);
      }

      case PNK_CALLSITEOBJ:
      {
        CallSiteNode* callSiteObj = &pn->as<CallSiteNode>();
        ListNode* rawNodes = callSiteObj->rawNodes();
        NodeVector raw(cx);
        if (!raw.reserve(rawNodes->count()))
            return false;
        for (ParseNode* item : rawNodes->contents()) {
            NameNode* rawItem = &item->as<NameNode>();
            MOZ_ASSERT(callSiteObj->pn_pos.encloses(rawItem->pn_pos));

            RootedValue expr(cx);
            expr.setString(rawItem->atom());
            raw.infallibleAppend(expr);
        }

        NodeVector cooked(cx);
        if (!cooked.reserve(callSiteObj->count() - 1))
            return false;

        for (ParseNode* cookedItem : callSiteObj->contentsFrom(rawNodes->pn_next)) {
            MOZ_ASSERT(callSiteObj->pn_pos.encloses(cookedItem->pn_pos));

            RootedValue expr(cx);
            if (cookedItem->isKind(PNK_RAW_UNDEFINED)) {
                expr.setUndefined();
            } else {
                MOZ_ASSERT(cookedItem->isKind(PNK_TEMPLATE_STRING));
                expr.setString(cookedItem->as<NameNode>().atom());
            }
            cooked.infallibleAppend(expr);
        }

        return builder.callSiteObj(raw, cooked, &callSiteObj->pn_pos, dst);
      }

      case PNK_ARRAY:
      {
        ListNode* array = &pn->as<ListNode>();
        NodeVector elts(cx);
        if (!elts.reserve(array->count()))
            return false;

        for (ParseNode* item : array->contents()) {
            MOZ_ASSERT(array->pn_pos.encloses(item->pn_pos));

            if (item->isKind(PNK_ELISION)) {
                elts.infallibleAppend(NullValue());
            } else {
                RootedValue expr(cx);
                if (!expression(item, &expr))
                    return false;
                elts.infallibleAppend(expr);
            }
        }

        return builder.arrayExpression(elts, &array->pn_pos, dst);
      }

      case PNK_SPREAD:
      {
          RootedValue expr(cx);
          return expression(pn->as<UnaryNode>().kid(), &expr) &&
                 builder.spreadExpression(expr, &pn->pn_pos, dst);
      }

      case PNK_COMPUTED_NAME:
      {
         RootedValue name(cx);
         return expression(pn->as<UnaryNode>().kid(), &name) &&
                builder.computedName(name, &pn->pn_pos, dst);
      }

      case PNK_OBJECT:
      {
        ListNode* obj = &pn->as<ListNode>();
        NodeVector elts(cx);
        if (!elts.reserve(obj->count()))
            return false;

        for (ParseNode* item : obj->contents()) {
            MOZ_ASSERT(obj->pn_pos.encloses(item->pn_pos));

            RootedValue prop(cx);
            if (!property(item, &prop))
                return false;
            elts.infallibleAppend(prop);
        }

        return builder.objectExpression(elts, &obj->pn_pos, dst);
      }

      case PNK_NAME:
        return identifier(&pn->as<NameNode>(), dst);

      case PNK_THIS:
        return builder.thisExpression(&pn->pn_pos, dst);

      case PNK_TEMPLATE_STRING_LIST:
      {
        ListNode* list = &pn->as<ListNode>();
        NodeVector elts(cx);
        if (!elts.reserve(list->count()))
            return false;

        for (ParseNode* item : list->contents()) {
            MOZ_ASSERT(list->pn_pos.encloses(item->pn_pos));

            RootedValue expr(cx);
            if (!expression(item, &expr))
                return false;
            elts.infallibleAppend(expr);
        }

        return builder.templateLiteral(elts, &list->pn_pos, dst);
      }

      case PNK_TEMPLATE_STRING:
      case PNK_STRING:
      case PNK_REGEXP:
      case PNK_NUMBER:
      case PNK_BIGINT:
      case PNK_TRUE:
      case PNK_FALSE:
      case PNK_NULL:
      case PNK_RAW_UNDEFINED:
        return literal(pn, dst);

      case PNK_YIELD_STAR:
      {
        UnaryNode* yieldNode = &pn->as<UnaryNode>();
        ParseNode* operand = yieldNode->kid();
        MOZ_ASSERT(yieldNode->pn_pos.encloses(operand->pn_pos));

        RootedValue arg(cx);
        return expression(operand, &arg) &&
               builder.yieldExpression(arg, Delegating, &yieldNode->pn_pos, dst);
      }

      case PNK_YIELD:
      {
        UnaryNode* yieldNode = &pn->as<UnaryNode>();
        ParseNode* operand = yieldNode->kid();
        MOZ_ASSERT_IF(operand, yieldNode->pn_pos.encloses(operand->pn_pos));

        RootedValue arg(cx);
        return optExpression(operand, &arg) &&
               builder.yieldExpression(arg, NotDelegating, &yieldNode->pn_pos, dst);
      }

      case PNK_ARRAYCOMP:
      {
        ListNode* literal = &pn->as<ListNode>();
        MOZ_ASSERT(pn->pn_pos.encloses(literal->head()->pn_pos));

        /* NB: it's no longer the case that pn_count could be 2. */
        LOCAL_ASSERT(literal->count() == 1);
        return comprehension(literal->head(), dst);
      }

      case PNK_CLASS:
        return classDefinition(&pn->as<ClassNode>(), true, dst);

      case PNK_NEWTARGET:
      case PNK_IMPORT_META:
      {
        BinaryNode* node = &pn->as<BinaryNode>();
        ParseNode* firstNode = node->left();
        MOZ_ASSERT(firstNode->isKind(PNK_POSHOLDER));
        MOZ_ASSERT(node->pn_pos.encloses(firstNode->pn_pos));

        ParseNode* secondNode = node->right();
        MOZ_ASSERT(secondNode->isKind(PNK_POSHOLDER));
        MOZ_ASSERT(node->pn_pos.encloses(secondNode->pn_pos));

        RootedValue firstIdent(cx);
        RootedValue secondIdent(cx);

        RootedAtom firstStr(cx);
        RootedAtom secondStr(cx);

        if (pn->getKind() == PNK_NEWTARGET) {
            firstStr = cx->names().new_;
            secondStr = cx->names().target;
        } else {
            firstStr = cx->names().import;
            secondStr = cx->names().meta;
        }

        return identifier(firstStr, &firstNode->pn_pos, &firstIdent) &&
               identifier(secondStr, &secondNode->pn_pos, &secondIdent) &&
               builder.metaProperty(firstIdent, secondIdent, &pn->pn_pos, dst);
      }

      case PNK_CALL_IMPORT:
      {
        BinaryNode* node = &pn->as<BinaryNode>();
        ParseNode* firstNode = node->left();
        MOZ_ASSERT(firstNode->isKind(PNK_POSHOLDER));
        MOZ_ASSERT(pn->pn_pos.encloses(firstNode->pn_pos));
        ParseNode* secondNode = node->right();
        MOZ_ASSERT(pn->pn_pos.encloses(secondNode->pn_pos));

        RootedValue ident(cx);
        RootedValue arg(cx);

        HandlePropertyName name = cx->names().import;
        return identifier(name, &firstNode->pn_pos, &ident) &&
               expression(secondNode, &arg) &&
               builder.callImportExpression(ident, arg, &pn->pn_pos, dst);
      }

      case PNK_SETTHIS: {
        // SETTHIS is used to assign the result of a super() call to |this|.
        // It's not part of the original AST, so just forward to the call.
        BinaryNode* node = &pn->as<BinaryNode>();
        MOZ_ASSERT(node->left()->isKind(PNK_NAME));
        return expression(node->right(), dst);
      }

      default:
        LOCAL_NOT_REACHED("unexpected expression type");
    }
}

bool
ASTSerializer::propertyName(ParseNode* key, MutableHandleValue dst)
{
    if (key->isKind(PNK_COMPUTED_NAME))
        return expression(key, dst);
    if (key->isKind(PNK_OBJECT_PROPERTY_NAME))
        return identifier(&key->as<NameNode>(), dst);

    LOCAL_ASSERT(key->isKind(PNK_STRING) || key->isKind(PNK_NUMBER));

    return literal(key, dst);
}

bool
ASTSerializer::property(ParseNode* pn, MutableHandleValue dst)
{
    if (pn->isKind(PNK_MUTATEPROTO)) {
        RootedValue val(cx);
        return expression(pn->as<UnaryNode>().kid(), &val) &&
               builder.prototypeMutation(val, &pn->pn_pos, dst);
    }
    if (pn->isKind(PNK_SPREAD))
        return expression(pn, dst);

    PropKind kind;
    switch (pn->getOp()) {
      case JSOP_INITPROP:
        kind = PROP_INIT;
        break;

      case JSOP_INITPROP_GETTER:
        kind = PROP_GETTER;
        break;

      case JSOP_INITPROP_SETTER:
        kind = PROP_SETTER;
        break;

      default:
        LOCAL_NOT_REACHED("unexpected object-literal property");
    }

    BinaryNode* node = &pn->as<BinaryNode>();
    ParseNode* keyNode = node->left();
    ParseNode* valNode = node->right();

    bool isShorthand = node->isKind(PNK_SHORTHAND);
    bool isMethod = valNode->is<FunctionNode>() &&
                    valNode->as<FunctionNode>().funbox()->function()->kind() == JSFunction::Method;
    RootedValue key(cx), val(cx);
    return propertyName(keyNode, &key) &&
           expression(valNode, &val) &&
           builder.propertyInitializer(key, val, kind, isShorthand, isMethod, &node->pn_pos, dst);
}

bool
ASTSerializer::literal(ParseNode* pn, MutableHandleValue dst)
{
    RootedValue val(cx);
    switch (pn->getKind()) {
      case PNK_TEMPLATE_STRING:
      case PNK_STRING:
        val.setString(pn->as<NameNode>().atom());
        break;

      case PNK_REGEXP:
      {
        RootedObject re1(cx, pn->as<RegExpLiteral>().objbox()->object());
        LOCAL_ASSERT(re1 && re1->is<RegExpObject>());

        RootedObject re2(cx, CloneRegExpObject(cx, re1));
        if (!re2)
            return false;

        val.setObject(*re2);
        break;
      }

      case PNK_NUMBER:
        val.setNumber(pn->as<NumericLiteral>().value());
        break;

      case PNK_BIGINT:
      {
        BigInt* x = pn->as<BigIntLiteral>().box()->value();
        val.setBigInt(x);
        break;
      }

      case PNK_NULL:
        val.setNull();
        break;

      case PNK_RAW_UNDEFINED:
        val.setUndefined();
        break;

      case PNK_TRUE:
        val.setBoolean(true);
        break;

      case PNK_FALSE:
        val.setBoolean(false);
        break;

      default:
        LOCAL_NOT_REACHED("unexpected literal type");
    }

    return builder.literal(val, &pn->pn_pos, dst);
}

bool
ASTSerializer::arrayPattern(ListNode* array, MutableHandleValue dst)
{
    MOZ_ASSERT(array->isKind(PNK_ARRAY));

    NodeVector elts(cx);
    if (!elts.reserve(array->count()))
        return false;

    for (ParseNode* item : array->contents()) {
        if (item->isKind(PNK_ELISION)) {
            elts.infallibleAppend(NullValue());
        } else if (item->isKind(PNK_SPREAD)) {
            RootedValue target(cx);
            RootedValue spread(cx);
            if (!pattern(item->as<UnaryNode>().kid(), &target))
                return false;
            if(!builder.spreadExpression(target, &item->pn_pos, &spread))
                return false;
            elts.infallibleAppend(spread);
        } else {
            RootedValue patt(cx);
            if (!pattern(item, &patt))
                return false;
            elts.infallibleAppend(patt);
        }
    }

    return builder.arrayPattern(elts, &array->pn_pos, dst);
}

bool
ASTSerializer::objectPattern(ListNode* obj, MutableHandleValue dst)
{
    MOZ_ASSERT(obj->isKind(PNK_OBJECT));

    NodeVector elts(cx);
    if (!elts.reserve(obj->count()))
        return false;

    for (ParseNode* propdef : obj->contents()) {
        if (propdef->isKind(PNK_SPREAD)) {
            RootedValue target(cx);
            RootedValue spread(cx);
            if (!pattern(propdef->as<UnaryNode>().kid(), &target))
                return false;
            if(!builder.spreadExpression(target, &propdef->pn_pos, &spread))
                return false;
            elts.infallibleAppend(spread);
            continue;
        }
        LOCAL_ASSERT(propdef->isKind(PNK_MUTATEPROTO) != propdef->isOp(JSOP_INITPROP));

        RootedValue key(cx);
        ParseNode* target;
        if (propdef->isKind(PNK_MUTATEPROTO)) {
            RootedValue pname(cx, StringValue(cx->names().proto));
            if (!builder.literal(pname, &propdef->pn_pos, &key))
                return false;
            target = propdef->as<UnaryNode>().kid();
        } else {
            BinaryNode* prop = &propdef->as<BinaryNode>();
            if (!propertyName(prop->left(), &key))
                return false;
            target = prop->right();
        }

        RootedValue patt(cx), prop(cx);
        if (!pattern(target, &patt) ||
            !builder.propertyPattern(key, patt, propdef->isKind(PNK_SHORTHAND), &propdef->pn_pos,
                                     &prop))
        {
            return false;
        }

        elts.infallibleAppend(prop);
    }

    return builder.objectPattern(elts, &obj->pn_pos, dst);
}

bool
ASTSerializer::pattern(ParseNode* pn, MutableHandleValue dst)
{
    JS_CHECK_RECURSION(cx, return false);
    switch (pn->getKind()) {
      case PNK_OBJECT:
        return objectPattern(&pn->as<ListNode>(), dst);

      case PNK_ARRAY:
        return arrayPattern(&pn->as<ListNode>(), dst);

      default:
        return expression(pn, dst);
    }
}

bool
ASTSerializer::identifier(HandleAtom atom, TokenPos* pos, MutableHandleValue dst)
{
    RootedValue atomContentsVal(cx, unrootedAtomContents(atom));
    return builder.identifier(atomContentsVal, pos, dst);
}

bool
ASTSerializer::identifier(NameNode* id, MutableHandleValue dst)
{
    LOCAL_ASSERT(id->atom());

    RootedAtom pnAtom(cx, id->atom());
    return identifier(pnAtom, &id->pn_pos, dst);
}

bool
ASTSerializer::function(FunctionNode* funNode, ASTType type, MutableHandleValue dst)
{
    FunctionBox* funbox = funNode->funbox();
    RootedFunction func(cx, funbox->function());

    GeneratorStyle generatorStyle =
        funbox->isStarGenerator()
        ? GeneratorStyle::ES6
        : funbox->isLegacyGenerator()
        ? GeneratorStyle::Legacy
        : GeneratorStyle::None;

    bool isAsync = funbox->isAsync();
    bool isExpression = funbox->isExprBody();

    RootedValue id(cx);
    RootedAtom funcAtom(cx, func->explicitName());
    if (!optIdentifier(funcAtom, nullptr, &id))
        return false;

    NodeVector args(cx);
    NodeVector defaults(cx);

    RootedValue body(cx), rest(cx);
    if (funbox->hasRest())
        rest.setUndefined();
    else
        rest.setNull();
    return functionArgsAndBody(funNode->body(), args, defaults, isAsync, isExpression,
                               &body, &rest) &&
           builder.function(type, &funNode->pn_pos, id, args, defaults, body,
                            rest, generatorStyle, isAsync, isExpression, dst);
}

bool
ASTSerializer::functionArgsAndBody(ParseNode* pn, NodeVector& args, NodeVector& defaults,
                                   bool isAsync, bool isExpression,
                                   MutableHandleValue body, MutableHandleValue rest)
{
    ListNode* argsList;
    ParseNode* bodyNode;

    /* Extract the args and body separately. */
    if (pn->isKind(PNK_PARAMSBODY)) {
        argsList = &pn->as<ListNode>();
        bodyNode = argsList->last();
    } else {
        argsList = nullptr;
        bodyNode = pn;
    }

    if (bodyNode->is<LexicalScopeNode>())
        bodyNode = bodyNode->as<LexicalScopeNode>().scopeBody();

    /* Serialize the arguments and body. */
    switch (bodyNode->getKind()) {
      case PNK_RETURN: /* expression closure, no destructured args */
        return functionArgs(pn, argsList, args, defaults, rest) &&
               expression(bodyNode->as<UnaryNode>().kid(), body);

      case PNK_STATEMENTLIST:     /* statement closure */
      {
        ParseNode* firstNode = bodyNode->as<ListNode>().head();

        // Skip over initial yield in generator.
        if (firstNode && firstNode->isKind(PNK_INITIALYIELD)) {
            MOZ_ASSERT(firstNode->getOp() == JSOP_INITIALYIELD);
            firstNode = firstNode->pn_next;
        }

        // Async arrow with expression body is converted into STATEMENTLIST
        // to insert initial yield.
        if (isAsync && isExpression) {
            MOZ_ASSERT(firstNode->getKind() == PNK_RETURN);
            return functionArgs(pn, argsList, args, defaults, rest) &&
                   expression(firstNode->as<UnaryNode>().kid(), body);
        }

        return functionArgs(pn, argsList, args, defaults, rest) &&
               functionBody(firstNode, &bodyNode->pn_pos, body);
      }

      default:
        LOCAL_NOT_REACHED("unexpected function contents");
    }
}

bool
ASTSerializer::functionArgs(ParseNode* pn, ListNode* argsList,
                            NodeVector& args, NodeVector& defaults,
                            MutableHandleValue rest)
{
    if (!argsList)
        return true;

    RootedValue node(cx);
    bool defaultsNull = true;
    MOZ_ASSERT(defaults.empty(),
               "must be initially empty for it to be proper to clear this "
               "when there are no defaults");

    for (ParseNode* arg : argsList->contentsTo(argsList->last())) {
        ParseNode* pat;
        ParseNode* defNode;
        if (arg->isKind(PNK_NAME) || arg->isKind(PNK_ARRAY) || arg->isKind(PNK_OBJECT)) {
            pat = arg;
            defNode = nullptr;
        } else {
            AssignmentNode* assignNode = &arg->as<AssignmentNode>();
            pat = assignNode->left();
            defNode = assignNode->right();
        }

        // Process the name or pattern.
        MOZ_ASSERT(pat->isKind(PNK_NAME) || pat->isKind(PNK_ARRAY) || pat->isKind(PNK_OBJECT));
        if (!pattern(pat, &node))
            return false;
        if (rest.isUndefined() && arg->pn_next == argsList->last()) {
            rest.setObject(node.toObject());
        } else {
            if (!args.append(node))
                return false;
        }

        // Process its default (or lack thereof).
        if (defNode) {
            defaultsNull = false;
            RootedValue def(cx);
            if (!expression(defNode, &def) || !defaults.append(def))
                return false;
        } else {
            if (!defaults.append(NullValue()))
                return false;
        }
    }
    MOZ_ASSERT(!rest.isUndefined(),
               "if a rest argument was present (signified by "
               "|rest.isUndefined()| initially), the rest node was properly "
               "recorded");

    if (defaultsNull)
        defaults.clear();

    return true;
}

bool
ASTSerializer::functionBody(ParseNode* pn, TokenPos* pos, MutableHandleValue dst)
{
    NodeVector elts(cx);

    /* We aren't sure how many elements there are up front, so we'll check each append. */
    for (ParseNode* next = pn; next; next = next->pn_next) {
        RootedValue child(cx);
        if (!sourceElement(next, &child) || !elts.append(child))
            return false;
    }

    return builder.blockStatement(elts, pos, dst);
}

static bool
reflect_parse(JSContext* cx, uint32_t argc, Value* vp)
{
    CallArgs args = CallArgsFromVp(argc, vp);

    if (args.length() < 1) {
        JS_ReportErrorNumberASCII(cx, GetErrorMessage, nullptr, JSMSG_MORE_ARGS_NEEDED,
                                  "Reflect.parse", "0", "s");
        return false;
    }

    RootedString src(cx, ToString<CanGC>(cx, args[0]));
    if (!src)
        return false;

    ScopedJSFreePtr<char> filename;
    uint32_t lineno = 1;
    bool loc = true;
    RootedObject builder(cx);
    ParseGoal target = ParseGoal::Script;

    RootedValue arg(cx, args.get(1));

    if (!arg.isNullOrUndefined()) {
        if (!arg.isObject()) {
            ReportValueErrorFlags(cx, JSREPORT_ERROR, JSMSG_UNEXPECTED_TYPE,
                                  JSDVG_SEARCH_STACK, arg, nullptr,
                                  "not an object", nullptr);
            return false;
        }

        RootedObject config(cx, &arg.toObject());

        RootedValue prop(cx);

        /* config.loc */
        RootedId locId(cx, NameToId(cx->names().loc));
        RootedValue trueVal(cx, BooleanValue(true));
        if (!GetPropertyDefault(cx, config, locId, trueVal, &prop))
            return false;

        loc = ToBoolean(prop);

        if (loc) {
            /* config.source */
            RootedId sourceId(cx, NameToId(cx->names().source));
            RootedValue nullVal(cx, NullValue());
            if (!GetPropertyDefault(cx, config, sourceId, nullVal, &prop))
                return false;

            if (!prop.isNullOrUndefined()) {
                RootedString str(cx, ToString<CanGC>(cx, prop));
                if (!str)
                    return false;

                filename = JS_EncodeString(cx, str);
                if (!filename)
                    return false;
            }

            /* config.line */
            RootedId lineId(cx, NameToId(cx->names().line));
            RootedValue oneValue(cx, Int32Value(1));
            if (!GetPropertyDefault(cx, config, lineId, oneValue, &prop) ||
                !ToUint32(cx, prop, &lineno)) {
                return false;
            }
        }

        /* config.builder */
        RootedId builderId(cx, NameToId(cx->names().builder));
        RootedValue nullVal(cx, NullValue());
        if (!GetPropertyDefault(cx, config, builderId, nullVal, &prop))
            return false;

        if (!prop.isNullOrUndefined()) {
            if (!prop.isObject()) {
                ReportValueErrorFlags(cx, JSREPORT_ERROR, JSMSG_UNEXPECTED_TYPE,
                                      JSDVG_SEARCH_STACK, prop, nullptr,
                                      "not an object", nullptr);
                return false;
            }
            builder = &prop.toObject();
        }

        /* config.target */
        RootedId targetId(cx, NameToId(cx->names().target));
        RootedValue scriptVal(cx, StringValue(cx->names().script));
        if (!GetPropertyDefault(cx, config, targetId, scriptVal, &prop))
            return false;

        if (!prop.isString()) {
            ReportValueErrorFlags(cx, JSREPORT_ERROR, JSMSG_UNEXPECTED_TYPE, JSDVG_SEARCH_STACK,
                                  prop, nullptr, "not 'script' or 'module'", nullptr);
            return false;
        }

        RootedString stringProp(cx, prop.toString());
        bool isScript = false;
        bool isModule = false;
        if (!EqualStrings(cx, stringProp, cx->names().script, &isScript))
            return false;

        if (!EqualStrings(cx, stringProp, cx->names().module, &isModule))
            return false;

        if (isScript) {
            target = ParseGoal::Script;
        } else if (isModule) {
            target = ParseGoal::Module;
        } else {
            JS_ReportErrorASCII(cx, "Bad target value, expected 'script' or 'module'");
            return false;
        }
    }

    /* Extract the builder methods first to report errors before parsing. */
    ASTSerializer serialize(cx, loc, filename, lineno);
    if (!serialize.init(builder))
        return false;

    JSLinearString* linear = src->ensureLinear(cx);
    if (!linear)
        return false;

    AutoStableStringChars linearChars(cx);
    if (!linearChars.initTwoByte(cx, linear))
        return false;

    CompileOptions options(cx);
    options.setFileAndLine(filename, lineno);
    options.setCanLazilyParse(false);
    mozilla::Range<const char16_t> chars = linearChars.twoByteRange();
    UsedNameTracker usedNames(cx);
    if (!usedNames.init())
        return false;
    Parser<FullParseHandler> parser(cx, cx->tempLifoAlloc(), options, chars.begin().get(),
                                    chars.length(), /* foldConstants = */ false, usedNames,
                                    nullptr, nullptr);
    if (!parser.checkOptions())
        return false;

    serialize.setParser(&parser);

    ParseNode* pn;
    if (target == ParseGoal::Script) {
        pn = parser.parse();
        if (!pn)
            return false;
    } else {
        if (!GlobalObject::ensureModulePrototypesCreated(cx, cx->global()))
            return false;

        Rooted<ModuleObject*> module(cx, ModuleObject::create(cx));
        if (!module)
            return false;

        ModuleBuilder builder(cx, module, parser.tokenStream);
        ModuleSharedContext modulesc(cx, module, &cx->global()->emptyGlobalScope(), builder);
        pn = parser.moduleBody(&modulesc);
        if (!pn)
            return false;

        pn = pn->as<ModuleNode>().body();
    }

    RootedValue val(cx);
    if (!serialize.program(&pn->as<ListNode>(), &val)) {
        args.rval().setNull();
        return false;
    }

    args.rval().set(val);
    return true;
}

JS_PUBLIC_API(bool)
JS_InitReflectParse(JSContext* cx, HandleObject global)
{
    RootedValue reflectVal(cx);
    if (!GetProperty(cx, global, global, cx->names().Reflect, &reflectVal))
        return false;
    if (!reflectVal.isObject()) {
        JS_ReportErrorASCII(cx, "JS_InitReflectParse must be called during global initialization");
        return false;
    }

    RootedObject reflectObj(cx, &reflectVal.toObject());
    return JS_DefineFunction(cx, reflectObj, "parse", reflect_parse, 1, 0);
}
