/**
*** \file preferenceswidget.h
*** \brief Application preferences widget.
*** \details A widget which allows the user to change preferences for the
***   application.
**/

/*
** This file is part of moonphase.
** Copyright (C) 2014-2018 by Alan Wise <awisesoftware@gmail.com>.
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


#ifndef   PREFERENCESWIDGET_H
/**
*** \brief preferenceswidget.h identifier.
*** \details Identifier for preferenceswidget.h.
**/
#define   PREFERENCESWIDGET_H


/****
*****
***** INCLUDES
*****
****/

#include  "ui_preferenceswidget.h"
#include  "moonanimation.h"
#include  "settings.h"

#include  <QTimer>


/****
*****
***** DEFINES
*****
****/


/****
*****
***** DATA TYPES
*****
****/

/**
*** \brief Application preferences widget class.
*** \details A widget class which allows the user to change preferences for the
***   application.
**/
class PREFERENCESWIDGET_C : public QWidget, public Ui::PREFERENCESWIDGETUI_C
{
  Q_OBJECT

  public:
    /**
    *** \brief Constructor.
    *** \details Constructor.
    *** \param pParent Pointer to parent widget.
    **/
    PREFERENCESWIDGET_C(QWidget *pParent=NULL);

    /**
    *** \brief Destructor.
    *** \details Destructor.
    **/
    ~PREFERENCESWIDGET_C(void);

    /**
    *** \brief Get preferences.
    *** \details Gets the preferences of the preferences panel.
    *** \returns Preferences.
    **/
    SETTINGS_C::APPLICATIONPREFERENCES_C GetPreferences(void);

    /**
    *** \brief Set preferences.
    *** \details Sets the preferences of the preferences panel.
    *** \param pSettings Preferences.
    **/
    void SetPreferences(
        SETTINGS_C::APPLICATIONPREFERENCES_C const *pPreferences);

  private:
    /**
    *** \brief Hide event.
    *** \details The widget is hidden.
    *** \param pEvent Event data.
    **/
    void hideEvent(QShowEvent *pEvent);

    /**
    *** \brief Show event.
    *** \details The widget is shown.
    *** \param pEvent Event data.
    **/
    void showEvent(QShowEvent *pEvent);

  private slots:
    /**
    *** \brief Animation pathname changed.
    *** \details The animation pathname has changed, attempt to read the file.
    **/
    void AnimationPathnameChangedSlot(void);

    /**
    *** \brief Animation timer triggered.
    *** \details The animation timer has been triggered, update the preview
    ***   image.
    **/
    void AnimationTimerTriggeredSlot(void);

    /**
    *** \brief Preferences changed.
    *** \details The preferences have changed.
    **/
    void PreferencesChangedSlot(void);

    /**
    *** \brief Shows/hides the control panel.
    *** \details Shows/hides the control panel, may also start or stop the
    ***   animation timer.
    *** \param VisibleFlag 0=Hide control panel,\n!0=Show control panel.
    **/
    void SetVisible(bool VisibleFlag);

    /**
    *** \brief Still Running Reminder widget clicked.
    *** \details The Still Running Reminder widget was clicked.
    **/
    void StillRunningReminderClickedSlot(void);

    /**
    *** \brief Use Opaque Background widget clicked.
    *** \details The Use Opaque Background widget was clicked.
    **/
    void UseOpaqueBackgroundClickedSlot(void);

  signals:
    /**
    *** \brief Preferences changed signal.
    *** \details Signal emitted when a preference has changed.
    **/
    void PreferencesChangedSignal(void);

  private:
    /**
    *** \brief Animation timer.
    *** \details Timer used to update the preview widget.
    **/
    QTimer m_AnimationTimer;

    /**
    *** \brief Original background color.
    *** \details Color of the background at program start or since last
    ***   settings save.
    **/
    QColor m_OriginalBackgroundColor;

    /**
    *** \brief Preview icon.
    *** \details The images displayed in the preview widget.
    **/
    MOONANIMATION_T m_MoonPreviewImages;

    /**
    *** \brief Preview percent counter.
    *** \details Counter used to animate the preview widget.
    **/
    unsigned int m_PreviewPercentCounter;

    /**
    *** \brief Original "Remind Once Per Session" flag.
    *** \details Status of the "Remind Once Per Session" flag at program start
    ***   or since last settings save.
    **/
    bool m_OriginalRemindOncePerSessionFlag;
};


/****
*****
***** DATA
*****
****/


/****
*****
***** VARIABLES
*****
****/


/****
*****
***** PROTOTYPES
*****
****/


#endif    /* PREFERENCESWIDGET_H */


/**
*** preferenceswidget.cpp
**/
