/*
 * Copyright @ 2015 Atlassian Pty Ltd
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.java.otr4j.crypto;

import java.math.BigInteger;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.PublicKey;

import javax.crypto.interfaces.DHPublicKey;

/**
 *
 * @author George Politis
 */
public interface OtrCryptoEngine {

	String MODULUS_TEXT = "00FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD129024E088A67CC74020BBEA63B139B22514A08798E3404DDEF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7EDEE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3DC2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F83655D23DCA3AD961C62F356208552BB9ED529077096966D670C354E4ABC9804F1746C08CA237327FFFFFFFFFFFFFFFF";
	BigInteger MODULUS = new BigInteger(MODULUS_TEXT, 16);
	BigInteger BIGINTEGER_TWO = BigInteger.valueOf(2);
	BigInteger MODULUS_MINUS_TWO = MODULUS
			.subtract(BIGINTEGER_TWO);

	String GENERATOR_TEXT = "2";
	BigInteger GENERATOR = new BigInteger(GENERATOR_TEXT, 10);

	int AES_KEY_BYTE_LENGTH = 16;
	int SHA256_HMAC_KEY_BYTE_LENGTH = 32;
	int DH_PRIVATE_KEY_MINIMUM_BIT_LENGTH = 320;
	byte[] ZERO_CTR = new byte[] { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

	int DSA_PUB_TYPE = 0;

	KeyPair generateDHKeyPair() throws OtrCryptoException;

	DHPublicKey getDHPublicKey(byte[] mpiBytes) throws OtrCryptoException;

	DHPublicKey getDHPublicKey(BigInteger mpi) throws OtrCryptoException;

	byte[] sha256Hmac(byte[] b, byte[] key) throws OtrCryptoException;

	byte[] sha256Hmac(byte[] b, byte[] key, int length)
			throws OtrCryptoException;

	byte[] sha1Hmac(byte[] b, byte[] key, int length) throws OtrCryptoException;

	byte[] sha256Hmac160(byte[] b, byte[] key) throws OtrCryptoException;

	byte[] sha256Hash(byte[] b) throws OtrCryptoException;

	byte[] sha1Hash(byte[] b) throws OtrCryptoException;

	byte[] aesDecrypt(byte[] key, byte[] ctr, byte[] b)
			throws OtrCryptoException;

	byte[] aesEncrypt(byte[] key, byte[] ctr, byte[] b)
			throws OtrCryptoException;

	BigInteger generateSecret(PrivateKey privKey, PublicKey pubKey)
			throws OtrCryptoException;

	byte[] sign(byte[] b, PrivateKey privatekey) throws OtrCryptoException;

	boolean verify(byte[] b, PublicKey pubKey, byte[] rs)
			throws OtrCryptoException;

	String getFingerprint(PublicKey pubKey) throws OtrCryptoException;

	byte[] getFingerprintRaw(PublicKey pubKey) throws OtrCryptoException;
}
