// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.chrome.browser.share.long_screenshots;

import android.content.Context;

import org.chromium.chrome.browser.tab.Tab;
import org.chromium.content_public.browser.RenderCoordinates;
import org.chromium.ui.display.DisplayAndroid;

import java.util.LinkedList;

/**
 * Entry manager responsible for managing all the of the {@LongScreenshotEntry}. This should be used
 * to generate and retrieve the needed bitmaps. The first bitmap can be generated by calling
 * {@link generateInitialBitmap}.
 */
public class EntryManager {
    private Context mContext;
    private Tab mTab;

    private LinkedList<LongScreenshotsEntry> mEntries;
    private int mDisplayHeightPx;

    /**
     * @param context An instance of current Android {@link Context}.
     * @param tab Tab to generate the bitmap for.
     */
    public EntryManager(Context context, Tab tab) {
        mContext = context;
        mTab = tab;
        mEntries = new LinkedList<LongScreenshotsEntry>();
        calculateClipHeight();
    }

    /**
     * Generates the first bitmap of the page that is the height of the phone display. Callers of
     * this function should add a listener to the returned entry to get that status of the
     * generation and retrieve the bitmap.
     */
    public LongScreenshotsEntry generateInitialEntry() {
        RenderCoordinates coords = RenderCoordinates.fromWebContents(mTab.getWebContents());
        int startY = coords.getScrollYPixInt() / coords.getPageScaleFactorInt();

        LongScreenshotsEntry entry =
                new LongScreenshotsEntry(mContext, mTab, startY, mDisplayHeightPx);
        entry.generateBitmap();
        mEntries.add(entry);
        return entry;
    }

    /**
     * Calculates the height of the phone used to determine the height of the bitmaps.
     */
    private void calculateClipHeight() {
        DisplayAndroid displayAndroid = DisplayAndroid.getNonMultiDisplay(mContext);
        mDisplayHeightPx = displayAndroid.getDisplayHeight();
    }
}
