// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/**
 * @fileoverview
 * 'crostini-shared-usb-devices' is the settings shared usb devices subpage for
 * Crostini.
 */

Polymer({
  is: 'settings-crostini-shared-usb-devices',

  behaviors: [I18nBehavior, WebUIListenerBehavior],

  properties: {
    /**
     * The USB Devices available for connection to a VM.
     * @private {Array<!CrostiniSharedUsbDevice>}
     */
    sharedUsbDevices_: Array,

    /**
     * The USB device which was toggled to be shared, but is already shared
     * with another VM. When non-null the reassign dialog is shown.
     * @private {?CrostiniSharedUsbDevice}
     */
    reassignDevice_: {
      type: Object,
      value: null,
    },
  },

  /** @private {settings.CrostiniBrowserProxy} */
  browserProxy_: null,

  /** @override */
  ready() {
    this.browserProxy_ = settings.CrostiniBrowserProxyImpl.getInstance();
    this.addWebUIListener(
        'crostini-shared-usb-devices-changed',
        this.onCrostiniSharedUsbDevicesChanged_.bind(this));
    this.browserProxy_.notifyCrostiniSharedUsbDevicesPageReady();
  },

  /**
   * @param {!Array<CrostiniSharedUsbDevice>} devices
   * @private
   */
  onCrostiniSharedUsbDevicesChanged_(devices) {
    this.sharedUsbDevices_ = devices;
  },

  /**
   * @param {!CustomEvent<!CrostiniSharedUsbDevice>} event
   * @private
   */
  onDeviceSharedChange_(event) {
    const device = event.model.item;
    // Show reassign dialog if device is already shared with another VM.
    if (event.target.checked && device.shareWillReassign) {
      event.target.checked = false;
      this.reassignDevice_ = device;
      return;
    }
    this.browserProxy_.setCrostiniUsbDeviceShared(
        device.guid, event.target.checked);
    settings.recordSettingChange();
  },

  /** @private */
  onReassignCancelClick_() {
    this.reassignDevice_ = null;
  },

  /** @private */
  onReassignContinueClick_() {
    this.browserProxy_.setCrostiniUsbDeviceShared(
        this.reassignDevice_.guid, true);
    this.reassignDevice_ = null;
    settings.recordSettingChange();
  },

  /**
   * @param {!CrostiniSharedUsbDevice} device USB device.
   * @private
   */
  getReassignDialogText_(device) {
    return this.i18n('crostiniSharedUsbDevicesReassign', device.label);
  },
});
