// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/nearby_sharing/file_attachment.h"

#include <utility>

#include "base/strings/string_util.h"
#include "chrome/browser/nearby_sharing/share_target.h"
#include "net/base/mime_util.h"

namespace {

FileAttachment::Type FileAttachmentTypeFromMimeType(
    const std::string& mime_type) {
  if (base::StartsWith(mime_type, "image/"))
    return FileAttachment::Type::kImage;

  if (base::StartsWith(mime_type, "video/"))
    return FileAttachment::Type::kVideo;

  if (base::StartsWith(mime_type, "audio/"))
    return FileAttachment::Type::kAudio;

  return FileAttachment::Type::kUnknown;
}

std::string MimeTypeFromPath(const base::FilePath& path) {
  std::string mime_type = "application/octet-stream";
  base::FilePath::StringType ext = path.Extension();
  if (!ext.empty())
    net::GetWellKnownMimeTypeFromExtension(ext.substr(1), &mime_type);

  return mime_type;
}

}  // namespace

FileAttachment::FileAttachment(base::FilePath file_path)
    : Attachment(Attachment::Family::kFile, /*size=*/0),
      file_name_(file_path.BaseName().AsUTF8Unsafe()),
      mime_type_(MimeTypeFromPath(file_path)),
      type_(FileAttachmentTypeFromMimeType(mime_type_)),
      file_path_(std::move(file_path)) {}

FileAttachment::FileAttachment(int64_t id,
                               int64_t size,
                               std::string file_name,
                               std::string mime_type,
                               Type type)
    : Attachment(id, Attachment::Family::kFile, size),
      file_name_(std::move(file_name)),
      mime_type_(std::move(mime_type)),
      type_(type) {}

FileAttachment::FileAttachment(const FileAttachment&) = default;

FileAttachment::FileAttachment(FileAttachment&&) = default;

FileAttachment& FileAttachment::operator=(const FileAttachment&) = default;

FileAttachment& FileAttachment::operator=(FileAttachment&&) = default;

FileAttachment::~FileAttachment() = default;

void FileAttachment::MoveToShareTarget(ShareTarget& share_target) {
  share_target.file_attachments.push_back(std::move(*this));
}

const std::string& FileAttachment::GetDescription() const {
  return file_name_;
}
