// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/google/google_brand_code_map_chromeos.h"

#include "base/containers/flat_map.h"
#include "base/logging.h"
#include "base/no_destructor.h"
#include "base/notreached.h"
#include "base/stl_util.h"

namespace google_brand {
namespace chromeos {

std::string GetRlzBrandCode(
    const std::string& static_brand_code,
    base::Optional<policy::MarketSegment> market_segment) {
  struct BrandCodeValueEntry {
    const char* unenrolled_brand_code;
    const char* education_enrolled_brand_code;
    const char* enterprise_enrolled_brand_code;
  };
  static const base::NoDestructor<
      base::flat_map<std::string, BrandCodeValueEntry>>
      kBrandCodeMap({{"ACAC", {"CFZM", "BEUH", "GUTN"}},
                     {"ACAG", {"KSOU", "MUHR", "YYJR"}},
                     {"ACAH", {"KEFG", "RYNH", "HHAZ"}},
                     {"ACAI", {"BKWQ", "CMVE", "VNFQ"}},
                     {"ACAJ", {"KVPC", "UHAI", "CPNG"}},
                     {"ACAK", {"PQNT", "MDWH", "AJKH"}},
                     {"ACAM", {"HBCZ", "ZGSZ", "MFUO"}},
                     {"ACAO", {"MWDF", "BNNY", "SYIY"}},
                     {"ACAP", {"LKNW", "SVFL", "FGKR"}},
                     {"ACAQ", {"JXWA", "PZLO", "AKLZ"}},
                     {"ACAR", {"EAQE", "UHHJ", "ZYFW"}},
                     {"ACAS", {"JEHD", "XUID", "FNGA"}},
                     {"ACAT", {"RJNJ", "CKCB", "VHGI"}},
                     {"ACAU", {"DXIN", "LPNB", "FIXM"}},
                     {"ACAV", {"TTSD", "XTQQ", "TIQC"}},
                     {"ACAX", {"CFKW", "QKXN", "VXIS"}},
                     {"ACAY", {"HKDC", "RYKK", "KSIY"}},
                     {"ACAZ", {"VHDQ", "AOTU", "WAMY"}},
                     {"ACBA", {"TVZD", "HLQR", "DOWV"}},
                     {"ACBB", {"ABIB", "LFJO", "ZQSG"}},
                     {"ACBC", {"UFPX", "WVQM", "MLYC"}},
                     {"ACBD", {"BRFU", "HBXU", "FAQM"}},
                     {"ACBE", {"JQFF", "GOJK", "ALHM"}},
                     {"ACBF", {"SSNP", "VHIH", "QMFD"}},
                     {"ADGK", {"PKUQ", "AEMI", "CUUL"}},
                     {"ADID", {"XDMY", "QHTP", "PBND"}},
                     {"AJIM", {"XQAQ", "WFLV", "AMBR"}},
                     {"ALRH", {"XDKE", "TDIH", "VLER"}},
                     {"AOKF", {"ZKSY", "KRPA", "QAZL"}},
                     {"AOPA", {"TTBL", "HHHA", "SADO"}},
                     {"AOPB", {"WFJJ", "ZQCN", "OMBM"}},
                     {"AOPC", {"CAKV", "AASR", "BXLX"}},
                     {"AOPD", {"PFAH", "NIUP", "MQIF"}},
                     {"AOPE", {"ZMYO", "BBQM", "UOUV"}},
                     {"APXY", {"JUFT", "VCUF", "TMSS"}},
                     {"ARBI", {"GFHA", "FBQD", "WRQR"}},
                     {"ASCT", {"CTRF", "LBBD", "YBND"}},
                     {"ASUA", {"IEIT", "JAIV", "MURN"}},
                     {"ASUB", {"QBBW", "RUGL", "UVVX"}},
                     {"ASUD", {"QLMM", "CRUA", "JSID"}},
                     {"ASUE", {"XLEN", "KECH", "HBGX"}},
                     {"ASUF", {"IVGE", "VNTM", "XELD"}},
                     {"ASUG", {"TSGW", "DZUL", "HFLO"}},
                     {"ASUH", {"DDNS", "OMSX", "VVWZ"}},
                     {"ASUJ", {"HJUL", "XWWL", "WSCY"}},
                     {"ASUK", {"RGUX", "OXBQ", "LDTL"}},
                     {"ASUL", {"ZJXP", "HGDC", "OFPC"}},
                     {"ASUN", {"ERAF", "HZQI", "JBDP"}},
                     {"ASUO", {"RCMY", "NLPS", "JOKT"}},
                     {"AYMH", {"BBMB", "VBWP", "BVTP"}},
                     {"BAQN", {"YJJJ", "LDCA", "QSJF"}},
                     {"BAUA", {"UWIF", "EOEW", "RPDR"}},
                     {"BCOL", {"YJDV", "GSIC", "BAUL"}},
                     {"BDIW", {"UDUG", "TRYQ", "PWFV"}},
                     {"BMNE", {"HLSA", "WXJQ", "TULR"}},
                     {"CBUY", {"POUW", "GHJY", "USXU"}},
                     {"CFUL", {"GIFL", "EDYW", "GOJE"}},
                     {"CLSF", {"OWOB", "RLJX", "OZWK"}},
                     {"CPPT", {"CQFF", "PCCZ", "HZEW"}},
                     {"CQFV", {"OJUW", "FCMY", "VCYR"}},
                     {"CQPQ", {"GATZ", "QAVU", "WRXC"}},
                     {"CSLV", {"BZSH", "ZDXA", "HGGZ"}},
                     {"CTIE", {"EURD", "HZJV", "WBJJ"}},
                     {"CYQR", {"XGJJ", "DRMC", "RUQD"}},
                     {"CYSQ", {"NHHD", "TAVM", "FHSA"}},
                     {"DBED", {"JUMI", "UTSY", "RXGS"}},
                     {"DEAA", {"HXUG", "BJUN", "IYTV"}},
                     {"DEAB", {"ARPQ", "MFRJ", "JWTH"}},
                     {"DEAC", {"DSMM", "IXET", "KQDV"}},
                     {"DEAD", {"QJXG", "AGGP", "GYQT"}},
                     {"DEAE", {"NZAS", "IHEL", "JSYE"}},
                     {"DEAF", {"TATK", "RWXF", "DQDT"}},
                     {"DEAG", {"JFEX", "CVLN", "UFWN"}},
                     {"DEAH", {"HRBU", "DJKF", "CMPZ"}},
                     {"DHAS", {"KEDN", "LUZR", "MHFN"}},
                     {"DISZ", {"PPAR", "VCPW", "NJKK"}},
                     {"DKJM", {"VRGL", "PZYF", "VBTW"}},
                     {"DRYI", {"LWTQ", "OLEY", "NWUA"}},
                     {"DUKI", {"FRGD", "SACE", "AAMW"}},
                     {"DVUG", {"HJHV", "KPAH", "DCQS"}},
                     {"DWCY", {"ZJQH", "JLCB", "QOAI"}},
                     {"DXVL", {"EBBY", "NMQL", "GTHA"}},
                     {"DXZT", {"WNSK", "WNDA", "DZWQ"}},
                     {"DYHT", {"YPAH", "NUKA", "EULJ"}},
                     {"EDFZ", {"VUMJ", "OMDW", "LCDF"}},
                     {"EDHM", {"NLAE", "JYDL", "BTWJ"}},
                     {"EGSC", {"DWAW", "FZRC", "PKWJ"}},
                     {"EKWL", {"PGWE", "JEHJ", "WQYW"}},
                     {"ELQA", {"GTJZ", "DTIH", "IXVN"}},
                     {"EOJH", {"GTAZ", "APYI", "UHAZ"}},
                     {"EUHF", {"RZPG", "FQYM", "IIBT"}},
                     {"EXCQ", {"LAOZ", "QTVX", "ZCLW"}},
                     {"FBTP", {"XLDO", "TLOM", "FYMS"}},
                     {"FCPG", {"WITB", "FOXJ", "YJQZ"}},
                     {"FCVS", {"HOBX", "YMDN", "GKTP"}},
                     {"FHYR", {"YKUD", "XTKX", "QFMD"}},
                     {"FIGU", {"VMWP", "SBFY", "IYUS"}},
                     {"FNVY", {"DLEJ", "DCNV", "XALG"}},
                     {"FQPJ", {"ZTQG", "ZNEO", "LYMZ"}},
                     {"FQZI", {"WPBA", "YZDA", "FXCI"}},
                     {"FRGW", {"ZPJY", "MYPP", "KQFE"}},
                     {"FRGW", {"ZPJY", "MYPP", "KQFE"}},
                     {"FSFR", {"ZDAR", "BERM", "COKX"}},
                     {"FSGY", {"PJQC", "RHZW", "POVI"}},
                     {"FWVK", {"MUTD", "GWKK", "SQSC"}},
                     {"FXEL", {"FVLL", "QOGS", "SVKH"}},
                     {"GBWE", {"DKLE", "OUDI", "VWJC"}},
                     {"GBXM", {"ONLL", "YBJS", "SOVT"}},
                     {"GFMQ", {"DRLH", "HVWY", "OYYM"}},
                     {"GFZE", {"HWCY", "NMLY", "QJJN"}},
                     {"GJZV", {"BUSA", "GIOS", "UYOM"}},
                     {"GLAR", {"RLLB", "UPQT", "OITD"}},
                     {"GMOO", {"GMRP", "QONY", "LOJX"}},
                     {"GNDV", {"UCEU", "GXKO", "HWDL"}},
                     {"GOKU", {"PRAG", "PQVF", "PIDI"}},
                     {"GVLR", {"HCKU", "VUNU", "FIRF"}},
                     {"GWDK", {"MQJZ", "WTMH", "ZOYJ"}},
                     {"GXYK", {"MLCI", "HWQK", "ERBL"}},
                     {"HDPY", {"JCUZ", "TMKK", "XMVQ"}},
                     {"HFAN", {"ZQNI", "RPSS", "VFHT"}},
                     {"HFKU", {"ILOF", "UXKA", "JQLI"}},
                     {"HFRG", {"YGYA", "IWET", "PSFN"}},
                     {"HHRN", {"IGZW", "ICRP", "QQKJ"}},
                     {"HKGT", {"EBMG", "KDZJ", "MELG"}},
                     {"HOMH", {"BXHI", "WXYD", "VRZY"}},
                     {"HOWM", {"MJNG", "XPYN", "IRWY"}},
                     {"HPZO", {"SICM", "XEGH", "TDJJ"}},
                     {"HPZP", {"NQDY", "QIMT", "QKAK"}},
                     {"HPZQ", {"XGER", "OLTF", "DVQA"}},
                     {"HPZR", {"ZAQH", "WPSK", "TCHA"}},
                     {"HPZS", {"QRFK", "SQGI", "VESI"}},
                     {"HPZT", {"IUCU", "WDAV", "LOLH"}},
                     {"HPZV", {"WAFN", "PQVW", "MJVM"}},
                     {"HPZW", {"TLLY", "WNPD", "XIFO"}},
                     {"HPZX", {"DNXN", "VHRG", "XTRZ"}},
                     {"HPZY", {"RAWP", "CNRC", "TPIA"}},
                     {"HPZZ", {"FJGP", "GMLT", "SZQX"}},
                     {"HRIZ", {"BJMA", "SKSL", "XBUU"}},
                     {"HTPV", {"LAEC", "NGRO", "BGEX"}},
                     {"HUIJ", {"EVJI", "RNMR", "JQZR"}},
                     {"HUUA", {"WTWZ", "DONX", "VRYO"}},
                     {"HVPU", {"HUTT", "JXOO", "HHMM"}},
                     {"HXZN", {"XTOL", "YHGP", "HMAG"}},
                     {"HYMD", {"LPEG", "UDVW", "KUBO"}},
                     {"HYPG", {"FSVQ", "PSWK", "RXGC"}},
                     {"HYZI", {"YBVF", "EUST", "WJVV"}},
                     {"IGRW", {"FORO", "KHEK", "BREP"}},
                     {"IHZG", {"MLLN", "EZTK", "GJEJ"}},
                     {"INUT", {"BRSN", "OJOO", "DWSP"}},
                     {"IULQ", {"ICMQ", "ZABS", "XMOU"}},
                     {"IXMM", {"DIJU", "LAUW", "XHLQ"}},
                     {"JBPA", {"VUZL", "XYPI", "XOWE"}},
                     {"JFZB", {"PFDC", "XJDX", "CPXX"}},
                     {"JICX", {"GUZK", "TIZA", "HTUW"}},
                     {"JLOF", {"IWFR", "CJHY", "DOPK"}},
                     {"JLRH", {"SAMJ", "GLJZ", "SKTN"}},
                     {"JOTV", {"QBNM", "NMWE", "IDTV"}},
                     {"JPUQ", {"OVKI", "AHZL", "YMJY"}},
                     {"JPZQ", {"CCBQ", "ABTW", "KFNE"}},
                     {"JQUD", {"CUTW", "DLJE", "DOON"}},
                     {"JRVR", {"WGPS", "YETD", "KBWB"}},
                     {"JVAN", {"FQPY", "WNTW", "XWXD"}},
                     {"JWGY", {"GMIM", "ZNPK", "RGAL"}},
                     {"JXIS", {"ZYZD", "TEIT", "ILLN"}},
                     {"JYXK", {"USZT", "XXPU", "LJHH"}},
                     {"KABJ", {"ISGW", "KOHG", "BPGB"}},
                     {"KBOV", {"PGBC", "IKKC", "AHSL"}},
                     {"KLKW", {"PIDD", "JIKU", "QTVN"}},
                     {"KOKS", {"XCGR", "ZFVG", "PPCB"}},
                     {"KXUH", {"RIFT", "DZUO", "ZSEI"}},
                     {"LASN", {"ILWC", "BQYG", "RROZ"}},
                     {"LEAA", {"DHUB", "OBDS", "YMSJ"}},
                     {"LEAB", {"LRHX", "EFFC", "SZFH"}},
                     {"LEAC", {"DMEA", "EXWD", "PBTU"}},
                     {"LEAD", {"QXLJ", "GDOH", "RJNB"}},
                     {"LEAE", {"QFVM", "GACH", "BMXB"}},
                     {"LEAF", {"KGXB", "OUVB", "GTLI"}},
                     {"LEAG", {"XTLW", "WLQO", "QVKP"}},
                     {"LEAH", {"QIDR", "XBTQ", "QYUO"}},
                     {"LEAI", {"KCSV", "PRBF", "FVDO"}},
                     {"LEAJ", {"OBPJ", "NJJS", "WOFS"}},
                     {"LEAK", {"CGWM", "ZLOS", "JGTD"}},
                     {"LEAL", {"EYPX", "SOCH", "PFPW"}},
                     {"LEAM", {"ZGEL", "KZQA", "PSAL"}},
                     {"LEAN", {"MEYH", "PXTT", "LFVK"}},
                     {"LEAO", {"MKOE", "YJSI", "QQMN"}},
                     {"LEAP", {"AEZG", "JOYE", "JHWK"}},
                     {"LGAA", {"YOGJ", "UGWO", "DAMU"}},
                     {"LIYT", {"CMCZ", "YUAJ", "MFPX"}},
                     {"LKSP", {"JVAJ", "ZERV", "YAYV"}},
                     {"LOEA", {"YYMF", "ZFDK", "KYJQ"}},
                     {"LOEB", {"HPPW", "LGZO", "NZIZ"}},
                     {"LOEC", {"FHUN", "VOTY", "IGUT"}},
                     {"LOED", {"WULK", "SEKY", "BWRY"}},
                     {"LOEE", {"GXPA", "MPFZ", "BAOI"}},
                     {"LOEF", {"DYDQ", "DBBP", "WNKL"}},
                     {"LOEG", {"WQVR", "VIMS", "XWTK"}},
                     {"LOEH", {"BLSW", "SRQW", "QJGU"}},
                     {"LOEI", {"WBJB", "HYVM", "QLRE"}},
                     {"LOEJ", {"JMPY", "RMIK", "CIPV"}},
                     {"LOEK", {"CFNY", "YTYX", "MFIU"}},
                     {"LOEL", {"KPTO", "AEKK", "PBSG"}},
                     {"LOEM", {"XQSP", "HYHH", "GQQF"}},
                     {"LOEN", {"FWRN", "XYNF", "TRTB"}},
                     {"LOEO", {"BNBG", "VSFX", "DMVB"}},
                     {"LOEP", {"KTKR", "JRUJ", "RYBH"}},
                     {"LOEQ", {"ZIEG", "IHSZ", "JXFB"}},
                     {"LOER", {"SOXE", "DIJG", "OHUN"}},
                     {"LOES", {"PAIY", "JXQE", "ZHPW"}},
                     {"LOET", {"CKLF", "TDYH", "HOES"}},
                     {"LOEU", {"LFQU", "ACJS", "DHDJ"}},
                     {"LOEV", {"YEZD", "CLSN", "JCDI"}},
                     {"LOEW", {"BYME", "GVQB", "ALXC"}},
                     {"LOEX", {"DTSE", "FFUO", "GOWI"}},
                     {"LOEY", {"UJLK", "PIZK", "ASMT"}},
                     {"LOEZ", {"LDCF", "MYHV", "OZLH"}},
                     {"LOFA", {"ZHIB", "KAWM", "RSJW"}},
                     {"LOFB", {"YIHY", "QXQD", "GDXE"}},
                     {"LOFC", {"IDAK", "FKMQ", "MHSL"}},
                     {"LOFD", {"PVSV", "WAEK", "JASG"}},
                     {"LOFE", {"BLWI", "ZXBI", "DMET"}},
                     {"LOFF", {"XVWG", "LLIC", "AAIM"}},
                     {"LOFG", {"SMQP", "RFJQ", "HDWV"}},
                     {"LOFH", {"LBUJ", "DGLT", "EHHF"}},
                     {"LOFI", {"DJQQ", "QKSW", "HWAJ"}},
                     {"LOFJ", {"WJXN", "IDHY", "GKCO"}},
                     {"LOFK", {"VGKG", "SQCD", "SLUY"}},
                     {"LOFL", {"UYXZ", "AZKR", "RDLY"}},
                     {"LOFM", {"CVLJ", "UCLO", "PADI"}},
                     {"LOFO", {"DMJS", "PYYK", "SKQO"}},
                     {"LOFP", {"DGNA", "ZXHN", "ARBG"}},
                     {"LOFQ", {"QGKD", "PRZN", "IPEQ"}},
                     {"LOFR", {"ZOWO", "ZSTS", "JXBM"}},
                     {"LOFS", {"QEKW", "TTKC", "MQUP"}},
                     {"LOFT", {"YXFQ", "QFJS", "BNPB"}},
                     {"LOFU", {"KSMM", "TJWT", "VBMW"}},
                     {"LOFV", {"NDDC", "BTQU", "HUZE"}},
                     {"LOFW", {"VLDG", "IPIN", "JAVJ"}},
                     {"LOFX", {"UWQQ", "IGRC", "GRUT"}},
                     {"LOFY", {"SCIO", "HJKR", "TBOP"}},
                     {"LOFZ", {"VSSO", "WSDE", "BHWL"}},
                     {"LOGB", {"BWRU", "YLCD", "RWLB"}},
                     {"LOGC", {"YUDR", "THXM", "NBVM"}},
                     {"LOGD", {"JFPI", "RDCY", "DGJD"}},
                     {"LOGE", {"ITOL", "GZEC", "INSH"}},
                     {"LOGF", {"OWNI", "ECYV", "JEFV"}},
                     {"LOGH", {"RTVE", "EJJV", "DNTX"}},
                     {"LOGI", {"OEYI", "IKUX", "TCEI"}},
                     {"LULQ", {"DEHI", "QYXC", "KAGT"}},
                     {"LYLN", {"XXWY", "JEUV", "RSOC"}},
                     {"MAII", {"EOHR", "XZOT", "VJJS"}},
                     {"MCDN", {"BAOV", "GLVV", "XHGO"}},
                     {"MCOO", {"IPNW", "CRSK", "QTAX"}},
                     {"MDPZ", {"AHBA", "ENTF", "IIMC"}},
                     {"MEXL", {"JFMC", "LBVP", "DERH"}},
                     {"MNZG", {"PPTP", "OFXE", "ROJJ"}},
                     {"MOIP", {"HCCZ", "PXCU", "MROE"}},
                     {"MQUZ", {"MFAZ", "GBNW", "MRMS"}},
                     {"MRFF", {"VHZM", "CBXS", "WHGR"}},
                     {"MXEQ", {"EKJV", "UWUR", "CPES"}},
                     {"MXUY", {"IRZH", "ADQR", "PCST"}},
                     {"MYQR", {"VMHK", "QHCZ", "HMFN"}},
                     {"MZVS", {"VUZM", "RIDT", "URTS"}},
                     {"NAMM", {"BFSS", "BKVK", "EBDV"}},
                     {"NBQS", {"KMJF", "MFWA", "UWRX"}},
                     {"NISD", {"MISA", "YDPG", "NCLQ"}},
                     {"NMOG", {"UYQU", "ZWTV", "TQFQ"}},
                     {"NOMD", {"GZLV", "UNZR", "FVOP"}},
                     {"NOMD", {"GZLV", "UNZR", "FVOP"}},
                     {"NPEC", {"BMGD", "YETH", "XAWJ"}},
                     {"NSXI", {"VYQS", "HGFQ", "SLFL"}},
                     {"NZRH", {"NOUG", "UDYG", "ZGAU"}},
                     {"OFPE", {"YFOO", "UIGY", "PFGZ"}},
                     {"OIFF", {"MLXE", "KFNX", "CRAQ"}},
                     {"OKWC", {"RGFB", "UPFP", "HUVK"}},
                     {"OYZI", {"WDBC", "NKZT", "QJZD"}},
                     {"PAZD", {"VARX", "KZSU", "WPLH"}},
                     {"PEVA", {"RBMX", "IBPY", "ALNV"}},
                     {"PGQF", {"USPJ", "SFKO", "KNBH"}},
                     {"PHYB", {"EGXD", "KHYC", "QUPU"}},
                     {"PIGM", {"FEBY", "YTML", "VFLZ"}},
                     {"PLKQ", {"EXXM", "LBZT", "SPDN"}},
                     {"PRYU", {"QFZV", "TZXL", "EPRT"}},
                     {"PWFL", {"WGJQ", "KMBF", "UKJV"}},
                     {"PXDO", {"ZXCF", "TQWC", "HOAL"}},
                     {"QAPN", {"EMNZ", "SJTH", "HJKU"}},
                     {"QBJC", {"WAQG", "MSEN", "FQYE"}},
                     {"QBTA", {"UDQV", "UIZV", "SGMN"}},
                     {"QCDF", {"HOUC", "PKTP", "APSD"}},
                     {"QJHH", {"TIHM", "SOII", "SXVL"}},
                     {"QKTA", {"USGV", "UPMS", "ZVTZ"}},
                     {"QLDV", {"BJRT", "ZICU", "URBL"}},
                     {"QLWW", {"LNZB", "JTVW", "XVCX"}},
                     {"QNDA", {"VFMY", "KTBL", "UOJY"}},
                     {"QQFU", {"ZUKV", "QBAU", "SIID"}},
                     {"QSIM", {"ZCML", "LEPJ", "QQEM"}},
                     {"QTMI", {"YMOW", "FZIR", "YKGT"}},
                     {"QYGU", {"FYBR", "QLFJ", "OLRV"}},
                     {"QZUX", {"HNBM", "BUJY", "FFDE"}},
                     {"RGDH", {"YWKM", "ZBAR", "RMQQ"}},
                     {"RGNF", {"SDGJ", "KEWA", "GITE"}},
                     {"RHDN", {"MGVK", "EQPB", "UAHY"}},
                     {"RIKG", {"VRBT", "LEPX", "VWIV"}},
                     {"RKRB", {"OPOY", "QMZZ", "FAGR"}},
                     {"RLGE", {"NTKV", "LOTA", "MJVG"}},
                     {"RVRM", {"MZJU", "IGXP", "DSJP"}},
                     {"RXGN", {"WHNA", "DWVK", "FRWP"}},
                     {"SBGV", {"ZNIN", "ZVZV", "BPJY"}},
                     {"SHAN", {"OERN", "XNHK", "GVYX"}},
                     {"SKIW", {"CLPF", "OTYY", "ZJVP"}},
                     {"SMAC", {"FDEX", "ZFXY", "DJMW"}},
                     {"SMAD", {"AADC", "URZK", "UBVE"}},
                     {"SMAE", {"SUUV", "QXWL", "LYKX"}},
                     {"SMAF", {"HKPA", "NFCE", "UBOP"}},
                     {"SMAG", {"DPGH", "PQFA", "ROEP"}},
                     {"SMAH", {"EXLB", "YYYY", "LLLA"}},
                     {"SMAI", {"PPDO", "ISMM", "BKNT"}},
                     {"SMAJ", {"PVCB", "UCIK", "XVBK"}},
                     {"SMAK", {"WOMZ", "OHAX", "JSTF"}},
                     {"SMAL", {"OWLX", "YXSA", "TXJR"}},
                     {"SOCA", {"AJGR", "IYZW", "NPDX"}},
                     {"SSLV", {"IUFZ", "NTYF", "TWGJ"}},
                     {"SSVR", {"NZKV", "NGLW", "LDCH"}},
                     {"SUCA", {"JSZT", "IBUF", "HMEZ"}},
                     {"SWLP", {"GLDC", "WZKJ", "GTXT"}},
                     {"TAAB", {"ZBMY", "NYDT", "CXYZ"}},
                     {"TAAC", {"YBVP", "RXXN", "HMDY"}},
                     {"TFIY", {"RVUF", "DHKE", "GFPK"}},
                     {"TJKH", {"ZHMG", "RBXM", "VIVU"}},
                     {"TKER", {"KOSM", "IUCL", "LIIM"}},
                     {"TKZT", {"KWCM", "APLN", "STGO"}},
                     {"TMSE", {"PSOE", "RFGT", "DVAS"}},
                     {"TMTX", {"CNAW", "BEDK", "HGOT"}},
                     {"TPHN", {"DGRC", "EDPM", "FLCE"}},
                     {"TQAU", {"PUVO", "MASK", "LJBB"}},
                     {"TSNX", {"NLLF", "DJAG", "FBBO"}},
                     {"TVRZ", {"XWBR", "VSOG", "WGJH"}},
                     {"TXMN", {"WTVY", "GJTZ", "KMRI"}},
                     {"TZIV", {"XWTU", "JFLV", "JLEU"}},
                     {"UGAY", {"YDHM", "HVCY", "ILHO"}},
                     {"UMAU", {"FKAK", "JCTZ", "GDUU"}},
                     {"UQDN", {"LWWF", "SCDS", "IKKY"}},
                     {"UQUC", {"YLQO", "IDZV", "PXQW"}},
                     {"URZD", {"QDAL", "YLWB", "XCCP"}},
                     {"UUCL", {"HELE", "KEDZ", "ZAAI"}},
                     {"VEUT", {"JDFA", "ALIR", "DDJM"}},
                     {"VGYW", {"AAXS", "SHZF", "HYJU"}},
                     {"VHUH", {"JYDF", "SFJY", "JMBU"}},
                     {"VICR", {"VNCX", "OLSV", "YCZO"}},
                     {"VRWC", {"OGMF", "GYJX", "NOBB"}},
                     {"VVUC", {"WQCU", "YUMW", "YHYC"}},
                     {"VYRC", {"VKSO", "NKTO", "ZPZX"}},
                     {"VZMB", {"YCKT", "WSPC", "SHYP"}},
                     {"WBZQ", {"LAYK", "LQDM", "QBFV"}},
                     {"WCLL", {"DALK", "WPRA", "TPTP"}},
                     {"WFIQ", {"KKHX", "UTHS", "HDSP"}},
                     {"WJOZ", {"BASQ", "BRTL", "CQAV"}},
                     {"WMVU", {"GMMR", "AVVS", "IMDF"}},
                     {"WNNA", {"ERXU", "TWMI", "ZOER"}},
                     {"WVRW", {"GJGN", "QQFA", "AGVP"}},
                     {"WXZG", {"IUGR", "JOEE", "PTHY"}},
                     {"XBWL", {"IQEI", "JEGU", "QSKW"}},
                     {"XFUX", {"UHAM", "NEHU", "SHMG"}},
                     {"XHVI", {"KVWL", "GQOJ", "JLLW"}},
                     {"XOGA", {"BIWO", "JPWZ", "YYDG"}},
                     {"XOKS", {"DEVR", "YKLR", "QYBF"}},
                     {"XVTK", {"TMUU", "BTWW", "THQH"}},
                     {"XVYQ", {"UAVB", "OEMI", "VQVK"}},
                     {"XWJE", {"KDZI", "IYPJ", "ERIM"}},
                     {"YAVR", {"DHAY", "KBWN", "BBPJ"}},
                     {"YFVF", {"NPWS", "PUZZ", "TTCZ"}},
                     {"YGHA", {"BMDT", "AUXW", "GYPE"}},
                     {"YHYU", {"CDLM", "QDXQ", "HPTE"}},
                     {"YLRO", {"ZQHU", "SFQD", "YNOL"}},
                     {"YMJL", {"LBTX", "YPBE", "LHMF"}},
                     {"YMMU", {"ZVIA", "CFKN", "ERLO"}},
                     {"YPCE", {"CCCC", "VHQK", "PYBL"}},
                     {"YPPO", {"TRIY", "TMUA", "AMPD"}},
                     {"YQWT", {"KEBH", "PAMG", "ACOF"}},
                     {"YTGY", {"PRXN", "QEZG", "FOSO"}},
                     {"YVRQ", {"LBMS", "AKKB", "UFNF"}},
                     {"YXBK", {"VKAU", "HUNQ", "AFRP"}},
                     {"YXED", {"KDUD", "MTUI", "WLHI"}},
                     {"ZBCF", {"BDTW", "MIQF", "VUNL"}},
                     {"ZDKS", {"UBRP", "AWQF", "GOVG"}},
                     {"ZFCZ", {"JQUA", "SEEH", "RJVV"}},
                     {"ZHKO", {"PEIC", "UYOS", "NVYS"}},
                     {"ZIWS", {"GSAE", "JJUF", "ZPRA"}},
                     {"ZJLO", {"HLMS", "OHWG", "HMAL"}},
                     {"ZKJH", {"OBDQ", "OUAQ", "SPYY"}},
                     {"ZLBC", {"DJCJ", "HNGZ", "IRYZ"}},
                     {"ZSKM", {"JPEZ", "FTUS", "ZFUF"}},
                     {"ZSLY", {"TQED", "GKPV", "BHWH"}},
                     {"ZZAB", {"WVIK", "IUXK", "ZCIK"}},
                     {"ZZAC", {"MBDD", "SMUW", "JEIY"}},
                     {"ZZAD", {"KSTH", "CBJY", "TSID"}},
                     {"ZZAF", {"OTWH", "RRNB", "VNXA"}},
                     {"ZZTB", {"MXQT", "JUUX", "FMFR"}}});

  const auto it = kBrandCodeMap->find(static_brand_code);
  if (it == kBrandCodeMap->end())
    return static_brand_code;
  const auto& entry = it->second;
  // An empty value indicates the device is not enrolled.
  if (!market_segment.has_value())
    return entry.unenrolled_brand_code;

  switch (market_segment.value()) {
    case policy::MarketSegment::EDUCATION:
      return entry.education_enrolled_brand_code;
    case policy::MarketSegment::ENTERPRISE:
    case policy::MarketSegment::UNKNOWN:
      // If the device is enrolled but market segment is unknown, it's fine to
      // treat it as enterprise enrolled.
      return entry.enterprise_enrolled_brand_code;
  }
  NOTREACHED();
  return static_brand_code;
}

}  // namespace chromeos
}  // namespace google_brand
