// Copyright 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/extensions/extension_tab_util.h"

#include "base/macros.h"
#include "chrome/browser/extensions/api/tabs/tabs_constants.h"
#include "chrome/common/extensions/api/tabs.h"
#include "extensions/common/extension.h"
#include "extensions/common/extension_builder.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace extensions {

namespace {

class ExtensionTabUtilTestDelegate : public ExtensionTabUtil::Delegate {
 public:
  ExtensionTabUtilTestDelegate() {}
  ~ExtensionTabUtilTestDelegate() override {}

  // ExtensionTabUtil::Delegate
  ExtensionTabUtil::ScrubTabBehaviorType GetScrubTabBehavior(
      const Extension* extension) override {
    return ExtensionTabUtil::kScrubTabUrlToOrigin;
  }

 private:
  DISALLOW_COPY_AND_ASSIGN(ExtensionTabUtilTestDelegate);
};

}  // namespace

// Test that the custom GetScrubTabBehavior delegate works - in this test it
// always returns kScrubTabUrlToOrigin
TEST(ExtensionTabUtilTest, Delegate) {
  ExtensionTabUtil::SetPlatformDelegate(
      std::make_unique<ExtensionTabUtilTestDelegate>());

  // Build an extension that passes the permission checks for the generic
  // GetScrubTabBehavior
  auto extension = ExtensionBuilder("test").AddPermission("tabs").Build();

  ExtensionTabUtil::ScrubTabBehavior scrub_tab_behavior =
      ExtensionTabUtil::GetScrubTabBehavior(
          extension.get(), Feature::Context::UNSPECIFIED_CONTEXT,
          GURL("http://www.google.com"));
  EXPECT_EQ(ExtensionTabUtil::kScrubTabUrlToOrigin,
            scrub_tab_behavior.committed_info);
  EXPECT_EQ(ExtensionTabUtil::kScrubTabUrlToOrigin,
            scrub_tab_behavior.pending_info);

  // Unset the delegate.
  ExtensionTabUtil::SetPlatformDelegate(nullptr);
}

TEST(ExtensionTabUtilTest, ScrubTabBehaviorForTabsPermission) {
  auto extension = ExtensionBuilder("Extension with tabs permission")
                       .AddPermission("tabs")
                       .Build();
  ExtensionTabUtil::ScrubTabBehavior scrub_tab_behavior =
      ExtensionTabUtil::GetScrubTabBehavior(
          extension.get(), Feature::Context::UNSPECIFIED_CONTEXT,
          GURL("http://www.google.com"));
  EXPECT_EQ(ExtensionTabUtil::kDontScrubTab, scrub_tab_behavior.committed_info);
  EXPECT_EQ(ExtensionTabUtil::kDontScrubTab, scrub_tab_behavior.pending_info);
}

TEST(ExtensionTabUtilTest, ScrubTabBehaviorForNoPermission) {
  auto extension = ExtensionBuilder("Extension with no permissions").Build();
  ExtensionTabUtil::ScrubTabBehavior scrub_tab_behavior =
      ExtensionTabUtil::GetScrubTabBehavior(
          extension.get(), Feature::Context::UNSPECIFIED_CONTEXT,
          GURL("http://www.google.com"));
  EXPECT_EQ(ExtensionTabUtil::kScrubTabFully,
            scrub_tab_behavior.committed_info);
  EXPECT_EQ(ExtensionTabUtil::kScrubTabFully, scrub_tab_behavior.pending_info);
}

TEST(ExtensionTabUtilTest, ScrubTabBehaviorForHostPermission) {
  auto extension = ExtensionBuilder("Extension with host permission")
                       .AddPermission("*://www.google.com/*")
                       .Build();
  ExtensionTabUtil::ScrubTabBehavior scrub_tab_behavior =
      ExtensionTabUtil::GetScrubTabBehavior(
          extension.get(), Feature::Context::UNSPECIFIED_CONTEXT,
          GURL("http://www.google.com/some/path"));
  EXPECT_EQ(ExtensionTabUtil::kDontScrubTab, scrub_tab_behavior.committed_info);
  EXPECT_EQ(ExtensionTabUtil::kDontScrubTab, scrub_tab_behavior.pending_info);
}

TEST(ExtensionTabUtilTest, ScrubTabBehaviorForNoExtension) {
  ExtensionTabUtil::ScrubTabBehavior scrub_tab_behavior =
      ExtensionTabUtil::GetScrubTabBehavior(
          nullptr, Feature::Context::UNSPECIFIED_CONTEXT,
          GURL("http://www.google.com"));
  EXPECT_EQ(ExtensionTabUtil::kScrubTabFully,
            scrub_tab_behavior.committed_info);
  EXPECT_EQ(ExtensionTabUtil::kScrubTabFully, scrub_tab_behavior.pending_info);
}

TEST(ExtensionTabUtilTest, ScrubTabBehaviorForWebUI) {
  ExtensionTabUtil::ScrubTabBehavior scrub_tab_behavior =
      ExtensionTabUtil::GetScrubTabBehavior(nullptr,
                                            Feature::Context::WEBUI_CONTEXT,
                                            GURL("http://www.google.com"));
  EXPECT_EQ(ExtensionTabUtil::kDontScrubTab, scrub_tab_behavior.committed_info);
  EXPECT_EQ(ExtensionTabUtil::kDontScrubTab, scrub_tab_behavior.pending_info);
}

TEST(ExtensionTabUtilTest, ScrubTabBehaviorForWebUIUntrusted) {
  ExtensionTabUtil::ScrubTabBehavior scrub_tab_behavior =
      ExtensionTabUtil::GetScrubTabBehavior(
          nullptr, Feature::Context::WEBUI_UNTRUSTED_CONTEXT,
          GURL("http://www.google.com"));
  EXPECT_EQ(ExtensionTabUtil::kScrubTabFully,
            scrub_tab_behavior.committed_info);
  EXPECT_EQ(ExtensionTabUtil::kScrubTabFully, scrub_tab_behavior.pending_info);
}

TEST(ExtensionTabUtilTest, ResolvePossiblyRelativeURL) {
  auto extension = ExtensionBuilder("test").Build();
  EXPECT_EQ(ExtensionTabUtil::ResolvePossiblyRelativeURL(
                "http://example.com/path", extension.get()),
            GURL("http://example.com/path"));
  EXPECT_EQ(
      ExtensionTabUtil::ResolvePossiblyRelativeURL("path", extension.get()),
      GURL("chrome-extension://jpignaibiiemhngfjkcpokkamffknabf/path"));
  EXPECT_EQ(ExtensionTabUtil::ResolvePossiblyRelativeURL("path", nullptr),
            GURL("path"));
}

TEST(ExtensionTabUtilTest, PrepareURLForNavigation) {
  auto extension = ExtensionBuilder("test").Build();
  // A fully qualified URL should return the same URL.
  {
    const std::string kTestUrl("http://google.com");
    std::string error;
    GURL url;
    EXPECT_TRUE(ExtensionTabUtil::PrepareURLForNavigation(
        kTestUrl, extension.get(), &url, &error));
    EXPECT_EQ(GURL(kTestUrl), url);
    EXPECT_EQ("", error);
  }
  // A relative path should return a URL relative to the extension's base URL.
  {
    const std::string kTestPath("foo");
    std::string error;
    GURL url;
    EXPECT_TRUE(ExtensionTabUtil::PrepareURLForNavigation(
        kTestPath, extension.get(), &url, &error));
    EXPECT_EQ(extension->GetResourceURL(kTestPath), url);
    EXPECT_EQ("", error);
  }
  // A kill URL should return false and set the error. There are several
  // different potential kill URLs and this just checks one of them.
  {
    const std::string kKillURL("chrome://crash");
    std::string error;
    GURL url;
    EXPECT_FALSE(ExtensionTabUtil::PrepareURLForNavigation(
        kKillURL, extension.get(), &url, &error));
    EXPECT_EQ(tabs_constants::kNoCrashBrowserError, error);
  }
}

}  // namespace extensions
