// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/android/reading_list/reading_list_notification_service_factory.h"

#include <memory>

#include "chrome/browser/notifications/scheduler/notification_schedule_service_factory.h"
#include "chrome/browser/profiles/incognito_helpers.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/profiles/profile_key.h"
#include "chrome/browser/reading_list/android/reading_list_notification_service.h"
#include "chrome/browser/ui/read_later/reading_list_model_factory.h"
#include "components/keyed_service/content/browser_context_dependency_manager.h"
#include "components/reading_list/features/reading_list_switches.h"

// static
ReadingListNotificationServiceFactory*
ReadingListNotificationServiceFactory::GetInstance() {
  return base::Singleton<ReadingListNotificationServiceFactory>::get();
}

// static
ReadingListNotificationService*
ReadingListNotificationServiceFactory::GetForBrowserContext(
    content::BrowserContext* context) {
  return static_cast<ReadingListNotificationService*>(
      GetInstance()->GetServiceForBrowserContext(context, /*create=*/true));
}

ReadingListNotificationServiceFactory::ReadingListNotificationServiceFactory()
    : BrowserContextKeyedServiceFactory(
          "ReadingListNotificationService",
          BrowserContextDependencyManager::GetInstance()) {
  DependsOn(ReadingListModelFactory::GetInstance());
  DependsOn(NotificationScheduleServiceFactory::GetInstance());
}

ReadingListNotificationServiceFactory::
    ~ReadingListNotificationServiceFactory() = default;

KeyedService* ReadingListNotificationServiceFactory::BuildServiceInstanceFor(
    content::BrowserContext* context) const {
  auto* reading_list_model =
      ReadingListModelFactory::GetForBrowserContext(context);
  Profile* profile = Profile::FromBrowserContext(context);
  auto* notification_scheduler =
      NotificationScheduleServiceFactory::GetForKey(profile->GetProfileKey());
  return new ReadingListNotificationService(reading_list_model,
                                            notification_scheduler);
}

content::BrowserContext*
ReadingListNotificationServiceFactory::GetBrowserContextToUse(
    content::BrowserContext* context) const {
  return chrome::GetBrowserContextRedirectedInIncognito(context);
}
