// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef BASE_CONTAINERS_FIXED_FLAT_SET_H_
#define BASE_CONTAINERS_FIXED_FLAT_SET_H_

#include <array>
#include <functional>

#include "base/check.h"
#include "base/containers/flat_set.h"
#include "base/containers/flat_tree.h"

namespace base {

// fixed_flat_set is a immutable container with a std::set-like interface that
// stores its contents in a sorted std::array.
//
// It is a special case of base::flat_set, and mostly useful as a look-up table.
//
// Please see //base/containers/README.md for an overview of which container
// to select.
//
// QUICK REFERENCE
//
// Most of the core functionality is inherited from flat_tree. Please see
// flat_tree.h for more details for most of these functions. As a quick
// reference, the functions available are:
//
// Constructors (inputs need to be sorted):
//   fixed_flat_set(const fixed_flat_set&);
//   fixed_flat_set(sorted_unique_t,
//                  const container_type& items,
//                  const Compare& compare = Compare());
//
// Size management functions:
//   size_t size() const;
//   size_t max_size() const;
//   bool   empty() const;
//
// Iterator functions:
//   const_iterator         begin() const;
//   const_iterator         cbegin() const;
//   const_iterator         end() const;
//   const_iterator         cend() const;
//   const reverse_iterator rbegin() const;
//   const_reverse_iterator crbegin() const;
//   const_reverse_iterator rend() const;
//   const_reverse_iterator crend() const;
//
// Comparators (see std::set documentation).
//   key_compare   key_comp() const;
//   value_compare value_comp() const;
//
// Search functions:
//   template <typename K> size_t                   count(const K&) const;
//   template <typename K> const_iterator           find(const K&) const;
//   template <typename K> bool                     contains(const K&) const;
//   template <typename K>
//       pair<const_iterator, const_iterator>       equal_range(K&) const;
//   template <typename K> const_iterator           lower_bound(const K&) const;
//   template <typename K> const_iterator           upper_bound(const K&) const;
//
// Non-member operators:
//   bool operator==(const fixed_flat_set&, const fixed_flat_set&);
//   bool operator!=(const fixed_flat_set&, const fixed_flat_set&);
//   bool operator<(const fixed_flat_set&, const fixed_flat_set&);
//   bool operator>(const fixed_flat_set&, const fixed_flat_set&);
//   bool operator>=(const fixed_flat_set&, const fixed_flat_set&);
//   bool operator<=(const fixed_flat_set&, const fixed_flat_set&);
//
template <class Key, size_t N, class Compare = std::less<>>
using fixed_flat_set = base::flat_set<Key, Compare, std::array<const Key, N>>;

// Utility function to simplify constructing a fixed_flat_set from a fixed list
// of keys. Requires that the passed in `data` is sorted and unique.
//
// Example usage:
//   constexpr auto kSet = base::MakeFixedFlatSet({1, 2, 3, 4});
template <class Key, size_t N, class Compare = std::less<>>
constexpr fixed_flat_set<Key, N, Compare> MakeFixedFlatSet(
    const Key (&data)[N],
    const Compare& comp = Compare()) {
  CHECK(internal::is_sorted_and_unique(data, comp));
  // Specify the value_type explicitly to ensure that the returned array has
  // immutable keys.
  return fixed_flat_set<Key, N, Compare>(
      sorted_unique, internal::ToArray<const Key>(data), comp);
}

}  // namespace base

#endif  // BASE_CONTAINERS_FIXED_FLAT_SET_H_
