# Copyright 2019 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Enforces command buffer autogen matches script output.

See http://dev.chromium.org/developers/how-tos/depottools/presubmit-scripts
for more details on the presubmit API built into depot_tools.
"""

import os.path


def CommonChecks(input_api, output_api):
  gles2_cmd_buffer_files = input_api.AffectedFiles(
      file_filter=lambda x: os.path.basename(x.LocalPath()) in [
          'build_cmd_buffer_lib.py', 'build_gles2_cmd_buffer.py',
          'gles2_cmd_buffer_functions.txt'])
  raster_cmd_buffer_files = input_api.AffectedFiles(
      file_filter=lambda x: os.path.basename(x.LocalPath()) in [
          'build_cmd_buffer_lib.py', 'build_raster_cmd_buffer.py',
          'raster_cmd_buffer_functions.txt'])
  webgpu_cmd_buffer_files = input_api.AffectedFiles(
      file_filter=lambda x: os.path.basename(x.LocalPath()) in [
          'build_cmd_buffer_lib.py', 'build_webgpu_cmd_buffer.py',
          'webgpu_cmd_buffer_functions.txt'])

  autogen_files = input_api.AffectedFiles(
      file_filter=lambda x: x.LocalPath().endswith('_autogen.h'))

  # Use input_api.change.AffectedFiles() to get files outside this directory.
  external_gl_headers = input_api.change.AffectedFiles(
      file_filter=lambda x: os.path.basename(x.LocalPath()) in [
          'gl2.h', 'gl2ext.h', 'gl3.h', 'gl31.h', 'gl2chromium.h',
          'gl2extchromium.h'
      ])

  messages = []

  if (len(autogen_files) > 0 and len(gles2_cmd_buffer_files) == 0 and
      len(external_gl_headers) == 0 and len(raster_cmd_buffer_files) == 0 and
      len(webgpu_cmd_buffer_files) == 0):
    long_text = 'Changed files:\n'
    for file in autogen_files:
      long_text += file.LocalPath() + '\n'
    long_text += '\n'
    messages.append(output_api.PresubmitError(
        'Command buffer autogenerated files changed but generators did not.',
        long_text=long_text))

  with input_api.temporary_directory() as temp_dir:
    commands = []
    if len(gles2_cmd_buffer_files) > 0:
      commands.append(input_api.Command(name='build_gles2_cmd_buffer',
          cmd=[input_api.python_executable, 'build_gles2_cmd_buffer.py',
               '--check', '--output-dir=' + temp_dir],
          kwargs={}, message=output_api.PresubmitError))
    if len(raster_cmd_buffer_files) > 0:
      commands.append(input_api.Command(name='build_raster_cmd_buffer',
          cmd=[input_api.python_executable, 'build_raster_cmd_buffer.py',
               '--check', '--output-dir=' + temp_dir],
          kwargs={}, message=output_api.PresubmitError))
    if len(webgpu_cmd_buffer_files) > 0:
      commands.append(input_api.Command(name='build_webgpu_cmd_buffer',
          cmd=[input_api.python_executable, 'build_webgpu_cmd_buffer.py',
               '--check', '--output-dir=' + temp_dir],
          kwargs={}, message=output_api.PresubmitError))
    if len(commands) > 0:
      messages.extend(input_api.RunTests(commands))

  return messages


def CheckChangeOnUpload(input_api, output_api):
  return CommonChecks(input_api, output_api)


def CheckChangeOnCommit(input_api, output_api):
  return CommonChecks(input_api, output_api)
