// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_NET_DNS_PROBE_RUNNER_H_
#define CHROME_BROWSER_NET_DNS_PROBE_RUNNER_H_

#include <memory>

#include "base/bind.h"
#include "base/callback.h"
#include "base/macros.h"
#include "base/sequence_checker.h"
#include "mojo/public/cpp/bindings/binding.h"
#include "services/network/public/cpp/resolve_host_client_base.h"
#include "services/network/public/mojom/host_resolver.mojom.h"

namespace network {
namespace mojom {
class NetworkContext;
}
}  // namespace network

namespace chrome_browser_net {

// Runs DNS probes using a HostResolver and evaluates the responses.
// (Currently requests A records for google.com and expects at least one IP
// address in the response.)
// Used by DnsProbeService to probe the system and public DNS configurations.
class DnsProbeRunner : public network::ResolveHostClientBase {
 public:
  static const char kKnownGoodHostname[];

  using NetworkContextGetter =
      base::RepeatingCallback<network::mojom::NetworkContext*(void)>;

  // Used in histograms; add new entries at the bottom, and don't remove any.
  enum Result {
    UNKNOWN,
    CORRECT,     // Response contains at least one A record.
    INCORRECT,   // Response claimed success but included no A records.
    FAILING,     // Response included an error or was malformed.
    UNREACHABLE  // No response received (timeout, network unreachable, etc.).
  };

  // Creates a probe runner that will use |dns_config_overrides| for the dns
  // configuration and will use |network_context_getter| to get the
  // NetworkContext to create the HostResolver.  The |network_context_getter|
  // may be called multiple times.
  DnsProbeRunner(net::DnsConfigOverrides dns_config_overrides,
                 const NetworkContextGetter& network_context_getter);
  ~DnsProbeRunner() override;

  // Starts a probe. |callback| will be called asynchronously when the result
  // is ready, and will not be called if the DnsProbeRunner is destroyed before
  // the probe finishes. Must not be called again until the callback is called,
  // but may be called during the callback.
  void RunProbe(base::OnceClosure callback);

  // Returns true if a probe is running.  Guaranteed to return true after
  // RunProbe returns, and false during and after the callback.
  bool IsRunning() const;

  // Returns the result of the last probe.
  Result result() const {
    DCHECK_CALLED_ON_VALID_SEQUENCE(sequence_checker_);
    return result_;
  }

  // network::ResolveHostClientBase impl:
  void OnComplete(
      int32_t result,
      const base::Optional<net::AddressList>& resolved_addresses) override;

 private:
  void CreateHostResolver();
  void OnMojoConnectionError();

  mojo::Binding<network::mojom::ResolveHostClient> binding_;

  net::DnsConfigOverrides dns_config_overrides_;
  NetworkContextGetter network_context_getter_;

  network::mojom::HostResolverPtr host_resolver_;

  // The callback passed to |RunProbe|.  Cleared right before calling the
  // callback.
  base::OnceClosure callback_;

  Result result_;

  SEQUENCE_CHECKER(sequence_checker_);

  DISALLOW_COPY_AND_ASSIGN(DnsProbeRunner);
};

}  // namespace chrome_browser_net

#endif  // CHROME_BROWSER_NET_DNS_PROBE_RUNNER_H_
