/*$
 Copyright (C) 2013-2018 Azel.

 This file is part of AzPainter.

 AzPainter is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 AzPainter is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/

/*****************************************
 * AppCursor
 *
 * アプリケーション用カーソル
 *****************************************/
/*
 * - 各スロットに、必要なカーソルを作成してセット。
 *   同じスロットで同じカーソルが要求された場合は再作成せずに済む。
 * - 描画用矢印と砂時計カーソルは頻繁に使うので、初期化時に作成済み。
 */

#include "mDef.h"
#include "mCursor.h"

#include "AppCursor.h"


//---------------------------

//描画
static const unsigned char g_cursor_draw[] = {
9,12, 0,0,
0x01, 0x00, 0x03, 0x00, 0x05, 0x00, 0x09, 0x00, 0x11, 0x00, 0x21, 0x00,
0x41, 0x00, 0x81, 0x00, 0xf9, 0x01, 0x05, 0x00, 0x03, 0x00, 0x01, 0x00,
0x01, 0x00, 0x03, 0x00, 0x07, 0x00, 0x0f, 0x00, 0x1f, 0x00, 0x3f, 0x00,
0x7f, 0x00, 0xff, 0x00, 0xff, 0x01, 0x07, 0x00, 0x03, 0x00, 0x01, 0x00 };

//ハンド
static const unsigned char g_cursor_hand[] = {
18,17, 9,9,
0x00, 0x02, 0x00, 0x20, 0x25, 0x00, 0x50, 0x55, 0x00, 0x50, 0x55, 0x01,
0x50, 0xcd, 0x02, 0x90, 0xad, 0x02, 0x90, 0x6d, 0x02, 0xa0, 0x6d, 0x02,
0x20, 0x20, 0x01, 0x20, 0x00, 0x01, 0x26, 0x00, 0x01, 0x19, 0x00, 0x01,
0x02, 0x00, 0x01, 0x04, 0x80, 0x00, 0x08, 0x80, 0x00, 0x10, 0x40, 0x00,
0xe0, 0x3f, 0x00,
0x00, 0x02, 0x00, 0x20, 0x27, 0x00, 0x70, 0x77, 0x00, 0x70, 0x77, 0x01,
0x70, 0xff, 0x03, 0xf0, 0xbf, 0x03, 0xf0, 0xff, 0x03, 0xe0, 0xff, 0x03,
0xe0, 0xff, 0x01, 0xe0, 0xff, 0x01, 0xe6, 0xff, 0x01, 0xff, 0xff, 0x01,
0xfe, 0xff, 0x01, 0xfc, 0xff, 0x00, 0xf8, 0xff, 0x00, 0xf0, 0x7f, 0x00,
0xe0, 0x3f, 0x00 };

//ハンドドラッグ時
static const unsigned char g_cursor_hand_drag[] = {
16,16, 8,9,
0x80, 0x01, 0x70, 0x0e, 0x48, 0x12, 0x48, 0x72, 0x48, 0x92, 0x4e, 0x90,
0x09, 0x80, 0x0d, 0x80, 0x0d, 0x80, 0x0d, 0x80, 0x09, 0x80, 0x02, 0x80,
0x02, 0x80, 0x04, 0x40, 0x08, 0x20, 0xf0, 0x1f,
0x80, 0x01, 0xf0, 0x0f, 0xf8, 0x1f, 0xf8, 0x7f, 0xf8, 0xff, 0xfe, 0xff,
0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff,
0xfe, 0xff, 0xfc, 0x7f, 0xf8, 0x3f, 0xf0, 0x1f };

//回転
static const unsigned char g_cursor_rotate[] = {
24,25, 0,0,
0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x05, 0x00, 0x00, 0x09, 0x00, 0x00,
0x11, 0x00, 0x00, 0x21, 0x00, 0x00, 0x41, 0x00, 0x00, 0x81, 0x00, 0x00,
0xf9, 0x01, 0x00, 0x05, 0x00, 0x00, 0x03, 0x00, 0x00, 0x01, 0xe1, 0x0f,
0x00, 0xf3, 0x1f, 0x00, 0x3f, 0x38, 0x00, 0x1f, 0x70, 0x00, 0x1f, 0x60,
0x00, 0x3f, 0x60, 0x00, 0x00, 0x60, 0x00, 0x3f, 0x60, 0x00, 0x1f, 0x60,
0x00, 0x1f, 0x70, 0x00, 0x3f, 0x38, 0x00, 0xf3, 0x1f, 0x00, 0xe1, 0x0f,
0x00, 0x00, 0x00,
0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x07, 0x00, 0x00, 0x0f, 0x00, 0x00,
0x1f, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x7f, 0x00, 0x00, 0xff, 0x00, 0x00,
0xff, 0x01, 0x00, 0x07, 0x00, 0x00, 0x03, 0xe1, 0x0f, 0x81, 0xf3, 0x1f,
0x80, 0xff, 0x3f, 0x80, 0xff, 0x7f, 0x80, 0x3f, 0xf8, 0x80, 0x3f, 0xf0,
0x80, 0x7f, 0xf0, 0x00, 0x3f, 0xf0, 0x80, 0x7f, 0xf0, 0x80, 0x3f, 0xf0,
0x80, 0x3f, 0xf8, 0x80, 0xff, 0x7f, 0x80, 0xff, 0x3f, 0x80, 0xf3, 0x1f,
0x00, 0xe1, 0x0f };

//砂時計
static const unsigned char g_cursor_wait[] = {
10,19, 5,9,
0xff, 0x03, 0x01, 0x02, 0xff, 0x03, 0x02, 0x01, 0x02, 0x01, 0x02, 0x01,
0x02, 0x01, 0x42, 0x01, 0xa4, 0x00, 0x48, 0x00, 0x84, 0x00, 0x02, 0x01,
0x02, 0x01, 0x02, 0x01, 0x56, 0x01, 0xaa, 0x01, 0xff, 0x03, 0x01, 0x02,
0xff, 0x03,
0xff, 0x03, 0xff, 0x03, 0xff, 0x03, 0xfe, 0x01, 0xfe, 0x01, 0xfe, 0x01,
0xfe, 0x01, 0xfe, 0x01, 0xfc, 0x00, 0x78, 0x00, 0xfc, 0x00, 0xfe, 0x01,
0xfe, 0x01, 0xfe, 0x01, 0xfe, 0x01, 0xfe, 0x01, 0xff, 0x03, 0xff, 0x03,
0xff, 0x03 };

//アイテムD&D移動
static const unsigned char g_cursor_itemmove[] = {
18,22, 0,0,
0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x05, 0x00, 0x00, 0x09, 0x00, 0x00,
0x11, 0x00, 0x00, 0x21, 0x00, 0x00, 0x41, 0x00, 0x00, 0x81, 0x00, 0x00,
0x01, 0x01, 0x00, 0x01, 0x02, 0x00, 0xe1, 0x07, 0x00, 0x49, 0x00, 0x00,
0x55, 0x00, 0x00, 0x93, 0xff, 0x03, 0xa1, 0x00, 0x02, 0xa0, 0x00, 0x02,
0xc0, 0x00, 0x02, 0x80, 0x00, 0x02, 0x80, 0x00, 0x02, 0x80, 0x00, 0x02,
0x80, 0x00, 0x02, 0x80, 0xff, 0x03,
0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x07, 0x00, 0x00, 0x0f, 0x00, 0x00,
0x1f, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x7f, 0x00, 0x00, 0xff, 0x00, 0x00,
0xff, 0x01, 0x00, 0xff, 0x03, 0x00, 0xff, 0x07, 0x00, 0x7f, 0x00, 0x00,
0x77, 0x00, 0x00, 0xf3, 0xff, 0x03, 0xe1, 0xff, 0x03, 0xe0, 0xff, 0x03,
0xc0, 0xff, 0x03, 0x80, 0xff, 0x03, 0x80, 0xff, 0x03, 0x80, 0xff, 0x03,
0x80, 0xff, 0x03, 0x80, 0xff, 0x03 };

//移動
static const unsigned char g_cursor_move[] = {
16,22, 0,0,
0x01,0x00,0x03,0x00,0x05,0x00,0x09,0x00,0x11,0x00,0x21,0x00,0x41,0x00,0x81,0x00,
0xf9,0x01,0x05,0x02,0x03,0x05,0x81,0x08,0xc0,0x1d,0x60,0x35,0xd0,0x5d,0x08,0x80,
0xd0,0x5d,0x60,0x35,0xc0,0x1d,0x80,0x08,0x00,0x05,0x00,0x02,0x01,0x00,0x03,0x00,
0x07,0x00,0x0f,0x00,0x1f,0x00,0x3f,0x00,0x7f,0x00,0xff,0x00,0xff,0x01,0x07,0x02,
0x03,0x07,0x81,0x0f,0xc0,0x1f,0x60,0x37,0xf0,0x7f,0xf8,0xff,0xf0,0x7f,0x60,0x37,
0xc0,0x1f,0x80,0x0f,0x00,0x07,0x00,0x02
};

//スポイト
static const unsigned char g_cursor_spoit[] = {
14,14, 1,12,
0x00, 0x00, 0x00, 0x0c, 0x00, 0x1e, 0x80, 0x1f, 0xc0, 0x0f, 0xc0, 0x07,
0x20, 0x07, 0x10, 0x03, 0x88, 0x00, 0x44, 0x00, 0x22, 0x00, 0x12, 0x00,
0x0d, 0x00, 0x02, 0x00,
0x00, 0x0c, 0x00, 0x1e, 0x80, 0x3f, 0xc0, 0x3f, 0xe0, 0x1f, 0xe0, 0x0f,
0xe0, 0x0f, 0xf0, 0x07, 0xf8, 0x03, 0x7c, 0x00, 0x3e, 0x00, 0x1e, 0x00,
0x0f, 0x00, 0x02, 0x00 };

//選択
static const unsigned char g_cursor_select[] = {
14,16, 0,0,
0x01,0x00,0x03,0x00,0x05,0x00,0x09,0x00,0x11,0x00,0x21,0x00,0x41,0x00,0x81,0x00,
0xf9,0x2f,0x25,0x20,0x23,0x20,0x21,0x20,0x20,0x20,0x20,0x20,0x00,0x20,0xe0,0x3f,
0x01,0x00,0x03,0x00,0x07,0x00,0x0f,0x00,0x1f,0x00,0x3f,0x00,0x7f,0x00,0xff,0x1f,
0xff,0x3f,0x37,0x30,0x33,0x30,0x31,0x30,0x30,0x30,0x30,0x30,0xf0,0x3f,0xe0,0x3f
};

//範囲移動
static const unsigned char g_cursor_selmove[] = {
15,18, 0,0,
0x01,0x00,0x03,0x00,0x05,0x00,0x09,0x00,0x11,0x00,0x21,0x00,0x41,0x00,0x81,0x00,
0xf9,0x5f,0x25,0x42,0x23,0x47,0xa1,0x4a,0xe0,0x5f,0xa0,0x4a,0x20,0x47,0x20,0x42,
0x00,0x40,0xe0,0x7f,0x01,0x00,0x03,0x00,0x07,0x00,0x0f,0x00,0x1f,0x00,0x3f,0x00,
0x7f,0x00,0xff,0x3f,0xff,0x7f,0x37,0x67,0xb3,0x6f,0xf1,0x7f,0xf0,0x7f,0xf0,0x7f,
0xb0,0x6f,0x30,0x67,0xf0,0x7f,0xe0,0x7f
};

//スタンプ
static const unsigned char g_cursor_stamp[] = {
13,19, 0,0,
0x01,0x00,0x03,0x00,0x05,0x00,0x09,0x00,0x11,0x00,0x21,0x00,0x41,0x00,0x81,0x00,
0xf9,0x01,0x85,0x07,0x83,0x04,0x81,0x04,0x80,0x07,0x00,0x03,0x00,0x03,0xe0,0x1f,
0x20,0x10,0xe0,0x1f,0xe0,0x1f,0x01,0x00,0x03,0x00,0x07,0x00,0x0f,0x00,0x1f,0x00,
0x3f,0x00,0x7f,0x00,0xff,0x00,0xff,0x01,0x87,0x07,0x83,0x07,0x81,0x07,0x80,0x07,
0x00,0x03,0x00,0x03,0xe0,0x1f,0xe0,0x1f,0xe0,0x1f,0xe0,0x1f
};

//テキスト
static const unsigned char g_cursor_text[] = {
7,16, 3,1,
0x3e,0x41,0x36,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x36,0x41,0x3e,
0x3e,0x7f,0x3e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x3e,0x7f,0x3e
};

//上下ドラッグ拡大縮小
static const unsigned char g_cursor_zoom_drag[]={
15,15,0,0,
0xfc,0x00,0x02,0x01,0x31,0x02,0x79,0x02,0x01,0x02,0x01,0x02,0x79,0x02,0x31,0x02,
0x02,0x01,0xfc,0x0e,0x00,0x1e,0x00,0x3e,0x00,0x7c,0x00,0x78,0x00,0x70,
0xfc,0x00,0xfe,0x01,0xff,0x03,0xff,0x03,0xff,0x03,0xff,0x03,0xff,0x03,0xff,0x03,
0xfe,0x01,0xfc,0x0e,0x00,0x1e,0x00,0x3e,0x00,0x7c,0x00,0x78,0x00,0x70 };

//リサイズ:左上
static const unsigned char g_cursor_lefttop[] = {
15,15, 8,8,
0xff,0x03,0x01,0x01,0x81,0x00,0x41,0x00,0x81,0x00,0x01,0x41,0x09,0x62,0x15,0x54,
0x23,0x48,0x41,0x40,0x80,0x40,0x00,0x41,0x80,0x40,0x40,0x40,0xe0,0x7f,0xff,0x03,
0xff,0x01,0xff,0x00,0x7f,0x00,0xff,0x00,0xff,0x41,0xff,0x63,0xf7,0x77,0xe3,0x7f,
0xc1,0x7f,0x80,0x7f,0x00,0x7f,0x80,0x7f,0xc0,0x7f,0xe0,0x7f
};

//リサイズ:右上
static const unsigned char g_cursor_righttop[] = {
15,15, 8,8,
0xe0,0x7f,0x40,0x40,0x80,0x40,0x00,0x41,0x80,0x40,0x41,0x40,0x23,0x48,0x15,0x54,
0x09,0x62,0x01,0x41,0x81,0x00,0x41,0x00,0x81,0x00,0x01,0x01,0xff,0x03,0xe0,0x7f,
0xc0,0x7f,0x80,0x7f,0x00,0x7f,0x80,0x7f,0xc1,0x7f,0xe3,0x7f,0xf7,0x77,0xff,0x63,
0xff,0x41,0xff,0x00,0x7f,0x00,0xff,0x00,0xff,0x01,0xff,0x03
};

//リサイズ:水平
static const unsigned char g_cursor_resizehorz[] = {
19,13, 10,7,
0x40,0x10,0x00,0x60,0x30,0x00,0x50,0x50,0x00,0x48,0x90,0x00,0xc4,0x1f,0x01,0x02,
0x00,0x02,0x01,0x00,0x04,0x02,0x00,0x02,0xc4,0x1f,0x01,0x48,0x90,0x00,0x50,0x50,
0x00,0x60,0x30,0x00,0x40,0x10,0x00,0x40,0x10,0x00,0x60,0x30,0x00,0x70,0x70,0x00,
0x78,0xf0,0x00,0xfc,0xff,0x01,0xfe,0xff,0x03,0xff,0xff,0x07,0xfe,0xff,0x03,0xfc,
0xff,0x01,0x78,0xf0,0x00,0x70,0x70,0x00,0x60,0x30,0x00,0x40,0x10,0x00
};

//リサイズ:垂直
static const unsigned char g_cursor_resizevert[] = {
13,19, 7,10,
0x40,0x00,0xa0,0x00,0x10,0x01,0x08,0x02,0x04,0x04,0x02,0x08,0x1f,0x1f,0x10,0x01,
0x10,0x01,0x10,0x01,0x10,0x01,0x10,0x01,0x1f,0x1f,0x02,0x08,0x04,0x04,0x08,0x02,
0x10,0x01,0xa0,0x00,0x40,0x00,0x40,0x00,0xe0,0x00,0xf0,0x01,0xf8,0x03,0xfc,0x07,
0xfe,0x0f,0xff,0x1f,0xf0,0x01,0xf0,0x01,0xf0,0x01,0xf0,0x01,0xf0,0x01,0xff,0x1f,
0xfe,0x0f,0xfc,0x07,0xf8,0x03,0xf0,0x01,0xe0,0x00,0x40,0x00
};

//---------------------------

enum
{
	SLOT_DRAW,
	SLOT_WAIT,
	SLOT_CANVAS,
	SLOT_DRAG,
	SLOT_DIALOG,

	SLOT_NUM
};

static const unsigned char *g_cursor_imgdat[] = {
	g_cursor_hand, g_cursor_hand_drag, g_cursor_rotate, g_cursor_itemmove,
	g_cursor_move, g_cursor_spoit, g_cursor_select, g_cursor_selmove,
	g_cursor_stamp, g_cursor_text, g_cursor_zoom_drag,
	g_cursor_lefttop, g_cursor_righttop, g_cursor_resizehorz, g_cursor_resizevert
};

/* slot : [0] 描画用矢印 [1] 砂時計 [2] キャンバス用 [3] ドラッグ用 [4] ダイアログ用 */

static mCursor g_slot_cursor[SLOT_NUM] = { 0,0,0,0,0 };
static int g_slot_no[SLOT_NUM] = { APP_CURSOR_DRAW, APP_CURSOR_WAIT, -1, -1, -1 };


//---------------------------



/** 番号からカーソル取得 */

static mCursor _get_cursor_slot(int slot,int cursorno)
{
	if(g_slot_no[slot] == cursorno)
		//現在のスロットに同じカーソルがある場合
		return g_slot_cursor[slot];
	else if(cursorno < 2)
		//固定カーソルの場合
		return g_slot_cursor[cursorno];
	else
	{
		//スロットのカーソルを入れ替え
		
		mCursorFree(g_slot_cursor[slot]);

		g_slot_cursor[slot] = mCursorCreateMono(g_cursor_imgdat[cursorno - 2]);
		g_slot_no[slot] = cursorno;

		return g_slot_cursor[slot];
	}
}


/** 初期化 */

void AppCursor_init(uint8_t *drawcursor)
{
	g_slot_cursor[SLOT_DRAW] = mCursorCreateMono((drawcursor)? drawcursor: g_cursor_draw);
	g_slot_cursor[SLOT_WAIT] = mCursorCreateMono(g_cursor_wait);
}

/** 終了時の解放 */

void AppCursor_free()
{
	int i;

	for(i = 0; i < SLOT_NUM; i++)
		mCursorFree(g_slot_cursor[i]);
}

/** 描画用カーソルを変更
 *
 * @param buf NULL でデフォルト */

void AppCursor_setDrawCursor(uint8_t *buf)
{
	mCursorFree(g_slot_cursor[SLOT_DRAW]);

	g_slot_cursor[SLOT_DRAW] = mCursorCreateMono((buf)? buf: g_cursor_draw);
}

/** 砂時計カーソル取得 */

mCursor AppCursor_getWaitCursor()
{
	return g_slot_cursor[SLOT_WAIT];
}

/** キャンバス用のカーソル取得 */

mCursor AppCursor_getForCanvas(int no)
{
	return _get_cursor_slot(SLOT_CANVAS, no);
}

/** ドラッグ中用のカーソル取得 */

mCursor AppCursor_getForDrag(int no)
{
	return _get_cursor_slot(SLOT_DRAG, no);
}

/** ダイアログ用のカーソル取得 */

mCursor AppCursor_getForDialog(int no)
{
	return _get_cursor_slot(SLOT_DIALOG, no);
}
