(function() {
  var CompositeDisposable, Emitter, File, LessCompileCache, ThemeManager, _, fs, path, ref;

  path = require('path');

  _ = require('underscore-plus');

  ref = require('event-kit'), Emitter = ref.Emitter, CompositeDisposable = ref.CompositeDisposable;

  File = require('pathwatcher').File;

  fs = require('fs-plus');

  LessCompileCache = require('./less-compile-cache');

  module.exports = ThemeManager = (function() {
    function ThemeManager(arg) {
      this.packageManager = arg.packageManager, this.config = arg.config, this.styleManager = arg.styleManager, this.notificationManager = arg.notificationManager, this.viewRegistry = arg.viewRegistry;
      this.emitter = new Emitter;
      this.styleSheetDisposablesBySourcePath = {};
      this.lessCache = null;
      this.initialLoadComplete = false;
      this.packageManager.registerPackageActivator(this, ['theme']);
      this.packageManager.onDidActivateInitialPackages((function(_this) {
        return function() {
          return _this.onDidChangeActiveThemes(function() {
            return _this.packageManager.reloadActivePackageStyleSheets();
          });
        };
      })(this));
    }

    ThemeManager.prototype.initialize = function(arg) {
      var devMode;
      this.resourcePath = arg.resourcePath, this.configDirPath = arg.configDirPath, this.safeMode = arg.safeMode, devMode = arg.devMode;
      this.lessSourcesByRelativeFilePath = null;
      if (devMode || typeof snapshotAuxiliaryData === 'undefined') {
        this.lessSourcesByRelativeFilePath = {};
        return this.importedFilePathsByRelativeImportPath = {};
      } else {
        this.lessSourcesByRelativeFilePath = snapshotAuxiliaryData.lessSourcesByRelativeFilePath;
        return this.importedFilePathsByRelativeImportPath = snapshotAuxiliaryData.importedFilePathsByRelativeImportPath;
      }
    };


    /*
    Section: Event Subscription
     */

    ThemeManager.prototype.onDidChangeActiveThemes = function(callback) {
      return this.emitter.on('did-change-active-themes', callback);
    };


    /*
    Section: Accessing Available Themes
     */

    ThemeManager.prototype.getAvailableNames = function() {
      return this.getLoadedNames();
    };


    /*
    Section: Accessing Loaded Themes
     */

    ThemeManager.prototype.getLoadedThemeNames = function() {
      var i, len, ref1, results, theme;
      ref1 = this.getLoadedThemes();
      results = [];
      for (i = 0, len = ref1.length; i < len; i++) {
        theme = ref1[i];
        results.push(theme.name);
      }
      return results;
    };

    ThemeManager.prototype.getLoadedThemes = function() {
      var i, len, pack, ref1, results;
      ref1 = this.packageManager.getLoadedPackages();
      results = [];
      for (i = 0, len = ref1.length; i < len; i++) {
        pack = ref1[i];
        if (pack.isTheme()) {
          results.push(pack);
        }
      }
      return results;
    };


    /*
    Section: Accessing Active Themes
     */

    ThemeManager.prototype.getActiveThemeNames = function() {
      var i, len, ref1, results, theme;
      ref1 = this.getActiveThemes();
      results = [];
      for (i = 0, len = ref1.length; i < len; i++) {
        theme = ref1[i];
        results.push(theme.name);
      }
      return results;
    };

    ThemeManager.prototype.getActiveThemes = function() {
      var i, len, pack, ref1, results;
      ref1 = this.packageManager.getActivePackages();
      results = [];
      for (i = 0, len = ref1.length; i < len; i++) {
        pack = ref1[i];
        if (pack.isTheme()) {
          results.push(pack);
        }
      }
      return results;
    };

    ThemeManager.prototype.activatePackages = function() {
      return this.activateThemes();
    };


    /*
    Section: Managing Enabled Themes
     */

    ThemeManager.prototype.warnForNonExistentThemes = function() {
      var i, len, ref1, results, themeName, themeNames;
      themeNames = (ref1 = this.config.get('core.themes')) != null ? ref1 : [];
      if (!_.isArray(themeNames)) {
        themeNames = [themeNames];
      }
      results = [];
      for (i = 0, len = themeNames.length; i < len; i++) {
        themeName = themeNames[i];
        if (!(themeName && typeof themeName === 'string' && this.packageManager.resolvePackagePath(themeName))) {
          results.push(console.warn("Enabled theme '" + themeName + "' is not installed."));
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    ThemeManager.prototype.getEnabledThemeNames = function() {
      var builtInThemeNames, ref1, themeNames;
      themeNames = (ref1 = this.config.get('core.themes')) != null ? ref1 : [];
      if (!_.isArray(themeNames)) {
        themeNames = [themeNames];
      }
      themeNames = themeNames.filter((function(_this) {
        return function(themeName) {
          if (themeName && typeof themeName === 'string') {
            if (_this.packageManager.resolvePackagePath(themeName)) {
              return true;
            }
          }
          return false;
        };
      })(this));
      if (themeNames.length < 2) {
        builtInThemeNames = ['atom-dark-syntax', 'atom-dark-ui', 'atom-light-syntax', 'atom-light-ui', 'base16-tomorrow-dark-theme', 'base16-tomorrow-light-theme', 'solarized-dark-syntax', 'solarized-light-syntax'];
        themeNames = _.intersection(themeNames, builtInThemeNames);
        if (themeNames.length === 0) {
          themeNames = ['atom-dark-syntax', 'atom-dark-ui'];
        } else if (themeNames.length === 1) {
          if (_.endsWith(themeNames[0], '-ui')) {
            themeNames.unshift('atom-dark-syntax');
          } else {
            themeNames.push('atom-dark-ui');
          }
        }
      }
      return themeNames.reverse();
    };


    /*
    Section: Private
     */

    ThemeManager.prototype.requireStylesheet = function(stylesheetPath, priority, skipDeprecatedSelectorsTransformation) {
      var content, fullPath;
      if (fullPath = this.resolveStylesheet(stylesheetPath)) {
        content = this.loadStylesheet(fullPath);
        return this.applyStylesheet(fullPath, content, priority, skipDeprecatedSelectorsTransformation);
      } else {
        throw new Error("Could not find a file at path '" + stylesheetPath + "'");
      }
    };

    ThemeManager.prototype.unwatchUserStylesheet = function() {
      var ref1, ref2;
      if ((ref1 = this.userStylsheetSubscriptions) != null) {
        ref1.dispose();
      }
      this.userStylsheetSubscriptions = null;
      this.userStylesheetFile = null;
      if ((ref2 = this.userStyleSheetDisposable) != null) {
        ref2.dispose();
      }
      return this.userStyleSheetDisposable = null;
    };

    ThemeManager.prototype.loadUserStylesheet = function() {
      var error, message, reloadStylesheet, userStylesheetContents, userStylesheetPath;
      this.unwatchUserStylesheet();
      userStylesheetPath = this.styleManager.getUserStyleSheetPath();
      if (!fs.isFileSync(userStylesheetPath)) {
        return;
      }
      try {
        this.userStylesheetFile = new File(userStylesheetPath);
        this.userStylsheetSubscriptions = new CompositeDisposable();
        reloadStylesheet = (function(_this) {
          return function() {
            return _this.loadUserStylesheet();
          };
        })(this);
        this.userStylsheetSubscriptions.add(this.userStylesheetFile.onDidChange(reloadStylesheet));
        this.userStylsheetSubscriptions.add(this.userStylesheetFile.onDidRename(reloadStylesheet));
        this.userStylsheetSubscriptions.add(this.userStylesheetFile.onDidDelete(reloadStylesheet));
      } catch (error1) {
        error = error1;
        message = "Unable to watch path: `" + (path.basename(userStylesheetPath)) + "`. Make sure\nyou have permissions to `" + userStylesheetPath + "`.\n\nOn linux there are currently problems with watch sizes. See\n[this document][watches] for more info.\n[watches]:https://github.com/atom/atom/blob/master/docs/build-instructions/linux.md#typeerror-unable-to-watch-path";
        this.notificationManager.addError(message, {
          dismissable: true
        });
      }
      try {
        userStylesheetContents = this.loadStylesheet(userStylesheetPath, true);
      } catch (error1) {
        return;
      }
      return this.userStyleSheetDisposable = this.styleManager.addStyleSheet(userStylesheetContents, {
        sourcePath: userStylesheetPath,
        priority: 2
      });
    };

    ThemeManager.prototype.loadBaseStylesheets = function() {
      return this.reloadBaseStylesheets();
    };

    ThemeManager.prototype.reloadBaseStylesheets = function() {
      return this.requireStylesheet('../static/atom', -2, true);
    };

    ThemeManager.prototype.stylesheetElementForId = function(id) {
      var escapedId;
      escapedId = id.replace(/\\/g, '\\\\');
      return document.head.querySelector("atom-styles style[source-path=\"" + escapedId + "\"]");
    };

    ThemeManager.prototype.resolveStylesheet = function(stylesheetPath) {
      if (path.extname(stylesheetPath).length > 0) {
        return fs.resolveOnLoadPath(stylesheetPath);
      } else {
        return fs.resolveOnLoadPath(stylesheetPath, ['css', 'less']);
      }
    };

    ThemeManager.prototype.loadStylesheet = function(stylesheetPath, importFallbackVariables) {
      if (path.extname(stylesheetPath) === '.less') {
        return this.loadLessStylesheet(stylesheetPath, importFallbackVariables);
      } else {
        return fs.readFileSync(stylesheetPath, 'utf8');
      }
    };

    ThemeManager.prototype.loadLessStylesheet = function(lessStylesheetPath, importFallbackVariables) {
      var baseVarImports, content, detail, digest, error, lessSource, message, relativeFilePath;
      if (importFallbackVariables == null) {
        importFallbackVariables = false;
      }
      if (this.lessCache == null) {
        this.lessCache = new LessCompileCache({
          resourcePath: this.resourcePath,
          lessSourcesByRelativeFilePath: this.lessSourcesByRelativeFilePath,
          importedFilePathsByRelativeImportPath: this.importedFilePathsByRelativeImportPath,
          importPaths: this.getImportPaths()
        });
      }
      try {
        if (importFallbackVariables) {
          baseVarImports = "@import \"variables/ui-variables\";\n@import \"variables/syntax-variables\";";
          relativeFilePath = path.relative(this.resourcePath, lessStylesheetPath);
          lessSource = this.lessSourcesByRelativeFilePath[relativeFilePath];
          if (lessSource != null) {
            content = lessSource.content;
            digest = lessSource.digest;
          } else {
            content = baseVarImports + '\n' + fs.readFileSync(lessStylesheetPath, 'utf8');
            digest = null;
          }
          return this.lessCache.cssForFile(lessStylesheetPath, content, digest);
        } else {
          return this.lessCache.read(lessStylesheetPath);
        }
      } catch (error1) {
        error = error1;
        error.less = true;
        if (error.line != null) {
          if (importFallbackVariables) {
            error.line -= 2;
          }
          message = "Error compiling Less stylesheet: `" + lessStylesheetPath + "`";
          detail = "Line number: " + error.line + "\n" + error.message;
        } else {
          message = "Error loading Less stylesheet: `" + lessStylesheetPath + "`";
          detail = error.message;
        }
        this.notificationManager.addError(message, {
          detail: detail,
          dismissable: true
        });
        throw error;
      }
    };

    ThemeManager.prototype.removeStylesheet = function(stylesheetPath) {
      var ref1;
      return (ref1 = this.styleSheetDisposablesBySourcePath[stylesheetPath]) != null ? ref1.dispose() : void 0;
    };

    ThemeManager.prototype.applyStylesheet = function(path, text, priority, skipDeprecatedSelectorsTransformation) {
      return this.styleSheetDisposablesBySourcePath[path] = this.styleManager.addStyleSheet(text, {
        priority: priority,
        skipDeprecatedSelectorsTransformation: skipDeprecatedSelectorsTransformation,
        sourcePath: path
      });
    };

    ThemeManager.prototype.activateThemes = function() {
      return new Promise((function(_this) {
        return function(resolve) {
          return _this.config.observe('core.themes', function() {
            var i, len, promises, ref1, themeName;
            _this.deactivateThemes();
            _this.warnForNonExistentThemes();
            _this.refreshLessCache();
            promises = [];
            ref1 = _this.getEnabledThemeNames();
            for (i = 0, len = ref1.length; i < len; i++) {
              themeName = ref1[i];
              if (_this.packageManager.resolvePackagePath(themeName)) {
                promises.push(_this.packageManager.activatePackage(themeName));
              } else {
                console.warn("Failed to activate theme '" + themeName + "' because it isn't installed.");
              }
            }
            return Promise.all(promises).then(function() {
              _this.addActiveThemeClasses();
              _this.refreshLessCache();
              _this.loadUserStylesheet();
              _this.reloadBaseStylesheets();
              _this.initialLoadComplete = true;
              _this.emitter.emit('did-change-active-themes');
              return resolve();
            });
          });
        };
      })(this));
    };

    ThemeManager.prototype.deactivateThemes = function() {
      var i, len, pack, ref1;
      this.removeActiveThemeClasses();
      this.unwatchUserStylesheet();
      ref1 = this.getActiveThemes();
      for (i = 0, len = ref1.length; i < len; i++) {
        pack = ref1[i];
        this.packageManager.deactivatePackage(pack.name);
      }
      return null;
    };

    ThemeManager.prototype.isInitialLoadComplete = function() {
      return this.initialLoadComplete;
    };

    ThemeManager.prototype.addActiveThemeClasses = function() {
      var i, len, pack, ref1, workspaceElement;
      if (workspaceElement = this.viewRegistry.getView(this.workspace)) {
        ref1 = this.getActiveThemes();
        for (i = 0, len = ref1.length; i < len; i++) {
          pack = ref1[i];
          workspaceElement.classList.add("theme-" + pack.name);
        }
      }
    };

    ThemeManager.prototype.removeActiveThemeClasses = function() {
      var i, len, pack, ref1, workspaceElement;
      workspaceElement = this.viewRegistry.getView(this.workspace);
      ref1 = this.getActiveThemes();
      for (i = 0, len = ref1.length; i < len; i++) {
        pack = ref1[i];
        workspaceElement.classList.remove("theme-" + pack.name);
      }
    };

    ThemeManager.prototype.refreshLessCache = function() {
      var ref1;
      return (ref1 = this.lessCache) != null ? ref1.setImportPaths(this.getImportPaths()) : void 0;
    };

    ThemeManager.prototype.getImportPaths = function() {
      var activeThemes, deprecatedPath, i, len, ref1, theme, themeName, themePath, themePaths;
      activeThemes = this.getActiveThemes();
      if (activeThemes.length > 0) {
        themePaths = (function() {
          var i, len, results;
          results = [];
          for (i = 0, len = activeThemes.length; i < len; i++) {
            theme = activeThemes[i];
            if (theme) {
              results.push(theme.getStylesheetsPath());
            }
          }
          return results;
        })();
      } else {
        themePaths = [];
        ref1 = this.getEnabledThemeNames();
        for (i = 0, len = ref1.length; i < len; i++) {
          themeName = ref1[i];
          if (themePath = this.packageManager.resolvePackagePath(themeName)) {
            deprecatedPath = path.join(themePath, 'stylesheets');
            if (fs.isDirectorySync(deprecatedPath)) {
              themePaths.push(deprecatedPath);
            } else {
              themePaths.push(path.join(themePath, 'styles'));
            }
          }
        }
      }
      return themePaths.filter(function(themePath) {
        return fs.isDirectorySync(themePath);
      });
    };

    return ThemeManager;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiL2hvbWUvdHJhdmlzL2J1aWxkL2F0b20vYXRvbS9vdXQvYXBwL3NyYy90aGVtZS1tYW5hZ2VyLmNvZmZlZSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTtBQUFBLE1BQUE7O0VBQUEsSUFBQSxHQUFPLE9BQUEsQ0FBUSxNQUFSOztFQUNQLENBQUEsR0FBSSxPQUFBLENBQVEsaUJBQVI7O0VBQ0osTUFBaUMsT0FBQSxDQUFRLFdBQVIsQ0FBakMsRUFBQyxxQkFBRCxFQUFVOztFQUNULE9BQVEsT0FBQSxDQUFRLGFBQVI7O0VBQ1QsRUFBQSxHQUFLLE9BQUEsQ0FBUSxTQUFSOztFQUNMLGdCQUFBLEdBQW1CLE9BQUEsQ0FBUSxzQkFBUjs7RUFLbkIsTUFBTSxDQUFDLE9BQVAsR0FDTTtJQUNTLHNCQUFDLEdBQUQ7TUFBRSxJQUFDLENBQUEscUJBQUEsZ0JBQWdCLElBQUMsQ0FBQSxhQUFBLFFBQVEsSUFBQyxDQUFBLG1CQUFBLGNBQWMsSUFBQyxDQUFBLDBCQUFBLHFCQUFxQixJQUFDLENBQUEsbUJBQUE7TUFDN0UsSUFBQyxDQUFBLE9BQUQsR0FBVyxJQUFJO01BQ2YsSUFBQyxDQUFBLGlDQUFELEdBQXFDO01BQ3JDLElBQUMsQ0FBQSxTQUFELEdBQWE7TUFDYixJQUFDLENBQUEsbUJBQUQsR0FBdUI7TUFDdkIsSUFBQyxDQUFBLGNBQWMsQ0FBQyx3QkFBaEIsQ0FBeUMsSUFBekMsRUFBK0MsQ0FBQyxPQUFELENBQS9DO01BQ0EsSUFBQyxDQUFBLGNBQWMsQ0FBQyw0QkFBaEIsQ0FBNkMsQ0FBQSxTQUFBLEtBQUE7ZUFBQSxTQUFBO2lCQUMzQyxLQUFDLENBQUEsdUJBQUQsQ0FBeUIsU0FBQTttQkFBRyxLQUFDLENBQUEsY0FBYyxDQUFDLDhCQUFoQixDQUFBO1VBQUgsQ0FBekI7UUFEMkM7TUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBQTdDO0lBTlc7OzJCQVNiLFVBQUEsR0FBWSxTQUFDLEdBQUQ7QUFDVixVQUFBO01BRFksSUFBQyxDQUFBLG1CQUFBLGNBQWMsSUFBQyxDQUFBLG9CQUFBLGVBQWUsSUFBQyxDQUFBLGVBQUEsVUFBVTtNQUN0RCxJQUFDLENBQUEsNkJBQUQsR0FBaUM7TUFDakMsSUFBRyxPQUFBLElBQVcsT0FBTyxxQkFBUCxLQUFnQyxXQUE5QztRQUNFLElBQUMsQ0FBQSw2QkFBRCxHQUFpQztlQUNqQyxJQUFDLENBQUEscUNBQUQsR0FBeUMsR0FGM0M7T0FBQSxNQUFBO1FBSUUsSUFBQyxDQUFBLDZCQUFELEdBQWlDLHFCQUFxQixDQUFDO2VBQ3ZELElBQUMsQ0FBQSxxQ0FBRCxHQUF5QyxxQkFBcUIsQ0FBQyxzQ0FMakU7O0lBRlU7OztBQVNaOzs7OzJCQVFBLHVCQUFBLEdBQXlCLFNBQUMsUUFBRDthQUN2QixJQUFDLENBQUEsT0FBTyxDQUFDLEVBQVQsQ0FBWSwwQkFBWixFQUF3QyxRQUF4QztJQUR1Qjs7O0FBR3pCOzs7OzJCQUlBLGlCQUFBLEdBQW1CLFNBQUE7YUFFakIsSUFBQyxDQUFBLGNBQUQsQ0FBQTtJQUZpQjs7O0FBSW5COzs7OzJCQUtBLG1CQUFBLEdBQXFCLFNBQUE7QUFDbkIsVUFBQTtBQUFBO0FBQUE7V0FBQSxzQ0FBQTs7cUJBQUEsS0FBSyxDQUFDO0FBQU47O0lBRG1COzsyQkFJckIsZUFBQSxHQUFpQixTQUFBO0FBQ2YsVUFBQTtBQUFBO0FBQUE7V0FBQSxzQ0FBQTs7WUFBMEQsSUFBSSxDQUFDLE9BQUwsQ0FBQTt1QkFBMUQ7O0FBQUE7O0lBRGU7OztBQUdqQjs7OzsyQkFLQSxtQkFBQSxHQUFxQixTQUFBO0FBQ25CLFVBQUE7QUFBQTtBQUFBO1dBQUEsc0NBQUE7O3FCQUFBLEtBQUssQ0FBQztBQUFOOztJQURtQjs7MkJBSXJCLGVBQUEsR0FBaUIsU0FBQTtBQUNmLFVBQUE7QUFBQTtBQUFBO1dBQUEsc0NBQUE7O1lBQTBELElBQUksQ0FBQyxPQUFMLENBQUE7dUJBQTFEOztBQUFBOztJQURlOzsyQkFHakIsZ0JBQUEsR0FBa0IsU0FBQTthQUFHLElBQUMsQ0FBQSxjQUFELENBQUE7SUFBSDs7O0FBRWxCOzs7OzJCQUlBLHdCQUFBLEdBQTBCLFNBQUE7QUFDeEIsVUFBQTtNQUFBLFVBQUEsNERBQTBDO01BQzFDLElBQUEsQ0FBaUMsQ0FBQyxDQUFDLE9BQUYsQ0FBVSxVQUFWLENBQWpDO1FBQUEsVUFBQSxHQUFhLENBQUMsVUFBRCxFQUFiOztBQUNBO1dBQUEsNENBQUE7O1FBQ0UsSUFBQSxDQUFBLENBQU8sU0FBQSxJQUFjLE9BQU8sU0FBUCxLQUFvQixRQUFsQyxJQUErQyxJQUFDLENBQUEsY0FBYyxDQUFDLGtCQUFoQixDQUFtQyxTQUFuQyxDQUF0RCxDQUFBO3VCQUNFLE9BQU8sQ0FBQyxJQUFSLENBQWEsaUJBQUEsR0FBa0IsU0FBbEIsR0FBNEIscUJBQXpDLEdBREY7U0FBQSxNQUFBOytCQUFBOztBQURGOztJQUh3Qjs7MkJBVTFCLG9CQUFBLEdBQXNCLFNBQUE7QUFDcEIsVUFBQTtNQUFBLFVBQUEsNERBQTBDO01BQzFDLElBQUEsQ0FBaUMsQ0FBQyxDQUFDLE9BQUYsQ0FBVSxVQUFWLENBQWpDO1FBQUEsVUFBQSxHQUFhLENBQUMsVUFBRCxFQUFiOztNQUNBLFVBQUEsR0FBYSxVQUFVLENBQUMsTUFBWCxDQUFrQixDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUMsU0FBRDtVQUM3QixJQUFHLFNBQUEsSUFBYyxPQUFPLFNBQVAsS0FBb0IsUUFBckM7WUFDRSxJQUFlLEtBQUMsQ0FBQSxjQUFjLENBQUMsa0JBQWhCLENBQW1DLFNBQW5DLENBQWY7QUFBQSxxQkFBTyxLQUFQO2FBREY7O2lCQUVBO1FBSDZCO01BQUEsQ0FBQSxDQUFBLENBQUEsSUFBQSxDQUFsQjtNQU9iLElBQUcsVUFBVSxDQUFDLE1BQVgsR0FBb0IsQ0FBdkI7UUFDRSxpQkFBQSxHQUFvQixDQUNsQixrQkFEa0IsRUFFbEIsY0FGa0IsRUFHbEIsbUJBSGtCLEVBSWxCLGVBSmtCLEVBS2xCLDRCQUxrQixFQU1sQiw2QkFOa0IsRUFPbEIsdUJBUGtCLEVBUWxCLHdCQVJrQjtRQVVwQixVQUFBLEdBQWEsQ0FBQyxDQUFDLFlBQUYsQ0FBZSxVQUFmLEVBQTJCLGlCQUEzQjtRQUNiLElBQUcsVUFBVSxDQUFDLE1BQVgsS0FBcUIsQ0FBeEI7VUFDRSxVQUFBLEdBQWEsQ0FBQyxrQkFBRCxFQUFxQixjQUFyQixFQURmO1NBQUEsTUFFSyxJQUFHLFVBQVUsQ0FBQyxNQUFYLEtBQXFCLENBQXhCO1VBQ0gsSUFBRyxDQUFDLENBQUMsUUFBRixDQUFXLFVBQVcsQ0FBQSxDQUFBLENBQXRCLEVBQTBCLEtBQTFCLENBQUg7WUFDRSxVQUFVLENBQUMsT0FBWCxDQUFtQixrQkFBbkIsRUFERjtXQUFBLE1BQUE7WUFHRSxVQUFVLENBQUMsSUFBWCxDQUFnQixjQUFoQixFQUhGO1dBREc7U0FkUDs7YUFzQkEsVUFBVSxDQUFDLE9BQVgsQ0FBQTtJQWhDb0I7OztBQWtDdEI7Ozs7MkJBYUEsaUJBQUEsR0FBbUIsU0FBQyxjQUFELEVBQWlCLFFBQWpCLEVBQTJCLHFDQUEzQjtBQUNqQixVQUFBO01BQUEsSUFBRyxRQUFBLEdBQVcsSUFBQyxDQUFBLGlCQUFELENBQW1CLGNBQW5CLENBQWQ7UUFDRSxPQUFBLEdBQVUsSUFBQyxDQUFBLGNBQUQsQ0FBZ0IsUUFBaEI7ZUFDVixJQUFDLENBQUEsZUFBRCxDQUFpQixRQUFqQixFQUEyQixPQUEzQixFQUFvQyxRQUFwQyxFQUE4QyxxQ0FBOUMsRUFGRjtPQUFBLE1BQUE7QUFJRSxjQUFVLElBQUEsS0FBQSxDQUFNLGlDQUFBLEdBQWtDLGNBQWxDLEdBQWlELEdBQXZELEVBSlo7O0lBRGlCOzsyQkFPbkIscUJBQUEsR0FBdUIsU0FBQTtBQUNyQixVQUFBOztZQUEyQixDQUFFLE9BQTdCLENBQUE7O01BQ0EsSUFBQyxDQUFBLDBCQUFELEdBQThCO01BQzlCLElBQUMsQ0FBQSxrQkFBRCxHQUFzQjs7WUFDRyxDQUFFLE9BQTNCLENBQUE7O2FBQ0EsSUFBQyxDQUFBLHdCQUFELEdBQTRCO0lBTFA7OzJCQU92QixrQkFBQSxHQUFvQixTQUFBO0FBQ2xCLFVBQUE7TUFBQSxJQUFDLENBQUEscUJBQUQsQ0FBQTtNQUVBLGtCQUFBLEdBQXFCLElBQUMsQ0FBQSxZQUFZLENBQUMscUJBQWQsQ0FBQTtNQUNyQixJQUFBLENBQWMsRUFBRSxDQUFDLFVBQUgsQ0FBYyxrQkFBZCxDQUFkO0FBQUEsZUFBQTs7QUFFQTtRQUNFLElBQUMsQ0FBQSxrQkFBRCxHQUEwQixJQUFBLElBQUEsQ0FBSyxrQkFBTDtRQUMxQixJQUFDLENBQUEsMEJBQUQsR0FBa0MsSUFBQSxtQkFBQSxDQUFBO1FBQ2xDLGdCQUFBLEdBQW1CLENBQUEsU0FBQSxLQUFBO2lCQUFBLFNBQUE7bUJBQUcsS0FBQyxDQUFBLGtCQUFELENBQUE7VUFBSDtRQUFBLENBQUEsQ0FBQSxDQUFBLElBQUE7UUFDbkIsSUFBQyxDQUFBLDBCQUEwQixDQUFDLEdBQTVCLENBQWdDLElBQUMsQ0FBQSxrQkFBa0IsQ0FBQyxXQUFwQixDQUFnQyxnQkFBaEMsQ0FBaEM7UUFDQSxJQUFDLENBQUEsMEJBQTBCLENBQUMsR0FBNUIsQ0FBZ0MsSUFBQyxDQUFBLGtCQUFrQixDQUFDLFdBQXBCLENBQWdDLGdCQUFoQyxDQUFoQztRQUNBLElBQUMsQ0FBQSwwQkFBMEIsQ0FBQyxHQUE1QixDQUFnQyxJQUFDLENBQUEsa0JBQWtCLENBQUMsV0FBcEIsQ0FBZ0MsZ0JBQWhDLENBQWhDLEVBTkY7T0FBQSxjQUFBO1FBT007UUFDSixPQUFBLEdBQVUseUJBQUEsR0FDZ0IsQ0FBQyxJQUFJLENBQUMsUUFBTCxDQUFjLGtCQUFkLENBQUQsQ0FEaEIsR0FDbUQseUNBRG5ELEdBRW1CLGtCQUZuQixHQUVzQztRQU1oRCxJQUFDLENBQUEsbUJBQW1CLENBQUMsUUFBckIsQ0FBOEIsT0FBOUIsRUFBdUM7VUFBQSxXQUFBLEVBQWEsSUFBYjtTQUF2QyxFQWhCRjs7QUFrQkE7UUFDRSxzQkFBQSxHQUF5QixJQUFDLENBQUEsY0FBRCxDQUFnQixrQkFBaEIsRUFBb0MsSUFBcEMsRUFEM0I7T0FBQSxjQUFBO0FBR0UsZUFIRjs7YUFLQSxJQUFDLENBQUEsd0JBQUQsR0FBNEIsSUFBQyxDQUFBLFlBQVksQ0FBQyxhQUFkLENBQTRCLHNCQUE1QixFQUFvRDtRQUFBLFVBQUEsRUFBWSxrQkFBWjtRQUFnQyxRQUFBLEVBQVUsQ0FBMUM7T0FBcEQ7SUE3QlY7OzJCQStCcEIsbUJBQUEsR0FBcUIsU0FBQTthQUNuQixJQUFDLENBQUEscUJBQUQsQ0FBQTtJQURtQjs7MkJBR3JCLHFCQUFBLEdBQXVCLFNBQUE7YUFDckIsSUFBQyxDQUFBLGlCQUFELENBQW1CLGdCQUFuQixFQUFxQyxDQUFDLENBQXRDLEVBQXlDLElBQXpDO0lBRHFCOzsyQkFHdkIsc0JBQUEsR0FBd0IsU0FBQyxFQUFEO0FBQ3RCLFVBQUE7TUFBQSxTQUFBLEdBQVksRUFBRSxDQUFDLE9BQUgsQ0FBVyxLQUFYLEVBQWtCLE1BQWxCO2FBQ1osUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFkLENBQTRCLGtDQUFBLEdBQW1DLFNBQW5DLEdBQTZDLEtBQXpFO0lBRnNCOzsyQkFJeEIsaUJBQUEsR0FBbUIsU0FBQyxjQUFEO01BQ2pCLElBQUcsSUFBSSxDQUFDLE9BQUwsQ0FBYSxjQUFiLENBQTRCLENBQUMsTUFBN0IsR0FBc0MsQ0FBekM7ZUFDRSxFQUFFLENBQUMsaUJBQUgsQ0FBcUIsY0FBckIsRUFERjtPQUFBLE1BQUE7ZUFHRSxFQUFFLENBQUMsaUJBQUgsQ0FBcUIsY0FBckIsRUFBcUMsQ0FBQyxLQUFELEVBQVEsTUFBUixDQUFyQyxFQUhGOztJQURpQjs7MkJBTW5CLGNBQUEsR0FBZ0IsU0FBQyxjQUFELEVBQWlCLHVCQUFqQjtNQUNkLElBQUcsSUFBSSxDQUFDLE9BQUwsQ0FBYSxjQUFiLENBQUEsS0FBZ0MsT0FBbkM7ZUFDRSxJQUFDLENBQUEsa0JBQUQsQ0FBb0IsY0FBcEIsRUFBb0MsdUJBQXBDLEVBREY7T0FBQSxNQUFBO2VBR0UsRUFBRSxDQUFDLFlBQUgsQ0FBZ0IsY0FBaEIsRUFBZ0MsTUFBaEMsRUFIRjs7SUFEYzs7MkJBTWhCLGtCQUFBLEdBQW9CLFNBQUMsa0JBQUQsRUFBcUIsdUJBQXJCO0FBQ2xCLFVBQUE7O1FBRHVDLDBCQUF3Qjs7O1FBQy9ELElBQUMsQ0FBQSxZQUFpQixJQUFBLGdCQUFBLENBQWlCO1VBQ2hDLGNBQUQsSUFBQyxDQUFBLFlBRGdDO1VBRWhDLCtCQUFELElBQUMsQ0FBQSw2QkFGZ0M7VUFHaEMsdUNBQUQsSUFBQyxDQUFBLHFDQUhnQztVQUlqQyxXQUFBLEVBQWEsSUFBQyxDQUFBLGNBQUQsQ0FBQSxDQUpvQjtTQUFqQjs7QUFPbEI7UUFDRSxJQUFHLHVCQUFIO1VBQ0UsY0FBQSxHQUFpQjtVQUlqQixnQkFBQSxHQUFtQixJQUFJLENBQUMsUUFBTCxDQUFjLElBQUMsQ0FBQSxZQUFmLEVBQTZCLGtCQUE3QjtVQUNuQixVQUFBLEdBQWEsSUFBQyxDQUFBLDZCQUE4QixDQUFBLGdCQUFBO1VBQzVDLElBQUcsa0JBQUg7WUFDRSxPQUFBLEdBQVUsVUFBVSxDQUFDO1lBQ3JCLE1BQUEsR0FBUyxVQUFVLENBQUMsT0FGdEI7V0FBQSxNQUFBO1lBSUUsT0FBQSxHQUFVLGNBQUEsR0FBaUIsSUFBakIsR0FBd0IsRUFBRSxDQUFDLFlBQUgsQ0FBZ0Isa0JBQWhCLEVBQW9DLE1BQXBDO1lBQ2xDLE1BQUEsR0FBUyxLQUxYOztpQkFPQSxJQUFDLENBQUEsU0FBUyxDQUFDLFVBQVgsQ0FBc0Isa0JBQXRCLEVBQTBDLE9BQTFDLEVBQW1ELE1BQW5ELEVBZEY7U0FBQSxNQUFBO2lCQWdCRSxJQUFDLENBQUEsU0FBUyxDQUFDLElBQVgsQ0FBZ0Isa0JBQWhCLEVBaEJGO1NBREY7T0FBQSxjQUFBO1FBa0JNO1FBQ0osS0FBSyxDQUFDLElBQU4sR0FBYTtRQUNiLElBQUcsa0JBQUg7VUFFRSxJQUFtQix1QkFBbkI7WUFBQSxLQUFLLENBQUMsSUFBTixJQUFjLEVBQWQ7O1VBRUEsT0FBQSxHQUFVLG9DQUFBLEdBQXFDLGtCQUFyQyxHQUF3RDtVQUNsRSxNQUFBLEdBQVMsZUFBQSxHQUNRLEtBQUssQ0FBQyxJQURkLEdBQ21CLElBRG5CLEdBRUwsS0FBSyxDQUFDLFFBUFo7U0FBQSxNQUFBO1VBVUUsT0FBQSxHQUFVLGtDQUFBLEdBQW1DLGtCQUFuQyxHQUFzRDtVQUNoRSxNQUFBLEdBQVMsS0FBSyxDQUFDLFFBWGpCOztRQWFBLElBQUMsQ0FBQSxtQkFBbUIsQ0FBQyxRQUFyQixDQUE4QixPQUE5QixFQUF1QztVQUFDLFFBQUEsTUFBRDtVQUFTLFdBQUEsRUFBYSxJQUF0QjtTQUF2QztBQUNBLGNBQU0sTUFsQ1I7O0lBUmtCOzsyQkE0Q3BCLGdCQUFBLEdBQWtCLFNBQUMsY0FBRDtBQUNoQixVQUFBOzJGQUFrRCxDQUFFLE9BQXBELENBQUE7SUFEZ0I7OzJCQUdsQixlQUFBLEdBQWlCLFNBQUMsSUFBRCxFQUFPLElBQVAsRUFBYSxRQUFiLEVBQXVCLHFDQUF2QjthQUNmLElBQUMsQ0FBQSxpQ0FBa0MsQ0FBQSxJQUFBLENBQW5DLEdBQTJDLElBQUMsQ0FBQSxZQUFZLENBQUMsYUFBZCxDQUN6QyxJQUR5QyxFQUV6QztRQUNFLFVBQUEsUUFERjtRQUVFLHVDQUFBLHFDQUZGO1FBR0UsVUFBQSxFQUFZLElBSGQ7T0FGeUM7SUFENUI7OzJCQVVqQixjQUFBLEdBQWdCLFNBQUE7YUFDVixJQUFBLE9BQUEsQ0FBUSxDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUMsT0FBRDtpQkFFVixLQUFDLENBQUEsTUFBTSxDQUFDLE9BQVIsQ0FBZ0IsYUFBaEIsRUFBK0IsU0FBQTtBQUM3QixnQkFBQTtZQUFBLEtBQUMsQ0FBQSxnQkFBRCxDQUFBO1lBRUEsS0FBQyxDQUFBLHdCQUFELENBQUE7WUFFQSxLQUFDLENBQUEsZ0JBQUQsQ0FBQTtZQUVBLFFBQUEsR0FBVztBQUNYO0FBQUEsaUJBQUEsc0NBQUE7O2NBQ0UsSUFBRyxLQUFDLENBQUEsY0FBYyxDQUFDLGtCQUFoQixDQUFtQyxTQUFuQyxDQUFIO2dCQUNFLFFBQVEsQ0FBQyxJQUFULENBQWMsS0FBQyxDQUFBLGNBQWMsQ0FBQyxlQUFoQixDQUFnQyxTQUFoQyxDQUFkLEVBREY7ZUFBQSxNQUFBO2dCQUdFLE9BQU8sQ0FBQyxJQUFSLENBQWEsNEJBQUEsR0FBNkIsU0FBN0IsR0FBdUMsK0JBQXBELEVBSEY7O0FBREY7bUJBTUEsT0FBTyxDQUFDLEdBQVIsQ0FBWSxRQUFaLENBQXFCLENBQUMsSUFBdEIsQ0FBMkIsU0FBQTtjQUN6QixLQUFDLENBQUEscUJBQUQsQ0FBQTtjQUNBLEtBQUMsQ0FBQSxnQkFBRCxDQUFBO2NBQ0EsS0FBQyxDQUFBLGtCQUFELENBQUE7Y0FDQSxLQUFDLENBQUEscUJBQUQsQ0FBQTtjQUNBLEtBQUMsQ0FBQSxtQkFBRCxHQUF1QjtjQUN2QixLQUFDLENBQUEsT0FBTyxDQUFDLElBQVQsQ0FBYywwQkFBZDtxQkFDQSxPQUFBLENBQUE7WUFQeUIsQ0FBM0I7VUFkNkIsQ0FBL0I7UUFGVTtNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBUjtJQURVOzsyQkEwQmhCLGdCQUFBLEdBQWtCLFNBQUE7QUFDaEIsVUFBQTtNQUFBLElBQUMsQ0FBQSx3QkFBRCxDQUFBO01BQ0EsSUFBQyxDQUFBLHFCQUFELENBQUE7QUFDQTtBQUFBLFdBQUEsc0NBQUE7O1FBQUEsSUFBQyxDQUFBLGNBQWMsQ0FBQyxpQkFBaEIsQ0FBa0MsSUFBSSxDQUFDLElBQXZDO0FBQUE7YUFDQTtJQUpnQjs7MkJBTWxCLHFCQUFBLEdBQXVCLFNBQUE7YUFBRyxJQUFDLENBQUE7SUFBSjs7MkJBRXZCLHFCQUFBLEdBQXVCLFNBQUE7QUFDckIsVUFBQTtNQUFBLElBQUcsZ0JBQUEsR0FBbUIsSUFBQyxDQUFBLFlBQVksQ0FBQyxPQUFkLENBQXNCLElBQUMsQ0FBQSxTQUF2QixDQUF0QjtBQUNFO0FBQUEsYUFBQSxzQ0FBQTs7VUFDRSxnQkFBZ0IsQ0FBQyxTQUFTLENBQUMsR0FBM0IsQ0FBK0IsUUFBQSxHQUFTLElBQUksQ0FBQyxJQUE3QztBQURGLFNBREY7O0lBRHFCOzsyQkFNdkIsd0JBQUEsR0FBMEIsU0FBQTtBQUN4QixVQUFBO01BQUEsZ0JBQUEsR0FBbUIsSUFBQyxDQUFBLFlBQVksQ0FBQyxPQUFkLENBQXNCLElBQUMsQ0FBQSxTQUF2QjtBQUNuQjtBQUFBLFdBQUEsc0NBQUE7O1FBQ0UsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLE1BQTNCLENBQWtDLFFBQUEsR0FBUyxJQUFJLENBQUMsSUFBaEQ7QUFERjtJQUZ3Qjs7MkJBTTFCLGdCQUFBLEdBQWtCLFNBQUE7QUFDaEIsVUFBQTttREFBVSxDQUFFLGNBQVosQ0FBMkIsSUFBQyxDQUFBLGNBQUQsQ0FBQSxDQUEzQjtJQURnQjs7MkJBR2xCLGNBQUEsR0FBZ0IsU0FBQTtBQUNkLFVBQUE7TUFBQSxZQUFBLEdBQWUsSUFBQyxDQUFBLGVBQUQsQ0FBQTtNQUNmLElBQUcsWUFBWSxDQUFDLE1BQWIsR0FBc0IsQ0FBekI7UUFDRSxVQUFBOztBQUFjO2VBQUEsOENBQUE7O2dCQUEwRDsyQkFBMUQsS0FBSyxDQUFDLGtCQUFOLENBQUE7O0FBQUE7O2FBRGhCO09BQUEsTUFBQTtRQUdFLFVBQUEsR0FBYTtBQUNiO0FBQUEsYUFBQSxzQ0FBQTs7VUFDRSxJQUFHLFNBQUEsR0FBWSxJQUFDLENBQUEsY0FBYyxDQUFDLGtCQUFoQixDQUFtQyxTQUFuQyxDQUFmO1lBQ0UsY0FBQSxHQUFpQixJQUFJLENBQUMsSUFBTCxDQUFVLFNBQVYsRUFBcUIsYUFBckI7WUFDakIsSUFBRyxFQUFFLENBQUMsZUFBSCxDQUFtQixjQUFuQixDQUFIO2NBQ0UsVUFBVSxDQUFDLElBQVgsQ0FBZ0IsY0FBaEIsRUFERjthQUFBLE1BQUE7Y0FHRSxVQUFVLENBQUMsSUFBWCxDQUFnQixJQUFJLENBQUMsSUFBTCxDQUFVLFNBQVYsRUFBcUIsUUFBckIsQ0FBaEIsRUFIRjthQUZGOztBQURGLFNBSkY7O2FBWUEsVUFBVSxDQUFDLE1BQVgsQ0FBa0IsU0FBQyxTQUFEO2VBQWUsRUFBRSxDQUFDLGVBQUgsQ0FBbUIsU0FBbkI7TUFBZixDQUFsQjtJQWRjOzs7OztBQXJUbEIiLCJzb3VyY2VzQ29udGVudCI6WyJwYXRoID0gcmVxdWlyZSAncGF0aCdcbl8gPSByZXF1aXJlICd1bmRlcnNjb3JlLXBsdXMnXG57RW1pdHRlciwgQ29tcG9zaXRlRGlzcG9zYWJsZX0gPSByZXF1aXJlICdldmVudC1raXQnXG57RmlsZX0gPSByZXF1aXJlICdwYXRod2F0Y2hlcidcbmZzID0gcmVxdWlyZSAnZnMtcGx1cydcbkxlc3NDb21waWxlQ2FjaGUgPSByZXF1aXJlICcuL2xlc3MtY29tcGlsZS1jYWNoZSdcblxuIyBFeHRlbmRlZDogSGFuZGxlcyBsb2FkaW5nIGFuZCBhY3RpdmF0aW5nIGF2YWlsYWJsZSB0aGVtZXMuXG4jXG4jIEFuIGluc3RhbmNlIG9mIHRoaXMgY2xhc3MgaXMgYWx3YXlzIGF2YWlsYWJsZSBhcyB0aGUgYGF0b20udGhlbWVzYCBnbG9iYWwuXG5tb2R1bGUuZXhwb3J0cyA9XG5jbGFzcyBUaGVtZU1hbmFnZXJcbiAgY29uc3RydWN0b3I6ICh7QHBhY2thZ2VNYW5hZ2VyLCBAY29uZmlnLCBAc3R5bGVNYW5hZ2VyLCBAbm90aWZpY2F0aW9uTWFuYWdlciwgQHZpZXdSZWdpc3RyeX0pIC0+XG4gICAgQGVtaXR0ZXIgPSBuZXcgRW1pdHRlclxuICAgIEBzdHlsZVNoZWV0RGlzcG9zYWJsZXNCeVNvdXJjZVBhdGggPSB7fVxuICAgIEBsZXNzQ2FjaGUgPSBudWxsXG4gICAgQGluaXRpYWxMb2FkQ29tcGxldGUgPSBmYWxzZVxuICAgIEBwYWNrYWdlTWFuYWdlci5yZWdpc3RlclBhY2thZ2VBY3RpdmF0b3IodGhpcywgWyd0aGVtZSddKVxuICAgIEBwYWNrYWdlTWFuYWdlci5vbkRpZEFjdGl2YXRlSW5pdGlhbFBhY2thZ2VzID0+XG4gICAgICBAb25EaWRDaGFuZ2VBY3RpdmVUaGVtZXMgPT4gQHBhY2thZ2VNYW5hZ2VyLnJlbG9hZEFjdGl2ZVBhY2thZ2VTdHlsZVNoZWV0cygpXG5cbiAgaW5pdGlhbGl6ZTogKHtAcmVzb3VyY2VQYXRoLCBAY29uZmlnRGlyUGF0aCwgQHNhZmVNb2RlLCBkZXZNb2RlfSkgLT5cbiAgICBAbGVzc1NvdXJjZXNCeVJlbGF0aXZlRmlsZVBhdGggPSBudWxsXG4gICAgaWYgZGV2TW9kZSBvciB0eXBlb2Ygc25hcHNob3RBdXhpbGlhcnlEYXRhIGlzICd1bmRlZmluZWQnXG4gICAgICBAbGVzc1NvdXJjZXNCeVJlbGF0aXZlRmlsZVBhdGggPSB7fVxuICAgICAgQGltcG9ydGVkRmlsZVBhdGhzQnlSZWxhdGl2ZUltcG9ydFBhdGggPSB7fVxuICAgIGVsc2VcbiAgICAgIEBsZXNzU291cmNlc0J5UmVsYXRpdmVGaWxlUGF0aCA9IHNuYXBzaG90QXV4aWxpYXJ5RGF0YS5sZXNzU291cmNlc0J5UmVsYXRpdmVGaWxlUGF0aFxuICAgICAgQGltcG9ydGVkRmlsZVBhdGhzQnlSZWxhdGl2ZUltcG9ydFBhdGggPSBzbmFwc2hvdEF1eGlsaWFyeURhdGEuaW1wb3J0ZWRGaWxlUGF0aHNCeVJlbGF0aXZlSW1wb3J0UGF0aFxuXG4gICMjI1xuICBTZWN0aW9uOiBFdmVudCBTdWJzY3JpcHRpb25cbiAgIyMjXG5cbiAgIyBFc3NlbnRpYWw6IEludm9rZSBgY2FsbGJhY2tgIHdoZW4gc3R5bGUgc2hlZXQgY2hhbmdlcyBhc3NvY2lhdGVkIHdpdGhcbiAgIyB1cGRhdGluZyB0aGUgbGlzdCBvZiBhY3RpdmUgdGhlbWVzIGhhdmUgY29tcGxldGVkLlxuICAjXG4gICMgKiBgY2FsbGJhY2tgIHtGdW5jdGlvbn1cbiAgb25EaWRDaGFuZ2VBY3RpdmVUaGVtZXM6IChjYWxsYmFjaykgLT5cbiAgICBAZW1pdHRlci5vbiAnZGlkLWNoYW5nZS1hY3RpdmUtdGhlbWVzJywgY2FsbGJhY2tcblxuICAjIyNcbiAgU2VjdGlvbjogQWNjZXNzaW5nIEF2YWlsYWJsZSBUaGVtZXNcbiAgIyMjXG5cbiAgZ2V0QXZhaWxhYmxlTmFtZXM6IC0+XG4gICAgIyBUT0RPOiBNYXliZSBzaG91bGQgY2hhbmdlIHRvIGxpc3QgYWxsIHRoZSBhdmFpbGFibGUgdGhlbWVzIG91dCB0aGVyZT9cbiAgICBAZ2V0TG9hZGVkTmFtZXMoKVxuXG4gICMjI1xuICBTZWN0aW9uOiBBY2Nlc3NpbmcgTG9hZGVkIFRoZW1lc1xuICAjIyNcblxuICAjIFB1YmxpYzogUmV0dXJucyBhbiB7QXJyYXl9IG9mIHtTdHJpbmd9cyBvZiBhbGwgdGhlIGxvYWRlZCB0aGVtZSBuYW1lcy5cbiAgZ2V0TG9hZGVkVGhlbWVOYW1lczogLT5cbiAgICB0aGVtZS5uYW1lIGZvciB0aGVtZSBpbiBAZ2V0TG9hZGVkVGhlbWVzKClcblxuICAjIFB1YmxpYzogUmV0dXJucyBhbiB7QXJyYXl9IG9mIGFsbCB0aGUgbG9hZGVkIHRoZW1lcy5cbiAgZ2V0TG9hZGVkVGhlbWVzOiAtPlxuICAgIHBhY2sgZm9yIHBhY2sgaW4gQHBhY2thZ2VNYW5hZ2VyLmdldExvYWRlZFBhY2thZ2VzKCkgd2hlbiBwYWNrLmlzVGhlbWUoKVxuXG4gICMjI1xuICBTZWN0aW9uOiBBY2Nlc3NpbmcgQWN0aXZlIFRoZW1lc1xuICAjIyNcblxuICAjIFB1YmxpYzogUmV0dXJucyBhbiB7QXJyYXl9IG9mIHtTdHJpbmd9cyBhbGwgdGhlIGFjdGl2ZSB0aGVtZSBuYW1lcy5cbiAgZ2V0QWN0aXZlVGhlbWVOYW1lczogLT5cbiAgICB0aGVtZS5uYW1lIGZvciB0aGVtZSBpbiBAZ2V0QWN0aXZlVGhlbWVzKClcblxuICAjIFB1YmxpYzogUmV0dXJucyBhbiB7QXJyYXl9IG9mIGFsbCB0aGUgYWN0aXZlIHRoZW1lcy5cbiAgZ2V0QWN0aXZlVGhlbWVzOiAtPlxuICAgIHBhY2sgZm9yIHBhY2sgaW4gQHBhY2thZ2VNYW5hZ2VyLmdldEFjdGl2ZVBhY2thZ2VzKCkgd2hlbiBwYWNrLmlzVGhlbWUoKVxuXG4gIGFjdGl2YXRlUGFja2FnZXM6IC0+IEBhY3RpdmF0ZVRoZW1lcygpXG5cbiAgIyMjXG4gIFNlY3Rpb246IE1hbmFnaW5nIEVuYWJsZWQgVGhlbWVzXG4gICMjI1xuXG4gIHdhcm5Gb3JOb25FeGlzdGVudFRoZW1lczogLT5cbiAgICB0aGVtZU5hbWVzID0gQGNvbmZpZy5nZXQoJ2NvcmUudGhlbWVzJykgPyBbXVxuICAgIHRoZW1lTmFtZXMgPSBbdGhlbWVOYW1lc10gdW5sZXNzIF8uaXNBcnJheSh0aGVtZU5hbWVzKVxuICAgIGZvciB0aGVtZU5hbWUgaW4gdGhlbWVOYW1lc1xuICAgICAgdW5sZXNzIHRoZW1lTmFtZSBhbmQgdHlwZW9mIHRoZW1lTmFtZSBpcyAnc3RyaW5nJyBhbmQgQHBhY2thZ2VNYW5hZ2VyLnJlc29sdmVQYWNrYWdlUGF0aCh0aGVtZU5hbWUpXG4gICAgICAgIGNvbnNvbGUud2FybihcIkVuYWJsZWQgdGhlbWUgJyN7dGhlbWVOYW1lfScgaXMgbm90IGluc3RhbGxlZC5cIilcblxuICAjIFB1YmxpYzogR2V0IHRoZSBlbmFibGVkIHRoZW1lIG5hbWVzIGZyb20gdGhlIGNvbmZpZy5cbiAgI1xuICAjIFJldHVybnMgYW4gYXJyYXkgb2YgdGhlbWUgbmFtZXMgaW4gdGhlIG9yZGVyIHRoYXQgdGhleSBzaG91bGQgYmUgYWN0aXZhdGVkLlxuICBnZXRFbmFibGVkVGhlbWVOYW1lczogLT5cbiAgICB0aGVtZU5hbWVzID0gQGNvbmZpZy5nZXQoJ2NvcmUudGhlbWVzJykgPyBbXVxuICAgIHRoZW1lTmFtZXMgPSBbdGhlbWVOYW1lc10gdW5sZXNzIF8uaXNBcnJheSh0aGVtZU5hbWVzKVxuICAgIHRoZW1lTmFtZXMgPSB0aGVtZU5hbWVzLmZpbHRlciAodGhlbWVOYW1lKSA9PlxuICAgICAgaWYgdGhlbWVOYW1lIGFuZCB0eXBlb2YgdGhlbWVOYW1lIGlzICdzdHJpbmcnXG4gICAgICAgIHJldHVybiB0cnVlIGlmIEBwYWNrYWdlTWFuYWdlci5yZXNvbHZlUGFja2FnZVBhdGgodGhlbWVOYW1lKVxuICAgICAgZmFsc2VcblxuICAgICMgVXNlIGEgYnVpbHQtaW4gc3ludGF4IGFuZCBVSSB0aGVtZSBhbnkgdGltZSB0aGUgY29uZmlndXJlZCB0aGVtZXMgYXJlIG5vdFxuICAgICMgYXZhaWxhYmxlLlxuICAgIGlmIHRoZW1lTmFtZXMubGVuZ3RoIDwgMlxuICAgICAgYnVpbHRJblRoZW1lTmFtZXMgPSBbXG4gICAgICAgICdhdG9tLWRhcmstc3ludGF4J1xuICAgICAgICAnYXRvbS1kYXJrLXVpJ1xuICAgICAgICAnYXRvbS1saWdodC1zeW50YXgnXG4gICAgICAgICdhdG9tLWxpZ2h0LXVpJ1xuICAgICAgICAnYmFzZTE2LXRvbW9ycm93LWRhcmstdGhlbWUnXG4gICAgICAgICdiYXNlMTYtdG9tb3Jyb3ctbGlnaHQtdGhlbWUnXG4gICAgICAgICdzb2xhcml6ZWQtZGFyay1zeW50YXgnXG4gICAgICAgICdzb2xhcml6ZWQtbGlnaHQtc3ludGF4J1xuICAgICAgXVxuICAgICAgdGhlbWVOYW1lcyA9IF8uaW50ZXJzZWN0aW9uKHRoZW1lTmFtZXMsIGJ1aWx0SW5UaGVtZU5hbWVzKVxuICAgICAgaWYgdGhlbWVOYW1lcy5sZW5ndGggaXMgMFxuICAgICAgICB0aGVtZU5hbWVzID0gWydhdG9tLWRhcmstc3ludGF4JywgJ2F0b20tZGFyay11aSddXG4gICAgICBlbHNlIGlmIHRoZW1lTmFtZXMubGVuZ3RoIGlzIDFcbiAgICAgICAgaWYgXy5lbmRzV2l0aCh0aGVtZU5hbWVzWzBdLCAnLXVpJylcbiAgICAgICAgICB0aGVtZU5hbWVzLnVuc2hpZnQoJ2F0b20tZGFyay1zeW50YXgnKVxuICAgICAgICBlbHNlXG4gICAgICAgICAgdGhlbWVOYW1lcy5wdXNoKCdhdG9tLWRhcmstdWknKVxuXG4gICAgIyBSZXZlcnNlIHNvIHRoZSBmaXJzdCAodG9wKSB0aGVtZSBpcyBsb2FkZWQgYWZ0ZXIgdGhlIG90aGVycy4gV2Ugd2FudFxuICAgICMgdGhlIGZpcnN0L3RvcCB0aGVtZSB0byBvdmVycmlkZSBsYXRlciB0aGVtZXMgaW4gdGhlIHN0YWNrLlxuICAgIHRoZW1lTmFtZXMucmV2ZXJzZSgpXG5cbiAgIyMjXG4gIFNlY3Rpb246IFByaXZhdGVcbiAgIyMjXG5cbiAgIyBSZXNvbHZlIGFuZCBhcHBseSB0aGUgc3R5bGVzaGVldCBzcGVjaWZpZWQgYnkgdGhlIHBhdGguXG4gICNcbiAgIyBUaGlzIHN1cHBvcnRzIGJvdGggQ1NTIGFuZCBMZXNzIHN0eWxzaGVldHMuXG4gICNcbiAgIyAqIGBzdHlsZXNoZWV0UGF0aGAgQSB7U3RyaW5nfSBwYXRoIHRvIHRoZSBzdHlsZXNoZWV0IHRoYXQgY2FuIGJlIGFuIGFic29sdXRlXG4gICMgICBwYXRoIG9yIGEgcmVsYXRpdmUgcGF0aCB0aGF0IHdpbGwgYmUgcmVzb2x2ZWQgYWdhaW5zdCB0aGUgbG9hZCBwYXRoLlxuICAjXG4gICMgUmV0dXJucyBhIHtEaXNwb3NhYmxlfSBvbiB3aGljaCBgLmRpc3Bvc2UoKWAgY2FuIGJlIGNhbGxlZCB0byByZW1vdmUgdGhlXG4gICMgcmVxdWlyZWQgc3R5bGVzaGVldC5cbiAgcmVxdWlyZVN0eWxlc2hlZXQ6IChzdHlsZXNoZWV0UGF0aCwgcHJpb3JpdHksIHNraXBEZXByZWNhdGVkU2VsZWN0b3JzVHJhbnNmb3JtYXRpb24pIC0+XG4gICAgaWYgZnVsbFBhdGggPSBAcmVzb2x2ZVN0eWxlc2hlZXQoc3R5bGVzaGVldFBhdGgpXG4gICAgICBjb250ZW50ID0gQGxvYWRTdHlsZXNoZWV0KGZ1bGxQYXRoKVxuICAgICAgQGFwcGx5U3R5bGVzaGVldChmdWxsUGF0aCwgY29udGVudCwgcHJpb3JpdHksIHNraXBEZXByZWNhdGVkU2VsZWN0b3JzVHJhbnNmb3JtYXRpb24pXG4gICAgZWxzZVxuICAgICAgdGhyb3cgbmV3IEVycm9yKFwiQ291bGQgbm90IGZpbmQgYSBmaWxlIGF0IHBhdGggJyN7c3R5bGVzaGVldFBhdGh9J1wiKVxuXG4gIHVud2F0Y2hVc2VyU3R5bGVzaGVldDogLT5cbiAgICBAdXNlclN0eWxzaGVldFN1YnNjcmlwdGlvbnM/LmRpc3Bvc2UoKVxuICAgIEB1c2VyU3R5bHNoZWV0U3Vic2NyaXB0aW9ucyA9IG51bGxcbiAgICBAdXNlclN0eWxlc2hlZXRGaWxlID0gbnVsbFxuICAgIEB1c2VyU3R5bGVTaGVldERpc3Bvc2FibGU/LmRpc3Bvc2UoKVxuICAgIEB1c2VyU3R5bGVTaGVldERpc3Bvc2FibGUgPSBudWxsXG5cbiAgbG9hZFVzZXJTdHlsZXNoZWV0OiAtPlxuICAgIEB1bndhdGNoVXNlclN0eWxlc2hlZXQoKVxuXG4gICAgdXNlclN0eWxlc2hlZXRQYXRoID0gQHN0eWxlTWFuYWdlci5nZXRVc2VyU3R5bGVTaGVldFBhdGgoKVxuICAgIHJldHVybiB1bmxlc3MgZnMuaXNGaWxlU3luYyh1c2VyU3R5bGVzaGVldFBhdGgpXG5cbiAgICB0cnlcbiAgICAgIEB1c2VyU3R5bGVzaGVldEZpbGUgPSBuZXcgRmlsZSh1c2VyU3R5bGVzaGVldFBhdGgpXG4gICAgICBAdXNlclN0eWxzaGVldFN1YnNjcmlwdGlvbnMgPSBuZXcgQ29tcG9zaXRlRGlzcG9zYWJsZSgpXG4gICAgICByZWxvYWRTdHlsZXNoZWV0ID0gPT4gQGxvYWRVc2VyU3R5bGVzaGVldCgpXG4gICAgICBAdXNlclN0eWxzaGVldFN1YnNjcmlwdGlvbnMuYWRkKEB1c2VyU3R5bGVzaGVldEZpbGUub25EaWRDaGFuZ2UocmVsb2FkU3R5bGVzaGVldCkpXG4gICAgICBAdXNlclN0eWxzaGVldFN1YnNjcmlwdGlvbnMuYWRkKEB1c2VyU3R5bGVzaGVldEZpbGUub25EaWRSZW5hbWUocmVsb2FkU3R5bGVzaGVldCkpXG4gICAgICBAdXNlclN0eWxzaGVldFN1YnNjcmlwdGlvbnMuYWRkKEB1c2VyU3R5bGVzaGVldEZpbGUub25EaWREZWxldGUocmVsb2FkU3R5bGVzaGVldCkpXG4gICAgY2F0Y2ggZXJyb3JcbiAgICAgIG1lc3NhZ2UgPSBcIlwiXCJcbiAgICAgICAgVW5hYmxlIHRvIHdhdGNoIHBhdGg6IGAje3BhdGguYmFzZW5hbWUodXNlclN0eWxlc2hlZXRQYXRoKX1gLiBNYWtlIHN1cmVcbiAgICAgICAgeW91IGhhdmUgcGVybWlzc2lvbnMgdG8gYCN7dXNlclN0eWxlc2hlZXRQYXRofWAuXG5cbiAgICAgICAgT24gbGludXggdGhlcmUgYXJlIGN1cnJlbnRseSBwcm9ibGVtcyB3aXRoIHdhdGNoIHNpemVzLiBTZWVcbiAgICAgICAgW3RoaXMgZG9jdW1lbnRdW3dhdGNoZXNdIGZvciBtb3JlIGluZm8uXG4gICAgICAgIFt3YXRjaGVzXTpodHRwczovL2dpdGh1Yi5jb20vYXRvbS9hdG9tL2Jsb2IvbWFzdGVyL2RvY3MvYnVpbGQtaW5zdHJ1Y3Rpb25zL2xpbnV4Lm1kI3R5cGVlcnJvci11bmFibGUtdG8td2F0Y2gtcGF0aFxuICAgICAgXCJcIlwiXG4gICAgICBAbm90aWZpY2F0aW9uTWFuYWdlci5hZGRFcnJvcihtZXNzYWdlLCBkaXNtaXNzYWJsZTogdHJ1ZSlcblxuICAgIHRyeVxuICAgICAgdXNlclN0eWxlc2hlZXRDb250ZW50cyA9IEBsb2FkU3R5bGVzaGVldCh1c2VyU3R5bGVzaGVldFBhdGgsIHRydWUpXG4gICAgY2F0Y2hcbiAgICAgIHJldHVyblxuXG4gICAgQHVzZXJTdHlsZVNoZWV0RGlzcG9zYWJsZSA9IEBzdHlsZU1hbmFnZXIuYWRkU3R5bGVTaGVldCh1c2VyU3R5bGVzaGVldENvbnRlbnRzLCBzb3VyY2VQYXRoOiB1c2VyU3R5bGVzaGVldFBhdGgsIHByaW9yaXR5OiAyKVxuXG4gIGxvYWRCYXNlU3R5bGVzaGVldHM6IC0+XG4gICAgQHJlbG9hZEJhc2VTdHlsZXNoZWV0cygpXG5cbiAgcmVsb2FkQmFzZVN0eWxlc2hlZXRzOiAtPlxuICAgIEByZXF1aXJlU3R5bGVzaGVldCgnLi4vc3RhdGljL2F0b20nLCAtMiwgdHJ1ZSlcblxuICBzdHlsZXNoZWV0RWxlbWVudEZvcklkOiAoaWQpIC0+XG4gICAgZXNjYXBlZElkID0gaWQucmVwbGFjZSgvXFxcXC9nLCAnXFxcXFxcXFwnKVxuICAgIGRvY3VtZW50LmhlYWQucXVlcnlTZWxlY3RvcihcImF0b20tc3R5bGVzIHN0eWxlW3NvdXJjZS1wYXRoPVxcXCIje2VzY2FwZWRJZH1cXFwiXVwiKVxuXG4gIHJlc29sdmVTdHlsZXNoZWV0OiAoc3R5bGVzaGVldFBhdGgpIC0+XG4gICAgaWYgcGF0aC5leHRuYW1lKHN0eWxlc2hlZXRQYXRoKS5sZW5ndGggPiAwXG4gICAgICBmcy5yZXNvbHZlT25Mb2FkUGF0aChzdHlsZXNoZWV0UGF0aClcbiAgICBlbHNlXG4gICAgICBmcy5yZXNvbHZlT25Mb2FkUGF0aChzdHlsZXNoZWV0UGF0aCwgWydjc3MnLCAnbGVzcyddKVxuXG4gIGxvYWRTdHlsZXNoZWV0OiAoc3R5bGVzaGVldFBhdGgsIGltcG9ydEZhbGxiYWNrVmFyaWFibGVzKSAtPlxuICAgIGlmIHBhdGguZXh0bmFtZShzdHlsZXNoZWV0UGF0aCkgaXMgJy5sZXNzJ1xuICAgICAgQGxvYWRMZXNzU3R5bGVzaGVldChzdHlsZXNoZWV0UGF0aCwgaW1wb3J0RmFsbGJhY2tWYXJpYWJsZXMpXG4gICAgZWxzZVxuICAgICAgZnMucmVhZEZpbGVTeW5jKHN0eWxlc2hlZXRQYXRoLCAndXRmOCcpXG5cbiAgbG9hZExlc3NTdHlsZXNoZWV0OiAobGVzc1N0eWxlc2hlZXRQYXRoLCBpbXBvcnRGYWxsYmFja1ZhcmlhYmxlcz1mYWxzZSkgLT5cbiAgICBAbGVzc0NhY2hlID89IG5ldyBMZXNzQ29tcGlsZUNhY2hlKHtcbiAgICAgIEByZXNvdXJjZVBhdGgsXG4gICAgICBAbGVzc1NvdXJjZXNCeVJlbGF0aXZlRmlsZVBhdGgsXG4gICAgICBAaW1wb3J0ZWRGaWxlUGF0aHNCeVJlbGF0aXZlSW1wb3J0UGF0aCxcbiAgICAgIGltcG9ydFBhdGhzOiBAZ2V0SW1wb3J0UGF0aHMoKVxuICAgIH0pXG5cbiAgICB0cnlcbiAgICAgIGlmIGltcG9ydEZhbGxiYWNrVmFyaWFibGVzXG4gICAgICAgIGJhc2VWYXJJbXBvcnRzID0gXCJcIlwiXG4gICAgICAgIEBpbXBvcnQgXCJ2YXJpYWJsZXMvdWktdmFyaWFibGVzXCI7XG4gICAgICAgIEBpbXBvcnQgXCJ2YXJpYWJsZXMvc3ludGF4LXZhcmlhYmxlc1wiO1xuICAgICAgICBcIlwiXCJcbiAgICAgICAgcmVsYXRpdmVGaWxlUGF0aCA9IHBhdGgucmVsYXRpdmUoQHJlc291cmNlUGF0aCwgbGVzc1N0eWxlc2hlZXRQYXRoKVxuICAgICAgICBsZXNzU291cmNlID0gQGxlc3NTb3VyY2VzQnlSZWxhdGl2ZUZpbGVQYXRoW3JlbGF0aXZlRmlsZVBhdGhdXG4gICAgICAgIGlmIGxlc3NTb3VyY2U/XG4gICAgICAgICAgY29udGVudCA9IGxlc3NTb3VyY2UuY29udGVudFxuICAgICAgICAgIGRpZ2VzdCA9IGxlc3NTb3VyY2UuZGlnZXN0XG4gICAgICAgIGVsc2VcbiAgICAgICAgICBjb250ZW50ID0gYmFzZVZhckltcG9ydHMgKyAnXFxuJyArIGZzLnJlYWRGaWxlU3luYyhsZXNzU3R5bGVzaGVldFBhdGgsICd1dGY4JylcbiAgICAgICAgICBkaWdlc3QgPSBudWxsXG5cbiAgICAgICAgQGxlc3NDYWNoZS5jc3NGb3JGaWxlKGxlc3NTdHlsZXNoZWV0UGF0aCwgY29udGVudCwgZGlnZXN0KVxuICAgICAgZWxzZVxuICAgICAgICBAbGVzc0NhY2hlLnJlYWQobGVzc1N0eWxlc2hlZXRQYXRoKVxuICAgIGNhdGNoIGVycm9yXG4gICAgICBlcnJvci5sZXNzID0gdHJ1ZVxuICAgICAgaWYgZXJyb3IubGluZT9cbiAgICAgICAgIyBBZGp1c3QgbGluZSBudW1iZXJzIGZvciBpbXBvcnQgZmFsbGJhY2tzXG4gICAgICAgIGVycm9yLmxpbmUgLT0gMiBpZiBpbXBvcnRGYWxsYmFja1ZhcmlhYmxlc1xuXG4gICAgICAgIG1lc3NhZ2UgPSBcIkVycm9yIGNvbXBpbGluZyBMZXNzIHN0eWxlc2hlZXQ6IGAje2xlc3NTdHlsZXNoZWV0UGF0aH1gXCJcbiAgICAgICAgZGV0YWlsID0gXCJcIlwiXG4gICAgICAgICAgTGluZSBudW1iZXI6ICN7ZXJyb3IubGluZX1cbiAgICAgICAgICAje2Vycm9yLm1lc3NhZ2V9XG4gICAgICAgIFwiXCJcIlxuICAgICAgZWxzZVxuICAgICAgICBtZXNzYWdlID0gXCJFcnJvciBsb2FkaW5nIExlc3Mgc3R5bGVzaGVldDogYCN7bGVzc1N0eWxlc2hlZXRQYXRofWBcIlxuICAgICAgICBkZXRhaWwgPSBlcnJvci5tZXNzYWdlXG5cbiAgICAgIEBub3RpZmljYXRpb25NYW5hZ2VyLmFkZEVycm9yKG1lc3NhZ2UsIHtkZXRhaWwsIGRpc21pc3NhYmxlOiB0cnVlfSlcbiAgICAgIHRocm93IGVycm9yXG5cbiAgcmVtb3ZlU3R5bGVzaGVldDogKHN0eWxlc2hlZXRQYXRoKSAtPlxuICAgIEBzdHlsZVNoZWV0RGlzcG9zYWJsZXNCeVNvdXJjZVBhdGhbc3R5bGVzaGVldFBhdGhdPy5kaXNwb3NlKClcblxuICBhcHBseVN0eWxlc2hlZXQ6IChwYXRoLCB0ZXh0LCBwcmlvcml0eSwgc2tpcERlcHJlY2F0ZWRTZWxlY3RvcnNUcmFuc2Zvcm1hdGlvbikgLT5cbiAgICBAc3R5bGVTaGVldERpc3Bvc2FibGVzQnlTb3VyY2VQYXRoW3BhdGhdID0gQHN0eWxlTWFuYWdlci5hZGRTdHlsZVNoZWV0KFxuICAgICAgdGV4dCxcbiAgICAgIHtcbiAgICAgICAgcHJpb3JpdHksXG4gICAgICAgIHNraXBEZXByZWNhdGVkU2VsZWN0b3JzVHJhbnNmb3JtYXRpb24sXG4gICAgICAgIHNvdXJjZVBhdGg6IHBhdGhcbiAgICAgIH1cbiAgICApXG5cbiAgYWN0aXZhdGVUaGVtZXM6IC0+XG4gICAgbmV3IFByb21pc2UgKHJlc29sdmUpID0+XG4gICAgICAjIEBjb25maWcub2JzZXJ2ZSBydW5zIHRoZSBjYWxsYmFjayBvbmNlLCB0aGVuIG9uIHN1YnNlcXVlbnQgY2hhbmdlcy5cbiAgICAgIEBjb25maWcub2JzZXJ2ZSAnY29yZS50aGVtZXMnLCA9PlxuICAgICAgICBAZGVhY3RpdmF0ZVRoZW1lcygpXG5cbiAgICAgICAgQHdhcm5Gb3JOb25FeGlzdGVudFRoZW1lcygpXG5cbiAgICAgICAgQHJlZnJlc2hMZXNzQ2FjaGUoKSAjIFVwZGF0ZSBjYWNoZSBmb3IgcGFja2FnZXMgaW4gY29yZS50aGVtZXMgY29uZmlnXG5cbiAgICAgICAgcHJvbWlzZXMgPSBbXVxuICAgICAgICBmb3IgdGhlbWVOYW1lIGluIEBnZXRFbmFibGVkVGhlbWVOYW1lcygpXG4gICAgICAgICAgaWYgQHBhY2thZ2VNYW5hZ2VyLnJlc29sdmVQYWNrYWdlUGF0aCh0aGVtZU5hbWUpXG4gICAgICAgICAgICBwcm9taXNlcy5wdXNoKEBwYWNrYWdlTWFuYWdlci5hY3RpdmF0ZVBhY2thZ2UodGhlbWVOYW1lKSlcbiAgICAgICAgICBlbHNlXG4gICAgICAgICAgICBjb25zb2xlLndhcm4oXCJGYWlsZWQgdG8gYWN0aXZhdGUgdGhlbWUgJyN7dGhlbWVOYW1lfScgYmVjYXVzZSBpdCBpc24ndCBpbnN0YWxsZWQuXCIpXG5cbiAgICAgICAgUHJvbWlzZS5hbGwocHJvbWlzZXMpLnRoZW4gPT5cbiAgICAgICAgICBAYWRkQWN0aXZlVGhlbWVDbGFzc2VzKClcbiAgICAgICAgICBAcmVmcmVzaExlc3NDYWNoZSgpICMgVXBkYXRlIGNhY2hlIGFnYWluIG5vdyB0aGF0IEBnZXRBY3RpdmVUaGVtZXMoKSBpcyBwb3B1bGF0ZWRcbiAgICAgICAgICBAbG9hZFVzZXJTdHlsZXNoZWV0KClcbiAgICAgICAgICBAcmVsb2FkQmFzZVN0eWxlc2hlZXRzKClcbiAgICAgICAgICBAaW5pdGlhbExvYWRDb21wbGV0ZSA9IHRydWVcbiAgICAgICAgICBAZW1pdHRlci5lbWl0ICdkaWQtY2hhbmdlLWFjdGl2ZS10aGVtZXMnXG4gICAgICAgICAgcmVzb2x2ZSgpXG5cbiAgZGVhY3RpdmF0ZVRoZW1lczogLT5cbiAgICBAcmVtb3ZlQWN0aXZlVGhlbWVDbGFzc2VzKClcbiAgICBAdW53YXRjaFVzZXJTdHlsZXNoZWV0KClcbiAgICBAcGFja2FnZU1hbmFnZXIuZGVhY3RpdmF0ZVBhY2thZ2UocGFjay5uYW1lKSBmb3IgcGFjayBpbiBAZ2V0QWN0aXZlVGhlbWVzKClcbiAgICBudWxsXG5cbiAgaXNJbml0aWFsTG9hZENvbXBsZXRlOiAtPiBAaW5pdGlhbExvYWRDb21wbGV0ZVxuXG4gIGFkZEFjdGl2ZVRoZW1lQ2xhc3NlczogLT5cbiAgICBpZiB3b3Jrc3BhY2VFbGVtZW50ID0gQHZpZXdSZWdpc3RyeS5nZXRWaWV3KEB3b3Jrc3BhY2UpXG4gICAgICBmb3IgcGFjayBpbiBAZ2V0QWN0aXZlVGhlbWVzKClcbiAgICAgICAgd29ya3NwYWNlRWxlbWVudC5jbGFzc0xpc3QuYWRkKFwidGhlbWUtI3twYWNrLm5hbWV9XCIpXG4gICAgICByZXR1cm5cblxuICByZW1vdmVBY3RpdmVUaGVtZUNsYXNzZXM6IC0+XG4gICAgd29ya3NwYWNlRWxlbWVudCA9IEB2aWV3UmVnaXN0cnkuZ2V0VmlldyhAd29ya3NwYWNlKVxuICAgIGZvciBwYWNrIGluIEBnZXRBY3RpdmVUaGVtZXMoKVxuICAgICAgd29ya3NwYWNlRWxlbWVudC5jbGFzc0xpc3QucmVtb3ZlKFwidGhlbWUtI3twYWNrLm5hbWV9XCIpXG4gICAgcmV0dXJuXG5cbiAgcmVmcmVzaExlc3NDYWNoZTogLT5cbiAgICBAbGVzc0NhY2hlPy5zZXRJbXBvcnRQYXRocyhAZ2V0SW1wb3J0UGF0aHMoKSlcblxuICBnZXRJbXBvcnRQYXRoczogLT5cbiAgICBhY3RpdmVUaGVtZXMgPSBAZ2V0QWN0aXZlVGhlbWVzKClcbiAgICBpZiBhY3RpdmVUaGVtZXMubGVuZ3RoID4gMFxuICAgICAgdGhlbWVQYXRocyA9ICh0aGVtZS5nZXRTdHlsZXNoZWV0c1BhdGgoKSBmb3IgdGhlbWUgaW4gYWN0aXZlVGhlbWVzIHdoZW4gdGhlbWUpXG4gICAgZWxzZVxuICAgICAgdGhlbWVQYXRocyA9IFtdXG4gICAgICBmb3IgdGhlbWVOYW1lIGluIEBnZXRFbmFibGVkVGhlbWVOYW1lcygpXG4gICAgICAgIGlmIHRoZW1lUGF0aCA9IEBwYWNrYWdlTWFuYWdlci5yZXNvbHZlUGFja2FnZVBhdGgodGhlbWVOYW1lKVxuICAgICAgICAgIGRlcHJlY2F0ZWRQYXRoID0gcGF0aC5qb2luKHRoZW1lUGF0aCwgJ3N0eWxlc2hlZXRzJylcbiAgICAgICAgICBpZiBmcy5pc0RpcmVjdG9yeVN5bmMoZGVwcmVjYXRlZFBhdGgpXG4gICAgICAgICAgICB0aGVtZVBhdGhzLnB1c2goZGVwcmVjYXRlZFBhdGgpXG4gICAgICAgICAgZWxzZVxuICAgICAgICAgICAgdGhlbWVQYXRocy5wdXNoKHBhdGguam9pbih0aGVtZVBhdGgsICdzdHlsZXMnKSlcblxuICAgIHRoZW1lUGF0aHMuZmlsdGVyICh0aGVtZVBhdGgpIC0+IGZzLmlzRGlyZWN0b3J5U3luYyh0aGVtZVBhdGgpXG4iXX0=
