(function() {
  var CompositeDisposable, Cursor, DecorationManager, Disposable, Emitter, Grim, GutterContainer, LanguageMode, MAX_SCREEN_LINE_LENGTH, Model, Point, Range, Selection, TextBuffer, TextEditor, TextEditorComponent, TextEditorElement, TextMateScopeSelector, TokenizedBuffer, ZERO_WIDTH_NBSP, _, fs, isDoubleWidthCharacter, isHalfWidthCharacter, isKoreanCharacter, isWrapBoundary, path, ref, ref1, ref2,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; },
    extend = function(child, parent) { for (var key in parent) { if (hasProp.call(parent, key)) child[key] = parent[key]; } function ctor() { this.constructor = child; } ctor.prototype = parent.prototype; child.prototype = new ctor(); child.__super__ = parent.prototype; return child; },
    hasProp = {}.hasOwnProperty,
    slice = [].slice;

  _ = require('underscore-plus');

  path = require('path');

  fs = require('fs-plus');

  Grim = require('grim');

  ref = require('event-kit'), CompositeDisposable = ref.CompositeDisposable, Disposable = ref.Disposable, Emitter = ref.Emitter;

  ref1 = TextBuffer = require('text-buffer'), Point = ref1.Point, Range = ref1.Range;

  LanguageMode = require('./language-mode');

  DecorationManager = require('./decoration-manager');

  TokenizedBuffer = require('./tokenized-buffer');

  Cursor = require('./cursor');

  Model = require('./model');

  Selection = require('./selection');

  TextMateScopeSelector = require('first-mate').ScopeSelector;

  GutterContainer = require('./gutter-container');

  TextEditorComponent = null;

  TextEditorElement = null;

  ref2 = require('./text-utils'), isDoubleWidthCharacter = ref2.isDoubleWidthCharacter, isHalfWidthCharacter = ref2.isHalfWidthCharacter, isKoreanCharacter = ref2.isKoreanCharacter, isWrapBoundary = ref2.isWrapBoundary;

  ZERO_WIDTH_NBSP = '\ufeff';

  MAX_SCREEN_LINE_LENGTH = 500;

  module.exports = TextEditor = (function(superClass) {
    extend(TextEditor, superClass);

    TextEditor.setClipboard = function(clipboard) {
      return this.clipboard = clipboard;
    };

    TextEditor.setScheduler = function(scheduler) {
      if (TextEditorComponent == null) {
        TextEditorComponent = require('./text-editor-component');
      }
      return TextEditorComponent.setScheduler(scheduler);
    };

    TextEditor.didUpdateStyles = function() {
      if (TextEditorComponent == null) {
        TextEditorComponent = require('./text-editor-component');
      }
      return TextEditorComponent.didUpdateStyles();
    };

    TextEditor.didUpdateScrollbarStyles = function() {
      if (TextEditorComponent == null) {
        TextEditorComponent = require('./text-editor-component');
      }
      return TextEditorComponent.didUpdateScrollbarStyles();
    };

    TextEditor.viewForItem = function(item) {
      var ref3;
      return (ref3 = item.element) != null ? ref3 : item;
    };

    TextEditor.prototype.serializationVersion = 1;

    TextEditor.prototype.buffer = null;

    TextEditor.prototype.languageMode = null;

    TextEditor.prototype.cursors = null;

    TextEditor.prototype.showCursorOnSelection = null;

    TextEditor.prototype.selections = null;

    TextEditor.prototype.suppressSelectionMerging = false;

    TextEditor.prototype.selectionFlashDuration = 500;

    TextEditor.prototype.gutterContainer = null;

    TextEditor.prototype.editorElement = null;

    TextEditor.prototype.verticalScrollMargin = 2;

    TextEditor.prototype.horizontalScrollMargin = 6;

    TextEditor.prototype.softWrapped = null;

    TextEditor.prototype.editorWidthInChars = null;

    TextEditor.prototype.lineHeightInPixels = null;

    TextEditor.prototype.defaultCharWidth = null;

    TextEditor.prototype.height = null;

    TextEditor.prototype.width = null;

    TextEditor.prototype.registered = false;

    TextEditor.prototype.atomicSoftTabs = true;

    TextEditor.prototype.invisibles = null;

    Object.defineProperty(TextEditor.prototype, "element", {
      get: function() {
        return this.getElement();
      }
    });

    Object.defineProperty(TextEditor.prototype, "editorElement", {
      get: function() {
        Grim.deprecate("`TextEditor.prototype.editorElement` has always been private, but now\nit is gone. Reading the `editorElement` property still returns a\nreference to the editor element but this field will be removed in a\nlater version of Atom, so we recommend using the `element` property instead.");
        return this.getElement();
      }
    });

    Object.defineProperty(TextEditor.prototype, 'displayBuffer', {
      get: function() {
        Grim.deprecate("`TextEditor.prototype.displayBuffer` has always been private, but now\nit is gone. Reading the `displayBuffer` property now returns a reference\nto the containing `TextEditor`, which now provides *some* of the API of\nthe defunct `DisplayBuffer` class.");
        return this;
      }
    });

    TextEditor.deserialize = function(state, atomEnvironment) {
      var disposable, editor, error;
      if (state.version !== this.prototype.serializationVersion && (state.displayBuffer != null)) {
        state.tokenizedBuffer = state.displayBuffer.tokenizedBuffer;
      }
      try {
        state.tokenizedBuffer = TokenizedBuffer.deserialize(state.tokenizedBuffer, atomEnvironment);
        state.tabLength = state.tokenizedBuffer.getTabLength();
      } catch (error1) {
        error = error1;
        if (error.syscall === 'read') {
          return;
        } else {
          throw error;
        }
      }
      state.buffer = state.tokenizedBuffer.buffer;
      state.assert = atomEnvironment.assert.bind(atomEnvironment);
      editor = new this(state);
      if (state.registered) {
        disposable = atomEnvironment.textEditors.add(editor);
        editor.onDidDestroy(function() {
          return disposable.dispose();
        });
      }
      return editor;
    };

    function TextEditor(params) {
      var displayLayerParams, grammar, initialColumn, initialLine, l, len, lineNumberGutterVisible, marker, ref3, ref4, ref5, suppressCursorCreation, tabLength;
      if (params == null) {
        params = {};
      }
      this.doBackgroundWork = bind(this.doBackgroundWork, this);
      if (this.constructor.clipboard == null) {
        throw new Error("Must call TextEditor.setClipboard at least once before creating TextEditor instances");
      }
      TextEditor.__super__.constructor.apply(this, arguments);
      this.softTabs = params.softTabs, this.initialScrollTopRow = params.initialScrollTopRow, this.initialScrollLeftColumn = params.initialScrollLeftColumn, initialLine = params.initialLine, initialColumn = params.initialColumn, tabLength = params.tabLength, this.softWrapped = params.softWrapped, this.decorationManager = params.decorationManager, this.selectionsMarkerLayer = params.selectionsMarkerLayer, this.buffer = params.buffer, suppressCursorCreation = params.suppressCursorCreation, this.mini = params.mini, this.placeholderText = params.placeholderText, lineNumberGutterVisible = params.lineNumberGutterVisible, this.showLineNumbers = params.showLineNumbers, this.largeFileMode = params.largeFileMode, this.assert = params.assert, grammar = params.grammar, this.showInvisibles = params.showInvisibles, this.autoHeight = params.autoHeight, this.autoWidth = params.autoWidth, this.scrollPastEnd = params.scrollPastEnd, this.scrollSensitivity = params.scrollSensitivity, this.editorWidthInChars = params.editorWidthInChars, this.tokenizedBuffer = params.tokenizedBuffer, this.displayLayer = params.displayLayer, this.invisibles = params.invisibles, this.showIndentGuide = params.showIndentGuide, this.softWrapped = params.softWrapped, this.softWrapAtPreferredLineLength = params.softWrapAtPreferredLineLength, this.preferredLineLength = params.preferredLineLength, this.showCursorOnSelection = params.showCursorOnSelection;
      if (this.assert == null) {
        this.assert = function(condition) {
          return condition;
        };
      }
      this.emitter = new Emitter;
      this.disposables = new CompositeDisposable;
      this.cursors = [];
      this.cursorsByMarkerId = new Map;
      this.selections = [];
      this.hasTerminatedPendingState = false;
      if (this.mini == null) {
        this.mini = false;
      }
      if (this.scrollPastEnd == null) {
        this.scrollPastEnd = false;
      }
      if (this.scrollSensitivity == null) {
        this.scrollSensitivity = 40;
      }
      if (this.showInvisibles == null) {
        this.showInvisibles = true;
      }
      if (this.softTabs == null) {
        this.softTabs = true;
      }
      if (tabLength == null) {
        tabLength = 2;
      }
      if (this.autoIndent == null) {
        this.autoIndent = true;
      }
      if (this.autoIndentOnPaste == null) {
        this.autoIndentOnPaste = true;
      }
      if (this.showCursorOnSelection == null) {
        this.showCursorOnSelection = true;
      }
      if (this.undoGroupingInterval == null) {
        this.undoGroupingInterval = 300;
      }
      if (this.nonWordCharacters == null) {
        this.nonWordCharacters = "/\\()\"':,.;<>~!@#$%^&*|+=[]{}`?-…";
      }
      if (this.softWrapped == null) {
        this.softWrapped = false;
      }
      if (this.softWrapAtPreferredLineLength == null) {
        this.softWrapAtPreferredLineLength = false;
      }
      if (this.preferredLineLength == null) {
        this.preferredLineLength = 80;
      }
      if (this.showLineNumbers == null) {
        this.showLineNumbers = true;
      }
      if (this.buffer == null) {
        this.buffer = new TextBuffer({
          shouldDestroyOnFileDelete: function() {
            return atom.config.get('core.closeDeletedFileTabs');
          }
        });
      }
      if (this.tokenizedBuffer == null) {
        this.tokenizedBuffer = new TokenizedBuffer({
          grammar: grammar,
          tabLength: tabLength,
          buffer: this.buffer,
          largeFileMode: this.largeFileMode,
          assert: this.assert
        });
      }
      if (this.displayLayer == null) {
        displayLayerParams = {
          invisibles: this.getInvisibles(),
          softWrapColumn: this.getSoftWrapColumn(),
          showIndentGuides: this.doesShowIndentGuide(),
          atomicSoftTabs: (ref3 = params.atomicSoftTabs) != null ? ref3 : true,
          tabLength: tabLength,
          ratioForCharacter: this.ratioForCharacter.bind(this),
          isWrapBoundary: isWrapBoundary,
          foldCharacter: ZERO_WIDTH_NBSP,
          softWrapHangingIndent: (ref4 = params.softWrapHangingIndentLength) != null ? ref4 : 0
        };
        if (this.displayLayer = this.buffer.getDisplayLayer(params.displayLayerId)) {
          this.displayLayer.reset(displayLayerParams);
          this.selectionsMarkerLayer = this.displayLayer.getMarkerLayer(params.selectionsMarkerLayerId);
        } else {
          this.displayLayer = this.buffer.addDisplayLayer(displayLayerParams);
        }
      }
      this.backgroundWorkHandle = requestIdleCallback(this.doBackgroundWork);
      this.disposables.add(new Disposable((function(_this) {
        return function() {
          if (_this.backgroundWorkHandle != null) {
            return cancelIdleCallback(_this.backgroundWorkHandle);
          }
        };
      })(this)));
      this.displayLayer.setTextDecorationLayer(this.tokenizedBuffer);
      this.defaultMarkerLayer = this.displayLayer.addMarkerLayer();
      this.disposables.add(this.defaultMarkerLayer.onDidDestroy((function(_this) {
        return function() {
          return _this.assert(false, "defaultMarkerLayer destroyed at an unexpected time");
        };
      })(this)));
      if (this.selectionsMarkerLayer == null) {
        this.selectionsMarkerLayer = this.addMarkerLayer({
          maintainHistory: true,
          persistent: true
        });
      }
      this.selectionsMarkerLayer.trackDestructionInOnDidCreateMarkerCallbacks = true;
      this.decorationManager = new DecorationManager(this);
      this.decorateMarkerLayer(this.selectionsMarkerLayer, {
        type: 'cursor'
      });
      if (!this.isMini()) {
        this.decorateCursorLine();
      }
      this.decorateMarkerLayer(this.displayLayer.foldsMarkerLayer, {
        type: 'line-number',
        "class": 'folded'
      });
      ref5 = this.selectionsMarkerLayer.getMarkers();
      for (l = 0, len = ref5.length; l < len; l++) {
        marker = ref5[l];
        this.addSelection(marker);
      }
      this.subscribeToBuffer();
      this.subscribeToDisplayLayer();
      if (this.cursors.length === 0 && !suppressCursorCreation) {
        initialLine = Math.max(parseInt(initialLine) || 0, 0);
        initialColumn = Math.max(parseInt(initialColumn) || 0, 0);
        this.addCursorAtBufferPosition([initialLine, initialColumn]);
      }
      this.languageMode = new LanguageMode(this);
      this.gutterContainer = new GutterContainer(this);
      this.lineNumberGutter = this.gutterContainer.addGutter({
        name: 'line-number',
        priority: 0,
        visible: lineNumberGutterVisible
      });
    }

    TextEditor.prototype.decorateCursorLine = function() {
      return this.cursorLineDecorations = [
        this.decorateMarkerLayer(this.selectionsMarkerLayer, {
          type: 'line',
          "class": 'cursor-line',
          onlyEmpty: true
        }), this.decorateMarkerLayer(this.selectionsMarkerLayer, {
          type: 'line-number',
          "class": 'cursor-line'
        }), this.decorateMarkerLayer(this.selectionsMarkerLayer, {
          type: 'line-number',
          "class": 'cursor-line-no-selection',
          onlyHead: true,
          onlyEmpty: true
        })
      ];
    };

    TextEditor.prototype.doBackgroundWork = function(deadline) {
      var previousLongestRow, ref3;
      previousLongestRow = this.getApproximateLongestScreenRow();
      if (this.displayLayer.doBackgroundWork(deadline)) {
        this.backgroundWorkHandle = requestIdleCallback(this.doBackgroundWork);
      } else {
        this.backgroundWorkHandle = null;
      }
      if (this.getApproximateLongestScreenRow() !== previousLongestRow) {
        return (ref3 = this.component) != null ? ref3.scheduleUpdate() : void 0;
      }
    };

    TextEditor.prototype.update = function(params) {
      var decoration, displayLayerParams, l, len, len1, m, param, ref3, ref4, ref5, ref6, ref7, ref8, value;
      displayLayerParams = {};
      ref3 = Object.keys(params);
      for (l = 0, len = ref3.length; l < len; l++) {
        param = ref3[l];
        value = params[param];
        switch (param) {
          case 'autoIndent':
            this.autoIndent = value;
            break;
          case 'autoIndentOnPaste':
            this.autoIndentOnPaste = value;
            break;
          case 'undoGroupingInterval':
            this.undoGroupingInterval = value;
            break;
          case 'nonWordCharacters':
            this.nonWordCharacters = value;
            break;
          case 'scrollSensitivity':
            this.scrollSensitivity = value;
            break;
          case 'encoding':
            this.buffer.setEncoding(value);
            break;
          case 'softTabs':
            if (value !== this.softTabs) {
              this.softTabs = value;
            }
            break;
          case 'atomicSoftTabs':
            if (value !== this.displayLayer.atomicSoftTabs) {
              displayLayerParams.atomicSoftTabs = value;
            }
            break;
          case 'tabLength':
            if ((value != null) && value !== this.tokenizedBuffer.getTabLength()) {
              this.tokenizedBuffer.setTabLength(value);
              displayLayerParams.tabLength = value;
            }
            break;
          case 'softWrapped':
            if (value !== this.softWrapped) {
              this.softWrapped = value;
              displayLayerParams.softWrapColumn = this.getSoftWrapColumn();
              this.emitter.emit('did-change-soft-wrapped', this.isSoftWrapped());
            }
            break;
          case 'softWrapHangingIndentLength':
            if (value !== this.displayLayer.softWrapHangingIndent) {
              displayLayerParams.softWrapHangingIndent = value;
            }
            break;
          case 'softWrapAtPreferredLineLength':
            if (value !== this.softWrapAtPreferredLineLength) {
              this.softWrapAtPreferredLineLength = value;
              displayLayerParams.softWrapColumn = this.getSoftWrapColumn();
            }
            break;
          case 'preferredLineLength':
            if (value !== this.preferredLineLength) {
              this.preferredLineLength = value;
              displayLayerParams.softWrapColumn = this.getSoftWrapColumn();
            }
            break;
          case 'mini':
            if (value !== this.mini) {
              this.mini = value;
              this.emitter.emit('did-change-mini', value);
              displayLayerParams.invisibles = this.getInvisibles();
              displayLayerParams.softWrapColumn = this.getSoftWrapColumn();
              displayLayerParams.showIndentGuides = this.doesShowIndentGuide();
              if (this.mini) {
                ref4 = this.cursorLineDecorations;
                for (m = 0, len1 = ref4.length; m < len1; m++) {
                  decoration = ref4[m];
                  decoration.destroy();
                }
                this.cursorLineDecorations = null;
              } else {
                this.decorateCursorLine();
              }
              if ((ref5 = this.component) != null) {
                ref5.scheduleUpdate();
              }
            }
            break;
          case 'placeholderText':
            if (value !== this.placeholderText) {
              this.placeholderText = value;
              this.emitter.emit('did-change-placeholder-text', value);
            }
            break;
          case 'lineNumberGutterVisible':
            if (value !== this.lineNumberGutterVisible) {
              if (value) {
                this.lineNumberGutter.show();
              } else {
                this.lineNumberGutter.hide();
              }
              this.emitter.emit('did-change-line-number-gutter-visible', this.lineNumberGutter.isVisible());
            }
            break;
          case 'showIndentGuide':
            if (value !== this.showIndentGuide) {
              this.showIndentGuide = value;
              displayLayerParams.showIndentGuides = this.doesShowIndentGuide();
            }
            break;
          case 'showLineNumbers':
            if (value !== this.showLineNumbers) {
              this.showLineNumbers = value;
              if ((ref6 = this.component) != null) {
                ref6.scheduleUpdate();
              }
            }
            break;
          case 'showInvisibles':
            if (value !== this.showInvisibles) {
              this.showInvisibles = value;
              displayLayerParams.invisibles = this.getInvisibles();
            }
            break;
          case 'invisibles':
            if (!_.isEqual(value, this.invisibles)) {
              this.invisibles = value;
              displayLayerParams.invisibles = this.getInvisibles();
            }
            break;
          case 'editorWidthInChars':
            if (value > 0 && value !== this.editorWidthInChars) {
              this.editorWidthInChars = value;
              displayLayerParams.softWrapColumn = this.getSoftWrapColumn();
            }
            break;
          case 'width':
            if (value !== this.width) {
              this.width = value;
              displayLayerParams.softWrapColumn = this.getSoftWrapColumn();
            }
            break;
          case 'scrollPastEnd':
            if (value !== this.scrollPastEnd) {
              this.scrollPastEnd = value;
              if ((ref7 = this.component) != null) {
                ref7.scheduleUpdate();
              }
            }
            break;
          case 'autoHeight':
            if (value !== this.autoHeight) {
              this.autoHeight = value;
            }
            break;
          case 'autoWidth':
            if (value !== this.autoWidth) {
              this.autoWidth = value;
            }
            break;
          case 'showCursorOnSelection':
            if (value !== this.showCursorOnSelection) {
              this.showCursorOnSelection = value;
              if ((ref8 = this.component) != null) {
                ref8.scheduleUpdate();
              }
            }
            break;
          default:
            if (param !== 'ref' && param !== 'key') {
              throw new TypeError("Invalid TextEditor parameter: '" + param + "'");
            }
        }
      }
      this.displayLayer.reset(displayLayerParams);
      if (this.component != null) {
        return this.component.getNextUpdatePromise();
      } else {
        return Promise.resolve();
      }
    };

    TextEditor.prototype.scheduleComponentUpdate = function() {
      var ref3;
      return (ref3 = this.component) != null ? ref3.scheduleUpdate() : void 0;
    };

    TextEditor.prototype.serialize = function() {
      var tokenizedBufferState;
      tokenizedBufferState = this.tokenizedBuffer.serialize();
      return {
        deserializer: 'TextEditor',
        version: this.serializationVersion,
        displayBuffer: {
          tokenizedBuffer: tokenizedBufferState
        },
        tokenizedBuffer: tokenizedBufferState,
        displayLayerId: this.displayLayer.id,
        selectionsMarkerLayerId: this.selectionsMarkerLayer.id,
        initialScrollTopRow: this.getScrollTopRow(),
        initialScrollLeftColumn: this.getScrollLeftColumn(),
        atomicSoftTabs: this.displayLayer.atomicSoftTabs,
        softWrapHangingIndentLength: this.displayLayer.softWrapHangingIndent,
        id: this.id,
        softTabs: this.softTabs,
        softWrapped: this.softWrapped,
        softWrapAtPreferredLineLength: this.softWrapAtPreferredLineLength,
        preferredLineLength: this.preferredLineLength,
        mini: this.mini,
        editorWidthInChars: this.editorWidthInChars,
        width: this.width,
        largeFileMode: this.largeFileMode,
        registered: this.registered,
        invisibles: this.invisibles,
        showInvisibles: this.showInvisibles,
        showIndentGuide: this.showIndentGuide,
        autoHeight: this.autoHeight,
        autoWidth: this.autoWidth
      };
    };

    TextEditor.prototype.subscribeToBuffer = function() {
      this.buffer.retain();
      this.disposables.add(this.buffer.onDidChangePath((function(_this) {
        return function() {
          _this.emitter.emit('did-change-title', _this.getTitle());
          return _this.emitter.emit('did-change-path', _this.getPath());
        };
      })(this)));
      this.disposables.add(this.buffer.onDidChangeEncoding((function(_this) {
        return function() {
          return _this.emitter.emit('did-change-encoding', _this.getEncoding());
        };
      })(this)));
      this.disposables.add(this.buffer.onDidDestroy((function(_this) {
        return function() {
          return _this.destroy();
        };
      })(this)));
      return this.disposables.add(this.buffer.onDidChangeModified((function(_this) {
        return function() {
          if (!_this.hasTerminatedPendingState && _this.buffer.isModified()) {
            return _this.terminatePendingState();
          }
        };
      })(this)));
    };

    TextEditor.prototype.terminatePendingState = function() {
      if (!this.hasTerminatedPendingState) {
        this.emitter.emit('did-terminate-pending-state');
      }
      return this.hasTerminatedPendingState = true;
    };

    TextEditor.prototype.onDidTerminatePendingState = function(callback) {
      return this.emitter.on('did-terminate-pending-state', callback);
    };

    TextEditor.prototype.subscribeToDisplayLayer = function() {
      this.disposables.add(this.tokenizedBuffer.onDidChangeGrammar(this.handleGrammarChange.bind(this)));
      this.disposables.add(this.displayLayer.onDidChangeSync((function(_this) {
        return function(e) {
          var ref3;
          _this.mergeIntersectingSelections();
          if ((ref3 = _this.component) != null) {
            ref3.didChangeDisplayLayer(e);
          }
          return _this.emitter.emit('did-change', e);
        };
      })(this)));
      this.disposables.add(this.displayLayer.onDidReset((function(_this) {
        return function() {
          var ref3;
          _this.mergeIntersectingSelections();
          if ((ref3 = _this.component) != null) {
            ref3.didResetDisplayLayer();
          }
          return _this.emitter.emit('did-change', {});
        };
      })(this)));
      this.disposables.add(this.selectionsMarkerLayer.onDidCreateMarker(this.addSelection.bind(this)));
      return this.disposables.add(this.selectionsMarkerLayer.onDidUpdate((function(_this) {
        return function() {
          var ref3;
          return (ref3 = _this.component) != null ? ref3.didUpdateSelections() : void 0;
        };
      })(this)));
    };

    TextEditor.prototype.destroyed = function() {
      var l, len, ref3, ref4, selection;
      this.disposables.dispose();
      this.displayLayer.destroy();
      this.tokenizedBuffer.destroy();
      ref3 = this.selections.slice();
      for (l = 0, len = ref3.length; l < len; l++) {
        selection = ref3[l];
        selection.destroy();
      }
      this.buffer.release();
      this.languageMode.destroy();
      this.gutterContainer.destroy();
      this.emitter.emit('did-destroy');
      this.emitter.clear();
      if ((ref4 = this.component) != null) {
        ref4.element.component = null;
      }
      this.component = null;
      return this.lineNumberGutter.element = null;
    };


    /*
    Section: Event Subscription
     */

    TextEditor.prototype.onDidChangeTitle = function(callback) {
      return this.emitter.on('did-change-title', callback);
    };

    TextEditor.prototype.onDidChangePath = function(callback) {
      return this.emitter.on('did-change-path', callback);
    };

    TextEditor.prototype.onDidChange = function(callback) {
      return this.emitter.on('did-change', callback);
    };

    TextEditor.prototype.onDidStopChanging = function(callback) {
      return this.getBuffer().onDidStopChanging(callback);
    };

    TextEditor.prototype.onDidChangeCursorPosition = function(callback) {
      return this.emitter.on('did-change-cursor-position', callback);
    };

    TextEditor.prototype.onDidChangeSelectionRange = function(callback) {
      return this.emitter.on('did-change-selection-range', callback);
    };

    TextEditor.prototype.onDidChangeSoftWrapped = function(callback) {
      return this.emitter.on('did-change-soft-wrapped', callback);
    };

    TextEditor.prototype.onDidChangeEncoding = function(callback) {
      return this.emitter.on('did-change-encoding', callback);
    };

    TextEditor.prototype.observeGrammar = function(callback) {
      callback(this.getGrammar());
      return this.onDidChangeGrammar(callback);
    };

    TextEditor.prototype.onDidChangeGrammar = function(callback) {
      return this.emitter.on('did-change-grammar', callback);
    };

    TextEditor.prototype.onDidChangeModified = function(callback) {
      return this.getBuffer().onDidChangeModified(callback);
    };

    TextEditor.prototype.onDidConflict = function(callback) {
      return this.getBuffer().onDidConflict(callback);
    };

    TextEditor.prototype.onWillInsertText = function(callback) {
      return this.emitter.on('will-insert-text', callback);
    };

    TextEditor.prototype.onDidInsertText = function(callback) {
      return this.emitter.on('did-insert-text', callback);
    };

    TextEditor.prototype.onDidSave = function(callback) {
      return this.getBuffer().onDidSave(callback);
    };

    TextEditor.prototype.onDidDestroy = function(callback) {
      return this.emitter.on('did-destroy', callback);
    };

    TextEditor.prototype.observeCursors = function(callback) {
      var cursor, l, len, ref3;
      ref3 = this.getCursors();
      for (l = 0, len = ref3.length; l < len; l++) {
        cursor = ref3[l];
        callback(cursor);
      }
      return this.onDidAddCursor(callback);
    };

    TextEditor.prototype.onDidAddCursor = function(callback) {
      return this.emitter.on('did-add-cursor', callback);
    };

    TextEditor.prototype.onDidRemoveCursor = function(callback) {
      return this.emitter.on('did-remove-cursor', callback);
    };

    TextEditor.prototype.observeSelections = function(callback) {
      var l, len, ref3, selection;
      ref3 = this.getSelections();
      for (l = 0, len = ref3.length; l < len; l++) {
        selection = ref3[l];
        callback(selection);
      }
      return this.onDidAddSelection(callback);
    };

    TextEditor.prototype.onDidAddSelection = function(callback) {
      return this.emitter.on('did-add-selection', callback);
    };

    TextEditor.prototype.onDidRemoveSelection = function(callback) {
      return this.emitter.on('did-remove-selection', callback);
    };

    TextEditor.prototype.observeDecorations = function(callback) {
      return this.decorationManager.observeDecorations(callback);
    };

    TextEditor.prototype.onDidAddDecoration = function(callback) {
      return this.decorationManager.onDidAddDecoration(callback);
    };

    TextEditor.prototype.onDidRemoveDecoration = function(callback) {
      return this.decorationManager.onDidRemoveDecoration(callback);
    };

    TextEditor.prototype.didAddDecoration = function(decoration) {
      var ref3;
      if (decoration.isType('block')) {
        return (ref3 = this.component) != null ? ref3.didAddBlockDecoration(decoration) : void 0;
      }
    };

    TextEditor.prototype.onDidChangePlaceholderText = function(callback) {
      return this.emitter.on('did-change-placeholder-text', callback);
    };

    TextEditor.prototype.onDidChangeScrollTop = function(callback) {
      Grim.deprecate("This is now a view method. Call TextEditorElement::onDidChangeScrollTop instead.");
      return this.getElement().onDidChangeScrollTop(callback);
    };

    TextEditor.prototype.onDidChangeScrollLeft = function(callback) {
      Grim.deprecate("This is now a view method. Call TextEditorElement::onDidChangeScrollLeft instead.");
      return this.getElement().onDidChangeScrollLeft(callback);
    };

    TextEditor.prototype.onDidRequestAutoscroll = function(callback) {
      return this.emitter.on('did-request-autoscroll', callback);
    };

    TextEditor.prototype.onDidChangeIcon = function(callback) {
      return this.emitter.on('did-change-icon', callback);
    };

    TextEditor.prototype.onDidUpdateDecorations = function(callback) {
      return this.decorationManager.onDidUpdateDecorations(callback);
    };

    TextEditor.prototype.getBuffer = function() {
      return this.buffer;
    };

    TextEditor.prototype.getURI = function() {
      return this.buffer.getUri();
    };

    TextEditor.prototype.copy = function() {
      var displayLayer, selectionsMarkerLayer, softTabs;
      displayLayer = this.displayLayer.copy();
      selectionsMarkerLayer = displayLayer.getMarkerLayer(this.buffer.getMarkerLayer(this.selectionsMarkerLayer.id).copy().id);
      softTabs = this.getSoftTabs();
      return new TextEditor({
        buffer: this.buffer,
        selectionsMarkerLayer: selectionsMarkerLayer,
        softTabs: softTabs,
        suppressCursorCreation: true,
        tabLength: this.tokenizedBuffer.getTabLength(),
        initialScrollTopRow: this.getScrollTopRow(),
        initialScrollLeftColumn: this.getScrollLeftColumn(),
        assert: this.assert,
        displayLayer: displayLayer,
        grammar: this.getGrammar(),
        autoWidth: this.autoWidth,
        autoHeight: this.autoHeight,
        showCursorOnSelection: this.showCursorOnSelection
      });
    };

    TextEditor.prototype.setVisible = function(visible) {
      return this.tokenizedBuffer.setVisible(visible);
    };

    TextEditor.prototype.setMini = function(mini) {
      this.update({
        mini: mini
      });
      return this.mini;
    };

    TextEditor.prototype.isMini = function() {
      return this.mini;
    };

    TextEditor.prototype.onDidChangeMini = function(callback) {
      return this.emitter.on('did-change-mini', callback);
    };

    TextEditor.prototype.setLineNumberGutterVisible = function(lineNumberGutterVisible) {
      return this.update({
        lineNumberGutterVisible: lineNumberGutterVisible
      });
    };

    TextEditor.prototype.isLineNumberGutterVisible = function() {
      return this.lineNumberGutter.isVisible();
    };

    TextEditor.prototype.onDidChangeLineNumberGutterVisible = function(callback) {
      return this.emitter.on('did-change-line-number-gutter-visible', callback);
    };

    TextEditor.prototype.observeGutters = function(callback) {
      return this.gutterContainer.observeGutters(callback);
    };

    TextEditor.prototype.onDidAddGutter = function(callback) {
      return this.gutterContainer.onDidAddGutter(callback);
    };

    TextEditor.prototype.onDidRemoveGutter = function(callback) {
      return this.gutterContainer.onDidRemoveGutter(callback);
    };

    TextEditor.prototype.setEditorWidthInChars = function(editorWidthInChars) {
      return this.update({
        editorWidthInChars: editorWidthInChars
      });
    };

    TextEditor.prototype.getEditorWidthInChars = function() {
      if ((this.width != null) && this.defaultCharWidth > 0) {
        return Math.max(0, Math.floor(this.width / this.defaultCharWidth));
      } else {
        return this.editorWidthInChars;
      }
    };


    /*
    Section: File Details
     */

    TextEditor.prototype.getTitle = function() {
      var ref3;
      return (ref3 = this.getFileName()) != null ? ref3 : 'untitled';
    };

    TextEditor.prototype.getLongTitle = function() {
      var allPathSegments, commonBase, directoryPath, fileName, firstSegment, l, len, len1, m, ourPathSegments, pathSegments, ref3, textEditor;
      if (this.getPath()) {
        fileName = this.getFileName();
        allPathSegments = [];
        ref3 = atom.workspace.getTextEditors();
        for (l = 0, len = ref3.length; l < len; l++) {
          textEditor = ref3[l];
          if (textEditor !== this) {
            if (textEditor.getFileName() === fileName) {
              directoryPath = fs.tildify(textEditor.getDirectoryPath());
              allPathSegments.push(directoryPath.split(path.sep));
            }
          }
        }
        if (allPathSegments.length === 0) {
          return fileName;
        }
        ourPathSegments = fs.tildify(this.getDirectoryPath()).split(path.sep);
        allPathSegments.push(ourPathSegments);
        while (true) {
          firstSegment = ourPathSegments[0];
          commonBase = _.all(allPathSegments, function(pathSegments) {
            return pathSegments.length > 1 && pathSegments[0] === firstSegment;
          });
          if (commonBase) {
            for (m = 0, len1 = allPathSegments.length; m < len1; m++) {
              pathSegments = allPathSegments[m];
              pathSegments.shift();
            }
          } else {
            break;
          }
        }
        return fileName + " \u2014 " + (path.join.apply(path, pathSegments));
      } else {
        return 'untitled';
      }
    };

    TextEditor.prototype.getPath = function() {
      return this.buffer.getPath();
    };

    TextEditor.prototype.getFileName = function() {
      var fullPath;
      if (fullPath = this.getPath()) {
        return path.basename(fullPath);
      } else {
        return null;
      }
    };

    TextEditor.prototype.getDirectoryPath = function() {
      var fullPath;
      if (fullPath = this.getPath()) {
        return path.dirname(fullPath);
      } else {
        return null;
      }
    };

    TextEditor.prototype.getEncoding = function() {
      return this.buffer.getEncoding();
    };

    TextEditor.prototype.setEncoding = function(encoding) {
      return this.buffer.setEncoding(encoding);
    };

    TextEditor.prototype.isModified = function() {
      return this.buffer.isModified();
    };

    TextEditor.prototype.isEmpty = function() {
      return this.buffer.isEmpty();
    };


    /*
    Section: File Operations
     */

    TextEditor.prototype.save = function() {
      return this.buffer.save();
    };

    TextEditor.prototype.saveAs = function(filePath) {
      return this.buffer.saveAs(filePath);
    };

    TextEditor.prototype.shouldPromptToSave = function(arg) {
      var projectHasPaths, ref3, windowCloseRequested;
      ref3 = arg != null ? arg : {}, windowCloseRequested = ref3.windowCloseRequested, projectHasPaths = ref3.projectHasPaths;
      if (windowCloseRequested && projectHasPaths && atom.stateStore.isConnected()) {
        return false;
      } else {
        return this.isModified() && !this.buffer.hasMultipleEditors();
      }
    };

    TextEditor.prototype.getSaveDialogOptions = function() {
      return {};
    };


    /*
    Section: Reading Text
     */

    TextEditor.prototype.getText = function() {
      return this.buffer.getText();
    };

    TextEditor.prototype.getTextInBufferRange = function(range) {
      return this.buffer.getTextInRange(range);
    };

    TextEditor.prototype.getLineCount = function() {
      return this.buffer.getLineCount();
    };

    TextEditor.prototype.getScreenLineCount = function() {
      return this.displayLayer.getScreenLineCount();
    };

    TextEditor.prototype.getApproximateScreenLineCount = function() {
      return this.displayLayer.getApproximateScreenLineCount();
    };

    TextEditor.prototype.getLastBufferRow = function() {
      return this.buffer.getLastRow();
    };

    TextEditor.prototype.getLastScreenRow = function() {
      return this.getScreenLineCount() - 1;
    };

    TextEditor.prototype.lineTextForBufferRow = function(bufferRow) {
      return this.buffer.lineForRow(bufferRow);
    };

    TextEditor.prototype.lineTextForScreenRow = function(screenRow) {
      var ref3;
      return (ref3 = this.screenLineForScreenRow(screenRow)) != null ? ref3.lineText : void 0;
    };

    TextEditor.prototype.logScreenLines = function(start, end) {
      var l, line, ref3, ref4, row;
      if (start == null) {
        start = 0;
      }
      if (end == null) {
        end = this.getLastScreenRow();
      }
      for (row = l = ref3 = start, ref4 = end; ref3 <= ref4 ? l <= ref4 : l >= ref4; row = ref3 <= ref4 ? ++l : --l) {
        line = this.lineTextForScreenRow(row);
        console.log(row, this.bufferRowForScreenRow(row), line, line.length);
      }
    };

    TextEditor.prototype.tokensForScreenRow = function(screenRow) {
      var currentTokenScopes, l, len, lineText, lineTextIndex, ref3, tag, tags, tokens;
      tokens = [];
      lineTextIndex = 0;
      currentTokenScopes = [];
      ref3 = this.screenLineForScreenRow(screenRow), lineText = ref3.lineText, tags = ref3.tags;
      for (l = 0, len = tags.length; l < len; l++) {
        tag = tags[l];
        if (this.displayLayer.isOpenTag(tag)) {
          currentTokenScopes.push(this.displayLayer.classNameForTag(tag));
        } else if (this.displayLayer.isCloseTag(tag)) {
          currentTokenScopes.pop();
        } else {
          tokens.push({
            text: lineText.substr(lineTextIndex, tag),
            scopes: currentTokenScopes.slice()
          });
          lineTextIndex += tag;
        }
      }
      return tokens;
    };

    TextEditor.prototype.screenLineForScreenRow = function(screenRow) {
      return this.displayLayer.getScreenLine(screenRow);
    };

    TextEditor.prototype.bufferRowForScreenRow = function(screenRow) {
      return this.displayLayer.translateScreenPosition(Point(screenRow, 0)).row;
    };

    TextEditor.prototype.bufferRowsForScreenRows = function(startScreenRow, endScreenRow) {
      return this.displayLayer.bufferRowsForScreenRows(startScreenRow, endScreenRow + 1);
    };

    TextEditor.prototype.screenRowForBufferRow = function(row) {
      return this.displayLayer.translateBufferPosition(Point(row, 0)).row;
    };

    TextEditor.prototype.getRightmostScreenPosition = function() {
      return this.displayLayer.getRightmostScreenPosition();
    };

    TextEditor.prototype.getApproximateRightmostScreenPosition = function() {
      return this.displayLayer.getApproximateRightmostScreenPosition();
    };

    TextEditor.prototype.getMaxScreenLineLength = function() {
      return this.getRightmostScreenPosition().column;
    };

    TextEditor.prototype.getLongestScreenRow = function() {
      return this.getRightmostScreenPosition().row;
    };

    TextEditor.prototype.getApproximateLongestScreenRow = function() {
      return this.getApproximateRightmostScreenPosition().row;
    };

    TextEditor.prototype.lineLengthForScreenRow = function(screenRow) {
      return this.displayLayer.lineLengthForScreenRow(screenRow);
    };

    TextEditor.prototype.bufferRangeForBufferRow = function(row, arg) {
      var includeNewline;
      includeNewline = (arg != null ? arg : {}).includeNewline;
      return this.buffer.rangeForRow(row, includeNewline);
    };

    TextEditor.prototype.getTextInRange = function(range) {
      return this.buffer.getTextInRange(range);
    };

    TextEditor.prototype.isBufferRowBlank = function(bufferRow) {
      return this.buffer.isRowBlank(bufferRow);
    };

    TextEditor.prototype.nextNonBlankBufferRow = function(bufferRow) {
      return this.buffer.nextNonBlankRow(bufferRow);
    };

    TextEditor.prototype.getEofBufferPosition = function() {
      return this.buffer.getEndPosition();
    };

    TextEditor.prototype.getCurrentParagraphBufferRange = function() {
      return this.getLastCursor().getCurrentParagraphBufferRange();
    };


    /*
    Section: Mutating Text
     */

    TextEditor.prototype.setText = function(text) {
      return this.buffer.setText(text);
    };

    TextEditor.prototype.setTextInBufferRange = function(range, text, options) {
      return this.getBuffer().setTextInRange(range, text, options);
    };

    TextEditor.prototype.insertText = function(text, options) {
      var groupingInterval;
      if (options == null) {
        options = {};
      }
      if (!this.emitWillInsertTextEvent(text)) {
        return false;
      }
      groupingInterval = options.groupUndo ? this.undoGroupingInterval : 0;
      if (options.autoIndentNewline == null) {
        options.autoIndentNewline = this.shouldAutoIndent();
      }
      if (options.autoDecreaseIndent == null) {
        options.autoDecreaseIndent = this.shouldAutoIndent();
      }
      return this.mutateSelectedText((function(_this) {
        return function(selection) {
          var didInsertEvent, range;
          range = selection.insertText(text, options);
          didInsertEvent = {
            text: text,
            range: range
          };
          _this.emitter.emit('did-insert-text', didInsertEvent);
          return range;
        };
      })(this), groupingInterval);
    };

    TextEditor.prototype.insertNewline = function(options) {
      return this.insertText('\n', options);
    };

    TextEditor.prototype["delete"] = function() {
      return this.mutateSelectedText(function(selection) {
        return selection["delete"]();
      });
    };

    TextEditor.prototype.backspace = function() {
      return this.mutateSelectedText(function(selection) {
        return selection.backspace();
      });
    };

    TextEditor.prototype.mutateSelectedText = function(fn, groupingInterval) {
      if (groupingInterval == null) {
        groupingInterval = 0;
      }
      return this.mergeIntersectingSelections((function(_this) {
        return function() {
          return _this.transact(groupingInterval, function() {
            var index, l, len, ref3, results, selection;
            ref3 = _this.getSelectionsOrderedByBufferPosition();
            results = [];
            for (index = l = 0, len = ref3.length; l < len; index = ++l) {
              selection = ref3[index];
              results.push(fn(selection, index));
            }
            return results;
          });
        };
      })(this));
    };

    TextEditor.prototype.moveLineUp = function() {
      var selections;
      selections = this.getSelectedBufferRanges().sort(function(a, b) {
        return a.compare(b);
      });
      if (selections[0].start.row === 0) {
        return;
      }
      if (selections[selections.length - 1].start.row === this.getLastBufferRow() && this.buffer.getLastLine() === '') {
        return;
      }
      return this.transact((function(_this) {
        return function() {
          var endRow, insertDelta, l, len, len1, lines, linesRange, m, newSelectionRanges, precedingRow, rangeToRefold, rangesToRefold, ref3, selection, selectionsToMove, startRow;
          newSelectionRanges = [];
          while (selections.length > 0) {
            selection = selections.shift();
            selectionsToMove = [selection];
            while (selection.end.row === ((ref3 = selections[0]) != null ? ref3.start.row : void 0)) {
              selectionsToMove.push(selections[0]);
              selection.end.row = selections[0].end.row;
              selections.shift();
            }
            startRow = selection.start.row;
            endRow = selection.end.row;
            if (selection.end.row > selection.start.row && selection.end.column === 0) {
              endRow--;
            }
            startRow = _this.displayLayer.findBoundaryPrecedingBufferRow(startRow);
            endRow = _this.displayLayer.findBoundaryFollowingBufferRow(endRow + 1);
            linesRange = new Range(Point(startRow, 0), Point(endRow, 0));
            precedingRow = _this.displayLayer.findBoundaryPrecedingBufferRow(startRow - 1);
            insertDelta = linesRange.start.row - precedingRow;
            rangesToRefold = _this.displayLayer.destroyFoldsIntersectingBufferRange(linesRange).map(function(range) {
              return range.translate([-insertDelta, 0]);
            });
            lines = _this.buffer.getTextInRange(linesRange);
            if (lines[lines.length - 1] !== '\n') {
              lines += _this.buffer.lineEndingForRow(linesRange.end.row - 2);
            }
            _this.buffer["delete"](linesRange);
            _this.buffer.insert([precedingRow, 0], lines);
            for (l = 0, len = rangesToRefold.length; l < len; l++) {
              rangeToRefold = rangesToRefold[l];
              _this.displayLayer.foldBufferRange(rangeToRefold);
            }
            for (m = 0, len1 = selectionsToMove.length; m < len1; m++) {
              selection = selectionsToMove[m];
              newSelectionRanges.push(selection.translate([-insertDelta, 0]));
            }
          }
          _this.setSelectedBufferRanges(newSelectionRanges, {
            autoscroll: false,
            preserveFolds: true
          });
          if (_this.shouldAutoIndent()) {
            _this.autoIndentSelectedRows();
          }
          return _this.scrollToBufferPosition([newSelectionRanges[0].start.row, 0]);
        };
      })(this));
    };

    TextEditor.prototype.moveLineDown = function() {
      var selections;
      selections = this.getSelectedBufferRanges();
      selections.sort(function(a, b) {
        return a.compare(b);
      });
      selections = selections.reverse();
      return this.transact((function(_this) {
        return function() {
          var endRow, followingRow, insertDelta, l, len, len1, lines, linesRange, m, newSelectionRanges, rangeToRefold, rangesToRefold, ref3, selection, selectionsToMove, startRow;
          _this.consolidateSelections();
          newSelectionRanges = [];
          while (selections.length > 0) {
            selection = selections.shift();
            selectionsToMove = [selection];
            while (selection.start.row === ((ref3 = selections[0]) != null ? ref3.end.row : void 0)) {
              selectionsToMove.push(selections[0]);
              selection.start.row = selections[0].start.row;
              selections.shift();
            }
            startRow = selection.start.row;
            endRow = selection.end.row;
            if (selection.end.row > selection.start.row && selection.end.column === 0) {
              endRow--;
            }
            startRow = _this.displayLayer.findBoundaryPrecedingBufferRow(startRow);
            endRow = _this.displayLayer.findBoundaryFollowingBufferRow(endRow + 1);
            linesRange = new Range(Point(startRow, 0), Point(endRow, 0));
            followingRow = Math.min(_this.buffer.getLineCount(), _this.displayLayer.findBoundaryFollowingBufferRow(endRow + 1));
            insertDelta = followingRow - linesRange.end.row;
            rangesToRefold = _this.displayLayer.destroyFoldsIntersectingBufferRange(linesRange).map(function(range) {
              return range.translate([insertDelta, 0]);
            });
            lines = _this.buffer.getTextInRange(linesRange);
            if (followingRow - 1 === _this.buffer.getLastRow()) {
              lines = "\n" + lines;
            }
            _this.buffer.insert([followingRow, 0], lines);
            _this.buffer["delete"](linesRange);
            for (l = 0, len = rangesToRefold.length; l < len; l++) {
              rangeToRefold = rangesToRefold[l];
              _this.displayLayer.foldBufferRange(rangeToRefold);
            }
            for (m = 0, len1 = selectionsToMove.length; m < len1; m++) {
              selection = selectionsToMove[m];
              newSelectionRanges.push(selection.translate([insertDelta, 0]));
            }
          }
          _this.setSelectedBufferRanges(newSelectionRanges, {
            autoscroll: false,
            preserveFolds: true
          });
          if (_this.shouldAutoIndent()) {
            _this.autoIndentSelectedRows();
          }
          return _this.scrollToBufferPosition([newSelectionRanges[0].start.row - 1, 0]);
        };
      })(this));
    };

    TextEditor.prototype.moveSelectionLeft = function() {
      var noSelectionAtStartOfLine, selections, translatedRanges, translationDelta;
      selections = this.getSelectedBufferRanges();
      noSelectionAtStartOfLine = selections.every(function(selection) {
        return selection.start.column !== 0;
      });
      translationDelta = [0, -1];
      translatedRanges = [];
      if (noSelectionAtStartOfLine) {
        return this.transact((function(_this) {
          return function() {
            var charTextToLeftOfSelection, charToLeftOfSelection, l, len, selection;
            for (l = 0, len = selections.length; l < len; l++) {
              selection = selections[l];
              charToLeftOfSelection = new Range(selection.start.translate(translationDelta), selection.start);
              charTextToLeftOfSelection = _this.buffer.getTextInRange(charToLeftOfSelection);
              _this.buffer.insert(selection.end, charTextToLeftOfSelection);
              _this.buffer["delete"](charToLeftOfSelection);
              translatedRanges.push(selection.translate(translationDelta));
            }
            return _this.setSelectedBufferRanges(translatedRanges);
          };
        })(this));
      }
    };

    TextEditor.prototype.moveSelectionRight = function() {
      var noSelectionAtEndOfLine, selections, translatedRanges, translationDelta;
      selections = this.getSelectedBufferRanges();
      noSelectionAtEndOfLine = selections.every((function(_this) {
        return function(selection) {
          return selection.end.column !== _this.buffer.lineLengthForRow(selection.end.row);
        };
      })(this));
      translationDelta = [0, 1];
      translatedRanges = [];
      if (noSelectionAtEndOfLine) {
        return this.transact((function(_this) {
          return function() {
            var charTextToRightOfSelection, charToRightOfSelection, l, len, selection;
            for (l = 0, len = selections.length; l < len; l++) {
              selection = selections[l];
              charToRightOfSelection = new Range(selection.end, selection.end.translate(translationDelta));
              charTextToRightOfSelection = _this.buffer.getTextInRange(charToRightOfSelection);
              _this.buffer["delete"](charToRightOfSelection);
              _this.buffer.insert(selection.start, charTextToRightOfSelection);
              translatedRanges.push(selection.translate(translationDelta));
            }
            return _this.setSelectedBufferRanges(translatedRanges);
          };
        })(this));
      }
    };

    TextEditor.prototype.duplicateLines = function() {
      return this.transact((function(_this) {
        return function() {
          var endRow, fold, foldRange, i, insertedRowCount, intersectingFolds, j, k, l, len, m, previousSelectionEndRow, previousSelectionRanges, previousSelectionStartRow, ref3, ref4, ref5, ref6, results, selections, start, startRow, textToDuplicate;
          selections = _this.getSelectionsOrderedByBufferPosition();
          previousSelectionRanges = [];
          i = selections.length - 1;
          results = [];
          while (i >= 0) {
            j = i;
            previousSelectionRanges[i] = selections[i].getBufferRange();
            if (selections[i].isEmpty()) {
              start = selections[i].getScreenRange().start;
              selections[i].setScreenRange([[start.row, 0], [start.row + 1, 0]], {
                preserveFolds: true
              });
            }
            ref3 = selections[i].getBufferRowRange(), startRow = ref3[0], endRow = ref3[1];
            endRow++;
            while (i > 0) {
              ref4 = selections[i - 1].getBufferRowRange(), previousSelectionStartRow = ref4[0], previousSelectionEndRow = ref4[1];
              if (previousSelectionEndRow === startRow) {
                startRow = previousSelectionStartRow;
                previousSelectionRanges[i - 1] = selections[i - 1].getBufferRange();
                i--;
              } else {
                break;
              }
            }
            intersectingFolds = _this.displayLayer.foldsIntersectingBufferRange([[startRow, 0], [endRow, 0]]);
            textToDuplicate = _this.getTextInBufferRange([[startRow, 0], [endRow, 0]]);
            if (endRow > _this.getLastBufferRow()) {
              textToDuplicate = '\n' + textToDuplicate;
            }
            _this.buffer.insert([endRow, 0], textToDuplicate);
            insertedRowCount = endRow - startRow;
            for (k = l = ref5 = i, ref6 = j; l <= ref6; k = l += 1) {
              selections[k].setBufferRange(previousSelectionRanges[k].translate([insertedRowCount, 0]));
            }
            for (m = 0, len = intersectingFolds.length; m < len; m++) {
              fold = intersectingFolds[m];
              foldRange = _this.displayLayer.bufferRangeForFold(fold);
              _this.displayLayer.foldBufferRange(foldRange.translate([insertedRowCount, 0]));
            }
            results.push(i--);
          }
          return results;
        };
      })(this));
    };

    TextEditor.prototype.replaceSelectedText = function(options, fn) {
      var selectWordIfEmpty;
      if (options == null) {
        options = {};
      }
      selectWordIfEmpty = options.selectWordIfEmpty;
      return this.mutateSelectedText(function(selection) {
        var range, text;
        selection.getBufferRange();
        if (selectWordIfEmpty && selection.isEmpty()) {
          selection.selectWord();
        }
        text = selection.getText();
        selection.deleteSelectedText();
        range = selection.insertText(fn(text));
        return selection.setBufferRange(range);
      });
    };

    TextEditor.prototype.splitSelectionsIntoLines = function() {
      return this.mergeIntersectingSelections((function(_this) {
        return function() {
          var end, l, len, range, ref3, row, selection, start;
          ref3 = _this.getSelections();
          for (l = 0, len = ref3.length; l < len; l++) {
            selection = ref3[l];
            range = selection.getBufferRange();
            if (range.isSingleLine()) {
              continue;
            }
            start = range.start, end = range.end;
            _this.addSelectionForBufferRange([start, [start.row, 2e308]]);
            row = start.row;
            while (++row < end.row) {
              _this.addSelectionForBufferRange([[row, 0], [row, 2e308]]);
            }
            if (end.column !== 0) {
              _this.addSelectionForBufferRange([[end.row, 0], [end.row, end.column]]);
            }
            selection.destroy();
          }
        };
      })(this));
    };

    TextEditor.prototype.transpose = function() {
      return this.mutateSelectedText(function(selection) {
        var text;
        if (selection.isEmpty()) {
          selection.selectRight();
          text = selection.getText();
          selection["delete"]();
          selection.cursor.moveLeft();
          return selection.insertText(text);
        } else {
          return selection.insertText(selection.getText().split('').reverse().join(''));
        }
      });
    };

    TextEditor.prototype.upperCase = function() {
      return this.replaceSelectedText({
        selectWordIfEmpty: true
      }, function(text) {
        return text.toUpperCase();
      });
    };

    TextEditor.prototype.lowerCase = function() {
      return this.replaceSelectedText({
        selectWordIfEmpty: true
      }, function(text) {
        return text.toLowerCase();
      });
    };

    TextEditor.prototype.toggleLineCommentsInSelection = function() {
      return this.mutateSelectedText(function(selection) {
        return selection.toggleLineComments();
      });
    };

    TextEditor.prototype.joinLines = function() {
      return this.mutateSelectedText(function(selection) {
        return selection.joinLines();
      });
    };

    TextEditor.prototype.insertNewlineBelow = function() {
      return this.transact((function(_this) {
        return function() {
          _this.moveToEndOfLine();
          return _this.insertNewline();
        };
      })(this));
    };

    TextEditor.prototype.insertNewlineAbove = function() {
      return this.transact((function(_this) {
        return function() {
          var bufferRow, indentLevel, onFirstLine;
          bufferRow = _this.getCursorBufferPosition().row;
          indentLevel = _this.indentationForBufferRow(bufferRow);
          onFirstLine = bufferRow === 0;
          _this.moveToBeginningOfLine();
          _this.moveLeft();
          _this.insertNewline();
          if (_this.shouldAutoIndent() && _this.indentationForBufferRow(bufferRow) < indentLevel) {
            _this.setIndentationForBufferRow(bufferRow, indentLevel);
          }
          if (onFirstLine) {
            _this.moveUp();
            return _this.moveToEndOfLine();
          }
        };
      })(this));
    };

    TextEditor.prototype.deleteToBeginningOfWord = function() {
      return this.mutateSelectedText(function(selection) {
        return selection.deleteToBeginningOfWord();
      });
    };

    TextEditor.prototype.deleteToPreviousWordBoundary = function() {
      return this.mutateSelectedText(function(selection) {
        return selection.deleteToPreviousWordBoundary();
      });
    };

    TextEditor.prototype.deleteToNextWordBoundary = function() {
      return this.mutateSelectedText(function(selection) {
        return selection.deleteToNextWordBoundary();
      });
    };

    TextEditor.prototype.deleteToBeginningOfSubword = function() {
      return this.mutateSelectedText(function(selection) {
        return selection.deleteToBeginningOfSubword();
      });
    };

    TextEditor.prototype.deleteToEndOfSubword = function() {
      return this.mutateSelectedText(function(selection) {
        return selection.deleteToEndOfSubword();
      });
    };

    TextEditor.prototype.deleteToBeginningOfLine = function() {
      return this.mutateSelectedText(function(selection) {
        return selection.deleteToBeginningOfLine();
      });
    };

    TextEditor.prototype.deleteToEndOfLine = function() {
      return this.mutateSelectedText(function(selection) {
        return selection.deleteToEndOfLine();
      });
    };

    TextEditor.prototype.deleteToEndOfWord = function() {
      return this.mutateSelectedText(function(selection) {
        return selection.deleteToEndOfWord();
      });
    };

    TextEditor.prototype.deleteLine = function() {
      this.mergeSelectionsOnSameRows();
      return this.mutateSelectedText(function(selection) {
        return selection.deleteLine();
      });
    };


    /*
    Section: History
     */

    TextEditor.prototype.undo = function() {
      this.avoidMergingSelections((function(_this) {
        return function() {
          return _this.buffer.undo();
        };
      })(this));
      return this.getLastSelection().autoscroll();
    };

    TextEditor.prototype.redo = function() {
      this.avoidMergingSelections((function(_this) {
        return function() {
          return _this.buffer.redo();
        };
      })(this));
      return this.getLastSelection().autoscroll();
    };

    TextEditor.prototype.transact = function(groupingInterval, fn) {
      return this.buffer.transact(groupingInterval, fn);
    };

    TextEditor.prototype.beginTransaction = function(groupingInterval) {
      Grim.deprecate('Transactions should be performed via TextEditor::transact only');
      return this.buffer.beginTransaction(groupingInterval);
    };

    TextEditor.prototype.commitTransaction = function() {
      Grim.deprecate('Transactions should be performed via TextEditor::transact only');
      return this.buffer.commitTransaction();
    };

    TextEditor.prototype.abortTransaction = function() {
      return this.buffer.abortTransaction();
    };

    TextEditor.prototype.createCheckpoint = function() {
      return this.buffer.createCheckpoint();
    };

    TextEditor.prototype.revertToCheckpoint = function(checkpoint) {
      return this.buffer.revertToCheckpoint(checkpoint);
    };

    TextEditor.prototype.groupChangesSinceCheckpoint = function(checkpoint) {
      return this.buffer.groupChangesSinceCheckpoint(checkpoint);
    };


    /*
    Section: TextEditor Coordinates
     */

    TextEditor.prototype.screenPositionForBufferPosition = function(bufferPosition, options) {
      if ((options != null ? options.clip : void 0) != null) {
        Grim.deprecate("The `clip` parameter has been deprecated and will be removed soon. Please, use `clipDirection` instead.");
        if (options.clipDirection == null) {
          options.clipDirection = options.clip;
        }
      }
      if ((options != null ? options.wrapAtSoftNewlines : void 0) != null) {
        Grim.deprecate("The `wrapAtSoftNewlines` parameter has been deprecated and will be removed soon. Please, use `clipDirection: 'forward'` instead.");
        if (options.clipDirection == null) {
          options.clipDirection = options.wrapAtSoftNewlines ? 'forward' : 'backward';
        }
      }
      if ((options != null ? options.wrapBeyondNewlines : void 0) != null) {
        Grim.deprecate("The `wrapBeyondNewlines` parameter has been deprecated and will be removed soon. Please, use `clipDirection: 'forward'` instead.");
        if (options.clipDirection == null) {
          options.clipDirection = options.wrapBeyondNewlines ? 'forward' : 'backward';
        }
      }
      return this.displayLayer.translateBufferPosition(bufferPosition, options);
    };

    TextEditor.prototype.bufferPositionForScreenPosition = function(screenPosition, options) {
      if ((options != null ? options.clip : void 0) != null) {
        Grim.deprecate("The `clip` parameter has been deprecated and will be removed soon. Please, use `clipDirection` instead.");
        if (options.clipDirection == null) {
          options.clipDirection = options.clip;
        }
      }
      if ((options != null ? options.wrapAtSoftNewlines : void 0) != null) {
        Grim.deprecate("The `wrapAtSoftNewlines` parameter has been deprecated and will be removed soon. Please, use `clipDirection: 'forward'` instead.");
        if (options.clipDirection == null) {
          options.clipDirection = options.wrapAtSoftNewlines ? 'forward' : 'backward';
        }
      }
      if ((options != null ? options.wrapBeyondNewlines : void 0) != null) {
        Grim.deprecate("The `wrapBeyondNewlines` parameter has been deprecated and will be removed soon. Please, use `clipDirection: 'forward'` instead.");
        if (options.clipDirection == null) {
          options.clipDirection = options.wrapBeyondNewlines ? 'forward' : 'backward';
        }
      }
      return this.displayLayer.translateScreenPosition(screenPosition, options);
    };

    TextEditor.prototype.screenRangeForBufferRange = function(bufferRange, options) {
      var end, start;
      bufferRange = Range.fromObject(bufferRange);
      start = this.screenPositionForBufferPosition(bufferRange.start, options);
      end = this.screenPositionForBufferPosition(bufferRange.end, options);
      return new Range(start, end);
    };

    TextEditor.prototype.bufferRangeForScreenRange = function(screenRange) {
      var end, start;
      screenRange = Range.fromObject(screenRange);
      start = this.bufferPositionForScreenPosition(screenRange.start);
      end = this.bufferPositionForScreenPosition(screenRange.end);
      return new Range(start, end);
    };

    TextEditor.prototype.clipBufferPosition = function(bufferPosition) {
      return this.buffer.clipPosition(bufferPosition);
    };

    TextEditor.prototype.clipBufferRange = function(range) {
      return this.buffer.clipRange(range);
    };

    TextEditor.prototype.clipScreenPosition = function(screenPosition, options) {
      if ((options != null ? options.clip : void 0) != null) {
        Grim.deprecate("The `clip` parameter has been deprecated and will be removed soon. Please, use `clipDirection` instead.");
        if (options.clipDirection == null) {
          options.clipDirection = options.clip;
        }
      }
      if ((options != null ? options.wrapAtSoftNewlines : void 0) != null) {
        Grim.deprecate("The `wrapAtSoftNewlines` parameter has been deprecated and will be removed soon. Please, use `clipDirection: 'forward'` instead.");
        if (options.clipDirection == null) {
          options.clipDirection = options.wrapAtSoftNewlines ? 'forward' : 'backward';
        }
      }
      if ((options != null ? options.wrapBeyondNewlines : void 0) != null) {
        Grim.deprecate("The `wrapBeyondNewlines` parameter has been deprecated and will be removed soon. Please, use `clipDirection: 'forward'` instead.");
        if (options.clipDirection == null) {
          options.clipDirection = options.wrapBeyondNewlines ? 'forward' : 'backward';
        }
      }
      return this.displayLayer.clipScreenPosition(screenPosition, options);
    };

    TextEditor.prototype.clipScreenRange = function(screenRange, options) {
      var end, start;
      screenRange = Range.fromObject(screenRange);
      start = this.displayLayer.clipScreenPosition(screenRange.start, options);
      end = this.displayLayer.clipScreenPosition(screenRange.end, options);
      return Range(start, end);
    };


    /*
    Section: Decorations
     */

    TextEditor.prototype.decorateMarker = function(marker, decorationParams) {
      return this.decorationManager.decorateMarker(marker, decorationParams);
    };

    TextEditor.prototype.decorateMarkerLayer = function(markerLayer, decorationParams) {
      return this.decorationManager.decorateMarkerLayer(markerLayer, decorationParams);
    };

    TextEditor.prototype.decorationsForScreenRowRange = function(startScreenRow, endScreenRow) {
      return this.decorationManager.decorationsForScreenRowRange(startScreenRow, endScreenRow);
    };

    TextEditor.prototype.decorationsStateForScreenRowRange = function(startScreenRow, endScreenRow) {
      return this.decorationManager.decorationsStateForScreenRowRange(startScreenRow, endScreenRow);
    };

    TextEditor.prototype.getDecorations = function(propertyFilter) {
      return this.decorationManager.getDecorations(propertyFilter);
    };

    TextEditor.prototype.getLineDecorations = function(propertyFilter) {
      return this.decorationManager.getLineDecorations(propertyFilter);
    };

    TextEditor.prototype.getLineNumberDecorations = function(propertyFilter) {
      return this.decorationManager.getLineNumberDecorations(propertyFilter);
    };

    TextEditor.prototype.getHighlightDecorations = function(propertyFilter) {
      return this.decorationManager.getHighlightDecorations(propertyFilter);
    };

    TextEditor.prototype.getOverlayDecorations = function(propertyFilter) {
      return this.decorationManager.getOverlayDecorations(propertyFilter);
    };


    /*
    Section: Markers
     */

    TextEditor.prototype.markBufferRange = function(bufferRange, options) {
      return this.defaultMarkerLayer.markBufferRange(bufferRange, options);
    };

    TextEditor.prototype.markScreenRange = function(screenRange, options) {
      return this.defaultMarkerLayer.markScreenRange(screenRange, options);
    };

    TextEditor.prototype.markBufferPosition = function(bufferPosition, options) {
      return this.defaultMarkerLayer.markBufferPosition(bufferPosition, options);
    };

    TextEditor.prototype.markScreenPosition = function(screenPosition, options) {
      return this.defaultMarkerLayer.markScreenPosition(screenPosition, options);
    };

    TextEditor.prototype.findMarkers = function(params) {
      return this.defaultMarkerLayer.findMarkers(params);
    };

    TextEditor.prototype.getMarker = function(id) {
      return this.defaultMarkerLayer.getMarker(id);
    };

    TextEditor.prototype.getMarkers = function() {
      return this.defaultMarkerLayer.getMarkers();
    };

    TextEditor.prototype.getMarkerCount = function() {
      return this.defaultMarkerLayer.getMarkerCount();
    };

    TextEditor.prototype.destroyMarker = function(id) {
      var ref3;
      return (ref3 = this.getMarker(id)) != null ? ref3.destroy() : void 0;
    };

    TextEditor.prototype.addMarkerLayer = function(options) {
      return this.displayLayer.addMarkerLayer(options);
    };

    TextEditor.prototype.getMarkerLayer = function(id) {
      return this.displayLayer.getMarkerLayer(id);
    };

    TextEditor.prototype.getDefaultMarkerLayer = function() {
      return this.defaultMarkerLayer;
    };


    /*
    Section: Cursors
     */

    TextEditor.prototype.getCursorBufferPosition = function() {
      return this.getLastCursor().getBufferPosition();
    };

    TextEditor.prototype.getCursorBufferPositions = function() {
      var cursor, l, len, ref3, results;
      ref3 = this.getCursors();
      results = [];
      for (l = 0, len = ref3.length; l < len; l++) {
        cursor = ref3[l];
        results.push(cursor.getBufferPosition());
      }
      return results;
    };

    TextEditor.prototype.setCursorBufferPosition = function(position, options) {
      return this.moveCursors(function(cursor) {
        return cursor.setBufferPosition(position, options);
      });
    };

    TextEditor.prototype.getCursorAtScreenPosition = function(position) {
      var selection;
      if (selection = this.getSelectionAtScreenPosition(position)) {
        if (selection.getHeadScreenPosition().isEqual(position)) {
          return selection.cursor;
        }
      }
    };

    TextEditor.prototype.getCursorScreenPosition = function() {
      return this.getLastCursor().getScreenPosition();
    };

    TextEditor.prototype.getCursorScreenPositions = function() {
      var cursor, l, len, ref3, results;
      ref3 = this.getCursors();
      results = [];
      for (l = 0, len = ref3.length; l < len; l++) {
        cursor = ref3[l];
        results.push(cursor.getScreenPosition());
      }
      return results;
    };

    TextEditor.prototype.setCursorScreenPosition = function(position, options) {
      if ((options != null ? options.clip : void 0) != null) {
        Grim.deprecate("The `clip` parameter has been deprecated and will be removed soon. Please, use `clipDirection` instead.");
        if (options.clipDirection == null) {
          options.clipDirection = options.clip;
        }
      }
      if ((options != null ? options.wrapAtSoftNewlines : void 0) != null) {
        Grim.deprecate("The `wrapAtSoftNewlines` parameter has been deprecated and will be removed soon. Please, use `clipDirection: 'forward'` instead.");
        if (options.clipDirection == null) {
          options.clipDirection = options.wrapAtSoftNewlines ? 'forward' : 'backward';
        }
      }
      if ((options != null ? options.wrapBeyondNewlines : void 0) != null) {
        Grim.deprecate("The `wrapBeyondNewlines` parameter has been deprecated and will be removed soon. Please, use `clipDirection: 'forward'` instead.");
        if (options.clipDirection == null) {
          options.clipDirection = options.wrapBeyondNewlines ? 'forward' : 'backward';
        }
      }
      return this.moveCursors(function(cursor) {
        return cursor.setScreenPosition(position, options);
      });
    };

    TextEditor.prototype.addCursorAtBufferPosition = function(bufferPosition, options) {
      this.selectionsMarkerLayer.markBufferPosition(bufferPosition, Object.assign({
        invalidate: 'never'
      }, options));
      if ((options != null ? options.autoscroll : void 0) !== false) {
        this.getLastSelection().cursor.autoscroll();
      }
      return this.getLastSelection().cursor;
    };

    TextEditor.prototype.addCursorAtScreenPosition = function(screenPosition, options) {
      this.selectionsMarkerLayer.markScreenPosition(screenPosition, {
        invalidate: 'never'
      });
      if ((options != null ? options.autoscroll : void 0) !== false) {
        this.getLastSelection().cursor.autoscroll();
      }
      return this.getLastSelection().cursor;
    };

    TextEditor.prototype.hasMultipleCursors = function() {
      return this.getCursors().length > 1;
    };

    TextEditor.prototype.moveUp = function(lineCount) {
      return this.moveCursors(function(cursor) {
        return cursor.moveUp(lineCount, {
          moveToEndOfSelection: true
        });
      });
    };

    TextEditor.prototype.moveDown = function(lineCount) {
      return this.moveCursors(function(cursor) {
        return cursor.moveDown(lineCount, {
          moveToEndOfSelection: true
        });
      });
    };

    TextEditor.prototype.moveLeft = function(columnCount) {
      return this.moveCursors(function(cursor) {
        return cursor.moveLeft(columnCount, {
          moveToEndOfSelection: true
        });
      });
    };

    TextEditor.prototype.moveRight = function(columnCount) {
      return this.moveCursors(function(cursor) {
        return cursor.moveRight(columnCount, {
          moveToEndOfSelection: true
        });
      });
    };

    TextEditor.prototype.moveToBeginningOfLine = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToBeginningOfLine();
      });
    };

    TextEditor.prototype.moveToBeginningOfScreenLine = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToBeginningOfScreenLine();
      });
    };

    TextEditor.prototype.moveToFirstCharacterOfLine = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToFirstCharacterOfLine();
      });
    };

    TextEditor.prototype.moveToEndOfLine = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToEndOfLine();
      });
    };

    TextEditor.prototype.moveToEndOfScreenLine = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToEndOfScreenLine();
      });
    };

    TextEditor.prototype.moveToBeginningOfWord = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToBeginningOfWord();
      });
    };

    TextEditor.prototype.moveToEndOfWord = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToEndOfWord();
      });
    };

    TextEditor.prototype.moveToTop = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToTop();
      });
    };

    TextEditor.prototype.moveToBottom = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToBottom();
      });
    };

    TextEditor.prototype.moveToBeginningOfNextWord = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToBeginningOfNextWord();
      });
    };

    TextEditor.prototype.moveToPreviousWordBoundary = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToPreviousWordBoundary();
      });
    };

    TextEditor.prototype.moveToNextWordBoundary = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToNextWordBoundary();
      });
    };

    TextEditor.prototype.moveToPreviousSubwordBoundary = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToPreviousSubwordBoundary();
      });
    };

    TextEditor.prototype.moveToNextSubwordBoundary = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToNextSubwordBoundary();
      });
    };

    TextEditor.prototype.moveToBeginningOfNextParagraph = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToBeginningOfNextParagraph();
      });
    };

    TextEditor.prototype.moveToBeginningOfPreviousParagraph = function() {
      return this.moveCursors(function(cursor) {
        return cursor.moveToBeginningOfPreviousParagraph();
      });
    };

    TextEditor.prototype.getLastCursor = function() {
      this.createLastSelectionIfNeeded();
      return _.last(this.cursors);
    };

    TextEditor.prototype.getWordUnderCursor = function(options) {
      return this.getTextInBufferRange(this.getLastCursor().getCurrentWordBufferRange(options));
    };

    TextEditor.prototype.getCursors = function() {
      this.createLastSelectionIfNeeded();
      return this.cursors.slice();
    };

    TextEditor.prototype.getCursorsOrderedByBufferPosition = function() {
      return this.getCursors().sort(function(a, b) {
        return a.compare(b);
      });
    };

    TextEditor.prototype.cursorsForScreenRowRange = function(startScreenRow, endScreenRow) {
      var cursor, cursors, l, len, marker, ref3;
      cursors = [];
      ref3 = this.selectionsMarkerLayer.findMarkers({
        intersectsScreenRowRange: [startScreenRow, endScreenRow]
      });
      for (l = 0, len = ref3.length; l < len; l++) {
        marker = ref3[l];
        if (cursor = this.cursorsByMarkerId.get(marker.id)) {
          cursors.push(cursor);
        }
      }
      return cursors;
    };

    TextEditor.prototype.addCursor = function(marker) {
      var cursor;
      cursor = new Cursor({
        editor: this,
        marker: marker,
        showCursorOnSelection: this.showCursorOnSelection
      });
      this.cursors.push(cursor);
      this.cursorsByMarkerId.set(marker.id, cursor);
      return cursor;
    };

    TextEditor.prototype.moveCursors = function(fn) {
      return this.transact((function(_this) {
        return function() {
          var cursor, l, len, ref3;
          ref3 = _this.getCursors();
          for (l = 0, len = ref3.length; l < len; l++) {
            cursor = ref3[l];
            fn(cursor);
          }
          return _this.mergeCursors();
        };
      })(this));
    };

    TextEditor.prototype.cursorMoved = function(event) {
      return this.emitter.emit('did-change-cursor-position', event);
    };

    TextEditor.prototype.mergeCursors = function() {
      var cursor, l, len, position, positions, ref3;
      positions = {};
      ref3 = this.getCursors();
      for (l = 0, len = ref3.length; l < len; l++) {
        cursor = ref3[l];
        position = cursor.getBufferPosition().toString();
        if (positions.hasOwnProperty(position)) {
          cursor.destroy();
        } else {
          positions[position] = true;
        }
      }
    };


    /*
    Section: Selections
     */

    TextEditor.prototype.getSelectedText = function() {
      return this.getLastSelection().getText();
    };

    TextEditor.prototype.getSelectedBufferRange = function() {
      return this.getLastSelection().getBufferRange();
    };

    TextEditor.prototype.getSelectedBufferRanges = function() {
      var l, len, ref3, results, selection;
      ref3 = this.getSelections();
      results = [];
      for (l = 0, len = ref3.length; l < len; l++) {
        selection = ref3[l];
        results.push(selection.getBufferRange());
      }
      return results;
    };

    TextEditor.prototype.setSelectedBufferRange = function(bufferRange, options) {
      return this.setSelectedBufferRanges([bufferRange], options);
    };

    TextEditor.prototype.setSelectedBufferRanges = function(bufferRanges, options) {
      var l, len, ref3, selection, selections;
      if (options == null) {
        options = {};
      }
      if (!bufferRanges.length) {
        throw new Error("Passed an empty array to setSelectedBufferRanges");
      }
      selections = this.getSelections();
      ref3 = selections.slice(bufferRanges.length);
      for (l = 0, len = ref3.length; l < len; l++) {
        selection = ref3[l];
        selection.destroy();
      }
      return this.mergeIntersectingSelections(options, (function(_this) {
        return function() {
          var bufferRange, i, len1, m;
          for (i = m = 0, len1 = bufferRanges.length; m < len1; i = ++m) {
            bufferRange = bufferRanges[i];
            bufferRange = Range.fromObject(bufferRange);
            if (selections[i]) {
              selections[i].setBufferRange(bufferRange, options);
            } else {
              _this.addSelectionForBufferRange(bufferRange, options);
            }
          }
        };
      })(this));
    };

    TextEditor.prototype.getSelectedScreenRange = function() {
      return this.getLastSelection().getScreenRange();
    };

    TextEditor.prototype.getSelectedScreenRanges = function() {
      var l, len, ref3, results, selection;
      ref3 = this.getSelections();
      results = [];
      for (l = 0, len = ref3.length; l < len; l++) {
        selection = ref3[l];
        results.push(selection.getScreenRange());
      }
      return results;
    };

    TextEditor.prototype.setSelectedScreenRange = function(screenRange, options) {
      return this.setSelectedBufferRange(this.bufferRangeForScreenRange(screenRange, options), options);
    };

    TextEditor.prototype.setSelectedScreenRanges = function(screenRanges, options) {
      var l, len, ref3, selection, selections;
      if (options == null) {
        options = {};
      }
      if (!screenRanges.length) {
        throw new Error("Passed an empty array to setSelectedScreenRanges");
      }
      selections = this.getSelections();
      ref3 = selections.slice(screenRanges.length);
      for (l = 0, len = ref3.length; l < len; l++) {
        selection = ref3[l];
        selection.destroy();
      }
      return this.mergeIntersectingSelections(options, (function(_this) {
        return function() {
          var i, len1, m, screenRange;
          for (i = m = 0, len1 = screenRanges.length; m < len1; i = ++m) {
            screenRange = screenRanges[i];
            screenRange = Range.fromObject(screenRange);
            if (selections[i]) {
              selections[i].setScreenRange(screenRange, options);
            } else {
              _this.addSelectionForScreenRange(screenRange, options);
            }
          }
        };
      })(this));
    };

    TextEditor.prototype.addSelectionForBufferRange = function(bufferRange, options) {
      var ref3;
      if (options == null) {
        options = {};
      }
      if (!options.preserveFolds) {
        this.destroyFoldsIntersectingBufferRange(bufferRange);
      }
      this.selectionsMarkerLayer.markBufferRange(bufferRange, {
        invalidate: 'never',
        reversed: (ref3 = options.reversed) != null ? ref3 : false
      });
      if (options.autoscroll !== false) {
        this.getLastSelection().autoscroll();
      }
      return this.getLastSelection();
    };

    TextEditor.prototype.addSelectionForScreenRange = function(screenRange, options) {
      if (options == null) {
        options = {};
      }
      return this.addSelectionForBufferRange(this.bufferRangeForScreenRange(screenRange), options);
    };

    TextEditor.prototype.selectToBufferPosition = function(position) {
      var lastSelection;
      lastSelection = this.getLastSelection();
      lastSelection.selectToBufferPosition(position);
      return this.mergeIntersectingSelections({
        reversed: lastSelection.isReversed()
      });
    };

    TextEditor.prototype.selectToScreenPosition = function(position, options) {
      var lastSelection;
      lastSelection = this.getLastSelection();
      lastSelection.selectToScreenPosition(position, options);
      if (!(options != null ? options.suppressSelectionMerge : void 0)) {
        return this.mergeIntersectingSelections({
          reversed: lastSelection.isReversed()
        });
      }
    };

    TextEditor.prototype.selectUp = function(rowCount) {
      return this.expandSelectionsBackward(function(selection) {
        return selection.selectUp(rowCount);
      });
    };

    TextEditor.prototype.selectDown = function(rowCount) {
      return this.expandSelectionsForward(function(selection) {
        return selection.selectDown(rowCount);
      });
    };

    TextEditor.prototype.selectLeft = function(columnCount) {
      return this.expandSelectionsBackward(function(selection) {
        return selection.selectLeft(columnCount);
      });
    };

    TextEditor.prototype.selectRight = function(columnCount) {
      return this.expandSelectionsForward(function(selection) {
        return selection.selectRight(columnCount);
      });
    };

    TextEditor.prototype.selectToTop = function() {
      return this.expandSelectionsBackward(function(selection) {
        return selection.selectToTop();
      });
    };

    TextEditor.prototype.selectToBottom = function() {
      return this.expandSelectionsForward(function(selection) {
        return selection.selectToBottom();
      });
    };

    TextEditor.prototype.selectAll = function() {
      return this.expandSelectionsForward(function(selection) {
        return selection.selectAll();
      });
    };

    TextEditor.prototype.selectToBeginningOfLine = function() {
      return this.expandSelectionsBackward(function(selection) {
        return selection.selectToBeginningOfLine();
      });
    };

    TextEditor.prototype.selectToFirstCharacterOfLine = function() {
      return this.expandSelectionsBackward(function(selection) {
        return selection.selectToFirstCharacterOfLine();
      });
    };

    TextEditor.prototype.selectToEndOfLine = function() {
      return this.expandSelectionsForward(function(selection) {
        return selection.selectToEndOfLine();
      });
    };

    TextEditor.prototype.selectToBeginningOfWord = function() {
      return this.expandSelectionsBackward(function(selection) {
        return selection.selectToBeginningOfWord();
      });
    };

    TextEditor.prototype.selectToEndOfWord = function() {
      return this.expandSelectionsForward(function(selection) {
        return selection.selectToEndOfWord();
      });
    };

    TextEditor.prototype.selectToPreviousSubwordBoundary = function() {
      return this.expandSelectionsBackward(function(selection) {
        return selection.selectToPreviousSubwordBoundary();
      });
    };

    TextEditor.prototype.selectToNextSubwordBoundary = function() {
      return this.expandSelectionsForward(function(selection) {
        return selection.selectToNextSubwordBoundary();
      });
    };

    TextEditor.prototype.selectLinesContainingCursors = function() {
      return this.expandSelectionsForward(function(selection) {
        return selection.selectLine();
      });
    };

    TextEditor.prototype.selectWordsContainingCursors = function() {
      return this.expandSelectionsForward(function(selection) {
        return selection.selectWord();
      });
    };

    TextEditor.prototype.selectToPreviousWordBoundary = function() {
      return this.expandSelectionsBackward(function(selection) {
        return selection.selectToPreviousWordBoundary();
      });
    };

    TextEditor.prototype.selectToNextWordBoundary = function() {
      return this.expandSelectionsForward(function(selection) {
        return selection.selectToNextWordBoundary();
      });
    };

    TextEditor.prototype.selectToBeginningOfNextWord = function() {
      return this.expandSelectionsForward(function(selection) {
        return selection.selectToBeginningOfNextWord();
      });
    };

    TextEditor.prototype.selectToBeginningOfNextParagraph = function() {
      return this.expandSelectionsForward(function(selection) {
        return selection.selectToBeginningOfNextParagraph();
      });
    };

    TextEditor.prototype.selectToBeginningOfPreviousParagraph = function() {
      return this.expandSelectionsBackward(function(selection) {
        return selection.selectToBeginningOfPreviousParagraph();
      });
    };

    TextEditor.prototype.selectMarker = function(marker) {
      var range;
      if (marker.isValid()) {
        range = marker.getBufferRange();
        this.setSelectedBufferRange(range);
        return range;
      }
    };

    TextEditor.prototype.getLastSelection = function() {
      this.createLastSelectionIfNeeded();
      return _.last(this.selections);
    };

    TextEditor.prototype.getSelectionAtScreenPosition = function(position) {
      var markers;
      markers = this.selectionsMarkerLayer.findMarkers({
        containsScreenPosition: position
      });
      if (markers.length > 0) {
        return this.cursorsByMarkerId.get(markers[0].id).selection;
      }
    };

    TextEditor.prototype.getSelections = function() {
      this.createLastSelectionIfNeeded();
      return this.selections.slice();
    };

    TextEditor.prototype.getSelectionsOrderedByBufferPosition = function() {
      return this.getSelections().sort(function(a, b) {
        return a.compare(b);
      });
    };

    TextEditor.prototype.selectionIntersectsBufferRange = function(bufferRange) {
      return _.any(this.getSelections(), function(selection) {
        return selection.intersectsBufferRange(bufferRange);
      });
    };

    TextEditor.prototype.addSelectionBelow = function() {
      return this.expandSelectionsForward(function(selection) {
        return selection.addSelectionBelow();
      });
    };

    TextEditor.prototype.addSelectionAbove = function() {
      return this.expandSelectionsBackward(function(selection) {
        return selection.addSelectionAbove();
      });
    };

    TextEditor.prototype.expandSelectionsForward = function(fn) {
      return this.mergeIntersectingSelections((function(_this) {
        return function() {
          var l, len, ref3, selection;
          ref3 = _this.getSelections();
          for (l = 0, len = ref3.length; l < len; l++) {
            selection = ref3[l];
            fn(selection);
          }
        };
      })(this));
    };

    TextEditor.prototype.expandSelectionsBackward = function(fn) {
      return this.mergeIntersectingSelections({
        reversed: true
      }, (function(_this) {
        return function() {
          var l, len, ref3, selection;
          ref3 = _this.getSelections();
          for (l = 0, len = ref3.length; l < len; l++) {
            selection = ref3[l];
            fn(selection);
          }
        };
      })(this));
    };

    TextEditor.prototype.finalizeSelections = function() {
      var l, len, ref3, selection;
      ref3 = this.getSelections();
      for (l = 0, len = ref3.length; l < len; l++) {
        selection = ref3[l];
        selection.finalize();
      }
    };

    TextEditor.prototype.selectionsForScreenRows = function(startRow, endRow) {
      return this.getSelections().filter(function(selection) {
        return selection.intersectsScreenRowRange(startRow, endRow);
      });
    };

    TextEditor.prototype.mergeIntersectingSelections = function() {
      var args;
      args = 1 <= arguments.length ? slice.call(arguments, 0) : [];
      return this.mergeSelections.apply(this, slice.call(args).concat([function(previousSelection, currentSelection) {
        var exclusive;
        exclusive = !currentSelection.isEmpty() && !previousSelection.isEmpty();
        return previousSelection.intersectsWith(currentSelection, exclusive);
      }]));
    };

    TextEditor.prototype.mergeSelectionsOnSameRows = function() {
      var args;
      args = 1 <= arguments.length ? slice.call(arguments, 0) : [];
      return this.mergeSelections.apply(this, slice.call(args).concat([function(previousSelection, currentSelection) {
        var screenRange;
        screenRange = currentSelection.getScreenRange();
        return previousSelection.intersectsScreenRowRange(screenRange.start.row, screenRange.end.row);
      }]));
    };

    TextEditor.prototype.avoidMergingSelections = function() {
      var args;
      args = 1 <= arguments.length ? slice.call(arguments, 0) : [];
      return this.mergeSelections.apply(this, slice.call(args).concat([function() {
        return false;
      }]));
    };

    TextEditor.prototype.mergeSelections = function() {
      var args, fn, head, mergePredicate, options, reducer, ref3, ref4, result, tail;
      args = 1 <= arguments.length ? slice.call(arguments, 0) : [];
      mergePredicate = args.pop();
      if (_.isFunction(_.last(args))) {
        fn = args.pop();
      }
      options = (ref3 = args.pop()) != null ? ref3 : {};
      if (this.suppressSelectionMerging) {
        return typeof fn === "function" ? fn() : void 0;
      }
      if (fn != null) {
        this.suppressSelectionMerging = true;
        result = fn();
        this.suppressSelectionMerging = false;
      }
      reducer = function(disjointSelections, selection) {
        var adjacentSelection;
        adjacentSelection = _.last(disjointSelections);
        if (mergePredicate(adjacentSelection, selection)) {
          adjacentSelection.merge(selection, options);
          return disjointSelections;
        } else {
          return disjointSelections.concat([selection]);
        }
      };
      ref4 = this.getSelectionsOrderedByBufferPosition(), head = ref4[0], tail = 2 <= ref4.length ? slice.call(ref4, 1) : [];
      _.reduce(tail, reducer, [head]);
      if (fn != null) {
        return result;
      }
    };

    TextEditor.prototype.addSelection = function(marker, options) {
      var cursor, l, len, ref3, selection, selectionBufferRange;
      if (options == null) {
        options = {};
      }
      cursor = this.addCursor(marker);
      selection = new Selection(Object.assign({
        editor: this,
        marker: marker,
        cursor: cursor
      }, options));
      this.selections.push(selection);
      selectionBufferRange = selection.getBufferRange();
      this.mergeIntersectingSelections({
        preserveFolds: options.preserveFolds
      });
      if (selection.destroyed) {
        ref3 = this.getSelections();
        for (l = 0, len = ref3.length; l < len; l++) {
          selection = ref3[l];
          if (selection.intersectsBufferRange(selectionBufferRange)) {
            return selection;
          }
        }
      } else {
        this.emitter.emit('did-add-cursor', cursor);
        this.emitter.emit('did-add-selection', selection);
        return selection;
      }
    };

    TextEditor.prototype.removeSelection = function(selection) {
      _.remove(this.cursors, selection.cursor);
      _.remove(this.selections, selection);
      this.cursorsByMarkerId["delete"](selection.cursor.marker.id);
      this.emitter.emit('did-remove-cursor', selection.cursor);
      return this.emitter.emit('did-remove-selection', selection);
    };

    TextEditor.prototype.clearSelections = function(options) {
      this.consolidateSelections();
      return this.getLastSelection().clear(options);
    };

    TextEditor.prototype.consolidateSelections = function() {
      var l, len, ref3, selection, selections;
      selections = this.getSelections();
      if (selections.length > 1) {
        ref3 = selections.slice(1, selections.length);
        for (l = 0, len = ref3.length; l < len; l++) {
          selection = ref3[l];
          selection.destroy();
        }
        selections[0].autoscroll({
          center: true
        });
        return true;
      } else {
        return false;
      }
    };

    TextEditor.prototype.selectionRangeChanged = function(event) {
      var ref3;
      if ((ref3 = this.component) != null) {
        ref3.didChangeSelectionRange();
      }
      return this.emitter.emit('did-change-selection-range', event);
    };

    TextEditor.prototype.createLastSelectionIfNeeded = function() {
      if (this.selections.length === 0) {
        return this.addSelectionForBufferRange([[0, 0], [0, 0]], {
          autoscroll: false,
          preserveFolds: true
        });
      }
    };


    /*
    Section: Searching and Replacing
     */

    TextEditor.prototype.scan = function(regex, options, iterator) {
      if (options == null) {
        options = {};
      }
      if (_.isFunction(options)) {
        iterator = options;
        options = {};
      }
      return this.buffer.scan(regex, options, iterator);
    };

    TextEditor.prototype.scanInBufferRange = function(regex, range, iterator) {
      return this.buffer.scanInRange(regex, range, iterator);
    };

    TextEditor.prototype.backwardsScanInBufferRange = function(regex, range, iterator) {
      return this.buffer.backwardsScanInRange(regex, range, iterator);
    };


    /*
    Section: Tab Behavior
     */

    TextEditor.prototype.getSoftTabs = function() {
      return this.softTabs;
    };

    TextEditor.prototype.setSoftTabs = function(softTabs1) {
      this.softTabs = softTabs1;
      return this.update({
        softTabs: this.softTabs
      });
    };

    TextEditor.prototype.hasAtomicSoftTabs = function() {
      return this.displayLayer.atomicSoftTabs;
    };

    TextEditor.prototype.toggleSoftTabs = function() {
      return this.setSoftTabs(!this.getSoftTabs());
    };

    TextEditor.prototype.getTabLength = function() {
      return this.tokenizedBuffer.getTabLength();
    };

    TextEditor.prototype.setTabLength = function(tabLength) {
      return this.update({
        tabLength: tabLength
      });
    };

    TextEditor.prototype.getInvisibles = function() {
      if (!this.mini && this.showInvisibles && (this.invisibles != null)) {
        return this.invisibles;
      } else {
        return {};
      }
    };

    TextEditor.prototype.doesShowIndentGuide = function() {
      return this.showIndentGuide && !this.mini;
    };

    TextEditor.prototype.getSoftWrapHangingIndentLength = function() {
      return this.displayLayer.softWrapHangingIndent;
    };

    TextEditor.prototype.usesSoftTabs = function() {
      var bufferRow, l, line, ref3, ref4;
      for (bufferRow = l = 0, ref3 = Math.min(1000, this.buffer.getLastRow()); 0 <= ref3 ? l <= ref3 : l >= ref3; bufferRow = 0 <= ref3 ? ++l : --l) {
        if ((ref4 = this.tokenizedBuffer.tokenizedLines[bufferRow]) != null ? ref4.isComment() : void 0) {
          continue;
        }
        line = this.buffer.lineForRow(bufferRow);
        if (line[0] === ' ') {
          return true;
        }
        if (line[0] === '\t') {
          return false;
        }
      }
      return void 0;
    };

    TextEditor.prototype.getTabText = function() {
      return this.buildIndentString(1);
    };

    TextEditor.prototype.normalizeTabsInBufferRange = function(bufferRange) {
      if (!this.getSoftTabs()) {
        return;
      }
      return this.scanInBufferRange(/\t/g, bufferRange, (function(_this) {
        return function(arg) {
          var replace;
          replace = arg.replace;
          return replace(_this.getTabText());
        };
      })(this));
    };


    /*
    Section: Soft Wrap Behavior
     */

    TextEditor.prototype.isSoftWrapped = function() {
      return this.softWrapped;
    };

    TextEditor.prototype.setSoftWrapped = function(softWrapped) {
      this.update({
        softWrapped: softWrapped
      });
      return this.isSoftWrapped();
    };

    TextEditor.prototype.getPreferredLineLength = function() {
      return this.preferredLineLength;
    };

    TextEditor.prototype.toggleSoftWrapped = function() {
      return this.setSoftWrapped(!this.isSoftWrapped());
    };

    TextEditor.prototype.getSoftWrapColumn = function() {
      if (this.isSoftWrapped() && !this.mini) {
        if (this.softWrapAtPreferredLineLength) {
          return Math.min(this.getEditorWidthInChars(), this.preferredLineLength);
        } else {
          return this.getEditorWidthInChars();
        }
      } else {
        return MAX_SCREEN_LINE_LENGTH;
      }
    };


    /*
    Section: Indentation
     */

    TextEditor.prototype.indentationForBufferRow = function(bufferRow) {
      return this.indentLevelForLine(this.lineTextForBufferRow(bufferRow));
    };

    TextEditor.prototype.setIndentationForBufferRow = function(bufferRow, newLevel, arg) {
      var endColumn, newIndentString, preserveLeadingWhitespace;
      preserveLeadingWhitespace = (arg != null ? arg : {}).preserveLeadingWhitespace;
      if (preserveLeadingWhitespace) {
        endColumn = 0;
      } else {
        endColumn = this.lineTextForBufferRow(bufferRow).match(/^\s*/)[0].length;
      }
      newIndentString = this.buildIndentString(newLevel);
      return this.buffer.setTextInRange([[bufferRow, 0], [bufferRow, endColumn]], newIndentString);
    };

    TextEditor.prototype.indentSelectedRows = function() {
      return this.mutateSelectedText(function(selection) {
        return selection.indentSelectedRows();
      });
    };

    TextEditor.prototype.outdentSelectedRows = function() {
      return this.mutateSelectedText(function(selection) {
        return selection.outdentSelectedRows();
      });
    };

    TextEditor.prototype.indentLevelForLine = function(line) {
      return this.tokenizedBuffer.indentLevelForLine(line);
    };

    TextEditor.prototype.autoIndentSelectedRows = function() {
      return this.mutateSelectedText(function(selection) {
        return selection.autoIndentSelectedRows();
      });
    };

    TextEditor.prototype.indent = function(options) {
      if (options == null) {
        options = {};
      }
      if (options.autoIndent == null) {
        options.autoIndent = this.shouldAutoIndent();
      }
      return this.mutateSelectedText(function(selection) {
        return selection.indent(options);
      });
    };

    TextEditor.prototype.buildIndentString = function(level, column) {
      var excessWhitespace, tabStopViolation;
      if (column == null) {
        column = 0;
      }
      if (this.getSoftTabs()) {
        tabStopViolation = column % this.getTabLength();
        return _.multiplyString(" ", Math.floor(level * this.getTabLength()) - tabStopViolation);
      } else {
        excessWhitespace = _.multiplyString(' ', Math.round((level - Math.floor(level)) * this.getTabLength()));
        return _.multiplyString("\t", Math.floor(level)) + excessWhitespace;
      }
    };


    /*
    Section: Grammars
     */

    TextEditor.prototype.getGrammar = function() {
      return this.tokenizedBuffer.grammar;
    };

    TextEditor.prototype.setGrammar = function(grammar) {
      return this.tokenizedBuffer.setGrammar(grammar);
    };

    TextEditor.prototype.reloadGrammar = function() {
      return this.tokenizedBuffer.reloadGrammar();
    };

    TextEditor.prototype.onDidTokenize = function(callback) {
      return this.tokenizedBuffer.onDidTokenize(callback);
    };


    /*
    Section: Managing Syntax Scopes
     */

    TextEditor.prototype.getRootScopeDescriptor = function() {
      return this.tokenizedBuffer.rootScopeDescriptor;
    };

    TextEditor.prototype.scopeDescriptorForBufferPosition = function(bufferPosition) {
      return this.tokenizedBuffer.scopeDescriptorForPosition(bufferPosition);
    };

    TextEditor.prototype.bufferRangeForScopeAtCursor = function(scopeSelector) {
      return this.bufferRangeForScopeAtPosition(scopeSelector, this.getCursorBufferPosition());
    };

    TextEditor.prototype.bufferRangeForScopeAtPosition = function(scopeSelector, position) {
      return this.tokenizedBuffer.bufferRangeForScopeAtPosition(scopeSelector, position);
    };

    TextEditor.prototype.isBufferRowCommented = function(bufferRow) {
      var match;
      if (match = this.lineTextForBufferRow(bufferRow).match(/\S/)) {
        if (this.commentScopeSelector == null) {
          this.commentScopeSelector = new TextMateScopeSelector('comment.*');
        }
        return this.commentScopeSelector.matches(this.scopeDescriptorForBufferPosition([bufferRow, match.index]).scopes);
      }
    };

    TextEditor.prototype.getCursorScope = function() {
      return this.getLastCursor().getScopeDescriptor();
    };

    TextEditor.prototype.tokenForBufferPosition = function(bufferPosition) {
      return this.tokenizedBuffer.tokenForPosition(bufferPosition);
    };


    /*
    Section: Clipboard Operations
     */

    TextEditor.prototype.copySelectedText = function() {
      var l, len, maintainClipboard, previousRange, ref3, selection;
      maintainClipboard = false;
      ref3 = this.getSelectionsOrderedByBufferPosition();
      for (l = 0, len = ref3.length; l < len; l++) {
        selection = ref3[l];
        if (selection.isEmpty()) {
          previousRange = selection.getBufferRange();
          selection.selectLine();
          selection.copy(maintainClipboard, true);
          selection.setBufferRange(previousRange);
        } else {
          selection.copy(maintainClipboard, false);
        }
        maintainClipboard = true;
      }
    };

    TextEditor.prototype.copyOnlySelectedText = function() {
      var l, len, maintainClipboard, ref3, selection;
      maintainClipboard = false;
      ref3 = this.getSelectionsOrderedByBufferPosition();
      for (l = 0, len = ref3.length; l < len; l++) {
        selection = ref3[l];
        if (!selection.isEmpty()) {
          selection.copy(maintainClipboard, false);
          maintainClipboard = true;
        }
      }
    };

    TextEditor.prototype.cutSelectedText = function() {
      var maintainClipboard;
      maintainClipboard = false;
      return this.mutateSelectedText(function(selection) {
        if (selection.isEmpty()) {
          selection.selectLine();
          selection.cut(maintainClipboard, true);
        } else {
          selection.cut(maintainClipboard, false);
        }
        return maintainClipboard = true;
      });
    };

    TextEditor.prototype.pasteText = function(options) {
      var clipboardText, metadata, ref3;
      if (options == null) {
        options = {};
      }
      ref3 = this.constructor.clipboard.readWithMetadata(), clipboardText = ref3.text, metadata = ref3.metadata;
      if (!this.emitWillInsertTextEvent(clipboardText)) {
        return false;
      }
      if (metadata == null) {
        metadata = {};
      }
      options.autoIndent = this.shouldAutoIndentOnPaste();
      return this.mutateSelectedText((function(_this) {
        return function(selection, index) {
          var containsNewlines, cursor, didInsertEvent, fullLine, indentBasis, newPosition, oldPosition, range, ref4, ref5, text;
          if (((ref4 = metadata.selections) != null ? ref4.length : void 0) === _this.getSelections().length) {
            ref5 = metadata.selections[index], text = ref5.text, indentBasis = ref5.indentBasis, fullLine = ref5.fullLine;
          } else {
            indentBasis = metadata.indentBasis, fullLine = metadata.fullLine;
            text = clipboardText;
          }
          delete options.indentBasis;
          cursor = selection.cursor;
          if (indentBasis != null) {
            containsNewlines = text.indexOf('\n') !== -1;
            if (containsNewlines || !cursor.hasPrecedingCharactersOnLine()) {
              if (options.indentBasis == null) {
                options.indentBasis = indentBasis;
              }
            }
          }
          range = null;
          if (fullLine && selection.isEmpty()) {
            oldPosition = selection.getBufferRange().start;
            selection.setBufferRange([[oldPosition.row, 0], [oldPosition.row, 0]]);
            range = selection.insertText(text, options);
            newPosition = oldPosition.translate([1, 0]);
            selection.setBufferRange([newPosition, newPosition]);
          } else {
            range = selection.insertText(text, options);
          }
          didInsertEvent = {
            text: text,
            range: range
          };
          return _this.emitter.emit('did-insert-text', didInsertEvent);
        };
      })(this));
    };

    TextEditor.prototype.cutToEndOfLine = function() {
      var maintainClipboard;
      maintainClipboard = false;
      return this.mutateSelectedText(function(selection) {
        selection.cutToEndOfLine(maintainClipboard);
        return maintainClipboard = true;
      });
    };

    TextEditor.prototype.cutToEndOfBufferLine = function() {
      var maintainClipboard;
      maintainClipboard = false;
      return this.mutateSelectedText(function(selection) {
        selection.cutToEndOfBufferLine(maintainClipboard);
        return maintainClipboard = true;
      });
    };


    /*
    Section: Folds
     */

    TextEditor.prototype.foldCurrentRow = function() {
      var bufferRow;
      bufferRow = this.bufferPositionForScreenPosition(this.getCursorScreenPosition()).row;
      return this.foldBufferRow(bufferRow);
    };

    TextEditor.prototype.unfoldCurrentRow = function() {
      var bufferRow;
      bufferRow = this.bufferPositionForScreenPosition(this.getCursorScreenPosition()).row;
      return this.unfoldBufferRow(bufferRow);
    };

    TextEditor.prototype.foldBufferRow = function(bufferRow) {
      return this.languageMode.foldBufferRow(bufferRow);
    };

    TextEditor.prototype.unfoldBufferRow = function(bufferRow) {
      return this.displayLayer.destroyFoldsIntersectingBufferRange(Range(Point(bufferRow, 0), Point(bufferRow, 2e308)));
    };

    TextEditor.prototype.foldSelectedLines = function() {
      var l, len, ref3, selection;
      ref3 = this.getSelections();
      for (l = 0, len = ref3.length; l < len; l++) {
        selection = ref3[l];
        selection.fold();
      }
    };

    TextEditor.prototype.foldAll = function() {
      return this.languageMode.foldAll();
    };

    TextEditor.prototype.unfoldAll = function() {
      this.languageMode.unfoldAll();
      return this.scrollToCursorPosition();
    };

    TextEditor.prototype.foldAllAtIndentLevel = function(level) {
      return this.languageMode.foldAllAtIndentLevel(level);
    };

    TextEditor.prototype.isFoldableAtBufferRow = function(bufferRow) {
      return this.tokenizedBuffer.isFoldableAtRow(bufferRow);
    };

    TextEditor.prototype.isFoldableAtScreenRow = function(screenRow) {
      return this.isFoldableAtBufferRow(this.bufferRowForScreenRow(screenRow));
    };

    TextEditor.prototype.toggleFoldAtBufferRow = function(bufferRow) {
      if (this.isFoldedAtBufferRow(bufferRow)) {
        return this.unfoldBufferRow(bufferRow);
      } else {
        return this.foldBufferRow(bufferRow);
      }
    };

    TextEditor.prototype.isFoldedAtCursorRow = function() {
      return this.isFoldedAtBufferRow(this.getCursorBufferPosition().row);
    };

    TextEditor.prototype.isFoldedAtBufferRow = function(bufferRow) {
      var range;
      range = Range(Point(bufferRow, 0), Point(bufferRow, this.buffer.lineLengthForRow(bufferRow)));
      return this.displayLayer.foldsIntersectingBufferRange(range).length > 0;
    };

    TextEditor.prototype.isFoldedAtScreenRow = function(screenRow) {
      return this.isFoldedAtBufferRow(this.bufferRowForScreenRow(screenRow));
    };

    TextEditor.prototype.foldBufferRowRange = function(startRow, endRow) {
      return this.foldBufferRange(Range(Point(startRow, 2e308), Point(endRow, 2e308)));
    };

    TextEditor.prototype.foldBufferRange = function(range) {
      return this.displayLayer.foldBufferRange(range);
    };

    TextEditor.prototype.destroyFoldsIntersectingBufferRange = function(bufferRange) {
      return this.displayLayer.destroyFoldsIntersectingBufferRange(bufferRange);
    };


    /*
    Section: Gutters
     */

    TextEditor.prototype.addGutter = function(options) {
      return this.gutterContainer.addGutter(options);
    };

    TextEditor.prototype.getGutters = function() {
      return this.gutterContainer.getGutters();
    };

    TextEditor.prototype.getLineNumberGutter = function() {
      return this.lineNumberGutter;
    };

    TextEditor.prototype.gutterWithName = function(name) {
      return this.gutterContainer.gutterWithName(name);
    };


    /*
    Section: Scrolling the TextEditor
     */

    TextEditor.prototype.scrollToCursorPosition = function(options) {
      var ref3;
      return this.getLastCursor().autoscroll({
        center: (ref3 = options != null ? options.center : void 0) != null ? ref3 : true
      });
    };

    TextEditor.prototype.scrollToBufferPosition = function(bufferPosition, options) {
      return this.scrollToScreenPosition(this.screenPositionForBufferPosition(bufferPosition), options);
    };

    TextEditor.prototype.scrollToScreenPosition = function(screenPosition, options) {
      return this.scrollToScreenRange(new Range(screenPosition, screenPosition), options);
    };

    TextEditor.prototype.scrollToTop = function() {
      Grim.deprecate("This is now a view method. Call TextEditorElement::scrollToTop instead.");
      return this.getElement().scrollToTop();
    };

    TextEditor.prototype.scrollToBottom = function() {
      Grim.deprecate("This is now a view method. Call TextEditorElement::scrollToTop instead.");
      return this.getElement().scrollToBottom();
    };

    TextEditor.prototype.scrollToScreenRange = function(screenRange, options) {
      var ref3, scrollEvent;
      if (options == null) {
        options = {};
      }
      if (options.clip !== false) {
        screenRange = this.clipScreenRange(screenRange);
      }
      scrollEvent = {
        screenRange: screenRange,
        options: options
      };
      if ((ref3 = this.component) != null) {
        ref3.didRequestAutoscroll(scrollEvent);
      }
      return this.emitter.emit("did-request-autoscroll", scrollEvent);
    };

    TextEditor.prototype.getHorizontalScrollbarHeight = function() {
      Grim.deprecate("This is now a view method. Call TextEditorElement::getHorizontalScrollbarHeight instead.");
      return this.getElement().getHorizontalScrollbarHeight();
    };

    TextEditor.prototype.getVerticalScrollbarWidth = function() {
      Grim.deprecate("This is now a view method. Call TextEditorElement::getVerticalScrollbarWidth instead.");
      return this.getElement().getVerticalScrollbarWidth();
    };

    TextEditor.prototype.pageUp = function() {
      return this.moveUp(this.getRowsPerPage());
    };

    TextEditor.prototype.pageDown = function() {
      return this.moveDown(this.getRowsPerPage());
    };

    TextEditor.prototype.selectPageUp = function() {
      return this.selectUp(this.getRowsPerPage());
    };

    TextEditor.prototype.selectPageDown = function() {
      return this.selectDown(this.getRowsPerPage());
    };

    TextEditor.prototype.getRowsPerPage = function() {
      var clientHeight, lineHeight;
      if (this.component != null) {
        clientHeight = this.component.getScrollContainerClientHeight();
        lineHeight = this.component.getLineHeight();
        return Math.max(1, Math.ceil(clientHeight / lineHeight));
      } else {
        return 1;
      }
    };

    Object.defineProperty(TextEditor.prototype, 'rowsPerPage', {
      get: function() {
        return this.getRowsPerPage();
      }
    });


    /*
    Section: Config
     */

    TextEditor.prototype.setScopedSettingsDelegate = function(scopedSettingsDelegate) {
      this.scopedSettingsDelegate = scopedSettingsDelegate;
    };

    TextEditor.prototype.getScopedSettingsDelegate = function() {
      return this.scopedSettingsDelegate;
    };

    TextEditor.prototype.shouldAutoIndent = function() {
      return this.autoIndent;
    };

    TextEditor.prototype.shouldAutoIndentOnPaste = function() {
      return this.autoIndentOnPaste;
    };

    TextEditor.prototype.getScrollPastEnd = function() {
      if (this.getAutoHeight()) {
        return false;
      } else {
        return this.scrollPastEnd;
      }
    };

    TextEditor.prototype.getScrollSensitivity = function() {
      return this.scrollSensitivity;
    };

    TextEditor.prototype.getShowCursorOnSelection = function() {
      return this.showCursorOnSelection;
    };

    TextEditor.prototype.doesShowLineNumbers = function() {
      return this.showLineNumbers;
    };

    TextEditor.prototype.getUndoGroupingInterval = function() {
      return this.undoGroupingInterval;
    };

    TextEditor.prototype.getNonWordCharacters = function(scopes) {
      var ref3, ref4;
      return (ref3 = (ref4 = this.scopedSettingsDelegate) != null ? typeof ref4.getNonWordCharacters === "function" ? ref4.getNonWordCharacters(scopes) : void 0 : void 0) != null ? ref3 : this.nonWordCharacters;
    };

    TextEditor.prototype.getCommentStrings = function(scopes) {
      var ref3;
      return (ref3 = this.scopedSettingsDelegate) != null ? typeof ref3.getCommentStrings === "function" ? ref3.getCommentStrings(scopes) : void 0 : void 0;
    };

    TextEditor.prototype.getIncreaseIndentPattern = function(scopes) {
      var ref3;
      return (ref3 = this.scopedSettingsDelegate) != null ? typeof ref3.getIncreaseIndentPattern === "function" ? ref3.getIncreaseIndentPattern(scopes) : void 0 : void 0;
    };

    TextEditor.prototype.getDecreaseIndentPattern = function(scopes) {
      var ref3;
      return (ref3 = this.scopedSettingsDelegate) != null ? typeof ref3.getDecreaseIndentPattern === "function" ? ref3.getDecreaseIndentPattern(scopes) : void 0 : void 0;
    };

    TextEditor.prototype.getDecreaseNextIndentPattern = function(scopes) {
      var ref3;
      return (ref3 = this.scopedSettingsDelegate) != null ? typeof ref3.getDecreaseNextIndentPattern === "function" ? ref3.getDecreaseNextIndentPattern(scopes) : void 0 : void 0;
    };

    TextEditor.prototype.getFoldEndPattern = function(scopes) {
      var ref3;
      return (ref3 = this.scopedSettingsDelegate) != null ? typeof ref3.getFoldEndPattern === "function" ? ref3.getFoldEndPattern(scopes) : void 0 : void 0;
    };


    /*
    Section: Event Handlers
     */

    TextEditor.prototype.handleGrammarChange = function() {
      this.unfoldAll();
      return this.emitter.emit('did-change-grammar', this.getGrammar());
    };


    /*
    Section: TextEditor Rendering
     */

    TextEditor.prototype.getElement = function() {
      if (this.component != null) {
        return this.component.element;
      } else {
        if (TextEditorComponent == null) {
          TextEditorComponent = require('./text-editor-component');
        }
        if (TextEditorElement == null) {
          TextEditorElement = require('./text-editor-element');
        }
        new TextEditorComponent({
          model: this,
          updatedSynchronously: TextEditorElement.prototype.updatedSynchronously,
          initialScrollTopRow: this.initialScrollTopRow,
          initialScrollLeftColumn: this.initialScrollLeftColumn
        });
        return this.component.element;
      }
    };

    TextEditor.prototype.getAllowedLocations = function() {
      return ['center'];
    };

    TextEditor.prototype.getPlaceholderText = function() {
      return this.placeholderText;
    };

    TextEditor.prototype.setPlaceholderText = function(placeholderText) {
      return this.update({
        placeholderText: placeholderText
      });
    };

    TextEditor.prototype.pixelPositionForBufferPosition = function(bufferPosition) {
      Grim.deprecate("This method is deprecated on the model layer. Use `TextEditorElement::pixelPositionForBufferPosition` instead");
      return this.getElement().pixelPositionForBufferPosition(bufferPosition);
    };

    TextEditor.prototype.pixelPositionForScreenPosition = function(screenPosition) {
      Grim.deprecate("This method is deprecated on the model layer. Use `TextEditorElement::pixelPositionForScreenPosition` instead");
      return this.getElement().pixelPositionForScreenPosition(screenPosition);
    };

    TextEditor.prototype.getVerticalScrollMargin = function() {
      var maxScrollMargin;
      maxScrollMargin = Math.floor(((this.height / this.getLineHeightInPixels()) - 1) / 2);
      return Math.min(this.verticalScrollMargin, maxScrollMargin);
    };

    TextEditor.prototype.setVerticalScrollMargin = function(verticalScrollMargin) {
      this.verticalScrollMargin = verticalScrollMargin;
      return this.verticalScrollMargin;
    };

    TextEditor.prototype.getHorizontalScrollMargin = function() {
      return Math.min(this.horizontalScrollMargin, Math.floor(((this.width / this.getDefaultCharWidth()) - 1) / 2));
    };

    TextEditor.prototype.setHorizontalScrollMargin = function(horizontalScrollMargin) {
      this.horizontalScrollMargin = horizontalScrollMargin;
      return this.horizontalScrollMargin;
    };

    TextEditor.prototype.getLineHeightInPixels = function() {
      return this.lineHeightInPixels;
    };

    TextEditor.prototype.setLineHeightInPixels = function(lineHeightInPixels) {
      this.lineHeightInPixels = lineHeightInPixels;
      return this.lineHeightInPixels;
    };

    TextEditor.prototype.getKoreanCharWidth = function() {
      return this.koreanCharWidth;
    };

    TextEditor.prototype.getHalfWidthCharWidth = function() {
      return this.halfWidthCharWidth;
    };

    TextEditor.prototype.getDoubleWidthCharWidth = function() {
      return this.doubleWidthCharWidth;
    };

    TextEditor.prototype.getDefaultCharWidth = function() {
      return this.defaultCharWidth;
    };

    TextEditor.prototype.ratioForCharacter = function(character) {
      if (isKoreanCharacter(character)) {
        return this.getKoreanCharWidth() / this.getDefaultCharWidth();
      } else if (isHalfWidthCharacter(character)) {
        return this.getHalfWidthCharWidth() / this.getDefaultCharWidth();
      } else if (isDoubleWidthCharacter(character)) {
        return this.getDoubleWidthCharWidth() / this.getDefaultCharWidth();
      } else {
        return 1;
      }
    };

    TextEditor.prototype.setDefaultCharWidth = function(defaultCharWidth, doubleWidthCharWidth, halfWidthCharWidth, koreanCharWidth) {
      if (doubleWidthCharWidth == null) {
        doubleWidthCharWidth = defaultCharWidth;
      }
      if (halfWidthCharWidth == null) {
        halfWidthCharWidth = defaultCharWidth;
      }
      if (koreanCharWidth == null) {
        koreanCharWidth = defaultCharWidth;
      }
      if (defaultCharWidth !== this.defaultCharWidth || doubleWidthCharWidth !== this.doubleWidthCharWidth && halfWidthCharWidth !== this.halfWidthCharWidth && koreanCharWidth !== this.koreanCharWidth) {
        this.defaultCharWidth = defaultCharWidth;
        this.doubleWidthCharWidth = doubleWidthCharWidth;
        this.halfWidthCharWidth = halfWidthCharWidth;
        this.koreanCharWidth = koreanCharWidth;
        if (this.isSoftWrapped()) {
          this.displayLayer.reset({
            softWrapColumn: this.getSoftWrapColumn()
          });
        }
      }
      return defaultCharWidth;
    };

    TextEditor.prototype.setHeight = function(height) {
      Grim.deprecate("This is now a view method. Call TextEditorElement::setHeight instead.");
      return this.getElement().setHeight(height);
    };

    TextEditor.prototype.getHeight = function() {
      Grim.deprecate("This is now a view method. Call TextEditorElement::getHeight instead.");
      return this.getElement().getHeight();
    };

    TextEditor.prototype.getAutoHeight = function() {
      var ref3;
      return (ref3 = this.autoHeight) != null ? ref3 : true;
    };

    TextEditor.prototype.getAutoWidth = function() {
      var ref3;
      return (ref3 = this.autoWidth) != null ? ref3 : false;
    };

    TextEditor.prototype.setWidth = function(width) {
      Grim.deprecate("This is now a view method. Call TextEditorElement::setWidth instead.");
      return this.getElement().setWidth(width);
    };

    TextEditor.prototype.getWidth = function() {
      Grim.deprecate("This is now a view method. Call TextEditorElement::getWidth instead.");
      return this.getElement().getWidth();
    };

    TextEditor.prototype.setFirstVisibleScreenRow = function(screenRow) {
      return this.setScrollTopRow(screenRow);
    };

    TextEditor.prototype.getFirstVisibleScreenRow = function() {
      return this.getElement().component.getFirstVisibleRow();
    };

    TextEditor.prototype.getLastVisibleScreenRow = function() {
      return this.getElement().component.getLastVisibleRow();
    };

    TextEditor.prototype.getVisibleRowRange = function() {
      return [this.getFirstVisibleScreenRow(), this.getLastVisibleScreenRow()];
    };

    TextEditor.prototype.setFirstVisibleScreenColumn = function(column) {
      return this.setScrollLeftColumn(column);
    };

    TextEditor.prototype.getFirstVisibleScreenColumn = function() {
      return this.getElement().component.getFirstVisibleColumn();
    };

    TextEditor.prototype.getScrollTop = function() {
      Grim.deprecate("This is now a view method. Call TextEditorElement::getScrollTop instead.");
      return this.getElement().getScrollTop();
    };

    TextEditor.prototype.setScrollTop = function(scrollTop) {
      Grim.deprecate("This is now a view method. Call TextEditorElement::setScrollTop instead.");
      return this.getElement().setScrollTop(scrollTop);
    };

    TextEditor.prototype.getScrollBottom = function() {
      Grim.deprecate("This is now a view method. Call TextEditorElement::getScrollBottom instead.");
      return this.getElement().getScrollBottom();
    };

    TextEditor.prototype.setScrollBottom = function(scrollBottom) {
      Grim.deprecate("This is now a view method. Call TextEditorElement::setScrollBottom instead.");
      return this.getElement().setScrollBottom(scrollBottom);
    };

    TextEditor.prototype.getScrollLeft = function() {
      Grim.deprecate("This is now a view method. Call TextEditorElement::getScrollLeft instead.");
      return this.getElement().getScrollLeft();
    };

    TextEditor.prototype.setScrollLeft = function(scrollLeft) {
      Grim.deprecate("This is now a view method. Call TextEditorElement::setScrollLeft instead.");
      return this.getElement().setScrollLeft(scrollLeft);
    };

    TextEditor.prototype.getScrollRight = function() {
      Grim.deprecate("This is now a view method. Call TextEditorElement::getScrollRight instead.");
      return this.getElement().getScrollRight();
    };

    TextEditor.prototype.setScrollRight = function(scrollRight) {
      Grim.deprecate("This is now a view method. Call TextEditorElement::setScrollRight instead.");
      return this.getElement().setScrollRight(scrollRight);
    };

    TextEditor.prototype.getScrollHeight = function() {
      Grim.deprecate("This is now a view method. Call TextEditorElement::getScrollHeight instead.");
      return this.getElement().getScrollHeight();
    };

    TextEditor.prototype.getScrollWidth = function() {
      Grim.deprecate("This is now a view method. Call TextEditorElement::getScrollWidth instead.");
      return this.getElement().getScrollWidth();
    };

    TextEditor.prototype.getMaxScrollTop = function() {
      Grim.deprecate("This is now a view method. Call TextEditorElement::getMaxScrollTop instead.");
      return this.getElement().getMaxScrollTop();
    };

    TextEditor.prototype.getScrollTopRow = function() {
      return this.getElement().component.getScrollTopRow();
    };

    TextEditor.prototype.setScrollTopRow = function(scrollTopRow) {
      return this.getElement().component.setScrollTopRow(scrollTopRow);
    };

    TextEditor.prototype.getScrollLeftColumn = function() {
      return this.getElement().component.getScrollLeftColumn();
    };

    TextEditor.prototype.setScrollLeftColumn = function(scrollLeftColumn) {
      return this.getElement().component.setScrollLeftColumn(scrollLeftColumn);
    };

    TextEditor.prototype.intersectsVisibleRowRange = function(startRow, endRow) {
      Grim.deprecate("This is now a view method. Call TextEditorElement::intersectsVisibleRowRange instead.");
      return this.getElement().intersectsVisibleRowRange(startRow, endRow);
    };

    TextEditor.prototype.selectionIntersectsVisibleRowRange = function(selection) {
      Grim.deprecate("This is now a view method. Call TextEditorElement::selectionIntersectsVisibleRowRange instead.");
      return this.getElement().selectionIntersectsVisibleRowRange(selection);
    };

    TextEditor.prototype.screenPositionForPixelPosition = function(pixelPosition) {
      Grim.deprecate("This is now a view method. Call TextEditorElement::screenPositionForPixelPosition instead.");
      return this.getElement().screenPositionForPixelPosition(pixelPosition);
    };

    TextEditor.prototype.pixelRectForScreenRange = function(screenRange) {
      Grim.deprecate("This is now a view method. Call TextEditorElement::pixelRectForScreenRange instead.");
      return this.getElement().pixelRectForScreenRange(screenRange);
    };


    /*
    Section: Utility
     */

    TextEditor.prototype.inspect = function() {
      return "<TextEditor " + this.id + ">";
    };

    TextEditor.prototype.emitWillInsertTextEvent = function(text) {
      var cancel, result, willInsertEvent;
      result = true;
      cancel = function() {
        return result = false;
      };
      willInsertEvent = {
        cancel: cancel,
        text: text
      };
      this.emitter.emit('will-insert-text', willInsertEvent);
      return result;
    };


    /*
    Section: Language Mode Delegated Methods
     */

    TextEditor.prototype.suggestedIndentForBufferRow = function(bufferRow, options) {
      return this.languageMode.suggestedIndentForBufferRow(bufferRow, options);
    };

    TextEditor.prototype.autoIndentBufferRow = function(bufferRow, options) {
      return this.languageMode.autoIndentBufferRow(bufferRow, options);
    };

    TextEditor.prototype.autoIndentBufferRows = function(startRow, endRow) {
      return this.languageMode.autoIndentBufferRows(startRow, endRow);
    };

    TextEditor.prototype.autoDecreaseIndentForBufferRow = function(bufferRow) {
      return this.languageMode.autoDecreaseIndentForBufferRow(bufferRow);
    };

    TextEditor.prototype.toggleLineCommentForBufferRow = function(row) {
      return this.languageMode.toggleLineCommentsForBufferRow(row);
    };

    TextEditor.prototype.toggleLineCommentsForBufferRows = function(start, end) {
      return this.languageMode.toggleLineCommentsForBufferRows(start, end);
    };

    return TextEditor;

  })(Model);

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
