(function() {
  var DefaultDirectoryProvider, Disposable, Emitter, GitRepositoryProvider, Model, Project, TextBuffer, _, fs, path, ref,
    extend = function(child, parent) { for (var key in parent) { if (hasProp.call(parent, key)) child[key] = parent[key]; } function ctor() { this.constructor = child; } ctor.prototype = parent.prototype; child.prototype = new ctor(); child.__super__ = parent.prototype; return child; },
    hasProp = {}.hasOwnProperty,
    indexOf = [].indexOf || function(item) { for (var i = 0, l = this.length; i < l; i++) { if (i in this && this[i] === item) return i; } return -1; },
    slice = [].slice;

  path = require('path');

  _ = require('underscore-plus');

  fs = require('fs-plus');

  ref = require('event-kit'), Emitter = ref.Emitter, Disposable = ref.Disposable;

  TextBuffer = require('text-buffer');

  DefaultDirectoryProvider = require('./default-directory-provider');

  Model = require('./model');

  GitRepositoryProvider = require('./git-repository-provider');

  module.exports = Project = (function(superClass) {
    extend(Project, superClass);


    /*
    Section: Construction and Destruction
     */

    function Project(arg) {
      var config, packageManager;
      this.notificationManager = arg.notificationManager, packageManager = arg.packageManager, config = arg.config, this.applicationDelegate = arg.applicationDelegate;
      this.emitter = new Emitter;
      this.buffers = [];
      this.rootDirectories = [];
      this.repositories = [];
      this.directoryProviders = [];
      this.defaultDirectoryProvider = new DefaultDirectoryProvider();
      this.repositoryPromisesByPath = new Map();
      this.repositoryProviders = [new GitRepositoryProvider(this, config)];
      this.loadPromisesByPath = {};
      this.consumeServices(packageManager);
    }

    Project.prototype.destroyed = function() {
      var buffer, j, k, len, len1, ref1, ref2, repository;
      ref1 = this.buffers.slice();
      for (j = 0, len = ref1.length; j < len; j++) {
        buffer = ref1[j];
        buffer.destroy();
      }
      ref2 = this.repositories.slice();
      for (k = 0, len1 = ref2.length; k < len1; k++) {
        repository = ref2[k];
        if (repository != null) {
          repository.destroy();
        }
      }
      this.rootDirectories = [];
      return this.repositories = [];
    };

    Project.prototype.reset = function(packageManager) {
      var buffer, j, len, ref1;
      this.emitter.dispose();
      this.emitter = new Emitter;
      ref1 = this.buffers;
      for (j = 0, len = ref1.length; j < len; j++) {
        buffer = ref1[j];
        if (buffer != null) {
          buffer.destroy();
        }
      }
      this.buffers = [];
      this.setPaths([]);
      this.loadPromisesByPath = {};
      return this.consumeServices(packageManager);
    };

    Project.prototype.destroyUnretainedBuffers = function() {
      var buffer, j, len, ref1;
      ref1 = this.getBuffers();
      for (j = 0, len = ref1.length; j < len; j++) {
        buffer = ref1[j];
        if (!buffer.isRetained()) {
          buffer.destroy();
        }
      }
    };


    /*
    Section: Serialization
     */

    Project.prototype.deserialize = function(state) {
      var bufferPromises, bufferState, error, j, len, ref1;
      bufferPromises = [];
      ref1 = state.buffers;
      for (j = 0, len = ref1.length; j < len; j++) {
        bufferState = ref1[j];
        if (fs.isDirectorySync(bufferState.filePath)) {
          continue;
        }
        if (bufferState.filePath) {
          try {
            fs.closeSync(fs.openSync(bufferState.filePath, 'r'));
          } catch (error1) {
            error = error1;
            if (error.code !== 'ENOENT') {
              continue;
            }
          }
        }
        if (bufferState.shouldDestroyOnFileDelete == null) {
          bufferState.shouldDestroyOnFileDelete = function() {
            return atom.config.get('core.closeDeletedFileTabs');
          };
        }
        bufferPromises.push(TextBuffer.deserialize(bufferState));
      }
      return Promise.all(bufferPromises).then((function(_this) {
        return function(buffers) {
          var buffer, k, len1, ref2;
          _this.buffers = buffers;
          ref2 = _this.buffers;
          for (k = 0, len1 = ref2.length; k < len1; k++) {
            buffer = ref2[k];
            _this.subscribeToBuffer(buffer);
          }
          return _this.setPaths(state.paths);
        };
      })(this));
    };

    Project.prototype.serialize = function(options) {
      if (options == null) {
        options = {};
      }
      return {
        deserializer: 'Project',
        paths: this.getPaths(),
        buffers: _.compact(this.buffers.map(function(buffer) {
          var isUnloading;
          if (buffer.isRetained()) {
            isUnloading = options.isUnloading === true;
            return buffer.serialize({
              markerLayers: isUnloading,
              history: isUnloading
            });
          }
        }))
      };
    };


    /*
    Section: Event Subscription
     */

    Project.prototype.onDidChangePaths = function(callback) {
      return this.emitter.on('did-change-paths', callback);
    };

    Project.prototype.onDidAddBuffer = function(callback) {
      return this.emitter.on('did-add-buffer', callback);
    };

    Project.prototype.observeBuffers = function(callback) {
      var buffer, j, len, ref1;
      ref1 = this.getBuffers();
      for (j = 0, len = ref1.length; j < len; j++) {
        buffer = ref1[j];
        callback(buffer);
      }
      return this.onDidAddBuffer(callback);
    };


    /*
    Section: Accessing the git repository
     */

    Project.prototype.getRepositories = function() {
      return this.repositories;
    };

    Project.prototype.repositoryForDirectory = function(directory) {
      var pathForDirectory, promise, promises;
      pathForDirectory = directory.getRealPathSync();
      promise = this.repositoryPromisesByPath.get(pathForDirectory);
      if (!promise) {
        promises = this.repositoryProviders.map(function(provider) {
          return provider.repositoryForDirectory(directory);
        });
        promise = Promise.all(promises).then((function(_this) {
          return function(repositories) {
            var ref1, repo;
            repo = (ref1 = _.find(repositories, function(repo) {
              return repo != null;
            })) != null ? ref1 : null;
            if (repo == null) {
              _this.repositoryPromisesByPath["delete"](pathForDirectory);
            }
            if (repo != null) {
              if (typeof repo.onDidDestroy === "function") {
                repo.onDidDestroy(function() {
                  return _this.repositoryPromisesByPath["delete"](pathForDirectory);
                });
              }
            }
            return repo;
          };
        })(this));
        this.repositoryPromisesByPath.set(pathForDirectory, promise);
      }
      return promise;
    };


    /*
    Section: Managing Paths
     */

    Project.prototype.getPaths = function() {
      var j, len, ref1, results, rootDirectory;
      ref1 = this.rootDirectories;
      results = [];
      for (j = 0, len = ref1.length; j < len; j++) {
        rootDirectory = ref1[j];
        results.push(rootDirectory.getPath());
      }
      return results;
    };

    Project.prototype.setPaths = function(projectPaths) {
      var j, k, len, len1, projectPath, ref1, repository;
      ref1 = this.repositories;
      for (j = 0, len = ref1.length; j < len; j++) {
        repository = ref1[j];
        if (repository != null) {
          repository.destroy();
        }
      }
      this.rootDirectories = [];
      this.repositories = [];
      for (k = 0, len1 = projectPaths.length; k < len1; k++) {
        projectPath = projectPaths[k];
        this.addPath(projectPath, {
          emitEvent: false
        });
      }
      return this.emitter.emit('did-change-paths', projectPaths);
    };

    Project.prototype.addPath = function(projectPath, options) {
      var directory, existingDirectory, j, k, len, len1, provider, ref1, ref2, repo;
      directory = this.getDirectoryForProjectPath(projectPath);
      if (!directory.existsSync()) {
        return;
      }
      ref1 = this.getDirectories();
      for (j = 0, len = ref1.length; j < len; j++) {
        existingDirectory = ref1[j];
        if (existingDirectory.getPath() === directory.getPath()) {
          return;
        }
      }
      this.rootDirectories.push(directory);
      repo = null;
      ref2 = this.repositoryProviders;
      for (k = 0, len1 = ref2.length; k < len1; k++) {
        provider = ref2[k];
        if (repo = typeof provider.repositoryForDirectorySync === "function" ? provider.repositoryForDirectorySync(directory) : void 0) {
          break;
        }
      }
      this.repositories.push(repo != null ? repo : null);
      if ((options != null ? options.emitEvent : void 0) !== false) {
        return this.emitter.emit('did-change-paths', this.getPaths());
      }
    };

    Project.prototype.getDirectoryForProjectPath = function(projectPath) {
      var directory, j, len, provider, ref1;
      directory = null;
      ref1 = this.directoryProviders;
      for (j = 0, len = ref1.length; j < len; j++) {
        provider = ref1[j];
        if (directory = typeof provider.directoryForURISync === "function" ? provider.directoryForURISync(projectPath) : void 0) {
          break;
        }
      }
      if (directory == null) {
        directory = this.defaultDirectoryProvider.directoryForURISync(projectPath);
      }
      return directory;
    };

    Project.prototype.removePath = function(projectPath) {
      var directory, i, indexToRemove, j, len, ref1, removedDirectory, removedRepository;
      if (indexOf.call(this.getPaths(), projectPath) < 0) {
        projectPath = this.defaultDirectoryProvider.normalizePath(projectPath);
      }
      indexToRemove = null;
      ref1 = this.rootDirectories;
      for (i = j = 0, len = ref1.length; j < len; i = ++j) {
        directory = ref1[i];
        if (directory.getPath() === projectPath) {
          indexToRemove = i;
          break;
        }
      }
      if (indexToRemove != null) {
        removedDirectory = this.rootDirectories.splice(indexToRemove, 1)[0];
        removedRepository = this.repositories.splice(indexToRemove, 1)[0];
        if (indexOf.call(this.repositories, removedRepository) < 0) {
          if (removedRepository != null) {
            removedRepository.destroy();
          }
        }
        this.emitter.emit("did-change-paths", this.getPaths());
        return true;
      } else {
        return false;
      }
    };

    Project.prototype.getDirectories = function() {
      return this.rootDirectories;
    };

    Project.prototype.resolvePath = function(uri) {
      var projectPath;
      if (!uri) {
        return;
      }
      if (uri != null ? uri.match(/[A-Za-z0-9+-.]+:\/\//) : void 0) {
        return uri;
      } else {
        if (fs.isAbsolute(uri)) {
          return this.defaultDirectoryProvider.normalizePath(fs.resolveHome(uri));
        } else if (projectPath = this.getPaths()[0]) {
          return this.defaultDirectoryProvider.normalizePath(fs.resolveHome(path.join(projectPath, uri)));
        } else {
          return void 0;
        }
      }
    };

    Project.prototype.relativize = function(fullPath) {
      return this.relativizePath(fullPath)[1];
    };

    Project.prototype.relativizePath = function(fullPath) {
      var j, len, ref1, relativePath, result, rootDirectory;
      result = [null, fullPath];
      if (fullPath != null) {
        ref1 = this.rootDirectories;
        for (j = 0, len = ref1.length; j < len; j++) {
          rootDirectory = ref1[j];
          relativePath = rootDirectory.relativize(fullPath);
          if ((relativePath != null ? relativePath.length : void 0) < result[1].length) {
            result = [rootDirectory.getPath(), relativePath];
          }
        }
      }
      return result;
    };

    Project.prototype.contains = function(pathToCheck) {
      return this.rootDirectories.some(function(dir) {
        return dir.contains(pathToCheck);
      });
    };


    /*
    Section: Private
     */

    Project.prototype.consumeServices = function(arg) {
      var serviceHub;
      serviceHub = arg.serviceHub;
      serviceHub.consume('atom.directory-provider', '^0.1.0', (function(_this) {
        return function(provider) {
          _this.directoryProviders.unshift(provider);
          return new Disposable(function() {
            return _this.directoryProviders.splice(_this.directoryProviders.indexOf(provider), 1);
          });
        };
      })(this));
      return serviceHub.consume('atom.repository-provider', '^0.1.0', (function(_this) {
        return function(provider) {
          _this.repositoryProviders.unshift(provider);
          if (indexOf.call(_this.repositories, null) >= 0) {
            _this.setPaths(_this.getPaths());
          }
          return new Disposable(function() {
            return _this.repositoryProviders.splice(_this.repositoryProviders.indexOf(provider), 1);
          });
        };
      })(this));
    };

    Project.prototype.getBuffers = function() {
      return this.buffers.slice();
    };

    Project.prototype.isPathModified = function(filePath) {
      var ref1;
      return (ref1 = this.findBufferForPath(this.resolvePath(filePath))) != null ? ref1.isModified() : void 0;
    };

    Project.prototype.findBufferForPath = function(filePath) {
      return _.find(this.buffers, function(buffer) {
        return buffer.getPath() === filePath;
      });
    };

    Project.prototype.findBufferForId = function(id) {
      return _.find(this.buffers, function(buffer) {
        return buffer.getId() === id;
      });
    };

    Project.prototype.bufferForPathSync = function(filePath) {
      var absoluteFilePath, existingBuffer;
      absoluteFilePath = this.resolvePath(filePath);
      if (filePath) {
        existingBuffer = this.findBufferForPath(absoluteFilePath);
      }
      return existingBuffer != null ? existingBuffer : this.buildBufferSync(absoluteFilePath);
    };

    Project.prototype.bufferForIdSync = function(id) {
      var existingBuffer;
      if (id) {
        existingBuffer = this.findBufferForId(id);
      }
      return existingBuffer != null ? existingBuffer : this.buildBufferSync();
    };

    Project.prototype.bufferForPath = function(absoluteFilePath) {
      var existingBuffer;
      if (absoluteFilePath != null) {
        existingBuffer = this.findBufferForPath(absoluteFilePath);
      }
      if (existingBuffer) {
        return Promise.resolve(existingBuffer);
      } else {
        return this.buildBuffer(absoluteFilePath);
      }
    };

    Project.prototype.shouldDestroyBufferOnFileDelete = function() {
      return atom.config.get('core.closeDeletedFileTabs');
    };

    Project.prototype.buildBufferSync = function(absoluteFilePath) {
      var buffer, params;
      params = {
        shouldDestroyOnFileDelete: this.shouldDestroyBufferOnFileDelete
      };
      if (absoluteFilePath != null) {
        buffer = TextBuffer.loadSync(absoluteFilePath, params);
      } else {
        buffer = new TextBuffer(params);
      }
      this.addBuffer(buffer);
      return buffer;
    };

    Project.prototype.buildBuffer = function(absoluteFilePath) {
      var base, params, promise;
      params = {
        shouldDestroyOnFileDelete: this.shouldDestroyBufferOnFileDelete
      };
      if (absoluteFilePath != null) {
        promise = (base = this.loadPromisesByPath)[absoluteFilePath] != null ? base[absoluteFilePath] : base[absoluteFilePath] = TextBuffer.load(absoluteFilePath, params)["catch"]((function(_this) {
          return function(error) {
            delete _this.loadPromisesByPath[absoluteFilePath];
            throw error;
          };
        })(this));
      } else {
        promise = Promise.resolve(new TextBuffer(params));
      }
      return promise.then((function(_this) {
        return function(buffer) {
          delete _this.loadPromisesByPath[absoluteFilePath];
          _this.addBuffer(buffer);
          return buffer;
        };
      })(this));
    };

    Project.prototype.addBuffer = function(buffer, options) {
      if (options == null) {
        options = {};
      }
      return this.addBufferAtIndex(buffer, this.buffers.length, options);
    };

    Project.prototype.addBufferAtIndex = function(buffer, index, options) {
      if (options == null) {
        options = {};
      }
      this.buffers.splice(index, 0, buffer);
      this.subscribeToBuffer(buffer);
      this.emitter.emit('did-add-buffer', buffer);
      return buffer;
    };

    Project.prototype.removeBuffer = function(buffer) {
      var index;
      index = this.buffers.indexOf(buffer);
      if (index !== -1) {
        return this.removeBufferAtIndex(index);
      }
    };

    Project.prototype.removeBufferAtIndex = function(index, options) {
      var buffer;
      if (options == null) {
        options = {};
      }
      buffer = this.buffers.splice(index, 1)[0];
      return buffer != null ? buffer.destroy() : void 0;
    };

    Project.prototype.eachBuffer = function() {
      var args, buffer, callback, j, len, ref1, subscriber;
      args = 1 <= arguments.length ? slice.call(arguments, 0) : [];
      if (args.length > 1) {
        subscriber = args.shift();
      }
      callback = args.shift();
      ref1 = this.getBuffers();
      for (j = 0, len = ref1.length; j < len; j++) {
        buffer = ref1[j];
        callback(buffer);
      }
      if (subscriber) {
        return subscriber.subscribe(this, 'buffer-created', function(buffer) {
          return callback(buffer);
        });
      } else {
        return this.on('buffer-created', function(buffer) {
          return callback(buffer);
        });
      }
    };

    Project.prototype.subscribeToBuffer = function(buffer) {
      buffer.onWillSave((function(_this) {
        return function(arg) {
          var path;
          path = arg.path;
          return _this.applicationDelegate.emitWillSavePath(path);
        };
      })(this));
      buffer.onDidSave((function(_this) {
        return function(arg) {
          var path;
          path = arg.path;
          return _this.applicationDelegate.emitDidSavePath(path);
        };
      })(this));
      buffer.onDidDestroy((function(_this) {
        return function() {
          return _this.removeBuffer(buffer);
        };
      })(this));
      buffer.onDidChangePath((function(_this) {
        return function() {
          if (!(_this.getPaths().length > 0)) {
            return _this.setPaths([path.dirname(buffer.getPath())]);
          }
        };
      })(this));
      return buffer.onWillThrowWatchError((function(_this) {
        return function(arg) {
          var error, handle;
          error = arg.error, handle = arg.handle;
          handle();
          return _this.notificationManager.addWarning("Unable to read file after file `" + error.eventType + "` event.\nMake sure you have permission to access `" + (buffer.getPath()) + "`.", {
            detail: error.message,
            dismissable: true
          });
        };
      })(this));
    };

    return Project;

  })(Model);

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
