(function() {
  var CommandRegistry, CompositeDisposable, Disposable, Emitter, InlineListener, SelectorBasedListener, SequenceCount, _, calculateSpecificity, ref, ref1, validateSelector,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  ref = require('event-kit'), Emitter = ref.Emitter, Disposable = ref.Disposable, CompositeDisposable = ref.CompositeDisposable;

  ref1 = require('clear-cut'), calculateSpecificity = ref1.calculateSpecificity, validateSelector = ref1.validateSelector;

  _ = require('underscore-plus');

  SequenceCount = 0;

  module.exports = CommandRegistry = (function() {
    function CommandRegistry() {
      this.handleCommandEvent = bind(this.handleCommandEvent, this);
      this.rootNode = null;
      this.clear();
    }

    CommandRegistry.prototype.clear = function() {
      this.registeredCommands = {};
      this.selectorBasedListenersByCommandName = {};
      this.inlineListenersByCommandName = {};
      return this.emitter = new Emitter;
    };

    CommandRegistry.prototype.attach = function(rootNode) {
      var command, results;
      this.rootNode = rootNode;
      for (command in this.selectorBasedListenersByCommandName) {
        this.commandRegistered(command);
      }
      results = [];
      for (command in this.inlineListenersByCommandName) {
        results.push(this.commandRegistered(command));
      }
      return results;
    };

    CommandRegistry.prototype.destroy = function() {
      var commandName;
      for (commandName in this.registeredCommands) {
        this.rootNode.removeEventListener(commandName, this.handleCommandEvent, true);
      }
    };

    CommandRegistry.prototype.add = function(target, commandName, callback, throwOnInvalidSelector) {
      var commands, disposable;
      if (throwOnInvalidSelector == null) {
        throwOnInvalidSelector = true;
      }
      if (typeof commandName === 'object') {
        commands = commandName;
        throwOnInvalidSelector = callback;
        disposable = new CompositeDisposable;
        for (commandName in commands) {
          callback = commands[commandName];
          disposable.add(this.add(target, commandName, callback, throwOnInvalidSelector));
        }
        return disposable;
      }
      if (typeof callback !== 'function') {
        throw new Error("Can't register a command with non-function callback.");
      }
      if (typeof target === 'string') {
        if (throwOnInvalidSelector) {
          validateSelector(target);
        }
        return this.addSelectorBasedListener(target, commandName, callback);
      } else {
        return this.addInlineListener(target, commandName, callback);
      }
    };

    CommandRegistry.prototype.addSelectorBasedListener = function(selector, commandName, callback) {
      var base, listener, listenersForCommand;
      if ((base = this.selectorBasedListenersByCommandName)[commandName] == null) {
        base[commandName] = [];
      }
      listenersForCommand = this.selectorBasedListenersByCommandName[commandName];
      listener = new SelectorBasedListener(selector, callback);
      listenersForCommand.push(listener);
      this.commandRegistered(commandName);
      return new Disposable((function(_this) {
        return function() {
          listenersForCommand.splice(listenersForCommand.indexOf(listener), 1);
          if (listenersForCommand.length === 0) {
            return delete _this.selectorBasedListenersByCommandName[commandName];
          }
        };
      })(this));
    };

    CommandRegistry.prototype.addInlineListener = function(element, commandName, callback) {
      var base, listener, listenersForCommand, listenersForElement;
      if ((base = this.inlineListenersByCommandName)[commandName] == null) {
        base[commandName] = new WeakMap;
      }
      listenersForCommand = this.inlineListenersByCommandName[commandName];
      if (!(listenersForElement = listenersForCommand.get(element))) {
        listenersForElement = [];
        listenersForCommand.set(element, listenersForElement);
      }
      listener = new InlineListener(callback);
      listenersForElement.push(listener);
      this.commandRegistered(commandName);
      return new Disposable(function() {
        listenersForElement.splice(listenersForElement.indexOf(listener), 1);
        if (listenersForElement.length === 0) {
          return listenersForCommand["delete"](element);
        }
      });
    };

    CommandRegistry.prototype.findCommands = function(arg) {
      var commandName, commandNames, commands, currentTarget, i, len, listener, listeners, name, ref2, ref3, ref4, target;
      target = arg.target;
      commandNames = new Set;
      commands = [];
      currentTarget = target;
      while (true) {
        ref2 = this.inlineListenersByCommandName;
        for (name in ref2) {
          listeners = ref2[name];
          if (listeners.has(currentTarget) && !commandNames.has(name)) {
            commandNames.add(name);
            commands.push({
              name: name,
              displayName: _.humanizeEventName(name)
            });
          }
        }
        ref3 = this.selectorBasedListenersByCommandName;
        for (commandName in ref3) {
          listeners = ref3[commandName];
          for (i = 0, len = listeners.length; i < len; i++) {
            listener = listeners[i];
            if (typeof currentTarget.webkitMatchesSelector === "function" ? currentTarget.webkitMatchesSelector(listener.selector) : void 0) {
              if (!commandNames.has(commandName)) {
                commandNames.add(commandName);
                commands.push({
                  name: commandName,
                  displayName: _.humanizeEventName(commandName)
                });
              }
            }
          }
        }
        if (currentTarget === window) {
          break;
        }
        currentTarget = (ref4 = currentTarget.parentNode) != null ? ref4 : window;
      }
      return commands;
    };

    CommandRegistry.prototype.dispatch = function(target, commandName, detail) {
      var event;
      event = new CustomEvent(commandName, {
        bubbles: true,
        detail: detail
      });
      Object.defineProperty(event, 'target', {
        value: target
      });
      return this.handleCommandEvent(event);
    };

    CommandRegistry.prototype.onWillDispatch = function(callback) {
      return this.emitter.on('will-dispatch', callback);
    };

    CommandRegistry.prototype.onDidDispatch = function(callback) {
      return this.emitter.on('did-dispatch', callback);
    };

    CommandRegistry.prototype.getSnapshot = function() {
      var commandName, listeners, ref2, snapshot;
      snapshot = {};
      ref2 = this.selectorBasedListenersByCommandName;
      for (commandName in ref2) {
        listeners = ref2[commandName];
        snapshot[commandName] = listeners.slice();
      }
      return snapshot;
    };

    CommandRegistry.prototype.restoreSnapshot = function(snapshot) {
      var commandName, listeners;
      this.selectorBasedListenersByCommandName = {};
      for (commandName in snapshot) {
        listeners = snapshot[commandName];
        this.selectorBasedListenersByCommandName[commandName] = listeners.slice();
      }
    };

    CommandRegistry.prototype.handleCommandEvent = function(event) {
      var currentTarget, dispatchedEvent, i, immediatePropagationStopped, j, key, len, listener, listeners, matched, propagationStopped, ref2, ref3, ref4, ref5, ref6, selectorBasedListeners;
      propagationStopped = false;
      immediatePropagationStopped = false;
      matched = false;
      currentTarget = event.target;
      dispatchedEvent = new CustomEvent(event.type, {
        bubbles: true,
        detail: event.detail
      });
      Object.defineProperty(dispatchedEvent, 'eventPhase', {
        value: Event.BUBBLING_PHASE
      });
      Object.defineProperty(dispatchedEvent, 'currentTarget', {
        get: function() {
          return currentTarget;
        }
      });
      Object.defineProperty(dispatchedEvent, 'target', {
        value: currentTarget
      });
      Object.defineProperty(dispatchedEvent, 'preventDefault', {
        value: function() {
          return event.preventDefault();
        }
      });
      Object.defineProperty(dispatchedEvent, 'stopPropagation', {
        value: function() {
          event.stopPropagation();
          return propagationStopped = true;
        }
      });
      Object.defineProperty(dispatchedEvent, 'stopImmediatePropagation', {
        value: function() {
          event.stopImmediatePropagation();
          propagationStopped = true;
          return immediatePropagationStopped = true;
        }
      });
      Object.defineProperty(dispatchedEvent, 'abortKeyBinding', {
        value: function() {
          return typeof event.abortKeyBinding === "function" ? event.abortKeyBinding() : void 0;
        }
      });
      ref2 = Object.keys(event);
      for (i = 0, len = ref2.length; i < len; i++) {
        key = ref2[i];
        dispatchedEvent[key] = event[key];
      }
      this.emitter.emit('will-dispatch', dispatchedEvent);
      while (true) {
        listeners = (ref3 = (ref4 = this.inlineListenersByCommandName[event.type]) != null ? ref4.get(currentTarget) : void 0) != null ? ref3 : [];
        if (currentTarget.webkitMatchesSelector != null) {
          selectorBasedListeners = ((ref5 = this.selectorBasedListenersByCommandName[event.type]) != null ? ref5 : []).filter(function(listener) {
            return currentTarget.webkitMatchesSelector(listener.selector);
          }).sort(function(a, b) {
            return a.compare(b);
          });
          listeners = selectorBasedListeners.concat(listeners);
        }
        if (listeners.length > 0) {
          matched = true;
        }
        for (j = listeners.length - 1; j >= 0; j += -1) {
          listener = listeners[j];
          if (immediatePropagationStopped) {
            break;
          }
          listener.callback.call(currentTarget, dispatchedEvent);
        }
        if (currentTarget === window) {
          break;
        }
        if (propagationStopped) {
          break;
        }
        currentTarget = (ref6 = currentTarget.parentNode) != null ? ref6 : window;
      }
      this.emitter.emit('did-dispatch', dispatchedEvent);
      return matched;
    };

    CommandRegistry.prototype.commandRegistered = function(commandName) {
      if ((this.rootNode != null) && !this.registeredCommands[commandName]) {
        this.rootNode.addEventListener(commandName, this.handleCommandEvent, true);
        return this.registeredCommands[commandName] = true;
      }
    };

    return CommandRegistry;

  })();

  SelectorBasedListener = (function() {
    function SelectorBasedListener(selector1, callback1) {
      this.selector = selector1;
      this.callback = callback1;
      this.specificity = calculateSpecificity(this.selector);
      this.sequenceNumber = SequenceCount++;
    }

    SelectorBasedListener.prototype.compare = function(other) {
      return this.specificity - other.specificity || this.sequenceNumber - other.sequenceNumber;
    };

    return SelectorBasedListener;

  })();

  InlineListener = (function() {
    function InlineListener(callback1) {
      this.callback = callback1;
    }

    return InlineListener;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
