(function() {
  var ApplicationDelegate, Disposable, _, getWindowLoadSettings, ipcHelpers, ipcRenderer, ref, remote, shell,
    slice = [].slice;

  _ = require('underscore-plus');

  ref = require('electron'), ipcRenderer = ref.ipcRenderer, remote = ref.remote, shell = ref.shell;

  ipcHelpers = require('./ipc-helpers');

  Disposable = require('event-kit').Disposable;

  getWindowLoadSettings = require('./get-window-load-settings');

  module.exports = ApplicationDelegate = (function() {
    function ApplicationDelegate() {}

    ApplicationDelegate.prototype.getWindowLoadSettings = function() {
      return getWindowLoadSettings();
    };

    ApplicationDelegate.prototype.open = function(params) {
      return ipcRenderer.send('open', params);
    };

    ApplicationDelegate.prototype.pickFolder = function(callback) {
      var responseChannel;
      responseChannel = "atom-pick-folder-response";
      ipcRenderer.on(responseChannel, function(event, path) {
        ipcRenderer.removeAllListeners(responseChannel);
        return callback(path);
      });
      return ipcRenderer.send("pick-folder", responseChannel);
    };

    ApplicationDelegate.prototype.getCurrentWindow = function() {
      return remote.getCurrentWindow();
    };

    ApplicationDelegate.prototype.closeWindow = function() {
      return ipcHelpers.call('window-method', 'close');
    };

    ApplicationDelegate.prototype.getTemporaryWindowState = function() {
      return ipcHelpers.call('get-temporary-window-state').then(function(stateJSON) {
        return JSON.parse(stateJSON);
      });
    };

    ApplicationDelegate.prototype.setTemporaryWindowState = function(state) {
      return ipcHelpers.call('set-temporary-window-state', JSON.stringify(state));
    };

    ApplicationDelegate.prototype.getWindowSize = function() {
      var height, ref1, width;
      ref1 = remote.getCurrentWindow().getSize(), width = ref1[0], height = ref1[1];
      return {
        width: width,
        height: height
      };
    };

    ApplicationDelegate.prototype.setWindowSize = function(width, height) {
      return ipcHelpers.call('set-window-size', width, height);
    };

    ApplicationDelegate.prototype.getWindowPosition = function() {
      var ref1, x, y;
      ref1 = remote.getCurrentWindow().getPosition(), x = ref1[0], y = ref1[1];
      return {
        x: x,
        y: y
      };
    };

    ApplicationDelegate.prototype.setWindowPosition = function(x, y) {
      return ipcHelpers.call('set-window-position', x, y);
    };

    ApplicationDelegate.prototype.centerWindow = function() {
      return ipcHelpers.call('center-window');
    };

    ApplicationDelegate.prototype.focusWindow = function() {
      return ipcHelpers.call('focus-window');
    };

    ApplicationDelegate.prototype.showWindow = function() {
      return ipcHelpers.call('show-window');
    };

    ApplicationDelegate.prototype.hideWindow = function() {
      return ipcHelpers.call('hide-window');
    };

    ApplicationDelegate.prototype.reloadWindow = function() {
      return ipcHelpers.call('window-method', 'reload');
    };

    ApplicationDelegate.prototype.restartApplication = function() {
      return ipcRenderer.send("restart-application");
    };

    ApplicationDelegate.prototype.minimizeWindow = function() {
      return ipcHelpers.call('window-method', 'minimize');
    };

    ApplicationDelegate.prototype.isWindowMaximized = function() {
      return remote.getCurrentWindow().isMaximized();
    };

    ApplicationDelegate.prototype.maximizeWindow = function() {
      return ipcHelpers.call('window-method', 'maximize');
    };

    ApplicationDelegate.prototype.unmaximizeWindow = function() {
      return ipcHelpers.call('window-method', 'unmaximize');
    };

    ApplicationDelegate.prototype.isWindowFullScreen = function() {
      return remote.getCurrentWindow().isFullScreen();
    };

    ApplicationDelegate.prototype.setWindowFullScreen = function(fullScreen) {
      if (fullScreen == null) {
        fullScreen = false;
      }
      return ipcHelpers.call('window-method', 'setFullScreen', fullScreen);
    };

    ApplicationDelegate.prototype.onDidEnterFullScreen = function(callback) {
      return ipcHelpers.on(ipcRenderer, 'did-enter-full-screen', callback);
    };

    ApplicationDelegate.prototype.onDidLeaveFullScreen = function(callback) {
      return ipcHelpers.on(ipcRenderer, 'did-leave-full-screen', callback);
    };

    ApplicationDelegate.prototype.openWindowDevTools = function() {
      return new Promise(process.nextTick).then(function() {
        return ipcHelpers.call('window-method', 'openDevTools');
      });
    };

    ApplicationDelegate.prototype.closeWindowDevTools = function() {
      return new Promise(process.nextTick).then(function() {
        return ipcHelpers.call('window-method', 'closeDevTools');
      });
    };

    ApplicationDelegate.prototype.toggleWindowDevTools = function() {
      return new Promise(process.nextTick).then(function() {
        return ipcHelpers.call('window-method', 'toggleDevTools');
      });
    };

    ApplicationDelegate.prototype.executeJavaScriptInWindowDevTools = function(code) {
      return ipcRenderer.send("execute-javascript-in-dev-tools", code);
    };

    ApplicationDelegate.prototype.setWindowDocumentEdited = function(edited) {
      return ipcHelpers.call('window-method', 'setDocumentEdited', edited);
    };

    ApplicationDelegate.prototype.setRepresentedFilename = function(filename) {
      return ipcHelpers.call('window-method', 'setRepresentedFilename', filename);
    };

    ApplicationDelegate.prototype.addRecentDocument = function(filename) {
      return ipcRenderer.send("add-recent-document", filename);
    };

    ApplicationDelegate.prototype.setRepresentedDirectoryPaths = function(paths) {
      return ipcHelpers.call('window-method', 'setRepresentedDirectoryPaths', paths);
    };

    ApplicationDelegate.prototype.setAutoHideWindowMenuBar = function(autoHide) {
      return ipcHelpers.call('window-method', 'setAutoHideMenuBar', autoHide);
    };

    ApplicationDelegate.prototype.setWindowMenuBarVisibility = function(visible) {
      return remote.getCurrentWindow().setMenuBarVisibility(visible);
    };

    ApplicationDelegate.prototype.getPrimaryDisplayWorkAreaSize = function() {
      return remote.screen.getPrimaryDisplay().workAreaSize;
    };

    ApplicationDelegate.prototype.getUserDefault = function(key, type) {
      return remote.systemPreferences.getUserDefault(key, type);
    };

    ApplicationDelegate.prototype.confirm = function(arg) {
      var buttonLabels, buttons, callback, chosen, detailedMessage, message;
      message = arg.message, detailedMessage = arg.detailedMessage, buttons = arg.buttons;
      if (buttons == null) {
        buttons = {};
      }
      if (_.isArray(buttons)) {
        buttonLabels = buttons;
      } else {
        buttonLabels = Object.keys(buttons);
      }
      chosen = remote.dialog.showMessageBox(remote.getCurrentWindow(), {
        type: 'info',
        message: message,
        detail: detailedMessage,
        buttons: buttonLabels
      });
      if (_.isArray(buttons)) {
        return chosen;
      } else {
        callback = buttons[buttonLabels[chosen]];
        return typeof callback === "function" ? callback() : void 0;
      }
    };

    ApplicationDelegate.prototype.showMessageDialog = function(params) {};

    ApplicationDelegate.prototype.showSaveDialog = function(params) {
      if (typeof params === 'string') {
        params = {
          defaultPath: params
        };
      }
      return this.getCurrentWindow().showSaveDialog(params);
    };

    ApplicationDelegate.prototype.playBeepSound = function() {
      return shell.beep();
    };

    ApplicationDelegate.prototype.onDidOpenLocations = function(callback) {
      var outerCallback;
      outerCallback = function(event, message, detail) {
        if (message === 'open-locations') {
          return callback(detail);
        }
      };
      ipcRenderer.on('message', outerCallback);
      return new Disposable(function() {
        return ipcRenderer.removeListener('message', outerCallback);
      });
    };

    ApplicationDelegate.prototype.onUpdateAvailable = function(callback) {
      var outerCallback;
      outerCallback = function(event, message, detail) {
        if (message === 'did-begin-downloading-update') {
          return callback(detail);
        }
      };
      ipcRenderer.on('message', outerCallback);
      return new Disposable(function() {
        return ipcRenderer.removeListener('message', outerCallback);
      });
    };

    ApplicationDelegate.prototype.onDidBeginDownloadingUpdate = function(callback) {
      return this.onUpdateAvailable(callback);
    };

    ApplicationDelegate.prototype.onDidBeginCheckingForUpdate = function(callback) {
      var outerCallback;
      outerCallback = function(event, message, detail) {
        if (message === 'checking-for-update') {
          return callback(detail);
        }
      };
      ipcRenderer.on('message', outerCallback);
      return new Disposable(function() {
        return ipcRenderer.removeListener('message', outerCallback);
      });
    };

    ApplicationDelegate.prototype.onDidCompleteDownloadingUpdate = function(callback) {
      var outerCallback;
      outerCallback = function(event, message, detail) {
        if (message === 'update-available') {
          return callback(detail);
        }
      };
      ipcRenderer.on('message', outerCallback);
      return new Disposable(function() {
        return ipcRenderer.removeListener('message', outerCallback);
      });
    };

    ApplicationDelegate.prototype.onUpdateNotAvailable = function(callback) {
      var outerCallback;
      outerCallback = function(event, message, detail) {
        if (message === 'update-not-available') {
          return callback(detail);
        }
      };
      ipcRenderer.on('message', outerCallback);
      return new Disposable(function() {
        return ipcRenderer.removeListener('message', outerCallback);
      });
    };

    ApplicationDelegate.prototype.onUpdateError = function(callback) {
      var outerCallback;
      outerCallback = function(event, message, detail) {
        if (message === 'update-error') {
          return callback(detail);
        }
      };
      ipcRenderer.on('message', outerCallback);
      return new Disposable(function() {
        return ipcRenderer.removeListener('message', outerCallback);
      });
    };

    ApplicationDelegate.prototype.onApplicationMenuCommand = function(callback) {
      var outerCallback;
      outerCallback = function() {
        var args, event;
        event = arguments[0], args = 2 <= arguments.length ? slice.call(arguments, 1) : [];
        return callback.apply(null, args);
      };
      ipcRenderer.on('command', outerCallback);
      return new Disposable(function() {
        return ipcRenderer.removeListener('command', outerCallback);
      });
    };

    ApplicationDelegate.prototype.onContextMenuCommand = function(callback) {
      var outerCallback;
      outerCallback = function() {
        var args, event;
        event = arguments[0], args = 2 <= arguments.length ? slice.call(arguments, 1) : [];
        return callback.apply(null, args);
      };
      ipcRenderer.on('context-command', outerCallback);
      return new Disposable(function() {
        return ipcRenderer.removeListener('context-command', outerCallback);
      });
    };

    ApplicationDelegate.prototype.onDidRequestUnload = function(callback) {
      var outerCallback;
      outerCallback = function(event, message) {
        return callback(event).then(function(shouldUnload) {
          return ipcRenderer.send('did-prepare-to-unload', shouldUnload);
        });
      };
      ipcRenderer.on('prepare-to-unload', outerCallback);
      return new Disposable(function() {
        return ipcRenderer.removeListener('prepare-to-unload', outerCallback);
      });
    };

    ApplicationDelegate.prototype.onDidChangeHistoryManager = function(callback) {
      var outerCallback;
      outerCallback = function(event, message) {
        return callback(event);
      };
      ipcRenderer.on('did-change-history-manager', outerCallback);
      return new Disposable(function() {
        return ipcRenderer.removeListener('did-change-history-manager', outerCallback);
      });
    };

    ApplicationDelegate.prototype.didChangeHistoryManager = function() {
      return ipcRenderer.send('did-change-history-manager');
    };

    ApplicationDelegate.prototype.openExternal = function(url) {
      return shell.openExternal(url);
    };

    ApplicationDelegate.prototype.checkForUpdate = function() {
      return ipcRenderer.send('command', 'application:check-for-update');
    };

    ApplicationDelegate.prototype.restartAndInstallUpdate = function() {
      return ipcRenderer.send('command', 'application:install-update');
    };

    ApplicationDelegate.prototype.getAutoUpdateManagerState = function() {
      return ipcRenderer.sendSync('get-auto-update-manager-state');
    };

    ApplicationDelegate.prototype.getAutoUpdateManagerErrorMessage = function() {
      return ipcRenderer.sendSync('get-auto-update-manager-error');
    };

    ApplicationDelegate.prototype.emitWillSavePath = function(path) {
      return ipcRenderer.sendSync('will-save-path', path);
    };

    ApplicationDelegate.prototype.emitDidSavePath = function(path) {
      return ipcRenderer.sendSync('did-save-path', path);
    };

    ApplicationDelegate.prototype.resolveProxy = function(requestId, url) {
      return ipcRenderer.send('resolve-proxy', requestId, url);
    };

    ApplicationDelegate.prototype.onDidResolveProxy = function(callback) {
      var outerCallback;
      outerCallback = function(event, requestId, proxy) {
        return callback(requestId, proxy);
      };
      ipcRenderer.on('did-resolve-proxy', outerCallback);
      return new Disposable(function() {
        return ipcRenderer.removeListener('did-resolve-proxy', outerCallback);
      });
    };

    return ApplicationDelegate;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
