(function() {
  var AddDialog, BufferedProcess, CompositeDisposable, CopyDialog, Directory, DirectoryView, Emitter, Minimatch, MoveDialog, RootDragAndDrop, TREE_VIEW_URI, TreeView, _, fs, getFullExtension, getStyleObject, nextId, path, ref, ref1, repoForPath, shell, toggleConfig, updateEditorsForPath,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; },
    slice = [].slice,
    indexOf = [].indexOf || function(item) { for (var i = 0, l = this.length; i < l; i++) { if (i in this && this[i] === item) return i; } return -1; };

  path = require('path');

  shell = require('electron').shell;

  _ = require('underscore-plus');

  ref = require('atom'), BufferedProcess = ref.BufferedProcess, CompositeDisposable = ref.CompositeDisposable, Emitter = ref.Emitter;

  ref1 = require("./helpers"), repoForPath = ref1.repoForPath, getStyleObject = ref1.getStyleObject, getFullExtension = ref1.getFullExtension, updateEditorsForPath = ref1.updateEditorsForPath;

  fs = require('fs-plus');

  AddDialog = require('./add-dialog');

  MoveDialog = require('./move-dialog');

  CopyDialog = require('./copy-dialog');

  Minimatch = null;

  Directory = require('./directory');

  DirectoryView = require('./directory-view');

  RootDragAndDrop = require('./root-drag-and-drop');

  TREE_VIEW_URI = 'atom://tree-view';

  toggleConfig = function(keyPath) {
    return atom.config.set(keyPath, !atom.config.get(keyPath));
  };

  nextId = 1;

  module.exports = TreeView = (function() {
    function TreeView(state) {
      this.onDragLeave = bind(this.onDragLeave, this);
      this.onDragEnter = bind(this.onDragEnter, this);
      this.onStylesheetsChanged = bind(this.onStylesheetsChanged, this);
      var observer;
      this.id = nextId++;
      this.element = document.createElement('div');
      this.element.classList.add('tool-panel', 'tree-view');
      this.element.tabIndex = -1;
      this.list = document.createElement('ol');
      this.list.classList.add('full-menu', 'list-tree', 'has-collapsable-children', 'focusable-panel');
      this.element.appendChild(this.list);
      this.disposables = new CompositeDisposable;
      this.emitter = new Emitter;
      this.roots = [];
      this.selectedPath = null;
      this.ignoredPatterns = [];
      this.useSyncFS = false;
      this.currentlyOpening = new Map;
      this.dragEventCounts = new WeakMap;
      this.rootDragAndDrop = new RootDragAndDrop(this);
      this.handleEvents();
      process.nextTick((function(_this) {
        return function() {
          var onStylesheetsChanged;
          _this.onStylesheetsChanged();
          onStylesheetsChanged = _.debounce(_this.onStylesheetsChanged, 100);
          _this.disposables.add(atom.styles.onDidAddStyleElement(onStylesheetsChanged));
          _this.disposables.add(atom.styles.onDidRemoveStyleElement(onStylesheetsChanged));
          return _this.disposables.add(atom.styles.onDidUpdateStyleElement(onStylesheetsChanged));
        };
      })(this));
      this.updateRoots(state.directoryExpansionStates);
      this.selectEntry(this.roots[0]);
      if (state.selectedPath) {
        this.selectEntryForPath(state.selectedPath);
      }
      if ((state.scrollTop != null) || (state.scrollLeft != null)) {
        observer = new IntersectionObserver((function(_this) {
          return function() {
            if (_this.isVisible()) {
              _this.element.scrollTop = state.scrollTop;
              _this.element.scrollLeft = state.scrollLeft;
              return observer.disconnect();
            }
          };
        })(this));
        observer.observe(this.element);
      }
      if (state.width > 0) {
        this.element.style.width = state.width + "px";
      }
      this.disposables.add(this.onEntryMoved(function(arg) {
        var initialPath, newPath;
        initialPath = arg.initialPath, newPath = arg.newPath;
        return updateEditorsForPath(initialPath, newPath);
      }));
      this.disposables.add(this.onEntryDeleted(function(arg) {
        var editor, j, len, path, ref2, ref3, results;
        path = arg.path;
        ref2 = atom.workspace.getTextEditors();
        results = [];
        for (j = 0, len = ref2.length; j < len; j++) {
          editor = ref2[j];
          if (editor != null ? (ref3 = editor.getPath()) != null ? ref3.startsWith(path) : void 0 : void 0) {
            results.push(editor.destroy());
          } else {
            results.push(void 0);
          }
        }
        return results;
      }));
    }

    TreeView.prototype.serialize = function() {
      var ref2;
      return {
        directoryExpansionStates: new (function(roots) {
          var j, len, root;
          for (j = 0, len = roots.length; j < len; j++) {
            root = roots[j];
            this[root.directory.path] = root.directory.serializeExpansionState();
          }
          return this;
        })(this.roots),
        deserializer: 'TreeView',
        selectedPath: (ref2 = this.selectedEntry()) != null ? ref2.getPath() : void 0,
        scrollLeft: this.element.scrollLeft,
        scrollTop: this.element.scrollTop,
        width: parseInt(this.element.style.width || 0)
      };
    };

    TreeView.prototype.destroy = function() {
      var j, len, ref2, root;
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        root.directory.destroy();
      }
      this.disposables.dispose();
      this.rootDragAndDrop.dispose();
      return this.emitter.emit('did-destroy');
    };

    TreeView.prototype.onDidDestroy = function(callback) {
      return this.emitter.on('did-destroy', callback);
    };

    TreeView.prototype.getTitle = function() {
      return "Project";
    };

    TreeView.prototype.getURI = function() {
      return TREE_VIEW_URI;
    };

    TreeView.prototype.getPreferredLocation = function() {
      if (atom.config.get('tree-view.showOnRightSide')) {
        return 'right';
      } else {
        return 'left';
      }
    };

    TreeView.prototype.getAllowedLocations = function() {
      return ["left", "right"];
    };

    TreeView.prototype.isPermanentDockItem = function() {
      return true;
    };

    TreeView.prototype.getPreferredWidth = function() {
      var result;
      this.list.style.width = 'min-content';
      result = this.list.offsetWidth;
      this.list.style.width = '';
      return result;
    };

    TreeView.prototype.onDirectoryCreated = function(callback) {
      return this.emitter.on('directory-created', callback);
    };

    TreeView.prototype.onEntryCopied = function(callback) {
      return this.emitter.on('entry-copied', callback);
    };

    TreeView.prototype.onEntryDeleted = function(callback) {
      return this.emitter.on('entry-deleted', callback);
    };

    TreeView.prototype.onEntryMoved = function(callback) {
      return this.emitter.on('entry-moved', callback);
    };

    TreeView.prototype.onFileCreated = function(callback) {
      return this.emitter.on('file-created', callback);
    };

    TreeView.prototype.handleEvents = function() {
      this.element.addEventListener('click', (function(_this) {
        return function(e) {
          if (e.target.classList.contains('entries')) {
            return;
          }
          if (!(e.shiftKey || e.metaKey || e.ctrlKey)) {
            return _this.entryClicked(e);
          }
        };
      })(this));
      this.element.addEventListener('mousedown', (function(_this) {
        return function(e) {
          return _this.onMouseDown(e);
        };
      })(this));
      this.element.addEventListener('dragstart', (function(_this) {
        return function(e) {
          return _this.onDragStart(e);
        };
      })(this));
      this.element.addEventListener('dragenter', (function(_this) {
        return function(e) {
          return _this.onDragEnter(e);
        };
      })(this));
      this.element.addEventListener('dragleave', (function(_this) {
        return function(e) {
          return _this.onDragLeave(e);
        };
      })(this));
      this.element.addEventListener('dragover', (function(_this) {
        return function(e) {
          return _this.onDragOver(e);
        };
      })(this));
      this.element.addEventListener('drop', (function(_this) {
        return function(e) {
          return _this.onDrop(e);
        };
      })(this));
      atom.commands.add(this.element, {
        'core:move-up': this.moveUp.bind(this),
        'core:move-down': this.moveDown.bind(this),
        'core:page-up': (function(_this) {
          return function() {
            return _this.pageUp();
          };
        })(this),
        'core:page-down': (function(_this) {
          return function() {
            return _this.pageDown();
          };
        })(this),
        'core:move-to-top': (function(_this) {
          return function() {
            return _this.scrollToTop();
          };
        })(this),
        'core:move-to-bottom': (function(_this) {
          return function() {
            return _this.scrollToBottom();
          };
        })(this),
        'tree-view:expand-item': (function(_this) {
          return function() {
            return _this.openSelectedEntry({
              pending: true
            }, true);
          };
        })(this),
        'tree-view:recursive-expand-directory': (function(_this) {
          return function() {
            return _this.expandDirectory(true);
          };
        })(this),
        'tree-view:collapse-directory': (function(_this) {
          return function() {
            return _this.collapseDirectory();
          };
        })(this),
        'tree-view:recursive-collapse-directory': (function(_this) {
          return function() {
            return _this.collapseDirectory(true);
          };
        })(this),
        'tree-view:open-selected-entry': (function(_this) {
          return function() {
            return _this.openSelectedEntry();
          };
        })(this),
        'tree-view:open-selected-entry-right': (function(_this) {
          return function() {
            return _this.openSelectedEntryRight();
          };
        })(this),
        'tree-view:open-selected-entry-left': (function(_this) {
          return function() {
            return _this.openSelectedEntryLeft();
          };
        })(this),
        'tree-view:open-selected-entry-up': (function(_this) {
          return function() {
            return _this.openSelectedEntryUp();
          };
        })(this),
        'tree-view:open-selected-entry-down': (function(_this) {
          return function() {
            return _this.openSelectedEntryDown();
          };
        })(this),
        'tree-view:move': (function(_this) {
          return function() {
            return _this.moveSelectedEntry();
          };
        })(this),
        'tree-view:copy': (function(_this) {
          return function() {
            return _this.copySelectedEntries();
          };
        })(this),
        'tree-view:cut': (function(_this) {
          return function() {
            return _this.cutSelectedEntries();
          };
        })(this),
        'tree-view:paste': (function(_this) {
          return function() {
            return _this.pasteEntries();
          };
        })(this),
        'tree-view:copy-full-path': (function(_this) {
          return function() {
            return _this.copySelectedEntryPath(false);
          };
        })(this),
        'tree-view:show-in-file-manager': (function(_this) {
          return function() {
            return _this.showSelectedEntryInFileManager();
          };
        })(this),
        'tree-view:open-in-new-window': (function(_this) {
          return function() {
            return _this.openSelectedEntryInNewWindow();
          };
        })(this),
        'tree-view:copy-project-path': (function(_this) {
          return function() {
            return _this.copySelectedEntryPath(true);
          };
        })(this),
        'tree-view:unfocus': (function(_this) {
          return function() {
            return _this.unfocus();
          };
        })(this),
        'tree-view:toggle-vcs-ignored-files': function() {
          return toggleConfig('tree-view.hideVcsIgnoredFiles');
        },
        'tree-view:toggle-ignored-names': function() {
          return toggleConfig('tree-view.hideIgnoredNames');
        },
        'tree-view:remove-project-folder': (function(_this) {
          return function(e) {
            return _this.removeProjectFolder(e);
          };
        })(this)
      });
      [0, 1, 2, 3, 4, 5, 6, 7, 8].forEach((function(_this) {
        return function(index) {
          return atom.commands.add(_this.element, "tree-view:open-selected-entry-in-pane-" + (index + 1), function() {
            return _this.openSelectedEntryInPane(index);
          });
        };
      })(this));
      this.disposables.add(atom.workspace.getCenter().onDidChangeActivePaneItem((function(_this) {
        return function() {
          _this.selectActiveFile();
          if (atom.config.get('tree-view.autoReveal')) {
            return _this.revealActiveFile(false);
          }
        };
      })(this)));
      this.disposables.add(atom.project.onDidChangePaths((function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.hideVcsIgnoredFiles', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.hideIgnoredNames', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('core.ignoredNames', (function(_this) {
        return function() {
          if (atom.config.get('tree-view.hideIgnoredNames')) {
            return _this.updateRoots();
          }
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.sortFoldersBeforeFiles', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      return this.disposables.add(atom.config.onDidChange('tree-view.squashDirectoryNames', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
    };

    TreeView.prototype.toggle = function() {
      return atom.workspace.toggle(this);
    };

    TreeView.prototype.show = function(focus) {
      return atom.workspace.open(this, {
        searchAllPanes: true,
        activatePane: false,
        activateItem: false
      }).then((function(_this) {
        return function() {
          atom.workspace.paneContainerForURI(_this.getURI()).show();
          if (focus) {
            return _this.focus();
          }
        };
      })(this));
    };

    TreeView.prototype.hide = function() {
      return atom.workspace.hide(this);
    };

    TreeView.prototype.focus = function() {
      return this.element.focus();
    };

    TreeView.prototype.unfocus = function() {
      return atom.workspace.getCenter().activate();
    };

    TreeView.prototype.hasFocus = function() {
      return document.activeElement === this.element;
    };

    TreeView.prototype.toggleFocus = function() {
      if (this.hasFocus()) {
        return this.unfocus();
      } else {
        return this.show(true);
      }
    };

    TreeView.prototype.entryClicked = function(e) {
      var entry, isRecursive;
      if (entry = e.target.closest('.entry')) {
        isRecursive = e.altKey || false;
        this.selectEntry(entry);
        if (entry.classList.contains('directory')) {
          return entry.toggleExpansion(isRecursive);
        } else if (entry.classList.contains('file')) {
          return this.fileViewEntryClicked(e);
        }
      }
    };

    TreeView.prototype.fileViewEntryClicked = function(e) {
      var alwaysOpenExisting, detail, filePath, openPromise, ref2;
      filePath = e.target.closest('.entry').getPath();
      detail = (ref2 = e.detail) != null ? ref2 : 1;
      alwaysOpenExisting = atom.config.get('tree-view.alwaysOpenExisting');
      if (detail === 1) {
        if (atom.config.get('core.allowPendingPaneItems')) {
          openPromise = atom.workspace.open(filePath, {
            pending: true,
            activatePane: false,
            searchAllPanes: alwaysOpenExisting
          });
          this.currentlyOpening.set(filePath, openPromise);
          return openPromise.then((function(_this) {
            return function() {
              return _this.currentlyOpening["delete"](filePath);
            };
          })(this));
        }
      } else if (detail === 2) {
        return this.openAfterPromise(filePath, {
          searchAllPanes: alwaysOpenExisting
        });
      }
    };

    TreeView.prototype.openAfterPromise = function(uri, options) {
      var promise;
      if (promise = this.currentlyOpening.get(uri)) {
        return promise.then(function() {
          return atom.workspace.open(uri, options);
        });
      } else {
        return atom.workspace.open(uri, options);
      }
    };

    TreeView.prototype.loadIgnoredPatterns = function() {
      var error, ignoredName, ignoredNames, j, len, ref2, results;
      this.ignoredPatterns.length = 0;
      if (!atom.config.get('tree-view.hideIgnoredNames')) {
        return;
      }
      if (Minimatch == null) {
        Minimatch = require('minimatch').Minimatch;
      }
      ignoredNames = (ref2 = atom.config.get('core.ignoredNames')) != null ? ref2 : [];
      if (typeof ignoredNames === 'string') {
        ignoredNames = [ignoredNames];
      }
      results = [];
      for (j = 0, len = ignoredNames.length; j < len; j++) {
        ignoredName = ignoredNames[j];
        if (ignoredName) {
          try {
            results.push(this.ignoredPatterns.push(new Minimatch(ignoredName, {
              matchBase: true,
              dot: true
            })));
          } catch (error1) {
            error = error1;
            results.push(atom.notifications.addWarning("Error parsing ignore pattern (" + ignoredName + ")", {
              detail: error.message
            }));
          }
        }
      }
      return results;
    };

    TreeView.prototype.updateRoots = function(expansionStates) {
      var directory, j, key, len, oldExpansionStates, projectPath, ref2, root, stats;
      if (expansionStates == null) {
        expansionStates = {};
      }
      oldExpansionStates = {};
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        oldExpansionStates[root.directory.path] = root.directory.serializeExpansionState();
        root.directory.destroy();
        root.remove();
      }
      this.loadIgnoredPatterns();
      return this.roots = (function() {
        var k, l, len1, len2, ref3, ref4, ref5, ref6, results;
        ref3 = atom.project.getPaths();
        results = [];
        for (k = 0, len1 = ref3.length; k < len1; k++) {
          projectPath = ref3[k];
          stats = fs.lstatSyncNoException(projectPath);
          if (!stats) {
            continue;
          }
          stats = _.pick.apply(_, [stats].concat(slice.call(_.keys(stats))));
          ref4 = ["atime", "birthtime", "ctime", "mtime"];
          for (l = 0, len2 = ref4.length; l < len2; l++) {
            key = ref4[l];
            stats[key] = stats[key].getTime();
          }
          directory = new Directory({
            name: path.basename(projectPath),
            fullPath: projectPath,
            symlink: false,
            isRoot: true,
            expansionState: (ref5 = (ref6 = expansionStates[projectPath]) != null ? ref6 : oldExpansionStates[projectPath]) != null ? ref5 : {
              isExpanded: true
            },
            ignoredPatterns: this.ignoredPatterns,
            useSyncFS: this.useSyncFS,
            stats: stats
          });
          root = new DirectoryView(directory).element;
          this.list.appendChild(root);
          results.push(root);
        }
        return results;
      }).call(this);
    };

    TreeView.prototype.getActivePath = function() {
      var ref2;
      return (ref2 = atom.workspace.getCenter().getActivePaneItem()) != null ? typeof ref2.getPath === "function" ? ref2.getPath() : void 0 : void 0;
    };

    TreeView.prototype.selectActiveFile = function() {
      var activeFilePath;
      if (activeFilePath = this.getActivePath()) {
        return this.selectEntryForPath(activeFilePath);
      }
    };

    TreeView.prototype.revealActiveFile = function(focus) {
      if (_.isEmpty(atom.project.getPaths())) {
        return Promise.resolve();
      }
      return this.show(focus != null ? focus : atom.config.get('tree-view.focusOnReveal')).then((function(_this) {
        return function() {
          var activeFilePath, activePathComponents, currentPath, entry, j, len, pathComponent, ref2, relativePath, results, rootPath;
          if (!(activeFilePath = _this.getActivePath())) {
            return;
          }
          ref2 = atom.project.relativizePath(activeFilePath), rootPath = ref2[0], relativePath = ref2[1];
          if (rootPath == null) {
            return;
          }
          activePathComponents = relativePath.split(path.sep);
          currentPath = rootPath;
          results = [];
          for (j = 0, len = activePathComponents.length; j < len; j++) {
            pathComponent = activePathComponents[j];
            currentPath += path.sep + pathComponent;
            entry = _this.entryForPath(currentPath);
            if (entry.classList.contains('directory')) {
              results.push(entry.expand());
            } else {
              _this.selectEntry(entry);
              results.push(_this.scrollToEntry(entry));
            }
          }
          return results;
        };
      })(this));
    };

    TreeView.prototype.copySelectedEntryPath = function(relativePath) {
      var pathToCopy;
      if (relativePath == null) {
        relativePath = false;
      }
      if (pathToCopy = this.selectedPath) {
        if (relativePath) {
          pathToCopy = atom.project.relativize(pathToCopy);
        }
        return atom.clipboard.write(pathToCopy);
      }
    };

    TreeView.prototype.entryForPath = function(entryPath) {
      var bestMatchEntry, bestMatchLength, entry, entryLength, j, len, ref2, ref3;
      bestMatchEntry = null;
      bestMatchLength = 0;
      ref2 = this.list.querySelectorAll('.entry');
      for (j = 0, len = ref2.length; j < len; j++) {
        entry = ref2[j];
        if (entry.isPathEqual(entryPath)) {
          return entry;
        }
        entryLength = entry.getPath().length;
        if (((ref3 = entry.directory) != null ? ref3.contains(entryPath) : void 0) && entryLength > bestMatchLength) {
          bestMatchEntry = entry;
          bestMatchLength = entryLength;
        }
      }
      return bestMatchEntry;
    };

    TreeView.prototype.selectEntryForPath = function(entryPath) {
      return this.selectEntry(this.entryForPath(entryPath));
    };

    TreeView.prototype.moveDown = function(event) {
      var nextEntry, selectedEntry;
      if (event != null) {
        event.stopImmediatePropagation();
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry != null) {
        if (selectedEntry.classList.contains('directory')) {
          if (this.selectEntry(selectedEntry.entries.children[0])) {
            this.scrollToEntry(this.selectedEntry());
            return;
          }
        }
        if (nextEntry = this.nextEntry(selectedEntry)) {
          this.selectEntry(nextEntry);
        }
      } else {
        this.selectEntry(this.roots[0]);
      }
      return this.scrollToEntry(this.selectedEntry());
    };

    TreeView.prototype.moveUp = function(event) {
      var entries, previousEntry, selectedEntry;
      event.stopImmediatePropagation();
      selectedEntry = this.selectedEntry();
      if (selectedEntry != null) {
        if (previousEntry = this.previousEntry(selectedEntry)) {
          this.selectEntry(previousEntry);
          if (previousEntry.classList.contains('directory')) {
            this.selectEntry(_.last(previousEntry.entries.children));
          }
        } else {
          this.selectEntry(selectedEntry.parentElement.closest('.directory'));
        }
      } else {
        entries = this.list.querySelectorAll('.entry');
        this.selectEntry(entries[entries.length - 1]);
      }
      return this.scrollToEntry(this.selectedEntry());
    };

    TreeView.prototype.nextEntry = function(entry) {
      var currentEntry;
      currentEntry = entry;
      while (currentEntry != null) {
        if (currentEntry.nextSibling != null) {
          currentEntry = currentEntry.nextSibling;
          if (currentEntry.matches('.entry')) {
            return currentEntry;
          }
        } else {
          currentEntry = currentEntry.parentElement.closest('.directory');
        }
      }
      return null;
    };

    TreeView.prototype.previousEntry = function(entry) {
      var currentEntry;
      currentEntry = entry;
      while (currentEntry != null) {
        currentEntry = currentEntry.previousSibling;
        if (currentEntry != null ? currentEntry.matches('.entry') : void 0) {
          return currentEntry;
        }
      }
      return null;
    };

    TreeView.prototype.expandDirectory = function(isRecursive) {
      var selectedEntry;
      if (isRecursive == null) {
        isRecursive = false;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      if (isRecursive === false && selectedEntry.isExpanded) {
        if (selectedEntry.directory.getEntries().length > 0) {
          return this.moveDown();
        }
      } else {
        return selectedEntry.expand(isRecursive);
      }
    };

    TreeView.prototype.collapseDirectory = function(isRecursive) {
      var directory, selectedEntry;
      if (isRecursive == null) {
        isRecursive = false;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      if (directory = selectedEntry.closest('.expanded.directory')) {
        directory.collapse(isRecursive);
        return this.selectEntry(directory);
      }
    };

    TreeView.prototype.openSelectedEntry = function(options, expandDirectory) {
      var selectedEntry;
      if (options == null) {
        options = {};
      }
      if (expandDirectory == null) {
        expandDirectory = false;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      if (selectedEntry.classList.contains('directory')) {
        if (expandDirectory) {
          return this.expandDirectory(false);
        } else {
          return selectedEntry.toggleExpansion();
        }
      } else if (selectedEntry.classList.contains('file')) {
        if (atom.config.get('tree-view.alwaysOpenExisting')) {
          options = Object.assign({
            searchAllPanes: true
          }, options);
        }
        return this.openAfterPromise(selectedEntry.getPath(), options);
      }
    };

    TreeView.prototype.openSelectedEntrySplit = function(orientation, side) {
      var pane, selectedEntry, split;
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      pane = atom.workspace.getCenter().getActivePane();
      if (pane && selectedEntry.classList.contains('file')) {
        if (atom.workspace.getCenter().getActivePaneItem()) {
          split = pane.split(orientation, side);
          return atom.workspace.openURIInPane(selectedEntry.getPath(), split);
        } else {
          return this.openSelectedEntry(true);
        }
      }
    };

    TreeView.prototype.openSelectedEntryRight = function() {
      return this.openSelectedEntrySplit('horizontal', 'after');
    };

    TreeView.prototype.openSelectedEntryLeft = function() {
      return this.openSelectedEntrySplit('horizontal', 'before');
    };

    TreeView.prototype.openSelectedEntryUp = function() {
      return this.openSelectedEntrySplit('vertical', 'before');
    };

    TreeView.prototype.openSelectedEntryDown = function() {
      return this.openSelectedEntrySplit('vertical', 'after');
    };

    TreeView.prototype.openSelectedEntryInPane = function(index) {
      var pane, selectedEntry;
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      pane = atom.workspace.getCenter().getPanes()[index];
      if (pane && selectedEntry.classList.contains('file')) {
        return atom.workspace.openURIInPane(selectedEntry.getPath(), pane);
      }
    };

    TreeView.prototype.moveSelectedEntry = function() {
      var dialog, entry, oldPath;
      if (this.hasFocus()) {
        entry = this.selectedEntry();
        if ((entry == null) || indexOf.call(this.roots, entry) >= 0) {
          return;
        }
        oldPath = entry.getPath();
      } else {
        oldPath = this.getActivePath();
      }
      if (oldPath) {
        dialog = new MoveDialog(oldPath, {
          onMove: (function(_this) {
            return function(arg) {
              var initialPath, newPath;
              initialPath = arg.initialPath, newPath = arg.newPath;
              return _this.emitter.emit('entry-moved', {
                initialPath: initialPath,
                newPath: newPath
              });
            };
          })(this)
        });
        return dialog.attach();
      }
    };

    TreeView.prototype.fileManagerCommandForPath = function(pathToOpen, isFile) {
      var args, command;
      switch (process.platform) {
        case 'darwin':
          return {
            command: 'open',
            label: 'Finder',
            args: ['-R', pathToOpen]
          };
        case 'win32':
          args = ["/select,\"" + pathToOpen + "\""];
          if (process.env.SystemRoot) {
            command = path.join(process.env.SystemRoot, 'explorer.exe');
          } else {
            command = 'explorer.exe';
          }
          return {
            command: command,
            label: 'Explorer',
            args: args
          };
        default:
          if (isFile) {
            pathToOpen = path.dirname(pathToOpen);
          }
          return {
            command: 'xdg-open',
            label: 'File Manager',
            args: [pathToOpen]
          };
      }
    };

    TreeView.prototype.openInFileManager = function(command, args, label, isFile) {
      var errorLines, exit, handleError, showProcess, stderr;
      handleError = function(errorMessage) {
        return atom.notifications.addError("Opening " + (isFile ? 'file' : 'folder') + " in " + label + " failed", {
          detail: errorMessage,
          dismissable: true
        });
      };
      errorLines = [];
      stderr = function(lines) {
        return errorLines.push(lines);
      };
      exit = function(code) {
        var errorMessage, failed;
        failed = code !== 0;
        errorMessage = errorLines.join('\n');
        if (process.platform === 'win32' && code === 1 && !errorMessage) {
          failed = false;
        }
        if (failed) {
          return handleError(errorMessage);
        }
      };
      showProcess = new BufferedProcess({
        command: command,
        args: args,
        stderr: stderr,
        exit: exit
      });
      showProcess.onWillThrowError(function(arg) {
        var error, handle;
        error = arg.error, handle = arg.handle;
        handle();
        return handleError(error != null ? error.message : void 0);
      });
      return showProcess;
    };

    TreeView.prototype.showSelectedEntryInFileManager = function() {
      var args, command, entry, isFile, label, ref2;
      if (!(entry = this.selectedEntry())) {
        return;
      }
      isFile = entry.classList.contains('file');
      ref2 = this.fileManagerCommandForPath(entry.getPath(), isFile), command = ref2.command, args = ref2.args, label = ref2.label;
      return this.openInFileManager(command, args, label, isFile);
    };

    TreeView.prototype.showCurrentFileInFileManager = function() {
      var args, command, editor, label, ref2;
      if (!(editor = atom.workspace.getCenter().getActiveTextEditor())) {
        return;
      }
      if (!editor.getPath()) {
        return;
      }
      ref2 = this.fileManagerCommandForPath(editor.getPath(), true), command = ref2.command, args = ref2.args, label = ref2.label;
      return this.openInFileManager(command, args, label, true);
    };

    TreeView.prototype.openSelectedEntryInNewWindow = function() {
      var pathToOpen, ref2;
      if (pathToOpen = (ref2 = this.selectedEntry()) != null ? ref2.getPath() : void 0) {
        return atom.open({
          pathsToOpen: [pathToOpen],
          newWindow: true
        });
      }
    };

    TreeView.prototype.copySelectedEntry = function() {
      var dialog, entry, oldPath;
      if (this.hasFocus()) {
        entry = this.selectedEntry();
        if (indexOf.call(this.roots, entry) >= 0) {
          return;
        }
        oldPath = entry != null ? entry.getPath() : void 0;
      } else {
        oldPath = this.getActivePath();
      }
      if (!oldPath) {
        return;
      }
      dialog = new CopyDialog(oldPath, {
        onCopy: (function(_this) {
          return function(arg) {
            var initialPath, newPath;
            initialPath = arg.initialPath, newPath = arg.newPath;
            return _this.emitter.emit('entry-copied', {
              initialPath: initialPath,
              newPath: newPath
            });
          };
        })(this)
      });
      return dialog.attach();
    };

    TreeView.prototype.removeSelectedEntries = function() {
      var activePath, j, len, ref2, ref3, root, selectedPaths;
      if (this.hasFocus()) {
        selectedPaths = this.selectedPaths();
      } else if (activePath = this.getActivePath()) {
        selectedPaths = [activePath];
      }
      if (!(selectedPaths && selectedPaths.length > 0)) {
        return;
      }
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        if (ref3 = root.getPath(), indexOf.call(selectedPaths, ref3) >= 0) {
          atom.confirm({
            message: "The root directory '" + root.directory.name + "' can't be removed.",
            buttons: ['OK']
          });
          return;
        }
      }
      return atom.confirm({
        message: "Are you sure you want to delete the selected " + (selectedPaths.length > 1 ? 'items' : 'item') + "?",
        detailedMessage: "You are deleting:\n" + (selectedPaths.join('\n')),
        buttons: {
          "Move to Trash": (function(_this) {
            return function() {
              var failedDeletions, k, len1, repo, selectedPath;
              failedDeletions = [];
              for (k = 0, len1 = selectedPaths.length; k < len1; k++) {
                selectedPath = selectedPaths[k];
                if (shell.moveItemToTrash(selectedPath)) {
                  _this.emitter.emit('entry-deleted', {
                    path: selectedPath
                  });
                } else {
                  failedDeletions.push("" + selectedPath);
                }
                if (repo = repoForPath(selectedPath)) {
                  repo.getPathStatus(selectedPath);
                }
              }
              if (failedDeletions.length > 0) {
                atom.notifications.addError(_this.formatTrashFailureMessage(failedDeletions), {
                  description: _this.formatTrashEnabledMessage(),
                  detail: "" + (failedDeletions.join('\n')),
                  dismissable: true
                });
              }
              if (atom.config.get('tree-view.squashDirectoryNames')) {
                return _this.updateRoots();
              }
            };
          })(this),
          "Cancel": null
        }
      });
    };

    TreeView.prototype.formatTrashFailureMessage = function(failedDeletions) {
      var fileText;
      fileText = failedDeletions.length > 1 ? 'files' : 'file';
      return "The following " + fileText + " couldn't be moved to the trash.";
    };

    TreeView.prototype.formatTrashEnabledMessage = function() {
      switch (process.platform) {
        case 'linux':
          return 'Is `gvfs-trash` installed?';
        case 'darwin':
          return 'Is Trash enabled on the volume where the files are stored?';
        case 'win32':
          return 'Is there a Recycle Bin on the drive where the files are stored?';
      }
    };

    TreeView.prototype.copySelectedEntries = function() {
      var selectedPaths;
      selectedPaths = this.selectedPaths();
      if (!(selectedPaths && selectedPaths.length > 0)) {
        return;
      }
      window.localStorage.removeItem('tree-view:cutPath');
      return window.localStorage['tree-view:copyPath'] = JSON.stringify(selectedPaths);
    };

    TreeView.prototype.cutSelectedEntries = function() {
      var selectedPaths;
      selectedPaths = this.selectedPaths();
      if (!(selectedPaths && selectedPaths.length > 0)) {
        return;
      }
      window.localStorage.removeItem('tree-view:copyPath');
      return window.localStorage['tree-view:cutPath'] = JSON.stringify(selectedPaths);
    };

    TreeView.prototype.pasteEntries = function() {
      var basePath, catchAndShowFileErrors, copiedPaths, cutPaths, extension, fileCounter, filePath, initialPath, initialPathIsDirectory, initialPaths, j, len, newPath, originalNewPath, ref2, results, selectedEntry;
      selectedEntry = this.selectedEntry();
      cutPaths = window.localStorage['tree-view:cutPath'] ? JSON.parse(window.localStorage['tree-view:cutPath']) : null;
      copiedPaths = window.localStorage['tree-view:copyPath'] ? JSON.parse(window.localStorage['tree-view:copyPath']) : null;
      initialPaths = copiedPaths || cutPaths;
      catchAndShowFileErrors = function(operation) {
        var error;
        try {
          return operation();
        } catch (error1) {
          error = error1;
          return atom.notifications.addWarning("Unable to paste paths: " + initialPaths, {
            detail: error.message
          });
        }
      };
      ref2 = initialPaths != null ? initialPaths : [];
      results = [];
      for (j = 0, len = ref2.length; j < len; j++) {
        initialPath = ref2[j];
        initialPathIsDirectory = fs.isDirectorySync(initialPath);
        if (selectedEntry && initialPath && fs.existsSync(initialPath)) {
          basePath = selectedEntry.getPath();
          if (selectedEntry.classList.contains('file')) {
            basePath = path.dirname(basePath);
          }
          newPath = path.join(basePath, path.basename(initialPath));
          if (copiedPaths) {
            fileCounter = 0;
            originalNewPath = newPath;
            while (fs.existsSync(newPath)) {
              if (initialPathIsDirectory) {
                newPath = "" + originalNewPath + fileCounter;
              } else {
                extension = getFullExtension(originalNewPath);
                filePath = path.join(path.dirname(originalNewPath), path.basename(originalNewPath, extension));
                newPath = "" + filePath + fileCounter + extension;
              }
              fileCounter += 1;
            }
            if (fs.isDirectorySync(initialPath)) {
              results.push(catchAndShowFileErrors((function(_this) {
                return function() {
                  fs.copySync(initialPath, newPath);
                  return _this.emitter.emit('entry-copied', {
                    initialPath: initialPath,
                    newPath: newPath
                  });
                };
              })(this)));
            } else {
              results.push(catchAndShowFileErrors((function(_this) {
                return function() {
                  fs.writeFileSync(newPath, fs.readFileSync(initialPath));
                  return _this.emitter.emit('entry-copied', {
                    initialPath: initialPath,
                    newPath: newPath
                  });
                };
              })(this)));
            }
          } else if (cutPaths) {
            if (!(fs.existsSync(newPath) || newPath.startsWith(initialPath))) {
              results.push(catchAndShowFileErrors((function(_this) {
                return function() {
                  fs.moveSync(initialPath, newPath);
                  return _this.emitter.emit('entry-moved', {
                    initialPath: initialPath,
                    newPath: newPath
                  });
                };
              })(this)));
            } else {
              results.push(void 0);
            }
          } else {
            results.push(void 0);
          }
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    TreeView.prototype.add = function(isCreatingFile) {
      var dialog, ref2, ref3, selectedEntry, selectedPath;
      selectedEntry = (ref2 = this.selectedEntry()) != null ? ref2 : this.roots[0];
      selectedPath = (ref3 = selectedEntry != null ? selectedEntry.getPath() : void 0) != null ? ref3 : '';
      dialog = new AddDialog(selectedPath, isCreatingFile);
      dialog.onDidCreateDirectory((function(_this) {
        return function(createdPath) {
          var ref4;
          if ((ref4 = _this.entryForPath(createdPath)) != null) {
            ref4.reload();
          }
          _this.selectEntryForPath(createdPath);
          if (atom.config.get('tree-view.squashDirectoryNames')) {
            _this.updateRoots();
          }
          return _this.emitter.emit('directory-created', {
            path: createdPath
          });
        };
      })(this));
      dialog.onDidCreateFile((function(_this) {
        return function(createdPath) {
          atom.workspace.open(createdPath);
          if (atom.config.get('tree-view.squashDirectoryNames')) {
            _this.updateRoots();
          }
          return _this.emitter.emit('file-created', {
            path: createdPath
          });
        };
      })(this));
      return dialog.attach();
    };

    TreeView.prototype.removeProjectFolder = function(e) {
      var pathToRemove, ref2, ref3;
      pathToRemove = (ref2 = e.target.closest(".project-root > .header")) != null ? (ref3 = ref2.querySelector(".name")) != null ? ref3.dataset.path : void 0 : void 0;
      if (atom.project.removePath != null) {
        if (pathToRemove != null) {
          return atom.project.removePath(pathToRemove);
        }
      }
    };

    TreeView.prototype.selectedEntry = function() {
      return this.list.querySelector('.selected');
    };

    TreeView.prototype.selectEntry = function(entry) {
      var selectedEntries;
      if (entry == null) {
        return;
      }
      this.selectedPath = entry.getPath();
      selectedEntries = this.getSelectedEntries();
      if (selectedEntries.length > 1 || selectedEntries[0] !== entry) {
        this.deselect(selectedEntries);
        entry.classList.add('selected');
      }
      return entry;
    };

    TreeView.prototype.getSelectedEntries = function() {
      return this.list.querySelectorAll('.selected');
    };

    TreeView.prototype.deselect = function(elementsToDeselect) {
      var j, len, selected;
      if (elementsToDeselect == null) {
        elementsToDeselect = this.getSelectedEntries();
      }
      for (j = 0, len = elementsToDeselect.length; j < len; j++) {
        selected = elementsToDeselect[j];
        selected.classList.remove('selected');
      }
      return void 0;
    };

    TreeView.prototype.scrollTop = function(top) {
      if (top != null) {
        return this.element.scrollTop = top;
      } else {
        return this.element.scrollTop;
      }
    };

    TreeView.prototype.scrollBottom = function(bottom) {
      if (bottom != null) {
        return this.element.scrollTop = bottom - this.element.offsetHeight;
      } else {
        return this.element.scrollTop + this.element.offsetHeight;
      }
    };

    TreeView.prototype.scrollToEntry = function(entry) {
      var element;
      element = (entry != null ? entry.classList.contains('directory') : void 0) ? entry.header : entry;
      return element != null ? element.scrollIntoViewIfNeeded(true) : void 0;
    };

    TreeView.prototype.scrollToBottom = function() {
      var lastEntry;
      if (lastEntry = _.last(this.list.querySelectorAll('.entry'))) {
        this.selectEntry(lastEntry);
        return this.scrollToEntry(lastEntry);
      }
    };

    TreeView.prototype.scrollToTop = function() {
      if (this.roots[0] != null) {
        this.selectEntry(this.roots[0]);
      }
      return this.element.scrollTop = 0;
    };

    TreeView.prototype.pageUp = function() {
      return this.element.scrollTop -= this.element.offsetHeight;
    };

    TreeView.prototype.pageDown = function() {
      return this.element.scrollTop += this.element.offsetHeight;
    };

    TreeView.prototype.moveEntry = function(initialPath, newDirectoryPath) {
      var entryName, error, newPath, repo;
      if (initialPath === newDirectoryPath) {
        return;
      }
      entryName = path.basename(initialPath);
      newPath = (newDirectoryPath + "/" + entryName).replace(/\s+$/, '');
      try {
        if (!fs.existsSync(newDirectoryPath)) {
          fs.makeTreeSync(newDirectoryPath);
        }
        fs.moveSync(initialPath, newPath);
        this.emitter.emit('entry-moved', {
          initialPath: initialPath,
          newPath: newPath
        });
        if (repo = repoForPath(newPath)) {
          repo.getPathStatus(initialPath);
          return repo.getPathStatus(newPath);
        }
      } catch (error1) {
        error = error1;
        return atom.notifications.addWarning("Failed to move entry " + initialPath + " to " + newDirectoryPath, {
          detail: error.message
        });
      }
    };

    TreeView.prototype.onStylesheetsChanged = function() {
      if (!this.isVisible()) {
        return;
      }
      this.element.style.display = 'none';
      this.element.offsetWidth;
      return this.element.style.display = '';
    };

    TreeView.prototype.onMouseDown = function(e) {
      var entryToSelect;
      if (entryToSelect = e.target.closest('.entry')) {
        e.stopPropagation();
        if (this.multiSelectEnabled() && e.target.classList.contains('selected') && (e.button === 2 || e.ctrlKey && process.platform === 'darwin')) {
          return;
        }
        if (e.shiftKey) {
          this.selectContinuousEntries(entryToSelect);
          return this.showMultiSelectMenu();
        } else if (e.metaKey || (e.ctrlKey && process.platform !== 'darwin')) {
          this.selectMultipleEntries(entryToSelect);
          if (this.selectedPaths().length > 1) {
            return this.showMultiSelectMenu();
          }
        } else {
          this.selectEntry(entryToSelect);
          return this.showFullMenu();
        }
      }
    };

    TreeView.prototype.selectedPaths = function() {
      var entry, j, len, ref2, results;
      ref2 = this.getSelectedEntries();
      results = [];
      for (j = 0, len = ref2.length; j < len; j++) {
        entry = ref2[j];
        results.push(entry.getPath());
      }
      return results;
    };

    TreeView.prototype.selectContinuousEntries = function(entry) {
      var currentSelectedEntry, element, elements, entries, entryIndex, i, j, len, parentContainer, selectedIndex;
      currentSelectedEntry = this.selectedEntry();
      parentContainer = entry.parentElement;
      if (parentContainer.contains(currentSelectedEntry)) {
        entries = Array.from(parentContainer.querySelectorAll('.entry'));
        entryIndex = entries.indexOf(entry);
        selectedIndex = entries.indexOf(currentSelectedEntry);
        elements = (function() {
          var j, ref2, ref3, results;
          results = [];
          for (i = j = ref2 = entryIndex, ref3 = selectedIndex; ref2 <= ref3 ? j <= ref3 : j >= ref3; i = ref2 <= ref3 ? ++j : --j) {
            results.push(entries[i]);
          }
          return results;
        })();
        this.deselect();
        for (j = 0, len = elements.length; j < len; j++) {
          element = elements[j];
          element.classList.add('selected');
        }
      }
      return elements;
    };

    TreeView.prototype.selectMultipleEntries = function(entry) {
      if (entry != null) {
        entry.classList.toggle('selected');
      }
      return entry;
    };

    TreeView.prototype.showFullMenu = function() {
      this.list.classList.remove('multi-select');
      return this.list.classList.add('full-menu');
    };

    TreeView.prototype.showMultiSelectMenu = function() {
      this.list.classList.remove('full-menu');
      return this.list.classList.add('multi-select');
    };

    TreeView.prototype.multiSelectEnabled = function() {
      return this.list.classList.contains('multi-select');
    };

    TreeView.prototype.onDragEnter = function(e) {
      var entry, header;
      if (header = e.target.closest('.entry.directory > .header')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.stopPropagation();
        entry = header.parentNode;
        if (!this.dragEventCounts.get(entry)) {
          this.dragEventCounts.set(entry, 0);
        }
        if (this.dragEventCounts.get(entry) === 0) {
          entry.classList.add('selected');
        }
        return this.dragEventCounts.set(entry, this.dragEventCounts.get(entry) + 1);
      }
    };

    TreeView.prototype.onDragLeave = function(e) {
      var entry, header;
      if (header = e.target.closest('.entry.directory > .header')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.stopPropagation();
        entry = header.parentNode;
        this.dragEventCounts.set(entry, this.dragEventCounts.get(entry) - 1);
        if (this.dragEventCounts.get(entry) === 0) {
          return entry.classList.remove('selected');
        }
      }
    };

    TreeView.prototype.onDragStart = function(e) {
      var entry, fileNameElement, initialPath, key, ref2, target, value;
      if (entry = e.target.closest('.entry')) {
        e.stopPropagation();
        if (this.rootDragAndDrop.canDragStart(e)) {
          return this.rootDragAndDrop.onDragStart(e);
        }
        target = entry.querySelector(".name");
        initialPath = target.dataset.path;
        fileNameElement = target.cloneNode(true);
        ref2 = getStyleObject(target);
        for (key in ref2) {
          value = ref2[key];
          fileNameElement.style[key] = value;
        }
        fileNameElement.style.position = 'absolute';
        fileNameElement.style.top = 0;
        fileNameElement.style.left = 0;
        fileNameElement.style.willChange = 'transform';
        document.body.appendChild(fileNameElement);
        e.dataTransfer.effectAllowed = "move";
        e.dataTransfer.setDragImage(fileNameElement, 0, 0);
        e.dataTransfer.setData("initialPath", initialPath);
        return window.requestAnimationFrame(function() {
          return fileNameElement.remove();
        });
      }
    };

    TreeView.prototype.onDragOver = function(e) {
      var entry;
      if (entry = e.target.closest('.entry')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (this.dragEventCounts.get(entry) > 0 && !entry.classList.contains('selected')) {
          return entry.classList.add('selected');
        }
      }
    };

    TreeView.prototype.onDrop = function(e) {
      var entry, file, initialPath, j, len, newDirectoryPath, ref2, ref3, results;
      if (entry = e.target.closest('.entry')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.preventDefault();
        e.stopPropagation();
        entry.classList.remove('selected');
        if (!entry.classList.contains('directory')) {
          return;
        }
        newDirectoryPath = (ref2 = entry.querySelector('.name')) != null ? ref2.dataset.path : void 0;
        if (!newDirectoryPath) {
          return false;
        }
        initialPath = e.dataTransfer.getData("initialPath");
        if (initialPath) {
          return this.moveEntry(initialPath, newDirectoryPath);
        } else {
          ref3 = e.dataTransfer.files;
          results = [];
          for (j = 0, len = ref3.length; j < len; j++) {
            file = ref3[j];
            results.push(this.moveEntry(file.path, newDirectoryPath));
          }
          return results;
        }
      }
    };

    TreeView.prototype.isVisible = function() {
      return this.element.offsetWidth !== 0 || this.element.offsetHeight !== 0;
    };

    return TreeView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
