(function() {
  var CompositeDisposable, Directory, DirectoryView, FileView, repoForPath;

  CompositeDisposable = require('event-kit').CompositeDisposable;

  Directory = require('./directory');

  FileView = require('./file-view');

  repoForPath = require('./helpers').repoForPath;

  module.exports = DirectoryView = (function() {
    function DirectoryView(directory) {
      var iconClass, ref, squashedDirectoryNameNode;
      this.directory = directory;
      this.subscriptions = new CompositeDisposable();
      this.subscriptions.add(this.directory.onDidDestroy((function(_this) {
        return function() {
          return _this.subscriptions.dispose();
        };
      })(this)));
      this.subscribeToDirectory();
      this.element = document.createElement('li');
      this.element.setAttribute('is', 'tree-view-directory');
      this.element.classList.add('directory', 'entry', 'list-nested-item', 'collapsed');
      this.header = document.createElement('div');
      this.header.classList.add('header', 'list-item');
      this.directoryName = document.createElement('span');
      this.directoryName.classList.add('name', 'icon');
      this.entries = document.createElement('ol');
      this.entries.classList.add('entries', 'list-tree');
      if (this.directory.symlink) {
        iconClass = 'icon-file-symlink-directory';
      } else {
        iconClass = 'icon-file-directory';
        if (this.directory.isRoot) {
          if ((ref = repoForPath(this.directory.path)) != null ? ref.isProjectAtRoot() : void 0) {
            iconClass = 'icon-repo';
          }
        } else {
          if (this.directory.submodule) {
            iconClass = 'icon-file-submodule';
          }
        }
      }
      this.directoryName.classList.add(iconClass);
      this.directoryName.dataset.path = this.directory.path;
      if (this.directory.squashedNames != null) {
        this.directoryName.dataset.name = this.directory.squashedNames.join('');
        this.directoryName.title = this.directory.squashedNames.join('');
        squashedDirectoryNameNode = document.createElement('span');
        squashedDirectoryNameNode.classList.add('squashed-dir');
        squashedDirectoryNameNode.textContent = this.directory.squashedNames[0];
        this.directoryName.appendChild(squashedDirectoryNameNode);
        this.directoryName.appendChild(document.createTextNode(this.directory.squashedNames[1]));
      } else {
        this.directoryName.dataset.name = this.directory.name;
        this.directoryName.title = this.directory.name;
        this.directoryName.textContent = this.directory.name;
      }
      this.element.appendChild(this.header);
      this.header.appendChild(this.directoryName);
      this.element.appendChild(this.entries);
      if (this.directory.isRoot) {
        this.element.classList.add('project-root');
        this.header.classList.add('project-root-header');
      } else {
        this.element.draggable = true;
        this.subscriptions.add(this.directory.onDidStatusChange((function(_this) {
          return function() {
            return _this.updateStatus();
          };
        })(this)));
        this.updateStatus();
      }
      if (this.directory.expansionState.isExpanded) {
        this.expand();
      }
      this.element.collapse = this.collapse.bind(this);
      this.element.expand = this.expand.bind(this);
      this.element.toggleExpansion = this.toggleExpansion.bind(this);
      this.element.reload = this.reload.bind(this);
      this.element.isExpanded = this.isExpanded;
      this.element.updateStatus = this.updateStatus.bind(this);
      this.element.isPathEqual = this.isPathEqual.bind(this);
      this.element.getPath = this.getPath.bind(this);
      this.element.directory = this.directory;
      this.element.header = this.header;
      this.element.entries = this.entries;
      this.element.directoryName = this.directoryName;
    }

    DirectoryView.prototype.updateStatus = function() {
      this.element.classList.remove('status-ignored', 'status-modified', 'status-added');
      if (this.directory.status != null) {
        return this.element.classList.add("status-" + this.directory.status);
      }
    };

    DirectoryView.prototype.subscribeToDirectory = function() {
      return this.subscriptions.add(this.directory.onDidAddEntries((function(_this) {
        return function(addedEntries) {
          var entry, i, insertionIndex, len, numberOfEntries, results, view;
          if (!_this.isExpanded) {
            return;
          }
          numberOfEntries = _this.entries.children.length;
          results = [];
          for (i = 0, len = addedEntries.length; i < len; i++) {
            entry = addedEntries[i];
            view = _this.createViewForEntry(entry);
            insertionIndex = entry.indexInParentDirectory;
            if (insertionIndex < numberOfEntries) {
              _this.entries.insertBefore(view.element, _this.entries.children[insertionIndex]);
            } else {
              _this.entries.appendChild(view.element);
            }
            results.push(numberOfEntries++);
          }
          return results;
        };
      })(this)));
    };

    DirectoryView.prototype.getPath = function() {
      return this.directory.path;
    };

    DirectoryView.prototype.isPathEqual = function(pathToCompare) {
      return this.directory.isPathEqual(pathToCompare);
    };

    DirectoryView.prototype.createViewForEntry = function(entry) {
      var subscription, view;
      if (entry instanceof Directory) {
        view = new DirectoryView(entry);
      } else {
        view = new FileView(entry);
      }
      subscription = this.directory.onDidRemoveEntries(function(removedEntries) {
        var removedEntry, removedName, results;
        results = [];
        for (removedName in removedEntries) {
          removedEntry = removedEntries[removedName];
          if (!(entry === removedEntry)) {
            continue;
          }
          view.element.remove();
          subscription.dispose();
          break;
        }
        return results;
      });
      this.subscriptions.add(subscription);
      return view;
    };

    DirectoryView.prototype.reload = function() {
      if (this.isExpanded) {
        return this.directory.reload();
      }
    };

    DirectoryView.prototype.toggleExpansion = function(isRecursive) {
      if (isRecursive == null) {
        isRecursive = false;
      }
      if (this.isExpanded) {
        return this.collapse(isRecursive);
      } else {
        return this.expand(isRecursive);
      }
    };

    DirectoryView.prototype.expand = function(isRecursive) {
      var entry, i, len, ref;
      if (isRecursive == null) {
        isRecursive = false;
      }
      if (!this.isExpanded) {
        this.isExpanded = true;
        this.element.isExpanded = this.isExpanded;
        this.element.classList.add('expanded');
        this.element.classList.remove('collapsed');
        this.directory.expand();
      }
      if (isRecursive) {
        ref = this.entries.children;
        for (i = 0, len = ref.length; i < len; i++) {
          entry = ref[i];
          if (entry.classList.contains('directory')) {
            entry.expand(true);
          }
        }
      }
      return false;
    };

    DirectoryView.prototype.collapse = function(isRecursive) {
      var entry, i, len, ref;
      if (isRecursive == null) {
        isRecursive = false;
      }
      this.isExpanded = false;
      this.element.isExpanded = false;
      if (isRecursive) {
        ref = this.entries.children;
        for (i = 0, len = ref.length; i < len; i++) {
          entry = ref[i];
          if (entry.isExpanded) {
            entry.collapse(true);
          }
        }
      }
      this.element.classList.remove('expanded');
      this.element.classList.add('collapsed');
      this.directory.collapse();
      return this.entries.innerHTML = '';
    };

    return DirectoryView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
