(function() {
  var CompositeDisposable, CorrectionsView, SpellCheckTask, SpellCheckView, _,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  _ = require('underscore-plus');

  CompositeDisposable = require('atom').CompositeDisposable;

  SpellCheckTask = require('./spell-check-task');

  CorrectionsView = null;

  module.exports = SpellCheckView = (function() {
    function SpellCheckView(editor, task, spellCheckModule, getInstance) {
      this.editor = editor;
      this.task = task;
      this.spellCheckModule = spellCheckModule;
      this.getInstance = getInstance;
      this.makeCorrection = bind(this.makeCorrection, this);
      this.addContextMenuEntries = bind(this.addContextMenuEntries, this);
      this.disposables = new CompositeDisposable;
      this.initializeMarkerLayer();
      this.taskWrapper = new SpellCheckTask(this.task);
      this.correctMisspellingCommand = atom.commands.add(atom.views.getView(this.editor), 'spell-check:correct-misspelling', (function(_this) {
        return function() {
          var marker, ref;
          if (marker = _this.markerLayer.findMarkers({
            containsBufferPosition: _this.editor.getCursorBufferPosition()
          })[0]) {
            if (CorrectionsView == null) {
              CorrectionsView = require('./corrections-view');
            }
            if ((ref = _this.correctionsView) != null) {
              ref.destroy();
            }
            _this.correctionsView = new CorrectionsView(_this.editor, _this.getCorrections(marker), marker, _this, _this.updateMisspellings);
            return _this.correctionsView.attach();
          }
        };
      })(this));
      atom.views.getView(this.editor).addEventListener('contextmenu', this.addContextMenuEntries);
      this.taskWrapper.onDidSpellCheck((function(_this) {
        return function(misspellings) {
          _this.destroyMarkers();
          if (_this.buffer != null) {
            return _this.addMarkers(misspellings);
          }
        };
      })(this));
      this.disposables.add(this.editor.onDidChangePath((function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.disposables.add(this.editor.onDidChangeGrammar((function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('editor.fontSize', (function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('spell-check.grammars', (function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.subscribeToBuffer();
      this.disposables.add(this.editor.onDidDestroy(this.destroy.bind(this)));
    }

    SpellCheckView.prototype.initializeMarkerLayer = function() {
      this.markerLayer = this.editor.addMarkerLayer({
        maintainHistory: false
      });
      return this.markerLayerDecoration = this.editor.decorateMarkerLayer(this.markerLayer, {
        type: 'highlight',
        "class": 'spell-check-misspelling',
        deprecatedRegionClass: 'misspelling'
      });
    };

    SpellCheckView.prototype.destroy = function() {
      var ref;
      this.unsubscribeFromBuffer();
      this.disposables.dispose();
      this.taskWrapper.terminate();
      this.markerLayer.destroy();
      this.markerLayerDecoration.destroy();
      this.correctMisspellingCommand.dispose();
      if ((ref = this.correctionsView) != null) {
        ref.destroy();
      }
      return this.clearContextMenuEntries();
    };

    SpellCheckView.prototype.unsubscribeFromBuffer = function() {
      this.destroyMarkers();
      if (this.buffer != null) {
        this.bufferDisposable.dispose();
        return this.buffer = null;
      }
    };

    SpellCheckView.prototype.subscribeToBuffer = function() {
      this.unsubscribeFromBuffer();
      if (this.spellCheckCurrentGrammar()) {
        this.buffer = this.editor.getBuffer();
        this.bufferDisposable = this.buffer.onDidStopChanging((function(_this) {
          return function() {
            return _this.updateMisspellings();
          };
        })(this));
        return this.updateMisspellings();
      }
    };

    SpellCheckView.prototype.spellCheckCurrentGrammar = function() {
      var grammar;
      grammar = this.editor.getGrammar().scopeName;
      return _.contains(atom.config.get('spell-check.grammars'), grammar);
    };

    SpellCheckView.prototype.destroyMarkers = function() {
      this.markerLayer.destroy();
      this.markerLayerDecoration.destroy();
      return this.initializeMarkerLayer();
    };

    SpellCheckView.prototype.addMarkers = function(misspellings) {
      var i, len, misspelling, results;
      results = [];
      for (i = 0, len = misspellings.length; i < len; i++) {
        misspelling = misspellings[i];
        results.push(this.markerLayer.markBufferRange(misspelling, {
          invalidate: 'touch'
        }));
      }
      return results;
    };

    SpellCheckView.prototype.updateMisspellings = function() {
      var error, ref;
      try {
        return this.taskWrapper.start(this.editor.buffer);
      } catch (error1) {
        error = error1;
        return console.warn('Error starting spell check task', (ref = error.stack) != null ? ref : error);
      }
    };

    SpellCheckView.prototype.getCorrections = function(marker) {
      var args, instance, misspelling, projectPath, ref, ref1, ref2, relativePath;
      projectPath = null;
      relativePath = null;
      if ((ref = this.buffer) != null ? (ref1 = ref.file) != null ? ref1.path : void 0 : void 0) {
        ref2 = atom.project.relativizePath(this.buffer.file.path), projectPath = ref2[0], relativePath = ref2[1];
      }
      args = {
        projectPath: projectPath,
        relativePath: relativePath
      };
      instance = this.getInstance();
      misspelling = this.editor.getTextInBufferRange(marker.getBufferRange());
      return instance.suggest(args, misspelling);
    };

    SpellCheckView.prototype.addContextMenuEntries = function(mouseEvent) {
      var commandName, contextMenuEntry, correction, correctionIndex, corrections, currentBufferPosition, currentScreenPosition, i, len, marker;
      this.clearContextMenuEntries();
      currentScreenPosition = atom.views.getView(this.editor).component.screenPositionForMouseEvent(mouseEvent);
      currentBufferPosition = this.editor.bufferPositionForScreenPosition(currentScreenPosition);
      if (marker = this.markerLayer.findMarkers({
        containsBufferPosition: currentBufferPosition
      })[0]) {
        corrections = this.getCorrections(marker);
        if (corrections.length > 0) {
          this.spellCheckModule.contextMenuEntries.push({
            menuItem: atom.contextMenu.add({
              'atom-text-editor': [
                {
                  type: 'separator'
                }
              ]
            })
          });
          correctionIndex = 0;
          for (i = 0, len = corrections.length; i < len; i++) {
            correction = corrections[i];
            contextMenuEntry = {};
            commandName = 'spell-check:correct-misspelling-' + correctionIndex;
            contextMenuEntry.command = (function(_this) {
              return function(correction, contextMenuEntry) {
                return atom.commands.add(atom.views.getView(_this.editor), commandName, function() {
                  _this.makeCorrection(correction, marker);
                  return _this.clearContextMenuEntries();
                });
              };
            })(this)(correction, contextMenuEntry);
            contextMenuEntry.menuItem = atom.contextMenu.add({
              'atom-text-editor': [
                {
                  label: correction.label,
                  command: commandName
                }
              ]
            });
            this.spellCheckModule.contextMenuEntries.push(contextMenuEntry);
            correctionIndex++;
          }
          return this.spellCheckModule.contextMenuEntries.push({
            menuItem: atom.contextMenu.add({
              'atom-text-editor': [
                {
                  type: 'separator'
                }
              ]
            })
          });
        }
      }
    };

    SpellCheckView.prototype.makeCorrection = function(correction, marker) {
      var args, projectPath, ref, ref1, ref2, relativePath;
      if (correction.isSuggestion) {
        this.editor.setSelectedBufferRange(marker.getBufferRange());
        return this.editor.insertText(correction.suggestion);
      } else {
        projectPath = null;
        relativePath = null;
        if ((ref = this.editor.buffer) != null ? (ref1 = ref.file) != null ? ref1.path : void 0 : void 0) {
          ref2 = atom.project.relativizePath(this.editor.buffer.file.path), projectPath = ref2[0], relativePath = ref2[1];
        }
        args = {
          id: this.id,
          projectPath: projectPath,
          relativePath: relativePath
        };
        correction.plugin.add(args, correction);
        return this.updateMisspellings.bind(this)();
      }
    };

    SpellCheckView.prototype.clearContextMenuEntries = function() {
      var entry, i, len, ref, ref1, ref2;
      ref = this.spellCheckModule.contextMenuEntries;
      for (i = 0, len = ref.length; i < len; i++) {
        entry = ref[i];
        if ((ref1 = entry.command) != null) {
          ref1.dispose();
        }
        if ((ref2 = entry.menuItem) != null) {
          ref2.dispose();
        }
      }
      return this.spellCheckModule.contextMenuEntries = [];
    };

    return SpellCheckView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
