'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _helpers = require('./helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _asyncToGenerator(fn) { return function () { var gen = fn.apply(this, arguments); return new Promise(function (resolve, reject) { function step(key, arg) { try { var info = gen[key](arg); var value = info.value; } catch (error) { reject(error); return; } if (info.done) { resolve(value); } else { return Promise.resolve(value).then(function (value) { step("next", value); }, function (err) { step("throw", err); }); } } return step("next"); }); }; } // Measure elapsed durations from specific beginning points.

// The maximum number of marks within a single DurationSet. A DurationSet will be automatically finished if this many
// marks are recorded.
const MAXIMUM_MARKS = 100;

// Flush all non-active DurationSets to disk each time that this many marks have been accumulated.
const PERSIST_INTERVAL = 1000;

// A sequence of durations measured from a fixed beginning point.
let DurationSet = class DurationSet {
  constructor(name) {
    this.name = name;
    this.startTimestamp = performance.now();
    this.marks = [];
    this.markCount = 0;

    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%cbegin %c%s:begin', 'font-weight: bold', 'font-weight: normal; font-style: italic; color: blue', this.name);
    }

    if (atom.config.get('github.performanceToProfile')) {
      // eslint-disable-next-line no-console
      console.profile(this.name);
    }
  }

  mark(eventName) {
    const duration = performance.now() - this.startTimestamp;

    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%cmark %c%s:%s %c%dms', 'font-weight: bold', 'font-weight: normal; font-style: italic; color: blue', this.name, eventName, 'font-style: normal; color: black', duration);
    }

    if (atom.config.get('github.performanceToDirectory') !== '') {
      this.marks.push({ eventName, duration });
    }

    this.markCount++;
    if (this.markCount >= MAXIMUM_MARKS) {
      this.finish();
    }
  }

  finish() {
    this.mark('finish');

    if (atom.config.get('github.performanceToProfile')) {
      // eslint-disable-next-line no-console
      console.profileEnd(this.name);
    }
  }

  serialize() {
    return {
      name: this.name,
      markers: this.marks
    };
  }

  getCount() {
    return this.marks.length;
  }
};


let durationSets = [];
let totalMarkCount = 0;
const activeSets = new Map();

function shouldCapture(seriesName, eventName) {
  const anyActive = ['Console', 'Directory', 'Profile'].some(kind => {
    const value = atom.config.get(`github.performanceTo${kind}`);
    return value !== '' && value !== false;
  });
  if (!anyActive) {
    return false;
  }

  const mask = new RegExp(atom.config.get('github.performanceMask'));
  if (!mask.test(`${seriesName}:${eventName}`)) {
    return false;
  }

  return true;
}

const yardstick = {
  save() {
    return _asyncToGenerator(function* () {
      const destDir = atom.config.get('github.performanceToDirectory');
      if (destDir === '' || destDir === undefined || destDir === null) {
        return;
      }
      const fileName = _path2.default.join(destDir, `performance-${Date.now()}.json`);

      yield new Promise(function (resolve, reject) {
        _fsExtra2.default.ensureDir(destDir, function (err) {
          return err ? reject(err) : resolve();
        });
      });

      const payload = JSON.stringify(durationSets.map(function (set) {
        return set.serialize();
      }));
      yield (0, _helpers.writeFile)(fileName, payload);

      if (atom.config.get('github.performanceToConsole')) {
        // eslint-disable-next-line no-console
        console.log('%csaved %c%d series to %s', 'font-weight: bold', 'font-weight: normal; color: black', durationSets.length, fileName);
      }

      durationSets = [];
    })();
  },

  begin(seriesName) {
    if (!shouldCapture(seriesName, 'begin')) {
      return;
    }

    const ds = new DurationSet(seriesName);
    activeSets.set(seriesName, ds);
  },

  mark(seriesName, eventName) {
    if (seriesName instanceof Array) {
      for (let i = 0; i < seriesName.length; i++) {
        this.mark(seriesName[i], eventName);
      }
      return;
    }

    if (!shouldCapture(seriesName, eventName)) {
      return;
    }

    const ds = activeSets.get(seriesName);
    if (ds === undefined) {
      return;
    }
    ds.mark(eventName);
  },

  finish(seriesName) {
    if (seriesName instanceof Array) {
      for (let i = 0; i < seriesName.length; i++) {
        this.finish(seriesName[i]);
      }
      return;
    }

    if (!shouldCapture(seriesName, 'finish')) {
      return;
    }

    const ds = activeSets.get(seriesName);
    if (ds === undefined) {
      return;
    }
    ds.finish();

    durationSets.push(ds);
    activeSets.delete(seriesName);

    totalMarkCount += ds.getCount();
    if (totalMarkCount >= PERSIST_INTERVAL) {
      totalMarkCount = 0;
      this.save();
    }
  },

  flush() {
    var _this = this;

    return _asyncToGenerator(function* () {
      durationSets.push(...activeSets.values());
      activeSets.clear();
      yield _this.save();
    })();
  }
};

exports.default = yardstick;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInlhcmRzdGljay5qcyJdLCJuYW1lcyI6WyJNQVhJTVVNX01BUktTIiwiUEVSU0lTVF9JTlRFUlZBTCIsIkR1cmF0aW9uU2V0IiwiY29uc3RydWN0b3IiLCJuYW1lIiwic3RhcnRUaW1lc3RhbXAiLCJwZXJmb3JtYW5jZSIsIm5vdyIsIm1hcmtzIiwibWFya0NvdW50IiwiYXRvbSIsImNvbmZpZyIsImdldCIsImNvbnNvbGUiLCJsb2ciLCJwcm9maWxlIiwibWFyayIsImV2ZW50TmFtZSIsImR1cmF0aW9uIiwicHVzaCIsImZpbmlzaCIsInByb2ZpbGVFbmQiLCJzZXJpYWxpemUiLCJtYXJrZXJzIiwiZ2V0Q291bnQiLCJsZW5ndGgiLCJkdXJhdGlvblNldHMiLCJ0b3RhbE1hcmtDb3VudCIsImFjdGl2ZVNldHMiLCJNYXAiLCJzaG91bGRDYXB0dXJlIiwic2VyaWVzTmFtZSIsImFueUFjdGl2ZSIsInNvbWUiLCJraW5kIiwidmFsdWUiLCJtYXNrIiwiUmVnRXhwIiwidGVzdCIsInlhcmRzdGljayIsInNhdmUiLCJkZXN0RGlyIiwidW5kZWZpbmVkIiwiZmlsZU5hbWUiLCJqb2luIiwiRGF0ZSIsIlByb21pc2UiLCJyZXNvbHZlIiwicmVqZWN0IiwiZW5zdXJlRGlyIiwiZXJyIiwicGF5bG9hZCIsIkpTT04iLCJzdHJpbmdpZnkiLCJtYXAiLCJzZXQiLCJiZWdpbiIsImRzIiwiQXJyYXkiLCJpIiwiZGVsZXRlIiwiZmx1c2giLCJ2YWx1ZXMiLCJjbGVhciJdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBRUE7Ozs7QUFDQTs7OztBQUNBOzs7OzJjQUpBOztBQU1BO0FBQ0E7QUFDQSxNQUFNQSxnQkFBZ0IsR0FBdEI7O0FBRUE7QUFDQSxNQUFNQyxtQkFBbUIsSUFBekI7O0FBRUE7SUFDTUMsVyxHQUFOLE1BQU1BLFdBQU4sQ0FBa0I7QUFDaEJDLGNBQVlDLElBQVosRUFBa0I7QUFDaEIsU0FBS0EsSUFBTCxHQUFZQSxJQUFaO0FBQ0EsU0FBS0MsY0FBTCxHQUFzQkMsWUFBWUMsR0FBWixFQUF0QjtBQUNBLFNBQUtDLEtBQUwsR0FBYSxFQUFiO0FBQ0EsU0FBS0MsU0FBTCxHQUFpQixDQUFqQjs7QUFFQSxRQUFJQyxLQUFLQyxNQUFMLENBQVlDLEdBQVosQ0FBZ0IsNkJBQWhCLENBQUosRUFBb0Q7QUFDbEQ7QUFDQUMsY0FBUUMsR0FBUixDQUFZLG9CQUFaLEVBQ0UsbUJBREYsRUFFRSxzREFGRixFQUUwRCxLQUFLVixJQUYvRDtBQUdEOztBQUVELFFBQUlNLEtBQUtDLE1BQUwsQ0FBWUMsR0FBWixDQUFnQiw2QkFBaEIsQ0FBSixFQUFvRDtBQUNsRDtBQUNBQyxjQUFRRSxPQUFSLENBQWdCLEtBQUtYLElBQXJCO0FBQ0Q7QUFDRjs7QUFFRFksT0FBS0MsU0FBTCxFQUFnQjtBQUNkLFVBQU1DLFdBQVdaLFlBQVlDLEdBQVosS0FBb0IsS0FBS0YsY0FBMUM7O0FBRUEsUUFBSUssS0FBS0MsTUFBTCxDQUFZQyxHQUFaLENBQWdCLDZCQUFoQixDQUFKLEVBQW9EO0FBQ2xEO0FBQ0FDLGNBQVFDLEdBQVIsQ0FBWSx1QkFBWixFQUNFLG1CQURGLEVBRUUsc0RBRkYsRUFFMEQsS0FBS1YsSUFGL0QsRUFFcUVhLFNBRnJFLEVBR0Usa0NBSEYsRUFHc0NDLFFBSHRDO0FBSUQ7O0FBRUQsUUFBSVIsS0FBS0MsTUFBTCxDQUFZQyxHQUFaLENBQWdCLCtCQUFoQixNQUFxRCxFQUF6RCxFQUE2RDtBQUMzRCxXQUFLSixLQUFMLENBQVdXLElBQVgsQ0FBZ0IsRUFBQ0YsU0FBRCxFQUFZQyxRQUFaLEVBQWhCO0FBQ0Q7O0FBRUQsU0FBS1QsU0FBTDtBQUNBLFFBQUksS0FBS0EsU0FBTCxJQUFrQlQsYUFBdEIsRUFBcUM7QUFDbkMsV0FBS29CLE1BQUw7QUFDRDtBQUNGOztBQUVEQSxXQUFTO0FBQ1AsU0FBS0osSUFBTCxDQUFVLFFBQVY7O0FBRUEsUUFBSU4sS0FBS0MsTUFBTCxDQUFZQyxHQUFaLENBQWdCLDZCQUFoQixDQUFKLEVBQW9EO0FBQ2xEO0FBQ0FDLGNBQVFRLFVBQVIsQ0FBbUIsS0FBS2pCLElBQXhCO0FBQ0Q7QUFDRjs7QUFFRGtCLGNBQVk7QUFDVixXQUFPO0FBQ0xsQixZQUFNLEtBQUtBLElBRE47QUFFTG1CLGVBQVMsS0FBS2Y7QUFGVCxLQUFQO0FBSUQ7O0FBRURnQixhQUFXO0FBQ1QsV0FBTyxLQUFLaEIsS0FBTCxDQUFXaUIsTUFBbEI7QUFDRDtBQTNEZSxDOzs7QUE4RGxCLElBQUlDLGVBQWUsRUFBbkI7QUFDQSxJQUFJQyxpQkFBaUIsQ0FBckI7QUFDQSxNQUFNQyxhQUFhLElBQUlDLEdBQUosRUFBbkI7O0FBRUEsU0FBU0MsYUFBVCxDQUF1QkMsVUFBdkIsRUFBbUNkLFNBQW5DLEVBQThDO0FBQzVDLFFBQU1lLFlBQVksQ0FBQyxTQUFELEVBQVksV0FBWixFQUF5QixTQUF6QixFQUFvQ0MsSUFBcEMsQ0FBeUNDLFFBQVE7QUFDakUsVUFBTUMsUUFBUXpCLEtBQUtDLE1BQUwsQ0FBWUMsR0FBWixDQUFpQix1QkFBc0JzQixJQUFLLEVBQTVDLENBQWQ7QUFDQSxXQUFPQyxVQUFVLEVBQVYsSUFBZ0JBLFVBQVUsS0FBakM7QUFDRCxHQUhpQixDQUFsQjtBQUlBLE1BQUksQ0FBQ0gsU0FBTCxFQUFnQjtBQUNkLFdBQU8sS0FBUDtBQUNEOztBQUVELFFBQU1JLE9BQU8sSUFBSUMsTUFBSixDQUFXM0IsS0FBS0MsTUFBTCxDQUFZQyxHQUFaLENBQWdCLHdCQUFoQixDQUFYLENBQWI7QUFDQSxNQUFJLENBQUN3QixLQUFLRSxJQUFMLENBQVcsR0FBRVAsVUFBVyxJQUFHZCxTQUFVLEVBQXJDLENBQUwsRUFBOEM7QUFDNUMsV0FBTyxLQUFQO0FBQ0Q7O0FBRUQsU0FBTyxJQUFQO0FBQ0Q7O0FBRUQsTUFBTXNCLFlBQVk7QUFDVkMsTUFBTixHQUFhO0FBQUE7QUFDWCxZQUFNQyxVQUFVL0IsS0FBS0MsTUFBTCxDQUFZQyxHQUFaLENBQWdCLCtCQUFoQixDQUFoQjtBQUNBLFVBQUk2QixZQUFZLEVBQVosSUFBa0JBLFlBQVlDLFNBQTlCLElBQTJDRCxZQUFZLElBQTNELEVBQWlFO0FBQy9EO0FBQ0Q7QUFDRCxZQUFNRSxXQUFXLGVBQUtDLElBQUwsQ0FBVUgsT0FBVixFQUFvQixlQUFjSSxLQUFLdEMsR0FBTCxFQUFXLE9BQTdDLENBQWpCOztBQUVBLFlBQU0sSUFBSXVDLE9BQUosQ0FBWSxVQUFDQyxPQUFELEVBQVVDLE1BQVYsRUFBcUI7QUFDckMsMEJBQUdDLFNBQUgsQ0FBYVIsT0FBYixFQUFzQjtBQUFBLGlCQUFRUyxNQUFNRixPQUFPRSxHQUFQLENBQU4sR0FBb0JILFNBQTVCO0FBQUEsU0FBdEI7QUFDRCxPQUZLLENBQU47O0FBSUEsWUFBTUksVUFBVUMsS0FBS0MsU0FBTCxDQUFlM0IsYUFBYTRCLEdBQWIsQ0FBaUI7QUFBQSxlQUFPQyxJQUFJakMsU0FBSixFQUFQO0FBQUEsT0FBakIsQ0FBZixDQUFoQjtBQUNBLFlBQU0sd0JBQVVxQixRQUFWLEVBQW9CUSxPQUFwQixDQUFOOztBQUVBLFVBQUl6QyxLQUFLQyxNQUFMLENBQVlDLEdBQVosQ0FBZ0IsNkJBQWhCLENBQUosRUFBb0Q7QUFDbEQ7QUFDQUMsZ0JBQVFDLEdBQVIsQ0FBWSwyQkFBWixFQUNFLG1CQURGLEVBRUUsbUNBRkYsRUFFdUNZLGFBQWFELE1BRnBELEVBRTREa0IsUUFGNUQ7QUFHRDs7QUFFRGpCLHFCQUFlLEVBQWY7QUFyQlc7QUFzQlosR0F2QmU7O0FBeUJoQjhCLFFBQU16QixVQUFOLEVBQWtCO0FBQ2hCLFFBQUksQ0FBQ0QsY0FBY0MsVUFBZCxFQUEwQixPQUExQixDQUFMLEVBQXlDO0FBQ3ZDO0FBQ0Q7O0FBRUQsVUFBTTBCLEtBQUssSUFBSXZELFdBQUosQ0FBZ0I2QixVQUFoQixDQUFYO0FBQ0FILGVBQVcyQixHQUFYLENBQWV4QixVQUFmLEVBQTJCMEIsRUFBM0I7QUFDRCxHQWhDZTs7QUFrQ2hCekMsT0FBS2UsVUFBTCxFQUFpQmQsU0FBakIsRUFBNEI7QUFDMUIsUUFBSWMsc0JBQXNCMkIsS0FBMUIsRUFBaUM7QUFDL0IsV0FBSyxJQUFJQyxJQUFJLENBQWIsRUFBZ0JBLElBQUk1QixXQUFXTixNQUEvQixFQUF1Q2tDLEdBQXZDLEVBQTRDO0FBQzFDLGFBQUszQyxJQUFMLENBQVVlLFdBQVc0QixDQUFYLENBQVYsRUFBeUIxQyxTQUF6QjtBQUNEO0FBQ0Q7QUFDRDs7QUFFRCxRQUFJLENBQUNhLGNBQWNDLFVBQWQsRUFBMEJkLFNBQTFCLENBQUwsRUFBMkM7QUFDekM7QUFDRDs7QUFFRCxVQUFNd0MsS0FBSzdCLFdBQVdoQixHQUFYLENBQWVtQixVQUFmLENBQVg7QUFDQSxRQUFJMEIsT0FBT2YsU0FBWCxFQUFzQjtBQUNwQjtBQUNEO0FBQ0RlLE9BQUd6QyxJQUFILENBQVFDLFNBQVI7QUFDRCxHQW5EZTs7QUFxRGhCRyxTQUFPVyxVQUFQLEVBQW1CO0FBQ2pCLFFBQUlBLHNCQUFzQjJCLEtBQTFCLEVBQWlDO0FBQy9CLFdBQUssSUFBSUMsSUFBSSxDQUFiLEVBQWdCQSxJQUFJNUIsV0FBV04sTUFBL0IsRUFBdUNrQyxHQUF2QyxFQUE0QztBQUMxQyxhQUFLdkMsTUFBTCxDQUFZVyxXQUFXNEIsQ0FBWCxDQUFaO0FBQ0Q7QUFDRDtBQUNEOztBQUVELFFBQUksQ0FBQzdCLGNBQWNDLFVBQWQsRUFBMEIsUUFBMUIsQ0FBTCxFQUEwQztBQUN4QztBQUNEOztBQUVELFVBQU0wQixLQUFLN0IsV0FBV2hCLEdBQVgsQ0FBZW1CLFVBQWYsQ0FBWDtBQUNBLFFBQUkwQixPQUFPZixTQUFYLEVBQXNCO0FBQ3BCO0FBQ0Q7QUFDRGUsT0FBR3JDLE1BQUg7O0FBRUFNLGlCQUFhUCxJQUFiLENBQWtCc0MsRUFBbEI7QUFDQTdCLGVBQVdnQyxNQUFYLENBQWtCN0IsVUFBbEI7O0FBRUFKLHNCQUFrQjhCLEdBQUdqQyxRQUFILEVBQWxCO0FBQ0EsUUFBSUcsa0JBQWtCMUIsZ0JBQXRCLEVBQXdDO0FBQ3RDMEIsdUJBQWlCLENBQWpCO0FBQ0EsV0FBS2EsSUFBTDtBQUNEO0FBQ0YsR0EvRWU7O0FBaUZWcUIsT0FBTixHQUFjO0FBQUE7O0FBQUE7QUFDWm5DLG1CQUFhUCxJQUFiLENBQWtCLEdBQUdTLFdBQVdrQyxNQUFYLEVBQXJCO0FBQ0FsQyxpQkFBV21DLEtBQVg7QUFDQSxZQUFNLE1BQUt2QixJQUFMLEVBQU47QUFIWTtBQUliO0FBckZlLENBQWxCOztrQkF3RmVELFMiLCJmaWxlIjoieWFyZHN0aWNrLmpzIiwic291cmNlUm9vdCI6Ii9ob21lL3RyYXZpcy9idWlsZC9hdG9tL2F0b20vb3V0L2FwcC9ub2RlX21vZHVsZXMvZ2l0aHViIiwic291cmNlc0NvbnRlbnQiOlsiLy8gTWVhc3VyZSBlbGFwc2VkIGR1cmF0aW9ucyBmcm9tIHNwZWNpZmljIGJlZ2lubmluZyBwb2ludHMuXG5cbmltcG9ydCBmcyBmcm9tICdmcy1leHRyYSc7XG5pbXBvcnQgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7d3JpdGVGaWxlfSBmcm9tICcuL2hlbHBlcnMnO1xuXG4vLyBUaGUgbWF4aW11bSBudW1iZXIgb2YgbWFya3Mgd2l0aGluIGEgc2luZ2xlIER1cmF0aW9uU2V0LiBBIER1cmF0aW9uU2V0IHdpbGwgYmUgYXV0b21hdGljYWxseSBmaW5pc2hlZCBpZiB0aGlzIG1hbnlcbi8vIG1hcmtzIGFyZSByZWNvcmRlZC5cbmNvbnN0IE1BWElNVU1fTUFSS1MgPSAxMDA7XG5cbi8vIEZsdXNoIGFsbCBub24tYWN0aXZlIER1cmF0aW9uU2V0cyB0byBkaXNrIGVhY2ggdGltZSB0aGF0IHRoaXMgbWFueSBtYXJrcyBoYXZlIGJlZW4gYWNjdW11bGF0ZWQuXG5jb25zdCBQRVJTSVNUX0lOVEVSVkFMID0gMTAwMDtcblxuLy8gQSBzZXF1ZW5jZSBvZiBkdXJhdGlvbnMgbWVhc3VyZWQgZnJvbSBhIGZpeGVkIGJlZ2lubmluZyBwb2ludC5cbmNsYXNzIER1cmF0aW9uU2V0IHtcbiAgY29uc3RydWN0b3IobmFtZSkge1xuICAgIHRoaXMubmFtZSA9IG5hbWU7XG4gICAgdGhpcy5zdGFydFRpbWVzdGFtcCA9IHBlcmZvcm1hbmNlLm5vdygpO1xuICAgIHRoaXMubWFya3MgPSBbXTtcbiAgICB0aGlzLm1hcmtDb3VudCA9IDA7XG5cbiAgICBpZiAoYXRvbS5jb25maWcuZ2V0KCdnaXRodWIucGVyZm9ybWFuY2VUb0NvbnNvbGUnKSkge1xuICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWNvbnNvbGVcbiAgICAgIGNvbnNvbGUubG9nKCclY2JlZ2luICVjJXM6YmVnaW4nLFxuICAgICAgICAnZm9udC13ZWlnaHQ6IGJvbGQnLFxuICAgICAgICAnZm9udC13ZWlnaHQ6IG5vcm1hbDsgZm9udC1zdHlsZTogaXRhbGljOyBjb2xvcjogYmx1ZScsIHRoaXMubmFtZSk7XG4gICAgfVxuXG4gICAgaWYgKGF0b20uY29uZmlnLmdldCgnZ2l0aHViLnBlcmZvcm1hbmNlVG9Qcm9maWxlJykpIHtcbiAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1jb25zb2xlXG4gICAgICBjb25zb2xlLnByb2ZpbGUodGhpcy5uYW1lKTtcbiAgICB9XG4gIH1cblxuICBtYXJrKGV2ZW50TmFtZSkge1xuICAgIGNvbnN0IGR1cmF0aW9uID0gcGVyZm9ybWFuY2Uubm93KCkgLSB0aGlzLnN0YXJ0VGltZXN0YW1wO1xuXG4gICAgaWYgKGF0b20uY29uZmlnLmdldCgnZ2l0aHViLnBlcmZvcm1hbmNlVG9Db25zb2xlJykpIHtcbiAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1jb25zb2xlXG4gICAgICBjb25zb2xlLmxvZygnJWNtYXJrICVjJXM6JXMgJWMlZG1zJyxcbiAgICAgICAgJ2ZvbnQtd2VpZ2h0OiBib2xkJyxcbiAgICAgICAgJ2ZvbnQtd2VpZ2h0OiBub3JtYWw7IGZvbnQtc3R5bGU6IGl0YWxpYzsgY29sb3I6IGJsdWUnLCB0aGlzLm5hbWUsIGV2ZW50TmFtZSxcbiAgICAgICAgJ2ZvbnQtc3R5bGU6IG5vcm1hbDsgY29sb3I6IGJsYWNrJywgZHVyYXRpb24pO1xuICAgIH1cblxuICAgIGlmIChhdG9tLmNvbmZpZy5nZXQoJ2dpdGh1Yi5wZXJmb3JtYW5jZVRvRGlyZWN0b3J5JykgIT09ICcnKSB7XG4gICAgICB0aGlzLm1hcmtzLnB1c2goe2V2ZW50TmFtZSwgZHVyYXRpb259KTtcbiAgICB9XG5cbiAgICB0aGlzLm1hcmtDb3VudCsrO1xuICAgIGlmICh0aGlzLm1hcmtDb3VudCA+PSBNQVhJTVVNX01BUktTKSB7XG4gICAgICB0aGlzLmZpbmlzaCgpO1xuICAgIH1cbiAgfVxuXG4gIGZpbmlzaCgpIHtcbiAgICB0aGlzLm1hcmsoJ2ZpbmlzaCcpO1xuXG4gICAgaWYgKGF0b20uY29uZmlnLmdldCgnZ2l0aHViLnBlcmZvcm1hbmNlVG9Qcm9maWxlJykpIHtcbiAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1jb25zb2xlXG4gICAgICBjb25zb2xlLnByb2ZpbGVFbmQodGhpcy5uYW1lKTtcbiAgICB9XG4gIH1cblxuICBzZXJpYWxpemUoKSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIG5hbWU6IHRoaXMubmFtZSxcbiAgICAgIG1hcmtlcnM6IHRoaXMubWFya3MsXG4gICAgfTtcbiAgfVxuXG4gIGdldENvdW50KCkge1xuICAgIHJldHVybiB0aGlzLm1hcmtzLmxlbmd0aDtcbiAgfVxufVxuXG5sZXQgZHVyYXRpb25TZXRzID0gW107XG5sZXQgdG90YWxNYXJrQ291bnQgPSAwO1xuY29uc3QgYWN0aXZlU2V0cyA9IG5ldyBNYXAoKTtcblxuZnVuY3Rpb24gc2hvdWxkQ2FwdHVyZShzZXJpZXNOYW1lLCBldmVudE5hbWUpIHtcbiAgY29uc3QgYW55QWN0aXZlID0gWydDb25zb2xlJywgJ0RpcmVjdG9yeScsICdQcm9maWxlJ10uc29tZShraW5kID0+IHtcbiAgICBjb25zdCB2YWx1ZSA9IGF0b20uY29uZmlnLmdldChgZ2l0aHViLnBlcmZvcm1hbmNlVG8ke2tpbmR9YCk7XG4gICAgcmV0dXJuIHZhbHVlICE9PSAnJyAmJiB2YWx1ZSAhPT0gZmFsc2U7XG4gIH0pO1xuICBpZiAoIWFueUFjdGl2ZSkge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIGNvbnN0IG1hc2sgPSBuZXcgUmVnRXhwKGF0b20uY29uZmlnLmdldCgnZ2l0aHViLnBlcmZvcm1hbmNlTWFzaycpKTtcbiAgaWYgKCFtYXNrLnRlc3QoYCR7c2VyaWVzTmFtZX06JHtldmVudE5hbWV9YCkpIHtcbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cblxuICByZXR1cm4gdHJ1ZTtcbn1cblxuY29uc3QgeWFyZHN0aWNrID0ge1xuICBhc3luYyBzYXZlKCkge1xuICAgIGNvbnN0IGRlc3REaXIgPSBhdG9tLmNvbmZpZy5nZXQoJ2dpdGh1Yi5wZXJmb3JtYW5jZVRvRGlyZWN0b3J5Jyk7XG4gICAgaWYgKGRlc3REaXIgPT09ICcnIHx8IGRlc3REaXIgPT09IHVuZGVmaW5lZCB8fCBkZXN0RGlyID09PSBudWxsKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIGNvbnN0IGZpbGVOYW1lID0gcGF0aC5qb2luKGRlc3REaXIsIGBwZXJmb3JtYW5jZS0ke0RhdGUubm93KCl9Lmpzb25gKTtcblxuICAgIGF3YWl0IG5ldyBQcm9taXNlKChyZXNvbHZlLCByZWplY3QpID0+IHtcbiAgICAgIGZzLmVuc3VyZURpcihkZXN0RGlyLCBlcnIgPT4gKGVyciA/IHJlamVjdChlcnIpIDogcmVzb2x2ZSgpKSk7XG4gICAgfSk7XG5cbiAgICBjb25zdCBwYXlsb2FkID0gSlNPTi5zdHJpbmdpZnkoZHVyYXRpb25TZXRzLm1hcChzZXQgPT4gc2V0LnNlcmlhbGl6ZSgpKSk7XG4gICAgYXdhaXQgd3JpdGVGaWxlKGZpbGVOYW1lLCBwYXlsb2FkKTtcblxuICAgIGlmIChhdG9tLmNvbmZpZy5nZXQoJ2dpdGh1Yi5wZXJmb3JtYW5jZVRvQ29uc29sZScpKSB7XG4gICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgbm8tY29uc29sZVxuICAgICAgY29uc29sZS5sb2coJyVjc2F2ZWQgJWMlZCBzZXJpZXMgdG8gJXMnLFxuICAgICAgICAnZm9udC13ZWlnaHQ6IGJvbGQnLFxuICAgICAgICAnZm9udC13ZWlnaHQ6IG5vcm1hbDsgY29sb3I6IGJsYWNrJywgZHVyYXRpb25TZXRzLmxlbmd0aCwgZmlsZU5hbWUpO1xuICAgIH1cblxuICAgIGR1cmF0aW9uU2V0cyA9IFtdO1xuICB9LFxuXG4gIGJlZ2luKHNlcmllc05hbWUpIHtcbiAgICBpZiAoIXNob3VsZENhcHR1cmUoc2VyaWVzTmFtZSwgJ2JlZ2luJykpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBjb25zdCBkcyA9IG5ldyBEdXJhdGlvblNldChzZXJpZXNOYW1lKTtcbiAgICBhY3RpdmVTZXRzLnNldChzZXJpZXNOYW1lLCBkcyk7XG4gIH0sXG5cbiAgbWFyayhzZXJpZXNOYW1lLCBldmVudE5hbWUpIHtcbiAgICBpZiAoc2VyaWVzTmFtZSBpbnN0YW5jZW9mIEFycmF5KSB7XG4gICAgICBmb3IgKGxldCBpID0gMDsgaSA8IHNlcmllc05hbWUubGVuZ3RoOyBpKyspIHtcbiAgICAgICAgdGhpcy5tYXJrKHNlcmllc05hbWVbaV0sIGV2ZW50TmFtZSk7XG4gICAgICB9XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgaWYgKCFzaG91bGRDYXB0dXJlKHNlcmllc05hbWUsIGV2ZW50TmFtZSkpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBjb25zdCBkcyA9IGFjdGl2ZVNldHMuZ2V0KHNlcmllc05hbWUpO1xuICAgIGlmIChkcyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIGRzLm1hcmsoZXZlbnROYW1lKTtcbiAgfSxcblxuICBmaW5pc2goc2VyaWVzTmFtZSkge1xuICAgIGlmIChzZXJpZXNOYW1lIGluc3RhbmNlb2YgQXJyYXkpIHtcbiAgICAgIGZvciAobGV0IGkgPSAwOyBpIDwgc2VyaWVzTmFtZS5sZW5ndGg7IGkrKykge1xuICAgICAgICB0aGlzLmZpbmlzaChzZXJpZXNOYW1lW2ldKTtcbiAgICAgIH1cbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBpZiAoIXNob3VsZENhcHR1cmUoc2VyaWVzTmFtZSwgJ2ZpbmlzaCcpKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgY29uc3QgZHMgPSBhY3RpdmVTZXRzLmdldChzZXJpZXNOYW1lKTtcbiAgICBpZiAoZHMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBkcy5maW5pc2goKTtcblxuICAgIGR1cmF0aW9uU2V0cy5wdXNoKGRzKTtcbiAgICBhY3RpdmVTZXRzLmRlbGV0ZShzZXJpZXNOYW1lKTtcblxuICAgIHRvdGFsTWFya0NvdW50ICs9IGRzLmdldENvdW50KCk7XG4gICAgaWYgKHRvdGFsTWFya0NvdW50ID49IFBFUlNJU1RfSU5URVJWQUwpIHtcbiAgICAgIHRvdGFsTWFya0NvdW50ID0gMDtcbiAgICAgIHRoaXMuc2F2ZSgpO1xuICAgIH1cbiAgfSxcblxuICBhc3luYyBmbHVzaCgpIHtcbiAgICBkdXJhdGlvblNldHMucHVzaCguLi5hY3RpdmVTZXRzLnZhbHVlcygpKTtcbiAgICBhY3RpdmVTZXRzLmNsZWFyKCk7XG4gICAgYXdhaXQgdGhpcy5zYXZlKCk7XG4gIH0sXG59O1xuXG5leHBvcnQgZGVmYXVsdCB5YXJkc3RpY2s7XG4iXX0=