'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _listSelection = require('./list-selection');

var _listSelection2 = _interopRequireDefault(_listSelection);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

let CompositeListSelection = class CompositeListSelection {
  constructor(_ref) {
    let listsByKey = _ref.listsByKey,
        idForItem = _ref.idForItem;

    this.keysBySelection = new Map();
    this.selections = [];
    this.idForItem = idForItem || (item => item);
    this.resolveNextUpdatePromise = () => {};

    for (const key in listsByKey) {
      const selection = new _listSelection2.default({ items: listsByKey[key] });
      this.keysBySelection.set(selection, key);
      this.selections.push(selection);
    }

    this.selectFirstNonEmptyList();
  }

  updateLists(listsByKey) {
    let wasChanged = false;

    const keys = Object.keys(listsByKey);
    for (let i = 0; i < keys.length; i++) {
      const newItems = listsByKey[keys[i]];
      const selection = this.selections[i];

      const oldItems = selection.getItems();
      if (!wasChanged) {
        if (newItems.length !== oldItems.length) {
          wasChanged = true;
        } else {
          for (let j = 0; j < oldItems.length; j++) {
            if (oldItems[j] !== newItems[j]) {
              wasChanged = true;
              break;
            }
          }
        }
      }

      const oldHeadItem = selection.getHeadItem();
      selection.setItems(newItems);
      let newHeadItem = null;
      if (oldHeadItem) {
        newHeadItem = newItems.find(item => this.idForItem(item) === this.idForItem(oldHeadItem));
      }
      if (newHeadItem) {
        selection.selectItem(newHeadItem);
      }
    }
    if (this.getActiveSelection().getItems().length === 0) {
      this.activateNextSelection() || this.activatePreviousSelection();
    }

    if (wasChanged) {
      this.resolveNextUpdatePromise();
    }
  }

  getNextUpdatePromise() {
    return new Promise((resolve, reject) => {
      this.resolveNextUpdatePromise = resolve;
    });
  }

  selectFirstNonEmptyList() {
    this.activeSelectionIndex = 0;
    for (let i = 0; i < this.selections.length; i++) {
      if (this.selections[i].getItems().length) {
        this.activeSelectionIndex = i;
        break;
      }
    }
  }

  getActiveListKey() {
    return this.keysBySelection.get(this.getActiveSelection());
  }

  getSelectedItems() {
    return this.getActiveSelection().getSelectedItems();
  }

  getHeadItem() {
    return this.getActiveSelection().getHeadItem();
  }

  getActiveSelection() {
    return this.selections[this.activeSelectionIndex];
  }

  activateSelection(selection) {
    const index = this.selections.indexOf(selection);
    if (index === -1) {
      throw new Error('Selection not found');
    }
    this.activeSelectionIndex = index;
  }

  activateNextSelection() {
    for (let i = this.activeSelectionIndex + 1; i < this.selections.length; i++) {
      if (this.selections[i].getItems().length > 0) {
        this.activeSelectionIndex = i;
        return true;
      }
    }
    return false;
  }

  activatePreviousSelection() {
    for (let i = this.activeSelectionIndex - 1; i >= 0; i--) {
      if (this.selections[i].getItems().length > 0) {
        this.activeSelectionIndex = i;
        return true;
      }
    }
    return false;
  }

  activateLastSelection() {
    for (let i = this.selections.length - 1; i >= 0; i--) {
      if (this.selections[i].getItems().length > 0) {
        this.activeSelectionIndex = i;
        return true;
      }
    }
    return false;
  }

  selectItem(item) {
    let preserveTail = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

    const selection = this.selectionForItem(item);
    if (!selection) {
      throw new Error(`No item found: ${item}`);
    }
    if (!preserveTail) {
      this.activateSelection(selection);
    }
    if (selection === this.getActiveSelection()) {
      selection.selectItem(item, preserveTail);
    }
  }

  addOrSubtractSelection(item) {
    const selection = this.selectionForItem(item);
    if (!selection) {
      throw new Error(`No item found: ${item}`);
    }

    if (selection === this.getActiveSelection()) {
      selection.addOrSubtractSelection(item);
    } else {
      this.activateSelection(selection);
      selection.selectItem(item);
    }
  }

  selectAllItems() {
    this.getActiveSelection().selectAllItems();
  }

  selectFirstItem(preserveTail) {
    this.getActiveSelection().selectFirstItem(preserveTail);
  }

  selectLastItem(preserveTail) {
    this.getActiveSelection().selectLastItem(preserveTail);
  }

  coalesce() {
    this.getActiveSelection().coalesce();
  }

  selectionForItem(item) {
    return this.selections.find(selection => selection.getItems().indexOf(item) > -1);
  }

  listKeyForItem(item) {
    return this.keysBySelection.get(this.selectionForItem(item));
  }

  selectNextItem() {
    let preserveTail = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

    if (!preserveTail && this.getActiveSelection().getHeadItem() === this.getActiveSelection().getLastItem()) {
      if (this.activateNextSelection()) {
        this.getActiveSelection().selectFirstItem();
        return true;
      } else {
        this.getActiveSelection().selectLastItem();
        return false;
      }
    } else {
      this.getActiveSelection().selectNextItem(preserveTail);
      return true;
    }
  }

  selectPreviousItem() {
    let preserveTail = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

    if (!preserveTail && this.getActiveSelection().getHeadItem() === this.getActiveSelection().getItems()[0]) {
      if (this.activatePreviousSelection()) {
        this.getActiveSelection().selectLastItem();
        return true;
      } else {
        this.getActiveSelection().selectFirstItem();
        return false;
      }
    } else {
      this.getActiveSelection().selectPreviousItem(preserveTail);
      return true;
    }
  }

  findItem(predicate) {
    for (let i = 0; i < this.selections.length; i++) {
      const selection = this.selections[i];
      const key = this.keysBySelection.get(selection);
      const found = selection.getItems().find(item => predicate(item, key));
      if (found !== undefined) {
        return found;
      }
    }
    return null;
  }
};
exports.default = CompositeListSelection;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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