'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _eventKit = require('event-kit');

/*
 * Register callbacks and construct Promises to wait for the next occurrence of specific events that occur throughout
 * the data refresh and rendering cycle.
 */
let Switchboard = class Switchboard {
  constructor() {
    this.promises = new Map();
    this.emitter = new _eventKit.Emitter();
  }

  /*
   * Invoke a callback each time that a desired event is observed. Return a Disposable that can be used to
   * unsubscribe from events.
   *
   * In general, you should use the more specific `onDidXyz` methods.
   */
  onDid(eventName, callback) {
    return this.emitter.on(`did-${eventName}`, callback);
  }

  /*
   * Indicate that a named event has been observed, firing any callbacks and resolving any Promises that were created
   * for this event. Optionally provide a payload with more information.
   *
   * In general, you should prefer the more specific `didXyz()` methods.
   */
  did(eventName, payload) {
    this.emitter.emit(`did-${eventName}`, payload);
  }

  /*
   * Retrieve a Promise that will be resolved the next time a desired event is observed.
   *
   * In general, you should prefer the more specific `getXyzPromise()` methods.
   */
  getPromise(eventName) {
    const existing = this.promises.get(eventName);
    if (existing !== undefined) {
      return existing;
    }

    const created = new Promise((resolve, reject) => {
      const subscription = this.onDid(eventName, payload => {
        subscription.dispose();
        this.promises.delete(eventName);
        resolve(payload);
      });
    });
    this.promises.set(eventName, created);
    return created;
  }
};
exports.default = Switchboard;


['UpdateRepository', 'BeginStageOperation', 'FinishStageOperation', 'ChangePatch', 'ScheduleActiveContextUpdate', 'BeginActiveContextUpdate', 'FinishActiveContextUpdate', 'FinishRender', 'FinishContextChangeRender'].forEach(eventName => {
  Switchboard.prototype[`did${eventName}`] = function (payload) {
    this.did(eventName, payload);
  };

  Switchboard.prototype[`get${eventName}Promise`] = function () {
    return this.getPromise(eventName);
  };

  Switchboard.prototype[`onDid${eventName}`] = function (callback) {
    return this.onDid(eventName, callback);
  };
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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