'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _desc, _value, _class;

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _eventKit = require('event-kit');

var _coreDecorators = require('core-decorators');

var _deferredCallbackQueue = require('../deferred-callback-queue');

var _deferredCallbackQueue2 = _interopRequireDefault(_deferredCallbackQueue);

var _repository = require('./repository');

var _repository2 = _interopRequireDefault(_repository);

var _resolutionProgress = require('./conflicts/resolution-progress');

var _resolutionProgress2 = _interopRequireDefault(_resolutionProgress);

var _fileSystemChangeObserver = require('./file-system-change-observer');

var _fileSystemChangeObserver2 = _interopRequireDefault(_fileSystemChangeObserver);

var _workspaceChangeObserver = require('./workspace-change-observer');

var _workspaceChangeObserver2 = _interopRequireDefault(_workspaceChangeObserver);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _asyncToGenerator(fn) { return function () { var gen = fn.apply(this, arguments); return new Promise(function (resolve, reject) { function step(key, arg) { try { var info = gen[key](arg); var value = info.value; } catch (error) { reject(error); return; } if (info.done) { resolve(value); } else { return Promise.resolve(value).then(function (value) { step("next", value); }, function (err) { step("throw", err); }); } } return step("next"); }); }; }

function _applyDecoratedDescriptor(target, property, decorators, descriptor, context) {
  var desc = {};
  Object['ke' + 'ys'](descriptor).forEach(function (key) {
    desc[key] = descriptor[key];
  });
  desc.enumerable = !!desc.enumerable;
  desc.configurable = !!desc.configurable;

  if ('value' in desc || desc.initializer) {
    desc.writable = true;
  }

  desc = decorators.slice().reverse().reduce(function (desc, decorator) {
    return decorator(target, property, desc) || desc;
  }, desc);

  if (context && desc.initializer !== void 0) {
    desc.value = desc.initializer ? desc.initializer.call(context) : void 0;
    desc.initializer = undefined;
  }

  if (desc.initializer === void 0) {
    Object['define' + 'Property'](target, property, desc);
    desc = null;
  }

  return desc;
}

const createRepoSym = Symbol('createRepo');

let absentWorkdirContext;

/*
 * Bundle of model objects associated with a git working directory.
 *
 * Provides synchronous access to each model in the form of a getter method that returns the model or `null` if it
 * has not yet been initialized, and asynchronous access in the form of a Promise generation method that will resolve
 * once the model is available. Initializes the platform-appropriate change observer and proxies select filesystem
 * change events.
 */
let WorkdirContext = (_class = class WorkdirContext {

  /*
   * Available options:
   * - `options.window`: Browser window global, used on Linux by the WorkspaceChangeObserver.
   * - `options.workspace`: Atom's workspace singleton, used on Linux by the WorkspaceChangeObserver.
   * - `options.promptCallback`: Callback used to collect information interactively through Atom.
   */
  constructor(directory) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    this.directory = directory;
    this.repository = (options[createRepoSym] || (() => new _repository2.default(directory)))();

    const theWindow = options.window,
          workspace = options.workspace,
          promptCallback = options.promptCallback;


    this.destroyed = false;
    this.emitter = new _eventKit.Emitter();
    this.subs = new _eventKit.CompositeDisposable();

    this.observer = this.useWorkspaceChangeObserver() ? new _workspaceChangeObserver2.default(theWindow, workspace, this.repository) : new _fileSystemChangeObserver2.default(this.repository);
    this.resolutionProgress = new _resolutionProgress2.default();
    this.deferredCallbackQueue = new _deferredCallbackQueue2.default(3000, collection => {
      this.repository.observeFilesystemChange(collection);
    });

    if (promptCallback) {
      this.repository.setPromptCallback(promptCallback);
    }

    // Wire up event forwarding among models
    this.subs.add(this.repository.onDidChangeState(this.repositoryChangedState));
    this.subs.add(this.observer.onDidChange(events => {
      const paths = events.map(e => e.special || _path2.default.join(e.directory, e.file || e.newFile));
      this.deferredCallbackQueue.push(...paths);
    }));
    this.subs.add(this.observer.onDidChangeWorkdirOrHead(() => this.emitter.emit('did-change-workdir-or-head')));
    this.subs.add(new _eventKit.Disposable(() => this.deferredCallbackQueue.destroy()));

    // If a pre-loaded Repository was provided, broadcast an initial state change event.
    this.repositoryChangedState({ from: null, to: this.repository.state });
  }

  static absent() {
    if (!absentWorkdirContext) {
      absentWorkdirContext = new AbsentWorkdirContext();
    }
    return absentWorkdirContext;
  }

  static destroyAbsent() {
    if (absentWorkdirContext) {
      absentWorkdirContext.destroy();
      absentWorkdirContext = null;
    }
  }

  static guess(options) {
    const projectPathCount = options.projectPathCount || 0;
    const initPathCount = options.initPathCount || 0;

    const createRepo = projectPathCount === 1 || projectPathCount === 0 && initPathCount === 1 ? () => _repository2.default.loadingGuess() : () => _repository2.default.absentGuess();

    return new WorkdirContext(null, { [createRepoSym]: createRepo });
  }

  /**
   * Respond to changes in `Repository` state. Load resolution progress and start the change observer when it becomes
   * present. Stop the change observer when it is destroyed. Re-broadcast the event to context subscribers
   * regardless.
   *
   * The ResolutionProgress will be loaded before the change event is re-broadcast, but change observer modifications
   * will not be complete.
   */

  repositoryChangedState(payload) {
    if (this.destroyed) {
      return;
    }

    if (this.repository.isPresent()) {
      this.observer.start().then(() => this.emitter.emit('did-start-observer'));
    } else if (this.repository.isDestroyed()) {
      this.emitter.emit('did-destroy-repository');

      this.observer.destroy();
    }

    this.emitter.emit('did-change-repository-state', payload);
  }

  isPresent() {
    return true;
  }

  isDestroyed() {
    return this.destroyed;
  }

  useWorkspaceChangeObserver() {
    return !!process.env.ATOM_GITHUB_WORKSPACE_OBSERVER || process.platform === 'linux';
  }

  // Event subscriptions

  onDidStartObserver(callback) {
    return this.emitter.on('did-start-observer', callback);
  }

  onDidChangeWorkdirOrHead(callback) {
    return this.emitter.on('did-change-workdir-or-head', callback);
  }

  onDidChangeRepositoryState(callback) {
    return this.emitter.on('did-change-repository-state', callback);
  }

  onDidUpdateRepository(callback) {
    return this.emitter.on('did-update-repository', callback);
  }

  onDidDestroyRepository(callback) {
    return this.emitter.on('did-destroy-repository', callback);
  }

  /**
   * Return a Promise that will resolve the next time that a Repository transitions to the requested state. Most
   * useful for test cases; most callers should prefer subscribing to `onDidChangeRepositoryState`.
   */
  getRepositoryStatePromise(stateName) {
    return new Promise(resolve => {
      const sub = this.onDidChangeRepositoryState(() => {
        if (this.repository.isInState(stateName)) {
          resolve();
          sub.dispose();
        }
      });
    });
  }

  /**
   * Return a Promise that will resolve the next time that a ChangeObserver successfully starts. Most useful for
   * test cases.
   */
  getObserverStartedPromise() {
    return new Promise(resolve => {
      const sub = this.onDidStartObserver(() => {
        resolve();
        sub.dispose();
      });
    });
  }

  getWorkingDirectory() {
    return this.directory;
  }

  getRepository() {
    return this.repository;
  }

  getChangeObserver() {
    return this.observer;
  }

  getResolutionProgress() {
    return this.resolutionProgress;
  }

  /*
   * Cleanly destroy any models that need to be cleaned, including stopping the filesystem watcher.
   */
  destroy() {
    var _this = this;

    return _asyncToGenerator(function* () {
      if (_this.destroyed) {
        return;
      }
      _this.destroyed = true;

      _this.subs.dispose();
      _this.repository.destroy();
      _this.emitter.dispose();

      yield _this.observer.destroy();
    })();
  }
}, (_applyDecoratedDescriptor(_class.prototype, 'repositoryChangedState', [_coreDecorators.autobind], Object.getOwnPropertyDescriptor(_class.prototype, 'repositoryChangedState'), _class.prototype)), _class);
exports.default = WorkdirContext;
let AbsentWorkdirContext = class AbsentWorkdirContext extends WorkdirContext {
  constructor() {
    super(null, { [createRepoSym]: () => _repository2.default.absent() });
  }

  isPresent() {
    return false;
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIndvcmtkaXItY29udGV4dC5qcyJdLCJuYW1lcyI6WyJjcmVhdGVSZXBvU3ltIiwiU3ltYm9sIiwiYWJzZW50V29ya2RpckNvbnRleHQiLCJXb3JrZGlyQ29udGV4dCIsImNvbnN0cnVjdG9yIiwiZGlyZWN0b3J5Iiwib3B0aW9ucyIsInJlcG9zaXRvcnkiLCJ0aGVXaW5kb3ciLCJ3aW5kb3ciLCJ3b3Jrc3BhY2UiLCJwcm9tcHRDYWxsYmFjayIsImRlc3Ryb3llZCIsImVtaXR0ZXIiLCJzdWJzIiwib2JzZXJ2ZXIiLCJ1c2VXb3Jrc3BhY2VDaGFuZ2VPYnNlcnZlciIsInJlc29sdXRpb25Qcm9ncmVzcyIsImRlZmVycmVkQ2FsbGJhY2tRdWV1ZSIsImNvbGxlY3Rpb24iLCJvYnNlcnZlRmlsZXN5c3RlbUNoYW5nZSIsInNldFByb21wdENhbGxiYWNrIiwiYWRkIiwib25EaWRDaGFuZ2VTdGF0ZSIsInJlcG9zaXRvcnlDaGFuZ2VkU3RhdGUiLCJvbkRpZENoYW5nZSIsImV2ZW50cyIsInBhdGhzIiwibWFwIiwiZSIsInNwZWNpYWwiLCJqb2luIiwiZmlsZSIsIm5ld0ZpbGUiLCJwdXNoIiwib25EaWRDaGFuZ2VXb3JrZGlyT3JIZWFkIiwiZW1pdCIsImRlc3Ryb3kiLCJmcm9tIiwidG8iLCJzdGF0ZSIsImFic2VudCIsIkFic2VudFdvcmtkaXJDb250ZXh0IiwiZGVzdHJveUFic2VudCIsImd1ZXNzIiwicHJvamVjdFBhdGhDb3VudCIsImluaXRQYXRoQ291bnQiLCJjcmVhdGVSZXBvIiwibG9hZGluZ0d1ZXNzIiwiYWJzZW50R3Vlc3MiLCJwYXlsb2FkIiwiaXNQcmVzZW50Iiwic3RhcnQiLCJ0aGVuIiwiaXNEZXN0cm95ZWQiLCJwcm9jZXNzIiwiZW52IiwiQVRPTV9HSVRIVUJfV09SS1NQQUNFX09CU0VSVkVSIiwicGxhdGZvcm0iLCJvbkRpZFN0YXJ0T2JzZXJ2ZXIiLCJjYWxsYmFjayIsIm9uIiwib25EaWRDaGFuZ2VSZXBvc2l0b3J5U3RhdGUiLCJvbkRpZFVwZGF0ZVJlcG9zaXRvcnkiLCJvbkRpZERlc3Ryb3lSZXBvc2l0b3J5IiwiZ2V0UmVwb3NpdG9yeVN0YXRlUHJvbWlzZSIsInN0YXRlTmFtZSIsIlByb21pc2UiLCJyZXNvbHZlIiwic3ViIiwiaXNJblN0YXRlIiwiZGlzcG9zZSIsImdldE9ic2VydmVyU3RhcnRlZFByb21pc2UiLCJnZXRXb3JraW5nRGlyZWN0b3J5IiwiZ2V0UmVwb3NpdG9yeSIsImdldENoYW5nZU9ic2VydmVyIiwiZ2V0UmVzb2x1dGlvblByb2dyZXNzIl0sIm1hcHBpbmdzIjoiOzs7Ozs7Ozs7QUFBQTs7OztBQUVBOztBQUNBOztBQUVBOzs7O0FBQ0E7Ozs7QUFDQTs7OztBQUNBOzs7O0FBQ0E7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7QUFFQSxNQUFNQSxnQkFBZ0JDLE9BQU8sWUFBUCxDQUF0Qjs7QUFFQSxJQUFJQyxvQkFBSjs7QUFFQTs7Ozs7Ozs7SUFRcUJDLGMsYUFBTixNQUFNQSxjQUFOLENBQXFCOztBQUVsQzs7Ozs7O0FBTUFDLGNBQVlDLFNBQVosRUFBcUM7QUFBQSxRQUFkQyxPQUFjLHVFQUFKLEVBQUk7O0FBQ25DLFNBQUtELFNBQUwsR0FBaUJBLFNBQWpCO0FBQ0EsU0FBS0UsVUFBTCxHQUFrQixDQUFDRCxRQUFRTixhQUFSLE1BQTJCLE1BQU0seUJBQWVLLFNBQWYsQ0FBakMsQ0FBRCxHQUFsQjs7QUFGbUMsVUFJcEJHLFNBSm9CLEdBSW9CRixPQUpwQixDQUk1QkcsTUFKNEI7QUFBQSxVQUlUQyxTQUpTLEdBSW9CSixPQUpwQixDQUlUSSxTQUpTO0FBQUEsVUFJRUMsY0FKRixHQUlvQkwsT0FKcEIsQ0FJRUssY0FKRjs7O0FBTW5DLFNBQUtDLFNBQUwsR0FBaUIsS0FBakI7QUFDQSxTQUFLQyxPQUFMLEdBQWUsdUJBQWY7QUFDQSxTQUFLQyxJQUFMLEdBQVksbUNBQVo7O0FBRUEsU0FBS0MsUUFBTCxHQUFnQixLQUFLQywwQkFBTCxLQUNaLHNDQUE0QlIsU0FBNUIsRUFBdUNFLFNBQXZDLEVBQWtELEtBQUtILFVBQXZELENBRFksR0FFWix1Q0FBNkIsS0FBS0EsVUFBbEMsQ0FGSjtBQUdBLFNBQUtVLGtCQUFMLEdBQTBCLGtDQUExQjtBQUNBLFNBQUtDLHFCQUFMLEdBQTZCLG9DQUEwQixJQUExQixFQUFnQ0MsY0FBYztBQUN6RSxXQUFLWixVQUFMLENBQWdCYSx1QkFBaEIsQ0FBd0NELFVBQXhDO0FBQ0QsS0FGNEIsQ0FBN0I7O0FBSUEsUUFBSVIsY0FBSixFQUFvQjtBQUNsQixXQUFLSixVQUFMLENBQWdCYyxpQkFBaEIsQ0FBa0NWLGNBQWxDO0FBQ0Q7O0FBRUQ7QUFDQSxTQUFLRyxJQUFMLENBQVVRLEdBQVYsQ0FBYyxLQUFLZixVQUFMLENBQWdCZ0IsZ0JBQWhCLENBQWlDLEtBQUtDLHNCQUF0QyxDQUFkO0FBQ0EsU0FBS1YsSUFBTCxDQUFVUSxHQUFWLENBQWMsS0FBS1AsUUFBTCxDQUFjVSxXQUFkLENBQTBCQyxVQUFVO0FBQ2hELFlBQU1DLFFBQVFELE9BQU9FLEdBQVAsQ0FBV0MsS0FBS0EsRUFBRUMsT0FBRixJQUFhLGVBQUtDLElBQUwsQ0FBVUYsRUFBRXhCLFNBQVosRUFBdUJ3QixFQUFFRyxJQUFGLElBQVVILEVBQUVJLE9BQW5DLENBQTdCLENBQWQ7QUFDQSxXQUFLZixxQkFBTCxDQUEyQmdCLElBQTNCLENBQWdDLEdBQUdQLEtBQW5DO0FBQ0QsS0FIYSxDQUFkO0FBSUEsU0FBS2IsSUFBTCxDQUFVUSxHQUFWLENBQWMsS0FBS1AsUUFBTCxDQUFjb0Isd0JBQWQsQ0FBdUMsTUFBTSxLQUFLdEIsT0FBTCxDQUFhdUIsSUFBYixDQUFrQiw0QkFBbEIsQ0FBN0MsQ0FBZDtBQUNBLFNBQUt0QixJQUFMLENBQVVRLEdBQVYsQ0FBYyx5QkFBZSxNQUFNLEtBQUtKLHFCQUFMLENBQTJCbUIsT0FBM0IsRUFBckIsQ0FBZDs7QUFFQTtBQUNBLFNBQUtiLHNCQUFMLENBQTRCLEVBQUNjLE1BQU0sSUFBUCxFQUFhQyxJQUFJLEtBQUtoQyxVQUFMLENBQWdCaUMsS0FBakMsRUFBNUI7QUFDRDs7QUFFRCxTQUFPQyxNQUFQLEdBQWdCO0FBQ2QsUUFBSSxDQUFDdkMsb0JBQUwsRUFBMkI7QUFDekJBLDZCQUF1QixJQUFJd0Msb0JBQUosRUFBdkI7QUFDRDtBQUNELFdBQU94QyxvQkFBUDtBQUNEOztBQUVELFNBQU95QyxhQUFQLEdBQXVCO0FBQ3JCLFFBQUl6QyxvQkFBSixFQUEwQjtBQUN4QkEsMkJBQXFCbUMsT0FBckI7QUFDQW5DLDZCQUF1QixJQUF2QjtBQUNEO0FBQ0Y7O0FBRUQsU0FBTzBDLEtBQVAsQ0FBYXRDLE9BQWIsRUFBc0I7QUFDcEIsVUFBTXVDLG1CQUFtQnZDLFFBQVF1QyxnQkFBUixJQUE0QixDQUFyRDtBQUNBLFVBQU1DLGdCQUFnQnhDLFFBQVF3QyxhQUFSLElBQXlCLENBQS9DOztBQUVBLFVBQU1DLGFBQWNGLHFCQUFxQixDQUFyQixJQUEyQkEscUJBQXFCLENBQXJCLElBQTBCQyxrQkFBa0IsQ0FBeEUsR0FDakIsTUFBTSxxQkFBV0UsWUFBWCxFQURXLEdBRWpCLE1BQU0scUJBQVdDLFdBQVgsRUFGUjs7QUFJQSxXQUFPLElBQUk5QyxjQUFKLENBQW1CLElBQW5CLEVBQXlCLEVBQUMsQ0FBQ0gsYUFBRCxHQUFpQitDLFVBQWxCLEVBQXpCLENBQVA7QUFDRDs7QUFFRDs7Ozs7Ozs7O0FBU0F2Qix5QkFBdUIwQixPQUF2QixFQUFnQztBQUM5QixRQUFJLEtBQUt0QyxTQUFULEVBQW9CO0FBQ2xCO0FBQ0Q7O0FBRUQsUUFBSSxLQUFLTCxVQUFMLENBQWdCNEMsU0FBaEIsRUFBSixFQUFpQztBQUMvQixXQUFLcEMsUUFBTCxDQUFjcUMsS0FBZCxHQUFzQkMsSUFBdEIsQ0FBMkIsTUFBTSxLQUFLeEMsT0FBTCxDQUFhdUIsSUFBYixDQUFrQixvQkFBbEIsQ0FBakM7QUFDRCxLQUZELE1BRU8sSUFBSSxLQUFLN0IsVUFBTCxDQUFnQitDLFdBQWhCLEVBQUosRUFBbUM7QUFDeEMsV0FBS3pDLE9BQUwsQ0FBYXVCLElBQWIsQ0FBa0Isd0JBQWxCOztBQUVBLFdBQUtyQixRQUFMLENBQWNzQixPQUFkO0FBQ0Q7O0FBRUQsU0FBS3hCLE9BQUwsQ0FBYXVCLElBQWIsQ0FBa0IsNkJBQWxCLEVBQWlEYyxPQUFqRDtBQUNEOztBQUVEQyxjQUFZO0FBQ1YsV0FBTyxJQUFQO0FBQ0Q7O0FBRURHLGdCQUFjO0FBQ1osV0FBTyxLQUFLMUMsU0FBWjtBQUNEOztBQUVESSwrQkFBNkI7QUFDM0IsV0FBTyxDQUFDLENBQUN1QyxRQUFRQyxHQUFSLENBQVlDLDhCQUFkLElBQWdERixRQUFRRyxRQUFSLEtBQXFCLE9BQTVFO0FBQ0Q7O0FBRUQ7O0FBRUFDLHFCQUFtQkMsUUFBbkIsRUFBNkI7QUFDM0IsV0FBTyxLQUFLL0MsT0FBTCxDQUFhZ0QsRUFBYixDQUFnQixvQkFBaEIsRUFBc0NELFFBQXRDLENBQVA7QUFDRDs7QUFFRHpCLDJCQUF5QnlCLFFBQXpCLEVBQW1DO0FBQ2pDLFdBQU8sS0FBSy9DLE9BQUwsQ0FBYWdELEVBQWIsQ0FBZ0IsNEJBQWhCLEVBQThDRCxRQUE5QyxDQUFQO0FBQ0Q7O0FBRURFLDZCQUEyQkYsUUFBM0IsRUFBcUM7QUFDbkMsV0FBTyxLQUFLL0MsT0FBTCxDQUFhZ0QsRUFBYixDQUFnQiw2QkFBaEIsRUFBK0NELFFBQS9DLENBQVA7QUFDRDs7QUFFREcsd0JBQXNCSCxRQUF0QixFQUFnQztBQUM5QixXQUFPLEtBQUsvQyxPQUFMLENBQWFnRCxFQUFiLENBQWdCLHVCQUFoQixFQUF5Q0QsUUFBekMsQ0FBUDtBQUNEOztBQUVESSx5QkFBdUJKLFFBQXZCLEVBQWlDO0FBQy9CLFdBQU8sS0FBSy9DLE9BQUwsQ0FBYWdELEVBQWIsQ0FBZ0Isd0JBQWhCLEVBQTBDRCxRQUExQyxDQUFQO0FBQ0Q7O0FBRUQ7Ozs7QUFJQUssNEJBQTBCQyxTQUExQixFQUFxQztBQUNuQyxXQUFPLElBQUlDLE9BQUosQ0FBWUMsV0FBVztBQUM1QixZQUFNQyxNQUFNLEtBQUtQLDBCQUFMLENBQWdDLE1BQU07QUFDaEQsWUFBSSxLQUFLdkQsVUFBTCxDQUFnQitELFNBQWhCLENBQTBCSixTQUExQixDQUFKLEVBQTBDO0FBQ3hDRTtBQUNBQyxjQUFJRSxPQUFKO0FBQ0Q7QUFDRixPQUxXLENBQVo7QUFNRCxLQVBNLENBQVA7QUFRRDs7QUFFRDs7OztBQUlBQyw4QkFBNEI7QUFDMUIsV0FBTyxJQUFJTCxPQUFKLENBQVlDLFdBQVc7QUFDNUIsWUFBTUMsTUFBTSxLQUFLVixrQkFBTCxDQUF3QixNQUFNO0FBQ3hDUztBQUNBQyxZQUFJRSxPQUFKO0FBQ0QsT0FIVyxDQUFaO0FBSUQsS0FMTSxDQUFQO0FBTUQ7O0FBRURFLHdCQUFzQjtBQUNwQixXQUFPLEtBQUtwRSxTQUFaO0FBQ0Q7O0FBRURxRSxrQkFBZ0I7QUFDZCxXQUFPLEtBQUtuRSxVQUFaO0FBQ0Q7O0FBRURvRSxzQkFBb0I7QUFDbEIsV0FBTyxLQUFLNUQsUUFBWjtBQUNEOztBQUVENkQsMEJBQXdCO0FBQ3RCLFdBQU8sS0FBSzNELGtCQUFaO0FBQ0Q7O0FBRUQ7OztBQUdNb0IsU0FBTixHQUFnQjtBQUFBOztBQUFBO0FBQ2QsVUFBSSxNQUFLekIsU0FBVCxFQUFvQjtBQUNsQjtBQUNEO0FBQ0QsWUFBS0EsU0FBTCxHQUFpQixJQUFqQjs7QUFFQSxZQUFLRSxJQUFMLENBQVV5RCxPQUFWO0FBQ0EsWUFBS2hFLFVBQUwsQ0FBZ0I4QixPQUFoQjtBQUNBLFlBQUt4QixPQUFMLENBQWEwRCxPQUFiOztBQUVBLFlBQU0sTUFBS3hELFFBQUwsQ0FBY3NCLE9BQWQsRUFBTjtBQVZjO0FBV2Y7QUF6TGlDLEM7a0JBQWZsQyxjO0lBNExmdUMsb0IsR0FBTixNQUFNQSxvQkFBTixTQUFtQ3ZDLGNBQW5DLENBQWtEO0FBQ2hEQyxnQkFBYztBQUNaLFVBQU0sSUFBTixFQUFZLEVBQUMsQ0FBQ0osYUFBRCxHQUFpQixNQUFNLHFCQUFXeUMsTUFBWCxFQUF4QixFQUFaO0FBQ0Q7O0FBRURVLGNBQVk7QUFDVixXQUFPLEtBQVA7QUFDRDtBQVArQyxDIiwiZmlsZSI6IndvcmtkaXItY29udGV4dC5qcyIsInNvdXJjZVJvb3QiOiIvaG9tZS90cmF2aXMvYnVpbGQvYXRvbS9hdG9tL291dC9hcHAvbm9kZV9tb2R1bGVzL2dpdGh1YiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBwYXRoIGZyb20gJ3BhdGgnO1xuXG5pbXBvcnQge0VtaXR0ZXIsIENvbXBvc2l0ZURpc3Bvc2FibGUsIERpc3Bvc2FibGV9IGZyb20gJ2V2ZW50LWtpdCc7XG5pbXBvcnQge2F1dG9iaW5kfSBmcm9tICdjb3JlLWRlY29yYXRvcnMnO1xuXG5pbXBvcnQgRGVmZXJyZWRDYWxsYmFja1F1ZXVlIGZyb20gJy4uL2RlZmVycmVkLWNhbGxiYWNrLXF1ZXVlJztcbmltcG9ydCBSZXBvc2l0b3J5IGZyb20gJy4vcmVwb3NpdG9yeSc7XG5pbXBvcnQgUmVzb2x1dGlvblByb2dyZXNzIGZyb20gJy4vY29uZmxpY3RzL3Jlc29sdXRpb24tcHJvZ3Jlc3MnO1xuaW1wb3J0IEZpbGVTeXN0ZW1DaGFuZ2VPYnNlcnZlciBmcm9tICcuL2ZpbGUtc3lzdGVtLWNoYW5nZS1vYnNlcnZlcic7XG5pbXBvcnQgV29ya3NwYWNlQ2hhbmdlT2JzZXJ2ZXIgZnJvbSAnLi93b3Jrc3BhY2UtY2hhbmdlLW9ic2VydmVyJztcblxuY29uc3QgY3JlYXRlUmVwb1N5bSA9IFN5bWJvbCgnY3JlYXRlUmVwbycpO1xuXG5sZXQgYWJzZW50V29ya2RpckNvbnRleHQ7XG5cbi8qXG4gKiBCdW5kbGUgb2YgbW9kZWwgb2JqZWN0cyBhc3NvY2lhdGVkIHdpdGggYSBnaXQgd29ya2luZyBkaXJlY3RvcnkuXG4gKlxuICogUHJvdmlkZXMgc3luY2hyb25vdXMgYWNjZXNzIHRvIGVhY2ggbW9kZWwgaW4gdGhlIGZvcm0gb2YgYSBnZXR0ZXIgbWV0aG9kIHRoYXQgcmV0dXJucyB0aGUgbW9kZWwgb3IgYG51bGxgIGlmIGl0XG4gKiBoYXMgbm90IHlldCBiZWVuIGluaXRpYWxpemVkLCBhbmQgYXN5bmNocm9ub3VzIGFjY2VzcyBpbiB0aGUgZm9ybSBvZiBhIFByb21pc2UgZ2VuZXJhdGlvbiBtZXRob2QgdGhhdCB3aWxsIHJlc29sdmVcbiAqIG9uY2UgdGhlIG1vZGVsIGlzIGF2YWlsYWJsZS4gSW5pdGlhbGl6ZXMgdGhlIHBsYXRmb3JtLWFwcHJvcHJpYXRlIGNoYW5nZSBvYnNlcnZlciBhbmQgcHJveGllcyBzZWxlY3QgZmlsZXN5c3RlbVxuICogY2hhbmdlIGV2ZW50cy5cbiAqL1xuZXhwb3J0IGRlZmF1bHQgY2xhc3MgV29ya2RpckNvbnRleHQge1xuXG4gIC8qXG4gICAqIEF2YWlsYWJsZSBvcHRpb25zOlxuICAgKiAtIGBvcHRpb25zLndpbmRvd2A6IEJyb3dzZXIgd2luZG93IGdsb2JhbCwgdXNlZCBvbiBMaW51eCBieSB0aGUgV29ya3NwYWNlQ2hhbmdlT2JzZXJ2ZXIuXG4gICAqIC0gYG9wdGlvbnMud29ya3NwYWNlYDogQXRvbSdzIHdvcmtzcGFjZSBzaW5nbGV0b24sIHVzZWQgb24gTGludXggYnkgdGhlIFdvcmtzcGFjZUNoYW5nZU9ic2VydmVyLlxuICAgKiAtIGBvcHRpb25zLnByb21wdENhbGxiYWNrYDogQ2FsbGJhY2sgdXNlZCB0byBjb2xsZWN0IGluZm9ybWF0aW9uIGludGVyYWN0aXZlbHkgdGhyb3VnaCBBdG9tLlxuICAgKi9cbiAgY29uc3RydWN0b3IoZGlyZWN0b3J5LCBvcHRpb25zID0ge30pIHtcbiAgICB0aGlzLmRpcmVjdG9yeSA9IGRpcmVjdG9yeTtcbiAgICB0aGlzLnJlcG9zaXRvcnkgPSAob3B0aW9uc1tjcmVhdGVSZXBvU3ltXSB8fCAoKCkgPT4gbmV3IFJlcG9zaXRvcnkoZGlyZWN0b3J5KSkpKCk7XG5cbiAgICBjb25zdCB7d2luZG93OiB0aGVXaW5kb3csIHdvcmtzcGFjZSwgcHJvbXB0Q2FsbGJhY2t9ID0gb3B0aW9ucztcblxuICAgIHRoaXMuZGVzdHJveWVkID0gZmFsc2U7XG4gICAgdGhpcy5lbWl0dGVyID0gbmV3IEVtaXR0ZXIoKTtcbiAgICB0aGlzLnN1YnMgPSBuZXcgQ29tcG9zaXRlRGlzcG9zYWJsZSgpO1xuXG4gICAgdGhpcy5vYnNlcnZlciA9IHRoaXMudXNlV29ya3NwYWNlQ2hhbmdlT2JzZXJ2ZXIoKVxuICAgICAgPyBuZXcgV29ya3NwYWNlQ2hhbmdlT2JzZXJ2ZXIodGhlV2luZG93LCB3b3Jrc3BhY2UsIHRoaXMucmVwb3NpdG9yeSlcbiAgICAgIDogbmV3IEZpbGVTeXN0ZW1DaGFuZ2VPYnNlcnZlcih0aGlzLnJlcG9zaXRvcnkpO1xuICAgIHRoaXMucmVzb2x1dGlvblByb2dyZXNzID0gbmV3IFJlc29sdXRpb25Qcm9ncmVzcygpO1xuICAgIHRoaXMuZGVmZXJyZWRDYWxsYmFja1F1ZXVlID0gbmV3IERlZmVycmVkQ2FsbGJhY2tRdWV1ZSgzMDAwLCBjb2xsZWN0aW9uID0+IHtcbiAgICAgIHRoaXMucmVwb3NpdG9yeS5vYnNlcnZlRmlsZXN5c3RlbUNoYW5nZShjb2xsZWN0aW9uKTtcbiAgICB9KTtcblxuICAgIGlmIChwcm9tcHRDYWxsYmFjaykge1xuICAgICAgdGhpcy5yZXBvc2l0b3J5LnNldFByb21wdENhbGxiYWNrKHByb21wdENhbGxiYWNrKTtcbiAgICB9XG5cbiAgICAvLyBXaXJlIHVwIGV2ZW50IGZvcndhcmRpbmcgYW1vbmcgbW9kZWxzXG4gICAgdGhpcy5zdWJzLmFkZCh0aGlzLnJlcG9zaXRvcnkub25EaWRDaGFuZ2VTdGF0ZSh0aGlzLnJlcG9zaXRvcnlDaGFuZ2VkU3RhdGUpKTtcbiAgICB0aGlzLnN1YnMuYWRkKHRoaXMub2JzZXJ2ZXIub25EaWRDaGFuZ2UoZXZlbnRzID0+IHtcbiAgICAgIGNvbnN0IHBhdGhzID0gZXZlbnRzLm1hcChlID0+IGUuc3BlY2lhbCB8fCBwYXRoLmpvaW4oZS5kaXJlY3RvcnksIGUuZmlsZSB8fCBlLm5ld0ZpbGUpKTtcbiAgICAgIHRoaXMuZGVmZXJyZWRDYWxsYmFja1F1ZXVlLnB1c2goLi4ucGF0aHMpO1xuICAgIH0pKTtcbiAgICB0aGlzLnN1YnMuYWRkKHRoaXMub2JzZXJ2ZXIub25EaWRDaGFuZ2VXb3JrZGlyT3JIZWFkKCgpID0+IHRoaXMuZW1pdHRlci5lbWl0KCdkaWQtY2hhbmdlLXdvcmtkaXItb3ItaGVhZCcpKSk7XG4gICAgdGhpcy5zdWJzLmFkZChuZXcgRGlzcG9zYWJsZSgoKSA9PiB0aGlzLmRlZmVycmVkQ2FsbGJhY2tRdWV1ZS5kZXN0cm95KCkpKTtcblxuICAgIC8vIElmIGEgcHJlLWxvYWRlZCBSZXBvc2l0b3J5IHdhcyBwcm92aWRlZCwgYnJvYWRjYXN0IGFuIGluaXRpYWwgc3RhdGUgY2hhbmdlIGV2ZW50LlxuICAgIHRoaXMucmVwb3NpdG9yeUNoYW5nZWRTdGF0ZSh7ZnJvbTogbnVsbCwgdG86IHRoaXMucmVwb3NpdG9yeS5zdGF0ZX0pO1xuICB9XG5cbiAgc3RhdGljIGFic2VudCgpIHtcbiAgICBpZiAoIWFic2VudFdvcmtkaXJDb250ZXh0KSB7XG4gICAgICBhYnNlbnRXb3JrZGlyQ29udGV4dCA9IG5ldyBBYnNlbnRXb3JrZGlyQ29udGV4dCgpO1xuICAgIH1cbiAgICByZXR1cm4gYWJzZW50V29ya2RpckNvbnRleHQ7XG4gIH1cblxuICBzdGF0aWMgZGVzdHJveUFic2VudCgpIHtcbiAgICBpZiAoYWJzZW50V29ya2RpckNvbnRleHQpIHtcbiAgICAgIGFic2VudFdvcmtkaXJDb250ZXh0LmRlc3Ryb3koKTtcbiAgICAgIGFic2VudFdvcmtkaXJDb250ZXh0ID0gbnVsbDtcbiAgICB9XG4gIH1cblxuICBzdGF0aWMgZ3Vlc3Mob3B0aW9ucykge1xuICAgIGNvbnN0IHByb2plY3RQYXRoQ291bnQgPSBvcHRpb25zLnByb2plY3RQYXRoQ291bnQgfHwgMDtcbiAgICBjb25zdCBpbml0UGF0aENvdW50ID0gb3B0aW9ucy5pbml0UGF0aENvdW50IHx8IDA7XG5cbiAgICBjb25zdCBjcmVhdGVSZXBvID0gKHByb2plY3RQYXRoQ291bnQgPT09IDEgfHwgKHByb2plY3RQYXRoQ291bnQgPT09IDAgJiYgaW5pdFBhdGhDb3VudCA9PT0gMSkpID9cbiAgICAgICgpID0+IFJlcG9zaXRvcnkubG9hZGluZ0d1ZXNzKCkgOlxuICAgICAgKCkgPT4gUmVwb3NpdG9yeS5hYnNlbnRHdWVzcygpO1xuXG4gICAgcmV0dXJuIG5ldyBXb3JrZGlyQ29udGV4dChudWxsLCB7W2NyZWF0ZVJlcG9TeW1dOiBjcmVhdGVSZXBvfSk7XG4gIH1cblxuICAvKipcbiAgICogUmVzcG9uZCB0byBjaGFuZ2VzIGluIGBSZXBvc2l0b3J5YCBzdGF0ZS4gTG9hZCByZXNvbHV0aW9uIHByb2dyZXNzIGFuZCBzdGFydCB0aGUgY2hhbmdlIG9ic2VydmVyIHdoZW4gaXQgYmVjb21lc1xuICAgKiBwcmVzZW50LiBTdG9wIHRoZSBjaGFuZ2Ugb2JzZXJ2ZXIgd2hlbiBpdCBpcyBkZXN0cm95ZWQuIFJlLWJyb2FkY2FzdCB0aGUgZXZlbnQgdG8gY29udGV4dCBzdWJzY3JpYmVyc1xuICAgKiByZWdhcmRsZXNzLlxuICAgKlxuICAgKiBUaGUgUmVzb2x1dGlvblByb2dyZXNzIHdpbGwgYmUgbG9hZGVkIGJlZm9yZSB0aGUgY2hhbmdlIGV2ZW50IGlzIHJlLWJyb2FkY2FzdCwgYnV0IGNoYW5nZSBvYnNlcnZlciBtb2RpZmljYXRpb25zXG4gICAqIHdpbGwgbm90IGJlIGNvbXBsZXRlLlxuICAgKi9cbiAgQGF1dG9iaW5kXG4gIHJlcG9zaXRvcnlDaGFuZ2VkU3RhdGUocGF5bG9hZCkge1xuICAgIGlmICh0aGlzLmRlc3Ryb3llZCkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGlmICh0aGlzLnJlcG9zaXRvcnkuaXNQcmVzZW50KCkpIHtcbiAgICAgIHRoaXMub2JzZXJ2ZXIuc3RhcnQoKS50aGVuKCgpID0+IHRoaXMuZW1pdHRlci5lbWl0KCdkaWQtc3RhcnQtb2JzZXJ2ZXInKSk7XG4gICAgfSBlbHNlIGlmICh0aGlzLnJlcG9zaXRvcnkuaXNEZXN0cm95ZWQoKSkge1xuICAgICAgdGhpcy5lbWl0dGVyLmVtaXQoJ2RpZC1kZXN0cm95LXJlcG9zaXRvcnknKTtcblxuICAgICAgdGhpcy5vYnNlcnZlci5kZXN0cm95KCk7XG4gICAgfVxuXG4gICAgdGhpcy5lbWl0dGVyLmVtaXQoJ2RpZC1jaGFuZ2UtcmVwb3NpdG9yeS1zdGF0ZScsIHBheWxvYWQpO1xuICB9XG5cbiAgaXNQcmVzZW50KCkge1xuICAgIHJldHVybiB0cnVlO1xuICB9XG5cbiAgaXNEZXN0cm95ZWQoKSB7XG4gICAgcmV0dXJuIHRoaXMuZGVzdHJveWVkO1xuICB9XG5cbiAgdXNlV29ya3NwYWNlQ2hhbmdlT2JzZXJ2ZXIoKSB7XG4gICAgcmV0dXJuICEhcHJvY2Vzcy5lbnYuQVRPTV9HSVRIVUJfV09SS1NQQUNFX09CU0VSVkVSIHx8IHByb2Nlc3MucGxhdGZvcm0gPT09ICdsaW51eCc7XG4gIH1cblxuICAvLyBFdmVudCBzdWJzY3JpcHRpb25zXG5cbiAgb25EaWRTdGFydE9ic2VydmVyKGNhbGxiYWNrKSB7XG4gICAgcmV0dXJuIHRoaXMuZW1pdHRlci5vbignZGlkLXN0YXJ0LW9ic2VydmVyJywgY2FsbGJhY2spO1xuICB9XG5cbiAgb25EaWRDaGFuZ2VXb3JrZGlyT3JIZWFkKGNhbGxiYWNrKSB7XG4gICAgcmV0dXJuIHRoaXMuZW1pdHRlci5vbignZGlkLWNoYW5nZS13b3JrZGlyLW9yLWhlYWQnLCBjYWxsYmFjayk7XG4gIH1cblxuICBvbkRpZENoYW5nZVJlcG9zaXRvcnlTdGF0ZShjYWxsYmFjaykge1xuICAgIHJldHVybiB0aGlzLmVtaXR0ZXIub24oJ2RpZC1jaGFuZ2UtcmVwb3NpdG9yeS1zdGF0ZScsIGNhbGxiYWNrKTtcbiAgfVxuXG4gIG9uRGlkVXBkYXRlUmVwb3NpdG9yeShjYWxsYmFjaykge1xuICAgIHJldHVybiB0aGlzLmVtaXR0ZXIub24oJ2RpZC11cGRhdGUtcmVwb3NpdG9yeScsIGNhbGxiYWNrKTtcbiAgfVxuXG4gIG9uRGlkRGVzdHJveVJlcG9zaXRvcnkoY2FsbGJhY2spIHtcbiAgICByZXR1cm4gdGhpcy5lbWl0dGVyLm9uKCdkaWQtZGVzdHJveS1yZXBvc2l0b3J5JywgY2FsbGJhY2spO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiBhIFByb21pc2UgdGhhdCB3aWxsIHJlc29sdmUgdGhlIG5leHQgdGltZSB0aGF0IGEgUmVwb3NpdG9yeSB0cmFuc2l0aW9ucyB0byB0aGUgcmVxdWVzdGVkIHN0YXRlLiBNb3N0XG4gICAqIHVzZWZ1bCBmb3IgdGVzdCBjYXNlczsgbW9zdCBjYWxsZXJzIHNob3VsZCBwcmVmZXIgc3Vic2NyaWJpbmcgdG8gYG9uRGlkQ2hhbmdlUmVwb3NpdG9yeVN0YXRlYC5cbiAgICovXG4gIGdldFJlcG9zaXRvcnlTdGF0ZVByb21pc2Uoc3RhdGVOYW1lKSB7XG4gICAgcmV0dXJuIG5ldyBQcm9taXNlKHJlc29sdmUgPT4ge1xuICAgICAgY29uc3Qgc3ViID0gdGhpcy5vbkRpZENoYW5nZVJlcG9zaXRvcnlTdGF0ZSgoKSA9PiB7XG4gICAgICAgIGlmICh0aGlzLnJlcG9zaXRvcnkuaXNJblN0YXRlKHN0YXRlTmFtZSkpIHtcbiAgICAgICAgICByZXNvbHZlKCk7XG4gICAgICAgICAgc3ViLmRpc3Bvc2UoKTtcbiAgICAgICAgfVxuICAgICAgfSk7XG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJuIGEgUHJvbWlzZSB0aGF0IHdpbGwgcmVzb2x2ZSB0aGUgbmV4dCB0aW1lIHRoYXQgYSBDaGFuZ2VPYnNlcnZlciBzdWNjZXNzZnVsbHkgc3RhcnRzLiBNb3N0IHVzZWZ1bCBmb3JcbiAgICogdGVzdCBjYXNlcy5cbiAgICovXG4gIGdldE9ic2VydmVyU3RhcnRlZFByb21pc2UoKSB7XG4gICAgcmV0dXJuIG5ldyBQcm9taXNlKHJlc29sdmUgPT4ge1xuICAgICAgY29uc3Qgc3ViID0gdGhpcy5vbkRpZFN0YXJ0T2JzZXJ2ZXIoKCkgPT4ge1xuICAgICAgICByZXNvbHZlKCk7XG4gICAgICAgIHN1Yi5kaXNwb3NlKCk7XG4gICAgICB9KTtcbiAgICB9KTtcbiAgfVxuXG4gIGdldFdvcmtpbmdEaXJlY3RvcnkoKSB7XG4gICAgcmV0dXJuIHRoaXMuZGlyZWN0b3J5O1xuICB9XG5cbiAgZ2V0UmVwb3NpdG9yeSgpIHtcbiAgICByZXR1cm4gdGhpcy5yZXBvc2l0b3J5O1xuICB9XG5cbiAgZ2V0Q2hhbmdlT2JzZXJ2ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMub2JzZXJ2ZXI7XG4gIH1cblxuICBnZXRSZXNvbHV0aW9uUHJvZ3Jlc3MoKSB7XG4gICAgcmV0dXJuIHRoaXMucmVzb2x1dGlvblByb2dyZXNzO1xuICB9XG5cbiAgLypcbiAgICogQ2xlYW5seSBkZXN0cm95IGFueSBtb2RlbHMgdGhhdCBuZWVkIHRvIGJlIGNsZWFuZWQsIGluY2x1ZGluZyBzdG9wcGluZyB0aGUgZmlsZXN5c3RlbSB3YXRjaGVyLlxuICAgKi9cbiAgYXN5bmMgZGVzdHJveSgpIHtcbiAgICBpZiAodGhpcy5kZXN0cm95ZWQpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgdGhpcy5kZXN0cm95ZWQgPSB0cnVlO1xuXG4gICAgdGhpcy5zdWJzLmRpc3Bvc2UoKTtcbiAgICB0aGlzLnJlcG9zaXRvcnkuZGVzdHJveSgpO1xuICAgIHRoaXMuZW1pdHRlci5kaXNwb3NlKCk7XG5cbiAgICBhd2FpdCB0aGlzLm9ic2VydmVyLmRlc3Ryb3koKTtcbiAgfVxufVxuXG5jbGFzcyBBYnNlbnRXb3JrZGlyQ29udGV4dCBleHRlbmRzIFdvcmtkaXJDb250ZXh0IHtcbiAgY29uc3RydWN0b3IoKSB7XG4gICAgc3VwZXIobnVsbCwge1tjcmVhdGVSZXBvU3ltXTogKCkgPT4gUmVwb3NpdG9yeS5hYnNlbnQoKX0pO1xuICB9XG5cbiAgaXNQcmVzZW50KCkge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxufVxuIl19