'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.CommitError = exports.AbortMergeError = undefined;

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _eventKit = require('event-kit');

var _compositeGitStrategy = require('../composite-git-strategy');

var _compositeGitStrategy2 = _interopRequireDefault(_compositeGitStrategy);

var _helpers = require('../helpers');

var _remote = require('./remote');

var _remote2 = _interopRequireDefault(_remote);

var _repositoryStates = require('./repository-states');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _asyncToGenerator(fn) { return function () { var gen = fn.apply(this, arguments); return new Promise(function (resolve, reject) { function step(key, arg) { try { var info = gen[key](arg); var value = info.value; } catch (error) { reject(error); return; } if (info.done) { resolve(value); } else { return Promise.resolve(value).then(function (value) { step("next", value); }, function (err) { step("throw", err); }); } } return step("next"); }); }; }

const MERGE_MARKER_REGEX = /^(>|<){7} \S+$/m;

// Internal option keys used to designate the desired initial state of a Repository.
const initialStateSym = Symbol('initialState');

let AbortMergeError = exports.AbortMergeError = class AbortMergeError extends Error {
  constructor(code, filePath) {
    super();
    this.message = `${code}: ${filePath}.`;
    this.code = code;
    this.path = filePath;
    this.stack = new Error().stack;
  }
};
let CommitError = exports.CommitError = class CommitError extends Error {
  constructor(code) {
    super();
    this.message = `Commit error: ${code}.`;
    this.code = code;
    this.stack = new Error().stack;
  }
};
let Repository = class Repository {
  constructor(workingDirectoryPath) {
    let gitStrategy = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;
    let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

    this.workingDirectoryPath = workingDirectoryPath;
    this.git = gitStrategy || _compositeGitStrategy2.default.create(workingDirectoryPath);

    this.emitter = new _eventKit.Emitter();

    this.loadPromise = new Promise(resolve => {
      const sub = this.onDidChangeState(() => {
        if (!this.isLoading()) {
          resolve();
          sub.dispose();
        } else if (this.isDestroyed()) {
          sub.dispose();
        }
      });
    });

    this.transitionTo(options[initialStateSym] || _repositoryStates.Loading);
  }

  static absent() {
    return new Repository(null, null, { [initialStateSym]: _repositoryStates.Absent });
  }

  static loadingGuess() {
    return new Repository(null, null, { [initialStateSym]: _repositoryStates.LoadingGuess });
  }

  static absentGuess() {
    return new Repository(null, null, { [initialStateSym]: _repositoryStates.AbsentGuess });
  }

  // State management //////////////////////////////////////////////////////////////////////////////////////////////////

  transition(currentState, StateConstructor) {
    if (currentState !== this.state) {
      // Attempted transition from a non-active state, most likely from an asynchronous start() method.
      return Promise.resolve();
    }

    for (var _len = arguments.length, payload = Array(_len > 2 ? _len - 2 : 0), _key = 2; _key < _len; _key++) {
      payload[_key - 2] = arguments[_key];
    }

    const nextState = new StateConstructor(this, ...payload);
    this.state = nextState;

    this.emitter.emit('did-change-state', { from: currentState, to: this.state });
    if (!this.isDestroyed()) {
      this.emitter.emit('did-update');
    }

    return this.state.start();
  }

  transitionTo(StateConstructor) {
    for (var _len2 = arguments.length, payload = Array(_len2 > 1 ? _len2 - 1 : 0), _key2 = 1; _key2 < _len2; _key2++) {
      payload[_key2 - 1] = arguments[_key2];
    }

    return this.transition(this.state, StateConstructor, ...payload);
  }

  getLoadPromise() {
    return this.isAbsent() ? Promise.reject(new Error('An absent repository will never load')) : this.loadPromise;
  }

  /*
   * Use `callback` to request user input from all git strategies.
   */
  setPromptCallback(callback) {
    this.git.getImplementers().forEach(strategy => strategy.setPromptCallback(callback));
  }

  // Event subscription ////////////////////////////////////////////////////////////////////////////////////////////////

  onDidDestroy(callback) {
    return this.emitter.on('did-destroy', callback);
  }

  onDidChangeState(callback) {
    return this.emitter.on('did-change-state', callback);
  }

  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  // State-independent actions /////////////////////////////////////////////////////////////////////////////////////////
  // Actions that use direct filesystem access or otherwise don't need `this.git` to be available.

  pathHasMergeMarkers(relativePath) {
    var _this = this;

    return _asyncToGenerator(function* () {
      try {
        const contents = yield (0, _helpers.readFile)(_path2.default.join(_this.getWorkingDirectoryPath(), relativePath), 'utf8');
        return MERGE_MARKER_REGEX.test(contents);
      } catch (e) {
        // EISDIR implies this is a submodule
        if (e.code === 'ENOENT' || e.code === 'EISDIR') {
          return false;
        } else {
          throw e;
        }
      }
    })();
  }

  getMergeMessage() {
    var _this2 = this;

    return _asyncToGenerator(function* () {
      try {
        const contents = yield (0, _helpers.readFile)(_path2.default.join(_this2.getGitDirectoryPath(), 'MERGE_MSG'), 'utf8');
        return contents;
      } catch (e) {
        return null;
      }
    })();
  }

  // State-independent accessors ///////////////////////////////////////////////////////////////////////////////////////

  getWorkingDirectoryPath() {
    return this.workingDirectoryPath;
  }

  setGitDirectoryPath(gitDirectoryPath) {
    this._gitDirectoryPath = gitDirectoryPath;
  }

  getGitDirectoryPath() {
    if (this._gitDirectoryPath) {
      return this._gitDirectoryPath;
    } else if (this.getWorkingDirectoryPath()) {
      return _path2.default.join(this.getWorkingDirectoryPath(), '.git');
    } else {
      // Absent/Loading/etc.
      return null;
    }
  }

  isInState(stateName) {
    return this.state.constructor.name === stateName;
  }

  toString() {
    return `Repository(state=${this.state.constructor.name}, workdir="${this.getWorkingDirectoryPath()}")`;
  }

  // Compound Getters //////////////////////////////////////////////////////////////////////////////////////////////////
  // Accessor methods for data derived from other, state-provided getters.

  getUnstagedChanges() {
    var _this3 = this;

    return _asyncToGenerator(function* () {
      var _ref = yield _this3.getStatusBundle();

      const unstagedFiles = _ref.unstagedFiles;

      return Object.keys(unstagedFiles).map(function (filePath) {
        return { filePath, status: unstagedFiles[filePath] };
      });
    })();
  }

  getStagedChanges() {
    var _this4 = this;

    return _asyncToGenerator(function* () {
      var _ref2 = yield _this4.getStatusBundle();

      const stagedFiles = _ref2.stagedFiles;

      return Object.keys(stagedFiles).map(function (filePath) {
        return { filePath, status: stagedFiles[filePath] };
      });
    })();
  }

  getMergeConflicts() {
    var _this5 = this;

    return _asyncToGenerator(function* () {
      var _ref3 = yield _this5.getStatusBundle();

      const mergeConflictFiles = _ref3.mergeConflictFiles;

      return Object.keys(mergeConflictFiles).map(function (filePath) {
        return { filePath, status: mergeConflictFiles[filePath] };
      });
    })();
  }

  isPartiallyStaged(fileName) {
    var _this6 = this;

    return _asyncToGenerator(function* () {
      var _ref4 = yield _this6.getStatusBundle();

      const unstagedFiles = _ref4.unstagedFiles,
            stagedFiles = _ref4.stagedFiles;

      const u = unstagedFiles[fileName];
      const s = stagedFiles[fileName];
      return u === 'modified' && s === 'modified' || u === 'modified' && s === 'added' || u === 'added' && s === 'deleted' || u === 'deleted' && s === 'modified';
    })();
  }

  getRemoteForBranch(branchName) {
    var _this7 = this;

    return _asyncToGenerator(function* () {
      const name = yield _this7.getConfig(`branch.${branchName}.remote`);
      if (name === null) {
        return _remote.nullRemote;
      } else {
        return new _remote2.default(name);
      }
    })();
  }

  saveDiscardHistory() {
    var _this8 = this;

    return _asyncToGenerator(function* () {
      if (_this8.isDestroyed()) {
        return;
      }

      const historySha = yield _this8.createDiscardHistoryBlob();
      if (_this8.isDestroyed()) {
        return;
      }
      yield _this8.setConfig('atomGithub.historySha', historySha);
    })();
  }
};

// The methods named here will be delegated to the current State.
//
// This list should match the methods decorated with @shouldDelegate in `lib/models/repository-states/state.js`. A test
// case in `test/models/repository.test.js` ensures that these sets match.
//
// Duplicated here rather than just using `expectedDelegates` directly so that this file is grep-friendly for answering
// the question of "what all can a Repository do exactly".

exports.default = Repository;
const delegates = ['isLoadingGuess', 'isAbsentGuess', 'isAbsent', 'isLoading', 'isEmpty', 'isPresent', 'isDestroyed', 'isUndetermined', 'showGitTabInit', 'showGitTabInitInProgress', 'showGitTabLoading', 'showStatusBarTiles', 'hasDirectory', 'init', 'clone', 'destroy', 'refresh', 'observeFilesystemChange', 'stageFiles', 'unstageFiles', 'stageFilesFromParentCommit', 'applyPatchToIndex', 'applyPatchToWorkdir', 'commit', 'merge', 'abortMerge', 'checkoutSide', 'mergeFile', 'writeMergeConflictToIndex', 'checkout', 'checkoutPathsAtRevision', 'fetch', 'pull', 'push', 'setConfig', 'createBlob', 'expandBlobToFile', 'createDiscardHistoryBlob', 'updateDiscardHistory', 'storeBeforeAndAfterBlobs', 'restoreLastDiscardInTempFiles', 'popDiscardHistory', 'clearDiscardHistory', 'discardWorkDirChangesForPaths', 'getStatusBundle', 'getStatusesForChangedFiles', 'getStagedChangesSinceParentCommit', 'getFilePatchForPath', 'readFileFromIndex', 'getLastCommit', 'getBranches', 'getCurrentBranch', 'isMerging', 'isRebasing', 'getRemotes', 'getAheadCount', 'getBehindCount', 'getConfig', 'unsetConfig', 'getBlobContents', 'hasDiscardHistory', 'getDiscardHistory', 'getLastHistorySnapshots'];

for (let i = 0; i < delegates.length; i++) {
  const delegate = delegates[i];

  Repository.prototype[delegate] = function () {
    return this.state[delegate](...arguments);
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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