'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
/*
 * Input adapter to facilitate parsing conflicts from text loaded into an Editor.
 */
let EditorAdapter = exports.EditorAdapter = class EditorAdapter {
  constructor(editor, startRow) {
    this.editor = editor;
    this.currentRow = startRow;
  }

  getCurrentRow() {
    return this.currentRow;
  }

  getCurrentLine() {
    return this.editor.lineTextForBufferRow(this.currentRow);
  }

  advanceRow() {
    this.currentRow++;
  }

  isAtEnd() {
    return this.currentRow > this.editor.getLastBufferRow();
  }
};

/*
 * Input adapter for parsing conflicts from a chunk of text arriving from a ReadStream.
 */

let ChunkAdapter = exports.ChunkAdapter = class ChunkAdapter {
  constructor(chunk) {
    this.chunk = chunk;

    this.lineEndRx = /\r?\n/g;

    this.lineStartPosition = 0;
    this.eof = false;
    this.advanceRow();
  }

  advanceTo(pattern) {
    if (this.eof) {
      return false;
    }

    const rx = new RegExp(pattern.source, 'gm');
    rx.lastIndex = this.lineStartPosition;

    const match = rx.exec(this.chunk);
    if (match) {
      this.lineEndRx.lastIndex = match.index;
      return true;
    } else {
      return false;
    }
  }

  getCurrentRow() {
    return undefined;
  }

  getCurrentLine() {
    return this.currentLine;
  }

  advanceRow() {
    this.lineStartPosition = this.lineEndRx.lastIndex;
    if (this.lineEndRx.test(this.chunk)) {
      this.currentLine = this.chunk.slice(this.lineStartPosition, this.lineEndRx.lastIndex);
    } else {
      this.currentLine = this.chunk.slice(this.lineStartPosition);
      this.eof = true;
    }
  }

  isAtEnd() {
    return this.eof;
  }

  getLastPartialMarker() {
    const match = /[<|>=]{1,7} ?[^\r\n]*\r?$/.exec(this.chunk);
    return match ? match[0] : '';
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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