'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _desc, _value, _class, _class2, _temp;

var _eventKit = require('event-kit');

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _coreDecorators = require('core-decorators');

var _compareSets2 = require('compare-sets');

var _compareSets3 = _interopRequireDefault(_compareSets2);

var _commands = require('../views/commands');

var _commands2 = _interopRequireDefault(_commands);

var _conflict = require('../models/conflicts/conflict');

var _conflict2 = _interopRequireDefault(_conflict);

var _conflictController = require('./conflict-controller');

var _conflictController2 = _interopRequireDefault(_conflictController);

var _source = require('../models/conflicts/source');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _toArray(arr) { return Array.isArray(arr) ? arr : Array.from(arr); }

function _applyDecoratedDescriptor(target, property, decorators, descriptor, context) {
  var desc = {};
  Object['ke' + 'ys'](descriptor).forEach(function (key) {
    desc[key] = descriptor[key];
  });
  desc.enumerable = !!desc.enumerable;
  desc.configurable = !!desc.configurable;

  if ('value' in desc || desc.initializer) {
    desc.writable = true;
  }

  desc = decorators.slice().reverse().reduce(function (desc, decorator) {
    return decorator(target, property, desc) || desc;
  }, desc);

  if (context && desc.initializer !== void 0) {
    desc.value = desc.initializer ? desc.initializer.call(context) : void 0;
    desc.initializer = undefined;
  }

  if (desc.initializer === void 0) {
    Object['define' + 'Property'](target, property, desc);
    desc = null;
  }

  return desc;
}

/**
 * Render a `ConflictController` for each conflict marker within an open TextEditor.
 */
let EditorConflictController = (_class = (_temp = _class2 = class EditorConflictController extends _react2.default.Component {

  constructor(props, context) {
    super(props, context);

    // this.layer = props.editor.addMarkerLayer({
    //   maintainHistory: true,
    //   persistent: false,
    // });

    this.layer = props.editor.getDefaultMarkerLayer();

    this.state = {
      conflicts: new Set(_conflict2.default.allFromEditor(props.editor, this.layer, props.isRebase))
    };

    this.subscriptions = new _eventKit.CompositeDisposable();

    this.updateMarkerCount();
  }

  componentDidMount() {
    const buffer = this.props.editor.getBuffer();

    this.subscriptions.add(this.props.editor.onDidStopChanging(() => this.forceUpdate()), this.props.editor.onDidDestroy(() => this.props.refreshResolutionProgress(this.props.editor.getPath())), buffer.onDidReload(() => this.reparseConflicts()));
  }

  render() {
    this.updateMarkerCount();

    return _react2.default.createElement(
      'div',
      null,
      this.state.conflicts.size > 0 && _react2.default.createElement(
        _commands2.default,
        { registry: this.props.commandRegistry, target: 'atom-text-editor' },
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-ours', callback: this.getResolverUsing([_source.OURS]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-theirs', callback: this.getResolverUsing([_source.THEIRS]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-base', callback: this.getResolverUsing([_source.BASE]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-ours-then-theirs', callback: this.getResolverUsing([_source.OURS, _source.THEIRS]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-theirs-then-ours', callback: this.getResolverUsing([_source.THEIRS, _source.OURS]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-current', callback: this.resolveAsCurrent }),
        _react2.default.createElement(_commands.Command, { command: 'github:revert-conflict-modifications', callback: this.revertConflictModifications }),
        _react2.default.createElement(_commands.Command, { command: 'github:dismiss-conflict', callback: this.dismissCurrent })
      ),
      Array.from(this.state.conflicts, c => _react2.default.createElement(_conflictController2.default, {
        key: c.getKey(),
        editor: this.props.editor,
        conflict: c,
        resolveAsSequence: sources => this.resolveAsSequence(c, sources),
        dismiss: () => this.dismissConflicts([c])
      }))
    );
  }

  componentWillUnmount() {
    // this.layer.destroy();
    this.subscriptions.dispose();
  }

  /*
   * Return an Array containing `Conflict` objects whose marked regions include any cursor position in the current
   * `TextEditor` and the `Sides` that contain a cursor within each.
   *
   * This method is written to have linear complexity with respect to the number of cursors and the number of
   * conflicts, to gracefully handle files with large numbers of both.
   */
  getCurrentConflicts() {
    const cursorPositions = this.props.editor.getCursorBufferPositions();
    cursorPositions.sort((a, b) => a.compare(b));
    const cursorIterator = cursorPositions[Symbol.iterator]();

    const conflictIterator = this.state.conflicts.keys();

    let currentCursor = cursorIterator.next();
    let currentConflict = conflictIterator.next();
    const activeConflicts = [];

    while (!currentCursor.done && !currentConflict.done) {
      // Advance currentCursor to the first cursor beyond the earliest conflict.
      const earliestConflictPosition = currentConflict.value.getRange().start;
      while (!currentCursor.done && currentCursor.value.isLessThan(earliestConflictPosition)) {
        currentCursor = cursorIterator.next();
      }

      // Advance currentConflict until the first conflict that begins at a position after the current cursor.
      // Compare each to the current cursor, and add it to activeConflicts if it contains it.
      while (!currentConflict.done && !currentCursor.done && currentConflict.value.getRange().start.isLessThan(currentCursor.value)) {
        if (currentConflict.value.includesPoint(currentCursor.value)) {
          // Hit; determine which sides of this conflict contain cursors.
          const conflict = currentConflict.value;
          const endPosition = conflict.getRange().end;
          const sides = new Set();
          while (!currentCursor.done && currentCursor.value.isLessThan(endPosition)) {
            const side = conflict.getSideContaining(currentCursor.value);
            if (side) {
              sides.add(side);
            }
            currentCursor = cursorIterator.next();
          }

          activeConflicts.push({ conflict, sides });
        }

        currentConflict = conflictIterator.next();
      }
    }

    return activeConflicts;
  }

  getResolverUsing(sequence) {
    return () => {
      this.getCurrentConflicts().forEach(match => this.resolveAsSequence(match.conflict, sequence));
    };
  }

  resolveAsCurrent() {
    this.getCurrentConflicts().forEach(match => {
      if (match.sides.size === 1) {
        const side = match.sides.keys().next().value;
        this.resolveAs(match.conflict, side.getSource());
      }
    });
  }

  revertConflictModifications() {
    this.getCurrentConflicts().forEach(match => {
      match.sides.forEach(side => {
        side.isModified() && side.revert();
        side.isBannerModified() && side.revertBanner();
      });
    });
  }

  dismissCurrent() {
    this.dismissConflicts(this.getCurrentConflicts().map(match => match.conflict));
  }

  dismissConflicts(conflicts) {
    this.setState((prevState, props) => {
      var _compareSets = (0, _compareSets3.default)(new Set(conflicts), prevState.conflicts);

      const added = _compareSets.added;

      return { conflicts: added };
    });
  }

  resolveAsSequence(conflict, sources) {
    var _sources$map$filter = sources.map(source => conflict.getSide(source)).filter(side => side),
        _sources$map$filter2 = _toArray(_sources$map$filter);

    const firstSide = _sources$map$filter2[0],
          restOfSides = _sources$map$filter2.slice(1);

    const textToAppend = restOfSides.map(side => side.getText()).join('');

    this.props.editor.transact(() => {
      // Append text from all but the first Side to the first Side. Adjust the following DisplayMarker so that only that
      // Side's marker includes the appended text, not the next one.
      const appendedRange = firstSide.appendText(textToAppend);
      const nextMarker = conflict.markerAfter(firstSide.getPosition());
      if (nextMarker) {
        nextMarker.setTailBufferPosition(appendedRange.end);
      }

      this.innerResolveAs(conflict, sources[0]);
    });
  }

  resolveAs(conflict, source) {
    this.props.editor.transact(() => {
      this.innerResolveAs(conflict, source);
    });
  }

  innerResolveAs(conflict, source) {
    conflict.resolveAs(source);

    const chosenSide = conflict.getChosenSide();
    if (!chosenSide.isBannerModified()) {
      chosenSide.deleteBanner();
    }

    const separator = conflict.getSeparator();
    if (!separator.isModified()) {
      separator.delete();
    }

    conflict.getUnchosenSides().forEach(side => {
      side.deleteBanner();
      side.delete();
    });

    this.updateMarkerCount();
  }

  reparseConflicts() {
    const newConflicts = new Set(_conflict2.default.allFromEditor(this.props.editor, this.layer, this.props.isRebase));
    this.setState({ conflicts: newConflicts });
  }

  updateMarkerCount() {
    this.props.resolutionProgress.reportMarkerCount(this.props.editor.getPath(), Array.from(this.state.conflicts, c => !c.isResolved()).filter(b => b).length);
  }
}, _class2.propTypes = {
  editor: _propTypes2.default.object.isRequired,
  commandRegistry: _propTypes2.default.object.isRequired,
  resolutionProgress: _propTypes2.default.object.isRequired,
  isRebase: _propTypes2.default.bool.isRequired,
  refreshResolutionProgress: _propTypes2.default.func
}, _class2.defaultProps = {
  refreshResolutionProgress: () => {}
}, _temp), (_applyDecoratedDescriptor(_class.prototype, 'resolveAsCurrent', [_coreDecorators.autobind], Object.getOwnPropertyDescriptor(_class.prototype, 'resolveAsCurrent'), _class.prototype), _applyDecoratedDescriptor(_class.prototype, 'revertConflictModifications', [_coreDecorators.autobind], Object.getOwnPropertyDescriptor(_class.prototype, 'revertConflictModifications'), _class.prototype), _applyDecoratedDescriptor(_class.prototype, 'dismissCurrent', [_coreDecorators.autobind], Object.getOwnPropertyDescriptor(_class.prototype, 'dismissCurrent'), _class.prototype)), _class);
exports.default = EditorConflictController;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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