(function() {
  var Emitter, Result, ResultsModel, TextEditor, _, binaryIndex, escapeHelper, ref, stringCompare,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  _ = require('underscore-plus');

  ref = require('atom'), Emitter = ref.Emitter, TextEditor = ref.TextEditor;

  escapeHelper = require('../escape-helper');

  Result = (function() {
    Result.create = function(result) {
      var matches, ref1;
      if (result != null ? (ref1 = result.matches) != null ? ref1.length : void 0 : void 0) {
        matches = result.matches.map(function(m) {
          return {
            matchText: m.matchText,
            lineText: m.lineText,
            lineTextOffset: m.lineTextOffset,
            range: m.range,
            leadingContextLines: m.leadingContextLines,
            trailingContextLines: m.trailingContextLines
          };
        });
        return new Result({
          filePath: result.filePath,
          matches: matches
        });
      } else {
        return null;
      }
    };

    function Result(result) {
      _.extend(this, result);
    }

    return Result;

  })();

  module.exports = ResultsModel = (function() {
    function ResultsModel(findOptions) {
      this.findOptions = findOptions;
      this.onContentsModified = bind(this.onContentsModified, this);
      this.emitter = new Emitter;
      atom.workspace.getCenter().observeActivePaneItem((function(_this) {
        return function(item) {
          if (item instanceof TextEditor) {
            return item.onDidStopChanging(function() {
              return _this.onContentsModified(item);
            });
          }
        };
      })(this));
      this.clear();
    }

    ResultsModel.prototype.onDidClear = function(callback) {
      return this.emitter.on('did-clear', callback);
    };

    ResultsModel.prototype.onDidClearSearchState = function(callback) {
      return this.emitter.on('did-clear-search-state', callback);
    };

    ResultsModel.prototype.onDidClearReplacementState = function(callback) {
      return this.emitter.on('did-clear-replacement-state', callback);
    };

    ResultsModel.prototype.onDidSearchPaths = function(callback) {
      return this.emitter.on('did-search-paths', callback);
    };

    ResultsModel.prototype.onDidErrorForPath = function(callback) {
      return this.emitter.on('did-error-for-path', callback);
    };

    ResultsModel.prototype.onDidStartSearching = function(callback) {
      return this.emitter.on('did-start-searching', callback);
    };

    ResultsModel.prototype.onDidCancelSearching = function(callback) {
      return this.emitter.on('did-cancel-searching', callback);
    };

    ResultsModel.prototype.onDidFinishSearching = function(callback) {
      return this.emitter.on('did-finish-searching', callback);
    };

    ResultsModel.prototype.onDidStartReplacing = function(callback) {
      return this.emitter.on('did-start-replacing', callback);
    };

    ResultsModel.prototype.onDidFinishReplacing = function(callback) {
      return this.emitter.on('did-finish-replacing', callback);
    };

    ResultsModel.prototype.onDidSearchPath = function(callback) {
      return this.emitter.on('did-search-path', callback);
    };

    ResultsModel.prototype.onDidReplacePath = function(callback) {
      return this.emitter.on('did-replace-path', callback);
    };

    ResultsModel.prototype.onDidAddResult = function(callback) {
      return this.emitter.on('did-add-result', callback);
    };

    ResultsModel.prototype.onDidRemoveResult = function(callback) {
      return this.emitter.on('did-remove-result', callback);
    };

    ResultsModel.prototype.clear = function() {
      this.clearSearchState();
      this.clearReplacementState();
      return this.emitter.emit('did-clear', this.getResultsSummary());
    };

    ResultsModel.prototype.clearSearchState = function() {
      this.pathCount = 0;
      this.matchCount = 0;
      this.regex = null;
      this.results = {};
      this.paths = [];
      this.active = false;
      this.searchErrors = null;
      if (this.inProgressSearchPromise != null) {
        this.inProgressSearchPromise.cancel();
        this.inProgressSearchPromise = null;
      }
      return this.emitter.emit('did-clear-search-state', this.getResultsSummary());
    };

    ResultsModel.prototype.clearReplacementState = function() {
      this.replacePattern = null;
      this.replacedPathCount = null;
      this.replacementCount = null;
      this.replacementErrors = null;
      return this.emitter.emit('did-clear-replacement-state', this.getResultsSummary());
    };

    ResultsModel.prototype.shoudldRerunSearch = function(findPattern, pathsPattern, replacePattern, options) {
      var onlyRunIfChanged;
      if (options == null) {
        options = {};
      }
      onlyRunIfChanged = options.onlyRunIfChanged;
      if (onlyRunIfChanged && (findPattern != null) && (pathsPattern != null) && findPattern === this.lastFindPattern && pathsPattern === this.lastPathsPattern) {
        return false;
      } else {
        return true;
      }
    };

    ResultsModel.prototype.search = function(findPattern, pathsPattern, replacePattern, options) {
      var keepReplacementState, leadingContextLineCount, onPathsSearched, searchPaths, trailingContextLineCount;
      if (options == null) {
        options = {};
      }
      if (!this.shoudldRerunSearch(findPattern, pathsPattern, replacePattern, options)) {
        return Promise.resolve();
      }
      keepReplacementState = options.keepReplacementState;
      if (keepReplacementState) {
        this.clearSearchState();
      } else {
        this.clear();
      }
      this.lastFindPattern = findPattern;
      this.lastPathsPattern = pathsPattern;
      this.findOptions.set(_.extend({
        findPattern: findPattern,
        replacePattern: replacePattern,
        pathsPattern: pathsPattern
      }, options));
      this.regex = this.findOptions.getFindPatternRegex();
      this.active = true;
      searchPaths = this.pathsArrayFromPathsPattern(pathsPattern);
      onPathsSearched = (function(_this) {
        return function(numberOfPathsSearched) {
          return _this.emitter.emit('did-search-paths', numberOfPathsSearched);
        };
      })(this);
      leadingContextLineCount = atom.config.get('find-and-replace.searchContextLineCountBefore');
      trailingContextLineCount = atom.config.get('find-and-replace.searchContextLineCountAfter');
      this.inProgressSearchPromise = atom.workspace.scan(this.regex, {
        paths: searchPaths,
        onPathsSearched: onPathsSearched,
        leadingContextLineCount: leadingContextLineCount,
        trailingContextLineCount: trailingContextLineCount
      }, (function(_this) {
        return function(result, error) {
          if (result) {
            return _this.setResult(result.filePath, Result.create(result));
          } else {
            if (_this.searchErrors == null) {
              _this.searchErrors = [];
            }
            _this.searchErrors.push(error);
            return _this.emitter.emit('did-error-for-path', error);
          }
        };
      })(this));
      this.emitter.emit('did-start-searching', this.inProgressSearchPromise);
      return this.inProgressSearchPromise.then((function(_this) {
        return function(message) {
          if (message === 'cancelled') {
            return _this.emitter.emit('did-cancel-searching');
          } else {
            _this.inProgressSearchPromise = null;
            return _this.emitter.emit('did-finish-searching', _this.getResultsSummary());
          }
        };
      })(this));
    };

    ResultsModel.prototype.replace = function(pathsPattern, replacePattern, replacementPaths) {
      var promise;
      if (!(this.findOptions.findPattern && (this.regex != null))) {
        return;
      }
      this.findOptions.set({
        replacePattern: replacePattern,
        pathsPattern: pathsPattern
      });
      if (this.findOptions.useRegex) {
        replacePattern = escapeHelper.unescapeEscapeSequence(replacePattern);
      }
      this.active = false;
      this.replacedPathCount = 0;
      this.replacementCount = 0;
      promise = atom.workspace.replace(this.regex, replacePattern, replacementPaths, (function(_this) {
        return function(result, error) {
          if (result) {
            if (result.replacements) {
              _this.replacedPathCount++;
              _this.replacementCount += result.replacements;
            }
            return _this.emitter.emit('did-replace-path', result);
          } else {
            if (_this.replacementErrors == null) {
              _this.replacementErrors = [];
            }
            _this.replacementErrors.push(error);
            return _this.emitter.emit('did-error-for-path', error);
          }
        };
      })(this));
      this.emitter.emit('did-start-replacing', promise);
      return promise.then((function(_this) {
        return function() {
          _this.emitter.emit('did-finish-replacing', _this.getResultsSummary());
          return _this.search(_this.findOptions.findPattern, _this.findOptions.pathsPattern, _this.findOptions.replacePattern, {
            keepReplacementState: true
          });
        };
      })(this))["catch"](function(e) {
        return console.error(e.stack);
      });
    };

    ResultsModel.prototype.setActive = function(isActive) {
      if ((isActive && this.findOptions.findPattern) || !isActive) {
        return this.active = isActive;
      }
    };

    ResultsModel.prototype.getActive = function() {
      return this.active;
    };

    ResultsModel.prototype.getFindOptions = function() {
      return this.findOptions;
    };

    ResultsModel.prototype.getLastFindPattern = function() {
      return this.lastFindPattern;
    };

    ResultsModel.prototype.getResultsSummary = function() {
      var findPattern, ref1, replacePattern;
      findPattern = (ref1 = this.lastFindPattern) != null ? ref1 : this.findOptions.findPattern;
      replacePattern = this.findOptions.replacePattern;
      return {
        findPattern: findPattern,
        replacePattern: replacePattern,
        pathCount: this.pathCount,
        matchCount: this.matchCount,
        searchErrors: this.searchErrors,
        replacedPathCount: this.replacedPathCount,
        replacementCount: this.replacementCount,
        replacementErrors: this.replacementErrors
      };
    };

    ResultsModel.prototype.getPathCount = function() {
      return this.pathCount;
    };

    ResultsModel.prototype.getMatchCount = function() {
      return this.matchCount;
    };

    ResultsModel.prototype.getPaths = function() {
      return this.paths;
    };

    ResultsModel.prototype.getResult = function(filePath) {
      return this.results[filePath];
    };

    ResultsModel.prototype.getResultAt = function(index) {
      return this.results[this.paths[index]];
    };

    ResultsModel.prototype.setResult = function(filePath, result) {
      if (result) {
        return this.addResult(filePath, result);
      } else {
        return this.removeResult(filePath);
      }
    };

    ResultsModel.prototype.addResult = function(filePath, result) {
      var filePathInsertedIndex;
      filePathInsertedIndex = null;
      if (this.results[filePath]) {
        this.matchCount -= this.results[filePath].matches.length;
      } else {
        this.pathCount++;
        filePathInsertedIndex = binaryIndex(this.paths, filePath, stringCompare);
        this.paths.splice(filePathInsertedIndex, 0, filePath);
      }
      this.matchCount += result.matches.length;
      this.results[filePath] = result;
      return this.emitter.emit('did-add-result', {
        filePath: filePath,
        result: result,
        filePathInsertedIndex: filePathInsertedIndex
      });
    };

    ResultsModel.prototype.removeResult = function(filePath) {
      if (this.results[filePath]) {
        this.pathCount--;
        this.matchCount -= this.results[filePath].matches.length;
        this.paths = _.without(this.paths, filePath);
        delete this.results[filePath];
        return this.emitter.emit('did-remove-result', {
          filePath: filePath
        });
      }
    };

    ResultsModel.prototype.onContentsModified = function(editor) {
      var leadingContextLineCount, matches, result, trailingContextLineCount;
      if (!(this.active && this.regex)) {
        return;
      }
      if (!editor.getPath()) {
        return;
      }
      matches = [];
      if (parseFloat(atom.getVersion()) >= 1.17) {
        leadingContextLineCount = atom.config.get('find-and-replace.searchContextLineCountBefore');
        trailingContextLineCount = atom.config.get('find-and-replace.searchContextLineCountAfter');
        editor.scan(this.regex, {
          leadingContextLineCount: leadingContextLineCount,
          trailingContextLineCount: trailingContextLineCount
        }, function(match) {
          return matches.push(match);
        });
      } else {
        editor.scan(this.regex, function(match) {
          return matches.push(match);
        });
      }
      result = Result.create({
        filePath: editor.getPath(),
        matches: matches
      });
      this.setResult(editor.getPath(), result);
      return this.emitter.emit('did-finish-searching', this.getResultsSummary());
    };

    ResultsModel.prototype.pathsArrayFromPathsPattern = function(pathsPattern) {
      var i, inputPath, len, ref1, results;
      ref1 = pathsPattern.trim().split(',');
      results = [];
      for (i = 0, len = ref1.length; i < len; i++) {
        inputPath = ref1[i];
        if (inputPath) {
          results.push(inputPath.trim());
        }
      }
      return results;
    };

    return ResultsModel;

  })();

  stringCompare = function(a, b) {
    return a.localeCompare(b);
  };

  binaryIndex = function(array, value, comparator) {
    var high, low, mid;
    low = 0;
    high = array.length;
    while (low < high) {
      mid = Math.floor((low + high) / 2);
      if (comparator(array[mid], value) < 0) {
        low = mid + 1;
      } else {
        high = mid;
      }
    }
    return low;
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
