(function() {
  var COMMENT_SELECTOR_REGEX, Range, SelfClosingTags, TAG_SELECTOR_REGEX, TagFinder, _, generateTagStartOrEndRegex, tagStartOrEndRegex;

  Range = require('atom').Range;

  _ = require('underscore-plus');

  SelfClosingTags = require('./self-closing-tags');

  TAG_SELECTOR_REGEX = /(\b|\.)(meta\.tag|punctuation\.definition\.tag)/;

  COMMENT_SELECTOR_REGEX = /(\b|\.)comment/;

  generateTagStartOrEndRegex = function(tagNameRegexStr) {
    var notSelfClosingTagEnd, re;
    notSelfClosingTagEnd = "(?:[^>\\/\"']|\"[^\"]*\"|'[^']*')*>";
    return re = new RegExp("<(" + tagNameRegexStr + ")" + notSelfClosingTagEnd + "|<\\/(" + tagNameRegexStr + ")>");
  };

  tagStartOrEndRegex = generateTagStartOrEndRegex("\\w[-\\w]*(?:\\:\\w[-\\w]*)?");

  module.exports = TagFinder = (function() {
    function TagFinder(editor) {
      this.editor = editor;
      this.tagPattern = /(<(\/?))([^\s>]+)([\s>]|$)/;
      this.wordRegex = /[^>\r\n]*/;
    }

    TagFinder.prototype.patternForTagName = function(tagName) {
      tagName = _.escapeRegExp(tagName);
      return new RegExp("(<" + tagName + "([\\s>]|$))|(</" + tagName + ">)", 'gi');
    };

    TagFinder.prototype.isRangeCommented = function(range) {
      return this.scopesForPositionMatchRegex(range.start, COMMENT_SELECTOR_REGEX);
    };

    TagFinder.prototype.isTagRange = function(range) {
      return this.scopesForPositionMatchRegex(range.start, TAG_SELECTOR_REGEX);
    };

    TagFinder.prototype.isCursorOnTag = function() {
      return this.scopesForPositionMatchRegex(this.editor.getCursorBufferPosition(), TAG_SELECTOR_REGEX);
    };

    TagFinder.prototype.scopesForPositionMatchRegex = function(position, regex) {
      var buffer, column, grammar, i, len, line, lineLength, nextColumn, ref, ref1, scopeIds, tag, tokenizedBuffer;
      ref = this.editor, tokenizedBuffer = ref.tokenizedBuffer, buffer = ref.buffer;
      grammar = tokenizedBuffer.grammar;
      column = 0;
      line = tokenizedBuffer.tokenizedLineForRow(position.row);
      if (line == null) {
        return false;
      }
      lineLength = buffer.lineLengthForRow(position.row);
      scopeIds = line.openScopes.slice();
      ref1 = line.tags;
      for (i = 0, len = ref1.length; i < len; i += 1) {
        tag = ref1[i];
        if (tag >= 0) {
          nextColumn = column + tag;
          if (nextColumn > position.column || nextColumn === lineLength) {
            break;
          }
          column = nextColumn;
        } else if ((tag & 1) === 1) {
          scopeIds.push(tag);
        } else {
          scopeIds.pop();
        }
      }
      return scopeIds.some(function(scopeId) {
        return regex.test(grammar.scopeForId(scopeId));
      });
    };

    TagFinder.prototype.findStartTag = function(tagName, endPosition) {
      var pattern, scanRange, startRange, unpairedCount;
      scanRange = new Range([0, 0], endPosition);
      pattern = this.patternForTagName(tagName);
      startRange = null;
      unpairedCount = 0;
      this.editor.backwardsScanInBufferRange(pattern, scanRange, (function(_this) {
        return function(arg) {
          var match, range, stop;
          match = arg.match, range = arg.range, stop = arg.stop;
          if (_this.isRangeCommented(range)) {
            return;
          }
          if (match[1]) {
            unpairedCount--;
            if (unpairedCount < 0) {
              startRange = range.translate([0, 1], [0, -match[2].length]);
              return stop();
            }
          } else {
            return unpairedCount++;
          }
        };
      })(this));
      return startRange;
    };

    TagFinder.prototype.findEndTag = function(tagName, startPosition) {
      var endRange, pattern, scanRange, unpairedCount;
      scanRange = new Range(startPosition, this.editor.buffer.getEndPosition());
      pattern = this.patternForTagName(tagName);
      endRange = null;
      unpairedCount = 0;
      this.editor.scanInBufferRange(pattern, scanRange, (function(_this) {
        return function(arg) {
          var match, range, stop;
          match = arg.match, range = arg.range, stop = arg.stop;
          if (_this.isRangeCommented(range)) {
            return;
          }
          if (match[1]) {
            return unpairedCount++;
          } else {
            unpairedCount--;
            if (unpairedCount < 0) {
              endRange = range.translate([0, 2], [0, -1]);
              return stop();
            }
          }
        };
      })(this));
      return endRange;
    };

    TagFinder.prototype.findStartEndTags = function() {
      var endPosition, ranges;
      ranges = null;
      endPosition = this.editor.getLastCursor().getCurrentWordBufferRange({
        wordRegex: this.wordRegex
      }).end;
      this.editor.backwardsScanInBufferRange(this.tagPattern, [[0, 0], endPosition], (function(_this) {
        return function(arg) {
          var endRange, entireMatch, isClosingTag, match, prefix, range, startRange, stop, suffix, tagName;
          match = arg.match, range = arg.range, stop = arg.stop;
          stop();
          entireMatch = match[0], prefix = match[1], isClosingTag = match[2], tagName = match[3], suffix = match[4];
          if (range.start.row === range.end.row) {
            startRange = range.translate([0, prefix.length], [0, -suffix.length]);
          } else {
            startRange = Range.fromObject([range.start.translate([0, prefix.length]), [range.start.row, 2e308]]);
          }
          if (isClosingTag) {
            endRange = _this.findStartTag(tagName, startRange.start);
          } else {
            endRange = _this.findEndTag(tagName, startRange.end);
          }
          if ((startRange != null) && (endRange != null)) {
            return ranges = {
              startRange: startRange,
              endRange: endRange
            };
          }
        };
      })(this));
      return ranges;
    };

    TagFinder.prototype.findEnclosingTags = function() {
      var ranges;
      if (ranges = this.findStartEndTags()) {
        if (this.isTagRange(ranges.startRange) && this.isTagRange(ranges.endRange)) {
          return ranges;
        }
      }
      return null;
    };

    TagFinder.prototype.findMatchingTags = function() {
      if (this.isCursorOnTag()) {
        return this.findStartEndTags();
      }
    };

    TagFinder.prototype.parseFragment = function(fragment, stack, matchExpr, cond) {
      var match, topElem;
      match = fragment.match(matchExpr);
      while (match && cond(stack)) {
        if (SelfClosingTags.indexOf(match[1]) === -1) {
          topElem = stack[stack.length - 1];
          if (match[2] && topElem === match[2]) {
            stack.pop();
          } else if (match[1]) {
            stack.push(match[1]);
          }
        }
        fragment = fragment.substr(match.index + match[0].length);
        match = fragment.match(matchExpr);
      }
      return stack;
    };

    TagFinder.prototype.tagsNotClosedInFragment = function(fragment) {
      return this.parseFragment(fragment, [], tagStartOrEndRegex, function() {
        return true;
      });
    };

    TagFinder.prototype.tagDoesNotCloseInFragment = function(tags, fragment) {
      var escapedTag, stack, stackLength, tag;
      if (tags.length === 0) {
        return false;
      }
      stack = tags;
      stackLength = stack.length;
      tag = tags[tags.length - 1];
      escapedTag = _.escapeRegExp(tag);
      stack = this.parseFragment(fragment, stack, generateTagStartOrEndRegex(escapedTag), function(s) {
        return s.length >= stackLength || s[s.length - 1] === tag;
      });
      return stack.length > 0 && stack[stack.length - 1] === tag;
    };

    TagFinder.prototype.closingTagForFragments = function(preFragment, postFragment) {
      var tag, tags;
      tags = this.tagsNotClosedInFragment(preFragment);
      tag = tags[tags.length - 1];
      if (this.tagDoesNotCloseInFragment(tags, postFragment)) {
        return tag;
      } else {
        return null;
      }
    };

    return TagFinder;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
