(function() {
  var BracketMatcher, CompositeDisposable, SelectorCache, _,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  _ = require('underscore-plus');

  CompositeDisposable = require('atom').CompositeDisposable;

  SelectorCache = require('./selector-cache');

  module.exports = BracketMatcher = (function() {
    function BracketMatcher(editor, editorElement, matchManager) {
      this.editor = editor;
      this.matchManager = matchManager;
      this.backspace = bind(this.backspace, this);
      this.insertNewline = bind(this.insertNewline, this);
      this.insertText = bind(this.insertText, this);
      this.subscriptions = new CompositeDisposable;
      this.bracketMarkers = [];
      _.adviseBefore(this.editor, 'insertText', this.insertText);
      _.adviseBefore(this.editor, 'insertNewline', this.insertNewline);
      _.adviseBefore(this.editor, 'backspace', this.backspace);
      this.subscriptions.add(atom.commands.add(editorElement, 'bracket-matcher:remove-brackets-from-selection', (function(_this) {
        return function(event) {
          if (!_this.removeBrackets()) {
            return event.abortKeyBinding();
          }
        };
      })(this)));
      this.subscriptions.add(this.editor.onDidDestroy((function(_this) {
        return function() {
          return _this.unsubscribe();
        };
      })(this)));
    }

    BracketMatcher.prototype.insertText = function(text, options) {
      var autoCompleteOpeningBracket, bracketMarker, cursorBufferPosition, hasEscapeCharacterBeforeCursor, hasEscapeSequenceBeforeCursor, hasQuoteBeforeCursor, hasWordAfterCursor, hasWordBeforeCursor, nextCharacter, pair, previousCharacter, previousCharacters, range, ref, ref1, ref2, skipOverExistingClosingBracket;
      if (!text) {
        return true;
      }
      if ((options != null ? options.select : void 0) || (options != null ? options.undo : void 0) === 'skip') {
        return true;
      }
      if (this.wrapSelectionInBrackets(text)) {
        return false;
      }
      if (this.editor.hasMultipleCursors()) {
        return true;
      }
      cursorBufferPosition = this.editor.getCursorBufferPosition();
      previousCharacters = this.editor.getTextInBufferRange([[cursorBufferPosition.row, 0], cursorBufferPosition]);
      nextCharacter = this.editor.getTextInBufferRange([cursorBufferPosition, cursorBufferPosition.traverse([0, 1])]);
      previousCharacter = previousCharacters.slice(-1);
      hasWordAfterCursor = /\w/.test(nextCharacter);
      hasWordBeforeCursor = /\w/.test(previousCharacter);
      hasQuoteBeforeCursor = this.isQuote(previousCharacter) && previousCharacter === text[0];
      hasEscapeCharacterBeforeCursor = ((ref = previousCharacters.match(/(\\+)$/)) != null ? ref[1].length : void 0) % 2 === 1;
      hasEscapeSequenceBeforeCursor = ((ref1 = previousCharacters.match(/(\\+)[^\\]$/)) != null ? ref1[1].length : void 0) % 2 === 1 || ((ref2 = previousCharacters.match(/(\\+)$/)) != null ? ref2[1].length : void 0) % 2 === 0;
      if (text === '#' && this.isCursorOnInterpolatedString()) {
        autoCompleteOpeningBracket = this.getScopedSetting('bracket-matcher.autocompleteBrackets') && !hasEscapeCharacterBeforeCursor;
        text += '{';
        pair = '}';
      } else {
        autoCompleteOpeningBracket = this.isOpeningBracket(text) && !hasWordAfterCursor && this.getScopedSetting('bracket-matcher.autocompleteBrackets') && !(this.isQuote(text) && (hasWordBeforeCursor || hasQuoteBeforeCursor || hasEscapeSequenceBeforeCursor)) && !hasEscapeCharacterBeforeCursor;
        pair = this.matchManager.pairedCharacters[text];
      }
      skipOverExistingClosingBracket = false;
      if (this.isClosingBracket(text) && nextCharacter === text && !hasEscapeCharacterBeforeCursor) {
        if (bracketMarker = _.find(this.bracketMarkers, function(marker) {
          return marker.isValid() && marker.getBufferRange().end.isEqual(cursorBufferPosition);
        })) {
          skipOverExistingClosingBracket = true;
        }
      }
      if (skipOverExistingClosingBracket) {
        bracketMarker.destroy();
        _.remove(this.bracketMarkers, bracketMarker);
        this.editor.moveRight();
        return false;
      } else if (autoCompleteOpeningBracket) {
        this.editor.insertText(text + pair);
        this.editor.moveLeft();
        range = [cursorBufferPosition, cursorBufferPosition.traverse([0, text.length])];
        this.bracketMarkers.push(this.editor.markBufferRange(range));
        return false;
      }
    };

    BracketMatcher.prototype.insertNewline = function() {
      var cursorBufferPosition, hasEscapeCharacterBeforeCursor, nextCharacter, previousCharacter, previousCharacters, ref;
      if (this.editor.hasMultipleCursors()) {
        return;
      }
      if (!this.editor.getLastSelection().isEmpty()) {
        return;
      }
      cursorBufferPosition = this.editor.getCursorBufferPosition();
      previousCharacters = this.editor.getTextInBufferRange([[cursorBufferPosition.row, 0], cursorBufferPosition]);
      nextCharacter = this.editor.getTextInBufferRange([cursorBufferPosition, cursorBufferPosition.traverse([0, 1])]);
      previousCharacter = previousCharacters.slice(-1);
      hasEscapeCharacterBeforeCursor = ((ref = previousCharacters.match(/(\\+)$/)) != null ? ref[1].length : void 0) % 2 === 1;
      if (this.matchManager.pairsWithExtraNewline[previousCharacter] === nextCharacter && !hasEscapeCharacterBeforeCursor) {
        this.editor.transact((function(_this) {
          return function() {
            var cursorRow;
            _this.editor.insertText("\n\n");
            _this.editor.moveUp();
            if (_this.getScopedSetting('editor.autoIndent')) {
              cursorRow = _this.editor.getCursorBufferPosition().row;
              return _this.editor.autoIndentBufferRows(cursorRow, cursorRow + 1);
            }
          };
        })(this));
        return false;
      }
    };

    BracketMatcher.prototype.backspace = function() {
      var cursorBufferPosition, hasEscapeCharacterBeforeCursor, nextCharacter, previousCharacter, previousCharacters, ref;
      if (this.editor.hasMultipleCursors()) {
        return;
      }
      if (!this.editor.getLastSelection().isEmpty()) {
        return;
      }
      cursorBufferPosition = this.editor.getCursorBufferPosition();
      previousCharacters = this.editor.getTextInBufferRange([[cursorBufferPosition.row, 0], cursorBufferPosition]);
      nextCharacter = this.editor.getTextInBufferRange([cursorBufferPosition, cursorBufferPosition.traverse([0, 1])]);
      previousCharacter = previousCharacters.slice(-1);
      hasEscapeCharacterBeforeCursor = ((ref = previousCharacters.match(/(\\+)$/)) != null ? ref[1].length : void 0) % 2 === 1;
      if ((this.matchManager.pairedCharacters[previousCharacter] === nextCharacter) && !hasEscapeCharacterBeforeCursor && this.getScopedSetting('bracket-matcher.autocompleteBrackets')) {
        this.editor.transact((function(_this) {
          return function() {
            _this.editor.moveLeft();
            _this.editor["delete"]();
            return _this.editor["delete"]();
          };
        })(this));
        return false;
      }
    };

    BracketMatcher.prototype.removeBrackets = function() {
      var bracketsRemoved;
      bracketsRemoved = false;
      this.editor.mutateSelectedText((function(_this) {
        return function(selection) {
          var options, range, selectionEnd, selectionStart, text;
          if (!_this.selectionIsWrappedByMatchingBrackets(selection)) {
            return;
          }
          range = selection.getBufferRange();
          options = {
            reversed: selection.isReversed()
          };
          selectionStart = range.start;
          if (range.start.row === range.end.row) {
            selectionEnd = range.end.traverse([0, -2]);
          } else {
            selectionEnd = range.end.traverse([0, -1]);
          }
          text = selection.getText();
          selection.insertText(text.substring(1, text.length - 1));
          selection.setBufferRange([selectionStart, selectionEnd], options);
          return bracketsRemoved = true;
        };
      })(this));
      return bracketsRemoved;
    };

    BracketMatcher.prototype.wrapSelectionInBrackets = function(bracket) {
      var pair, selectionWrapped;
      if (bracket === '#') {
        if (!this.isCursorOnInterpolatedString()) {
          return false;
        }
        bracket = '#{';
        pair = '}';
      } else {
        if (!this.isOpeningBracket(bracket)) {
          return false;
        }
        pair = this.matchManager.pairedCharacters[bracket];
      }
      if (!this.editor.selections.some(function(s) {
        return !s.isEmpty();
      })) {
        return false;
      }
      if (!this.getScopedSetting('bracket-matcher.wrapSelectionsInBrackets')) {
        return false;
      }
      selectionWrapped = false;
      this.editor.mutateSelectedText(function(selection) {
        var options, range, selectionEnd, selectionStart;
        if (selection.isEmpty()) {
          return;
        }
        if (bracket === '#{' && !selection.getBufferRange().isSingleLine()) {
          return;
        }
        selectionWrapped = true;
        range = selection.getBufferRange();
        options = {
          reversed: selection.isReversed()
        };
        selection.insertText("" + bracket + (selection.getText()) + pair);
        selectionStart = range.start.traverse([0, bracket.length]);
        if (range.start.row === range.end.row) {
          selectionEnd = range.end.traverse([0, bracket.length]);
        } else {
          selectionEnd = range.end;
        }
        return selection.setBufferRange([selectionStart, selectionEnd], options);
      });
      return selectionWrapped;
    };

    BracketMatcher.prototype.isQuote = function(string) {
      return /['"`]/.test(string);
    };

    BracketMatcher.prototype.isCursorOnInterpolatedString = function() {
      var segments;
      if (this.interpolatedStringSelector == null) {
        segments = ['*.*.*.interpolated.ruby', 'string.interpolated.ruby', 'string.regexp.interpolated.ruby', 'string.quoted.double.coffee', 'string.unquoted.heredoc.ruby', 'string.quoted.double.livescript', 'string.quoted.double.heredoc.livescript', 'string.quoted.double.elixir', 'string.quoted.double.heredoc.elixir', 'comment.documentation.heredoc.elixir'];
        this.interpolatedStringSelector = SelectorCache.get(segments.join(' | '));
      }
      return this.interpolatedStringSelector.matches(this.editor.getLastCursor().getScopeDescriptor().getScopesArray());
    };

    BracketMatcher.prototype.isOpeningBracket = function(string) {
      return this.matchManager.pairedCharacters.hasOwnProperty(string);
    };

    BracketMatcher.prototype.isClosingBracket = function(string) {
      return this.matchManager.pairedCharactersInverse.hasOwnProperty(string);
    };

    BracketMatcher.prototype.selectionIsWrappedByMatchingBrackets = function(selection) {
      var firstCharacter, lastCharacter, selectedText;
      if (selection.isEmpty()) {
        return false;
      }
      selectedText = selection.getText();
      firstCharacter = selectedText[0];
      lastCharacter = selectedText[selectedText.length - 1];
      return this.matchManager.pairedCharacters[firstCharacter] === lastCharacter;
    };

    BracketMatcher.prototype.unsubscribe = function() {
      return this.subscriptions.dispose();
    };

    BracketMatcher.prototype.getScopedSetting = function(key) {
      return atom.config.get(key, {
        scope: this.editor.getRootScopeDescriptor()
      });
    };

    return BracketMatcher;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
