(function() {
  var BracketMatcherView, CompositeDisposable, MAX_ROWS_TO_SCAN, MAX_ROWS_TO_SCAN_BACKWARD_TRAVERSAL, MAX_ROWS_TO_SCAN_FORWARD_TRAVERSAL, ONE_CHAR_BACKWARD_TRAVERSAL, ONE_CHAR_FORWARD_TRAVERSAL, Point, Range, TWO_CHARS_BACKWARD_TRAVERSAL, TagFinder, _, ref,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  CompositeDisposable = require('atom').CompositeDisposable;

  _ = require('underscore-plus');

  ref = require('atom'), Range = ref.Range, Point = ref.Point;

  TagFinder = require('./tag-finder');

  MAX_ROWS_TO_SCAN = 10000;

  ONE_CHAR_FORWARD_TRAVERSAL = Object.freeze(Point(0, 1));

  ONE_CHAR_BACKWARD_TRAVERSAL = Object.freeze(Point(0, -1));

  TWO_CHARS_BACKWARD_TRAVERSAL = Object.freeze(Point(0, -2));

  MAX_ROWS_TO_SCAN_FORWARD_TRAVERSAL = Object.freeze(Point(MAX_ROWS_TO_SCAN, 0));

  MAX_ROWS_TO_SCAN_BACKWARD_TRAVERSAL = Object.freeze(Point(-MAX_ROWS_TO_SCAN, 0));

  module.exports = BracketMatcherView = (function() {
    function BracketMatcherView(editor, editorElement, matchManager) {
      this.editor = editor;
      this.matchManager = matchManager;
      this.updateMatch = bind(this.updateMatch, this);
      this.destroy = bind(this.destroy, this);
      this.subscriptions = new CompositeDisposable;
      this.tagFinder = new TagFinder(this.editor);
      this.pairHighlighted = false;
      this.tagHighlighted = false;
      this.subscriptions.add(this.editor.onDidTokenize(this.updateMatch));
      this.subscriptions.add(this.editor.getBuffer().onDidChangeText(this.updateMatch));
      this.subscriptions.add(this.editor.onDidChangeGrammar(this.updateMatch));
      this.subscriptions.add(this.editor.onDidChangeSelectionRange(this.updateMatch));
      this.subscriptions.add(this.editor.onDidAddCursor(this.updateMatch));
      this.subscriptions.add(this.editor.onDidRemoveCursor(this.updateMatch));
      this.subscriptions.add(atom.commands.add(editorElement, 'bracket-matcher:go-to-matching-bracket', (function(_this) {
        return function() {
          return _this.goToMatchingPair();
        };
      })(this)));
      this.subscriptions.add(atom.commands.add(editorElement, 'bracket-matcher:go-to-enclosing-bracket', (function(_this) {
        return function() {
          return _this.goToEnclosingPair();
        };
      })(this)));
      this.subscriptions.add(atom.commands.add(editorElement, 'bracket-matcher:select-inside-brackets', (function(_this) {
        return function() {
          return _this.selectInsidePair();
        };
      })(this)));
      this.subscriptions.add(atom.commands.add(editorElement, 'bracket-matcher:close-tag', (function(_this) {
        return function() {
          return _this.closeTag();
        };
      })(this)));
      this.subscriptions.add(atom.commands.add(editorElement, 'bracket-matcher:remove-matching-brackets', (function(_this) {
        return function() {
          return _this.removeMatchingBrackets();
        };
      })(this)));
      this.subscriptions.add(this.editor.onDidDestroy(this.destroy));
      this.updateMatch();
    }

    BracketMatcherView.prototype.destroy = function() {
      return this.subscriptions.dispose();
    };

    BracketMatcherView.prototype.updateMatch = function() {
      var currentPair, endRange, highlightPair, highlightTag, matchPosition, matchingPair, pair, position, ref1, ref2, startRange;
      if (this.pairHighlighted) {
        this.editor.destroyMarker(this.startMarker.id);
        this.editor.destroyMarker(this.endMarker.id);
      }
      this.pairHighlighted = false;
      this.tagHighlighted = false;
      if (!this.editor.getLastSelection().isEmpty()) {
        return;
      }
      ref1 = this.findCurrentPair(false), position = ref1.position, currentPair = ref1.currentPair, matchingPair = ref1.matchingPair;
      if (position) {
        matchPosition = this.findMatchingEndPair(position, currentPair, matchingPair);
      } else {
        ref2 = this.findCurrentPair(true), position = ref2.position, currentPair = ref2.currentPair, matchingPair = ref2.matchingPair;
        if (position) {
          matchPosition = this.findMatchingStartPair(position, matchingPair, currentPair);
        }
      }
      startRange = null;
      endRange = null;
      highlightTag = false;
      highlightPair = false;
      if ((position != null) && (matchPosition != null)) {
        startRange = Range(position, position.traverse(ONE_CHAR_FORWARD_TRAVERSAL));
        endRange = Range(matchPosition, matchPosition.traverse(ONE_CHAR_FORWARD_TRAVERSAL));
        highlightPair = true;
      } else {
        if (pair = this.tagFinder.findMatchingTags()) {
          startRange = pair.startRange;
          endRange = pair.endRange;
          highlightPair = true;
          highlightTag = true;
        }
      }
      if (!(highlightTag || highlightPair)) {
        return;
      }
      if (this.editor.isFoldedAtCursorRow()) {
        return;
      }
      if (this.isCursorOnCommentOrString()) {
        return;
      }
      this.startMarker = this.createMarker(startRange);
      this.endMarker = this.createMarker(endRange);
      this.pairHighlighted = highlightPair;
      return this.tagHighlighted = highlightTag;
    };

    BracketMatcherView.prototype.removeMatchingBrackets = function() {
      if (this.editor.hasMultipleCursors()) {
        return this.editor.backspace();
      }
      return this.editor.transact((function(_this) {
        return function() {
          var currentPair, matchPosition, matchingPair, position, ref1, ref2, text;
          if (_this.editor.getLastSelection().isEmpty()) {
            _this.editor.selectLeft();
          }
          text = _this.editor.getSelectedText();
          _this.editor.moveRight();
          if (_this.matchManager.pairedCharacters.hasOwnProperty(text) || _this.matchManager.pairedCharactersInverse.hasOwnProperty(text)) {
            ref1 = _this.findCurrentPair(false), position = ref1.position, currentPair = ref1.currentPair, matchingPair = ref1.matchingPair;
            if (position) {
              matchPosition = _this.findMatchingEndPair(position, currentPair, matchingPair);
            } else {
              ref2 = _this.findCurrentPair(true), position = ref2.position, currentPair = ref2.currentPair, matchingPair = ref2.matchingPair;
              if (position) {
                matchPosition = _this.findMatchingStartPair(position, matchingPair, currentPair);
              }
            }
            if ((position != null) && (matchPosition != null)) {
              _this.editor.setCursorBufferPosition(matchPosition);
              _this.editor["delete"]();
              if (position.row === matchPosition.row && _this.matchManager.pairedCharactersInverse.hasOwnProperty(currentPair)) {
                position = position.traverse(ONE_CHAR_BACKWARD_TRAVERSAL);
              }
              _this.editor.setCursorBufferPosition(position);
              return _this.editor["delete"]();
            } else {
              return _this.editor.backspace();
            }
          } else {
            return _this.editor.backspace();
          }
        };
      })(this));
    };

    BracketMatcherView.prototype.findMatchingEndPair = function(startPairPosition, startPair, endPair) {
      var endPairPosition, scanRange, unpairedCount;
      if (startPair === endPair) {
        return;
      }
      scanRange = new Range(startPairPosition.traverse(ONE_CHAR_FORWARD_TRAVERSAL), startPairPosition.traverse(MAX_ROWS_TO_SCAN_FORWARD_TRAVERSAL));
      endPairPosition = null;
      unpairedCount = 0;
      this.editor.scanInBufferRange(this.matchManager.pairRegexes[startPair], scanRange, (function(_this) {
        return function(result) {
          if (_this.isRangeCommentedOrString(result.range)) {
            return;
          }
          switch (result.match[0]) {
            case startPair:
              return unpairedCount++;
            case endPair:
              unpairedCount--;
              if (unpairedCount < 0) {
                endPairPosition = result.range.start;
                return result.stop();
              }
          }
        };
      })(this));
      return endPairPosition;
    };

    BracketMatcherView.prototype.findMatchingStartPair = function(endPairPosition, startPair, endPair) {
      var scanRange, startPairPosition, unpairedCount;
      if (startPair === endPair) {
        return;
      }
      scanRange = new Range(endPairPosition.traverse(MAX_ROWS_TO_SCAN_BACKWARD_TRAVERSAL), endPairPosition);
      startPairPosition = null;
      unpairedCount = 0;
      this.editor.backwardsScanInBufferRange(this.matchManager.pairRegexes[startPair], scanRange, (function(_this) {
        return function(result) {
          if (_this.isRangeCommentedOrString(result.range)) {
            return;
          }
          switch (result.match[0]) {
            case startPair:
              unpairedCount--;
              if (unpairedCount < 0) {
                startPairPosition = result.range.start;
                return result.stop();
              }
              break;
            case endPair:
              return unpairedCount++;
          }
        };
      })(this));
      return startPairPosition;
    };

    BracketMatcherView.prototype.findAnyStartPair = function(cursorPosition) {
      var combinedRegExp, endPair, endPairRegExp, scanRange, startPair, startPairRegExp, startPosition, unpairedCount;
      scanRange = new Range(Point.ZERO, cursorPosition);
      startPair = _.escapeRegExp(_.keys(this.matchManager.pairedCharacters).join(''));
      endPair = _.escapeRegExp(_.keys(this.matchManager.pairedCharactersInverse).join(''));
      combinedRegExp = new RegExp("[" + startPair + endPair + "]", 'g');
      startPairRegExp = new RegExp("[" + startPair + "]", 'g');
      endPairRegExp = new RegExp("[" + endPair + "]", 'g');
      startPosition = null;
      unpairedCount = 0;
      this.editor.backwardsScanInBufferRange(combinedRegExp, scanRange, (function(_this) {
        return function(result) {
          if (_this.isRangeCommentedOrString(result.range)) {
            return;
          }
          if (result.match[0].match(endPairRegExp)) {
            return unpairedCount++;
          } else if (result.match[0].match(startPairRegExp)) {
            unpairedCount--;
            if (unpairedCount < 0) {
              startPosition = result.range.start;
              return result.stop();
            }
          }
        };
      })(this));
      return startPosition;
    };

    BracketMatcherView.prototype.createMarker = function(bufferRange) {
      var marker;
      marker = this.editor.markBufferRange(bufferRange);
      this.editor.decorateMarker(marker, {
        type: 'highlight',
        "class": 'bracket-matcher',
        deprecatedRegionClass: 'bracket-matcher'
      });
      return marker;
    };

    BracketMatcherView.prototype.findCurrentPair = function(isInverse) {
      var currentPair, matches, matchingPair, position;
      position = this.editor.getCursorBufferPosition();
      if (isInverse) {
        matches = this.matchManager.pairedCharactersInverse;
        position = position.traverse(ONE_CHAR_BACKWARD_TRAVERSAL);
      } else {
        matches = this.matchManager.pairedCharacters;
      }
      currentPair = this.editor.getTextInRange(Range.fromPointWithDelta(position, 0, 1));
      if (!matches[currentPair]) {
        if (isInverse) {
          position = position.traverse(ONE_CHAR_FORWARD_TRAVERSAL);
        } else {
          position = position.traverse(ONE_CHAR_BACKWARD_TRAVERSAL);
        }
        currentPair = this.editor.getTextInRange(Range.fromPointWithDelta(position, 0, 1));
      }
      if (matchingPair = matches[currentPair]) {
        return {
          position: position,
          currentPair: currentPair,
          matchingPair: matchingPair
        };
      } else {
        return {};
      }
    };

    BracketMatcherView.prototype.goToMatchingPair = function() {
      var endPosition, endRange, position, previousPosition, ref1, startPosition, startRange, tagCharacterOffset, tagLength;
      if (!this.pairHighlighted) {
        return this.goToEnclosingPair();
      }
      position = this.editor.getCursorBufferPosition();
      if (this.tagHighlighted) {
        startRange = this.startMarker.getBufferRange();
        tagLength = startRange.end.column - startRange.start.column;
        endRange = this.endMarker.getBufferRange();
        if (startRange.compare(endRange) > 0) {
          ref1 = [endRange, startRange], startRange = ref1[0], endRange = ref1[1];
        }
        startRange = new Range(startRange.start.traverse(ONE_CHAR_BACKWARD_TRAVERSAL), endRange.end.traverse(ONE_CHAR_BACKWARD_TRAVERSAL));
        endRange = new Range(endRange.start.traverse(TWO_CHARS_BACKWARD_TRAVERSAL), endRange.end.traverse(TWO_CHARS_BACKWARD_TRAVERSAL));
        if (position.isLessThan(endRange.start)) {
          tagCharacterOffset = position.column - startRange.start.column;
          if (tagCharacterOffset > 0) {
            tagCharacterOffset++;
          }
          tagCharacterOffset = Math.min(tagCharacterOffset, tagLength + 2);
          return this.editor.setCursorBufferPosition(endRange.start.traverse([0, tagCharacterOffset]));
        } else {
          tagCharacterOffset = position.column - endRange.start.column;
          if (tagCharacterOffset > 1) {
            tagCharacterOffset--;
          }
          tagCharacterOffset = Math.min(tagCharacterOffset, tagLength + 1);
          return this.editor.setCursorBufferPosition(startRange.start.traverse([0, tagCharacterOffset]));
        }
      } else {
        previousPosition = position.traverse(ONE_CHAR_BACKWARD_TRAVERSAL);
        startPosition = this.startMarker.getStartBufferPosition();
        endPosition = this.endMarker.getStartBufferPosition();
        if (position.isEqual(startPosition)) {
          return this.editor.setCursorBufferPosition(endPosition.traverse(ONE_CHAR_FORWARD_TRAVERSAL));
        } else if (previousPosition.isEqual(startPosition)) {
          return this.editor.setCursorBufferPosition(endPosition);
        } else if (position.isEqual(endPosition)) {
          return this.editor.setCursorBufferPosition(startPosition.traverse(ONE_CHAR_FORWARD_TRAVERSAL));
        } else if (previousPosition.isEqual(endPosition)) {
          return this.editor.setCursorBufferPosition(startPosition);
        }
      }
    };

    BracketMatcherView.prototype.goToEnclosingPair = function() {
      var endRange, matchPosition, pair, ref1, startRange;
      if (this.pairHighlighted) {
        return;
      }
      if (matchPosition = this.findAnyStartPair(this.editor.getCursorBufferPosition())) {
        return this.editor.setCursorBufferPosition(matchPosition);
      } else if (pair = this.tagFinder.findEnclosingTags()) {
        startRange = pair.startRange, endRange = pair.endRange;
        if (startRange.compare(endRange) > 0) {
          ref1 = [endRange, startRange], startRange = ref1[0], endRange = ref1[1];
        }
        return this.editor.setCursorBufferPosition(pair.startRange.start);
      }
    };

    BracketMatcherView.prototype.selectInsidePair = function() {
      var endPosition, endRange, pair, rangeToSelect, ref1, ref2, startPair, startPosition, startRange;
      if (this.pairHighlighted) {
        startRange = this.startMarker.getBufferRange();
        endRange = this.endMarker.getBufferRange();
        if (startRange.compare(endRange) > 0) {
          ref1 = [endRange, startRange], startRange = ref1[0], endRange = ref1[1];
        }
        if (this.tagHighlighted) {
          startPosition = startRange.end;
          endPosition = endRange.start.traverse(TWO_CHARS_BACKWARD_TRAVERSAL);
        } else {
          startPosition = startRange.start;
          endPosition = endRange.start;
        }
      } else {
        if (startPosition = this.findAnyStartPair(this.editor.getCursorBufferPosition())) {
          startPair = this.editor.getTextInRange(Range.fromPointWithDelta(startPosition, 0, 1));
          endPosition = this.findMatchingEndPair(startPosition, startPair, this.matchManager.pairedCharacters[startPair]);
        } else if (pair = this.tagFinder.findEnclosingTags()) {
          startRange = pair.startRange, endRange = pair.endRange;
          if (startRange.compare(endRange) > 0) {
            ref2 = [endRange, startRange], startRange = ref2[0], endRange = ref2[1];
          }
          startPosition = startRange.end;
          endPosition = endRange.start.traverse(TWO_CHARS_BACKWARD_TRAVERSAL);
        }
      }
      if ((startPosition != null) && (endPosition != null)) {
        rangeToSelect = new Range(startPosition.traverse(ONE_CHAR_FORWARD_TRAVERSAL), endPosition);
        return this.editor.setSelectedBufferRange(rangeToSelect);
      }
    };

    BracketMatcherView.prototype.closeTag = function() {
      var cursorPosition, postFragment, preFragment, tag;
      cursorPosition = this.editor.getCursorBufferPosition();
      preFragment = this.editor.getTextInBufferRange([Point.ZERO, cursorPosition]);
      postFragment = this.editor.getTextInBufferRange([cursorPosition, Point.INFINITY]);
      if (tag = this.tagFinder.closingTagForFragments(preFragment, postFragment)) {
        return this.editor.insertText("</" + tag + ">");
      }
    };

    BracketMatcherView.prototype.isCursorOnCommentOrString = function() {
      return this.isScopeCommentedOrString(this.editor.getLastCursor().getScopeDescriptor().getScopesArray());
    };

    BracketMatcherView.prototype.isRangeCommentedOrString = function(range) {
      return this.isScopeCommentedOrString(this.editor.scopeDescriptorForBufferPosition(range.start).getScopesArray());
    };

    BracketMatcherView.prototype.isScopeCommentedOrString = function(scopesArray) {
      var i, len, ref1, scope;
      ref1 = scopesArray.reverse();
      for (i = 0, len = ref1.length; i < len; i++) {
        scope = ref1[i];
        scope = scope.split('.');
        if (scope.includes('embedded') && scope.includes('source')) {
          return false;
        }
        if (scope.includes('comment') || scope.includes('string')) {
          return true;
        }
      }
      return false;
    };

    return BracketMatcherView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
