(function() {
  var ApplicationMenu, AtomApplication, AtomProtocolHandler, AtomWindow, AutoUpdateManager, BrowserWindow, CompositeDisposable, Config, ConfigSchema, Disposable, EventEmitter, FileRecoveryService, FindParentDir, LocationSuffixRegExp, Menu, Resolve, StorageFolder, _, app, dialog, fs, ipcHelpers, ipcMain, net, os, path, ref, ref1, screen, shell, url,
    slice = [].slice;

  AtomWindow = require('./atom-window');

  ApplicationMenu = require('./application-menu');

  AtomProtocolHandler = require('./atom-protocol-handler');

  AutoUpdateManager = require('./auto-update-manager');

  StorageFolder = require('../storage-folder');

  Config = require('../config');

  FileRecoveryService = require('./file-recovery-service');

  ipcHelpers = require('../ipc-helpers');

  ref = require('electron'), BrowserWindow = ref.BrowserWindow, Menu = ref.Menu, app = ref.app, dialog = ref.dialog, ipcMain = ref.ipcMain, shell = ref.shell, screen = ref.screen;

  ref1 = require('event-kit'), CompositeDisposable = ref1.CompositeDisposable, Disposable = ref1.Disposable;

  fs = require('fs-plus');

  path = require('path');

  os = require('os');

  net = require('net');

  url = require('url');

  EventEmitter = require('events').EventEmitter;

  _ = require('underscore-plus');

  FindParentDir = null;

  Resolve = null;

  ConfigSchema = require('../config-schema');

  LocationSuffixRegExp = /(:\d+)(:\d+)?$/;

  module.exports = AtomApplication = (function() {
    Object.assign(AtomApplication.prototype, EventEmitter.prototype);

    AtomApplication.open = function(options) {
      var client, userNameSafe;
      if (options.socketPath == null) {
        if (process.platform === 'win32') {
          userNameSafe = new Buffer(process.env.USERNAME).toString('base64');
          options.socketPath = "\\\\.\\pipe\\atom-" + options.version + "-" + userNameSafe + "-" + process.arch + "-sock";
        } else {
          options.socketPath = path.join(os.tmpdir(), "atom-" + options.version + "-" + process.env.USER + ".sock");
        }
      }
      if ((process.platform !== 'win32' && !fs.existsSync(options.socketPath)) || options.test || options.benchmark || options.benchmarkTest) {
        new AtomApplication(options).initialize(options);
        return;
      }
      client = net.connect({
        path: options.socketPath
      }, function() {
        return client.write(JSON.stringify(options), function() {
          client.end();
          return app.quit();
        });
      });
      return client.on('error', function() {
        return new AtomApplication(options).initialize(options);
      });
    };

    AtomApplication.prototype.windows = null;

    AtomApplication.prototype.applicationMenu = null;

    AtomApplication.prototype.atomProtocolHandler = null;

    AtomApplication.prototype.resourcePath = null;

    AtomApplication.prototype.version = null;

    AtomApplication.prototype.quitting = false;

    AtomApplication.prototype.exit = function(status) {
      return app.exit(status);
    };

    function AtomApplication(options) {
      this.resourcePath = options.resourcePath, this.devResourcePath = options.devResourcePath, this.version = options.version, this.devMode = options.devMode, this.safeMode = options.safeMode, this.socketPath = options.socketPath, this.logFile = options.logFile, this.userDataDir = options.userDataDir;
      if (options.test || options.benchmark || options.benchmarkTest) {
        this.socketPath = null;
      }
      this.pidsToOpenWindows = {};
      this.windows = [];
      this.config = new Config({
        enablePersistence: true
      });
      this.config.setSchema(null, {
        type: 'object',
        properties: _.clone(ConfigSchema)
      });
      ConfigSchema.projectHome = {
        type: 'string',
        "default": path.join(fs.getHomeDirectory(), 'github'),
        description: 'The directory where projects are assumed to be located. Packages created using the Package Generator will be stored here by default.'
      };
      this.config.initialize({
        configDirPath: process.env.ATOM_HOME,
        resourcePath: this.resourcePath,
        projectHomeSchema: ConfigSchema.projectHome
      });
      this.config.load();
      this.fileRecoveryService = new FileRecoveryService(path.join(process.env.ATOM_HOME, "recovery"));
      this.storageFolder = new StorageFolder(process.env.ATOM_HOME);
      this.autoUpdateManager = new AutoUpdateManager(this.version, options.test || options.benchmark || options.benchmarkTest, this.config);
      this.disposable = new CompositeDisposable;
      this.handleEvents();
    }

    AtomApplication.prototype.initialize = function(options) {
      global.atomApplication = this;
      if (process.platform === 'darwin' && this.config.get('core.useCustomTitleBar')) {
        this.config.unset('core.useCustomTitleBar');
        this.config.set('core.titleBar', 'custom');
      }
      this.config.onDidChange('core.titleBar', this.promptForRestart.bind(this));
      process.nextTick((function(_this) {
        return function() {
          return _this.autoUpdateManager.initialize();
        };
      })(this));
      this.applicationMenu = new ApplicationMenu(this.version, this.autoUpdateManager);
      this.atomProtocolHandler = new AtomProtocolHandler(this.resourcePath, this.safeMode);
      this.listenForArgumentsFromNewProcess();
      this.setupDockMenu();
      return this.launch(options);
    };

    AtomApplication.prototype.destroy = function() {
      var windowsClosePromises;
      windowsClosePromises = this.windows.map(function(window) {
        window.close();
        return window.closedPromise;
      });
      return Promise.all(windowsClosePromises).then((function(_this) {
        return function() {
          return _this.disposable.dispose();
        };
      })(this));
    };

    AtomApplication.prototype.launch = function(options) {
      var ref2, ref3;
      if (((ref2 = options.pathsToOpen) != null ? ref2.length : void 0) > 0 || ((ref3 = options.urlsToOpen) != null ? ref3.length : void 0) > 0 || options.test || options.benchmark || options.benchmarkTest) {
        if (this.config.get('core.restorePreviousWindowsOnStart') === 'always') {
          this.loadState(_.deepClone(options));
        }
        return this.openWithOptions(options);
      } else {
        return this.loadState(options) || this.openPath(options);
      }
    };

    AtomApplication.prototype.openWithOptions = function(options) {
      var addToLastWindow, benchmark, benchmarkTest, clearWindowState, devMode, env, executedFrom, i, initialPaths, len, logFile, newWindow, pathsToOpen, pidToKillWhenClosed, profileStartup, results, safeMode, test, timeout, urlToOpen, urlsToOpen;
      initialPaths = options.initialPaths, pathsToOpen = options.pathsToOpen, executedFrom = options.executedFrom, urlsToOpen = options.urlsToOpen, benchmark = options.benchmark, benchmarkTest = options.benchmarkTest, test = options.test, pidToKillWhenClosed = options.pidToKillWhenClosed, devMode = options.devMode, safeMode = options.safeMode, newWindow = options.newWindow, logFile = options.logFile, profileStartup = options.profileStartup, timeout = options.timeout, clearWindowState = options.clearWindowState, addToLastWindow = options.addToLastWindow, env = options.env;
      app.focus();
      if (test) {
        return this.runTests({
          headless: true,
          devMode: devMode,
          resourcePath: this.resourcePath,
          executedFrom: executedFrom,
          pathsToOpen: pathsToOpen,
          logFile: logFile,
          timeout: timeout,
          env: env
        });
      } else if (benchmark || benchmarkTest) {
        return this.runBenchmarks({
          headless: true,
          test: benchmarkTest,
          resourcePath: this.resourcePath,
          executedFrom: executedFrom,
          pathsToOpen: pathsToOpen,
          timeout: timeout,
          env: env
        });
      } else if (pathsToOpen.length > 0) {
        return this.openPaths({
          initialPaths: initialPaths,
          pathsToOpen: pathsToOpen,
          executedFrom: executedFrom,
          pidToKillWhenClosed: pidToKillWhenClosed,
          newWindow: newWindow,
          devMode: devMode,
          safeMode: safeMode,
          profileStartup: profileStartup,
          clearWindowState: clearWindowState,
          addToLastWindow: addToLastWindow,
          env: env
        });
      } else if (urlsToOpen.length > 0) {
        results = [];
        for (i = 0, len = urlsToOpen.length; i < len; i++) {
          urlToOpen = urlsToOpen[i];
          results.push(this.openUrl({
            urlToOpen: urlToOpen,
            devMode: devMode,
            safeMode: safeMode,
            env: env
          }));
        }
        return results;
      } else {
        return this.openPath({
          initialPaths: initialPaths,
          pidToKillWhenClosed: pidToKillWhenClosed,
          newWindow: newWindow,
          devMode: devMode,
          safeMode: safeMode,
          profileStartup: profileStartup,
          clearWindowState: clearWindowState,
          addToLastWindow: addToLastWindow,
          env: env
        });
      }
    };

    AtomApplication.prototype.removeWindow = function(window) {
      var ref2, ref3;
      this.windows.splice(this.windows.indexOf(window), 1);
      if (this.windows.length === 0) {
        if ((ref2 = this.applicationMenu) != null) {
          ref2.enableWindowSpecificItems(false);
        }
        if ((ref3 = process.platform) === 'win32' || ref3 === 'linux') {
          app.quit();
          return;
        }
      }
      if (!window.isSpec) {
        return this.saveState(true);
      }
    };

    AtomApplication.prototype.addWindow = function(window) {
      var blurHandler, focusHandler, ref2;
      this.windows.push(window);
      if ((ref2 = this.applicationMenu) != null) {
        ref2.addWindow(window.browserWindow);
      }
      window.once('window:loaded', (function(_this) {
        return function() {
          var ref3;
          return (ref3 = _this.autoUpdateManager) != null ? ref3.emitUpdateAvailableEvent(window) : void 0;
        };
      })(this));
      if (!window.isSpec) {
        focusHandler = (function(_this) {
          return function() {
            return _this.lastFocusedWindow = window;
          };
        })(this);
        blurHandler = (function(_this) {
          return function() {
            return _this.saveState(false);
          };
        })(this);
        window.browserWindow.on('focus', focusHandler);
        window.browserWindow.on('blur', blurHandler);
        window.browserWindow.once('closed', (function(_this) {
          return function() {
            if (window === _this.lastFocusedWindow) {
              _this.lastFocusedWindow = null;
            }
            window.browserWindow.removeListener('focus', focusHandler);
            return window.browserWindow.removeListener('blur', blurHandler);
          };
        })(this));
        return window.browserWindow.webContents.once('did-finish-load', (function(_this) {
          return function() {
            return _this.saveState(false);
          };
        })(this));
      }
    };

    AtomApplication.prototype.listenForArgumentsFromNewProcess = function() {
      var server;
      if (this.socketPath == null) {
        return;
      }
      this.deleteSocketFile();
      server = net.createServer((function(_this) {
        return function(connection) {
          var data;
          data = '';
          connection.on('data', function(chunk) {
            return data = data + chunk;
          });
          return connection.on('end', function() {
            var options;
            options = JSON.parse(data);
            return _this.openWithOptions(options);
          });
        };
      })(this));
      server.listen(this.socketPath);
      return server.on('error', function(error) {
        return console.error('Application server failed', error);
      });
    };

    AtomApplication.prototype.deleteSocketFile = function() {
      var error;
      if (process.platform === 'win32' || (this.socketPath == null)) {
        return;
      }
      if (fs.existsSync(this.socketPath)) {
        try {
          return fs.unlinkSync(this.socketPath);
        } catch (error1) {
          error = error1;
          if (error.code !== 'ENOENT') {
            throw error;
          }
        }
      }
    };

    AtomApplication.prototype.handleEvents = function() {
      var clipboard, getLoadSettings;
      getLoadSettings = (function(_this) {
        return function() {
          var ref2, ref3;
          return {
            devMode: (ref2 = _this.focusedWindow()) != null ? ref2.devMode : void 0,
            safeMode: (ref3 = _this.focusedWindow()) != null ? ref3.safeMode : void 0
          };
        };
      })(this);
      this.on('application:quit', function() {
        return app.quit();
      });
      this.on('application:new-window', function() {
        return this.openPath(getLoadSettings());
      });
      this.on('application:new-file', function() {
        var ref2;
        return ((ref2 = this.focusedWindow()) != null ? ref2 : this).openPath();
      });
      this.on('application:open-dev', function() {
        return this.promptForPathToOpen('all', {
          devMode: true
        });
      });
      this.on('application:open-safe', function() {
        return this.promptForPathToOpen('all', {
          safeMode: true
        });
      });
      this.on('application:inspect', function(arg) {
        var atomWindow, x, y;
        x = arg.x, y = arg.y, atomWindow = arg.atomWindow;
        if (atomWindow == null) {
          atomWindow = this.focusedWindow();
        }
        return atomWindow != null ? atomWindow.browserWindow.inspectElement(x, y) : void 0;
      });
      this.on('application:open-documentation', function() {
        return shell.openExternal('http://flight-manual.atom.io/');
      });
      this.on('application:open-discussions', function() {
        return shell.openExternal('https://discuss.atom.io');
      });
      this.on('application:open-faq', function() {
        return shell.openExternal('https://atom.io/faq');
      });
      this.on('application:open-terms-of-use', function() {
        return shell.openExternal('https://atom.io/terms');
      });
      this.on('application:report-issue', function() {
        return shell.openExternal('https://github.com/atom/atom/blob/master/CONTRIBUTING.md#reporting-bugs');
      });
      this.on('application:search-issues', function() {
        return shell.openExternal('https://github.com/search?q=+is%3Aissue+user%3Aatom');
      });
      this.on('application:install-update', (function(_this) {
        return function() {
          _this.quitting = true;
          return _this.autoUpdateManager.install();
        };
      })(this));
      this.on('application:check-for-update', (function(_this) {
        return function() {
          return _this.autoUpdateManager.check();
        };
      })(this));
      if (process.platform === 'darwin') {
        this.on('application:bring-all-windows-to-front', function() {
          return Menu.sendActionToFirstResponder('arrangeInFront:');
        });
        this.on('application:hide', function() {
          return Menu.sendActionToFirstResponder('hide:');
        });
        this.on('application:hide-other-applications', function() {
          return Menu.sendActionToFirstResponder('hideOtherApplications:');
        });
        this.on('application:minimize', function() {
          return Menu.sendActionToFirstResponder('performMiniaturize:');
        });
        this.on('application:unhide-all-applications', function() {
          return Menu.sendActionToFirstResponder('unhideAllApplications:');
        });
        this.on('application:zoom', function() {
          return Menu.sendActionToFirstResponder('zoom:');
        });
      } else {
        this.on('application:minimize', function() {
          var ref2;
          return (ref2 = this.focusedWindow()) != null ? ref2.minimize() : void 0;
        });
        this.on('application:zoom', function() {
          var ref2;
          return (ref2 = this.focusedWindow()) != null ? ref2.maximize() : void 0;
        });
      }
      this.openPathOnEvent('application:about', 'atom://about');
      this.openPathOnEvent('application:show-settings', 'atom://config');
      this.openPathOnEvent('application:open-your-config', 'atom://.atom/config');
      this.openPathOnEvent('application:open-your-init-script', 'atom://.atom/init-script');
      this.openPathOnEvent('application:open-your-keymap', 'atom://.atom/keymap');
      this.openPathOnEvent('application:open-your-snippets', 'atom://.atom/snippets');
      this.openPathOnEvent('application:open-your-stylesheet', 'atom://.atom/stylesheet');
      this.openPathOnEvent('application:open-license', path.join(process.resourcesPath, 'LICENSE.md'));
      this.disposable.add(ipcHelpers.on(app, 'before-quit', (function(_this) {
        return function(event) {
          var resolveBeforeQuitPromise, windowUnloadPromises;
          resolveBeforeQuitPromise = null;
          _this.lastBeforeQuitPromise = new Promise(function(resolve) {
            return resolveBeforeQuitPromise = resolve;
          });
          if (_this.quitting) {
            return resolveBeforeQuitPromise();
          } else {
            event.preventDefault();
            _this.quitting = true;
            windowUnloadPromises = _this.windows.map(function(window) {
              return window.prepareToUnload();
            });
            return Promise.all(windowUnloadPromises).then(function(windowUnloadedResults) {
              var didUnloadAllWindows;
              didUnloadAllWindows = windowUnloadedResults.every(function(didUnloadWindow) {
                return didUnloadWindow;
              });
              if (didUnloadAllWindows) {
                app.quit();
              }
              return resolveBeforeQuitPromise();
            });
          }
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(app, 'will-quit', (function(_this) {
        return function() {
          _this.killAllProcesses();
          return _this.deleteSocketFile();
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(app, 'open-file', (function(_this) {
        return function(event, pathToOpen) {
          event.preventDefault();
          return _this.openPath({
            pathToOpen: pathToOpen
          });
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(app, 'open-url', (function(_this) {
        return function(event, urlToOpen) {
          event.preventDefault();
          return _this.openUrl({
            urlToOpen: urlToOpen,
            devMode: _this.devMode,
            safeMode: _this.safeMode
          });
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(app, 'activate', (function(_this) {
        return function(event, hasVisibleWindows) {
          if (!hasVisibleWindows) {
            if (event != null) {
              event.preventDefault();
            }
            return _this.emit('application:new-window');
          }
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'restart-application', (function(_this) {
        return function() {
          return _this.restart();
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'resolve-proxy', function(event, requestId, url) {
        return event.sender.session.resolveProxy(url, function(proxy) {
          if (!event.sender.isDestroyed()) {
            return event.sender.send('did-resolve-proxy', requestId, proxy);
          }
        });
      }));
      this.disposable.add(ipcHelpers.on(ipcMain, 'did-change-history-manager', (function(_this) {
        return function(event) {
          var atomWindow, i, len, ref2, results, webContents;
          ref2 = _this.windows;
          results = [];
          for (i = 0, len = ref2.length; i < len; i++) {
            atomWindow = ref2[i];
            webContents = atomWindow.browserWindow.webContents;
            if (webContents !== event.sender) {
              results.push(webContents.send('did-change-history-manager'));
            } else {
              results.push(void 0);
            }
          }
          return results;
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'open', (function(_this) {
        return function(event, options) {
          var ref2, window;
          window = _this.atomWindowForEvent(event);
          if (options != null) {
            if (typeof options.pathsToOpen === 'string') {
              options.pathsToOpen = [options.pathsToOpen];
            }
            if (((ref2 = options.pathsToOpen) != null ? ref2.length : void 0) > 0) {
              options.window = window;
              return _this.openPaths(options);
            } else {
              return new AtomWindow(_this, _this.fileRecoveryService, options);
            }
          } else {
            return _this.promptForPathToOpen('all', {
              window: window
            });
          }
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'update-application-menu', (function(_this) {
        return function(event, template, keystrokesByCommand) {
          var ref2, win;
          win = BrowserWindow.fromWebContents(event.sender);
          return (ref2 = _this.applicationMenu) != null ? ref2.update(win, template, keystrokesByCommand) : void 0;
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'run-package-specs', (function(_this) {
        return function(event, packageSpecPath) {
          return _this.runTests({
            resourcePath: _this.devResourcePath,
            pathsToOpen: [packageSpecPath],
            headless: false
          });
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'run-benchmarks', (function(_this) {
        return function(event, benchmarksPath) {
          return _this.runBenchmarks({
            resourcePath: _this.devResourcePath,
            pathsToOpen: [benchmarksPath],
            headless: false,
            test: false
          });
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'command', (function(_this) {
        return function(event, command) {
          return _this.emit(command);
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'open-command', (function(_this) {
        return function() {
          var args, command, defaultPath, event;
          event = arguments[0], command = arguments[1], args = 3 <= arguments.length ? slice.call(arguments, 2) : [];
          if (args.length > 0) {
            defaultPath = args[0];
          }
          switch (command) {
            case 'application:open':
              return _this.promptForPathToOpen('all', getLoadSettings(), defaultPath);
            case 'application:open-file':
              return _this.promptForPathToOpen('file', getLoadSettings(), defaultPath);
            case 'application:open-folder':
              return _this.promptForPathToOpen('folder', getLoadSettings(), defaultPath);
            default:
              return console.log("Invalid open-command received: " + command);
          }
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'window-command', function() {
        var args, command, event, win;
        event = arguments[0], command = arguments[1], args = 3 <= arguments.length ? slice.call(arguments, 2) : [];
        win = BrowserWindow.fromWebContents(event.sender);
        return win.emit.apply(win, [command].concat(slice.call(args)));
      }));
      this.disposable.add(ipcHelpers.respondTo('window-method', (function(_this) {
        return function() {
          var args, browserWindow, method, ref2;
          browserWindow = arguments[0], method = arguments[1], args = 3 <= arguments.length ? slice.call(arguments, 2) : [];
          return (ref2 = _this.atomWindowForBrowserWindow(browserWindow)) != null ? ref2[method].apply(ref2, args) : void 0;
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'pick-folder', (function(_this) {
        return function(event, responseChannel) {
          return _this.promptForPath("folder", function(selectedPaths) {
            return event.sender.send(responseChannel, selectedPaths);
          });
        };
      })(this)));
      this.disposable.add(ipcHelpers.respondTo('set-window-size', function(win, width, height) {
        return win.setSize(width, height);
      }));
      this.disposable.add(ipcHelpers.respondTo('set-window-position', function(win, x, y) {
        return win.setPosition(x, y);
      }));
      this.disposable.add(ipcHelpers.respondTo('center-window', function(win) {
        return win.center();
      }));
      this.disposable.add(ipcHelpers.respondTo('focus-window', function(win) {
        return win.focus();
      }));
      this.disposable.add(ipcHelpers.respondTo('show-window', function(win) {
        return win.show();
      }));
      this.disposable.add(ipcHelpers.respondTo('hide-window', function(win) {
        return win.hide();
      }));
      this.disposable.add(ipcHelpers.respondTo('get-temporary-window-state', function(win) {
        return win.temporaryState;
      }));
      this.disposable.add(ipcHelpers.respondTo('set-temporary-window-state', function(win, state) {
        return win.temporaryState = state;
      }));
      clipboard = require('../safe-clipboard');
      this.disposable.add(ipcHelpers.on(ipcMain, 'write-text-to-selection-clipboard', function(event, selectedText) {
        return clipboard.writeText(selectedText, 'selection');
      }));
      this.disposable.add(ipcHelpers.on(ipcMain, 'write-to-stdout', function(event, output) {
        return process.stdout.write(output);
      }));
      this.disposable.add(ipcHelpers.on(ipcMain, 'write-to-stderr', function(event, output) {
        return process.stderr.write(output);
      }));
      this.disposable.add(ipcHelpers.on(ipcMain, 'add-recent-document', function(event, filename) {
        return app.addRecentDocument(filename);
      }));
      this.disposable.add(ipcHelpers.on(ipcMain, 'execute-javascript-in-dev-tools', function(event, code) {
        var ref2;
        return (ref2 = event.sender.devToolsWebContents) != null ? ref2.executeJavaScript(code) : void 0;
      }));
      this.disposable.add(ipcHelpers.on(ipcMain, 'get-auto-update-manager-state', (function(_this) {
        return function(event) {
          return event.returnValue = _this.autoUpdateManager.getState();
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'get-auto-update-manager-error', (function(_this) {
        return function(event) {
          return event.returnValue = _this.autoUpdateManager.getErrorMessage();
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'will-save-path', (function(_this) {
        return function(event, path) {
          _this.fileRecoveryService.willSavePath(_this.atomWindowForEvent(event), path);
          return event.returnValue = true;
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'did-save-path', (function(_this) {
        return function(event, path) {
          _this.fileRecoveryService.didSavePath(_this.atomWindowForEvent(event), path);
          return event.returnValue = true;
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'did-change-paths', (function(_this) {
        return function() {
          return _this.saveState(false);
        };
      })(this)));
      return this.disposable.add(this.disableZoomOnDisplayChange());
    };

    AtomApplication.prototype.setupDockMenu = function() {
      var dockMenu;
      if (process.platform === 'darwin') {
        dockMenu = Menu.buildFromTemplate([
          {
            label: 'New Window',
            click: (function(_this) {
              return function() {
                return _this.emit('application:new-window');
              };
            })(this)
          }
        ]);
        return app.dock.setMenu(dockMenu);
      }
    };

    AtomApplication.prototype.sendCommand = function() {
      var args, command, focusedWindow;
      command = arguments[0], args = 2 <= arguments.length ? slice.call(arguments, 1) : [];
      if (!this.emit.apply(this, [command].concat(slice.call(args)))) {
        focusedWindow = this.focusedWindow();
        if (focusedWindow != null) {
          return focusedWindow.sendCommand.apply(focusedWindow, [command].concat(slice.call(args)));
        } else {
          return this.sendCommandToFirstResponder(command);
        }
      }
    };

    AtomApplication.prototype.sendCommandToWindow = function() {
      var args, atomWindow, command;
      command = arguments[0], atomWindow = arguments[1], args = 3 <= arguments.length ? slice.call(arguments, 2) : [];
      if (!this.emit.apply(this, [command].concat(slice.call(args)))) {
        if (atomWindow != null) {
          return atomWindow.sendCommand.apply(atomWindow, [command].concat(slice.call(args)));
        } else {
          return this.sendCommandToFirstResponder(command);
        }
      }
    };

    AtomApplication.prototype.sendCommandToFirstResponder = function(command) {
      if (process.platform !== 'darwin') {
        return false;
      }
      switch (command) {
        case 'core:undo':
          Menu.sendActionToFirstResponder('undo:');
          break;
        case 'core:redo':
          Menu.sendActionToFirstResponder('redo:');
          break;
        case 'core:copy':
          Menu.sendActionToFirstResponder('copy:');
          break;
        case 'core:cut':
          Menu.sendActionToFirstResponder('cut:');
          break;
        case 'core:paste':
          Menu.sendActionToFirstResponder('paste:');
          break;
        case 'core:select-all':
          Menu.sendActionToFirstResponder('selectAll:');
          break;
        default:
          return false;
      }
      return true;
    };

    AtomApplication.prototype.openPathOnEvent = function(eventName, pathToOpen) {
      return this.on(eventName, function() {
        var window;
        if (window = this.focusedWindow()) {
          return window.openPath(pathToOpen);
        } else {
          return this.openPath({
            pathToOpen: pathToOpen
          });
        }
      });
    };

    AtomApplication.prototype.windowForPaths = function(pathsToOpen, devMode) {
      return _.find(this.windows, function(atomWindow) {
        return atomWindow.devMode === devMode && atomWindow.containsPaths(pathsToOpen);
      });
    };

    AtomApplication.prototype.atomWindowForEvent = function(arg) {
      var sender;
      sender = arg.sender;
      return this.atomWindowForBrowserWindow(BrowserWindow.fromWebContents(sender));
    };

    AtomApplication.prototype.atomWindowForBrowserWindow = function(browserWindow) {
      return this.windows.find(function(atomWindow) {
        return atomWindow.browserWindow === browserWindow;
      });
    };

    AtomApplication.prototype.focusedWindow = function() {
      return _.find(this.windows, function(atomWindow) {
        return atomWindow.isFocused();
      });
    };

    AtomApplication.prototype.getWindowOffsetForCurrentPlatform = function() {
      var offsetByPlatform, ref2;
      offsetByPlatform = {
        darwin: 22,
        win32: 26
      };
      return (ref2 = offsetByPlatform[process.platform]) != null ? ref2 : 0;
    };

    AtomApplication.prototype.getDimensionsForNewWindow = function() {
      var dimensions, offset, ref2, ref3, ref4, ref5;
      if ((ref2 = (ref3 = this.focusedWindow()) != null ? ref3 : this.lastFocusedWindow) != null ? ref2.isMaximized() : void 0) {
        return;
      }
      dimensions = (ref4 = (ref5 = this.focusedWindow()) != null ? ref5 : this.lastFocusedWindow) != null ? ref4.getDimensions() : void 0;
      offset = this.getWindowOffsetForCurrentPlatform();
      if ((dimensions != null) && (offset != null)) {
        dimensions.x += offset;
        dimensions.y += offset;
      }
      return dimensions;
    };

    AtomApplication.prototype.openPath = function(arg) {
      var addToLastWindow, clearWindowState, devMode, env, initialPaths, newWindow, pathToOpen, pidToKillWhenClosed, profileStartup, ref2, safeMode, window;
      ref2 = arg != null ? arg : {}, initialPaths = ref2.initialPaths, pathToOpen = ref2.pathToOpen, pidToKillWhenClosed = ref2.pidToKillWhenClosed, newWindow = ref2.newWindow, devMode = ref2.devMode, safeMode = ref2.safeMode, profileStartup = ref2.profileStartup, window = ref2.window, clearWindowState = ref2.clearWindowState, addToLastWindow = ref2.addToLastWindow, env = ref2.env;
      return this.openPaths({
        initialPaths: initialPaths,
        pathsToOpen: [pathToOpen],
        pidToKillWhenClosed: pidToKillWhenClosed,
        newWindow: newWindow,
        devMode: devMode,
        safeMode: safeMode,
        profileStartup: profileStartup,
        window: window,
        clearWindowState: clearWindowState,
        addToLastWindow: addToLastWindow,
        env: env
      });
    };

    AtomApplication.prototype.openPaths = function(arg) {
      var addToLastWindow, clearWindowState, currentWindow, devMode, env, executedFrom, existingWindow, initialPaths, locationToOpen, locationsToOpen, newWindow, openedWindow, pathToOpen, pathsToOpen, pidToKillWhenClosed, profileStartup, ref2, resourcePath, safeMode, stats, window, windowDimensions, windowInitializationScript;
      ref2 = arg != null ? arg : {}, initialPaths = ref2.initialPaths, pathsToOpen = ref2.pathsToOpen, executedFrom = ref2.executedFrom, pidToKillWhenClosed = ref2.pidToKillWhenClosed, newWindow = ref2.newWindow, devMode = ref2.devMode, safeMode = ref2.safeMode, windowDimensions = ref2.windowDimensions, profileStartup = ref2.profileStartup, window = ref2.window, clearWindowState = ref2.clearWindowState, addToLastWindow = ref2.addToLastWindow, env = ref2.env;
      if ((pathsToOpen == null) || pathsToOpen.length === 0) {
        return;
      }
      if (env == null) {
        env = process.env;
      }
      devMode = Boolean(devMode);
      safeMode = Boolean(safeMode);
      clearWindowState = Boolean(clearWindowState);
      locationsToOpen = (function() {
        var i, len, results;
        results = [];
        for (i = 0, len = pathsToOpen.length; i < len; i++) {
          pathToOpen = pathsToOpen[i];
          results.push(this.locationForPathToOpen(pathToOpen, executedFrom, addToLastWindow));
        }
        return results;
      }).call(this);
      pathsToOpen = (function() {
        var i, len, results;
        results = [];
        for (i = 0, len = locationsToOpen.length; i < len; i++) {
          locationToOpen = locationsToOpen[i];
          results.push(locationToOpen.pathToOpen);
        }
        return results;
      })();
      if (!(pidToKillWhenClosed || newWindow)) {
        existingWindow = this.windowForPaths(pathsToOpen, devMode);
        stats = (function() {
          var i, len, results;
          results = [];
          for (i = 0, len = pathsToOpen.length; i < len; i++) {
            pathToOpen = pathsToOpen[i];
            results.push(fs.statSyncNoException(pathToOpen));
          }
          return results;
        })();
        if (existingWindow == null) {
          if (currentWindow = window != null ? window : this.lastFocusedWindow) {
            if (addToLastWindow || currentWindow.devMode === devMode && (stats.every(function(stat) {
              return typeof stat.isFile === "function" ? stat.isFile() : void 0;
            }) || stats.some(function(stat) {
              return (typeof stat.isDirectory === "function" ? stat.isDirectory() : void 0) && !currentWindow.hasProjectPath();
            }))) {
              existingWindow = currentWindow;
            }
          }
        }
      }
      if (existingWindow != null) {
        openedWindow = existingWindow;
        openedWindow.openLocations(locationsToOpen);
        if (openedWindow.isMinimized()) {
          openedWindow.restore();
        } else {
          openedWindow.focus();
        }
        openedWindow.replaceEnvironment(env);
      } else {
        if (devMode) {
          try {
            windowInitializationScript = require.resolve(path.join(this.devResourcePath, 'src', 'initialize-application-window'));
            resourcePath = this.devResourcePath;
          } catch (error1) {}
        }
        if (windowInitializationScript == null) {
          windowInitializationScript = require.resolve('../initialize-application-window');
        }
        if (resourcePath == null) {
          resourcePath = this.resourcePath;
        }
        if (windowDimensions == null) {
          windowDimensions = this.getDimensionsForNewWindow();
        }
        openedWindow = new AtomWindow(this, this.fileRecoveryService, {
          initialPaths: initialPaths,
          locationsToOpen: locationsToOpen,
          windowInitializationScript: windowInitializationScript,
          resourcePath: resourcePath,
          devMode: devMode,
          safeMode: safeMode,
          windowDimensions: windowDimensions,
          profileStartup: profileStartup,
          clearWindowState: clearWindowState,
          env: env
        });
        openedWindow.focus();
        this.lastFocusedWindow = openedWindow;
      }
      if (pidToKillWhenClosed != null) {
        this.pidsToOpenWindows[pidToKillWhenClosed] = openedWindow;
      }
      openedWindow.browserWindow.once('closed', (function(_this) {
        return function() {
          return _this.killProcessForWindow(openedWindow);
        };
      })(this));
      return openedWindow;
    };

    AtomApplication.prototype.killAllProcesses = function() {
      var pid;
      for (pid in this.pidsToOpenWindows) {
        this.killProcess(pid);
      }
    };

    AtomApplication.prototype.killProcessForWindow = function(openedWindow) {
      var pid, ref2, trackedWindow;
      ref2 = this.pidsToOpenWindows;
      for (pid in ref2) {
        trackedWindow = ref2[pid];
        if (trackedWindow === openedWindow) {
          this.killProcess(pid);
        }
      }
    };

    AtomApplication.prototype.killProcess = function(pid) {
      var error, parsedPid, ref2;
      try {
        parsedPid = parseInt(pid);
        if (isFinite(parsedPid)) {
          process.kill(parsedPid);
        }
      } catch (error1) {
        error = error1;
        if (error.code !== 'ESRCH') {
          console.log("Killing process " + pid + " failed: " + ((ref2 = error.code) != null ? ref2 : error.message));
        }
      }
      return delete this.pidsToOpenWindows[pid];
    };

    AtomApplication.prototype.saveState = function(allowEmpty) {
      var i, len, ref2, states, window;
      if (allowEmpty == null) {
        allowEmpty = false;
      }
      if (this.quitting) {
        return;
      }
      states = [];
      ref2 = this.windows;
      for (i = 0, len = ref2.length; i < len; i++) {
        window = ref2[i];
        if (!window.isSpec) {
          states.push({
            initialPaths: window.representedDirectoryPaths
          });
        }
      }
      if (states.length > 0 || allowEmpty) {
        this.storageFolder.storeSync('application.json', states);
        return this.emit('application:did-save-state');
      }
    };

    AtomApplication.prototype.loadState = function(options) {
      var i, len, ref2, ref3, results, state, states;
      if (((ref2 = this.config.get('core.restorePreviousWindowsOnStart')) === 'yes' || ref2 === 'always') && ((ref3 = (states = this.storageFolder.load('application.json'))) != null ? ref3.length : void 0) > 0) {
        results = [];
        for (i = 0, len = states.length; i < len; i++) {
          state = states[i];
          results.push(this.openWithOptions(Object.assign(options, {
            initialPaths: state.initialPaths,
            pathsToOpen: state.initialPaths.filter(function(directoryPath) {
              return fs.isDirectorySync(directoryPath);
            }),
            urlsToOpen: [],
            devMode: this.devMode,
            safeMode: this.safeMode
          })));
        }
        return results;
      } else {
        return null;
      }
    };

    AtomApplication.prototype.openUrl = function(arg) {
      var PackageManager, devMode, env, pack, packageName, packagePath, safeMode, urlToOpen, windowDimensions, windowInitializationScript;
      urlToOpen = arg.urlToOpen, devMode = arg.devMode, safeMode = arg.safeMode, env = arg.env;
      if (this.packages == null) {
        PackageManager = require('../package-manager');
        this.packages = new PackageManager({});
        this.packages.initialize({
          configDirPath: process.env.ATOM_HOME,
          devMode: devMode,
          resourcePath: this.resourcePath
        });
      }
      packageName = url.parse(urlToOpen).host;
      pack = _.find(this.packages.getAvailablePackageMetadata(), function(arg1) {
        var name;
        name = arg1.name;
        return name === packageName;
      });
      if (pack != null) {
        if (pack.urlMain) {
          packagePath = this.packages.resolvePackagePath(packageName);
          windowInitializationScript = path.resolve(packagePath, pack.urlMain);
          windowDimensions = this.getDimensionsForNewWindow();
          return new AtomWindow(this, this.fileRecoveryService, {
            windowInitializationScript: windowInitializationScript,
            resourcePath: this.resourcePath,
            devMode: devMode,
            safeMode: safeMode,
            urlToOpen: urlToOpen,
            windowDimensions: windowDimensions,
            env: env
          });
        } else {
          return console.log("Package '" + pack.name + "' does not have a url main: " + urlToOpen);
        }
      } else {
        return console.log("Opening unknown url: " + urlToOpen);
      }
    };

    AtomApplication.prototype.runTests = function(arg) {
      var devMode, env, error, executedFrom, headless, i, isSpec, legacyTestRunnerPath, len, logFile, pathToOpen, pathsToOpen, resourcePath, safeMode, testPaths, testRunnerPath, timeout, timeoutHandler, timeoutInSeconds, windowInitializationScript;
      headless = arg.headless, resourcePath = arg.resourcePath, executedFrom = arg.executedFrom, pathsToOpen = arg.pathsToOpen, logFile = arg.logFile, safeMode = arg.safeMode, timeout = arg.timeout, env = arg.env;
      if (resourcePath !== this.resourcePath && !fs.existsSync(resourcePath)) {
        resourcePath = this.resourcePath;
      }
      timeoutInSeconds = Number.parseFloat(timeout);
      if (!Number.isNaN(timeoutInSeconds)) {
        timeoutHandler = function() {
          console.log("The test suite has timed out because it has been running for more than " + timeoutInSeconds + " seconds.");
          return process.exit(124);
        };
        setTimeout(timeoutHandler, timeoutInSeconds * 1000);
      }
      try {
        windowInitializationScript = require.resolve(path.resolve(this.devResourcePath, 'src', 'initialize-test-window'));
      } catch (error1) {
        error = error1;
        windowInitializationScript = require.resolve(path.resolve(__dirname, '..', '..', 'src', 'initialize-test-window'));
      }
      testPaths = [];
      if (pathsToOpen != null) {
        for (i = 0, len = pathsToOpen.length; i < len; i++) {
          pathToOpen = pathsToOpen[i];
          testPaths.push(path.resolve(executedFrom, fs.normalize(pathToOpen)));
        }
      }
      if (testPaths.length === 0) {
        process.stderr.write('Error: Specify at least one test path\n\n');
        process.exit(1);
      }
      legacyTestRunnerPath = this.resolveLegacyTestRunnerPath();
      testRunnerPath = this.resolveTestRunnerPath(testPaths[0]);
      devMode = true;
      isSpec = true;
      if (safeMode == null) {
        safeMode = false;
      }
      return new AtomWindow(this, this.fileRecoveryService, {
        windowInitializationScript: windowInitializationScript,
        resourcePath: resourcePath,
        headless: headless,
        isSpec: isSpec,
        devMode: devMode,
        testRunnerPath: testRunnerPath,
        legacyTestRunnerPath: legacyTestRunnerPath,
        testPaths: testPaths,
        logFile: logFile,
        safeMode: safeMode,
        env: env
      });
    };

    AtomApplication.prototype.runBenchmarks = function(arg) {
      var benchmarkPaths, devMode, env, error, executedFrom, headless, i, isSpec, len, pathToOpen, pathsToOpen, resourcePath, safeMode, test, windowInitializationScript;
      headless = arg.headless, test = arg.test, resourcePath = arg.resourcePath, executedFrom = arg.executedFrom, pathsToOpen = arg.pathsToOpen, env = arg.env;
      if (resourcePath !== this.resourcePath && !fs.existsSync(resourcePath)) {
        resourcePath = this.resourcePath;
      }
      try {
        windowInitializationScript = require.resolve(path.resolve(this.devResourcePath, 'src', 'initialize-benchmark-window'));
      } catch (error1) {
        error = error1;
        windowInitializationScript = require.resolve(path.resolve(__dirname, '..', '..', 'src', 'initialize-benchmark-window'));
      }
      benchmarkPaths = [];
      if (pathsToOpen != null) {
        for (i = 0, len = pathsToOpen.length; i < len; i++) {
          pathToOpen = pathsToOpen[i];
          benchmarkPaths.push(path.resolve(executedFrom, fs.normalize(pathToOpen)));
        }
      }
      if (benchmarkPaths.length === 0) {
        process.stderr.write('Error: Specify at least one benchmark path.\n\n');
        process.exit(1);
      }
      devMode = true;
      isSpec = true;
      safeMode = false;
      return new AtomWindow(this, this.fileRecoveryService, {
        windowInitializationScript: windowInitializationScript,
        resourcePath: resourcePath,
        headless: headless,
        test: test,
        isSpec: isSpec,
        devMode: devMode,
        benchmarkPaths: benchmarkPaths,
        safeMode: safeMode,
        env: env
      });
    };

    AtomApplication.prototype.resolveTestRunnerPath = function(testPath) {
      var packageMetadata, packageRoot, testRunnerPath;
      if (FindParentDir == null) {
        FindParentDir = require('find-parent-dir');
      }
      if (packageRoot = FindParentDir.sync(testPath, 'package.json')) {
        packageMetadata = require(path.join(packageRoot, 'package.json'));
        if (packageMetadata.atomTestRunner) {
          if (Resolve == null) {
            Resolve = require('resolve');
          }
          if (testRunnerPath = Resolve.sync(packageMetadata.atomTestRunner, {
            basedir: packageRoot,
            extensions: Object.keys(require.extensions)
          })) {
            return testRunnerPath;
          } else {
            process.stderr.write("Error: Could not resolve test runner path '" + packageMetadata.atomTestRunner + "'");
            process.exit(1);
          }
        }
      }
      return this.resolveLegacyTestRunnerPath();
    };

    AtomApplication.prototype.resolveLegacyTestRunnerPath = function() {
      var error;
      try {
        return require.resolve(path.resolve(this.devResourcePath, 'spec', 'jasmine-test-runner'));
      } catch (error1) {
        error = error1;
        return require.resolve(path.resolve(__dirname, '..', '..', 'spec', 'jasmine-test-runner'));
      }
    };

    AtomApplication.prototype.locationForPathToOpen = function(pathToOpen, executedFrom, forceAddToWindow) {
      var initialColumn, initialLine, match;
      if (executedFrom == null) {
        executedFrom = '';
      }
      if (!pathToOpen) {
        return {
          pathToOpen: pathToOpen
        };
      }
      pathToOpen = pathToOpen.replace(/[:\s]+$/, '');
      match = pathToOpen.match(LocationSuffixRegExp);
      if (match != null) {
        pathToOpen = pathToOpen.slice(0, -match[0].length);
        if (match[1]) {
          initialLine = Math.max(0, parseInt(match[1].slice(1)) - 1);
        }
        if (match[2]) {
          initialColumn = Math.max(0, parseInt(match[2].slice(1)) - 1);
        }
      } else {
        initialLine = initialColumn = null;
      }
      if (url.parse(pathToOpen).protocol == null) {
        pathToOpen = path.resolve(executedFrom, fs.normalize(pathToOpen));
      }
      return {
        pathToOpen: pathToOpen,
        initialLine: initialLine,
        initialColumn: initialColumn,
        forceAddToWindow: forceAddToWindow
      };
    };

    AtomApplication.prototype.promptForPathToOpen = function(type, arg, path) {
      var devMode, safeMode, window;
      devMode = arg.devMode, safeMode = arg.safeMode, window = arg.window;
      if (path == null) {
        path = null;
      }
      return this.promptForPath(type, ((function(_this) {
        return function(pathsToOpen) {
          return _this.openPaths({
            pathsToOpen: pathsToOpen,
            devMode: devMode,
            safeMode: safeMode,
            window: window
          });
        };
      })(this)), path);
    };

    AtomApplication.prototype.promptForPath = function(type, callback, path) {
      var openOptions, parentWindow, properties;
      properties = (function() {
        switch (type) {
          case 'file':
            return ['openFile'];
          case 'folder':
            return ['openDirectory'];
          case 'all':
            return ['openFile', 'openDirectory'];
          default:
            throw new Error(type + " is an invalid type for promptForPath");
        }
      })();
      parentWindow = process.platform === 'darwin' ? null : BrowserWindow.getFocusedWindow();
      openOptions = {
        properties: properties.concat(['multiSelections', 'createDirectory']),
        title: (function() {
          switch (type) {
            case 'file':
              return 'Open File';
            case 'folder':
              return 'Open Folder';
            default:
              return 'Open';
          }
        })()
      };
      if (path != null) {
        openOptions.defaultPath = path;
      }
      return dialog.showOpenDialog(parentWindow, openOptions, callback);
    };

    AtomApplication.prototype.promptForRestart = function() {
      var chosen;
      chosen = dialog.showMessageBox(BrowserWindow.getFocusedWindow(), {
        type: 'warning',
        title: 'Restart required',
        message: "You will need to restart Atom for this change to take effect.",
        buttons: ['Restart Atom', 'Cancel']
      });
      if (chosen === 0) {
        return this.restart();
      }
    };

    AtomApplication.prototype.restart = function() {
      var args;
      args = [];
      if (this.safeMode) {
        args.push("--safe");
      }
      if (this.logFile != null) {
        args.push("--log-file=" + this.logFile);
      }
      if (this.socketPath != null) {
        args.push("--socket-path=" + this.socketPath);
      }
      if (this.userDataDir != null) {
        args.push("--user-data-dir=" + this.userDataDir);
      }
      if (this.devMode) {
        args.push('--dev');
        args.push("--resource-path=" + this.resourcePath);
      }
      app.relaunch({
        args: args
      });
      return app.quit();
    };

    AtomApplication.prototype.disableZoomOnDisplayChange = function() {
      var outerCallback;
      outerCallback = (function(_this) {
        return function() {
          var i, len, ref2, results, window;
          ref2 = _this.windows;
          results = [];
          for (i = 0, len = ref2.length; i < len; i++) {
            window = ref2[i];
            results.push(window.disableZoom());
          }
          return results;
        };
      })(this);
      screen.on('display-added', outerCallback);
      screen.on('display-removed', outerCallback);
      return new Disposable(function() {
        screen.removeListener('display-added', outerCallback);
        return screen.removeListener('display-removed', outerCallback);
      });
    };

    return AtomApplication;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
